//---------------------------------------------------------------------------
// File  :  \CBproj\SoundUtl\NoiseBlanker.h
// Date  :  2002-05-05  (yyyy-mm-dd)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Interface for the CNoiseBlanker class.
//
// Revision history :  See *.cpp
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef NoiseBlankerH
#define NoiseBlankerH

#define NOISEBLANKER_USE_FFT  0  // FFT was too complicated and slow

#include "SoundTab.h"  // frequently used types and tables for audio processing

#if(NOISEBLANKER_USE_FFT)
  #include "fft.h"     // fast fourier transform needed for noise detection
#endif // NOISEBLANKER_USE_FFT


/*** See the quoted mail from DF6NM's mail to the RSGB LF group in *.cpp ! ***/



//----------------------- Constants ---------------------------------------

    // caution: the NOISEBLANKER TYPE may be used somewhere in a combo list
#define NBL_TYPE_OFF             0
#define NBL_TYPE_CRUDE_CLIPPER   1


#define NBL_FFT_SIZE 128


//----------------------- Data Types --------------------------------------

#ifndef _T_CPLX_
#define _T_CPLX_
typedef struct
{
  double re;
  double im;
} T_CPLX;
#endif // _T_CPLX_




//***************************************************************************
//  Definition of the  CNoiseBlanker   class
//***************************************************************************

class CNoiseBlanker
{
public:
   CNoiseBlanker();            // constructor without parameters
   virtual ~CNoiseBlanker();   // destructor (cleans up)

   BOOL Init(  int  iMode, T_Float fltSampleRate);  // samples per second

   BOOL ProcessSamples( T_Float *pdblSource, T_Float *pdblDest,
               int iNrSamplePoints );
        // Called by the audio-processing thread.
        // Be careful not to call this while the Init()-method
        // is busy - may crash when modifying the buffer sze .
        // All 'Set'-methods below may be called while ProcessSamples()
        //   is working (to change parameters on-the-fly) .
        // Note: pdblSource and pdblDest may point to the same
        //       memory location .


   T_Float GetRampTime(void);
   BOOL    SetRampTime(T_Float fltRampTime);
        // A good starting value for the ramp time is 10 milliseconds,
        //  or a little more to reduce the spectral spreading of signals
        //  caused by the 'amplitude modulation' from the blanker
        //  which affects all signals; coherent or not !

   BOOL SetTriggerLevel( T_Float fltMinPeakToAvrg_dB );
        // Set 'trigger level' for noiseblanker in DECIBEL .
        //  Everything which does not exceed this level
        //  above the peak value of the past <ramp time> seconds
        //  is not 'considered harmful' and will not be notched.


   BOOL SetMaxPulseWidth( T_Float fltNoisePulseWidth_sec );
        // Set 'maximum noise pulse width' in seconds
        // (everything LONGER than this may be strong morse-code dots,
        //  'beeps' etc.)

   BOOL SetMaxPulseRiseTime( T_Float fltNoiseRiseTime_sec );
        // Set the 'slowest pulse rise time' in seconds
        //  (signals with a slower rise are not considered noise,
        //   which usually separates narrow-band signals
        //   from lightning strokes, which rise within a few usec).

   T_Float GetNormalizedAvrgNoiseLevel(void);
   T_Float GetNormalizedPeakNoiseLevel(void);



private:
   T_Float m_fltSampleRate;
   WORD   m_wFftInputIndex;
   T_Float *m_pFltBuffer;   // circular buffer ..
   DWORD    m_dwBufferSize;
   DWORD    m_dwBufIndex;
    // ( need some time to detect pulse noise, and to start smooth ramp
    //   so there is the proper attenuation at the pulses peak ! )

#if(NOISEBLANKER_USE_FFT)
   Cfft *m_pFft;  // the utterly cluttered FFT class  --- no longer used  ;-)
   T_Float m_fltFftInput[NBL_FFT_SIZE];
   float  m_fltFftOutput[NBL_FFT_SIZE];
#endif // NOISEBLANKER_USE_FFT

   T_Float m_fltCurrNoiseLevel;
   #define NBL_AVRG_LP_STAGES 20                // good enough for this purpose ?
   T_Float m_fltAvLowpass[NBL_AVRG_LP_STAGES];  // lowpass function can be checked with SCOPE
   T_Float m_fltAvrgNoisePower, m_fltPeakNoisePower;
   T_Float m_fltCurrBlankingMultiplier;
   T_Float m_fltMinPeak2AvrgPowerRatio;

   T_Float m_fltRampTime, m_fltOldRampTime;
   int     m_iRampLength, m_iRampIndex, m_iRampDir;
   
   void ApplyNewRampTime(void);


}; // end class CNoiseBlanker


#endif // NoiseBlankerH