/***************************************************************************/
/*  JasFileIO.h :                                                          */
/*   Unit to import and export *.jas - files.                              */
/*   - by DL4YHF August '2002                                              */
/*   - based on the description of the JasDataHdr by Alberto (I2PHD)       */
/*     which is a specialized audio logging format with I/Q-samples        */
/*     as double precision floats, means 2*8 bytes per sample.             */
/*-------------------------------------------------------------------------*/

#ifndef JasFileIO_H
#define JasFileIO_H


#include "SoundUps.h"   // upsampling audio buffer with optional I/Q processor


#ifndef  SWI_FLOAT_PRECISION
 #define SWI_FLOAT_PRECISION 1 /* 1=single precision, 2=double precision */
#endif

#if SWI_FLOAT_PRECISION==1
 #define T_Float float
 // 2003-07-07: "float" instead of "double" saves CPU power on stoneage notebooks
#else
 #define T_Float double
#endif



/*----------- Constant Definitions ----------------------------------------*/




/*----------- Data Types --------------------------------------------------*/


// JASON file header structure :
typedef struct
{
   char tag[8];       // 'JasData '
   char hdrsize[4];   // '0068'
   char when[24];     // 'Fri Aug 09 21:20:05 2002'
   char nco_freq[16]; // '1003.23339843750'
   char samprate[16]; // '43.0664062500000'
} T_JasDataHdr;


/*----------- Definition of the C_JasFileIO  class -------------------------*/
class C_JasFileIO
{
public:
   C_JasFileIO();
   virtual ~C_JasFileIO();

   // Public member functions for object's users.
   /************************************************************************/
   BOOL InOpen( char *pszFileName, T_Float dblReadingSampleRate );
     /* Opens a wave file for READING it's samples.
      * Returns TRUE on success and FALSE on any error.
      */


   /************************************************************************/
   LONG GetCurrentSampleIndex(void);
     /* Reads the current sample index that will be used on the next
      * READ- or WRITE- access to the audio samples in the opened file.
      * Returns a negative value on any error.
      */


   /************************************************************************/
   BOOL WindToSampleIndex(long new_sample_index);
     /* Sets the current sample index that will be used on the next
      * READ- or WRITE- access to the audio samples an opened file.
      * Returns TRUE if ok ,  otherwise FALSE.
      * As long as you don't use this routine/method,
      * the audio file will be played in a strictly "sequential" way.
      */


   /************************************************************************/
   LONG ReadSamples( BYTE* pData, LONG Length );
     /* Reads some samples from a wave file which has been opened for READING.
      * Returns the NUMBER OF AUDIO SAMPLES if successful,
      *         or a negative value if errors occurred.
      * Note: "Length" is the size of the caller's buffer capacity in BYTES.
      *       The actual number of AUDIO SAMPLES will depend on the sample
      *       format. For 16-bit mono recordings, the number of samples
      *       will only be half the "Length".
      *       Generally, the number of audio samples that will be read is:
      *         n_samples = Length / m_ChunkFormat.wBlockAlign.
      */


   /************************************************************************/
   LONG ReadSampleBlocks(
         int channel_nr,     // channel_nr for 1st destination block
         int nr_samples,     // nr_samples
         T_Float *pFltDest1,  // 1st destination block (~"LEFT")
         T_Float *pFltDest2); // 2nd destination block (~"RIGHT")
     /* Reads some samples from a wave file which has been opened for READING
      * and converts them to double (floats) in the range of +- 32768.0
      * Input parameters:
      *   channel_nr: 0=first channel (LEFT if stereo recording)
      *    nr_samples: Max number of samples which can be stored in T_Float dest[0..nr_samples-1]
      *   *dest: Pointer to destination array.
      * Returns the NUMBER OF AUDIO SAMPLES if successful,
      *         or a negative value if errors occurred.
      * Note that returning less samples than you may expect is not an error
      * but indicates reaching the file's end !
      */



   /************************************************************************/
   BOOL OutOpen( char *pszFileName,
               double dblFileSampleRate,
               double dblFeedingSampleRate,
               double dblNcoFrequency );
     /* Creates and Opens an audio file for WRITING audio samples.
      *         Returns TRUE on success and FALSE on any error.
      *         Detailed description of arguments in WaveIO.cpp only.
      */


   /************************************************************************/
   LONG WriteSamples( BYTE* pData, LONG Length );
     /* Writes some samples to a wave file which has been opened for WRITING.
      * Returns TRUE on success and FALSE on any error.
      * Note: "Length" is the size of the caller's buffer capacity in BYTES.
      *       The actual number of AUDIO SAMPLES will depend on the sample
      *       format. For 16-bit mono recordings, the number of samples
      *       will only be half the "Length" etc.
      */


   /************************************************************************/
   LONG GetCurrentFileSize(void);
     /* Returns the current file size (if there is an opened WAVE file).
      * Useful especially when logging audio samples  to keep track
      * of the used disk size (stop before Windows is getting into trouble).
      */

   /************************************************************************/
   // Some more 'Get' - and 'Set' - routines for the AUDIO FILE class ..
   int    GetFileFormat(void);
   void   GetFileName(char *pszDest, int iMaxLen);

   double GetSampleRate(void);
   void   SetSampleRate(double dblSampleRate);

   int    GetNrChannels(void);
   inline int GetBitsPerSample(void) { return 8 * sizeof(T_Complex); }
   double GetFreqOffset(void);


   double GetNcoFrequency(void);
   BOOL   SetNcoFrequency(double dblNcoFrequency);

   double GetCurrentRecordingTime(void); // .. depending on current sample index
   double GetRecordingStartTime(void);
   void   SetRecordingStartTime(double dblStartTime);

   bool   AllParamsIncluded(void);

   /************************************************************************/
   BOOL CloseFile( void );
     /* Closes the WAV-file (if opened for READING exor WRITING.
      * Returns TRUE on success and FALSE on any error.
      */


   // Public properties for simplicity .. we know what we're doing, right ?
   T_JasDataHdr       m_JasDataHdr;

   double m_dblStartTime;  // seconds elapsed since 00:00:00 GMT, January 1, 1970.
   int    m_iFileFormat;
   char   m_cErrorString[81];
   BOOL   m_OpenedForReading;
   BOOL   m_OpenedForWriting;
   double m_dbl_NcoFrequency;

private:
   HANDLE m_hFile;
   LONG  m_lFilePos_Data;   // file-internal position of 1st audio sample
   LONG  m_lCurrFilePos;    // current file access position
   LONG  m_lFileDataSize;   
   char  m_sz255FileName[256];
   double m_dblFileSampleRate;    

   CSoundUpsamplingBuffer UpsamplingAudioBuffer;

}; /* end class C_JasFileIO */


#endif // JasFileIO_H

/* EOF <JasFileIO.h> */

