//---------------------------------------------------------------------------
// File  :  C:\CBproj\SoundUtl\GpsPulseDetector.h
// Date  :  2012-02-23 (ISO 8601, YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//    Header file for the 'GPS pulse detector', first used by Spectrum Lab .
//    Used to continuously 'calibrate' the soundcard's sample rate
//    with the 1-PPS (pulse-per-second) output from a suitable GPS receiver .
//    Later drilled up to provide high-resolution timestamps, using
//    the GPS'es NMEA output added to the PPS signal with a passive combiner
//    ( see  http://www.qsl.net/dl4yhf/speclab/frqcalib.htm#gps_pps_output )
//
//
//  Copyright (c) 2012..2015, Wolfang Buescher
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//
//  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
//  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
//  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Revision history :  See *.cpp !
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef _GpsPulseDetector_H_
#define _GpsPulseDetector_H_

#ifdef BUILDING_THE_DLL  /* defined somewhere under 'Project..Options'  */
# define DLLIMPEX __declspec(dllexport)  /* here: DLLIMPEX = DLL EXPORT */
#else
# define DLLIMPEX __declspec(dllimport)  /* here: DLLIMPEX = DLL IMPORT */
#endif


#include "SoundTab.h"    // global tables, most important : cosine lookup table
#include "SoundMaths.h"  // some frequently used math subroutines, FFT, IFFT, ..
#include "CircuitDefs.h" // definitions for the 'activity indicators' / circuit states in SL
  // (CircuitDefs.h may be commented out when not available.. see evil #ifdefs somewhere)

#include "SoundDec.h"    // Complex decimating audio buffer, here: used as narrow-band phase meter
#include "gps_dec.h"     // general GPS/NMEA decoder/parser

//----------------------- Constants ---------------------------------------


// other 'frequently needed' stuff..
#ifndef  C_PI
 #define C_PI  (3.1415926535897932384626)
#endif


//----------------------------------------------------------------------------
//  Data Types
//----------------------------------------------------------------------------

typedef struct
{ double a[4];
  double mu;
} T_CubicInterpolator;

typedef struct tGpsDet  // detector for GPS-sync-pulses, and optional NMEA decoder
{
  // Parameters:
  double t_sample;          // time of one (audio-) sample [sec]; ASSUMED CONSTANT
  double t_sync_interval;   // almost always 1.0 [seconds per GPS-sync-pulse, AND NMEA-frame]
  double dblPulseWidth_sec; // EXPECTED width of the GPS "PPS" sync pulse in seconds, usually 0 = auto-detect
                            // (Trimble Thunderbolt E uses 10 microseconds - less than t_sample,
                            //  such strange timings must be ENTERED MANUALLY in the SL config)
  double t_sync_pulse_length;  // measured, typically 25..100 milliseconds (Garmin GPS 18x LVC)
  double t_NMEA_bit;   // serial bit time for the NMEA stream [sec]. ZERO means "don't decode NMEA".
                           // A standard NMEA output would use 4800 baud, i.e. 1/4800 sec .
                           // (But, with only 4800 bits/second, the NMEA data would overlap
                           //  the 1pps-signal, so use a LARGER bitrate, for example 19200 bit/sec)
  double dblUpdateCycle_sec;   // here: average cycle (to reduce short-term jitter)
  double dblMaxFreqOffset_ppm; // max frequency offset in ppm to either side (+/-, parts per million)
  int    NMEA_one_pulse_late;  // Added 2015-01-11, because some receivers were 'one second late', others weren't.
                               // If NMEA_one_pulse_late == 1, ONE SYNC INTERVAL (usually one second)
                               // is *ADDED* to the date-and-time decoded from the NMEA frame.
                               // In SL's "SRCalibCtrl.cpp", controlled via checkmark "GPS 1s late".
  BOOL   m_fAlternativeAlgorithm; // SL : UConfig.iSRCalib_AlternativeAlgorithm


  // Process variables:
  LONGLONG i64TSC; // 'Total Sample Counter' of all ADC samples entering GPS_Det_ProcessSamples().
           // Comparable with ("subtractable from") other i64TSC_..-variables in this struct.
           // The above sample counter is used as 'common timebase'
           // to convert timestamps for samples in various arrays (further below),
           // by using the 'Total Sample Counter'(TSC)-difference, divided by dblAveragedSampleRate.
  LONGLONG i64TSC_FirstEntryInSyncdWaveform;

  long double ldblTime_from_last_valid_NMEA; // time+date decoded from the last received NMEA sentence, converted into UNIX format,
                                             // decoded even without a valid sync pulse (!)
  LONGLONG    i64TSC_at_last_valid_NMEA;     // "Total Sample Counter at the last VALID NMEA sentence"
                                             // (comparable against other i64TSC-values;
                                             //  details in GPS_Det_ProcessSamples() )
  long double ldblTime_at_last_valid_sync_pulse; // Precise date and time ("as Unix second") at the last "good" sync pulse
                                                 // (per definition, an integer multiple of the sync pulse period;
                                                 //  BUT contains a 'fractional offset' of a few microseconds
                                                 //  as explained in GpsDet_InterpolateAndProcessSyncPulse() ).
  LONGLONG    i64TSC_at_last_valid_sync_pulse;   // "Total Sample Counter at the last VALID SYNC PULSE"

  // Internal and 'diagnostic' stuff:
  T_Float fltPpsPositiveThreshold, fltPpsNegativeThreshold;  // .. in the HIGH-PASS-FILTERED(!) input signal
  T_Float fltPpsPositivePeak,      fltPpsNegativePeak;       // normalized (-1 .. +1)
  T_Float fltPpsNewPositivePeak,   fltPpsNewNegativePeak;
  BOOL    peak_levels_valid;       // ...from the previous one-second-interval
  BOOL    pps_polarity_inverted;   // FALSE = normal polarity, TRUE = inverted (by soundcard)
  BOOL    samplerate_too_far_off;  // another internal error flag
  int     iCoarseDetState;         // 0: below threshold; before rising edge, 1: between rising edge and peak, 2: between peak and threshold on falling edge
  T_Float fltNewPeakMin, fltNewPeakMax; // similar, for the new detection interval
                                        // from the previous one-second-interval
  T_Float lp1_y_old, hp1_y_old, hp1_x_old;  // for 1st order low- and highpass
  int     iCountdownUntilFrameProcessing;
  double  dblPeakDetTimer, dblBadAmplTimer; // all timers are in SECONDS (!)
  double  dblUnixDateAndTimeFromChunkInfo;  // used in ProcessSamples(), but only to SHOW DEBUG MESSAGES
  double  dblSystemTimeMinusTimeFromGPS;    // MEASURED, should be well below 0.5 seconds
  double  dblGoodTimestampsSince;           // <- to suppress warnings shortly after start
  long double ldblUnixDateAndTimeForChunkInfo;
  long double ldblNextExpectedOutputTimestamp;
  long        nBadOutputTimestamps;
  int     nSamplesPerChunk;        // only for debugging (GpsDet_DumpInternalVariables)
  int     iSoftwareTestCommand;    // 0 for normal operation, or one of the following (will be cleared when "done").
  // Tests are "injected" from SRCalibCtrl.cpp, test results documented there.
# define GPSDET_TEST_NONE                      0
# define GPSDET_TEST_DONE                      1
# define GPSDET_TEST_SIMULATE_LOST_SAMPLE      2
# define GPSDET_TEST_SIMULATE_LOST_SYNC_PULSE  3
# define GPSDET_TEST_DUMP_INTERNAL_VARS        4

  // Suitable length of the filter kernel found by experimentation;
  //  visible improvement when stepping from 400 to 800
  //     (by observing the standard deviation of measured sample rate),
  //  and a slight improvement when stepping from 800 to 2000 :
  //  waveform remains the same when position of the GPS-pulse changes
  //  within one sample interval of the soundcard. Ok then.
  //  No problem because this 'slow' filter doesn't run continuously.
  //  It only runs ONCE every second, with
  //  C_GPSDET_INTERPOLATION_RATIO * C_GPSDET_FIFO_LEN_NETTO samples
  //  convolved with C_GPSDET_LOWPASS_KERNEL_LENGTH coefficients,
  //  requiring something like 4*16*2000 = 128000 MACs every second.
  #define C_GPSDET_LOWPASS_KERNEL_LENGTH  2000 /* must be an EVEN number ! */
  #define C_GPSDET_INTERPOLATION_RATIO    4 /* ex: 4; 8 also works but didn't give more accuracy for EDGE detection */
  #define C_GPSDET_FIFO_LEN_NETTO 16  // 'netto' samples to be processed,
                           // not depending on the filter kernel length .
  #define C_GPSDET_FIFO_LEN_BRUTTO (C_GPSDET_FIFO_LEN_NETTO+C_GPSDET_LOWPASS_KERNEL_LENGTH/C_GPSDET_INTERPOLATION_RATIO)
                           // -> circa ( 16+2000) / 4 = 504 samples
  T_Float fltInputPulse[C_GPSDET_FIFO_LEN_BRUTTO];
  long double ldblDateAndTimeAtInputPulse_ArrayIndexZero; // added 2015-01-31 for DebugPrintf()


  #define C_GPSDET_INTERPOLATION_BUFFER_LENGTH (C_GPSDET_FIFO_LEN_BRUTTO*C_GPSDET_INTERPOLATION_RATIO)
  T_Float fltInterpolationBuffer[C_GPSDET_INTERPOLATION_BUFFER_LENGTH];
          // The 'brutto' input for the interpolation (fltInterpolationBuffer)
          // must be MUCH LONGER than the interpolated result (fltInterpolatedStep)
          // because of the long windowed-sinc filter kernel.
          // Details about the contents of the interpolation buffer in *.cpp !
  #define C_GPSDET_INTERPOLATED_STEP_LENGTH (C_GPSDET_FIFO_LEN_NETTO/*16*/*C_GPSDET_INTERPOLATION_RATIO/*4*/)
  T_Float fltInterpolatedStep[C_GPSDET_INTERPOLATED_STEP_LENGTH];
          // This "interpolated step" only covers a few microseconds on both sides
          // of the relevant edge of the sync pulse. It can be viewed on the SR calib panel,
          // with the 'Scope' mode set to 'Interpolated PPS Signal'.
          // When running at 192 kSamples/sec, fltInterpolatedStep[] contains 16*4 = 64 samples
          // interpolated to fs = 4*192kHz = 768 kHz -> time span = -40 us .. +40 us.

  long double ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero;   // added 2015-01-31 for debugging

  T_Float fltLowpassKernel[C_GPSDET_LOWPASS_KERNEL_LENGTH+1];
  T_Float fltLowpassKernelCoeffMin, fltLowpassKernelCoeffMax;


  int  iDCdetFirstIndex, iDCdetLastIndex;
  double dblInterpolatedStepZeroCrossingIndex; // 0..1 * C_GPSDET_INTERPOLATED_STEP_LENGTH;
    // ideally 0.5*...C_GPSDET_INTERPOLATED_STEP_LENGTH = center of fltInterpolatedStep[].
    // To convert into SECONDS, multiply with (t_sample/C_GPSDET_INTERPOLATION_RATIO) .
  double dblPrevInterpolatedStepZeroCrossingIndex;
  T_CubicInterpolator CubicInterpolator;
  double dblCurrentMeasuredSampleRate; // number of audio samples (with fractional part)
                                       // between two rising edges of the GPS-pps-signal;
                                       // scaled for one-pulse-per-second GPSes .
                                       // May be ZERO if no valid PPS signal is present !
                                       // NOT smoothed, interpolated, or low-pass filtered -
                                       // THIS VALUE STEPS ONCE PER SECOND !
  double dblAveragedSampleRate; // Average of the last 10 (?) "good" measuring cycles .
                                // Always nonzero, no need to check before divide .
  double dblSampleRateDrift;    // [Hz per second], [1/sec^2]; used for prediction and to avoid step changes in the T_ChunkInfo
  double dblChunkInfo_SamplingRateDrift; // value used for the application (copied into T_ChunkInfo)
  BOOL   fExcessiveDrift;       // flag to enter 'excessive drift warning' only ONCE.


  #define GPSDEC_SR_HISTORY_BUF_LEN 60  // enough for one minute of data...
  double dblMeasuredSampleRateHistory[GPSDEC_SR_HISTORY_BUF_LEN]; // [0]=oldest, [59]=newest entry
  int    iMeasuredSampleRateHistoryLength;
  double dblStdDevInSampleRateHistory;  // .. within the last 60 minutes, IGNORING older history
  double dblMeanSampleRateFromHistory;  // .. within the last 60 minutes, IGNORING older history
  int    nAveragesWanted;  // copied from UConfig.iSRCalibAverages (through a long and winding path)
            // ("wanted" to emphasize that we are resticted to <GPSDEC_SR_HISTORY_BUF_LEN> averages)

  // For debugging / display on the 'sample rate calibrator control window' :
  #define GPSDEC_DEBUG_MESSAGE_BUF_LEN 64
  char sz127DebugMessageBuffer[GPSDEC_DEBUG_MESSAGE_BUF_LEN][128];
  int  iDebugMessageBufferHeadIndex, iDebugMessageBufferTailIndex; // simple circular FIFO
  long i32SyncPulseCounter, i32SyncErrorCounter;
  char sz80Info1[84], sz80Info2[84], sz80Info3[84], sz80TimeInfo[84];
  int  iUpdateCount;
  int  iScopeOption; // here: selects display of the interpolated step,
                     // or the filter's impulse response (aka "kernel"), etc :
#      define GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS 0
#      define GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE 1
#      define GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE      2
#      define GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM  3
#      define GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM   4
  BOOL fScopeDisplayPaused;
  char sz255StatusText[256];
  int  iStatusTextUpdateCount,  iStatusTextModifiedInLine;
  int  iCircuitState;  // CIRCUIT_STATE_PASSIVE, CIRCUIT_STATE_ACTIVE, CIRCUIT_STATE_WARNING, CIRCUIT_STATE_ERROR, ..
  int  iSourceLineOfLastDebugMessage;
  double dblPrevDebugTimestamp;

  // Sample buffer for the software UART / NMEA decoder, but also for the 'scope' display
  //  (contains an entire pulse train with slightly over 1 second of samples) :
  T_Float *pfltSyncdInputWaveform;    // dynamically allocated; contains ONE SECOND of samples [dwAllocdInputWaveformLength]
  long double ldblGpsTime_FirstEntryInSyncdWaveform; // added 2015-01-31 for DebugPrintf()
  T_Float *pfltProcessedInputForUART;
  T_Float *pfltSyncdInputWave4Plot;   // a copy of the above for the SCOPE display
  int     iSyncdInputWave4PlotLength; // number of samples currently in pfltSyncdInputWave4Plot[]
  int     iSyncdSampleIndex4Plot_PPS;

  // Internal stuff, used inside the blackbox "GpsDet_InterpolateAndProcessSyncPulse()":
  LONGLONG i64SyncdInputWaveformSampleIndex; // not related to .i64TSC ("TotalSampleCounter") !
  LONGLONG i64PreviousSampleIndexOfPPS;      //  "    "     "    "  !
  double  dblSyncPulseLength4Scope; // approx t_sync_pulse_length, but rarely changing
  int     iFirstNmeaSampleIndex4Scope, iLastNmeaSampleIndex4Scope;
  double  dblNmeaDataThrshold4Scope;
  T_Float Vpeak_nmea;
  DWORD   dwAllocdInputWaveformLength; // usable : pfltSyncdInputWaveform[0..dwAllocdInputWaveformLength-1]
  DWORD   dwSyncdWaveformSampleIndex;  // index into pfltSyncdInputWaveform[] to fill the array
  DWORD   dwUsedInputWaveformLength;
  T_Float fltSyncPulsePeak;            // used to detect the COARSE peak in 'short' (Dirac-like) pulses
  int     iSampleIndexOfPPS[3];  // [0] = rising edge, [1] = falling edge, [2] = 'important' edge for interpolation
  long double ldbDateAndTimeAtPPSEdges[3];  // for "NMEA-less" operation, same indices as above
  DWORD   dwMaxSamplesPerSyncInterval;
  int     iSerialDecoderErrors;
  T_GpsPosInfo *pGpsPosInfo; // type defined in C:\cbproj\gps_decoder\gps_dec.h


  // Members to fill out the output 'chunk info' (?\SoundUtl\ChunkInfo.h : T_ChunkInfo )
  //  in each call of GPS_Det_ProcessSamples() .
  //  All members in T_ChunkInfo for which there is no equivalent below
  //  are NOT modified in any call of GPS_Det_ProcessSamples() .
  double dblChunkInfo_Lat_deg;  // geographic latitude in degrees, positive = EAST, <-999 = INVALID
  double dblChunkInfo_Lon_deg;  // geographic longitude in degrees, positive = NORTH
  double dblChunkInfo_mASL;    // meters above sea level (from GPS receiver)
  double dblChunkInfo_Vel_kmh;  // GPS velocity in km/h
  // ex: double dblFractionalTimeOffset; // fractional time offset (less than one sample), from PPS interpolation, [seconds]
  BOOL   fChunkInfo_DateAndTimeFromGPSValid;

  DWORD dwMagic70710678; // 'magic number' to make this structure valid
        // (avoids trying to free memory if the struct is filled with garbage)
} T_GPS_Detector;   // used as component 'gps_det' in T_FREQ_CALIBRATOR (-> SOUND_SR_Calibrator)


extern BOOL SRCalibCtrl_fEmitDebugOutputNow; // implemented in SRCalibCtrl.cpp ..
extern BOOL SRCalibCtrl_fDebugOutputPaused;


//**************************************************************************
//    Functions (no shiny class methods)
//**************************************************************************

extern void  GpsPulseDet_DebugPrintf( double dblUnixDateAndTime, char *pszFormat, ... );
extern char* GpsPulseDet_GetDebugMessage( void ); // may be called from a different thread !
extern void  GpsPulseDet_ClearDebugMessages( void );


extern BOOL GpsGUI_AppendReceivedBytesToLogfile( BYTE *pbRxData, DWORD dwNumBytesRcvd );  // extern !

/***************************************************************************/
void   CubicIpol_Init( T_CubicInterpolator *pInterpolator,
                            double y0, double y1, double y2, double y3);
double CubicInterpolate( T_CubicInterpolator *pInterpolator, double mu);
double CubicIpol_FindZero( T_CubicInterpolator *pInterpolator );


/***************************************************************************/
void GPS_Det_Init( // GPS sync pulse detector / decoder..
        T_GPS_Detector *pGPSdet,      // [in] address of structure to be initialized
        double dblInitialSampleRate,  // [in] initial sample rate in Hz
        double dblPulsesPerSecond,    // [in] pulses per second, usually 1.0
        double dblPulseWidth_sec,     // [in] expected width of the GPS "PPS" sync pulse in seconds, 0 = auto-detect
        double dblUpdateCycle_sec,    // [in] update cycle / DFT length in seconds (default: 10)
        double dblMaxFreqOffset_ppm,  // [in] max frequency offset in ppm to either side (+/-, parts per million)
        int    iNMEA_bitrate,         // [in] NMEA bitrate (0=don't decode NMEA)
        T_GpsPosInfo *pGpsPosInfo);   // [in] address of the GPS/NMEA parser
  // Other, less important(?) parameters may be set AFTER GPS_Det_Init(),
  //        for example T_GPS_Detector.nAveragesWanted .


/***************************************************************************/
void GPS_Det_Delete(  // should be called on exit, to avoid memory leaks etc
        T_GPS_Detector *pGPSdet );

/***************************************************************************/
BOOL   GPS_Det_ProcessSamples( T_GPS_Detector *pGPSdet,T_Float *pAudioChunk,T_ChunkInfo *pChunkInfo);

/***************************************************************************/
// All the rest are more or less 'internal' functions, or for testing :
void   GPS_Det_InitLowpass(T_GPS_Detector *pGPSdet,double fc );
int    GPS_Det_GetNumSamplesForScope(T_GPS_Detector *pGPSdet );
double GPS_Det_ScopeSampleIndexToPhysicalUnit( T_GPS_Detector *pGPSdet, int iScopeSampleIndex, int *piScaleUnit );
void   GPS_Det_GetAmplRangeForScope( T_GPS_Detector *pGPSdet,
          double *pdblAmplMin, double *pdblAmplMax, int *piAmplUnit );
void GPS_Det_GetXAxisRangeForScope( T_GPS_Detector *pGPSdet,
          double *pdblPhysValueMin, double *pdblPhysValueMax, int *piPhysUnit );
double GPS_Det_GetSampleForScope( T_GPS_Detector *pGPSdet, int iScopeSampleIndex);
double GPS_Det_PhysicalValueToScopeSampleIndex( T_GPS_Detector *pGPSdet,double dblHorzPhysicalValue);
double GPS_Det_GetSampleForScope( T_GPS_Detector *pGPSdet, int iScopeSampleIndex);


#endif // _GpsPulseDetector_H_