//---------------------------------------------------------------------------
// File  :  C:\cbproj\SoundUtl\GpsPulseDetector.cpp
// Date  :  2016-10-30 (ISO 8601, YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//    Implementation of the 'GPS pulse detector' for Spectrum Lab .
//    Used to continuously 'calibrate' the soundcard's sample rate
//    with the 1-PPS (pulse-per-second) output from a suitable GPS receiver .
//
//    Provides high-resolution timestamps, using
//    the GPS'es NMEA output added to the PPS signal with a passive combiner
//    ( see  http://www.qsl.net/dl4yhf/speclab/frqcalib.htm#gps_pps_output )
//
//    Once a part of FreqCalibrator.cpp, and still invoked from there .
//
//  Copyright (c) 2011 .. 2016, Wolfang Buescher (DL4YHF) .
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//
//  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ''AS IS'' AND ANY EXPRESS OR
//  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
//  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Revision history (YYYY-MM-DD):
//   2016-10-29  Added the pulse-center detection (centroid calculation)
//               for extremely short GPS sync pulses. For example, a Trimble
//               Thunderbolt E sends 10 us (!) long pulses, which are often
//               SHORTER than a single sample for the soundcard.
//               Most of that is in GPS_Det_ProcessInterpolatedShortPulse().
//
//   2014-12-20  Tried to make this module suitable for a certain Motorola
//               Oncore GPS receiver, where the NMEA data were quite close
//               to the 1pps sync signal (both sent over the "same wire").
//               Details: C:/postarchiv/df6nm/2014_12_20_SpecLab_NMEA_timing.htm
//
//   2012-02-10  Added the samplerate drift calculation
//               ( -> T_ChunkInfo.dblSamplingRateDrift )
//   2011-04-06  Separated the GPS-Pulse-Detector from the 'Frequency Calibrator'
//               because the module grew too large (> 2500 lines of sourcecode).
//   2011-03-24  Added support for GPS receivers with 1-pps-output .
//
//---------------------------------------------------------------------------


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h>   // Do we still need this junk (DWORD, BYTE, WORD) ?
#include <math.h>
#include <stdio.h>     // no "standard I/O" used here, just sprintf

#pragma hdrstop   // BORLAND stuff: no precompiled headers after this point

#include "SoundTab.h"  // some required tables, filter coeffs, etc.
#include "utility1.h"  // uses UTL_NamedMalloc instead of malloc (for debugging), and to *GUARANTEE* 8-byte alignment

#include "GpsPulseDetector.h"  // header for THIS module (GPS pulse detector, for the frequency calibrator)



//**************************************************************************
//    Global Variables
//**************************************************************************

extern double TIM_dblPerformanceCounterToTimeOffset; // Unix seconds when TIM_QueryHRTimer_sec() was 'zero' 


// Debugging stuff ...
int  GpsPulseDet_iSourceCodeLine = __LINE__;  // last source code line (WATCH this!)
int  GpsPulseDet_iDebugParam1 = 0;
#define DOBINI(iDebugParam1) GpsPulseDet_iSourceCodeLine=__LINE__,GpsPulseDet_iDebugParam1=iDebugParam1

BYTE GpsPulseDet_bNumErrorsShownInLine[4000];



//**************************************************************************
//    Internal forward references
//**************************************************************************
static double GpsDet_SampleCountertoTimeDiff( T_GPS_Detector *pGPSdet, LONGLONG i64TSC_diff );
static void GpsDet_CopySamplesForPlotting( T_GPS_Detector *pGPSdet );
static void GpsDet_RemoveOldEntriesInSyncdInputWaveform( T_GPS_Detector *  pGPSdet );
static void GpsDet_UpdateSyncPulseThresholds( T_GPS_Detector *pGPSdet, float fltCurrSample, double tPeakDet );
static BOOL GpsDet_InterpolateAndProcessSyncPulse( T_GPS_Detector * pGPSdet );
static BOOL GpsDet_DecodeSerialData( T_GPS_Detector *pGPSdet,
                               T_Float *pFltSource,
                               double dblHRTimerValueAtSyncPulse_sec,
                               int iFirstNmeaSampleIndex,
                               int iLastNmeaSampleIndex );


//**************************************************************************
//    Functions (no class methods)
//**************************************************************************

// Lock-free buffer, filled by GpsPulseDet_DebugPrintf(), drained somewhere else:
#define GPS_PULSE_DET_DEBUG_FIFO_LEN 256 /* must be a power of two ! */
static char GpsPulseDet_sz80DebugMsgFifo[GPS_PULSE_DET_DEBUG_FIFO_LEN][88];
static int  GpsPulseDet_iDebugFifoHead = 0, GpsPulseDet_iDebugFifoTail = 0;

void GpsPulseDet_DebugPrintf( double dblUnixDateAndTime, char *pszFormat, ... )
{
  va_list arglist;
  char *cp;

  // To make this reasonably thread-safe, don't pass the "printed output"
  //         directly to a fancy Borland VCL / Windows gimmick.
  // Instead, 'sprintf' the message to a kind of FIFO
  // from which a display thread (for example in SL : SRCalibCtrl.cpp) reads it.
  cp = GpsPulseDet_sz80DebugMsgFifo[ GpsPulseDet_iDebugFifoHead & (GPS_PULSE_DET_DEBUG_FIFO_LEN-1) ];
  // As long as "printing" isn't finished, don't increment the FIFO head index !

  // Format the message, then let someone else append it to a text window...
  va_start(arglist, pszFormat);

  // Use the same date-and-time format in GpsPulseDetector.cpp and TimedResampler.c !
  if( dblUnixDateAndTime > 0.0 )
   { UTL_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss.ss ", dblUnixDateAndTime, cp );
     cp += strlen(cp);
   }
  vsprintf(cp, pszFormat, arglist);
  va_end(arglist);

  GpsPulseDet_iDebugFifoHead = ( GpsPulseDet_iDebugFifoHead + 1) & (GPS_PULSE_DET_DEBUG_FIFO_LEN-1);

} // end GpsPulseDet_DebugPrintf()

char* GpsPulseDet_GetDebugMessage( void ) // may be called from a different thread !
  // Returns NULL if no new 'debug message' has been printed into the FIFO,
  //         otherwise retrieves the next string from that buffer (no-copy).
{
  char *cp = NULL;
  if( GpsPulseDet_iDebugFifoHead != GpsPulseDet_iDebugFifoTail )
   { cp = GpsPulseDet_sz80DebugMsgFifo[ GpsPulseDet_iDebugFifoTail & (GPS_PULSE_DET_DEBUG_FIFO_LEN-1) ];
     GpsPulseDet_iDebugFifoTail = (GpsPulseDet_iDebugFifoTail+1) & (GPS_PULSE_DET_DEBUG_FIFO_LEN-1);
   }
  return cp;
} // end GpsPulseDet_GetDebugMessage()

//---------------------------------------------------------------------------
void  GpsPulseDet_ClearDebugMessages( void )
{
  // Called from same thread as GpsPulseDet->GetDebugMessage(), so don't modify GpsPulseDet_iDebugFifoHead here:
  GpsPulseDet_iDebugFifoTail = GpsPulseDet_iDebugFifoHead;

  // To allow all debug messages again, clear their individual counters:
  memset( GpsPulseDet_bNumErrorsShownInLine, 0, sizeof(GpsPulseDet_bNumErrorsShownInLine) );
} // end GpsPulseDet_ClearDebugMessages()

//---------------------------------------------------------------------------
void GpsDet_DumpInternalVariables( T_GPS_Detector *pGPSdet )
{
  double   d,t = pGPSdet->ldblNextExpectedOutputTimestamp;
  LONGLONG i64TSC = pGPSdet->i64TSC;
  char sz60Temp[64];

  GpsPulseDet_DebugPrintf( 0, "" );
  GpsPulseDet_DebugPrintf( 0, "---- GPS Pulse Detector state ----" );
  UTL_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss.sss", t + 0.1e-3, sz60Temp );
  GpsPulseDet_DebugPrintf( 0, " Timestamp = %s", sz60Temp );
  GpsPulseDet_DebugPrintf( 0, " Averaged S.rate = %.3lf Hz",(double)pGPSdet->dblAveragedSampleRate);
  GpsPulseDet_DebugPrintf( 0, " Samplerate drift  = %.6lf Hz/s",(double)pGPSdet->dblSampleRateDrift);
  GpsPulseDet_DebugPrintf( 0, " Processing chunk  = %d samples", (int)pGPSdet->nSamplesPerChunk );
  if( pGPSdet->nSamplesPerChunk > 0 )
   { d = (double)i64TSC / (double)pGPSdet->nSamplesPerChunk;
   }
  else
   { d = 0.0;
   }
  GpsPulseDet_DebugPrintf( 0, " Total samples processed = %.3lf chunks",(double)d );
  if( pGPSdet->ldblTime_at_last_valid_sync_pulse > 0.0 )
   { UTL_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss.sss", pGPSdet->ldblTime_at_last_valid_sync_pulse+0.1e-3, sz60Temp );
     GpsPulseDet_DebugPrintf( 0, " Time at last valid sync pulse = %s", sz60Temp );
   }
  else
   { GpsPulseDet_DebugPrintf( 0, " No timestamped 'sync pulses' yet !" );
   }
  if( pGPSdet->ldblTime_from_last_valid_NMEA > 0.0 )
   { UTL_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss.sss", pGPSdet->ldblTime_from_last_valid_NMEA+0.1e-3, sz60Temp );
     GpsPulseDet_DebugPrintf( 0, " Time from last valid NMEA = %s", sz60Temp );
   }
  else
   { GpsPulseDet_DebugPrintf( 0, " No time decoded from NMEA yet !" );
   }
  GpsPulseDet_DebugPrintf( 0, " age of last valid sync pulse= %.3lf s",(double)GpsDet_SampleCountertoTimeDiff( pGPSdet, i64TSC-pGPSdet->i64TSC_at_last_valid_sync_pulse) );
  GpsPulseDet_DebugPrintf( 0, " age of synch'd waveform= %.3lf s",(double)GpsDet_SampleCountertoTimeDiff( pGPSdet, i64TSC-pGPSdet->i64TSC_FirstEntryInSyncdWaveform) );
  GpsPulseDet_DebugPrintf( 0, " age of last NMEA burst = %.3lf s", (double)GpsDet_SampleCountertoTimeDiff( pGPSdet, i64TSC-pGPSdet->i64TSC_at_last_valid_NMEA ) );
  GpsPulseDet_DebugPrintf( 0, " Sync pulse peaks: %.3f .. %.3f", (float)pGPSdet->fltPpsNegativePeak,      (float)pGPSdet->fltPpsPositivePeak );
  GpsPulseDet_DebugPrintf( 0, " Pulse thresholds: %.3f .. %.3f", (float)pGPSdet->fltPpsNegativeThreshold, (float)pGPSdet->fltPpsPositiveThreshold );
  GpsPulseDet_DebugPrintf( 0, " Polarity inverted = %d", (int)pGPSdet->pps_polarity_inverted );
  if( pGPSdet->dblGoodTimestampsSince > 0.0 )
   { GpsPulseDet_DebugPrintf( 0, " Good timestamps since >= %.1lf s",(double)(t-pGPSdet->dblGoodTimestampsSince) );
   }
  else
   { GpsPulseDet_DebugPrintf( 0, " No 'good' timestamps yet !" );
   }
  GpsPulseDet_DebugPrintf( 0, " Bad timestamp counter    = %ld",(long)pGPSdet->nBadOutputTimestamps );
  GpsPulseDet_DebugPrintf( 0, " PPS edge times (delta t) = %.3lf %.3lf %.3lf s",
                    (double)(t-pGPSdet->ldbDateAndTimeAtPPSEdges[0]),
                    (double)(t-pGPSdet->ldbDateAndTimeAtPPSEdges[1]),
                    (double)(t-pGPSdet->ldbDateAndTimeAtPPSEdges[2]) );
  UTL_FloatToTechNotation( sz60Temp, 20/*iMaxLen*/,
                    4/*digits*/ + UTL_FMT_OPTION_SPACE_BEFORE_UNIT,
                    SCALE_UNIT_SECONDS,
                    pGPSdet->dblStdDevInSampleRateHistory * pGPSdet->t_sample );
  GpsPulseDet_DebugPrintf( 0, " Standard deviation in %2d pulses = %s/s",
                      (int)pGPSdet->iMeasuredSampleRateHistoryLength, sz60Temp);
     // Some examples for comparison:
     // (a)  E-MU 0202, Garmin GPS18 LVC, fs=48 kHz, 100 ms pulses / edge detection
     // (b)  E-MU 0202, Garmin GPS18 LVC, fs=192 kHz, 100 ms pulses / edge detection
     //
     // Configuration | StdDev in 60 pulses |
     // --------------+---------------------+----
     //     (a)       |  190 .. 220 ns/s    |
     //
     //
  SRCalibCtrl_fEmitDebugOutputNow = TRUE; // emit the above even when message-output is paused

} // end GpsDet_DumpInternalVariables()

//---------------------------------------------------------------------------
BOOL GpsDet_DebugCheckTimestamp( T_GPS_Detector *pGPSdet, double dblUnixDateAndTime )
{ // Compares timestamp between current and previous call. Only for debugging.
  BOOL fOk = TRUE;
  double dt;
  if( pGPSdet->dblPrevDebugTimestamp > 0 )
   {
     dt = dblUnixDateAndTime - pGPSdet->dblPrevDebugTimestamp;
     if( dt<0 || dt>1.0 )
      { GpsPulseDet_DebugPrintf( dblUnixDateAndTime,
          "PulseDet: TS discontinuity, dt=%.3lf", dt );
        fOk = FALSE;
      }
   }
  pGPSdet->dblPrevDebugTimestamp = dblUnixDateAndTime;

  return fOk;
} // end GpsDet_DebugCheckTimestamp()


/***************************************************************************/
void GPS_Det_Init(
        T_GPS_Detector *pGPSdet,      // [in] address of structure to be initialized
        double dblInitialSampleRate,  // [in] initial sample rate in Hz (required for allocation)
        double dblPulsesPerSecond,    // [in] pulses per second, usually 1.0
        double dblPulseWidth_sec,     // [in] expected width of the GPS "PPS" sync pulse in seconds, 0 = auto-detect
        double dblUpdateCycle_sec,    // [in] here: average cycle (to reduce short-term jitter)
        double dblMaxFreqOffset_ppm,  // [in] max frequency offset in ppm to either side (+/-, parts per million)
        int    iNMEA_bitrate,         // [in] NMEA bitrate (0=don't decode NMEA, only use the "PPS" sync pulse)
        T_GpsPosInfo *pGpsPosInfo)    // [in] address of the GPS/NMEA parser
{
   DOBINI( (int)dblInitialSampleRate );
   if( pGPSdet->dwMagic70710678 == 0x70710678)
    { GPS_Det_Delete(pGPSdet);
    }
   memset( GpsPulseDet_bNumErrorsShownInLine, 0, sizeof(GpsPulseDet_bNumErrorsShownInLine) );
   memset( pGPSdet, 0, sizeof(T_GPS_Detector) );
   pGPSdet->iDebugMessageBufferHeadIndex = pGPSdet->iDebugMessageBufferTailIndex = 0;
   pGPSdet->ldblNextExpectedOutputTimestamp = -1;
   pGPSdet->nBadOutputTimestamps    = -1;   // negative: "don't count BAD timestamps yet" !
   pGPSdet->dblGoodTimestampsSince  = 0.0;  // "not running with GOOD timestamps yet"
   pGPSdet->fltPpsNegativeThreshold = pGPSdet->fltPpsPositiveThreshold
    = pGPSdet->fltNewPeakMin = pGPSdet->fltNewPeakMax = 0.0;
   pGPSdet->fltPpsPositivePeak = pGPSdet->fltPpsNewPositivePeak = 0.0;
   pGPSdet->fltPpsNegativePeak = pGPSdet->fltPpsNewNegativePeak = 0.0;
   pGPSdet->dblSyncPulseLength4Scope = pGPSdet->dblPeakDetTimer = 0.0;
   pGPSdet->lp1_y_old = pGPSdet->hp1_y_old = pGPSdet->hp1_x_old = 0.0;
   pGPSdet->peak_levels_valid = FALSE; // not ready to "look for peaks" yet
   pGPSdet->iSampleIndexOfPPS[0] = pGPSdet->iSampleIndexOfPPS[1] = pGPSdet->iSampleIndexOfPPS[2] = -1;
   pGPSdet->fltSyncPulsePeak = 0.0;
   pGPSdet->ldbDateAndTimeAtPPSEdges[0] = pGPSdet->ldbDateAndTimeAtPPSEdges[1] = 0.0;
   if( dblInitialSampleRate <= 0.0 )
    {  dblInitialSampleRate = 1.0;  // must NEVER be zero (to avoid div-by-zero) !
    }
   pGPSdet->dblAveragedSampleRate = dblInitialSampleRate; // < must NEVER be zero !    
   pGPSdet->t_sample = 1.0 / dblInitialSampleRate;
   if( dblPulsesPerSecond > 0.0 )
        pGPSdet->t_sync_interval = 1.0 / dblPulsesPerSecond;
   else pGPSdet->t_sync_interval = 0.0;
   pGPSdet->dblPulseWidth_sec    = dblPulseWidth_sec;
   pGPSdet->dblUpdateCycle_sec   = dblUpdateCycle_sec;
   pGPSdet->dblMaxFreqOffset_ppm = dblMaxFreqOffset_ppm;
   if( iNMEA_bitrate>0 )
        pGPSdet->t_NMEA_bit = 1.0 / (double)iNMEA_bitrate;
   else pGPSdet->t_NMEA_bit = 0.0;  // don't try to decode the NMEA stream
   GPS_Det_InitLowpass(pGPSdet,
      0.5/(double)C_GPSDET_INTERPOLATION_RATIO ); // [in] normalized cutoff frequency, 0 ... 0.5
      // (cutoff frequency at half the ORIGINAL sampling rate)
   pGPSdet->dblInterpolatedStepZeroCrossingIndex = 0.5 * C_GPSDET_INTERPOLATED_STEP_LENGTH;
   pGPSdet->dblPrevInterpolatedStepZeroCrossingIndex = 0.0;
   pGPSdet->iCountdownUntilFrameProcessing = (int)(dblInitialSampleRate*1.1);
   pGPSdet->dblCurrentMeasuredSampleRate = 0.0;
   pGPSdet->iMeasuredSampleRateHistoryLength = 0;
   pGPSdet->dblStdDevInSampleRateHistory = 0.0;
   pGPSdet->dblMeanSampleRateFromHistory = 0.0;

   // For the software UART / NMEA decoder, but also for the 'scope' display (entire pulse train) :
   // Make the array 10% longer than the one-second-cycle !
   pGPSdet->dwAllocdInputWaveformLength = (DWORD)(dblInitialSampleRate*1.1);
   pGPSdet->pfltSyncdInputWaveform = (T_Float*)UTL_NamedMalloc( "GPSdec",
                 pGPSdet->dwAllocdInputWaveformLength * sizeof( T_Float ) );
   pGPSdet->pfltProcessedInputForUART = (T_Float*)UTL_NamedMalloc("GPSrxd",
                 pGPSdet->dwAllocdInputWaveformLength * sizeof( T_Float ) );
   pGPSdet->pfltSyncdInputWave4Plot= (T_Float*)UTL_NamedMalloc( "GPSplt",
                 pGPSdet->dwAllocdInputWaveformLength * sizeof( T_Float ) );
   pGPSdet->dwSyncdWaveformSampleIndex = 0;  // index into pfltSyncdInputWaveform[] to fill the array
   pGPSdet->i64SyncdInputWaveformSampleIndex = 0; // what a name ;o)
   pGPSdet->i64PreviousSampleIndexOfPPS = 0;
   pGPSdet->dwUsedInputWaveformLength = 0;
   pGPSdet->iSyncdInputWave4PlotLength= 0;
   pGPSdet->pGpsPosInfo = pGpsPosInfo;  // address of a T_GpsPosInfo, see gps_dec.h (NMEA parser)

   pGPSdet->dblChunkInfo_Lat_deg = C_CHUNK_INFO_LAT_INVALID;
   pGPSdet->dblChunkInfo_Lon_deg = C_CHUNK_INFO_LON_INVALID;
   pGPSdet->dblChunkInfo_mASL   = 0.0;
   pGPSdet->dblChunkInfo_Vel_kmh = 0.0;
   pGPSdet->ldblTime_from_last_valid_NMEA      = 0.0;  // always decoded, even without valid sync pulse
   pGPSdet->ldblTime_at_last_valid_sync_pulse = 0.0;  // beware the difference to the 'current date and time' !
   pGPSdet->i64TSC_at_last_valid_sync_pulse = 0; // "Total Sample Counter at the last VALID SYNC PULSE"
   pGPSdet->dblChunkInfo_SamplingRateDrift  = 0.0;
   pGPSdet->fExcessiveDrift = TRUE; // << suppress "excessive drift" warning message right after start

   pGPSdet->dwMagic70710678 = 0x70710678;
} // end GPS_Det_Init()


/***************************************************************************/
void GPS_Det_Delete(  // should be called on exit, to avoid memory leaks etc
        T_GPS_Detector *pGPSdet )
{
  DOBINI(0);
  if( pGPSdet->dwMagic70710678 == 0x70710678 )
   { // Has this instance been properly initialized ?
     //  (if not, DON'T TRY TO FREE THE DYNAMICALLY ALLOCATED ARRAYS
     //   because they will most likely just be 'pointers to garbage' ! )
     if( pGPSdet->dwAllocdInputWaveformLength > 0 )
      {
        pGPSdet->dwAllocdInputWaveformLength = 0;
        Sleep(10);  // kludge against multi-threading issues ("array still being used" ?)
        UTL_free( pGPSdet->pfltSyncdInputWaveform );
        pGPSdet->pfltSyncdInputWaveform = NULL;
        UTL_free( pGPSdet->pfltProcessedInputForUART );
        pGPSdet->pfltProcessedInputForUART = NULL;
        UTL_free( pGPSdet->pfltSyncdInputWave4Plot);
        pGPSdet->pfltSyncdInputWave4Plot= NULL;
      }

     pGPSdet->dwMagic70710678 = 0;
   } // end if < pGPSdet->dwMagic70710678 .. >
  DOBINI(0);
} // end GPS_Det_Delete()  [ kind of 'destructor' if this was a C++ class ]

/***************************************************************************/
double GpsDet_SampleCountertoTimeDiff( T_GPS_Detector *pGPSdet, LONGLONG i64TSC_diff )
  // Converts an (ADC-) Sample Counter difference into a time in seconds,
  // using the 'best available' (measured, averaged) sampling rate.
{
  double d = 0.0;
  if( pGPSdet->dblAveragedSampleRate > 0 )
   { d = (double)i64TSC_diff / pGPSdet->dblAveragedSampleRate;
   }
  return d;
} // end GpsDet_SampleCountertoTimeDiff()

/***************************************************************************/
double GPS_Det_GetTimeDiff( T_GPS_Detector *pGPSdet, double t1, double t2 )
  // Calculates a time difference, which must be LESS THAN HALF the pps-interval
  //  (otherwise deal with the wrap-around of the sync interval)
{
  double tdiff = t2 - t1;    // -> time difference, and ideally less than 0.5 seconds
  if( tdiff < 0.0 )  // timer must have wrapped between t1 and t2
   { // example: t1=0.9, t2=0.1, pGPSdet->t_sync_interval=1.0 -> tdiff must be 0.2 sec (not -0.8)
     tdiff += pGPSdet->t_sync_interval;
   }
  return tdiff;
} // end GPS_Det_GetTimeDiff()

/***************************************************************************/
void GPS_Det_InitLowpass(  // Initializes the windowed-sinc lowpass, used for interpolation
        T_GPS_Detector *pGPSdet, // [in] address of structure to be initialized
        double fc )              // [in] normalized cutoff frequency, 0 ... 0.5
{
  int i,j; float f,sum;

  DOBINI( (int)fc );
  // Initialize the filter kernel for the interpolating windowed-sinc lowpass.
  // Note that C_GPSDET_LOWPASS_KERNEL_LENGTH is an even number (usually 200),
  // but the coefficient array indices fltLowpassKernel[i] run from 0..200 !!
  for(i=0; i<=/*!!*/C_GPSDET_LOWPASS_KERNEL_LENGTH/*200?*/; ++i )
   { j = i-(C_GPSDET_LOWPASS_KERNEL_LENGTH/2);
     if( j==0 ) f = 2.0 * C_PI * fc;
     else       f = sin( 2.0 * C_PI * fc * (double)j ) / (double)j;
     // Multiply the sinc coeff with a Blackman window:
     f *= (0.42 - 0.5*cos(2.0*C_PI*(double)i/(double)C_GPSDET_LOWPASS_KERNEL_LENGTH)
               + 0.08*cos(4.0*C_PI*(double)i/(double)C_GPSDET_LOWPASS_KERNEL_LENGTH));
     pGPSdet->fltLowpassKernel[i] = f;
   }
  // Normalize the low-pass filter kernel for unity gain at DC:
  for(sum=0.0,i=0; i<=C_GPSDET_LOWPASS_KERNEL_LENGTH; ++i )
   { sum += pGPSdet->fltLowpassKernel[i];
   }
  pGPSdet->fltLowpassKernelCoeffMin = 0.0;
  pGPSdet->fltLowpassKernelCoeffMax = 0.0;
  for(i=0; i<=C_GPSDET_LOWPASS_KERNEL_LENGTH; ++i )
   { f = pGPSdet->fltLowpassKernel[i] / sum;
     pGPSdet->fltLowpassKernel[i] = f;
     if( f <  pGPSdet->fltLowpassKernelCoeffMin )
      {  pGPSdet->fltLowpassKernelCoeffMin = f;  // to show the filter's IMPULSE RESPONSE (aka "kernel") on the scope
      }
     if( f >  pGPSdet->fltLowpassKernelCoeffMax )
      {  pGPSdet->fltLowpassKernelCoeffMax = f;
      }
   }
  DOBINI( (int)fc );
} // end GPS_Det_InitLowpass()

/***************************************************************************/
static void GpsDet_InterpolateSyncPulseEdge( T_GPS_Detector *pGPSdet )
  // Calculates the interpolated leading edge of the sync pulse,
  //            or an interpolated 'Dirac-like' waveform for very short pulses,
  //  using a lowpass with fcutoff/fsample = 1/8
  //
  // Based on "The Scientist and Engineer's Guide to Digital Signal Processing"
  //          by Steven W. Smith;  chapter 16, Windowed-Sinc Filters.
  //  [in]  pGPSdet->fltInterpolationBuffer[0...C_GPSDET_INTERPOLATION_BUFFER_LENGTH_BRUTTO-1]
  //          contains *ZERO STUFFED* samples (see caller, GPS_Det_ProcessSamples() ).
  //        pGPSdet->ldblDateAndTimeAtInputPulse_ArrayIndexZero :
  //          approximate timestamp of the first sample in the above buffer
  //          (the precise timestamp will be known after finding the zero-crossing)
  // [out] pGPSdet->fltInterpolatedStep[0..C_GPSDET_INTERPOLATED_STEP_LENGTH-1],
  //        pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero .
  //
  // This function is called when we have enough samples
  //      on BOTH SIDES of the 'interesting center' (rising edge of GPS sync pulse),
  //      long enough to be convolved with the filter kernel.
  //      ONLY THOSE (sixteen times four?) SAMPLES AROUND THE RISING EDGE are actually
  //      calculated, but the rest
  //       (C_GPSDET_LOWPASS_KERNEL_LENGTH/2 samples to the left and the right)
  //      is also necessary, because with only 16 input samples
  //      it would be impossible to convolve with a 201-tap filter.
{
  T_Float y;
  int iSourceSample, iCoeff, iDestSample;

  for( iDestSample=0;
       iDestSample<C_GPSDET_INTERPOLATED_STEP_LENGTH;
     ++iDestSample )
   {
     y = 0.0;
     iSourceSample = iDestSample;
     for( iCoeff=0/*!*/; iCoeff<=/*!!!*/C_GPSDET_LOWPASS_KERNEL_LENGTH; ++iCoeff )
      { y +=  pGPSdet->fltInterpolationBuffer[iSourceSample++]
            * pGPSdet->fltLowpassKernel[iCoeff];
      }
     pGPSdet->fltInterpolatedStep[iDestSample]= y;
     if( pGPSdet->dwAllocdInputWaveformLength == 0 )
      { // kludge for "fast termination" (see GPS_Det_Delete)
        DOBINI(iDestSample);
        return;
      }
   } // end for < all samples >

  pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero =
    pGPSdet->ldblDateAndTimeAtInputPulse_ArrayIndexZero;
  // When running at 192 kSamples/sec, fltInterpolatedStep[] contains 16*4 = 64 samples
  // interpolated to fs = 4*192kHz = 768 kHz -> time span = -40 us .. +40 us.
  // Thus, the first sample (at fltInterpolatedStep[0]) timestamp should be
  // approximately(!) 40 microseconds "before a full second".
  // 2015-01-31 : Got here with iDestSample=64 and
  //    pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero = 1422738831.99866,
  //             i.e. 1.34 ms "before the second" ! Bugged ?
  // Check THIS : will the above low-pass-filtered interpolation
  //        create an additional delay, which would have to be
  //        considered in the above copying of the timestamp ?

} // end GpsDet_InterpolateSyncPulseEdge()

//---------------------------------------------------------------------------
int GpsDet_FindZeroCrossing( T_Float *pfltData, // [in] source array (1D)
     int from_index,            // [in] begin of the searching range
     int to_index,              // [in] end of the searching range
     double *pdblZeroXingIndex) // [out,optional] precise zero-crossing index
{ double y1 = pfltData[from_index];
  double y2;
  double zi;
  int  i;

  for( i = from_index+1; i<= to_index; i++ )
   { y2 = pfltData[i];
     if( ( (y1<0.0) && (y2>=0.0) )  ||  ( (y1>0.0) && (y2<=0.0) ) )
      { //
        // Figure 1 :
        // ----------
        //  Linear interpolation to find the array index of the zero-crossing,
        //         used for 'long' pulses, focused on the LEADING PULSE EDGE.
        //
        //                     __
        //          y2......../  \__
        //                   /|
        //           0 ...../ |
        //                 /| |
        //          y1..../ | |
        //            \__/| | |
        //                | | |
        //              i-1 | i
        //                  |
        //                 zi = i - (y2 / (y2-y1) )
        //
        zi = (double)i - (y2 / (y2-y1) );
        if( pdblZeroXingIndex != NULL )
         { *pdblZeroXingIndex = zi;
         }
        return (int)(zi+0.5); // ex: return i; but we want the integer index CLOSEST to the 'zero' !
      }
     y1 = y2;
   }
  return -1;  // no zero-crossing between from_index and to_index !
} // end GpsDet_FindZeroCrossing()

/***************************************************************************/
void CubicIpol_Init( T_CubicInterpolator *pInterpolator,
                     double y0, double y1, double y2, double y3 )
  // Initialises a cubic interpolator. Used for the peak-detection.
{

#if(0)  // Poor algorithm; turns a perfect straight line (y0=0 y1=1 y2=2 y3=3)
        // into an oscillating mess !
  // Found at http://paulbourke.net/miscellaneous/interpolation/ : CubicInterpolate() .
  pInterpolator->a[0] = (y3 - y2) - (y0 - y1);
  pInterpolator->a[1] = (y0 - y1) - pInterpolator->a[0];
  pInterpolator->a[2] = y2 - y0;
  pInterpolator->a[3] = y1;
     // With the coeffs a[0] .. a[3], the interpolated function is :
     //    y(mu) = a[0]*mu^3 + a[1]*mu^2 + a[2]*mu + a[3];
     //    with mu=0.0 : y=y1;  ...  mu=1.0 : y=y2 .
#else  // Tried this, from http://www.ureader.de/msg/145617834.aspx ,
       //    suggested by 'Olli' (beware of the indices y1..y4 below) :
       //   f(t) = 1/2(-y1+3*y2-3*y3+y4)t^3
       //        + 1/2(2*y1-5*y2+4*y3-y4)t^2
       //        + 1/2(-y1+y3)t
       //        + y2
       //  These polynomial coefficients gave a perfect straight line
       //        with y0=0, y1=1, y2=2, y3=3 .
       //  On a second glance, this is identical to the algorithm
       //  proposed by Paul Breeuwsma :
  pInterpolator->a[0] = 0.5 * (    -y0 + 3.0*y1 - 3.0*y2 + y3 );
  pInterpolator->a[1] = 0.5 * ( 2.0*y0 - 5.0*y1 + 4.0*y2 - y3 );
  pInterpolator->a[2] = 0.5 * (    -y0 + y2 );
  pInterpolator->a[3] = y1;
#endif
} // end CubicIpol_Init()

/***************************************************************************/
double CubicInterpolate( T_CubicInterpolator *pInterpolator, double mu)
  // Interpolates a value between y1 (mu=0) and y2 (mu=1) .
  //       y(mu) = a[0]*mu^3 + a[1]*mu^2 + a[2]*mu + a[3];
{
  double mu2 = mu*mu;
  return  pInterpolator->a[0]*mu2*mu
         +pInterpolator->a[1]*mu2
         +pInterpolator->a[2]*mu
         +pInterpolator->a[3];
} // end CubicInterpolate()


/***************************************************************************/
double CubicIpol_FindZero( T_CubicInterpolator *pInterpolator )
  // Finds the zero in the specified polynomial (from cubic interpolation) .
  // For details, see  http://de.wikipedia.org/wiki/Regula_Falsi [2011-03-26].
  // Return value:  mu  for which y(mu) == 0 .
{
  // > Das Regula-falsi-Verfahren startet mit zwei Stellen (in der Nhe
  // > der Nullstelle) a0 und b0, deren Funktionsauswertungen f(a0), f(b0)
  // > unterschiedliche Vorzeichen haben. In dem Intervall [a,b] befindet sich
  // > somit nach dem Zwischenwertsatz (fr stetiges f) eine Nullstelle.
  // > Nun verkleinert man in mehreren Iterationsschritten das Intervall
  // > und bekommt so eine immer genauere Nherung fr die Nullstelle.
  double a,ya,b,yb,c,yc,denom;
  int nLoops;
  // Start: a=a0=left edge of the interval,  b=b0=right edge of the interval .
  //   Actually, there's no "x" but "mu" here, but the principle remains the same.
  a = 0.0;
  ya= CubicInterpolate( pInterpolator, a );   // f(a0)
  b = 1.0;
  yb= CubicInterpolate( pInterpolator, b );   // f(b0)
  c = 0.5*(a+b); // just in case iteration is impossible
  for(nLoops=0;nLoops<50;++nLoops)
   { denom = yb - ya;
     if( fabs(denom) < 1e-10 )
      { break; // cannot safely divide by this denominator -> stop iteration
      }
     // ex: c = (a*yb - b*ya) / denom;  // Regula Falsi, basic form...
     // faster converging:
     c = a + ya * (a-b) / denom;
     yc= CubicInterpolate( pInterpolator, c );  // f(c[k])
     // > Nun whlt man a[k], b[k] folgendermaen:
     // >  * a[k]=c[k], b[k]=b[k-1]
     //         falls f(a[k-1]) und f(c[k]) gleiches Vorzeichen haben sowie
     // >  * a[k]=a[k-1], b[k]=c[k]
     //         falls f(b[k-1]) und f(c[k]) gleiches Vorzeichen haben.
     if( (ya * yc ) > 0.0 )      // same sign in f(a[k-1]) and f(c[k])
      { a = c;  ya = yc;
      }
     else if( (yb * yc ) > 0.0 ) // same sign in f(b[k-1]) and f(c[k])
      { b = c;  yb = yc;
      }
     else // not sure if we ever get here, but don't assume anything..
      { break;  // stop iteration !
      }
   } // end for
  // Arrived here, 'c' is actually the 'mu' (0...1) of the zero-crossing
  //               between x1/y1 and x2/y2 in the cubic interpolation .
  return c;
} // end CubicIpol_FindZero()

/***************************************************************************/
double CubicIpol_FindLocalMaximum( T_CubicInterpolator *pInterpolator,
                                   double *pdblOutMu )
  // Finds the 'local' maximum in the specified polynomial (cubic interpolation),
  //      between y[1] and y[2], i.e. the two 'mid' points (see CubicIpol_Init) .
  // Return value:  value of the maximum (y) .
  //  optional output:  mu  for which y(mu) == maximum .
  //                mu = 0.0 would be y1 (out of interpolation values y0..y3);
  //                mu = 1.0 would be y2 .
  // (if the input is 'noisy', mu may be slightly below zero, or slightly above one)
{
  // Funtion:      y(mu) =   a[0]*mu^3 +   a[1]*mu^2 + a[2]*mu + a[3]
  // IF there REALLY is a maximum between mu=0 and mu=1, the 1st derivate
  //              y'(mu) = 3*a[0]*mu^2 + 2*a[1]*mu   + a[2]
  //              must be zero at that point.
  // Principle: Save the original coeffs, replace the function with the 1st derivative,
  //            find the zero (of the 1st derivative), and restore the original coeffs.
  int i;
  double mu,y_max;
  double old_a[4];
  for(i=0; i<=3; ++i)
   { old_a[i] = pInterpolator->a[i];
   }
  pInterpolator->a[0] = 0.0;  // no coeff for x^3 in the derivative
  for(i=1; i<=3; ++i)
   { pInterpolator->a[i] = old_a[i-1] * (double)(4-i); // -> a[1,2,3] = 3,2,1 .
   }
  mu = CubicIpol_FindZero(pInterpolator);
  for(i=0; i<=3; ++i)  // restore the original coeffs
   { pInterpolator->a[i] = old_a[i];
   }
  y_max = CubicInterpolate( pInterpolator, mu );
  if( pdblOutMu != NULL )
   { *pdblOutMu = mu;
   }
  return y_max;
} // end CubicIpol_FindLocalMaximum()

/***************************************************************************/
BOOL SimpleLinearRegression( // determines the 'best' fitting regression line
       T_Float *pfltY,    // [in]            array with 'Y' values
       T_Float *pfltX,    // [in,optional]   array with 'X' values (NULL when samples in pfltY are 'equally spaced', x[i]=i)
       int    from_index, // [in] start index into the array(s)
       int    to_index,   // [in] end index into the array(s)
       double *pdblAlpha, double *pdblBeta) // [out] parameters for y = alpha + x * beta
{
  // wiki/Simple_linear_regression  // Lineare_Regression // "Ordinary Least Squares-Estimator" :
  // > Fitting the regression line
  // >   Suppose there are n data points {(xi, yi), i = 1, ..., n}.
  // >   The function that describes x and y is:
  // >     y[i] = alpha + beta * x[i] + epsilon[i] .
  // >   The goal is to find the equation of the straight line
  // >     y = alpha + beta * x ,
  // >   which would provide a "best" fit for the data points.
  // >   Here the "best" will be understood as in the least-squares approach:
  // >   a line that minimizes the sum of squared residuals of the linear regression model.
  //   (..)
  // >             n * sum( x[i]*y[i] ) - sum(x) * sum(y)
  // >  beta  =  ----------------------------------------------------------
  // >             n * sum( x[i]  )    - ( sum(x) )
  // >
  // >  alpha =    avrg_y - beta * avrg_x
  //
  //  (the german wiki used 'beta2' instead of 'beta' and 'beta1' instead of 'alpha')
  double x, y, sum_x, sum_y, sum_xy, sum_xx, denom;
  double n = (double)(1 + to_index - from_index);  // number of points
  double alpha = 0.0;
  double beta  = 0.0;
  int    i;
  BOOL   fResult = FALSE;
  if( n>=2.0 ) // can only calculate parameters for a line (Gerade) with at least TWO points
   {
     sum_x = sum_y = sum_xy = sum_xx = 0.0;
     for( i=from_index; i<=to_index; ++i )
      { x = ( pfltX != NULL ) ? pfltX[i] : i;
        y = pfltY[i];
        sum_x  += x;
        sum_y  += y;
        sum_xy += x * y;
        sum_xx += x * x;
      }
     denom = n * sum_xx - sum_x * sum_x; // denominator for the 'beta' formula
     if( denom != 0.0 )
      { beta = ( n * sum_xy - sum_x * sum_y) / denom;
        fResult = TRUE;
      }
     alpha = ( sum_y - beta * sum_x ) / n;
   }
  if( pdblAlpha != NULL )
   { *pdblAlpha = alpha;
   }
  if( pdblBeta  != NULL )
   { *pdblBeta = beta;
   }
  return fResult;          // returns TRUE when successful
} // end SimpleLinearRegression()


/***************************************************************************/
void GPS_Det_GetFsMinAndMax(
       T_GPS_Detector *pGPSdet, // [in,out] instance data for the GPS sync detector/decoder
       double *fs_min, double *fs_max )
{
  *fs_min = (1.0 - 1e-6*pGPSdet->dblMaxFreqOffset_ppm) / pGPSdet->t_sample;
  *fs_max = (1.0 + 1e-6*pGPSdet->dblMaxFreqOffset_ppm) / pGPSdet->t_sample;
}

/***************************************************************************/
BOOL GpsDet_IsShortSyncPulse( T_GPS_Detector *pGPSdet ) // decides which 'algorithm' to use TRUE; for "very short" or "normal" sync pulses
{ return (pGPSdet->dblPulseWidth_sec>0.0) && (pGPSdet->dblPulseWidth_sec<(2.0 * pGPSdet->t_sample) );
}

/***************************************************************************/
void GpsDet_ApplyNewMeasuredSampleRate( T_GPS_Detector *pGPSdet ) // [in] pGPSdet->dblCurrentMeasuredSampleRate
  // Common part of GPS_Det_ProcessInterpolatedStep()
  //            and GPS_Det_ProcessInterpolatedShortPulse() .
{
  int    n,i;
  double d,avrg,sum_of_squares;
  double f_sample, fs_min, fs_max;

  GPS_Det_GetFsMinAndMax( pGPSdet, &fs_min, &fs_max ); // -> min. and max tolerable sampling rate [Hz]


  //-----------------------------------------------------------------
  // Store the momentary (non-filtered) measured sampling rate in an array.
  // This will be used for averaging and to calculate the standard deviation.
  // dblMeasuredSampleRateHistory[GPSDEC_SR_HISTORY_BUF_LEN]; // [0]=oldest, [59]=newest entry
  f_sample = pGPSdet->dblCurrentMeasuredSampleRate;
  if( ( f_sample >= fs_min ) && ( f_sample <= fs_max ) )  // only "good readings" for the average / history / drift rate calculation !
   {
     if( pGPSdet->iMeasuredSampleRateHistoryLength<GPSDEC_SR_HISTORY_BUF_LEN )
      { pGPSdet->dblMeasuredSampleRateHistory[pGPSdet->iMeasuredSampleRateHistoryLength++] = f_sample;
      }
     else // history buffer already full -> shift up the SR-history-buffer,
      {   // so the OLDEST entry is at index zero, and the NEWEST at the end.
        for(i=0; i<GPSDEC_SR_HISTORY_BUF_LEN-1; ++i)
         { pGPSdet->dblMeasuredSampleRateHistory[i] = pGPSdet->dblMeasuredSampleRateHistory[i+1];
         }
        pGPSdet->dblMeasuredSampleRateHistory[GPSDEC_SR_HISTORY_BUF_LEN-1] = f_sample;
        pGPSdet->iMeasuredSampleRateHistoryLength=GPSDEC_SR_HISTORY_BUF_LEN;
      }
     // Use averaging for the sampling rate emitted in the T_ChunkInfo struct.
     //  The resulting timestamps must remain consistent with the indicated
     //  sampling rate; and 'steps' must be avoided, otherwise the output
     //  from the 'timed resampler' (TimedResampler.c) will not be spectrally
     //  'clean', and not suitable for phase measurements.
     //  BTW the resampling takes place
     //    in c:\cbproj\SpecLab\SoundThd.cpp :  TSoundThread::Execute()
     //    by calling TResampler_WriteInput() / TResampler_ReadOutput() .
     d = 0.0;
     n = pGPSdet->nAveragesWanted; // ideal number of averages to calculate the "averaged" SR
                     // (ex: n = 5; since 2015-02-22 the number of averages
                     //  is adjustable via UConfig.iSRCalibAverages
                     //   -> FreqCalibrator.cpp :: FreqCalib_Init( iAverages )
                     //    -> GpsPulseDetector.h : T_GPS_Detector.nAveragesWanted .
     if( n<2 )
      {  n=2;
      }
     if( n> pGPSdet->iMeasuredSampleRateHistoryLength )  // typically limited to 60 [entries, seconds]
      { n = pGPSdet->iMeasuredSampleRateHistoryLength;
      }
     for(i=1; i<=n; ++i) // take the average of the LATEST "n" ENTRIES in the history:
      { d += pGPSdet->dblMeasuredSampleRateHistory[pGPSdet->iMeasuredSampleRateHistoryLength-i];
      }
     avrg = d / (double)n;
     d = (pGPSdet->dblAveragedSampleRate - avrg) // SR difference between 'new' and 'previous' measurement
          / pGPSdet->t_sync_interval;  // -> converted into a 'drift rate' [Hz/sec = Hz^2]
     pGPSdet->dblAveragedSampleRate = avrg;
     // Is the calculated sample rate drift (in Hz / second, or 1/sec^2) plausible ?
     // With a measured standard deviation of less than 500 ns / second = 0.5ppm in the 1-second pulses,
     // the sampling rate shouldn't drift by more than 0.5ppm * 48 kHz = 0.025 Hz per second.
     // (with n=5, d was actually -0.0051; -0.00012; -0.0013;  -0.0007 Hz/sec, etc... )
     if( (d>=-0.025) && (d<=0.025) )
      { pGPSdet->dblChunkInfo_SamplingRateDrift = d;
        pGPSdet->fExcessiveDrift = FALSE;
      }
     else // cannot calculate a 'sample rate drift' (rate) yet :
      { pGPSdet->dblChunkInfo_SamplingRateDrift = 0.0;
        if( ! pGPSdet->fExcessiveDrift )
         { sprintf( pGPSdet->sz255StatusText, "PulseDet: Excessive SR drift (%.3lf Hz)",(double)d );
           GpsPulseDet_DebugPrintf( pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero,
                                    pGPSdet->sz255StatusText );
           // 2015-01-26 : Got here occasionally, reason unknown, with..
           // > 2015-01-26 22:14:23  ProcessSync: Excessive SR drift (0.067)
           // > 2015-01-26 22:14:28  ProcessSync: Excessive SR drift (-0.066)
           // (measured sampling rate "stepped forward by quite a lot,
           //                          then back again by the same amount".
           //  Most likely reason: slighly "misplaced" sync pulse )

         }
        pGPSdet->fExcessiveDrift = TRUE;
      }

     //-----------------------------------------------------------------
     // For comparisons and testing, the standard deviation
     //   of the measured sampling rates
     //   is calculated for the last 60 measuring cycles :
     // dblMeasuredSampleRateHistory[GPSDEC_SR_HISTORY_BUF_LEN]; // [0]=oldest, [59]=newest entry
     if( pGPSdet->iMeasuredSampleRateHistoryLength >= 2 )
      {
        avrg = 0.0; sum_of_squares = 0.0;
        for( i=0;i<pGPSdet->iMeasuredSampleRateHistoryLength; ++i )
         { avrg += pGPSdet->dblMeasuredSampleRateHistory[i];  // mean or average ? ... vote for the simplest
         }
        avrg /= (double)pGPSdet->iMeasuredSampleRateHistoryLength;
        pGPSdet->dblMeanSampleRateFromHistory = avrg;
        for( i=0;i<pGPSdet->iMeasuredSampleRateHistoryLength; ++i )
         { d = pGPSdet->dblMeasuredSampleRateHistory[i] - avrg;
           sum_of_squares += d*d;
         }
        // "population standard deviation" (http://en.wikipedia.org/wiki/Standard_deviation) :
        pGPSdet->dblStdDevInSampleRateHistory = sqrt( sum_of_squares
                 / (double)pGPSdet->iMeasuredSampleRateHistoryLength );
      } // end if < history is 'long enough' for the statistics >
   } // end if(  ( f_sample >= fs_min ) && ( f_sample <= fs_max ) )
  else // called GpsDet_ApplyNewMeasuredSampleRate() but f_sample (measured) is too far off ?
   { DEBUGGER_BREAK();
   }
} // end GpsDet_ApplyNewMeasuredSampleRate()

/***************************************************************************/
BOOL GPS_Det_ProcessInterpolatedShortPulse( T_GPS_Detector *pGPSdet,
                long nAudioSamplesBetweenThisAndPreviousCall )
  // Processes the interpolated sync signal from a GPS receiver (1-pps-signal),
  //       for "short" pulses (sometimes even shorter than a SINGLE sample from the soundcard).
  //  [in]  pGPSdet->fltInterpolatedStep[0...C_GPSDET_INTERPOLATED_STEP_LENGTH-1], (*)
  //        pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero .
  //  [out] pGPSdet->dblCurrentMeasuredSampleRate ,
  //        pGPSdet->dblInterpolatedStepZeroCrossingIndex , etc .
  //  (same for GPS_Det_ProcessInterpolatedStep() and GPS_Det_ProcessInterpolatedShortPulse() )
  //
  // Returns TRUE when the measurement *looks* valid .
  //
  // Note: (*) spGPSdet->fltInterpolatedStep[] was sampled
  //           at C_GPSDET_INTERPOLATION_RATIO * pGPSdet->dblCurrentMeasuredSampleRate,
  //           that's up to 4 * 192 kHz = 768 kHz !
  //
{
  int n = C_GPSDET_INTERPOLATED_STEP_LENGTH;
  int i,a,b,c;   // running index, index of peak, index range for centroid calculation
  double y,yc,M,R;
  double d;
  double za, zb, zi, step, mu;
  BOOL fResultOk = TRUE;
  T_CubicInterpolator *pInterpolator = &pGPSdet->CubicInterpolator;

  DOBINI( nAudioSamplesBetweenThisAndPreviousCall );

  // Prepare finding the centroid (Schwerpunkt) of the pulse ...
  //______________________________________________________________
  //                                                              |
  // Figure 2 :   Idealized *interpolated* waveform               |
  // ----------      of a 'very narrow' pulse (10 us,             |
  //                similar can be seen in SL's scope window)     |
  //                                                              |
  //                              _                               |__ yc
  //                             / \                              |
  //                            /   \                             |
  //                           /     \                            |
  //  _____________---__      /       \      __---_______________ |___ 0
  //                    --___-         -___--                     |
  //      |                   |   |   |                     |     |
  //      |                   a   c   b                     |     |
  // i : [0] ..................[center].................. [n-1]   |
  //______________________________________________________________|
  //
  // The centroid detection (center of gravity) only uses the 'center'
  //    of the main peak, between the innermost zero crossings (a .. b),
  //    but skips the 'ripple' before and after the pulse
  //
  // Similar centroid detection range also in Paul Nicholson's VLF-RX tools (vttime.c),
  //    but the pulses used there are much wider (stretched with an RC lowpass).
  //
  // Step 1:  Find the index of the peak (within the INTERPOLATED waveform sketched above)
  yc = pGPSdet->fltInterpolatedStep[0];
  c  = -1;  // no 'center peak' found .. "complete silence" ?!
  for(i=0; i<n; ++i )
   { y = pGPSdet->fltInterpolatedStep[i];
     if( y > yc )
      { c  = i;
        yc = y;
      }
   }
  if( c<0 )
   { GpsPulseDet_DebugPrintf( pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero,
              "No 'center' peak found in interpolated sync pulse." );
     return FALSE;
     // Example : E-MU 0202, line-in gain pot set to 50 %, 10 us pulse with 250 mV,
     //           would reach 50 % of clipping level when sampled at 192 kHz
     //           but reached yp = 0.2 (20 %) when sampled at 48 kHz .
   }
  // Step 2:  Beginning at the peak, look for the first zero-crossings on both sides..
  a = b = -1;
  for(i=c; i>=0; --i )
   { if( pGPSdet->fltInterpolatedStep[i] <= 0.0 )
      { a = i;    // index of zero-crossing on the LEFT pulse edge
        break;
      }
   }
  for(i=c; i<n; ++i )
   { if( pGPSdet->fltInterpolatedStep[i] <= 0.0 )
      { b = i;    // index of zero-crossing on the RIGHT pulse edge
        break;
      }
   }
  // Step 3:  If there are no zero-crossings on both sides of the peak,
  //          (for example because the "sync pulse" is a low-pass filtered triangle)
  //          find the 1% locations of the pulse amplitude as in vttime.c .
  //   (never happend with a 10-us pulse: typical values were a=22, b=32, c=27, n=64 )
  if( a<0 || b<0 )
   {
     y = 0.01 * yc;
     a = b = -1;
     for(i=c; i<n; ++i )
      { if( pGPSdet->fltInterpolatedStep[i] < y )
         { a = i;
           break;
         }
      }
     for(i=c; i>=0; --i )
      { if( pGPSdet->fltInterpolatedStep[i] < y )
         { b = i;
           break;
         }
      }
   }
  if( a<0 || b<0 )  // still no reasonable range for centroid calculation ? ?
   { GpsPulseDet_DebugPrintf( pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero,
              "Interpolated sync pulse too wide." );
     return FALSE;
   }

  // Centroid calculation (Schwerpunkt) ..
  //   consider each element in fltInterpolatedStep[i = a..b] being a 'particle', P[i],
  //   located on coordinate r[i] = i,  with mass m[i] .
  //   Then, the coordinate R of the center of gravity is according to en.wikipedia.org/wiki/Center_of_mass :
  // >
  // >   R = 1 / M * sum(i=a..b, m[i] * r[i] ) ,
  // >   where M is the sum of masses of all the particles .
  // >
  //  (boiling this down, WB arrived at a similar algorithm as in Paul's vttime.c)
  M = R = 0.0;
#if(0)  // "simple" centroid calculation, using integer array indices as X-coords
  // Worked acceptably for 10 us pulses with 192 kHz sampling, but not 48 kHz:
  for(i=a; i<=b; ++i)
   { d =  pGPSdet->fltInterpolatedStep[i];
     R += (double)i * d;
     M += d;  // "sum of masses of all the particles"
   }
#elif(1)  // because the above had a too large std dev at low sampling rates (see comparisons below) : "advanced" centroid
  // Improve the accuracy of the centroid detection by using sub-sampling,
  // and using a loop with fractional steps / interpolation to let the
  // centroid detection area begin and end EXACTLY at the zero-crossings:
  //___________________________________________________________________
  //       |<- centroid calculation area->|                            |
  //       .             _____            .                            |
  //       .         ___/  |  \__         .                            |
  //       za     __/  .   |   . \__     zb (zero crossing coords,     |
  //       .    _/ |   .   |   .  | \_    .        floating point)     |
  //       .   /.  |   .   |   .  |  .\   .                            |
  //       .  / .  |   .   |   .  |  . \  .                            |
  //       . /  .  |   .   |   .  |  .  \ .                            |
  //       ./|  .  |   .   |   .  |  .  |\.                            |
  //  _____/_|_____|_______|______|_____|_\_______________  0          |
  //     _/| |  .  |   .   |   .  |  .  | |\_      ___/                |
  //   _/                                    \____/                    |
  //                                                                   |
  //  i :    a    a+1     a+2    a+3   b=a+4 (integer array indices)   |
  //         |<--->| one sample in pGPSdet->fltInterpolatedStep[],     |
  //                 divided into even smaller sub-samples (dotted)    |
  //                 using CUBIC interpolation                         |
  //                 (good approximation of a sinusoidal shape)        |
  //___________________________________________________________________|

  // Find the precise (fractional) index of the zero-crossings near 'a' and 'b'
  if( a<1 ) a=1;
  a = GpsDet_FindZeroCrossing( pGPSdet->fltInterpolatedStep, a-1, a+1, &za );
  if( b<1 ) b=1;
  b = GpsDet_FindZeroCrossing( pGPSdet->fltInterpolatedStep, b-1, b+1, &zb );
  if( a<0 || b<0 )
   { GpsPulseDet_DebugPrintf( pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero,
              "No zero-crossing in interpolated sync pulse." );
     return FALSE;
   }

  // Since we have the cubic interpolation already, the special cases
  // ('triangles' at the centroid area's ends, zero crossings za,zb) can be
  // avoided by using FRACTIONAL, NARROW STEPS. The loop starts at 'za',
  // ends *precisely* at 'zb', and uses a stepwidth of *APPROXIMATELY* 0.5 :
  step = 0.5 * ( zb - za ) / (double)(b-a);
  zi = za;
  while( zi < (zb+step/2) ) // beware of floating point rounding
   { i = (int)zi;  // <- TRUNCATES, thus offsets -1,0,+1,+2 for interpolation !
     CubicIpol_Init( pInterpolator,  // <- doesn't hurt to call even if 'i' didn't change
            /* y0 */ pGPSdet->fltInterpolatedStep[i-1],
            /* y1 */ pGPSdet->fltInterpolatedStep[i  ],
            /* y2 */ pGPSdet->fltInterpolatedStep[i+1],
            /* y3 */ pGPSdet->fltInterpolatedStep[i+21] );
     mu = zi - (double)i; // -> mu = 0.0 .. 1.0
     d  = CubicInterpolate( pInterpolator, mu );
     R += zi * d;  // same centroid calculation as above, but with "high resolution" and fractional steps
     M += d;       // "sum of masses of all the particles"

     zi += step; // next 'fractional' array index for the cubic interpolator
   }

#else   // FOR COMPARISON ONLY : Use the *INTEGER* center index only, no centroid calculation, no interpolation:
  R = c;   // crude test for the standard deviation (~jitter) WITHOUT centroid calculation..
  M = 1.0; // dummy 'mass' to avoid div-by-zero further below
#endif
  if( M>0.0 )
   { R /= M;  // e.g. R = 27.05 when c (center, integer) was 27 .
   }
  else  // oops.. no masses at all ?
   { R = 0.5 * (double)(a+b);
   }
  // At this point, R ("coordinate of the center of gravity") is in fact
  //         a fractional array index, stepping by 1 / f_sample_INTERPOLATED.
  // For compatibility with GPS_Det_ProcessInterpolatedStep(), store the result HERE:
  pGPSdet->dblInterpolatedStepZeroCrossingIndex = R;

  // Calculate the number of ADC samples (fractional) between THIS pulse (center) and the previous one:
  d = (double)nAudioSamplesBetweenThisAndPreviousCall       // <- INTEGER part
             + ( pGPSdet->dblInterpolatedStepZeroCrossingIndex // <- fractional part..
                -pGPSdet->dblPrevInterpolatedStepZeroCrossingIndex)
                / (double) C_GPSDET_INTERPOLATION_RATIO;
  if( pGPSdet->t_sync_interval > 0.0 ) // almost always 1.0 [seconds per GPS-sync-pulse]
   { d /= pGPSdet->t_sync_interval;    // number of ADC samples -> samplerate
   }
  pGPSdet->dblCurrentMeasuredSampleRate = d;
  pGPSdet->dblPrevInterpolatedStepZeroCrossingIndex = pGPSdet->dblInterpolatedStepZeroCrossingIndex;

  // Sanity check (only use the result if the results calculated above are plausible):
  d = 1e6 * (  pGPSdet->dblCurrentMeasuredSampleRate * pGPSdet->t_sample - 1.0/*ideal*/ ); // -> ~ppm
  if( (d<-pGPSdet->dblMaxFreqOffset_ppm) || (d>pGPSdet->dblMaxFreqOffset_ppm) )
   { if( ! pGPSdet->samplerate_too_far_off ) // avoid flooding the error history..
      {
        d = pGPSdet->ldblUnixDateAndTimeForChunkInfo - pGPSdet->dblGoodTimestampsSince; // running "long enough" on "good timestamps" ?
        //  After starting, dblGoodTimestampsSince = 0.0 ->
        //  d is a very LARGE number, which means "NO good timestamps yet" !
        if( d>=10.0 && d<100.0 ) // suppress warning "when no good timestamps" yet (10 seconds to settle after start)
         { GpsPulseDet_DebugPrintf( pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero,
             "PulseDet: Measured SR off by %.2lf ppm",(double)d);
         }
      }
     pGPSdet->samplerate_too_far_off = TRUE;
     fResultOk = FALSE;
   }
  else
   { pGPSdet->samplerate_too_far_off = FALSE;
   }
  if( fResultOk )
   { GpsDet_ApplyNewMeasuredSampleRate( pGPSdet ); // [in] pGPSdet->dblCurrentMeasuredSampleRate .
     // This also measures the standard deviation in the last 60 pulses. Results listed below.
   }
  else
   { ++pGPSdet->i32SyncErrorCounter;
   }
  return fResultOk;

  // Comparison of different methods to measure time for 'short' pulses :
  // * Standard deviation in 60 sync pulses, rectangular pulses, width = 10 us,
  //          on E-MU 0202 at f_sample = 48 .. 192 kHz (ADC sampling rate),
  //          processed with INTERPOLATION_RATIO=4 and the above simple centroid calculation
  //          with INTEGER array limits (no precise zero crossings) :
  //     StdDev60  = ~ 600 ns/second at f_sample =  48 kHz,  (??!)
  //     StdDev60  = ~  26..30 ns/second at f_sample = 192 kHz .
  //                   (on day with strong solar wind / K=6 in Kiruna,
  //                    StdDev60 didn't drop below 37 ns/sec.. Did Mrs Garmin suffer from it ?)
  // * Same again, but now with with INTERPOLATION_RATIO = 8
  //     StdDev60  = ~ 190 ns/second at f_sample =  48 kHz,  ("simple" centroid)
  //     StdDev60  = ~  52 ns/second at f_sample =  48 kHz,  ("advanced" centroid)
  //     StdDev60  = ~  22..30 ns/second at f_sample = 192 kHz . (no improvement)
  // * Similar, but back to INTERPOLATION_RATIO = 4 with "advanced" centroid :
  //     StdDev60  = ~  52 ns/second at f_sample =  48 kHz (no need for 8-fold interpolation with the "advanced" centroid)
  //                   (on a day with strong solar wind / K=6 in Kiruna,
  //                    StdDev60 = 63 instead of 52 ns/sec !)
  //
  // * Same hardware, 10 us pulses, with R=c ("crude test" without centroid calculation at all),
  //          processed with with INTERPOLATION_RATIO=4
  //     StdDev60  = ~2590 ns/second (at f_sample = 48 kHz)
  //     StdDev60  = ~ 650 ns/second (at f_sample = 192 kHz;
  //          "theoretic jitter" = 1 / ( INTERPOLATION_RATIO*f_sample ) = 1300 ns;
  //          standard deviation = half of the "theoretic jitter". Nice. )
  //
  // * Same.., but 100 ms sync pulses and *interpolated edge* :
  //     StdDev60  = ~ 200 ns/second at f_sample = 48 kHz,
  //     StdDev60  = ~  50 ns/second at f_sample = 192 kHz,
  //        see test results listed in GpsDet_DumpInternalVariables() .
  //        The 'short' pulses / centroid detection gave slightly better std dev than
  //        the 'long' pulses / edge detection, most likely because the
  //        (old) edge detection algorithm only looked at FOUR interpolated samples
  //        to find the zero-crossing in the DC-free pulse edge,
  //        while the centroid detection looked at FIFTEEN interpolated samples.
  //
  // * Same.., but 100 ms sync pulses differentiated via software + centroid calculation :
  //     StdDev60  = ~  60 ns/second at f_sample = 48 kHz,  (!)
  //     StdDev60  = ~  55 ns/second at f_sample = 192 kHz .

} // end GPS_Det_ProcessInterpolatedShortPulse()


/***************************************************************************/
BOOL GPS_Det_ProcessInterpolatedStep( T_GPS_Detector *pGPSdet,
                long nAudioSamplesBetweenThisAndPreviousCall )
  // Processes the interpolated RISING EDGE of a GPS receiver's 1-pps-signal
  //       with "long" pulses (longer than a dozen samples from the soundcard).
  //  [in]  pGPSdet->fltInterpolatedStep[0...C_GPSDET_INTERPOLATED_STEP_LENGTH-1], (*)
  //        pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero .
  //  [out] pGPSdet->dblCurrentMeasuredSampleRate ,
  //        pGPSdet->dblInterpolatedStepZeroCrossingIndex , etc .
  //  (same for GPS_Det_ProcessInterpolatedStep() and GPS_Det_ProcessInterpolatedShortPulse() )
  //
  // Returns TRUE when the measurement *looks* valid .
  //
  // Note: (*) pGPSdet->fltInterpolatedStep[] is sampled
  //           at C_GPSDET_INTERPOLATION_RATIO * pGPSdet->dblCurrentMeasuredSampleRate,
  //           that's up to 4 * 192 kHz = 768 kHz !
  //
{ int    n,i,iZeroXingIndex;
  double dblZeroXingIndex = 0.0;
  double dblZeroXingIndex2;
  double dblApproxTimeAtZeroXing; // .. for DebugPrintf()
  double mu_zero,d,y,y1,y2,sign;
  double dt; // ,alpha,beta;
  int xm[2]; double mu[2], ym[2];
  T_CubicInterpolator *pInterpolator = &pGPSdet->CubicInterpolator;
  BOOL fResultOk = TRUE;

  DOBINI( nAudioSamplesBetweenThisAndPreviousCall );



  // For 'long' pulses, the interpolated, DC-free signal crosses zero near the middle
  // of pGPSdet->fltInterpolatedStep[C_GPSDET_INTERPOLATED_STEP_LENGTH] .
  //    To find the 50 percent "threshold", use the 'LOW' and 'HIGH' levels
  //    a few ten microseconds away from the rising slope (instead of relying
  //    on the 'DC removal' - because there may be some remaining hum, etc) .
  //
  // Step 1:  Find the index where the interpolated step crosses zero
  iZeroXingIndex = GpsDet_FindZeroCrossing( pGPSdet->fltInterpolatedStep,
                         C_GPSDET_INTERPOLATED_STEP_LENGTH/4, // [in] from index
                       3*C_GPSDET_INTERPOLATED_STEP_LENGTH/4, // [in] to index
                       &dblZeroXingIndex ); // [out] precise zero-crossing index (fractional)

  // The following timestamp is just an APPROXIMATION, used for the timestamps in DebugPrintf().
  dblApproxTimeAtZeroXing = pGPSdet->ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero
       + dblZeroXingIndex / (C_GPSDET_INTERPOLATION_RATIO * pGPSdet->dblAveragedSampleRate);

  if( pGPSdet->iSoftwareTestCommand==GPSDET_TEST_SIMULATE_LOST_SYNC_PULSE ) // SOFTWARE TEST ?
   { // This test can be "injected" anytime from SRCalibCtrl.cpp, test results documented there.
     iZeroXingIndex = -1;  // "lost sync pulse test", should always be detected (and result in an entry in the error history)
     pGPSdet->iSoftwareTestCommand = GPSDET_TEST_DONE; // here: after GPSDET_TEST_SIMULATE_LOST_SYNC_PULSE
   } // SOFTWARE TEST ?

  pGPSdet->dblInterpolatedStepZeroCrossingIndex = dblZeroXingIndex;

  if( iZeroXingIndex <= 0 )
   { // error: no zero-crossing found where it should be !
     // (this may be a GPS glitch, which must be handled properly AT A HIGHER LEVEL..)
     fResultOk = FALSE;
     // Time shown here (in GPS_Det_ProcessInterpolatedStep) applies to the 'expected' sync pulse zero crossing:
     GpsPulseDet_DebugPrintf( dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
       "PulseDet: Bad zero-crossing" );
   }
  else // coarse zero-crossing-detection was successful ...
   {
     // To eliminate the DC offset *near the important rising edge*,
     //    measure the DC offset again (to remove it further below),
     // SYMMETRICALLY around the coarsely detected zero-crossing (integer index) .
     // Figure 3 :   Interpolation range at the "long" sync pulse's leading edge
     // ----------
     //
     //                     __      /_ _ _ _ _ _ Interpolated maximum
     //                    /y2\__   \                     ( -> y2 )
     //                   /
     //                  /          /_ _ _ _ _ _ measured "DC" offset
     //                 /.          \               = (ymin+ymax)/2
     //           _    / .
     //            \__/  .          /_ _ _ _ _ _ Interpolated minimum
     //             y1   .          \                     ( -> y1 )
     //                  .
     //                  begin searching at the 'coarse' zero-crossing index
     //
     //  At fsample=192 kHz, the 'negative overshoot' (y1) occurred about
     //      6 us ( ~~ 1/fsample) before the step;
     //      the positive overshoot (t2) is 6 us after the step .
     //      -> Looking <2*C_GPSDET_INTERPOLATION_RATIO> samples
     //         'to the left' and 'to the right' (should be y1 and y2) .
     xm[0] = xm[1] = iZeroXingIndex;
     y = pGPSdet->fltInterpolatedStep[iZeroXingIndex];
     n = iZeroXingIndex-2*C_GPSDET_INTERPOLATION_RATIO;
     if( n<1 )
      {  n=1;
         fResultOk = FALSE;
         GpsPulseDet_DebugPrintf( dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
                "PPS zero-crossing too early" );
      }
     for( i=iZeroXingIndex-1; i>n; --i )  // COARSE search for the negative peak (y1)
      { if( pGPSdet->fltInterpolatedStep[i] < y )
         { xm[0] = i;
         }
        y = pGPSdet->fltInterpolatedStep[i];
      }
     y = pGPSdet->fltInterpolatedStep[iZeroXingIndex];
     n = iZeroXingIndex+2*C_GPSDET_INTERPOLATION_RATIO;
     if( n>=C_GPSDET_INTERPOLATED_STEP_LENGTH )
      {  n= C_GPSDET_INTERPOLATED_STEP_LENGTH-1;
         fResultOk = FALSE;
         GpsPulseDet_DebugPrintf( dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
               "PPS zero-crossing too late" );
      }
     for( i=iZeroXingIndex+1; i<n; ++i )  // COARSE search for the posivite peak (y2)
      { if( pGPSdet->fltInterpolatedStep[i] > y )
         { xm[1] = i;
         }
        y = pGPSdet->fltInterpolatedStep[i];
      }

     // At this point xm[0] and xm[1] are the integer array indices of the
     // 'overshoot extremes'. For the cubic interpolation / peak value detection,
     // interpolate between xm and its LEFT or RIGHT neighbour,
     // whichever is closer.  Example:
     //
     // Figure 4 :    "Interpolate with LEFT or RIGHT neighbour ?"
     // ----------
     //                 *
     //                    *     ---->  interpolate between xm and xm+1
     //            *
     //           xm-1 xm  xm+1
     // --------------------------------------------------------------------
     //                 *
     //            *             ---->  interpolate between xm-1 and xm
     //                    *
     //           xm-1 xm  xm+1
     //
     //
     for(i=0; i<=1; ++i )   // i=0:negative peak (minimum),  i=1:positive peak (maximum)
      { if( pGPSdet->dwAllocdInputWaveformLength == 0 )
         { // kludge for "fast termination" (see GPS_Det_Delete)
           DOBINI( i );
           return FALSE;
         }
        sign = (i==0) ? -1.0 : +1.0;
        y1 = sign * pGPSdet->fltInterpolatedStep[xm[i]-1];  // check neighbours..
        y2 = sign * pGPSdet->fltInterpolatedStep[xm[i]+1];
        if( y1>y2 )  // left neighbour closer to the (positive-made) peak than right neighour..
         { --xm[i];  // move towards the LEFT neighbour
         }
        // Prepare the cubic interpolator for the peak-detection :
        CubicIpol_Init( pInterpolator,
         /* y0 */ sign * pGPSdet->fltInterpolatedStep[xm[i]-1],
         /* y1 */ sign * pGPSdet->fltInterpolatedStep[xm[i]  ],
         /* y2 */ sign * pGPSdet->fltInterpolatedStep[xm[i]+1],
         /* y3 */ sign * pGPSdet->fltInterpolatedStep[xm[i]+2] );
        ym[i] = sign * CubicIpol_FindLocalMaximum( pInterpolator, &mu[i] );
        if( mu[i]<-0.2 || mu[i]>1.2 ) // this should NEVER happen ! ('mu' is ideally 0.5) ->
         { DEBUGGER_BREAK(); // set breakpoint here, or use the 'common' breakpoint in DebugU1.cpp : DEBUG_Breakpoint()
           fResultOk = FALSE;
           GpsPulseDet_DebugPrintf( dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
              "PPS zero-crossing too flat" );
         }
      } // for(i=0; ..  [loop to process negative + positive peak]
     // Typical values seen here (after measuring the negative+positive peak):
     //  xm[0]=28,      xm[1]=37 ,    iZeroXingIndex = 34 (half way between xm[0] and xm[1])
     //  mu[0]=0.98,    mu[1]=0.59
     //  ym[0]=-0.46,   ym[1]=0.47

     y1 = 0.5 * (ym[0] + ym[1] );  // -> DC offset (average)
    //  y1 = 0.0;  // TEST without 'precise' DC removal
     // Subtract the DC offset from all interpolated samples, also for the scope display:
     for( i=0; i<C_GPSDET_INTERPOLATED_STEP_LENGTH; ++i )
      { pGPSdet->fltInterpolatedStep[i] -= y1;
      }

     // TWO different approaches to find the precise zero-crossing index ..
     if( ! pGPSdet->m_fAlternativeAlgorithm ) // SL : UConfig.iSRCalib_AlternativeAlgorithm
      { // Best method used as 'default', since 2016-11-05 :
        //   High-pass filter, or even differentiate the pulse edge to turn it
        //          into a 'short' pulse. Then use the centroid detection method.
        //  [in]  fltInterpolatedStep[i] : interpolated 'rising' (significant) pulse edge
        for( i=C_GPSDET_INTERPOLATED_STEP_LENGTH-1; i>=1; --i ) // differentiate..
         { // ex: pGPSdet->fltInterpolatedStep[i] -= pGPSdet->fltInterpolatedStep[i-1];
           // Due to the FOURFOLD interpolation, the slope isn't steep.
           // With amplitude = -0.2 to +0.2 ( 'peak to peak' = 0.4 ),
           // the peak of the DIFFERENTIATED slope was only 0.1 (as expected),
           // so apply a bit of "gain" here to compensate this :
           pGPSdet->fltInterpolatedStep[i] = C_GPSDET_INTERPOLATION_RATIO * ( pGPSdet->fltInterpolatedStep[i] - pGPSdet->fltInterpolatedStep[i-1] );
         }
        pGPSdet->fltInterpolatedStep[0] = 0.0; // <- only for the scope display : slope of 1st sample in array is assumed to be zero
           //  (we don't use the samples at the begin and end of the array for calculation anyway)
        return GPS_Det_ProcessInterpolatedShortPulse( pGPSdet, nAudioSamplesBetweenThisAndPreviousCall );
        // This simple methode (run the "edge" through a differentiator, and use
        // the pulse center detection on the result) worked surprisingly well:
        //  StdDev60 = about 50 to 70 ns/sec at 48ks/sec, old method gave 200 to 220 ns/sec !

      }
     else // UConfig.iSRCalib_AlternativeAlgorithm ...
      { // 2016-10: Tried something different, to reduce the standard deviation for the sync edge detection
        //          (doesn't work properly yet, the input range for the linear regression itself
        //           must be chosen "very carefully centered" / too much effort .
#      if(1) // the following algorithm was once the default edge-detection algorithm;
             // but since 2016-11 it's only kept as "alternative" algorithm
             // (as a fall back, just in case differentiating the edge is too noisy,
             //  which may happen if the GPS sync pulse has travelled over a lossy, compressed stream):

        // Prepare a cubic interpolation for the samples between
        //   iZeroXingIndex-1 and iZeroXingIndex .
        // Based on http://paulbourke.net/miscellaneous/interpolation/ : CubicInterpolate() :
        // > Cubic interpolation is the simplest method that offers true continuity
        // > between the segments. As such it requires more than just the two
        // > endpoints of the segment but also the two points on either side of them.
        // > So the function requires 4 points in all labeled y0, y1, y2, and y3,
        // > in the code below.
        // The zero-crossing is between y1(mu=0)  and y2(mu=1)  .
        CubicIpol_Init( pInterpolator,
            /* y0 */ pGPSdet->fltInterpolatedStep[iZeroXingIndex-2],
            /* y1 */ pGPSdet->fltInterpolatedStep[iZeroXingIndex-1],
            /* y2 */ pGPSdet->fltInterpolatedStep[iZeroXingIndex  ],
            /* y3 */ pGPSdet->fltInterpolatedStep[iZeroXingIndex+1] );
        // Typical values seen here: y0=-0.27 y1=-0.098 y2=+0.087 y3=+0.256 .
        //   (zero crossing between y1 and y2, which is the interpolatable range)
        // With the coeffs a[0] .. a[3], the interpolated function is :
        //    y(mu) = a[0]*mu^3 + a[1]*mu^2 + a[2]*mu + a[3];
        //    with mu=0.0 : y=y1;  ...  mu=1.0 : y=y2 .
        // Solve that formula (polynomial) for y=0.0000 . The resulting 'mu'
        // should be very close to the fractional part of dblZeroXingIndex
        // (which is based on the simple LINEAR interpolation, before DC removal)
        mu_zero = CubicIpol_FindZero( pInterpolator );
        dblZeroXingIndex2 = (double)iZeroXingIndex - 1.0 + mu_zero;
        // The result from the
        // 'simple' (linear) interpolation, dblZeroXingIndex,
        // and the cubic interpolation,     dblZeroXingIndex2,  should be almost equal.
        // Typical values : iZeroXingIndex = 32 [ = integer index after the zero-crossing ]
        //   dblZeroXingIndex = 31.8822 [ from linear interpolation, BEFORE accurate DC-removal]
        //   dblZeroXingIndex2= 31.9575 [ from cubic interpolation, AFTER accurate DC-removal]
        if( ( fabs(dblZeroXingIndex-dblZeroXingIndex2) < 1.2/*ex:1.0*/ )  )
         { // ok, use the "better" value
           pGPSdet->dblInterpolatedStepZeroCrossingIndex = dblZeroXingIndex2;
           // Note: pGPSdet->dblInterpolatedStepZeroCrossingIndex is an index
           //       into  pGPSdet->fltInterpolatedStep[] - nothing else !
           //  To use this in combination with .i64TSC ("TotalSampleCounter") ,
           //  the 'total sample counter' of the FIRST ENTRY in pGPSdet->fltInterpolatedStep[]
           //  must taken into account !
         }
        else
         { DEBUGGER_BREAK();  // << should never happen - set breakpoint here
           fResultOk = FALSE;
           GpsPulseDet_DebugPrintf(  dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
              "Pulse interpolation failed, zero crossing at idx %.2lf, mu=%.2lf",
              (double)dblZeroXingIndex, (double)mu_zero );
         }

#      else // other "alternative algorithms" ...
        // From ALL interpolated samples on the 'linear' part of the slope,
        //      perform a linear regression (best fit for least error squares).
        //      Then calculate the zero-crossing of the regression line (Interpolationsgerade).
        //
        //
        //                        ...
        //                      .. | ..   ..........
        //                     .   |   ...
        //                    .    |
        //                   /     |
        //                  /      |   ______________ y = 0
        //                 /|      |
        //                / |      |
        //     ...       .  |      |
        //        ..   ..   |      |
        //          ...     |      |
        //                  |      |
        //           |      |      |
        // [in]  x= xm[0]   |     xm[1]     ( xm[1] - xm[09 = ca. 8 )
        //                  |
        //           x = iZeroXingIndex (coarse detection, integer array index)
        //
        //  Example (with 4-fold oversampled data in pGPSdet->fltInterpolatedStep) :
        //    index i :   19     20     21     22     23     24    25    26    27    28      29
        //                    (=xm[0])                     (zXing)                 (=xm[1])
        //    Y[i]   :  -0.235 -0.247 -0.238 -0.198 -0.127 -0.033 0.067 0.155 0.216 0.244   0.234
        //                                   |____ "quite linear" slope _____|
        //    slope (approximately 'beta' after the linear regression) :
        //    beta  = delta y / delta x = (0.244- -0.247 ) / ( 28 - 20 ) = 0.061 [over the full range];
        //                            but (0.155- -0.198 ) / ( 26 - 22 ) = 0.088 [for a narrow range].
        n  = ( xm[1] - xm[0] ) / 4; // only use the "almost linear slope" for the regression line
        if( SimpleLinearRegression( // successfully determined the 'best' fitting regression line ?
            pGPSdet->fltInterpolatedStep, // [in] array with 'Y' values
            NULL,  // [in,optional]  array with 'X' values (NULL to use x[i]=i)
            iZeroXingIndex - n, // [in] from_index
            iZeroXingIndex + n, // [in] to_index
            &alpha, &beta) )    // [out] parameters for straight line, y = alpha + x * beta
         { // Wanted: fractional array index of the precise zero crossing,
           //         y = 0 = alpha + x * beta .
           //         The solution (x=-alpha/beta) would dividide  by zero if the
           //         regression line was "completely flat" (y=constant) :
           if( beta == 0.0 )
            { GpsPulseDet_DebugPrintf(  dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
                 "Pulse edge detection failed, beta = 0 ?!" );
              fResultOk = FALSE;
            }
           else
            { pGPSdet->dblInterpolatedStepZeroCrossingIndex = -alpha/beta;
            }
         }
        else
         { fResultOk = FALSE;
           GpsPulseDet_DebugPrintf(  dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
              "Linear regression failed, can't detect pulse edge" );
         }
        // 2016-11: When tested at fs=48 kHz, the above 'alternative' algorithm
        //          gave a StdDev60 about 10 to 20 % less than the old
        //          pulse-edge detection further above.
#      endif // which of the different 'alternative' algorithms to use ?
      } // end else < use an 'alternative' algorithm instead of the default EDGE DETECTION > ?

     // Calculate the number of ADC samples (fractional) between THIS pulse and the previous one:
     d = (double)nAudioSamplesBetweenThisAndPreviousCall       // <- INTEGER part
             + ( pGPSdet->dblInterpolatedStepZeroCrossingIndex // <- fractional part..
                -pGPSdet->dblPrevInterpolatedStepZeroCrossingIndex)
                / (double) C_GPSDET_INTERPOLATION_RATIO;
     if( pGPSdet->t_sync_interval > 0.0 ) // almost always 1.0 [seconds per GPS-sync-pulse]
      { d /= pGPSdet->t_sync_interval;    // number of ADC samples -> samplerate
      }
     pGPSdet->dblCurrentMeasuredSampleRate = d;
     pGPSdet->dblPrevInterpolatedStepZeroCrossingIndex = pGPSdet->dblInterpolatedStepZeroCrossingIndex;

     // Sanity check (only use the result if the results calculated above are plausible):
     d = 1e6 * (  pGPSdet->dblCurrentMeasuredSampleRate * pGPSdet->t_sample - 1.0/*ideal*/ ); // -> ~ppm
     if( (d<-pGPSdet->dblMaxFreqOffset_ppm) || (d>pGPSdet->dblMaxFreqOffset_ppm) )
      { if( ! pGPSdet->samplerate_too_far_off ) // avoid flooding the error history..
         {
           dt = pGPSdet->ldblUnixDateAndTimeForChunkInfo - pGPSdet->dblGoodTimestampsSince; // running "long enough" on "good timestamps" ?
           //  After starting, dblGoodTimestampsSince = 0.0 ->
           //  dt is a very LARGE number, which means "NO good timestamps yet" !
           if( dt>=10.0 && dt<100.0 ) // suppress warning "when no good timestamps" yet (10 seconds to settle after start)
            { GpsPulseDet_DebugPrintf(  dblApproxTimeAtZeroXing, // ex: pGPSdet->dblUnixDateAndTimeFromChunkInfo
                "PulseDet: Measured SR off by %.2lf ppm",(double)d);
            }
         }
        pGPSdet->samplerate_too_far_off = TRUE;
        fResultOk = FALSE;
        // 2015-01-26 : Got here after triggering the "lost sample test"
        //        with pGPSdet->dblCurrentMeasuredSampleRate = 191997.66 Hz
        //         and pGPSdet->dblMeanSampleRateFromHistory = 191998.65 Hz. Ok.
        // But the emitted timestamp made a 1-second "jump forward", which is not Ok.
      }
     else
      { pGPSdet->samplerate_too_far_off = FALSE;
      }
     if( fResultOk )
      { GpsDet_ApplyNewMeasuredSampleRate(pGPSdet);
      } // end if < pGPSdet->dblCurrentMeasuredSampleRate "plausible" ? >
     else // ! fResultOk ..
      { // In this case the caller (GPS_Det_ProcessSamples -> GpsDet_InterpolateAndProcessSyncPulse)
        // must switch from "GPS-based timestamps"
        //               to "Sample-count-based timestamps",
        //      using the best available SAMPLING RATE for the calculation.
      }
   } // end else < coarse zero-crossing detection successful >

  if( ! fResultOk )
   { ++pGPSdet->i32SyncErrorCounter;
   }
  DOBINI( iZeroXingIndex );
  return fResultOk;
} // end GPS_Det_ProcessInterpolatedStep()




/***************************************************************************/
BOOL GPS_Det_ProcessSamples( // process 1-pps-signal with optional NMEA stream (on the "same wire")
       T_GPS_Detector *pGPSdet, // [in,out] instance data for the GPS sync detector/decoder
       T_Float *pAudioChunk,    // [in,sometimes out] real samples from soundcard
       T_ChunkInfo *pChunkInfo) // [in] number of samples, approximate sample rate;
                                // [in/out] UNIX time, precise sample rate, SR drift,
                                //          GPS data, etc
  // Returns TRUE when a new one-pps-measurement is finished .
  // In that case, pGPSdet->i64TSC_at_last_valid_sync_pulse
  //  can be used to 'locate' the most recent GPS sync pulse .
  //
  // Call Stack (in Spectrum Lab) :
  //  SndThd_CheckSourceAndRunFreqCalib() -> FreqCalib_ProcessSamples() -> GPS_Det_ProcessSamples()
{
  int     i,k,n,iSample,nSamples;
  T_Float fltCurrSample, x, fltNormFactor;
  T_Float tau, hp_alpha, hp_y;  // crude highpass with a time-constant of X samples
  T_Float lp_alpha, lp_y;
  T_Float fltThreshold;
  double  d, dt, tPeakDet;
  double  f_sample;
  double  t_sample = pGPSdet->t_sample;
  double  dblHRTimerValueAtSyncPulse_sec;
  BOOL    fShortSyncPulse;
  BOOL    fWasOk, fPulseOk = FALSE;
  BOOL    fSetTimeInChunkInfo      = FALSE;
  BOOL    fSetTimeForSyncdWaveform = TRUE;

#ifdef __BORLANDC__
  (void)fSetTimeInChunkInfo;
#endif


  if( (pGPSdet->iSoftwareTestCommand==GPSDET_TEST_SIMULATE_LOST_SAMPLE)  // SOFTWARE TEST ?
    &&(pChunkInfo->dwNrOfSamplePoints>1) )
   { // This test can be "injected" anytime from SRCalibCtrl.cpp, test results documented there.
     --pChunkInfo->dwNrOfSamplePoints; // "lost sample test", here in GpsPulseDetector.cpp
     pGPSdet->iSoftwareTestCommand = GPSDET_TEST_DONE; // here: deliberately dropped a single sample
   } // SOFTWARE TEST ?


  nSamples = (int)pChunkInfo->dwNrOfSamplePoints;
  pGPSdet->nSamplesPerChunk = nSamples;  // only for debugging (GpsDet_DumpInternalVariables)

  DOBINI( nSamples );

  pGPSdet->dblUnixDateAndTimeFromChunkInfo = pChunkInfo->ldblUnixDateAndTime; // for debug-messages

  // 2015-11-25 : GPS_Det_ProcessSamples() 'destroyed' the timestamps
  //              when there was no valid sync pulse from the GPS.
  //    That happened with pGPSdet->i64TSC_at_last_valid_sync_pulse = 0,
  //    which means no valid sync-pulse yet, and no date-and-time from the GPS as well.
  //    Fixed using the Unix-time provided as *INPUT* from Spectrum Lab (SoundThd.cpp).
  //    (Hint: timestamps can be examined at any node in SL's CIRCUIT WINDOW)
  if( pChunkInfo->dwValidityFlags & CHUNK_INFO_TIMESTAMPS_VALID/*8*/ )
   { // SL declared the *input* timestamps are valid (read by input device or system time),
     // so if pGPSdet->dblUnixDateAndTimeForChunkInfo (*For*, not *From*) is not valid yet,
     // make it valid now:
     dt = pGPSdet->ldblUnixDateAndTimeForChunkInfo - pChunkInfo->ldblUnixDateAndTime;
     if(  ( dt < -60.0) || (dt > 60.0)  )   /* (*) */
      { // The usually 'free-running' Unix time (based on the number of samples at the "calibrated" S.R.)
        // is too far off, so warp to the timestamp provided 'as input' by SpecLab:
        pGPSdet->ldblUnixDateAndTimeForChunkInfo = pChunkInfo->ldblUnixDateAndTime;
        // (*) we need this large tolerance because without NTP,
        //     a PC's system time can be INCREDIBLY FAR OFF after a few hours.
        //     That's most likely not the fault of the battery-backed-up RTC
        //     but the stupid way Windows implements its own system time
        //      - possibly using a timer interrupt service handler with low priority.
      }
   }


#define DEBUG_CHECK_TIMESTAMP_AT  2  /* 0=not at all, 1=on input, 2=on sync detection, ... */
#if( DEBUG_CHECK_TIMESTAMP_AT==1)
  GpsDet_DebugCheckTimestamp( pGPSdet, pChunkInfo->ldblUnixDateAndTime  );
#endif // DEBUG_CHECK_TIMESTAMP_AT==1

  if(  (t_sample<=0) || (pGPSdet->t_sync_interval<=0) || (pGPSdet->t_sync_interval>120.0) )  // something wrong with the configuration !
   { ++pGPSdet->i32SyncErrorCounter;
     DOBINI( pGPSdet->i32SyncErrorCounter );
     if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
      { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        strcpy( pGPSdet->sz255StatusText, "PulseDet: Bad configuration" );
        GpsPulseDet_DebugPrintf(pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/, pGPSdet->sz255StatusText );
        ++pGPSdet->iStatusTextUpdateCount;
#      ifdef CIRCUIT_STATE_ERROR  /* from CircuitDefs.h, optional */
        pGPSdet->iCircuitState = CIRCUIT_STATE_ERROR;
#      endif
      }
     return FALSE;   // something isn't properly initialized (BUG ?)
   }
  // Note: Throughout the 'lifetime' of the detector, we don't adjust
  //       the 'input' sample rate & time. So DON'T DO THIS :
  // No-No: pGPSdet->t_sample = 1.0 / pChunkInfo->dblPrecSamplingRate;
  tPeakDet= 2.0 * pGPSdet->t_sync_interval; // -> 2.0 seconds for the peak-detecing interval
  if( pChunkInfo->dblSampleValueRange > 0.0 )
   { fltNormFactor = 1.0 / pChunkInfo->dblSampleValueRange;
     // factor to normalize the input into a +/- 1 value range
   }
  else
   { fltNormFactor = 1.0;
   }

  fShortSyncPulse = GpsDet_IsShortSyncPulse( pGPSdet ); // decide which 'algorithm' to use, for "short" or "long" pulses

  // Prepare a 1st-order low pass filter, to reduce the effect of overshoot.
  //  See http://en.wikipedia.org/wiki/Low-pass_filter,
  //  "algorithmic implementation of a time-discrete filter":
  //  alpha = t_sample / ( tau * t_sample )
  //  y[i]  = y[i-1] + alpha * (x[i] - y[i-1])   // refactored lowpass formula
  tau = 2.0 * t_sample;  // RC time constant for the lowpass : two samples
  lp_alpha = t_sample / (tau + t_sample );


  // Prepare a 1st-order high pass filter (not an 'ideal' differentiator)
  tau = 1e-3;   // RC time constant for the highpass : 1 millisecond
  hp_alpha = tau / (tau + t_sample );

  // Prinzip : Vorschlag von Markus, DF6NM :
  // > Hier eine von mir frher verwendete, relativ sparsame Methode
  // > zur Interpolation einer Flanke:
  // > - berschreiten der Triggerschwelle abwarten
  // > - nur einen kleinen Bereich darum herum (z.B. 16 Samples) extrahieren
  // > - mit diesen Daten und einem FIR sinc Filter ab dem Zeitpunkt
  // >   vor dem Triggern auf vierfache Abtastfrequenz interpolieren,
  // >   solange bis das Ergebnis die Schwelle berschreitet.
  // > - zwischen den beiden letzten hochgesampelten Punkten linear interpolieren.
  // > Eine feste Schwelle hat vielleicht den Nachteil,
  // > dass sie auf Offsetdrift, Brumm etc reagiert. Alternativ knntest Du
  // > auch den steilsten Punkt raussuchen, indem Du die hochgesampelten Daten
  // > differenzierst, und dann die Position des Maximums aus drei Punkten
  // > mit Parabel-Fit extrahierst.
  // WB: An efficient interpolation algorithm is described here:
  //    http://www.dspguru.com/dsp/faqs/multirate/interpolation  .
  //    Anyway, only 16 out of 48000...192000 samples per second are
  //    interpolated by zero stuffing and a windowed-sinc lowpass,
  //    so the 'reduced computation' described at dspguru isn't required here.
  for(iSample=0; iSample<nSamples; ++iSample)
   {
     if( pGPSdet->dwAllocdInputWaveformLength == 0 )
      { return FALSE;    // kludge for "fast termination" from another thread (see GPS_Det_Delete)
      }
     DOBINI( iSample );  // -> GpsPulseDet_iSourceLine = __LINE__
     fltCurrSample = pAudioChunk[iSample] * fltNormFactor;  // get the next input sample, normalized to +/- 1.0

     // Detect the absolute peaks, because it's important to avoid 'analog' clipping
     if( fltCurrSample > pGPSdet->fltPpsNewPositivePeak )
      {  pGPSdet->fltPpsNewPositivePeak = fltCurrSample;
      }
     if( fltCurrSample < pGPSdet->fltPpsNewNegativePeak )
      {  pGPSdet->fltPpsNewNegativePeak = fltCurrSample;
      }


     // Save the input sample in pfltSyncdInputWaveform[], large enough for a complete sync cycle,
     //   synchronized by the PPS (rising edge of 'long' pulses, or peak of 'short' pulses) .
     //   This will be used for the scope display AND for the NMEA decoder,
     //   but not for the option to 'subtract the sync-pulse from the input' :
     if( (pGPSdet->pfltSyncdInputWaveform!=NULL) && (pGPSdet->dwAllocdInputWaveformLength>0) )
      { if( pGPSdet->dwSyncdWaveformSampleIndex >= pGPSdet->dwAllocdInputWaveformLength )
         { // buffer "capacity" exceeded : Remove 5 percent of the oldest samples;
           // we don't need them anymore. The buffer is 10 % longer than the 1-second cycle.
           GpsDet_RemoveOldEntriesInSyncdInputWaveform( pGPSdet );
           fSetTimeForSyncdWaveform = TRUE;
         }
        // Append the newest sample to the 1.x-second-'waveform buffer' :
        pGPSdet->pfltSyncdInputWaveform[ pGPSdet->dwSyncdWaveformSampleIndex] = fltCurrSample;
        // To retrieve precise timestamps later, save the 'global sample counter'
        //    of the first (=oldest) entry in pGPSdet->pfltSyncdInputWaveform[] :
        pGPSdet->i64TSC_FirstEntryInSyncdWaveform =
                  pGPSdet->i64TSC + (LONGLONG)iSample
                - pGPSdet->dwSyncdWaveformSampleIndex;
        pGPSdet->dwSyncdWaveformSampleIndex++; // index into pGPSdet->pfltSyncdInputWaveform[] for next sample

        // Set the APPROXIMATE timestamp of the first sample in pGPSdet->pfltSyncdInputWaveform[] .
        //  Added 2015-01-31 for DebugPrintf().  We may not have a 'precise time' here yet !
        if( fSetTimeForSyncdWaveform )
         { pGPSdet->ldblGpsTime_FirstEntryInSyncdWaveform =
                               pGPSdet->ldblTime_at_last_valid_sync_pulse
              + (long double)( pGPSdet->i64TSC_FirstEntryInSyncdWaveform
                              -pGPSdet->i64TSC_at_last_valid_sync_pulse)
                             / pGPSdet->dblAveragedSampleRate;
           fSetTimeForSyncdWaveform = FALSE;
         }
      } // end if( (pGPSdet->pfltSyncdInputWaveform!=NULL) && ...

     // COARSE pps-detection, step 1, performed on EACH input sample :
     //   Search for the largest peaks (+ and -) in the differentiated input signal,
     //         which must occur once every second.
     //   Note: Some soundcards (like the E-MU 0202) seem to invert the polarity,
     //         so the important RISING EDGE from the GPS 1-pps-output
     //         may actually be a FALLING EDGE here. Too bad !
     // For simplicity, use the negative and positive peak VOLTAGES from the LAST
     //  peak-detection interval, while already detecting the NEXT one.
     //  To make sure we catch the 'real' peaks (of the PPS pulse),
     //  the peak-detection interval is TWO seconds long
     //  (but of course the PPS-detection takes place ONCE per second).
     //
     //    Use a 1st order high-pass filter, RC time constant 1 ms (not a real differentiator)
     //    to detect the height of the POSITIVE and NEGATIVE slope
     //    of the 1-pps sync pulse.
     //  Using the 'refactored' algorithm from wiki, High-pass_filter:
     //    y[i] := alpha * (y[i-1] + x[i] - x[i-1]) // refactored highpass formula
     // ex:  x = fltCurrSample;
     x = fltCurrSample;
     hp_y = hp_alpha * ( pGPSdet->hp1_y_old + x - pGPSdet->hp1_x_old );
     pGPSdet->hp1_x_old = x;
     pGPSdet->hp1_y_old = hp_y;

     // To measure the peak voltages WITHOUT overshoot, run the highpass output
     //    through a LOW-PASS filter, with tau = 2 samples .
     // But since 2016-10, due to the Thunderbolt E's incredibly short sync pulse (10 us),
     //  this 'overshoot removal' filter may be skipped :
     if( fShortSyncPulse )
      { // extremely short sync pulse, a lowpass would 'kill' it for the peak-measurement, so:
        lp_y = hp_y; // bypass the lowpass (only use the highpass, for DC removal)
      }
     else // pulse sufficiently long -> run through the lowpass (with tau = 2 samples)
      { // Using the refactored algorithm from wiki, Low-pass_filter :
        //  y[i]  = y[i-1] + alpha * (x[i] - y[i-1])   // refactored lowpass formula
        x = hp_y;
        lp_y = pGPSdet->lp1_y_old + lp_alpha * ( x-pGPSdet->lp1_y_old );
        pGPSdet->lp1_y_old = lp_y;
      }

     if( lp_y < pGPSdet->fltNewPeakMin )
      { pGPSdet->fltNewPeakMin = lp_y;
        // NO TIMING HERE .. just get the peak values
      }
     if( lp_y > pGPSdet->fltNewPeakMax )
      { pGPSdet->fltNewPeakMax = lp_y;
      }
     pGPSdet->dblPeakDetTimer += t_sample;
     if( pGPSdet->dblPeakDetTimer >= tPeakDet/*2 seconds for SAFE detection*/ )
      { // Finished another peak-detection interval, use the NEW threshold values:
        GpsDet_UpdateSyncPulseThresholds( pGPSdet, fltCurrSample, tPeakDet );
        // -> [out]   pGPSdet->fltPpsNegativeThreshold ,  pGPSdet->fltPpsPositiveThreshold
        // 2016-10-29 : With a 10us sync pulse, amplitude about 50 % of the soundcard's clipping level,
        //              only reached ~15 % of the clipping level when digitized at 48 kS/sec, i.e.
        //              dblPulseWidth_sec=10us but t_sample=20.83us . Sounds realistic.
        //             See SL/Screenshots/10us_pulse_EMU0202_48kS.png / ..._192kS.png .
        //  With that 'simulated' Thunderbolt E pulse, and the 'inverting' E-MU 0202,
        //  got here with peak_levels_valid=TRUE,
        //                pGPSdet->fltPpsPositiveThreshold =  0.019 ,
        //                pGPSdet->fltPpsNegativeThreshold = -0.084 .
      }
     //-----------------------------------------------------------------------
     // ABOVE: peak-level analysis to catch the amplitude of the PPS signal (NMEA data must be weaker!).
     // BELOW: COARSE timing analysis. Only possible when the PPS peak levels are known !
     //-----------------------------------------------------------------------
     //        Note that the TIMING analysis uses the highpass output,
     //        not the output of the additional lowpass !
     k = 0;  // no flag to trigger the block-processing yet ...
     if( pGPSdet->peak_levels_valid ) // ready for the coarse pulse timing ?
      { // Watch for the RISING and FALLING slope the 1-pps-signal .
        // This is active over the ENTIRE 1-second-cycle, because we also
        // want to check for 'lost samples' from the soundcard this way !
        // A bit tricky due to the AC-coupling of the soundcard
        //  (so we process the output of the HIGH-PASS FILTER here),
        // and due the fact that some audio devices INVERT the polarity,
        // and due to the unknown length of the sync pulse itself.
        //
        // Figure 5 :  Typical (but idealized) oscillogram with NMEA and 'long' sync pulse
        // ----------
        //                                                                   .
        //            |<-------------- 1 second ---------->|                 .
        //                                                                   .
        //            |                                    |                 .
        //  peak_max  |\                                   |\                .
        //            | \              |||||               | \               .
        //   hp_y     |  \             ||||||||||||        |  \              .
        //   ---------|   |   __-------||||||||||||--------|   |   __------  .
        //            |   |  /                |||||            |  /          .
        //                | /     UTC second valid here -->|   | /           .
        //   peak_min     |/                                   |/            .
        //                |                                    |             .
        //            Sync-        optional NMEA-Data       Next             .
        //             pulse     (9600 or 19200 bit/sec)    Sync-Pulse       .
        //                                                                   .
        //                 |     pfltSyncdInputWaveform[]        |           .
        //                 |_____________________________________|           .
        //                  (shown on the SR detector's 'Scope')             .
        //                                                                   .
        //            |                   |                |   |             .
        //            |                   |                |   |             .
        //           t=-1000 ms          t=-500 ms        t=0  t=+100 ms     .
        //                                                 |   |
        //                               iSampleIndexOfPPS[0] [1]
        //
        if( ! fShortSyncPulse )
         {  // here: for "long enough" sync pulses (figure 5) ...
           if( hp_y > pGPSdet->fltPpsPositiveThreshold  )
            { if( pGPSdet->iSampleIndexOfPPS[0] < 0 ) // *JUST* saw the rising edge
               { // (which may, depending on the soundcard, be the END of the sync pulse)
                 pGPSdet->iSampleIndexOfPPS[0] = pGPSdet->dwSyncdWaveformSampleIndex;
                 pGPSdet->ldbDateAndTimeAtPPSEdges[0] = pChunkInfo->ldblUnixDateAndTime + (long double)iSample * t_sample;
               }
            } // end if < PPS rising edge JUST detected >
           if( hp_y < pGPSdet->fltPpsNegativeThreshold  )
            { if( pGPSdet->iSampleIndexOfPPS[1] < 0 ) // *JUST* saw the falling edge
               { // (which may, depending on the soundcard, the BEGIN of the sync pulse)
                 pGPSdet->iSampleIndexOfPPS[1] = pGPSdet->dwSyncdWaveformSampleIndex;
                 pGPSdet->ldbDateAndTimeAtPPSEdges[1] = pChunkInfo->ldblUnixDateAndTime + (long double)iSample * t_sample;
               }
            }

           if( (pGPSdet->iSampleIndexOfPPS[0] >= 0 )  // saw BOTH edges of the ('long') PPS-pulse ?
            && (pGPSdet->iSampleIndexOfPPS[1] >= 0 ) )
            { // Both edge times are (roughly) known by now . Time for 'action',
              //  including the decoding of the NMEA data burst received **BEFORE** / **AFTER**
              //  the pulse.  From the Rockwell Jupiter datasheet:
              // > .. the UTC Time Mark Pulse Output message preceeds the TMARK pulse ..
              //          ( "UTC in the NMEA sentences"      preceeds       "PPS" ?? )
              // Obviously not, at least not for Garmin GPS18x LVC :
              //  Here, the UTC time encoded in the NMEA sentence seems to apply
              //  to the preceeding sync pulse, not vice versa.
              //  In other words, ONE SECOND must be added to the decoded UTC time
              //  way further below (otherwise it would be one second late) .
              DOBINI( iSample );
              // Check for the correct polarity, maybe the soundcard has inverted it.
              n = (int)( 0.5*pGPSdet->t_sync_interval / t_sample ); // max pulse duration [samples]
              i = pGPSdet->iSampleIndexOfPPS[1] - pGPSdet->iSampleIndexOfPPS[0];
              if( (i < 0) && (i>-n) )
               { // The pulse seems to be VALID,
                 // but the polarity is obviously inverted ( a "feature" of the E-MU 0202 ? )
                 pGPSdet->t_sync_pulse_length = (double)-i * t_sample;
                 pGPSdet->pps_polarity_inverted = TRUE;
                 pGPSdet->iSampleIndexOfPPS[2] = pGPSdet->iSampleIndexOfPPS[1]; // 'important' slope here
                 pGPSdet->ldbDateAndTimeAtPPSEdges[2] = pGPSdet->ldbDateAndTimeAtPPSEdges[1];
                 k = 1;  // flag to trigger the block-processing
               }
              else if( (i > 0) && (i < n) )
               { // The pulse seems to be VALID, and the polarity is NOT inverted:
                 pGPSdet->t_sync_pulse_length = (double)i * t_sample;
                 pGPSdet->pps_polarity_inverted = FALSE;
                 pGPSdet->iSampleIndexOfPPS[2] = pGPSdet->iSampleIndexOfPPS[0]; // 'important' slope here
                 pGPSdet->ldbDateAndTimeAtPPSEdges[2] = pGPSdet->ldbDateAndTimeAtPPSEdges[0];
                 k = 1;  // flag to trigger the block-processing
               }
              else
               { // No valid sync-pulse detected at this point (edges too far apart) .
                 // If processing began at the 'wrong moment', t_sync_pulse_length
                 //   would be 0.9 instead of 0.1 [seconds]. In that case,
                 //   wait a bit longer; until the next (rising or falling) edge is detected.
                 k = 0;
                 // 2015-01-15 : Got here with i = 37021,  n = 23999 (E-MU 0202 + GPS18LVC)
               }
              // Forget the OLDER of the two edges  :
              if( pGPSdet->iSampleIndexOfPPS[0] < pGPSdet->iSampleIndexOfPPS[1] )
               { pGPSdet->iSampleIndexOfPPS[0] = -1;
               }
              else
               { pGPSdet->iSampleIndexOfPPS[1] = -1;
               }
            } // end if < both PPS pulse edges valid >

         } // end if ( ! fShortSyncPulse )
        else  // the 'expected' sync pulse is too short for the 'highpass-filtered rectangle' detection from Figure 5 (above)..
         {    //      .. so use a different 'trigger detection' :
           //
           //________________________________________________________________
           // Figure 6 : waveform with a 'very short' sync pulse, no NMEA,   |
           // ----------  not inverted (unlike 10us_pulse_EMU0202_48kS.png)  |
           //                                                                |
           //   previous pulse           current pulse                       |
           //        _                      _                                |
           //       / \                    / \                  Sync Pulse   |
           //      /   \          _       /   \       _     <-- Threshold    |
           //  __ /     \ ...___-  -_    /     \    _- -__   __---_________  |
           //    -                   -__/       \__-      ---                |
           //        |                   |  |  |                             |
           //        t=-1000ms           | t=0 |                             |
           //                            |  |  |                             |
           //          iCoarseDetState:0011122200000000....                  |
           //          iSampleIndexOfPPS[0][2][1]                            |
           //                     |<- interpolated    |                      |
           //                     |    range (later)->|                      |
           //________________________________________________________________|
           //  In THIS case, the sync pulse may be SHORTER than t_sample,
           //     but the delta-sigma ADC's inherent (and almost ideal) low-pass
           //     will not let it look like a 'Dirac' even at this state,
           //     before the interpolation .
           //
           //  Find the index of the STRONGEST peak (iSampleIndexOfPPS) :
           if( fabs(pGPSdet->fltPpsPositiveThreshold) < fabs( pGPSdet->fltPpsNegativeThreshold ) )
            { // The soundcard has invert the sync pulse .. so invert the signal for peak-measurement
              hp_y = -hp_y;
              fltThreshold = -pGPSdet->fltPpsNegativeThreshold;
              pGPSdet->pps_polarity_inverted = TRUE;
            }
           else
            { fltThreshold =  pGPSdet->fltPpsPositiveThreshold;
              pGPSdet->pps_polarity_inverted = FALSE;
            }
           switch( pGPSdet->iCoarseDetState )
            { case 0: // below threshold; before rising edge
                 if( hp_y > fltThreshold  )       // just reached the threshold ?
                  { pGPSdet->iCoarseDetState = 1;
                    pGPSdet->iSampleIndexOfPPS[0] = pGPSdet->iSampleIndexOfPPS[1] = pGPSdet->iSampleIndexOfPPS[2]
                      = pGPSdet->dwSyncdWaveformSampleIndex;
                    pGPSdet->ldbDateAndTimeAtPPSEdges[0] = pGPSdet->ldbDateAndTimeAtPPSEdges[1] = pGPSdet->ldbDateAndTimeAtPPSEdges[2]
                      = pChunkInfo->ldblUnixDateAndTime + (long double)iSample * t_sample;
                    pGPSdet->fltSyncPulsePeak = hp_y;
                  }
                 break;
              case 1: // between rising edge and peak (or, for extremely short pulses, already BELOW the threshold again)
                 if( hp_y > pGPSdet->fltSyncPulsePeak ) // amplitude still RISING ->
                  { pGPSdet->fltSyncPulsePeak = hp_y;
                    pGPSdet->iSampleIndexOfPPS[2] = pGPSdet->dwSyncdWaveformSampleIndex; // new 'temporary' peak
                    pGPSdet->ldbDateAndTimeAtPPSEdges[2] = pChunkInfo->ldblUnixDateAndTime + (long double)iSample * t_sample;
                  }
                 else // reached the top, now descending.. but usually still above the threshold
                  { pGPSdet->iCoarseDetState = 2;
                  }
                 // NO BREAK HERE !
              case 2: // somewhere between peak and threshold on falling edge
                 if( hp_y < fltThreshold  ) // oops, descended below the threshold within ONE sample !
                  { pGPSdet->iCoarseDetState = 0; // ready for the next COARSE sync pulse detection
                    pGPSdet->iSampleIndexOfPPS[1] = pGPSdet->dwSyncdWaveformSampleIndex;
                    pGPSdet->ldbDateAndTimeAtPPSEdges[1] = pChunkInfo->ldblUnixDateAndTime + (long double)iSample * t_sample;
                    pGPSdet->t_sync_pulse_length = (double)(pGPSdet->iSampleIndexOfPPS[1] - pGPSdet->iSampleIndexOfPPS[0]) * t_sample;  // -> pulse length in seconds,
                    // stretched by soundcard's low pass filter (hopefully delta-sigma type) for 'very' short pulses.
                    //
                    // Example ( with a steep 200 mV, 10 us rectangular pulse,
                    //           fed into an E-MU 0202, gain pot centered, fs=48 kHz):
                    //
                    // fltThreshold         = ~0.1   (varies greatly, depending on pulse location within t_sample)
                    // fltSyncPulsePeak     = ~0.16  (amplitude *BEFORE* interpolation varies greatly, but CONSTANT afterwards, here: 0.20)
                    // iSampleIndexOfPPS[0] = 50516  (rising pulse edge)
                    // iSampleIndexOfPPS[2] = 50516  (coarse peak position)
                    // iSampleIndexOfPPS[1] = 50517  (falling pulse edge)
                    //
                    k = 1;  // flag to trigger the interpolated block-processing further below
                            // (when enough samples are available for interpolation,
                            //  which may not be the case HERE yet, thus : )
                    if( pGPSdet->iCountdownUntilFrameProcessing < C_GPSDET_FIFO_LEN_BRUTTO )
                     {  pGPSdet->iCountdownUntilFrameProcessing = C_GPSDET_FIFO_LEN_BRUTTO;
                        // (need a few extra samples before calling GpsDet_InterpolateAndProcessSyncPulse() )
                     }
                  }
                 break;
            } // end switch( pGPSdet->iCoarseDetState )
         } // end else < "short" sync pulses, may be even shorter than t_sample >

      } // end if < valid peak levels >
     else
      { // No-no : (already warned about 'too weak' levels) ..
        // if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
        //  { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        //    strcpy( pGPSdet->sz255StatusText, "PPS peak levels invalid !" );
        //    GpsPulseDet_DebugPrintf(pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/, pGPSdet->sz255StatusText );
        //    ++pGPSdet->iStatusTextUpdateCount;
        //    pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
        //  }
      } // end else < ! pGPSdet->peak_levels_valid >


     if( k ) // flag to trigger the processing of another sync-cycle .
      { // -> pGPSdet->t_sync_pulse_length = 25ms (Jupiter) .. 100ms (Garmin) ... 10us (Thunderbolt E)
        DOBINI( iSample );
        ++pGPSdet->i32SyncPulseCounter;  // if all works well, this should increment each second
        // Collect a few more samples, then 'do the hard work' (further BELOW) :
        if( pGPSdet->iCountdownUntilFrameProcessing > C_GPSDET_FIFO_LEN_BRUTTO )
         {  pGPSdet->iCountdownUntilFrameProcessing =  C_GPSDET_FIFO_LEN_BRUTTO; // 'triggered' processing
           // (with a few extra samples for GpsDet_InterpolateAndProcessSyncPulse() )
         }
      }

     //-----------------------------------------------------------------------
     // ABOVE: COARSE sync-timing analysis ("find valid sync pulse") .
     // BELOW: Decode NMEA (GPS nav data) and PRECISE sync-timing analysis (PPS)
     //-----------------------------------------------------------------------

     --pGPSdet->iCountdownUntilFrameProcessing;
     if( pGPSdet->iCountdownUntilFrameProcessing <= 0 )
      { // Just reached the end of a full PPS / NMEA cycle .
        // Time has come to ...
        //  * decode NMEA (GPS navigation data) which may be carried on the 'same wire'.
        //  * synchronise timestamps with the GPS Sync Pulse (1 PPS)
        //  * copy the waveform data into extra arrays for plotting,
        //  * measure the (relative) position of the sync pulse
        //    with a resolution which exceeds the soundcard's sampling interval (by far!)
        //
        pGPSdet->iCountdownUntilFrameProcessing = // -> timer for next 'free-running' decode
            (int)( 1.05 * pGPSdet->t_sync_interval / t_sample );

        // Finish peak detection, and prepare the next :
        pGPSdet->fltPpsPositivePeak = pGPSdet->fltPpsNewPositivePeak;
        pGPSdet->fltPpsNegativePeak = pGPSdet->fltPpsNewNegativePeak;
        pGPSdet->fltPpsNewPositivePeak = pGPSdet->fltPpsNewNegativePeak = 0.0;

        // Copy the samples into pfltSyncdInputWave4Plot[] for "plotting"
        if( (! pGPSdet->fScopeDisplayPaused) && (pGPSdet->pfltSyncdInputWave4Plot!= NULL) )
         { // Copy the 'synchronized waveform' for plotting (in the SR Calib control panel)
           GpsDet_CopySamplesForPlotting( pGPSdet ); // [in] pGPSdet->pfltSyncdInputWaveform[], dwSyncdWaveformSampleIndex, iSampleIndexOfPPS[2] .
         }

        // Try to decode the NMEA stream from the past cycle ?
        //   (this also copies pGPSdet->pfltSyncdInputWaveform
        //    into pfltSyncdInputWave4Plot)
        // Note: Don't make too many assumptions about the timing relation
        //       between the SYNC PULSE (processed further above)
        //       and the serial NMEA data (processed further below) !
        //       DECODING OF THE NMEA DATA (below) LAGS THE PULSE DETECTION !
        //  If the previous sync pulse was discarded,
        //       (which means i64TSC_at_last_valid_sync_pulse has NOT been incremented)
        //  then the GPS time *possibly* decoded in GpsDet_DecodeSerialData() below
        //  must not overwrite
        //
        if( pGPSdet->t_NMEA_bit > 0 ) // try to decode NMEA (serial data) with date+time ?
         { BOOL fDecodedTime = FALSE;
           // Start looking for the NMEA data after the end of the 'falling edge'
           //  of the *PREVIOUS* sync pulse (if any). How to find this :
           //  pGPSdet->iSampleIndexOfPPS[2] = position (sample index) of the RECENT sync pulse, rising edge.
           //
           n = // number of samples in pfltSyncdInputWaveform[] which MAY be NMEA data
            (int)( 0.96 * ( pGPSdet->t_sync_interval - pGPSdet->t_sync_pulse_length ) / t_sample );
               // to stay clear of the falling edge of the PPS, and overshoot/"ringing",
               // even if the real sampling rate is 3 % off the "nominal value" .
               // Modified 2014-12-20. In older versions the factor was 0.9 instead of 0.97,
               // see C:/postarchiv/df6nm/2014_12_20_SpecLab_NMEA_timing.htm :
               // > 1. Wre es mglich, den Beginn des NMEA Auswertefensters
               // >    weiter nach vorne zu legen, also z.B. schon 40 ms statt
               // >    100 ms nach dem 1pps Ende ?
               //   -> begin of the NMEA window now ~ 0.96 seconds
               //      before the 1pps pulse's "rising" edge .
           i = pGPSdet->iSampleIndexOfPPS[2] - n; // -> i = index of 1st sample in pfltSyncdInputWaveform[] which MAY be NMEA
           if( n > ((int)pGPSdet->dwUsedInputWaveformLength - i) )
            {  n = ((int)pGPSdet->dwUsedInputWaveformLength - i); // don't process 'garbage'
            }
           if( (i>=0) && ((i+n)<(int)pGPSdet->dwUsedInputWaveformLength) )
            { // Example (Garmin GPS 18x LVC, 192kS/s, 100 ms pulse, 1 s cycle) :
              //          i=9814, n=172799 .
              DOBINI( i );
              dblHRTimerValueAtSyncPulse_sec = pGPSdet->ldbDateAndTimeAtPPSEdges[2] - TIM_dblPerformanceCounterToTimeOffset; // <- since 2016-04
              if( GpsDet_DecodeSerialData( pGPSdet, pGPSdet->pfltSyncdInputWaveform,
                                 dblHRTimerValueAtSyncPulse_sec,
                                 i,      // [in] iFirstNmeaSampleIndex,
                                 i+n ))  // [in] iLastNmeaSampleIndex
                                         // [out] pGPSdet->pGpsPosInfo
               { // Successfully decoded an NMEA telegram -> save all we need
                 //     for the outbound components of the T_ChunkInfo :
                 pGPSdet->dblChunkInfo_Lat_deg = C_CHUNK_INFO_LAT_INVALID;
                 pGPSdet->dblChunkInfo_Lon_deg = C_CHUNK_INFO_LON_INVALID;
                 pGPSdet->dblChunkInfo_mASL   = 0.0;
                 pGPSdet->dblChunkInfo_Vel_kmh = 0.0;
                 pGPSdet->fChunkInfo_DateAndTimeFromGPSValid = pGPSdet->pGpsPosInfo->time_valid
                                                            && pGPSdet->pGpsPosInfo->date_valid;
                 // BOTH "date" and "time" fields must be valid in the NMEA burst.
                 // (for simplicity: 'ldblTime' contains BOTH. "date" are just the MSBits of "time")
                 if( pGPSdet->fChunkInfo_DateAndTimeFromGPSValid )
                  { pGPSdet->ldblTime_from_last_valid_NMEA = UTL_ConvertYMDhmsToUnix(
                       pGPSdet->pGpsPosInfo->year, pGPSdet->pGpsPosInfo->month, pGPSdet->pGpsPosInfo->day,
                       pGPSdet->pGpsPosInfo->hour,  pGPSdet->pGpsPosInfo->minute,
                        (double)pGPSdet->pGpsPosInfo->second
                       +(double)pGPSdet->pGpsPosInfo->millisecs*1e-3);
                    // pGPSdet->ldblTime_from_last_valid_NMEA must always be used
                    //    in combination with the 'total sample counter' of reception,
                    //    pGPSdet->i64TSC_at_last_valid_NMEA, which is the only way
                    //    to tell "the age of reception of that timestamp"
                    //    because the time+date FROM THE NMEA may be wrong,
                    //    thus comparing pGPSdet->ldblTime_from_last_valid_NMEA
                    //              with pGPSdet->ldblTime_at_last_valid_sync_pulse
                    //           or even pGPSdet->ldblUnixDateAndTimeForChunkInfo
                    //    is meaningless, or may lead to "wrong conclusions" !
                    // Thus, to apply the "time FROM THE LAST VALID NMEA sentence"
                    //    later, keep track of the TSC (Total Sample Counter)
                    //    at which it was received. This TSC only needs to be
                    //    accurate to a few hundred milliseconds, because the actual
                    //    time of transmission (of the NMEA-burst) is not
                    //    as important as the TSC of the leading edge of the sync pulse.
                    //    which "belonged to the NMEA burst".
                    // GpsDet_DecodeSerialData() was instructed
                    //    to operate on the samples in
                    //      pGPSdet->pfltSyncdInputWaveform[ i ... i+n ],
                    //    to the TSC of the "center of the NMEA burst" is:
                    pGPSdet->i64TSC_at_last_valid_NMEA = // <- "Total Sample Counter" in the middle of the last VALID NMEA burst
                       pGPSdet->i64TSC_FirstEntryInSyncdWaveform + i + n/2;

                    // Not sure if the GPS receiver emits the NMEA frame
                    //  *before* or *after* the sync-pulse for the second
                    //  indicated in the NMEA time+date (string).
                    // Differenent receivers may behave differently;
                    //   thus since 2015-01-11 the *user* can 'correct' this (*) :
                    if( pGPSdet->NMEA_one_pulse_late != 0)
                     { pGPSdet->ldblTime_from_last_valid_NMEA += (double)pGPSdet->NMEA_one_pulse_late * pGPSdet->t_sync_interval;
                     }
                    // Beware, the following (nice-sounding) descripting
                    // is NOT how Garmin, Jupiter, Motorola and co operate:
                    // > " The GPSClock 200 has an RS-232 output that provides NMEA time codes
                    // >   and a PPS output signal. About a half-second before,
                    // >   it outputs the time of the next PPS pulse in either GPRMC or GPZDA format.
                    // >   Within one microsecond of the beginning of the UTC second,
                    // >   it brings the PPS output high for about 500 ms."
                    // It later turned out that NONE of the recent GPS receivers
                    // with PPS output behaves like the above "GPSClock 200";
                    // instead, they all(!) send the time (NMEA) about a half-second *AFTER*
                    //   a PPS pulse which represents that second-number !
                    //
                    //  ------------------------------------------------------
                    // | Figure 7 : "NMEA output **BEFORE** the sync pulse" ! |
                    // | ----------    (NOT compatible with Garmin & Co !)    |
                    // |                                                      |
                    // |             _                _                       |
                    // |            | |     NMEA     | |                      |
                    // |          __| |_____||||_____| |____                  |
                    // |                             PPS                      |
                    // |                     /|\    /|\                       |
                    // |                      |      |                        |
                    // |  "Beim nchsten Ton ist es  |                        |
                    // |   12 Uhr, 0 Minuten,        |                        |
                    // |       und 0 Sekunden".      "Piep!"                  |
                    //  ------------------------------------------------------
                    //
                    // BUT: Other manufacturers describe their receivers as follows
                    //      (DON'T MODIFY THIS):
                    // > For the GPS 18x LVC: Regardless of the selected baud rate,
                    // > the information transmitted by the GPS sensor is referenced
                    // > to the one-pulse-per-second output immediately *PRECEDING*
                    // > the GPRMC sentence, or whichever sentence is output first
                    // > in the burst (see Table 2 in the GPS18x 'Technical Specifications').
                    //
                    //  -----------------------------------------------
                    // | ("PRECEDE" ? Vorangehen ? ?  )                |
                    // |                                               |
                    // |             _                _                |
                    // |            | |     NMEA     | |               |
                    // |          __| |_____||||_____| |____           |
                    // |            PPS                                |
                    // |            /|\      /|\                       |
                    // |             |        |                        |
                    // |             |        |                        |
                    // |             |        |                        |
                    // |           "Piep!"    |                        |
                    // |                      |                        |
                    // |  "Beim vorherigen Ton WAR ES 12 Uhr,          |
                    // |      0 Minuten, und 0 Sekunden ?"  Och nee !  |
                    //  -----------------------------------------------
                    //
                    // (*) The above can be checked by measuring the difference
                    //     between the PC's system- and the decoded GPS time,
                    //     displayed somewhere on the 'SR Calibrator' panel;
                    //  or in a fast-running spectrogram, with DCF77 or HBG in one channel
                    //     and the combined NMEA/PPS signal in the other channel :
                    //  At HH:MM:59.0, there's the leading PPS edge, and NO AMPLITUDE DROP on DCF77 !
                    //  Details in file:///C:/cbproj/SpecLab/html/frqcalib.htm#checking_the_GPS_NMEA_timing .
                  }
                 // Already set (even without NMEA output) :
                 // pGPSdet->dblAveragedSampleRate = xyz;
                 // pGPSdet->i64TSC_at_last_valid_sync_pulse
               } // end if( GpsDet_DecodeSerialData(..)
              else
               {
                 if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
                  { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
                    GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/,
                      "PulseDet: Could not decode NMEA" );
                  }
               }
            } // end if( (i>=0) && ((i+n)<(int)pGPSdet->dwUsedInputWaveformLength) )
#         ifdef __BORLANDC__
           (void)fDecodedTime;
#         endif
         }
        else //  pGPSdet->t_NMEA_bit <= 0  :  Use date and time from the input 'audio chunk info',
         {   // *NOT* from the GPS receiver's NMEA output .
           int year,month,day,hour,minute;
           double second;
           long double date_and_time = pGPSdet->ldbDateAndTimeAtPPSEdges[2]; // ex: pChunkInfo->ldblUnixDateAndTime;
           // Round the 'input timestamp' (based on the PC's system time)
           // to the nearest full second, which is hopefully close enough
           // to the GPS sync pulse. This usually requires an internet time server.
           // See notes in html/frqcalib.htm#GPS_sync_without_NMEA !
           date_and_time += 0.5 * pGPSdet->t_sync_interval;
           date_and_time -= fmodl( date_and_time, pGPSdet->t_sync_interval );
           // If the PC's system time is synchronized via internet,
           //  date_and_time should only differ by a few ten milliseconds
           //  from pGPSdet->ldbDateAndTimeAtPPSEdges[2] . Test:
           second = date_and_time - pGPSdet->ldbDateAndTimeAtPPSEdges[2];
           pGPSdet->ldblTime_from_last_valid_NMEA = date_and_time;
           UTL_SplitUnixDateTimeToYMDhms(
               date_and_time,  // [in] date and time in Unix format
               &year, &month, &day, &hour, &minute, &second ); // [out]
           pGPSdet->pGpsPosInfo->year   = year;
           pGPSdet->pGpsPosInfo->month  = month;
           pGPSdet->pGpsPosInfo->day    = day;
           pGPSdet->pGpsPosInfo->hour   = hour;
           pGPSdet->pGpsPosInfo->minute = minute;
           pGPSdet->pGpsPosInfo->second = second;
           pGPSdet->fChunkInfo_DateAndTimeFromGPSValid
             = pGPSdet->pGpsPosInfo->time_valid
             = pGPSdet->pGpsPosInfo->date_valid = TRUE;

           // Geographic latidude and longitude is meaningless in this case;
           // the application (Spectrum Lab) may use 'configurable dummies' instead..
           pGPSdet->dblChunkInfo_Lat_deg = C_CHUNK_INFO_LAT_INVALID;
           pGPSdet->dblChunkInfo_Lon_deg = C_CHUNK_INFO_LON_INVALID;
           pGPSdet->dblChunkInfo_mASL   = 0.0;
           pGPSdet->dblChunkInfo_Vel_kmh = 0.0;

         } // end if( pGPSdet->t_NMEA_bit > 0 ) ?

     //-----------------------------------------------------------------------
     // ABOVE: Decode NMEA (GPS nav data); may (or may not) contain date+time
     // BELOW: Precise sync-timing analysis (PPS), with high-accuracy edge detection.
     //-----------------------------------------------------------------------


        // Analyse the GPS sync pulse (when it's time to... a few dozen samples AFTER the leading edge)
        fPulseOk = FALSE;
        fWasOk   = pGPSdet->fChunkInfo_DateAndTimeFromGPSValid;
        if( pGPSdet->iSampleIndexOfPPS[2] >= 0 )
         {
           // Perform the interpolation..
           fPulseOk = GpsDet_InterpolateAndProcessSyncPulse( pGPSdet );
           //   [in]  pGPSdet->pfltSyncdInputWaveform[],
           //         pGPSdet->i64TSC_FirstEntryInSyncdWaveform
           //         pGPSdet->ldblGpsTime_FirstEntryInSyncdWaveform, ....
           //   [out] pGPSdet->i64TSC_at_last_valid_sync_pulse, [sample counter]
           //         pGPSdet->ldblTime_at_last_valid_sync_pulse  [Unix timestamp for the above sample index]
         } // end if < enough samples 'around' the rising PPS edge for interpolation >
        if( fWasOk && !fPulseOk ) // there doesn't seem to be a PPS signal anymore:
         { // ToDo: Reset the coarse detection, and switch to free-running mode ?
           strcpy( pGPSdet->sz80Info1, "Sync-pulse missing or discarded" );
           if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
            { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
              GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/,
                "PulseDet: %s", pGPSdet->sz80Info1 );
            }
         }
        if( fPulseOk && !fWasOk ) // valid PPS signal just "returned" :
         { strcpy( pGPSdet->sz80Info1, "Sync-pulse valid" );
           if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
            { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
              GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/,
                "PulseDet: %s", pGPSdet->sz80Info1 );
            }
         }
        // Finish acquisition of the PPS- (and optionally NMEA-) cycle:
        n = pGPSdet->dwSyncdWaveformSampleIndex; // -> number of samples collected for approx. 1.1 (!) seconds
        // No-No: pGPSdet->dwSyncdWaveformSampleIndex = 0;
        if(  (DWORD)n > pGPSdet->dwAllocdInputWaveformLength )
         {   (DWORD)n = pGPSdet->dwAllocdInputWaveformLength; // safety first!
           // (should never be necessary because the allocated waveform
           //  is about 10 % than for the 'nominal' sampling rate )
           if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
            { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
              GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/,
                "PulseDet: Sync'd waveform buffer too small !" );
            }
         }
        if(  n < (((int)pGPSdet->dwAllocdInputWaveformLength*9)/10) )
         {
           if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
            { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
              GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/,
                "PulseDet: Unexpected low number of samples (%d) in sync'd waveform buffer" );
            }
           n = n;  // << shouldn't happen - set breakpoint here!
           // Got here with Freddy's "Jupiter" sample (48kS/s)
           // with n=3652, and pAudioChunk[iSample-4] .. [iSample+4] =
           //  -3561, -3902, -1870, -22002, -32767, -31593, -32768, -32523, -32696 .
           GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/,
             "PulseDet: Unexpected low number of samples (%d) in sync'd waveform buffer" );
         }
        pGPSdet->dwUsedInputWaveformLength = n;    // limited value

        ++pGPSdet->iUpdateCount;  // polled in the GUI to update the display
      } // end if( pGPSdet->iCountdownUntilFrameProcessing <= 0 )

   } // end for < all samples in the chunk >

  DOBINI( 0 );

  // Set the timestamp, momentary sample rate, and SR drift rate
  //      in the OUTPUT T_ChunkInfo structure ?
  //   Note: At this point, i64TSC (Total Sample Counter) still applies
  //         to the FIRST sample in the chunk !
  //         pGPSdet->i64TSC += pChunkInfo->dwNrOfSamplePoints  follows FURTHER BELOW !

  fSetTimeInChunkInfo = FALSE;
  if( pGPSdet->dblAveragedSampleRate > 0 )
   {
     // Inputs:
     // *  pGPSdet->ldblTime_at_last_valid_sync_pulse
     //    = Time decoded from NMEA, which applies to the "last valid" sync-pulse,
     // * pGPSdet->i64TSC_at_last_valid_sync_pulse
     //    = reconstructed global sample counter at the last VALID sync pulse
     //      (which may be MORE THAN JUST ONE SECOND away),
     f_sample = pGPSdet->dblAveragedSampleRate;
     // f_sample = 32099.851778;    // TEST: constant sample rate -> there shouldn't be any phase jumps now
     dt = ((double)( pGPSdet->i64TSC/*Total Sample Counter (input samples from ADC)*/
                   - pGPSdet->i64TSC_at_last_valid_sync_pulse )
                   / f_sample );
     // dt = number of seconds between the last properly decoded GPS timestamp
     //      (valid at the sync pulse) and the sample in pAudioChunk[0].
     //      Should be positive, because the last sync is OLDER than the most recent sample from the ADC.
     //      Be tolerant: 'iron out' up to 10(?) missing sync pulses..
     if( (dt>=-1.0) && (dt<=2.5 + pGPSdet->t_sync_interval) )
      {
        pChunkInfo->dblPrecSamplingRate = f_sample;

        if( pGPSdet->fChunkInfo_DateAndTimeFromGPSValid )
         {
           pGPSdet->ldblUnixDateAndTimeForChunkInfo = pGPSdet->ldblTime_at_last_valid_sync_pulse + (long double)dt;
           pGPSdet->dblSystemTimeMinusTimeFromGPS = pChunkInfo->ldblUnixDateAndTime - pGPSdet->ldblUnixDateAndTimeForChunkInfo;
           //  pGPSdet->dblSystemTimeMinusTimeFromGPS was used just for *display* in SRCalibCtrl.cpp,
           //  but it can OPTIONALLY be applied in SpecMain.cpp : TSpectrumLab::Timer1Timer()
           //  to CORRECT THE PC's SYSTEM TIME IN UTC as a replacement for the
           //  non-functional, or poor-performing windows time sync service.
           //  See more notes THERE (near the call of TIM_SetDateAndTimeInUTC())
           //      for details about the SYSTEM TIME SYNCHRONISATION,
           //      and about the role that dblSystemTimeMinusTimeFromGPS plays,
           //      and how it is replaced in configurations WITHOUT a "sync pulse".
           //  With a proper sync pulse from the GPS and NTP with 11 ms 'round trip',
           //  dblSystemTimeMinusTimeFromGPS (displayed as "System minus GPS" in
           //  SRCalibCtrl.cpp and GPSReceiverGUI.cpp) was always below 100 ms .
           //  Since 2016-05, the above difference can also be used to
           //  synchronize the PC's system time to the GPS time we decoded here.
           //  SetSystemTime() may be called from SpecMain.cpp occasionally
           //  to eliminate the above difference, as far as the "OS" permits.
           pChunkInfo->ldblUnixDateAndTime = pGPSdet->ldblUnixDateAndTimeForChunkInfo;
           fSetTimeInChunkInfo = TRUE;

           // Declare the timestamps (in the T_ChunkInfo struct) valid:
           pChunkInfo->dwValidityFlags |=
                 ( CHUNK_INFO_TIMESTAMPS_VALID   // here: timestamps derived from GPS sync pulse..
                 | CHUNK_INFO_TIMESTAMPS_PRECISE // .. so their accuracy should be 'better than one audio sample period' !
                 | CHUNK_INFO_SAMPLE_RATE_VALID
                 | CHUNK_INFO_SAMPLE_RATE_CALIBRATED );

           // To suppress certain warnings shortly after start ('before settling'):
           if(  ( pGPSdet->dblGoodTimestampsSince >  pGPSdet->ldblUnixDateAndTimeForChunkInfo )
              ||( pGPSdet->dblGoodTimestampsSince < (pGPSdet->ldblUnixDateAndTimeForChunkInfo-61.0) ) )
            { pGPSdet->dblGoodTimestampsSince = pGPSdet->ldblUnixDateAndTimeForChunkInfo;
            }
           if( pGPSdet->dblGoodTimestampsSince < (pGPSdet->ldblUnixDateAndTimeForChunkInfo-60.0) )
            { pGPSdet->dblGoodTimestampsSince = pGPSdet->ldblUnixDateAndTimeForChunkInfo-60.0;
            }
         } // end if( pGPSdet->fChunkInfo_DateAndTimeFromGPSValid )
        else
         {
      //   if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
      //    { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
      //      GpsPulseDet_DebugPrintf( pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/,
      //          "PulseDet: Invalid date and time from GPS" );
      //    }
         }
      } // end if < delta t between audio sample timestamp and GPS time reasonably small >
     else // last valid sync-pulse is 'too far off' !
      { // (usually a subsequent error, the real reason may be
        //  something like "Measured SAMPLERATE too far off" ! )
        dt = dt; // <<< set breakpoint here !
        if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
         { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
           GpsPulseDet_DebugPrintf( pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/,
             "PulseDet: Sync pulse ignored (dt=%.3lf)", (double)dt );
         }

        if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
         { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
           if( pGPSdet->samplerate_too_far_off )
            { sprintf( pGPSdet->sz255StatusText, "Samplerate too far off (%.2lf ppm) !",
                   (double)1e6 * (pGPSdet->dblCurrentMeasuredSampleRate * pGPSdet->t_sample - 1.0 ) ); // -> ~ppm
            }
           else
            { sprintf( pGPSdet->sz255StatusText, "Last sync too old (dt=%.2lf s) !", (double)dt );
            }
           ++pGPSdet->iStatusTextUpdateCount;
#         ifdef CIRCUIT_STATE_WARNING  /* from CircuitDefs.h, optional */
           pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
#         endif
           // Emit the above warning also to the error history / debug messages ?
           dt = pGPSdet->ldblUnixDateAndTimeForChunkInfo - pGPSdet->dblGoodTimestampsSince; // running "long enough" on "good timestamps" ?
           //  After starting, pTR->dblGoodTimestampsSince = 0.0 ->
           //  dt is a very LARGE number, which means "NO good timestamps yet" !
           if( dt>=10.0 && dt<100.0 ) // suppress warning "when no good timestamps" yet (10 seconds to settle after start)
            { GpsPulseDet_DebugPrintf( pGPSdet->ldblUnixDateAndTimeForChunkInfo/*?*/, pGPSdet->sz255StatusText );
            }
         }
      } // end if < last successful NMEA decode sufficiently 'new' >
   } // end if < 'precise' sample rate is known >

  pGPSdet->i64TSC += pChunkInfo->dwNrOfSamplePoints;

  if( ! fSetTimeInChunkInfo ) // oops.. temporarily switch to "free running" timestamps
   { pChunkInfo->ldblUnixDateAndTime = pGPSdet->ldblUnixDateAndTimeForChunkInfo;
   }
  pGPSdet->ldblUnixDateAndTimeForChunkInfo += (long double)pChunkInfo->dwNrOfSamplePoints
                                            / (long double)pGPSdet->dblAveragedSampleRate;


  if( pGPSdet->iSoftwareTestCommand==GPSDET_TEST_DUMP_INTERNAL_VARS )
   {  GpsDet_DumpInternalVariables( pGPSdet );
      pGPSdet->iSoftwareTestCommand=GPSDET_TEST_DONE;
   }

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Self-test : Are emitted consecutive timestamps plausible ?
  if (pChunkInfo->dblPrecSamplingRate > 0 ) // avoid div-by-zero..
   {
     if( pGPSdet->ldblNextExpectedOutputTimestamp>0 )
      { d = pChunkInfo->ldblUnixDateAndTime - pGPSdet->ldblNextExpectedOutputTimestamp;
        // d should be close to ZERO at this point !
        // 2016-10-30 : Got here with d = 1, i.e. "one second missing"
        //              in pGPSdet->ldblNextExpectedOutputTimestamp
        //              Fixed the crude way (to avoid the error messages):
        if( d > 0.999 )
         {  d -= 1.0;
         }
        if( d < -0.999 )
         {  d += 1.0;
         }

        d *= pChunkInfo->dblPrecSamplingRate; // -> d = number of SAMPLES (!)
        // Similar test criteria in GpsPulseDetector.cpp and TimedResampler.c :
        if( (d < -0.25) || (d > 0.25) )  // 'd' is a FRACTIONAL NUMBER OF SAMPLES !
         { // Relatively "bad" timestamp (too much jitter or noise ?)
           // 2015-01-29 : Got here when a single sample was missed (per GPSDET_TEST_SIMULATE_LOST_SAMPLE)
           //              with d = 1.000, q.e.d.
           // 2016-10-19 : Got here with d = the approximate sampling rate,
           //              looks like a GPS sync pulse was missed for reasons yet-to-be discovered.
           // 2016-10-24 : When the GPS sync pulses were 'emulated' with SL's test signal generator,
           //              d (=jitter measured in SAMPLES) was sometimes almost 1.0 ,
           //              caused by the crude implementation of the square wave generator
           //              which could only place the pulse edge at integer multiples of sample intervals.
           dt = pGPSdet->ldblUnixDateAndTimeForChunkInfo - pGPSdet->dblGoodTimestampsSince; // running "long enough" on "good timestamps" ?
           //  After starting, dblGoodTimestampsSince = 0.0 ->
           //  dt is a very LARGE number, which means "NO good timestamps yet" !
           if( dt>=5.0 && dt<100.0 ) // suppress warning shortly after start (without "good timestamps")
            { if(  pGPSdet->nBadOutputTimestamps >= 0 )   // count "bad" timestamps ? [after there was at least one "good" timestamp]
               { ++pGPSdet->nBadOutputTimestamps;
                 GpsPulseDet_DebugPrintf( pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/,
                   "PulseDet: bad TS #%d, d=%.3lf samples", (int)pGPSdet->nBadOutputTimestamps, (double)d );
               }
            }
           pGPSdet->dblGoodTimestampsSince = 0.0; // "no good timestamps now" (because we just emitted a BAD one)
           if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
            { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
              sprintf( pGPSdet->sz255StatusText,
                 "PulseDet: bad TS, d=%.3lf samples", (double)d );
              ++pGPSdet->iStatusTextUpdateCount;
            } // end if < running with "good timestamps" since at least 10 seconds ? >
#         ifdef CIRCUIT_STATE_WARNING  /* from CircuitDefs.h, optional */
           pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
#         endif
         } // end if < obviously a TIMING BREAK (bad timestamp)>
        else // "good timestamp" : enable counting BAD TIMESTAMPS from now on
         { if( pGPSdet->nBadOutputTimestamps < 0 )
            {  pGPSdet->nBadOutputTimestamps = 0;  // enable start counting BAD timestamps (!)
            }
         }
      }
     pGPSdet->ldblNextExpectedOutputTimestamp = pChunkInfo->ldblUnixDateAndTime
           +   (long double)pChunkInfo->dwNrOfSamplePoints
             / (long double)pChunkInfo->dblPrecSamplingRate;
   } // end if (pChunkInfo->dblPrecSamplingRate > 0 )
  else
   { GpsPulseDet_DebugPrintf( pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/,
       "PulseDet: Illegal SR, %.3lf Hz", (double)pChunkInfo->dblPrecSamplingRate );
   }
  // - - - - - - - - - - - - end Self-test - - - - - - - - - - - - - - - - -


  return fPulseOk;

} // end GPS_Det_ProcessSamples()


//---------------------------------------------------------------------------
void GpsDet_RemoveOldEntriesInSyncdInputWaveform( T_GPS_Detector *  pGPSdet )
   // Removes the oldest 5 percent of the 'pulse-synchronized input waveform' ;
   // we don't need them anymore. The buffer is 10 % longer than the 1-second cycle.
{
  int i,n = pGPSdet->dwAllocdInputWaveformLength / 20;  // number of samples to "throw out"
  T_Float *pFltSrc = &pGPSdet->pfltSyncdInputWaveform[n];
  T_Float *pFltDst = &pGPSdet->pfltSyncdInputWaveform[0];

  pGPSdet->dwSyncdWaveformSampleIndex -= n;
  for(i=0; i<(int)pGPSdet->dwSyncdWaveformSampleIndex; ++i)
   { *pFltDst++ = *pFltSrc++;
   }
  // Adjust all members of T_GPS_Detector which are contain INDICES
  //        into pfltSyncdInputWaveform[]:
  pGPSdet->i64SyncdInputWaveformSampleIndex += n; // offset for an 'absolute' sample counter
  // (hasn't got anything to do with the 'global' sample counter in .i64TSC (*) )
  for(i=0; i<=2; ++i )
   { pGPSdet->iSampleIndexOfPPS[i] -= n;
     if( pGPSdet->iSampleIndexOfPPS[i] < 0 ) // sample index not valid anymore ->
      {  pGPSdet->iSampleIndexOfPPS[i] = -1; // avoid integer overflow, just keep index negative (=invalid)
      }
   }
} // end GpsDet_RemoveOldEntriesInSyncdInputWaveform()

//---------------------------------------------------------------------------
void GpsDet_UpdateSyncPulseThresholds( T_GPS_Detector *pGPSdet, float fltCurrSample, double tPeakDet )
{
  T_Float fltRequiredPulseAmplitude = 0.15; // .. 1.0 would be "unipolar full-swing" (which would mean CLIPPING)

  pGPSdet->fltPpsNegativeThreshold = 0.9 * pGPSdet->fltNewPeakMin; // must not be too low, to avoid triggering on NMEA !
  pGPSdet->fltPpsPositiveThreshold = 0.9 * pGPSdet->fltNewPeakMax;

  if( (pGPSdet->dblPulseWidth_sec>0.0) && (pGPSdet->dblPulseWidth_sec<(4.0 * pGPSdet->t_sample) ) )
   { // extremely short sync pulse, will not reach the true amplitude due to low-pass filtering, so:
     fltRequiredPulseAmplitude = 0.05; // a 10us, 50% ampl pulse only reached 15% when digitized on an E-MU0202 with 48kS/sec,
     // and the amplitude varied greatly, depending on the relative position of the pulse within t_sample !
     // See multiple superimposed pulses in C:/cbproj/SpecLab/screenshots/10us_pulse_EMU0202_48kS.png .
     // The max amplitude (of a needle-like pulse) was ~19%, the lowest (trapezoid) was ~11% . Thus:
     pGPSdet->fltPpsNegativeThreshold = 0.5 * pGPSdet->fltNewPeakMin; // in this case, NMEA must be "much weaker" !
     pGPSdet->fltPpsPositiveThreshold = 0.5 * pGPSdet->fltNewPeakMax; // (if NMEA is possible on the same wire at all)
   }

  pGPSdet->fltNewPeakMin = pGPSdet->fltNewPeakMax = fltCurrSample;
  pGPSdet->dblPeakDetTimer -= tPeakDet;  // 'restart' for the next 2(?) seconds
  if( (pGPSdet->fltPpsPositivePeak>0.9)||(pGPSdet->fltPpsNegativePeak<-0.9) ) // dangerously close to the clipping point (1.0) ?
   { // peak-levels (of PPS, most likely) are CRITICALLY HIGH but still valid ..
     if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
      { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        strcpy( pGPSdet->sz255StatusText, "PPS level too high" );
        ++pGPSdet->iStatusTextUpdateCount;
#      ifdef CIRCUIT_STATE_ACTIVE  /* from CircuitDefs.h, optional */
        pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
#      endif
      }
     pGPSdet->peak_levels_valid = TRUE;
   }
  else
  if( (pGPSdet->fltPpsPositiveThreshold - pGPSdet->fltPpsNegativeThreshold ) >  fltRequiredPulseAmplitude/*ex:0.15*/ )
   { // peak-levels (of PPS, most likely) are NOW valid ..
     if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
      { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        strcpy( pGPSdet->sz255StatusText, "PPS peaks ok" );
        ++pGPSdet->iStatusTextUpdateCount;
#      ifdef CIRCUIT_STATE_ACTIVE  /* from CircuitDefs.h, optional */
        pGPSdet->iCircuitState = CIRCUIT_STATE_ACTIVE;
#      endif
      }
     pGPSdet->peak_levels_valid = TRUE;
   }
  else // peak-levels are now INVALID !
   { if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
      { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        strcpy( pGPSdet->sz255StatusText, "Sync pulses too weak !" );
        ++pGPSdet->iStatusTextUpdateCount;
#      ifdef CIRCUIT_STATE_WARNING  /* from CircuitDefs.h, optional */
        pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
#      endif
        if(  GpsPulseDet_bNumErrorsShownInLine[__LINE__] < 10 )
         { ++GpsPulseDet_bNumErrorsShownInLine[__LINE__-1];
           GpsPulseDet_DebugPrintf( pGPSdet->dblUnixDateAndTimeFromChunkInfo/*?*/, pGPSdet->sz255StatusText );
         }
      }
     pGPSdet->peak_levels_valid = FALSE;
   }
} // end GpsDet_UpdateSyncPulseThresholds()


//---------------------------------------------------------------------------
BOOL GpsDet_InterpolateAndProcessSyncPulse( T_GPS_Detector * pGPSdet )
  // Performs the PPS-INTERPOLATION .
  // Called a few samples after the (coarsely detected) leading edge of the sync pulse.
  // Returns TRUE when the pulse appears "ok", otherwise FALSE.
  //
  //   [in]  pGPSdet->pfltSyncdInputWaveform[],
  //                  pGPSdet->iSampleIndexOfPPS[2] ("coarsely detected"),
  //                  i64TSC_FirstEntryInSyncdWaveform
  //                  ldblGpsTime_FirstEntryInSyncdWaveform, ....
  //   [out] pGPSdet->i64TSC_at_last_valid_sync_pulse, [sample counter at the last (latest) valid sync pulse, for which a valid GPS-time was decoded from NMEA]
  //                  ldblTime_at_last_valid_sync_pulse  [GPS time in Unix format for the above point]
  //
  // Fasten seat belt, this code is ugly due to its 'grown structure' !
{
  int     i,j,k;
  long     i32;
  LONGLONG i64;
  double  d, deltaI, age_sec;
  double  t_sample = pGPSdet->t_sample;
  char   *cp;
  T_Float factor;
  T_Float sample, *pflt, dc_offset;

  BOOL fPulseOk = FALSE;

  // pGPSdet->pfltSyncdInputWaveform[ pGPSdet->iSampleIndexOfPPS[2] ]
  //    may be ONE SAMPLE AHEAD or BEHIND the 'true' PPS edge,
  //    but the rising edge shall be approximately centered
  //    in pGPSdet->fltInputPulse[0..C_GPSDET_FIFO_LEN_BRUTTO-1] .
#   if ( 0 )  // TEST 2011-12-30 :
              //   Overwrite the input with an 'ideally centered'
              //   rising edge, which should give deltaT = 0 later (*) :
              //
              i = C_GPSDET_FIFO_LEN_BRUTTO; // -> 516 (an EVEN array size) !
              //   thus the 'ideal step' below is symmetrical,
              //   and the 'center of the step' falls right in the middle
              //   between two simulated samples from the soundcard. That's ok;
              //   but remember 'deltaI' will be at sample-index 0.5, not 0.0 !
              for(i=-C_GPSDET_FIFO_LEN_BRUTTO/2; i<=C_GPSDET_FIFO_LEN_BRUTTO/2; ++i)
               { pGPSdet->pfltSyncdInputWaveform[ i + pGPSdet->iSampleIndexOfPPS[2] ]
                          = (i<0) ? 0.0 : 1.0 ;
                 // The negative 'i' is is safe because pGPSdet->iSampleIndexOfPPS[2]
                 //  is 'in the middle' of pGPSdet->pfltSyncdInputWaveform[]
               }
              pGPSdet->pps_polarity_inverted = FALSE; // !
              //
              // Test-waveform in pGPSdet->pfltSyncdInputWaveform[i] now :
              //
              //                  ____________ 1
              //                 |
              //                 | ........... 0.5
              //   0 ____________|
              //
              //                 ^
              //           |     |___ sample at i =  pGPSdet->iSampleIndexOfPPS[2] already '1' !
              //           |
              //           j = 'array index of the first 'interesting' sample' further below .
              //
              //    For the *LONG* pulse detection the center of the slope (0.5),
              //      after interpolation, should be close to
              //      pfltSyncdInputWaveform[ iSampleIndexOfPPS[2] - 0.5 ],
              //                                                     !!!
              //
              // pGPSdet->pfltSyncdInputWaveform[ pGPSdet->iSampleIndexOfPPS[2]-1] = 0.0
              // pGPSdet->pfltSyncdInputWaveform[ pGPSdet->iSampleIndexOfPPS[2]  ] = 1.0
              // pGPSdet->pfltSyncdInputWaveform[ pGPSdet->iSampleIndexOfPPS[2]+1] = 1.0
              //
              // With the above test pattern, after GPS_Det_ProcessInterpolatedStep() :
              //    pGPSdet->dblInterpolatedStepZeroCrossingIndex = 30.000000899 .
              //    pGPSdet->fltInterpolatedStep[29] = -0.2805521 --
              //    pGPSdet->fltInterpolatedStep[30] = -0.0000002   |-- Rounding
              //    pGPSdet->fltInterpolatedStep[31] = +0.2805514 --    error ?
              // Check 2011-12-30 11:00 :  OK .
#   endif // TEST with synthetic data, 2011-12-30 11:00

  j = // source array index of the first 'interesting' sample in pGPSdet->pfltSyncdInputWaveform[] :
      (int)pGPSdet->iSampleIndexOfPPS[2] - C_GPSDET_FIFO_LEN_BRUTTO/2;
  if( (j>=0) && (j<((int)pGPSdet->dwSyncdWaveformSampleIndex-C_GPSDET_FIFO_LEN_BRUTTO/*ca 504*/)) )
   {
     factor = C_GPSDET_INTERPOLATION_RATIO; // 4.0;
     if ( pGPSdet->pps_polarity_inverted )
      { factor = -factor;
      }
     dc_offset = 0.0;
     for(i=0; i<C_GPSDET_FIFO_LEN_BRUTTO; ++i)
      { sample = pGPSdet->pfltSyncdInputWaveform[ j+i ];
        pGPSdet->fltInputPulse[i] = sample;  // save for the 'raw' scope display
        dc_offset += sample;
      }
     dc_offset /= C_GPSDET_FIFO_LEN_BRUTTO;

     // Update the *APPROXIMATE* timestamp of the sample in pGPSdet->fltInputPulse[0]:
     pGPSdet->ldblDateAndTimeAtInputPulse_ArrayIndexZero =
      pGPSdet->ldblGpsTime_FirstEntryInSyncdWaveform
         + (double)j / pGPSdet->dblAveragedSampleRate;

#   if ( 0 )  // 2011-12-30: Check the extra delay caused by zero-stuffing for interpolation.
              //            Fill the 'input pulse' with another step function:
              for(i=0; i<C_GPSDET_FIFO_LEN_BRUTTO; ++i)
               { pGPSdet->fltInputPulse[i] =  // DC-free 'ideal step' before zero-stuffing
                    (i<C_GPSDET_FIFO_LEN_BRUTTO/2) ? -1.0 : 1.0 ;
               }
              factor = C_GPSDET_INTERPOLATION_RATIO; // 4.0;
              dc_offset = 0.0;  // there's no DC offset in the above 'ideal' input.
              // At this point: i=C_GPSDET_FIFO_LEN_BRUTTO = 516 (!)
              //  pGPSdet->fltInputPulse[ 0 ..257] = -1
              //  pGPSdet->fltInputPulse[258..515] =  1  .
              // With THIS test pattern, after GpsDet_InterpolateSyncPulseEdge() :
              //   pGPSdet->fltInterpolatedStep[27] = -1.239
              //   pGPSdet->fltInterpolatedStep[28] = -1.000
              //   pGPSdet->fltInterpolatedStep[29] = -0.561
              //   pGPSdet->fltInterpolatedStep[30] =  0.000  <<< zero crossing (*)
              //   pGPSdet->fltInterpolatedStep[31] =  0.561
              //   pGPSdet->fltInterpolatedStep[28] =  1.000
              //   pGPSdet->fltInterpolatedStep[32] =  1.239
              // (*) GPS_Det_ProcessInterpolatedStep() detected the zero-crossing
              //     at pGPSdet->dblInterpolatedStepZeroCrossingIndex = 30.000000899 ;
              //     which confirms that by appending THREE zeroes per sample (in the code further below)
              //     reduces dblInterpolatedStepZeroCrossingIndex by 3/2 ,
              //     3/2 = 0.5 * (C_GPSDET_INTERPOLATION_RATIO-1) [interpolated samples] .
              //     To compensate this, 0.5 * (C_GPSDET_INTERPOLATION_RATIO-1)
              //     will be added to pGPSdet->dblInterpolatedStepZeroCrossingIndex
              //     somewhere below ("compensate zero-stuffing, 2011-12-30") .
#   endif // TEST to find the group delay caused by interpolation (FIR lowpass)


     // Now scale and copy the 'real' data, with three padded zeroes per input sample
     // to interpolate the input sampling rate by FOUR (or more ?) .
     // On this occasion, also remove most of the DC offset to simplify
     //        the detection of the zero-crossing (for "long pulse edge detection")
     //        or the detection of the centroid (for "short pulse center detection").
     pflt = pGPSdet->fltInterpolationBuffer;
     // Note: The timestamp of the sample in pGPSdet->fltInterpolationBuffer[0]
     //       is the same as for the sample in pGPSdet->fltInputPulse[0],
     //       which is pGPSdet->ldblDateAndTimeAtInputPulse_ArrayIndexZero .
     for(i=0; i<C_GPSDET_FIFO_LEN_BRUTTO; ++i)
      { *pflt++ =  // samples 'near the rising pulse edge', normalized to 0 .. 1 :
                factor * (pGPSdet->fltInputPulse[i] - dc_offset);
        for( k=0; k<(C_GPSDET_INTERPOLATION_RATIO-1); ++k)
         { *pflt++ = 0.0;   // for example, stuff in THREE zeroes per sample,
           // to interpolate by four. The windowed sinc lowpass does the rest.
           // The result, at this stage, looks terrible on the scope but that's ok.
         }
      }

     // How many ADC samples have passed between 'this'
     //  and the previous call of GpsDet_InterpolateSyncPulseEdge() ?
     i32 = /* -> nAudioSamplesBetweenThisAndPreviousCall, usually close to the SAMPLING RATE */
       (long)(  pGPSdet->i64SyncdInputWaveformSampleIndex
             + (LONGLONG)pGPSdet->iSampleIndexOfPPS[2]
             - pGPSdet->i64PreviousSampleIndexOfPPS );
     pGPSdet->i64PreviousSampleIndexOfPPS = pGPSdet->i64SyncdInputWaveformSampleIndex
             + (LONGLONG)pGPSdet->iSampleIndexOfPPS[2];
     // Wenn beim aktuellen Durchlauf 100 Samples MEHR in pfltSyncdInputWaveform[]
     // vorhanden sind als beim vorherigen Durchlauf, dann ist z.B. (bei fs=192 kHz) :
     //   pGPSdet->iSampleIndexOfPPS[2] = 189574 (Index innerhalb pfltSyncdInputWaveform[])
     // beim vorherigen Durchlauf aber (da WENIGER Samples im Puffer waren)
     //   pGPSdet->iSampleIndexOfPPS[2] = 189474 .

#   define TEST_INTERPOLATOR_WITH_IDEAL_STEP 0  /* 0 = normal compilation, 1 = TEST */
#   if (   TEST_INTERPOLATOR_WITH_IDEAL_STEP )
     // TEST 2011-12-29: Find out the group delay introduced by the interpolator (it's zero).
     // Fill the input for GpsDet_InterpolateSyncPulseEdge() with an ideal step function:
     for(i=0; i<C_GPSDET_INTERPOLATION_BUFFER_LENGTH; ++i)
      { pGPSdet->fltInterpolationBuffer[i] = (i<C_GPSDET_INTERPOLATION_BUFFER_LENGTH/2) ? -1.0 : 1.0 ;
      }
     // At this point: i=C_GPSDET_INTERPOLATION_BUFFER_LENGTH = 2064 (!)
     //  pGPSdet->fltInterpolationBuffer[ 0   .. 1031] = -1
     //  pGPSdet->fltInterpolationBuffer[1032 .. 2063] =  1
     // Note: The above array has an EVEN length, so the zero-crossing
     //       is in the middle between index 1031 and 1032 (i.e. "1031.5"),
     //       which explains the "strange" value of deltaI (-0.125) further below.
     // Result AFTER CALLING GpsDet_InterpolateSyncPulseEdge() AND GPS_Det_ProcessInterpolatedStep() :
     //  pGPSdet->dblInterpolatedStepZeroCrossingIndex = 31.49999857 . Ok, because:
     //  The 'center array index' in pGPSdet->fltInterpolatedStep[0..63]
     //  would ideally be 63/2=31.5 (check: 0,1,2,3 -> center index = 1.5 = 3/2);
     //  this interpolation has an immediate effect on the accuracy of the
     //  timestamp detection, and thus on the measurement of the sampling rate !
#   endif // TEST_INTERPOLATOR_WITH_IDEAL_STEP ?

     GpsDet_InterpolateSyncPulseEdge( pGPSdet );  // interpolation with a 'long' FIR lowpass
        // [in]  pGPSdet->fltInterpolationBuffer[C_GPSDET_INTERPOLATION_BUFFER_LENGTH],
        //                ldblDateAndTimeAtInputPulse_ArrayIndexZero .
        // [out] pGPSdet->fltInterpolatedStep[C_GPSDET_INTERPOLATED_STEP_LENGTH],
        //                ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero .

     // For 'long' pulses: Ideally, the interpolated, DC-free signal crosses zero in the middle
     //         of pGPSdet->fltInterpolatedStep[C_GPSDET_INTERPOLATED_STEP_LENGTH] .
     //         (it does, confirmed by TEST_INTERPOLATOR_WITH_IDEAL_STEP)
     // C_GPSDET_INTERPOLATED_STEP_LENGTH = (C_GPSDET_FIFO_LEN_NETTO*C_GPSDET_INTERPOLATION_RATIO) = 16 * 4 .
     // 2011-12-29: With the 'very ideal' input from above, the INTERPOLATED step was:
     //  pGPSdet->fltInterpolatedStep[30..33] = -0.70, -0.25, +0.26, +0.70 ;
     //                        zero crossing in the center __|__ at index 31.5 .
     //  i.e. the interpolated step was 'centered' in the array as it should.
     //
     // Locate the precise position of the sync pulse in the interpolated waveform:
     //  [in]  pGPSdet->fltInterpolatedStep[0...C_GPSDET_INTERPOLATED_STEP_LENGTH-1],
     //                 ldblDateAndTimeAtInterpolatedStep_ArrayIndexZero .
     //  [out] pGPSdet->dblCurrentMeasuredSampleRate ,
     //        pGPSdet->dblInterpolatedStepZeroCrossingIndex , etc .
     if( GpsDet_IsShortSyncPulse( pGPSdet ) ) // decide which 'algorithm' to use, "short" or "long" pulses ?
      { // for "short" pulses, the interpolated waveform contains the entire pulse;
        //     there's no flat top here because the original GPS sync pulse
        //     may be shorter than a single sample interval from the soundcard !
        // See INTERPOLATED waveforms sketched in GPS_Det_ProcessInterpolatedShortPulse() .
        fPulseOk = GPS_Det_ProcessInterpolatedShortPulse( pGPSdet,
                i32/*nAudioSamplesBetweenThisAndPreviousCall*/ );
      }
     else // for "long" pulses, the interpolated waveform only contains the RISING edge (step):
      { fPulseOk = GPS_Det_ProcessInterpolatedStep( pGPSdet,
                i32/*nAudioSamplesBetweenThisAndPreviousCall*/ );
        // 2011-12-29: With the synthetic TEST_INTERPOLATOR_WITH_IDEAL_STEP,
        //  pGPSdet->dblInterpolatedStepZeroCrossingIndex was ~ 31.5 (=63/2) .
        //  In GPS_Det_ProcessInterpolatedStep() there was no difference
        //  between the linear and cubic interpolation with the 'ideal step'.
        // 2016-11-05: To compare different algorithms, even the "long" pulse processing
        //  may use centroid detection now - depending on m_fAlternativeAlgorithm .
      } // end else < "long" pulses, where only the significant rising edge is interpolated >

   } // end if( (j>=0) && (j<((int)pGPSdet->dwSyncdWaveformSampleIndex-C_GPSDET_FIFO_LEN_BRUTTO)) )


  if( fPulseOk ) // Convert the interpolated pulse 'location' [array index] into a timestamp.
   {
     // Determine the sample-index-difference 'deltaI' between
     //      the sample with the rising edge of the GPS sync pulse
     //                 in the 'interpolated' pulse waveform
     //      and the sample-index in .i64TSC (TSC=Total Sample Counter).
     // This is used to calculate the *precise timestamps* in the chunk headers later.
     deltaI = ( pGPSdet->dblInterpolatedStepZeroCrossingIndex
             - ( C_GPSDET_INTERPOLATED_STEP_LENGTH/*64*/ / 2) // offset for center of array
              )  / (double)C_GPSDET_INTERPOLATION_RATIO/*4*/;
     // The FRACTIONAL part of 'deltaI' (='sub-sample-index') cannot be
     // taken into account by adjusting i64TSC_at_last_valid_sync_pulse
     //  (because any 'Total Sample Counter' is not a floating-point number).
     // The following "fractional offset" (d) is only a few microseconds.
     //    It is taken into account in the emitted timestamps   (later)
     //    by slightly adjusting ldblTime_at_last_valid_sync_pulse (below):
     d = fmod( deltaI/*sample index*/, 1.0)      // -> "fractional offset" [as a time in seconds]
       / pGPSdet->dblCurrentMeasuredSampleRate;  //    taken into account by tweaking ldblTime_at_last_valid_sync_pulse

     //  With TEST_INTERPOLATOR_WITH_IDEAL_STEP,
     //       pGPSdet->dblInterpolatedStepZeroCrossingIndex = 31.49999857,
     //      the result 'deltaI' was
     //         ( 31.5 - (64/2) ) / 4 = -0.125 (!)
     //      at this point. This is ok, because the center (zero crossing)
     //         of the leading edge was between
     //         pfltSyncdInputWaveform[ iSampleIndexOfPPS[2] - 1 ]
     //      and
     //         pfltSyncdInputWaveform[ iSampleIndexOfPPS[2] ] .
     // Years ago, this was checked with the 'dirac'-test in SoundThd.cpp,
     //      based on the Unix-timestamp, which coincided
     //      with the center of the rising edge of the PPS signal.
     //
     // Summary: deltaI = non-integer "sample counter difference"
     //      between the sample with the rising edge of the GPS sync pulse
     //      and the sample-index in .i64TSC (TSC=Total Sample Counter).
     //          (calculated from the 'interpolated' pulse waveform at 4 * f_sample,
     //           but indexing applies to the SAMPLES FROM THE ADC, i.e. f_sample),



     // Determine TSC at the new "last VALID SYNC PULSE" :
     pGPSdet->i64TSC_at_last_valid_sync_pulse = // <- this is a SAMPLE COUNTER, not a time in seconds !
        pGPSdet->i64TSC_FirstEntryInSyncdWaveform
      + (LONGLONG)pGPSdet->iSampleIndexOfPPS[2]  /* index offset of 1st sample in fltInputPulse[] */
      + (LONGLONG)deltaI; /* offset from interpolation, truncated(!!) to integer */

     if( pGPSdet->t_NMEA_bit > 0 ) // > ZERO means "don't decode NMEA"...
      { // only if the NMEA decoder shall be active, too ... :

        // To reconstruct the "complete date and time" later,
        // take the possible "age" of the last successfully decoded NMEA-burst into account:
        age_sec = (double)(pGPSdet->i64TSC_at_last_valid_sync_pulse - pGPSdet->i64TSC_at_last_valid_NMEA)
          / pGPSdet->dblCurrentMeasuredSampleRate;
        // Example: "center" of the last SUCCESSFULLY DECODED burst 1.5 seconds older
        //          than the sync pulse currently being processed :
        //          -> age_sec =
        // Because the NMEA-burst may be "anywere between two sync pulses",
        //  only integer multiples of pGPSdet->t_sync_interval are important :
        age_sec -= fmod( age_sec, pGPSdet->t_sync_interval );
        // Example: -> age_sec = 1.0

        i64 = ( pGPSdet->ldblTime_from_last_valid_NMEA // <- always decoded (or at least 'emulated'), even without valid sync pulse
            + age_sec                            // "age", in seconds, between last successfully decoded NMEA and the current sync pulse
            + 0.5 * pGPSdet->t_sync_interval )   // <- round to the nearest sync-interval, which is...
          / pGPSdet->t_sync_interval; // ...almost always 1.0 [seconds per GPS-sync-pulse, AND NMEA-frame] (*)
      } // end if( pGPSdet->t_NMEA_bit > 0 )
     else  // for installations without NMEA (but GPS sync pulse only) :
      {
        i64 = ( pGPSdet->dblUnixDateAndTimeFromChunkInfo // <- actually the PC's current "system time", resolution 100(?)ms, accuracy must be better than 500 ms
            + 0.5 * pGPSdet->t_sync_interval )  // <- round to the nearest sync-interval, which is...
          / pGPSdet->t_sync_interval; // ...almost always 1.0 [seconds per GPS-sync-pulse, AND NMEA-frame] (*)
      }

     pGPSdet->ldblTime_at_last_valid_sync_pulse = // GPS date and time ("as Unix second") at the last "good" sync pulse
       (long double)pGPSdet->t_sync_interval * (long double)i64 // <- "integer part" [Unix time, but an integer multiple of the PPS-interval]
        - d; // <- fractional offset [few us], because the PPS edge wasn't centered in the ADC sampling time
     // Example : f_sample = 192 kHz (unit for deltaI, already divided
     //                      from the INTERPOLATOR'S sampling rate to the ADC sampling rate),
     //  d = +0.5 * 5.2us -> ldblTime_at_last_valid_sync_pulse falls "half into the NEXT sample"
     //                      (that's why 'd' is SUBTRACTED, not ADDED above)
     // If       pGPSdet->ldblTime_from_last_valid_NMEA is "long ago", the difference
     // between  pGPSdet->i64TSC and i64TSC_at_last_valid_NMEA would increase, thus the timestamps.
     //          (not a perfect solution if the sampling rate drifts,
     //           but if there are lots of non-decoded NMEA bursts, something else must be fixed)
     //
   } // end if( fPulseOk )
  else  // fPulseOk==FALSE ...
   { // Pulse detection NOT ok (or implausible 'delta t' somewhere) :
     // The caller (GPS_Det_ProcessSamples) can still calculate timestamps
     //   from pGPSdet->ldblTime_at_last_valid_sync_pulse
     //    and pGPSdet->i64TSC ("Total Sample Counter" for input from the ADC)
     //  minus pGPSdet->i64TSC_at_last_valid_sync_pulse !
   }

  // Diagnostic display (shown on the 'SR calibrator' control panel)
  sprintf( pGPSdet->sz80Info2, "SR= %6.3lf Hz; Mean%d=%8.5lf Hz",
                   (double)pGPSdet->dblCurrentMeasuredSampleRate,
                      (int)pGPSdet->iMeasuredSampleRateHistoryLength,
                   (double)pGPSdet->dblMeanSampleRateFromHistory );
  sprintf( pGPSdet->sz80Info3, "StdDev%d=%8.5lf Hz ~~ ",
                      (int)pGPSdet->iMeasuredSampleRateHistoryLength,
                   (double)pGPSdet->dblStdDevInSampleRateHistory );
         // Also shown in GpsDet_DumpInternalVariables(). Test results THERE.
  cp = pGPSdet->sz80Info3 + strlen( pGPSdet->sz80Info3 );
  UTL_FloatToTechNotation( cp, 20/*iMaxLen*/,
                    4/*digits*/ + UTL_FMT_OPTION_SPACE_BEFORE_UNIT,
                    SCALE_UNIT_SECONDS,
                    pGPSdet->dblStdDevInSampleRateHistory * t_sample );
  cp += strlen( cp );  strcpy(cp,"/second");

  sprintf( pGPSdet->sz80Info1, "GPS-Pulse: tH=%5.1lfms ampl: +%d%% %d%% NMEA: %d%%",
                   (double)(1e3 * pGPSdet->t_sync_pulse_length),
                      (int)(pGPSdet->fltPpsPositivePeak*100.0),
                      (int)(pGPSdet->fltPpsNegativePeak*100.0),
                      (int)(pGPSdet->Vpeak_nmea*100.0));
  sprintf( pGPSdet->sz80TimeInfo, "System - GPS time: %5.1lf ms", // added 2012-03-18
                   (double)(1e3 * pGPSdet->dblSystemTimeMinusTimeFromGPS) );
  return fPulseOk;
} // end GpsDet_InterpolateAndProcessSyncPulse()



//---------------------------------------------------------------------------
BOOL GpsDet_DecodeSerialData( T_GPS_Detector * pGPSdet,
                  T_Float *pFltSource,  // begin of the sample block (WITHOUT sync pulse)
                  double dblHRTimerValueAtSyncPulse_sec, // [in] high-res timer value at the associated sync pulse
                  int iFirstNmeaSampleIndex,
                  int iLastNmeaSampleIndex )
  // Tries to decode the serial bitstream (with NMEA sentences)
  // in the waveform of the previous second-cycle .
  // Input: pFltSource[0..nSamples-1] = waveform with the entire NMEA burst,
  //        usually beginning(!) with PREVIOUS GPS sync pulse (pps),
  //        amplitudes normalized to +/- 1.0,  NOT corrected for polarity,
  //        and still suffering from the 'AC coupling'.
  // Output: pGPSdet->pGpsPosInfo
  //
  // Call Stack (in SL) :
  //   TSoundThread::Execute() -> SOUND_RunProcessingChain()
  //   -> SndThd_CheckSourceAndRunFreqCalib()
  //      -> FreqCalib_ProcessSamples()
  //         -> GPS_Det_ProcessSamples()
  //            -> GpsDet_DecodeSerialData() <------ [THIS function] !
  //               -> GpsDec_DecodeNmeaString() [in C:\cbproj\gps_decoder\gps_dec.c]
{
  int iSample,iSample2,uart_state, uart_bit_number, uart_shift_reg, uart_errors;
  T_Float fltSamplesPerBit;

  T_Float tau, vc1, Vpeak_nmea, Vclamp, Vthrsh, Vnmea_max;
  T_Float x, fltClampedSample, sign, fltOffset;
  T_Float mu, y[4]; // for bit sync / interpolation
  double  fltSampleIndex;
  T_Float hp_alpha, hp_x_old, hp_y_old, hp_y;  // crude highpass with a time-constant of ~0.5*t_NMEA_bit
  // T_Float hp2_y, hp2_y_old, hp2_x_old;         // 2nd highpass (experimental, for 2nd-order HP)
  T_Float lp_alpha, lp_y, lp_y_old;  // crude lowpass with a time-constant of a few audio samples
  T_Float cl_alpha;
  BOOL    fResult;
  int     iFirstGoodSampleIndex, inhibit_countdown;
  T_CubicInterpolator interpolator;
  T_Float bit_voltage;
  BYTE    bRxCharBuf[1024];  // UART 'receive' buffer.  Typical usage: < 400 characters.
                             // Long enough for MULTIPLE NMEA SENTENCES in a burst.
  int     iUARTrxBufIdx = 0; // index into the "  "  "
  int     iNmeaSentenceType;
  unsigned char *cp, *cpNMEAStart;

  fResult = FALSE;

  if( (pGPSdet->t_NMEA_bit<=pGPSdet->t_sample) || (pGPSdet->t_sample<=0) )
   { DOBINI( 0 );
     return FALSE;  // something fishy, refuse to do anything
   }
  if( (iLastNmeaSampleIndex-iFirstNmeaSampleIndex) < 1000 )
   { // the available waveform is too short. Wait for the next time.
     DOBINI( iLastNmeaSampleIndex-iFirstNmeaSampleIndex );
     return FALSE;
   }
  fltSamplesPerBit = pGPSdet->t_NMEA_bit / pGPSdet->t_sample;
  if( fltSamplesPerBit < 2.0 )  // farewell, due to Nyquist / Shannon ..
   { if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
      { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        strcpy( pGPSdet->sz255StatusText, "Too few samples per bit, can't decode NMEA !" );
        ++pGPSdet->iStatusTextUpdateCount;
#      ifdef CIRCUIT_STATE_WARNING  /* from CircuitDefs.h, optional */
        pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
#      endif        
      }
     return FALSE;
   }

  // Increment the 'NMEA sample index' until the (absolute) input signal
  //   has dropped below 25% of the sync pulse amplitude .
  //  (without this, the detection of the "serial data voltage" might fail
  //   due to the slow exponential decay *after* the PPS; see oscillogram) .
  // From the manual (frqcalib.htm) :
  // > PPS signal (synchronisation pulse) : signal amplitude between
  // >             40 and 90 % of the soundcard's clipping level .
  // >  NMEA data (serial output, if used at all) : signal amplitude between
  // >             25 and 50 % of the  *PPS signal* (!) .
  x =  0.25 * pGPSdet->fltPpsPositiveThreshold;
  Vthrsh = x;   // threshold voltage for the loop below
  x = -0.25 * pGPSdet->fltPpsNegativeThreshold;
  if (x > Vthrsh )
   {  Vthrsh = x;
   }
  iFirstGoodSampleIndex = iFirstNmeaSampleIndex;
  while( iFirstGoodSampleIndex<iLastNmeaSampleIndex )
   { x = pFltSource[iFirstGoodSampleIndex];
     if( x<0 ) x=-x;
     if( x< Vthrsh )
      { break;   // end of the annoying 'exponential decay'
      }
     ++iFirstGoodSampleIndex;
   }


  // Measure the 'voltage' of the NMEA pulses,
  //         without overshoot, and without the influence of the slow
  //         exponential decay (after the strong sync pulse - ) .
#define L_TEST_VOLTAGE_MEASUREMENT 0  /* 0=normal compilation, 1=TEST. DON'T REMOVE THIS. */
  // Principle: Lowpass (R1,C1) with fg~~bitrate to defeat the overshoot,
  //            highpass (R2,C2) to remove the slow exponential decay,
  //            and a diode rectifier. This is NOT a voltage doubler:
  //
  //  IN  ,----,     ||C2            |\|D2
  //   o--| R1 |--*--||---*------*-----|----O  out: Vpeak_nmea
  //      '----'  |  ||  _|_     |   |/|    |
  //           ___|___  |   |  __|__     ___|___
  //           _______  |R2 |   /|\      _______
  //              |     |___|  /_|_\        |
  //           C1 |       |   D1 |          |
  //             _|_     _|_    _|_        _|_
  //
  //
  // Prepare a 1st-order low pass filter to remove 'overshoot' for the peak-detection.
  //  time constant : about a half serial bit time, but at least one audio sample.
  //  Principle: see http://en.wikipedia.org/wiki/Low-pass_filter,
  //  "algorithmic implementation of a time-discrete filter":
  //  alpha = t_sample / ( tau * t_sample )
  //  y[i]  = y[i-1] + alpha * (x[i] - y[i-1])   // refactored formula
  tau = 1.0 * pGPSdet->t_sample;   // time constant for the lowpass R1 * C1,  unit: seconds
  lp_alpha = pGPSdet->t_sample / (tau + pGPSdet->t_sample );
  lp_y = lp_y_old = pFltSource[iFirstGoodSampleIndex];
#ifdef __BORLANDC__
  (void)lp_y;  // shut up ..
#endif

  // Prepare a 1st-order high pass filter; time constant = NN bit intervals.
  tau = 10e-3; // time constant for the highpass : R2*C2,  unit: seconds
               // ( 10 milliseconds were STILL TOO MUCH for Freddy's sample,
               //   see file 'GPS_nmea_via_mic_input.png :
               //   the transient effect took about 10 (!) milliseconds,
               //   so a high-pass filter with a large cutoff frequency
               //   of ~100 Hz may work better. But such a filter would turn
               //   the rectangular wave into a slanted waveform, causing errors
               //   in the peak-voltage detection. A better way to defeat
               //   the initial transient was to simply ignore the first few bits )
  hp_alpha = tau / (tau + pGPSdet->t_sample );
  hp_y = hp_y_old = 0.0;
  hp_x_old = pFltSource[iFirstGoodSampleIndex];
  // hp2_y = hp2_y_old = hp2_x_old = 0.0;  // 2nd highpass (experimental, for 2nd-order HP)

#ifdef __BORLANDC__
  (void)hp_y;
#endif

  vc1 = Vpeak_nmea = Vnmea_max = 0.0;  // prepare the dumb "rectifier"
  uart_state = uart_bit_number = 0;    // abused to 'ignore the first bits' here
  inhibit_countdown = 200; // ignore the first XXX samples - reasons below .
    // (an NMEA sentence contains ~40 chars * 10 bit/chr * 4 samples/bit = 1600 samples,
    //  but usually MUCH MORE than that.. so ignoring another 200 samples here
    //  to eliminate the transient effect in Freddy's sample should be ok...)
#ifdef __BORLANDC__
  (void)vc1;  // shut up ..
#endif

  for( iSample=iFirstGoodSampleIndex; iSample<iLastNmeaSampleIndex-20; ++iSample )
   {
     x = pFltSource[iSample];

#if(1)
     // To measure the peak voltage WITHOUT overshoot, run the highpass output
     // through the LOW-PASS filter (R1,C2) :
     // Using the refactored algorithm from wiki, Low-pass_filter :
     //  y[i]  = y[i-1] + alpha * (x[i] - y[i-1])   // refactored lowpass formula
     lp_y = lp_y_old + lp_alpha * ( x-lp_y_old );
     lp_y_old = lp_y;
     x = lp_y;   // input for the next stage
#endif // use LOWPASS to reject 'overshoot' ?

#if(0) // TEST: use a 2nd order highpass; same corner frequency to keep it simple
     hp2_y = hp_alpha * ( hp2_y_old + x - hp2_x_old );
     hp2_x_old = x;
     hp2_y_old = hp2_y;
     x = hp2_y;  // input for the next stage
#endif // try two-stage highpass filters in series ?

     // Run the signal through the highpass (R2, C2),
     //   Speciality: Diode D1 across R2 forces the output "never negative":
     hp_y = hp_alpha * ( hp_y_old + x - hp_x_old );
     if( hp_y < 0.0 )  // diode D2 in action.. clamps to zero volts
      { // ex: hp_y = 0.0;   // modified 2011-04-22, to reduce the effect of
                             //  'short negative spikes' :
        hp_y *= 0.25;  // found by trial-and-error, with L_TEST_VOLTAGE_MEASUREMENT
      }
     hp_x_old = x;
     hp_y_old = hp_y;
     x = hp_y;  // input for the next stage

#if(L_TEST_VOLTAGE_MEASUREMENT) // TEST ! :
     if( ! pGPSdet->fScopeDisplayPaused )
      { pGPSdet->pfltSyncdInputWave4Plot[iSample] = x; // TEST ! !
        pGPSdet->pfltSyncdInputWave4Plot[iFirstGoodSampleIndex] = 1.0;
        pGPSdet->pfltSyncdInputWave4Plot[iLastNmeaSampleIndex]  = 1.0;
      } // end if < TEST >
#endif

     if( inhibit_countdown > 0 )
      { --inhibit_countdown;
      }
     else
      {
        // Throw away the first 20 bits .. there will be plenty enough of them.
        // This avoids measuring Vpeak_nmea while we're still in the annoying
        // 'transient area' shortly after the sync pulse - see Freddy's "Jupiter" sample.
        if( x > 1.1*Vthrsh )
         { uart_state = 1;
         }
        else if( x < 0.9*Vthrsh )
         { if( uart_state==1 )
            { ++uart_bit_number;  // just a COARSE measure at this point
            }
           uart_state = 0;
         }
        if( uart_bit_number > 100 ) // enough pulses have passed... start measuring the serial signal voltage:
         { // 2011-04-22 : Got here with ..
           //               iSample               =  8644,
           //               iFirstGoodSampleIndex =  5004,
           //               iLastNmeaSampleIndex  = 50441 .
           if( x > Vpeak_nmea )  // D2 conducts, charging "Vnmea_max" :
            { Vpeak_nmea = x;
            }
         }
      }
   } // end for( iSample..    to measure V_nmea_max )
  pGPSdet->Vpeak_nmea = Vnmea_max = Vpeak_nmea;
  // Test: Vpeak_nmea should be around 0.47 in Freddy's sample (48kS/s, 9k6),
  //       but due to hum & overshoot, the program measured 0.56 .
  //       After modifying the D1 - clamping ("series resistor"),
  //             using tau = 1 sample for the lowpass filter.
  //             and the 'ignore the first 100 pulses'-trick,
  //       Vpeak_nmea was measured as 0.49 (which is "good enough") .
  //  The same parameters even worked with a 19k2 bitstream sampled at 48kS/s,
  //  because there was a sufficient number of 000-111-000 sequences in there
  //  so the measured signal voltage was ok even though the lowpass-filtered
  //  signal looked ugly (a 0-1-0-1 bit sequence only reached 50 % amplitude).
  //  -> Keep it that way, at least for a start .   WB 2011-02-22 .
#if( L_TEST_VOLTAGE_MEASUREMENT )
  return FALSE;
#endif

  // Next step : Find the "first startbit", and
  //             find out the serial data 'polarity' automagically !
  // A UART's 'idle line state' is logic HIGH, which is the LOW voltage here.
  //       But some GPS receivers (for example, Rockwell "Jupiter")
  //       with a TTL output use POSITIVE logic; logic low = low voltage.
  //       Furthermore, some soundcards invert the input, others don't .
  //  - a STARTBIT (which looks like a logic ZERO) is LOW VOLTAGE (otherwise, invert),
  //  - a LOGIC ONE in the serial data is HIGH (positive) voltage,
  //  - a LOGIC ZERO in the serial data is LOW (negative) voltage,
  //  - a STOPBIT, and an IDLE LINE (which look like a logic ONE), are HIGH VOLTAGE
  // Prepare a highpass to identify the 'polarity':
  tau = pGPSdet->t_NMEA_bit; // time constant for the highpass : one bit-time (not critical)
  hp_alpha = tau / (tau + pGPSdet->t_sample );
  hp_y = hp_y_old = 0.0;
#ifdef __BORLANDC__
  (void)hp_y;  // shut up ..
#endif

  hp_x_old = pFltSource[iFirstNmeaSampleIndex];
  for( iSample=iFirstNmeaSampleIndex; iSample<iLastNmeaSampleIndex; ++iSample )
   {
     x = pFltSource[iSample];
     // Run the signal (x) through the highpass :
     hp_y = hp_alpha * ( hp_y_old + x - hp_x_old );
     hp_x_old = x;
     hp_y_old = hp_y;
     x = hp_y;  // input for the next stage

     if( iSample >= iFirstGoodSampleIndex )
      { // Examine the high-pass filtered input signal .
        // The first pulse which exceeds the threshold is the STARTBIT .
        // Again, the STARTBIT looks like a LOGIC ONE - not a logic ZERO !
        if( x > Vthrsh ) // startbit (logic ZERO!) seems to have POSITIVE voltage
         { sign = -1.0;  // multiply the input with -1 to have POSITIVE LOGIC for the databits
           if( iSample > iFirstGoodSampleIndex )
            { iFirstGoodSampleIndex = iSample-2;
            }
           break;
         }
        if( x < -Vthrsh ) // startbit (logic ZERO) seems to have NEGATIVE voltage
         { sign = 1.0;   // multiply the input with +1 to have POSITIVE LOGIC for the databits
           if( iSample > iFirstGoodSampleIndex )
            { iFirstGoodSampleIndex = iSample-2;
            }
           break;
         }
      } // end if( iSample>iFirstGoodSampleIndex )
   } // end for



  // Principle to restore a 'DC coupled' signal (to simplify the decoder) :
  //  Clamp the 'low' level to zero (like a diode circuit)
  //  to defeat the high-pass characteristic of the soundcard .
  //  Clamping circit equivalent with 'DC'-restored voltage at node 'X' :
  //
  //       Vclamp O--*-------                                               .
  //               __|__    _|_                                             .
  //            D1  /|\    |R1 |                                            .
  //               /_|_\   |   |                                            .
  //                 |     |___|                                            .
  //  IN     ||      |       |                                              .
  //   o-----||------*-------*------O  out: 'clamped' voltage               .
  //      -  || +    |                      (fltClampedSample)              .
  //        C1     __|__                                                    .
  //            D2  /|\                                                     .
  //               /_|_\                                                    .
  //                 |                                                      .
  //                _|_                                                     .
  //
  // Idealized 'clamped' voltage around the first received character ("$") :
  //
  //             |
  //   Vclamp ___| _________          __       __       __
  //             | (idle=H) |S |  |  | H|  |  | H|  |  | H|
  //             |          |T |  |  |  |  |  |  |  |  |S |
  //   Vthrsh ___|          |A |b0|b1|b2|b3|b4|b5|b6|b7|T |
  //             |          |R |  |  |  |  |  |  |  |  |O |next
  //             |          |T | L| L|  | L| L|  | L| L|P |startbit
  //     0 V  ___|          |__|__|__|  |__|__|  |__|__|  |..
  //             |
  //                           "$" = 0x24 = 00100100 bin
  //                      ( first character in NMEA sentence )
  //
  // calculate the voltage level, using Vnmea_max from the pre-analysis loop:
  Vclamp = 1.01 * Vnmea_max;  // should ideally NEVER clamp (via diodes), but..
  Vthrsh = 0.5  * Vnmea_max;
  if( ! pGPSdet->fScopeDisplayPaused )
   { pGPSdet->dblNmeaDataThrshold4Scope = Vthrsh;
   }

  // To avoid seeing startbits "in the noise" :
  // Refuse to decode if the automatic threshold is suspiciously low (whatever that is)
  if( Vthrsh < 0.01 )   // typically, Vthrsh is above 0.1 (on a 0...1 scale)
   { if(pGPSdet->iStatusTextModifiedInLine != __LINE__  )
      { pGPSdet->iStatusTextModifiedInLine = __LINE__-1;
        strcpy( pGPSdet->sz255StatusText, "Amplitude of serial data too low !" );
        ++pGPSdet->iStatusTextUpdateCount;
#      ifdef CIRCUIT_STATE_WARNING  /* from CircuitDefs.h, optional */
        pGPSdet->iCircuitState = CIRCUIT_STATE_WARNING;
#      endif
      }
     return FALSE;
   }

  // Prepare the 'slow charging action' of C1 via R1 (see clamping circuit diagram above)
  tau = 200.0 * pGPSdet->t_NMEA_bit;   // time constant for R1 * C1,  unit: seconds
  cl_alpha = pGPSdet->t_sample / (tau + pGPSdet->t_sample );
  fltClampedSample = Vclamp;
  vc1 = Vclamp - x; // initial voltage across C1 for the initial input voltage (see schematics!):
                    // the first clamped sample, fltClampedSample = x + vc1, shall be Vclamp .
  // Copy the 'DC clamped' waveform for the software UART.
  //  (to make things easy, the decoder needs 'random access' to this array)
  if( pGPSdet->pfltProcessedInputForUART == NULL )
   { return FALSE;
   }
  for( iSample=0; iSample<(int)pGPSdet->dwAllocdInputWaveformLength; ++iSample )
   { pGPSdet->pfltProcessedInputForUART[iSample] = fltClampedSample; // clear 'unused' junk
   }
  for(iSample=iFirstNmeaSampleIndex; iSample<iLastNmeaSampleIndex; ++iSample )
   {
     x = sign * pFltSource[iSample];

     // Generate the 'clamped' (but NOT LOWPASS-FILTERED) waveform :
     fltClampedSample = x + vc1;  // -> 'DC clamped' voltage, ideally 0...Vclamp
     vc1 += cl_alpha/*0.00X*/ * ( Vclamp - fltClampedSample/*ex:vc1*/ ); // R1 charges C1 (SLOWLY)
     if( fltClampedSample > Vclamp )  // output voltage too high, D1 starts to conduct..
      { // ex: fltClampedSample = Vclamp;
        // ex: vc1 = fltClampedSample - x;  // vc1 = new voltage across C1
        // kind of "limit the diode current", to reduce the effect of overshoot:
        vc1 -= 0.25 * (fltClampedSample - Vclamp); // -> vc1 gets lower (C1 discharged via D1)
      }
     else if( fltClampedSample < 0.0 ) // output voltage negative, D2 starts to conduct..
      { // Note that the lower (zero volt") clamping level is ALWAYS active .
        // "Limit the diode current", to reduce the effect of overshoot:
        vc1 -= 0.1 * fltClampedSample; // -> vc1 gets MORE POSITIVE (C1 charged via D2)
      }
     fltClampedSample = x + vc1;  // 2nd time, now with vc1 possibly modified
     pGPSdet->pfltProcessedInputForUART[iSample] = fltClampedSample;

     // Also copy the 'DC clamped' waveform for the oscilloscope (for debugging) ?
     if( (pGPSdet->iScopeOption == GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM )
       && (! pGPSdet->fScopeDisplayPaused) && (pGPSdet->pfltSyncdInputWave4Plot!= NULL) )
      { // copy this 'clamped' UART input signal for the scope display :
        pGPSdet->pfltSyncdInputWave4Plot[iSample] = fltClampedSample;
      }

     if( pGPSdet->dwAllocdInputWaveformLength == 0 )
      { DOBINI( 0 );    // kludge for "fast termination" (see GPS_Det_Delete)
        return FALSE;
      }
   } // end for ( iSample.. )

  // Prepare the software UART, and decode the bitstream into CHARACTERS :
  uart_errors = 0;
  uart_state = uart_bit_number = uart_shift_reg = 0;
  iSample=iFirstGoodSampleIndex; // ex: iFirstNmeaSampleIndex;
  while( iSample<(iLastNmeaSampleIndex-20/*20 samples ? too few for 10 bits*/) )
   {
     fltClampedSample = pGPSdet->pfltProcessedInputForUART[iSample];

     // Test for the next startbit (logic ZERO!)  :
     if( fltClampedSample < Vthrsh ) // yes, looks like a startbit
      { // For debugging, WATCH this (with Borland C++) :
        //   pGPSdet->pfltProcessedInputForUART[iSample-2]
        //  (set 'Repeat Count' to 4 for that expression) .
        if( (iUARTrxBufIdx==0) && (!pGPSdet->fScopeDisplayPaused) )
         { pGPSdet->iFirstNmeaSampleIndex4Scope = iSample;
         }
        // At this point, iSample is the array index of the FIRST SAMPLE
        //   which belongs to the startbit ( voltage < 0.5 * Vnmea_max ) .
        // But as a floating point number, the sample-index where the voltage
        //   was exactly Vthrsh will be a bit LESS than that
        //   (see calculation of fltSampleIndex further below).
        //
        // To find the begin of the startbit with
        //  sub-sample resolution, use a cubic interpolation
        //  with FOUR adjacent samples. The threshold, Vthrsh,
        //  is crossed somewhere between y1 and y2 .
        // (if there's a large number of samples per "data bit",
        //  this is total overkill, but with only 2.5 samples per bit,
        //  e.g. 19200 bit/second and 48000 samples/second, it's necessary)
        fltOffset = 0.0;
        for(int i=0; i<=3; ++i )
         { fltOffset += pGPSdet->pfltProcessedInputForUART[iSample-2+i];
         }
        fltOffset *= 0.25;  // to make y[0]..y[3] "completely DC-free" :
        for(int i=0; i<=3; ++i )
         { y[i] = pGPSdet->pfltProcessedInputForUART[iSample-2+i] - fltOffset;
         }
        // Watch y[0] (with repeat count = 4)
        // in Borland's integrated debugger . Typical values:
        // y[0]=0.15 y[1]=0.13 y[2]=-0.12 y[3]=-0.17  Vpeak_nmea=0.24
        CubicIpol_Init( &interpolator, y[0],y[1],y[2],y[3] );
        // Solve that formula (polynomial) for y=0.0000 (in fact Vthrsh).
        // The resulting 'mu' is between 0.0 (~y1) and 1.0 (~y2) .
        // Ideally, mu=0.5 which means the interpolated signal
        // crosses zero in the middle between y[1] and y[2] .
        mu = CubicIpol_FindZero( &interpolator );
        if( mu<-0.2 || mu>1.2 )  // ex: ( mu<0 || mu>1 )
         { // This should never happen, but it does !
           // If it happens, the 'rising edge' is NOT a startbit,
           //  but (for EXAMLE) :
           //  - a pulse larger than the 'high' voltage in the NMEA burst (from pass zero).
           //  - just "noise" near the threshold voltage (but not a real step)
           mu = mu;   //  <<< set breakpoint here, and WATCH y[0]..y[3]
           ++iSample; // skip this junk; advance array-index for the next sample
#         ifdef __BORLANDC__
           (void)mu;  // shut up ..
#         endif
         }
        else  // mu approximately between 0 and 1 : Looks like a valid startbit ...
         { // The BEGIN of the startbit is somewhere between
           //         y[1] ~ pGPSdet->pfltProcessedInputForUART[iSample-2+1] = [iSample-1]
           //     and y[2] ~ pGPSdet->pfltProcessedInputForUART[iSample-2+2] = [iSample] .
           // Example:  mu=0.0 -> bit begins exactly at y[1] = sig[iSample-1],
           //           mu=0.5 -> bit begins right between y[1] and y[2], i.e. at index iSample-0.5,
           //           mu=1.0 -> bit begins exactly at y[2] = sig[iSample] .
           // Determine the theoretic sample index for the CENTER of the startbit:
           fltSampleIndex = (double)iSample - 1.0 // -> source index of y[1] (above)
                          + mu;                   // -> offset between y[1] and y[2]
           // At this point, fltSampleIndex is the 'theoretic' index of the BEGIN of the startbit,
           //    i.e. where the negative slope of the signal voltage JUST crosses Vthrsh=Ampl/2 .
           // Example: iSample=28644, mu=0.56, fltSampleIndex = 28643.56 ( LESS than iSample! )
           fltSampleIndex += 0.5 * fltSamplesPerBit; // -> now it's the index of the CENTER of the STARTbit
           // Measure the 'voltages at the center of the databits',
           //    using interpolation (because we may be close to the Shannon/Nyquist limit).
           // For simplicity, the stopbit is treated as the 9-th databit. Handled later.
           uart_shift_reg = 0;
           for( uart_bit_number = -1/*startbit*/;
                uart_bit_number <= 8/*8=stopbit*/;
              ++uart_bit_number )
            {
              // fltSampleIndex = ideal index (floating point!) to sample the exact CENTER of the databit.
              iSample2 = (int)fltSampleIndex; // array index for 'center' sample (INTEGER, "left neighbour")
              mu = fltSampleIndex - (T_Float)iSample2; // -> mu = 0.0 .. 1.0
              // Prepare the cubic interpolation to retrieve the 'center' voltage:
              CubicIpol_Init( &interpolator,
                 pGPSdet->pfltProcessedInputForUART[iSample2-1],  // y[0]
                 pGPSdet->pfltProcessedInputForUART[iSample2  ],  // y[1] (mu=0, "left neighbour")
                 pGPSdet->pfltProcessedInputForUART[iSample2+1],  // y[2] (mu=1, "right neighbour")
                 pGPSdet->pfltProcessedInputForUART[iSample2+2]); // y[3]
              bit_voltage = CubicInterpolate( &interpolator, mu );
              if( bit_voltage > Vthrsh ) // logic H ?
               { if( uart_bit_number==-1 ) // it's the STARTBIT.. it's bad
                  { ++uart_errors; // bug: saw a falling slope, but it's NOT a startbit ?!
                  }
                 else // it's one of the databits or the stopbit : shift in..
                  { uart_shift_reg |= 1<<uart_bit_number;
                  }
               }
              fltSampleIndex += fltSamplesPerBit; // inc index (floating point!) for the center of the next bit
            } // end for < NINE bits (8 databits plus one stopbit) >
           // check the STOP BIT, it must be 'logic high' !
           if( uart_shift_reg & 0x0100 )
            { // Bingo, at least from the UART's point of view
              // the format is correct. Append the received byte
              // to the software-UART's receive buffer:
              if( iUARTrxBufIdx < sizeof(bRxCharBuf) )
               { bRxCharBuf[iUARTrxBufIdx++] = (BYTE)(uart_shift_reg&0x00FF);
               }
            }
           else  // bad stop bit !?
            { // 2011-04-10: Got here at iUARTrxBufIdx=5,
              // Garmin GPS 18x at 19k2, bRxCharBuf[0..4] was "$GPRS" (not ok)
              ++uart_errors;
#            ifdef __BORLANDC__
              (void)uart_errors;
#            endif
            }
           // For the outer loop, set iSample to the BEGIN(!) of the current STOPBIT,
           // because otherwise, the sampling point of the *NEXT* startbit
           // cannot be detected precisely (the interpolator NEEDS the previous stopbit).
  // ex:iSample += (int)( +0.5 - 1.0 + 9.0/*except the STOPBIT!*/ * fltSamplesPerBit );
  //                  // + 0.5 to round, - 1.0 because
  //                  //      iSample will be incremented by one below .
           // Note: after the bit-loop above, fltSampleIndex already points
           //       into the center of the NEXT STARTBIT.
           // But we want the BEGIN OF THE STOPBIT again, so:
           iSample = (int)( fltSampleIndex - fltSamplesPerBit );
           if( !pGPSdet->fScopeDisplayPaused )
            { pGPSdet->iLastNmeaSampleIndex4Scope = iSample;
            }
           // At this point, iSample is in the center of the stopbit .
           //  ( stopbit = high level, same as the "idle state") .
           // Decrement the index by one or two, as long as the previous sample is H-level.
           // This makes sure that the interpolator for the next H-L transition works properly.
           for(int i=0; i<2; ++i)
            { fltClampedSample = pGPSdet->pfltProcessedInputForUART[iSample-1];
              if( fltClampedSample < Vthrsh ) // doesn't look like the stopbit..
               { break;
               }
              else
               { --iSample;
               }
            }
         } // end else < "mu" from the stopbit-analysis ok >
      } // end if( fltClampedSample < Vthrsh ) .... may be a startbit
     else // it's NOT a startbit.. but the line is IDLE (=high level)
      { ++iSample;  //  array-index for the next sample
      }

    if( pGPSdet->dwAllocdInputWaveformLength == 0 )
     { DOBINI( 0 );    // kludge for "fast termination" (see GPS_Det_Delete)
       return FALSE;
     }
   } // end for ( iSample.. )

  // DECODE the NMEA strings, using  C:\cbproj\gps_decoder\gps_dec.c  .
  //  The input for that decoder is an array of characters,
  //  same as if the input was received from a serial port (hardware).
  if( iUARTrxBufIdx < sizeof(bRxCharBuf) )
   { bRxCharBuf[iUARTrxBufIdx] = 0;  // terminate the string with a zero byte
     if( iUARTrxBufIdx>0 )
      { // Record the 'raw' NMEA strings, regardless of being decodable or not ?
        GpsGUI_AppendReceivedBytesToLogfile( bRxCharBuf, iUARTrxBufIdx ); // implemented in GPSReceiverGUI.cpp!
        // Note: Since 2020-04-04 (when wrong DATES were parsed from a GPS18LVC),
        //       GpsGUI_AppendReceivedBytesToLogfile() can optionally dump the
        //       NMEA to a text editor control, from where it can be copied .
      }
     cp = (unsigned char*)bRxCharBuf;
     while(*cp>0)
      {
        DOBINI( iUARTrxBufIdx );
        cpNMEAStart = cp;
        iNmeaSentenceType = GpsDec_DecodeNmeaString( // here: called after receiving NMEA *via soundcard*
          &cp, // [in] NMEA source pointer (will be skipped until, and including, the CR+NL)
          pGPSdet->pGpsPosInfo );  // [out] GPS position, time and date, and other info
        DOBINI( iUARTrxBufIdx );
        if( iNmeaSentenceType != C_GPS_RESULT_FALSE )  // Recognized ANY kind of NMEA ?
         { if( cp>cpNMEAStart ) // truncate a single NMEA sentence (there may be MORE THAN ONE in a row)
            {  cp[-1] = '\0';   // ugly but no problem, because the source-buffer is LOCAL ("ours")
            }
           GpsGUI_CallbackForReceivedNMEASentences( cpNMEAStart ); // optionally show in the GUI
         }
        if( ( (iNmeaSentenceType==C_GPS_RESULT_OK_RMC)
           || (iNmeaSentenceType==C_GPS_RESULT_OK_GGA) )
          &&   pGPSdet->pGpsPosInfo->time_valid
          &&   pGPSdet->pGpsPosInfo->date_valid )
         { fResult = TRUE;
           // Because GpsDec_DecodeNmeaString() doesn't care about the sync pulse's time of reception,
           // care for the 'high resolution timer' HERE, so we can always retrieve
           // the GPS-based *CURRENT* DATE AND TIME including the 'high-resolution timer offset' .
           // In this case, do NOT call TIM_QueryHRTimer_sec() but use the 'high-res timer value'
           // of the GPS sync pulse (PPS signal) related to the emitted NMEA sentence(s) :
           pGPSdet->pGpsPosInfo->dblHRTimerValueAtReception_sec = dblHRTimerValueAtSyncPulse_sec;
         }
        if( iNmeaSentenceType<=0 ) // erroneous sentence or checksum error:
         { ++pGPSdet->iSerialDecoderErrors;
           break;
         }
      }
   }

  DOBINI( 0 );
  return fResult;
} // end GpsDet_DecodeSerialData()

void GpsDet_CopySamplesForPlotting( T_GPS_Detector *pGPSdet )
{ int i, n = pGPSdet->dwSyncdWaveformSampleIndex;
  if( n>(int)pGPSdet->dwAllocdInputWaveformLength )
   {  n=(int)pGPSdet->dwAllocdInputWaveformLength; // just for safety..
   }
  DOBINI( n );
  for(i=0; i<n; ++i)
   {
     pGPSdet->pfltSyncdInputWave4Plot[i] = pGPSdet->pfltSyncdInputWaveform[i];
     // 2012-03-23: Crashed here when the audio-processing-thread
     //             consumed too much CPU time (?) .
     // 2012-03-23: Crashed here after switching from 'mono' to 'stereo'.
     //
   }
  pGPSdet->iSyncdInputWave4PlotLength = n;
  pGPSdet->iSyncdSampleIndex4Plot_PPS = pGPSdet->iSampleIndexOfPPS[2];
  if(  (pGPSdet->dblSyncPulseLength4Scope < (pGPSdet->t_sync_pulse_length - 1e-3) )
     ||(pGPSdet->dblSyncPulseLength4Scope > (pGPSdet->t_sync_pulse_length + 1e-3) ) )
   {    pGPSdet->dblSyncPulseLength4Scope = pGPSdet->t_sync_pulse_length;
   }
} // end GpsDet_CopySamplesForPlotting()


//---------------------------------------------------------------------------
int GPS_Det_GetNumSamplesForScope( T_GPS_Detector * pGPSdet )
{
   switch( pGPSdet->iScopeOption )
    { case GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS:
      default:  // this will most likely be the 'default' scope-mode :
         return C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO;
      case GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE:
         return C_GPSDET_LOWPASS_KERNEL_LENGTH;
      case GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE     :
         return C_GPSDET_FIFO_LEN_BRUTTO; // fltInputPulse[C_GPSDET_FIFO_LEN_BRUTTO];
      case GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM :
      case GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM  :
         return pGPSdet->dwAllocdInputWaveformLength;
    } // end switch( pGPSdet->iScopeOption )
} // end GPS_Det_GetNumSamplesForScope()

double GPS_Det_ScopeSampleIndexToPhysicalUnit( T_GPS_Detector *pGPSdet,
          int iScopeSampleIndex, // [in] sample index
          int *piScaleUnit )     // [out,optional] SCALE_UNIT_HERTZ or .._SECONDS, etc
  // Caution: For the GPS PULSE DETECTOR, the relationship between
  //   "scope sample index" and "physical value for the X-axis"
  //   may permanently change, due to the sub-sampling process !
  //   For the same 'sample index', we may get a different result each second !
{
  double d;
  switch( pGPSdet->iScopeOption )
    { case GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS:
      default:  // this will most likely be the 'default' scope-mode :
        if( piScaleUnit != NULL )
         { *piScaleUnit = SCALE_UNIT_SECONDS;
         }
        // <C_GPSDET_INTERPOLATION_RATIO> interpolated samples are ignored here !
        // The input, iScopeSampleIndex, ranges from zero
        //  to C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO - 1 .
        // EX (doesn't precisely CENTER the curve at the 'measured' zero-crossing):
        // d = (double)iScopeSampleIndex - 0.5 * (C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO);
        // return d * pGPSdet->t_sample / C_GPSDET_INTERPOLATION_RATIO; // -> SECONDS (not Hertz!)
        // The CENTER element, iScopeSampleIndex
        // == (C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO)/2 ,
        // shall contain the zero crossing, i.e. InterpGPSdet->dblInterpolatedStepZeroCrossingIndex .
        // pGPSdet->dblInterpolatedStepZeroCrossingIndex is 'ideally' 0.5* C_GPSDET_INTERPOLATED_STEP_LENGTH .
        // One "scope sample index" spans pGPSdet->t_sample / (double)C_GPSDET_INTERPOLATION_RATIO seconds,
        // because the scope plots the *INTERPOLATED* step .
        // The result shall be scaled into seconds for plotting (in SRCalibCtrl.cpp),
        // ranging from  dblPhysValueMin to dblPhysValueMax ( see  FreqCalib_GetXAxisRangeForScope ).
        d = (double)iScopeSampleIndex/*runs from ZERO to C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO-1 */
            - pGPSdet->dblInterpolatedStepZeroCrossingIndex; // ~~ 0.5* C_GPSDET_INTERPOLATED_STEP_LENGTH
        return d * pGPSdet->t_sample / (double)C_GPSDET_INTERPOLATION_RATIO; // array index -> SECONDS
     case GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE:
        if( piScaleUnit != NULL )
         { *piScaleUnit = SCALE_UNIT_NONE;
         }
        return iScopeSampleIndex;
     case GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE     :
        if( piScaleUnit != NULL )
         { *piScaleUnit = SCALE_UNIT_SECONDS;
         }
        return (double)(iScopeSampleIndex-C_GPSDET_FIFO_LEN_BRUTTO/2) * pGPSdet->t_sample;
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM :
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM  :
        if( piScaleUnit != NULL )
         { *piScaleUnit = SCALE_UNIT_SECONDS;
         }
        return (double)(iScopeSampleIndex-pGPSdet->iSyncdSampleIndex4Plot_PPS)
                      * pGPSdet->t_sample; // -> SECONDS (not Hertz!)
    } // end switch( pGPSdet->iScopeOption )

} // end GPS_Det_ScopeSampleIndexToPhysicalUnit()

double GPS_Det_PhysicalValueToScopeSampleIndex( T_GPS_Detector *pGPSdet,
          double dblHorzPhysicalValue ) // [in] physical value (horizontal axis)
  // Inverse to GPS_Det_ScopeSampleIndexToPhysicalUnit() .
  //   Used for the cursor-readout-function in the 'scope display' .
{
  double d;
  switch( pGPSdet->iScopeOption )
    { case GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS:
      default:  // this will most likely be the 'default' scope-mode :
        // inverse to :
        // d = (double)iScopeSampleIndex/*runs from ZERO to C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO-1 */
        //      - pGPSdet->dblInterpolatedStepZeroCrossingIndex; // ~~ 0.5* C_GPSDET_INTERPOLATED_STEP_LENGTH
        // return d * pGPSdet->t_sample / (double)C_GPSDET_INTERPOLATION_RATIO; // array index -> SECONDS
        if( pGPSdet->t_sample > 0 )
         { d = dblHorzPhysicalValue * (double)C_GPSDET_INTERPOLATION_RATIO / pGPSdet->t_sample;
           return d + pGPSdet->dblInterpolatedStepZeroCrossingIndex;
         }
        break;
     case GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE:
        return dblHorzPhysicalValue;
     case GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE     :
        // Inverse to:
        //  dblHorzPhysicalValue = (double)(iScopeSampleIndex-C_GPSDET_FIFO_LEN_BRUTTO/2) * pGPSdet->t_sample;
        if( pGPSdet->t_sample > 0 )
         { return dblHorzPhysicalValue / pGPSdet->t_sample + 0.5*C_GPSDET_FIFO_LEN_BRUTTO;
         }
        break;
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM :
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM  :
        if( pGPSdet->t_sample > 0 )
         { return dblHorzPhysicalValue / pGPSdet->t_sample
                 + pGPSdet->iSyncdSampleIndex4Plot_PPS ;
         }
        break;
    } // end switch( pGPSdet->iScopeOption )
  return 0.0;
} // end GPS_Det_PhysicalValueToScopeSampleIndex()


void GPS_Det_GetAmplRangeForScope( T_GPS_Detector *pGPSdet,
         double *pdblAmplMin, double *pdblAmplMax, int *piAmplUnit )
{
   switch( pGPSdet->iScopeOption )
    { case GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS:
      default:  // this will most likely be the 'default' scope-mode :
         if( pdblAmplMin!=NULL ) { *pdblAmplMin = -1.1; }
         if( pdblAmplMax!=NULL ) { *pdblAmplMax =  1.1; }
         if( piAmplUnit !=NULL ) { *piAmplUnit  = SCALE_UNIT_NONE; }
         break;
      case GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE:
         if( pdblAmplMin!=NULL ) { *pdblAmplMin = pGPSdet->fltLowpassKernelCoeffMin; }
         if( pdblAmplMax!=NULL ) { *pdblAmplMax = pGPSdet->fltLowpassKernelCoeffMax; }
         if( piAmplUnit !=NULL ) { *piAmplUnit  = SCALE_UNIT_NONE; }
         break;
      case GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE     :
         if( pdblAmplMin!=NULL ) { *pdblAmplMin = -1.0; }
         if( pdblAmplMax!=NULL ) { *pdblAmplMax =  1.0; }
         if( piAmplUnit !=NULL ) { *piAmplUnit  = SCALE_UNIT_NONE; }
         break;
      case GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM :
      case GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM  :
         if( pdblAmplMin!=NULL ) { *pdblAmplMin = -1.0; }
         if( pdblAmplMax!=NULL ) { *pdblAmplMax =  1.0; }
         if( piAmplUnit !=NULL ) { *piAmplUnit  = SCALE_UNIT_NONE; }
         break;
    } // end switch( pGPSdet->iScopeOption )
} // end GPS_Det_GetAmplRangeForScope()

void GPS_Det_GetXAxisRangeForScope( T_GPS_Detector *pGPSdet,
           double *pdblPhysValueMin, double *pdblPhysValueMax, int *piPhysUnit )
{
  double d;
  double dblPhysValueMin=0.0;
  double dblPhysValueMax=1.0;
  int    iScaleUnit = SCALE_UNIT_SECONDS;
#ifdef __BORLANDC__
  (void)dblPhysValueMin; (void)dblPhysValueMax;  // shut up ..
#endif

  switch( pGPSdet->iScopeOption )
   { case GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS:
     default:  // this will most likely be the 'default' scope-mode :
                 d = 0.5 * (double)(C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO)
                         * pGPSdet->t_sample / (double)C_GPSDET_INTERPOLATION_RATIO; // -> SECONDS (not Hertz!)
                 dblPhysValueMin = -d;
                 dblPhysValueMax = d;
                 break;
     case GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE:
                 iScaleUnit = SCALE_UNIT_NONE;
                 dblPhysValueMin = 0;
                 dblPhysValueMax = GPS_Det_GetNumSamplesForScope( pGPSdet );
                 break;
     case GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE     :
                 d = 0.5 * (double)C_GPSDET_FIFO_LEN_BRUTTO * pGPSdet->t_sample; // -> SECONDS (not Hertz!)
                 dblPhysValueMin = -d;
                 dblPhysValueMax = d;
                 break;
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM :
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM  :
                 dblPhysValueMax = pGPSdet->dblSyncPulseLength4Scope;
                 dblPhysValueMin = dblPhysValueMax - 1.01 * pGPSdet->t_sync_interval;
                 break;
   } // end switch( pGPSdet->iScopeOption )
  if( pdblPhysValueMin != NULL )
   { *pdblPhysValueMin = dblPhysValueMin;
   }
  if( pdblPhysValueMax != NULL )
   { *pdblPhysValueMax = dblPhysValueMax;
   }
  if( piPhysUnit != NULL )
   { *piPhysUnit = iScaleUnit;
   }
} // end GPS_Det_GetXAxisRangeForScope()

double GPS_Det_GetSampleForScope( T_GPS_Detector *pGPSdet, int iScopeSampleIndex )
{
   switch( pGPSdet->iScopeOption )
    { case GPSDET_SCOPE_OPTION_SHOW_INTERPOLATED_PPS:
      default:  // this will most likely be the 'default' scope-mode :
        // The input, iScopeSampleIndex, ranges from zero
        //  to C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO - 1 .
        // The CENTER element, iScopeSampleIndex
        // == (C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO)/2 ,
        // shall contain the zero crossing, i.e. pGPSdet->dblInterpolatedStepZeroCrossingIndex :
        // ex: d = (double)iScopeSampleIndex
        //         - 0.5 * (C_GPSDET_INTERPOLATED_STEP_LENGTH-C_GPSDET_INTERPOLATION_RATIO)
        //         + pGPSdet->dblInterpolatedStepZeroCrossingIndex;
        //     iScopeSampleIndex = (int)(d + 0.5);
        // The 'horizontal movement' of the graph depending on dblInterpolatedStepZeroCrossingIndex
        // is now achieved in GPS_Det_ScopeSampleIndexToPhysicalUnit() !
        if( iScopeSampleIndex>=0 && iScopeSampleIndex<C_GPSDET_INTERPOLATED_STEP_LENGTH )
         { return pGPSdet->fltInterpolatedStep[iScopeSampleIndex];
         }
        break;
     case GPSDET_SCOPE_OPTION_SHOW_IMPULSE_RESPONSE:
        if( iScopeSampleIndex>=0 && iScopeSampleIndex<C_GPSDET_LOWPASS_KERNEL_LENGTH )
         { return pGPSdet->fltLowpassKernel[iScopeSampleIndex];
         }
        break;
     case GPSDET_SCOPE_OPTION_SHOW_INPUT_PULSE     :
        if( iScopeSampleIndex>=0 && iScopeSampleIndex<C_GPSDET_FIFO_LEN_BRUTTO )
         { return pGPSdet->fltInputPulse[iScopeSampleIndex];
         }
        break;
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_INPUT_WAVEFORM :
     case GPSDET_SCOPE_OPTION_SHOW_SYNCD_PROC_WAVEFORM  :
        if( (iScopeSampleIndex>=0)
         && ((DWORD)iScopeSampleIndex<pGPSdet->dwAllocdInputWaveformLength)
         && ( iScopeSampleIndex<pGPSdet->iSyncdInputWave4PlotLength)
         && ( pGPSdet->pfltSyncdInputWave4Plot != NULL )  )
         { return pGPSdet->pfltSyncdInputWave4Plot[iScopeSampleIndex];
         }
        break;

    } // end    switch( pGPSdet->iScopeOption )
  return C_SND_MATH_INVALID_FLOAT_VALUE;
} // end GPS_Det_GetSampleForScope()

#if( __LINE__ > 3990 )
#  error "This sourcecode gets too long." // because of GpsPulseDet_bNumErrorsShownInLine[]
#endif
/* EOF < C:\cbproj\SoundUtl\GpsPulseDetector.cpp > */





