//---------------------------------------------------------------------------
// File  :  FftFilter.h
// Date  :  2003-11-16   (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//   Implementation of an FFT-based filter for real time processing
//
// Revision history :  See *.CPP .
//      Not duplicated here to avoid inconsistency !
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef FftFilterH
#define FftFilterH

#include "SoundTab.h"   // some often uses tables for audio processing

//----------------------- Constants ---------------------------------------


#define C_FftFilt_MaxFftSize 32768

// Other 'often needed' stuff..
#ifndef  C_PI
 #define C_PI  (3.1415926535897932384626)
#endif


// DON'T CHANGE these options - they are explained in the manual,
// and people may use them through an interpreter function already !
#define C_FftFilt_OPTIONS_NONE      0
#define C_FftFilt_OPTIONS_DENOISER  0x0001
#define C_FftFilt_OPTIONS_AUTONOTCH 0x0002
#define C_FftFilt_OPTIONS_LIMITER   0x0004  /* since 2005-07 */
#define C_FftFilt_OPTIONS_RESERVED  0x0008  /* withdrawn, may be re-used */
#define C_FftFilt_OPTIONS_F_INVERT  0x0010  /* invert (mirror) frequencies in a range*/
#define C_FftFilt_OPTIONS_RESERVED2 0x0020  /* planned: automatic "gentle transition" for inv-range */
#define C_FftFilt_OPTIONS_SHIFT_UP  0x0040  /* if not set, shift DOWN (RX) */
#define C_FftFilt_OPTIONS_IQ_IN     0x0100  /* filter with I- and Q-input  */
#define C_FftFilt_OPTIONS_IQ_OUT    0x0200  /* filter with I- and Q-output */
#define C_FftFilt_OPTIONS_NO_NEG_FREQS 0x0400 /* reject negative frequencies*/
                         /* (only combined with C_FftFilt_OPTIONS_IQ_OUT ) */
#define C_FftFilt_OPTIONS_CHIRP     0x1000  /* chirp filter (delay proportional to frequency) */

//**************************************************************************
//    Structures and Datatypes used in this module
//**************************************************************************

typedef void (*T_FftFilterCallback)( float *pfltRe, float *pfltIm, int iLength );



//**************************************************************************
//    Global Vars (yucc)
//**************************************************************************

  // none (hooray ;)



//**************************************************************************
//    Prototypes (no class methods)
//**************************************************************************

#ifndef CPROT
// Note on "extern": do not trust the lousy example from Borland !!
//   >extern "c" = wrong (not recognized by Borland C++ V4.0)
//   >extern "C" = right (though the help system says something different)
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif


//***************************************************************************
//  Definition of the  CFftFilter   class
//***************************************************************************

class CFftFilter
{
public:
  CFftFilter();           // constructor without parameters
  virtual ~CFftFilter();  // destructor (cleans up)

  void   SetSampleRate(double dblSamplesPerSecond);
  double GetSampleRate(void);
  int    FrequencyToBinIndex( double dblFreq_Hz );
  BOOL   SetFreqShift_Hz( double dblFreq_Hz );
  double GetFreqShift_Hz( void );
  BOOL   SetFreqInversion( double dblFromFreq, double dblToFreq );

  BOOL SetRealFrequencyResponse( // Sets the filter's frequency response: AMPLITUDES only (no phase)
         float *pfltResponse,        // frequency response, real parts ONLY
         int   iFreqResponseLength); // length of pfltResponseXX[], must be (2^n + 1)
  int  GetRealFrequencyResponse( float *pfltResponse, int iMaxResponseLength);

  BOOL SetThresholdValues(       // to limit (clip) individual FFT bins
         float *pfltThresholdValues, // one threshold value for every FFT bin (!)
         int   iNumberOfValues);     // length of pfltThresholdValues[],

  int  GetNumberOfFftFrequencyBins(void);

  BOOL RegisterInputSpectrumCallback( T_FftFilterCallback pSpectrumReadyCallback, BOOL fTurnOn );  // callback for source of plot data
  BOOL RegisterOutputSpectrumCallback(T_FftFilterCallback pSpectrumReadyCallback, BOOL fTurnOn );

  BOOL SetOptions( int iOptions );   // C_FftFilt_OPTIONS_xxxx
  int  GetOptions( void );

  BOOL ProcessSamples(
                  T_Float *pfltSource,   // real INPUT (or I-component)
                  T_Float *pfltSourceQ,  // quadrature component for input (optional)
                  T_Float *pfltDest,
                  T_Float *pfltDestQ,    // quadrature component for output (optional)
                  long lNrSamples);
       // Runs a number of audio samples through the filter .
       // T_Float is either 'float' or 'double',
       //   defined in c:\CBproj\SoundUtl\Sound.h, SoundMaths.h, and a few other modules,
       //   depending on the compilation switch 'SWI_FLOAT_PRECISION' .


  // Properties...
  int   m_iOptions;       // bitcombination of C_FftFilt_OPTIONS_ xxx (PUBLIC for testing)
  double m_dblSamplesPerSecond;
  double m_dblChirpRate;  // in Hz/second (use with  m_iOptions & C_FftFilt_OPTIONS_CHIRP)
  float *pfltFactors;     // [m_iFftLength]  for noise reduction and auto-notch (PUBLIC for testing)
  DWORD dwFreqRespModifyCount;
  float m_fltAutonotchSpeed;
  float m_fltDenoiserLevel;
  DWORD m_dwFftCounter;   // required for frequency-shift with odd bin-count  


private:
  void  KillBuffers(void);
  BOOL  AllocBuffersIfNeeded(void);
  void  InvertFrequencies(T_Float *pfltFreqBinsRe,T_Float *pfltFreqBinsIm,int iNumFrequBins);
  void  ShiftFrequencies(T_Float *pfltFreqBinsRe,T_Float *pfltFreqBinsIm, int iNumFrequBins, int iShiftOffset );

   BOOL m_fUseComplexFilterResponse;
   BOOL m_fUseComplexInput;
   BOOL m_fUseComplexOutput;
   BOOL m_fOpenForBusiness;

   int m_iSegmentLength;       // length of a processing segment (or chunk)
   int m_iFilterKernelLength;  // length of the original filter kernel (not necessarily a power of two!)
   int m_iFftLength;           // actually used FFT length, defines some buffer sizes
   int m_iTimeDomainBufIndex;  // buffer head index into pfltTimeDomainSig[]

   int m_iOverlapLength;      // length of the overlap from one segment to the next
   T_Float *pfltOverlapRe;    // [m_iOverlapLength] holds overlapping samples from segment to segment
   T_Float *pfltOverlapIm;    // [m_iOverlapLength] " " ", imaginary part, if INPUT is complex
   T_Float *m_pfltFreqBinsRe; // [m_iFftLength] real time domain signal, AND real part of the frequency domain (for the FFT)
   T_Float *m_pfltFreqBinsIm; // [m_iFftLength] imaginary part of the frequency domain (for the FFT)
   T_Float *m_pfltReFreqResp; // [m_iFftLength/2] real part of the filter's frequency response
   T_Float *m_pfltImFreqResp; // [m_iFftLength/2] imaginary part of the filter's frequency response
   int m_iFreqResponseLength; // length of the frequency response (may be LESS the number of frequency bins; see SetFrequencyResponse)

   T_Float *pfltCos2Window;   // [m_iFftLength]   FFT windowing function in an array
   T_Float *m_pfltAverages;   // [m_iFftLength]  for noise reduction and auto-notch (sometimes PUBLIC for testing)
   T_Float *m_pfltThresholds; // [m_iFftLength]  for frequency-selective LIMITER
   int      m_iThresholdLength; // number of allocated m_pfltThresholds[]-values
 //  int m_iOptions;        // (sometimes PUBLIC for testing)

   int   m_iFreqShift_Bins;   // optional frequency shift, expressed as NUMBER OF FFT-BINS
         // (0=don't shift, >0=shift frequencies UP, <0=shift frequencies DOWN)
   int   m_iFreqInvertFromBin, m_iFreqInvertToBin;  // range for optional frequency inversion


   T_FftFilterCallback m_pInputSpectrumReadyCallback; // optional callback for "new spectrum ready"
   T_FftFilterCallback m_pOutputSpectrumReadyCallback; // same for "output spectrum" (AFTER processing)

}; // end class CFftFilter


#endif // FftFilterH