// Special, ad-hoc noiseblanker for AMSAT EVE project .



#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !

#include "windows.h"
#include "stddef.h"    //
#include "math.h"

#include "Utility1.h"  // UTL_NamedMalloc(8-byte-aligned) and other stuff by WB

#include "CLI_ROUT.H"   // the application's own COMMAND LINE INTERPRETER

#include "Config.h"      // global configuration structure for SpecLab

#include "EveNB.h"

typedef struct
{ T_Float re;
  T_Float im;
} T_EVE_NB_COMPLEX;

#define EVE_NB_FIFO_SIZE 8192 // must be a power of two, and large enough for SL's "chunk size" !
T_EVE_NB_COMPLEX EveNB_cpxFifo[EVE_NB_FIFO_SIZE];
int EveNB_iFifoHeadIdx = 0;
double EveNB_dblAmpAvrg = 0.0;  // current, slightly averaged, amplitude (pk)
int    EveNB_iBlankCountdown = 0;
double EveNB_dblBlankedPercent=0;
int    EveNB_iBlankPcntTimer = 0;
int    EveNB_iBlankPcntCount = 0;

//------------------------------------------------------------------------
void EveNB_Blank( int iFromIndex, int iToIndex )
{
  int i;
  while( iFromIndex < iToIndex )
   { i = iFromIndex;
     if( i<0 ) i+=EVE_NB_FIFO_SIZE;
     i &= (EVE_NB_FIFO_SIZE-1);
     if( (EveNB_cpxFifo[i].re!=0.0) || (EveNB_cpxFifo[i].im!=0.0) )
      {  // only count this sample as "blanked" for the statistics
         // if it HAS NOT BEEN BLANKED already:
         EveNB_iBlankPcntCount++; // for statistics only !
      }
     EveNB_cpxFifo[i].re = 0.0;
     EveNB_cpxFifo[i].im = 0.0;
     ++iFromIndex;
   }
}

//------------------------------------------------------------------------
void EveNB_ProcessSamples(
        int i_or_q,           // [in] 0=I (first), 1=Q (second), -1="real input, no I/Q"
        double dblSampleRate, // [in] samples per second
        T_Float *pdblSource,  // [in] sample block, EITHER I or Q (separate blocks)
        T_Float *pdblDest,    // [out] sample block, EITHER I or Q (separate blocks)
        int iNrOfSamples )    // [in] number of sample points in the block, typ. 1024
{
   int i;
   int delay_n_samples;
   int tv_n_samples, tn_n_samples;
   int fifo_head, proc_index, fifo_tail;
   T_EVE_NB_COMPLEX *pCplx;
   T_Float re,im, amp;

   // Ntige Vorlaufzeit zwischen berschreiten der Triggerschwelle
   //   und dem Beginn des Austastintervalls: Typisch 1 ms,
   //   bei 55 kHz (vom SDR-IQ) also ~55 Samples !
   // Tv = Vorlaufzeit (Beginn der Detektion bis Beginn des Blanking-Intervalls)
   //      Bei typischer Strimpuls-Lnge von 2 ms und "Anstiegszeit" von < 0.5ms
   //      sollte 1 ms "Rckschau-Puffer" ausreichen .
   // Tn = Nachlaufzeit (Von Detektion bis Ende des Blanking-Intervalls)
   tv_n_samples = (int)(dblSampleRate *  UConfig.Blackbox[0].dblNbPreTriggerTime + 0.99);
   tn_n_samples = (int)(dblSampleRate *  UConfig.Blackbox[0].dblNbPostTriggerTime + 0.99);

   // Gesamte ntige Verzgerung zwischen Ein- und Ausgang:
   // Wegen der getrennten Einspeisung von "I" und "Q" muss der Puffer
   // mindestens <iNrOfSamples> Abtastwerte gross sein,
   //    zustzlich "tv" (Vorlaufzeit, tv_n_samples) .
   // Prinzip: Es werden grundstzlich ZWEI Chunks im FIFO gehalten,
   //          z.B. bei 1024 Samples pro Chunk 2048 Werte im FIFO.
   delay_n_samples = 2*iNrOfSamples/*chunk_size*/;
   fifo_head = EveNB_iFifoHeadIdx & (EVE_NB_FIFO_SIZE-1); // start index for "input"
   proc_index = fifo_head;
   fifo_tail = (fifo_head-delay_n_samples);  // start index for "output"
   if( fifo_tail < 0 )
    {  fifo_tail += EVE_NB_FIFO_SIZE;
    }
   fifo_tail &= (EVE_NB_FIFO_SIZE-1);

   switch( i_or_q )
    {
      case 0:   // I-component: only put this into the FIFO,
        // must wait for the Q-component until we can process the I/Q samples !

        // Enter the "input" I-component in our fifo :
        for(i=0; i<iNrOfSamples; ++i)
         { EveNB_cpxFifo[fifo_head].re = pdblSource[i];
           fifo_head = (fifo_head+1) & (EVE_NB_FIFO_SIZE-1);
         }
        // Read the "output" I-component from our fifo :
        for(i=0; i<iNrOfSamples; ++i)
         { pdblDest[i] = EveNB_cpxFifo[fifo_tail].re;
           fifo_tail = (fifo_tail+1) & (EVE_NB_FIFO_SIZE-1);
         }
        // do NOT change EveNB_iFifoHeadIdx here !
        break;
     case 1: // i_or_q ==1,  which means we're called for the "Q"-part :
        // Enter the "input" Q-component in our fifo :
        for(i=0; i<iNrOfSamples; ++i)
         { EveNB_cpxFifo[fifo_head].im = pdblSource[i];
           fifo_head = (fifo_head+1) & (EVE_NB_FIFO_SIZE-1);
         }
        // Read the (older!) "output" Q-component from our fifo :
        // (cannot read the "newer" samples here, because of the I-channel)
        for(i=0; i<iNrOfSamples; ++i)
         { pdblDest[i] = EveNB_cpxFifo[fifo_tail].im;
           fifo_tail = (fifo_tail+1) & (EVE_NB_FIFO_SIZE-1);
         }


        // After sending out the "old" samples (which have been processed earlier),
        // process the new block (which have, at this point, just been entered
        //  in the buffer) :
        for(i=0; i<iNrOfSamples; ++i)
         { pCplx = &EveNB_cpxFifo[proc_index];
           re = pCplx->re;
           im = pCplx->im;

           // Envelope detector with complex input :
           amp = sqrt(re*re + im*im);
           EveNB_dblAmpAvrg = 0.99*EveNB_dblAmpAvrg + 0.01*amp; // how fast ?

#if(0)     // TEST: replace the complex sample with the 'amplitude',
           //       to check for proper FIFO operation:
           pCplx->re = amp;
           pCplx->im = amp;
#else    // normal compilaton:
           //ex:  if( amp > (9.0 * EveNB_dblAmpAvrg) )
           if( amp > (UConfig.Blackbox[0].EveNB_dblTrigLevel) )
            { EveNB_Blank( proc_index-tv_n_samples, proc_index );
              EveNB_iBlankCountdown = tn_n_samples;
            }
           if( EveNB_iBlankCountdown > 0 )
            { if( (pCplx->re!=0.0) || (pCplx->im!=0.0) )
               {  // only count this sample as "blanked" for the statistics
                  // if it HAS NOT BEEN BLANKED already:
                  EveNB_iBlankPcntCount++; // for statistics only !
               }
             pCplx->re = 0.0;
              pCplx->im = 0.0;
              --EveNB_iBlankCountdown;
            }
#endif   // (0,1) TEST / normal mode


           proc_index = (proc_index+1) & (EVE_NB_FIFO_SIZE-1);
         } // end for < processing loop >

        // Set the new FIFO-head- and tail index AFTER processing "Q":
        EveNB_iFifoHeadIdx = fifo_head;

        // Update the noiseblanker 'statistics' when enough samples collected
        //  (roughly every second) :
        EveNB_iBlankPcntTimer += iNrOfSamples;
        if( EveNB_iBlankPcntTimer > dblSampleRate )
         { EveNB_dblBlankedPercent = 100.0 * (double)EveNB_iBlankPcntCount / (double)EveNB_iBlankPcntTimer;
           EveNB_iBlankPcntCount = EveNB_iBlankPcntTimer = 0;
         }
        break; // end case i_or_q==1
     default:  // real input, real output
        // Enter the real input in the fifo :
        for(i=0; i<iNrOfSamples; ++i)
         { EveNB_cpxFifo[fifo_head].re = pdblSource[i];
           fifo_head = (fifo_head+1) & (EVE_NB_FIFO_SIZE-1);
         }
        // Read the (older!) "output" from the fifo :
        for(i=0; i<iNrOfSamples; ++i)
         { pdblDest[i] = EveNB_cpxFifo[fifo_tail].re;
           fifo_tail = (fifo_tail+1) & (EVE_NB_FIFO_SIZE-1);
         }

        // After sending out the "old" samples (which have been processed earlier),
        // process the new block (which have, at this point, just been entered
        //  in the buffer) :
        for(i=0; i<iNrOfSamples; ++i)
         { pCplx = &EveNB_cpxFifo[proc_index];
           re = pCplx->re;

           // Envelope detector :
           amp = sqrt(re*re);
           EveNB_dblAmpAvrg = 0.99*EveNB_dblAmpAvrg + 0.01*amp; // how fast ?

           if( amp > (UConfig.Blackbox[0].EveNB_dblTrigLevel) )
            { EveNB_Blank( proc_index-tv_n_samples, proc_index );
              EveNB_iBlankCountdown = tn_n_samples;
            }
           if( EveNB_iBlankCountdown > 0 )
            { if( pCplx->re!=0.0 )
               {  // only count this sample as "blanked" for the statistics
                  // if it HAS NOT BEEN BLANKED already:
                  EveNB_iBlankPcntCount++; // for statistics only !
               }
             pCplx->re = 0.0;
              --EveNB_iBlankCountdown;
            }

           proc_index = (proc_index+1) & (EVE_NB_FIFO_SIZE-1);
         } // end for < processing loop >

        // Set the new FIFO-head- and tail index AFTER processing "Q":
        EveNB_iFifoHeadIdx = fifo_head;

        // Update the noiseblanker 'statistics' when enough samples collected
        //  (roughly every second) :
        EveNB_iBlankPcntTimer += iNrOfSamples;
        if( EveNB_iBlankPcntTimer > dblSampleRate )
         { EveNB_dblBlankedPercent = 100.0 * (double)EveNB_iBlankPcntCount / (double)EveNB_iBlankPcntTimer;
           EveNB_iBlankPcntCount = EveNB_iBlankPcntTimer = 0;
         }
        break; // end case i_or_q==-1 (real signal, not complex)
    } // end swich < I or Q or REAL input >


} // end EveNB_ProcessSamples()

//---------------------------------------------------------------------------
int CliCmd_ENB(T_CLI_CONTEXT *pCC, T_CLI_VALUE *pvDest )
  /* Interpreter stub for all "enb"-procedures and -functions (enb = Extended Noise Blanker, EVE-Noise-Blanker)
   *
   * The source pointer will be incremented to the end of the command.
   *                       The string "enb." has already been skipped.
   * Return: Negative value = Error Code.
   *         Zero           = Ok, command executed.
   *        Positive values are reserved for future applications,
   *          like loops and program branches.
   */
{
 CLI_FLOAT_TYPE cli_float_value;
 int  error_code = CLI_ERROR_NO_ERROR;
 int  token_idx;
 double *pdblParam = NULL;
 char *cp;

  CLI_SkipChar( pCC, '.' );  // skip '.' between group- and component name

  token_idx = CLI_CheckAndSkipNameFromList( pCC,
    // idx: 0  1  2  3
           "tv,tn,tl,pcnt" );
  switch(token_idx) // all 32-bit integer parameters:
   { case 0: pdblParam = &UConfig.Blackbox[0].dblNbPreTriggerTime;
             break;
     case 1: pdblParam = &UConfig.Blackbox[0].dblNbPostTriggerTime;
             break;
     case 2: pdblParam = &UConfig.Blackbox[0].EveNB_dblTrigLevel;
             break;
     case 3: pdblParam = &EveNB_dblBlankedPercent;
             break;
     default:
        break;
   }
  if( pdblParam != NULL )   // get or set a parameter:
   {
     if(pvDest)
      { pvDest->type = CLI_DT_FLOAT;
        pvDest->flt  = *pdblParam;
      }
     else  // not 'GET' but 'SET' :
      { if( CLI_SkipAssignmentOperator( pCC ) )
         {
           error_code = CLI_CalcSum_Float( pCC, &cli_float_value);
           if( error_code != CLI_ERROR_NO_ERROR )
               return error_code;
           if( pdblParam != NULL )
            { *pdblParam = (long)cli_float_value;
            }
         }
      }
     return error_code;
   } // end if < parameter >


  // NONE of the above functions ...
  //   The error code 'CLI_ERROR_UNKNOWN_FUNCTION' should be treated
  //   as a special case by the CALLER, and the displayed error message
  //   should give a hint (thus copy the 'unknown' thing to pCC->sz40UnknownFunctionName).
  cp = pCC->pszSource;
  strcpy( pCC->sz40UnknownFunctionName, "enb." );
  CLI_GetNameOrKeyword( &cp, pCC->sz40UnknownFunctionName+4, 39-4 );
  return CLI_ERROR_UNKNOWN_FUNCTION;

} // end CliCmd_ENB()


/* EOF < EveNB.cpp > */
