/***************************************************************************/
/*  File:   c:\cbproj\SoundUtl\ChunkInfo2.c                                */
/*  Copies: Unknown (2014-02-22) .                                         */
/*   More helpers for the "AUDIO CHUNK" info ( T_ChunkInfo ),              */
/*   besides the basic ones like ChunkInfo_Init() from ChunkInfo.c .       */
/*   - by DL4YHF 2014-02-22                                                */
/*  Since 2011-06, the T_ChunkInfo structure is ALSO used in the           */
/*     Audio-I/O-DLLs, so be EXTREMELY CAREFUL when modifying it !         */
/*  Since 2014-02, to remove some dependencies in ChunkInfo.c, rarely used */
/*     functions like ChunkInfo_AdjustInfoForSampleIndexDiff(),            */
/*                    ChunkInfo_Interpolate(),                             */
/*                    ChunkInfo_GetInterpolatedEntryFromArray(),           */
/*                    ChunkInfoToString() and StringToChunkInfo()          */
/*     are not in ChunkInfo.c anymore, but in ChunkInfo2.c  !              */
/***************************************************************************/

#include "stdio.h"  // no STANDARD I/O but sprintf ..
#include "math.h"
#include "string.h" // "C"-the multi-million-header language. memset() in HERE !

#pragma hdrstop     // precompiled headers don't work here

#include "QFile.h"  // uses the simple parsing functions from WB's QuickFile module
                    // (if the compiler doesn't find this include file,
                    //  add this to the list of annoying "include directories":
                    //  C:\cbproj\YHF_Tools )
#include "Utility1.h"     // DEBUGGER_BREAK(), etc                    
#include "ChunkInfo.h"    // header for THIS module


/***************************************************************************/
void ChunkInfoArray_Init(
       T_ChunkArray *pChunkArray)     // [out] an array of ChunkInfo entries
{
  int i;
  if( pChunkArray != NULL )
   {
     memset( pChunkArray, 0, sizeof(T_ChunkArray) );
     for(i=0; i<CHUNK_ARRAY_MAX_ENTRIES; ++i)
      { ChunkInfo_Init( &pChunkArray->chunk_info[i] );
      }
     pChunkArray->iNewestEntry = -1;  // because of PRE-INCREMENT in ChunkInfoArray_Append(), 1st valid entry at index zero
   }
} // end ChunkInfoArray_Init()


/***************************************************************************/
T_ChunkInfo *ChunkInfoArray_Append(
        T_ChunkArray *pChunkArray,     // [out] an array of ChunkInfo entries
        T_ChunkInfo  *pChunkInfo,      // [in] precise sampling rate, sample index, date+time, "radio" frequency, GPS data,....
        double       dblHistoryLen_s)  // [in] length, in SECONDS, of the history to be squeezed into the array,
                                       //      required to tell how many old info chunks must remain available in the history.
  // Appends a new (="the most recent") T_ChunkInfo from a stream
  //         into an array ("history") for interpolation,
  //         if the previous entry (from the previous call)
  //         is sufficiently old (so that the history ALWAYS covers
  //         at least <dblHistoryLen_s> seconds of data for interpolation.
  //     When called 'very frequently', with pChunkInfo->i64TotalSampleCounter
  //     only differing by a few sample points from the previous entry,
  //     the call is IGNORED.
  // Returns the address of the new entry IN THE ARRAY,
  //     so the caller can modify it afterwards (for example, see SoundDec.cpp)
  // Returns NULL if something went wrong.
{
  T_ChunkInfo *pDstChunkInfo;
  double d;
  if( pChunkArray != NULL )
   {
     // There is only a limited number of such structs in the array,
     // but since GPS emits new data only once per second, having 32 buffers
     // should be enough to retrieve this info later, matching the SAMPLE INDEX .
     // Since 2015, the caller's SAMPLE ARRAY LENGTH (measured in SECONDS)
     // is used here to determine if a new entry may be stored.
     // The decision is trivial:
     //   The SAMPLE INDEX DIFFERENCE between two chunk-infos in the array
     //   shall not be less than the caller's SAMLE ARRAY LENGTH / CHUNK_ARRAY_MAX_ENTRIES :
     d = pChunkInfo->i64TotalSampleCounter - pChunkArray->i64PrevSampleCounter;
     if( pChunkInfo->dblPrecSamplingRate > 0.0 )
      { d /= pChunkInfo->dblPrecSamplingRate; // -> difference between this and the previous entry in SECONDS
        // (don't rely on pChunkInfo->ldblUnixDateAndTime; it may 'hop around'!)
        // Example: f_sample = 32kHz,
        //          ChunkInfoArray_Append() last called 8192 input samples ago,
        //          d = 0.256 seconds,
        //          dblHistoryLen_s = 11 seconds,
        //          CHUNK_ARRAY_MAX_ENTRIES = 32  -> min write interval = 11 s / 32 = 0.343 seconds
        //    -> "do NOT increment iNewestEntry yet" !
        //          NEXT call : d = 0.512 seconds -> ok, append to array this time.
      }
     if( (pChunkArray->nUsedEntries <= 0 ) // FIRST entry in the chunk history, or..
        || (d<0.0) || (d>=(dblHistoryLen_s/CHUNK_ARRAY_MAX_ENTRIES) ) ) // previous increment of iNewestEntry is sufficiently old
      { ++pChunkArray->iNewestEntry;
        pChunkArray->i64PrevSampleCounter = pChunkInfo->i64TotalSampleCounter;
        if(  pChunkArray->nUsedEntries < CHUNK_ARRAY_MAX_ENTRIES )
         { ++pChunkArray->nUsedEntries;
         }
      }
     // Keep iNewestEntry in valid bounds even if NOT incremented :
     if( pChunkArray->iNewestEntry<0
       ||pChunkArray->iNewestEntry>= CHUNK_ARRAY_MAX_ENTRIES )
      {  pChunkArray->iNewestEntry = 0;
      }
     // Copy the caller's NEW chunk info into the array,
     //      EVEN IF THE INDEX (iNewestEntry) WAS NOT INCREMENTED.
     //      This way, the interpolation can always operate with the most recent info.
     pDstChunkInfo  = &pChunkArray->chunk_info[pChunkArray->iNewestEntry];
     *pDstChunkInfo = *pChunkInfo;
     return pDstChunkInfo; // ALWAYS return the address of the buffered info,
                           // even if pChunkArray->iNewestEntry was NOT incremented.
                           // That way, the caller can overwrite the 'last appended' entry
                           // with the most recent data (GPS, sampling rate,..) if he needs to.
   }
  return NULL;

} // end ChunkInfoArray_Append()


/***************************************************************************/
BOOL ChunkInfoArray_GetInterpolatedEntry(
        T_ChunkArray *pChunkArray,     // [in] an array of ChunkInfo entries
        T_ChunkInfo  *pChunkInfo,      // [out] precise sampling rate, date+time, "radio" frequency, GPS data
        LONGLONG i64WantedSampleIndex) // [in] compared with pChunkInfoArray->i64TotalSampleCounter
{
  int i;
  T_ChunkInfo *pciSrc;
  T_ChunkInfo *pci1 = NULL;  // closest possible 'left neighbour'
  T_ChunkInfo *pci2 = NULL;  // closest possible 'right neighbour'
  LONGLONG i64SampleIndex1=-1, i64SampleIndex2=-1;
  double dblRelativeIndex;

  if( pChunkInfo==NULL )
   {  return FALSE;
   }

  // Search the history of 'most recent' entries in the timestamp history
  // for the closest neighbours (for the requested sample index) .
  //    The danger of accessing an entry in the same moment as it is
  //    being overwritten in a different thread is low, because
  //    the 'overwritten' entry will be VERY OLD
  //    and never used if nArrayEntries is sufficiently large.
  //         In Spectrum Lab, SoundDec.cpp ENTERS chunk-info from the audio thread,
  //         while SoundThd_WaveSaveProcess() READS chunk-info from the queue
  //         a few hundred milliseconds after they were entered.
  // But 2016-10-28, the chunk-info requested by CSoundDecimatingBuffer()
  //     didn't seem to be in this array anymore : all entries
  //     had  pciSrc->i64TotalSampleCounter  >  i64WantedSampleIndex,
  //     when called from Sound_WaveSaveProcess() -> CSoundDecimatingBuffer::GetSamplePoints() .
  //     Not fixable HERE, but fixed in C:\CBproj\SoundUtl\SoundDec.cpp .
  //     (the caller, GetSamplePoints(), now limits i64WantedSampleIndex
  //       *before* passing that sample-index to ChunkInfoArray_GetInterpolatedEntry() )
  for(i=0; i<pChunkArray->nUsedEntries; ++i )
   {
     pciSrc = &pChunkArray->chunk_info[i];
     if(  (pciSrc->dwSizeOfStruct == sizeof(T_ChunkInfo) ) // array entry currently valid ?
       && (pciSrc->dwNrOfSamplePoints > 0 )                // .. to exclude 'unused' entries
       )
      { if( pciSrc->i64TotalSampleCounter <= i64WantedSampleIndex )
         { // this buffer entry is OLDER than the one requested by the caller (or "on spot")
           if(   (pciSrc->i64TotalSampleCounter > i64SampleIndex1 )
              || (i64SampleIndex1 < 0 ) ) // .. and it's a CLOSER "left neighbour"
            { i64SampleIndex1 = pciSrc->i64TotalSampleCounter;
              pci1 = pciSrc;
            }
         }
        if( pciSrc->i64TotalSampleCounter > i64WantedSampleIndex )
         { // this buffer entry is NEWER than the one requested by the caller:
           if(   (pciSrc->i64TotalSampleCounter < i64SampleIndex2 )
              || (i64SampleIndex2 < 0 ) ) // .. and it's a CLOSER "right neighbour"
            { i64SampleIndex2 = pciSrc->i64TotalSampleCounter;
              pci2 = pciSrc;
            }
         }
      }
   } // end for

  // At this point:
  //   i64SampleIndex1 = sample index close to (but sometimes a bit OLDER) than i64WantedSampleIndex,
  //   i64SampleIndex2 = sample index close to (but  always   a bit NEWER) than i64WantedSampleIndex.

  if( (pci1!=NULL) && (pci2==NULL) )
   { // didn't find a NEWER entry than the requested one
     // -> cannot INTERPOLATE, so must EXTRAPOLATE from two OLDER entries .
     // Find the 2nd 'most recent' entry in the array.
     // pci1 remains the closer neighbour.
     // pci2 will have a slightly lower sample index than that (i.e. "a bit older").
     i64SampleIndex2 = -1;
     for(i=0; i<pChunkArray->nUsedEntries; ++i )
      {
        pciSrc = &pChunkArray->chunk_info[i];
        if(  (pciSrc->dwSizeOfStruct == sizeof(T_ChunkInfo) ) // array entry currently valid ?
          && (pciSrc->dwNrOfSamplePoints > 0 ) ) // added 2012-05-04 to exclude 'unused' entries
         { if( pciSrc->i64TotalSampleCounter < pci1->i64TotalSampleCounter )
            { // this buffer entry is OLDER than the first found neighbour..
              if(   (pciSrc->i64TotalSampleCounter > i64SampleIndex2 )
                 || (i64SampleIndex2 < 0 ) ) // .. and it's a 'better' than the previous..
               { i64SampleIndex2 = pciSrc->i64TotalSampleCounter;
                 pci2 = pciSrc;
               }
            }
         }
      } // end for
   } // end if( (pci1!=NULL) && (pci2==NULL) )

  if( (pci1 != NULL) && (pci2 != NULL) ) // found 'two neighbours' so INTERPOLATE (or extrapolate):
   {
     if( pci1 == pci2 )
      { DEBUGGER_BREAK();  // <<< set breakpoint here <<< ("assertion failed")
      }
     if( pci1->i64TotalSampleCounter != pci2->i64TotalSampleCounter )
      { dblRelativeIndex = (double)(i64WantedSampleIndex - pci1->i64TotalSampleCounter)
                  / (double)(pci2->i64TotalSampleCounter - pci1->i64TotalSampleCounter);
        // dblRelativeIndex :
        //   0.0 would be 'the left neighbour';
        //   1.0 would be 'the right neighbour';
      }
     else
      { dblRelativeIndex = 0.0;  // no need to interpolate !
      }
     return ChunkInfo_Interpolate( // Added 2011-12 for timestamped VLF streams
              pci1,  // [in] 'older' chunk info than the one requested by the caller
              pci2,  // [in] 'newer' chunk info than the one requested by the caller
              dblRelativeIndex,    // [in] explained below
              pChunkInfo);         // [out] interpolated T_ChunkInfo
   }
  else if( pci1 != NULL )
   { ChunkInfo_CopyFromTo( pci1,   // [in] T_ChunkInfo
              pChunkInfo, pChunkInfo->dwSizeOfStruct ); // [out]
     return TRUE;
   }
  else // complete failure; set a breakpoint in the line below
   { return FALSE; // << should NEVER get here (but "shit happens")
   }
} // end ChunkInfoArray_GetInterpolatedEntry()


//---------------------------------------------------------------------------
BOOL ChunkInfoToString( char *pszDest, int iMaxLen,
             T_ChunkInfo *pChunkInfo ) // [in] see \cbproj\SoundUtl\ChunkInfo.h
  // Converts a 'chunk info' (header) into a string. Used, amonst others,
  // to produce the 'additional info' field in the header of a written audio file.
  //
  // Called when writing certain audio files, for example
  //        from c:\cbproj\SoundUtl\CWaveIO.cpp :: C_WaveIO::WriteHeader().
  //
  // The 'additional info string' may contain (depending on the config) :
  //    * Audio sampling rate (more precise than the wave format header permits)
  //    * precise GPS time and date of the first recorded audio sample
  //       (or, at least, UTC time and date),
  //    * current GPS longitude, latitude, and meters above sea level,
  //    * the name of a GPS ASCII logfile, if one is written along with the
  //      audio logfile (using the precise timestamps as "common index") .
{
  char *cp;
  if( (pszDest!=NULL) && (iMaxLen>80) )
   { memset(pszDest, 0, iMaxLen); // prevents trailing junk when aligning for 4 bytes (in wave header),
                                  // and doesn't cost too much here .
     cp = pszDest;
     sprintf(cp, "sr=%.12lg",  pChunkInfo->dblPrecSamplingRate ); // see details below on the resolution (*)
     cp += strlen(cp);

     if( pChunkInfo->dblRadioFrequency != 0.0 ) // append 'Radio Frequency' (tuner freq. offset) ?
      { sprintf(cp, " rf=%.9lf", pChunkInfo->dblRadioFrequency ); // see details below on the resolution (*)
        cp += strlen(cp);
      }
     // (*) About the required frequency resolution (above):
     //     To measure the phase of DCF77 signal over 24 hours with <= 10 "phase error",
     //     one RF carrier cycle = 1 / 77.5 kHz       = 13 microseconds.
     //     10/360rd of a cycle  = 10 / (360*77.5kHz) = 360 nanoseconds.
     //     max "tolerable" timing error in 24 hours  = 360 ns;
     //        Ratio = (24*60*60) s / 360ns = 249 000 000 000 (2.4e+11) .
     //                                      |_______________| 12 digits
     //     77.5kHz / 2.4e11 = 323 nHz (Nano-Hertz!),
     //               requires at least one more decimal place ("after the point")
     //               so to get it straight "rf" and "sr" have 1 nHz resolution now.
     //    Justified for "double" (64 bit total, 53 bit mantissa) ? Of course:
     //  > Eine Przision von 53 Bit bedeutet umgerechnet ins Dezimalsystem
     //  > eine ungefhre Genauigkeit auf 16 Stellen im Dezimalsystem
     //  > (53 * log10(2) = approx 16)

     if( pChunkInfo->ldblUnixDateAndTime > 0.0 ) // append "Unix date and time" ?
      { // Unit : "seconds passed since Januar 1st, 1970, 00:00:00.0" (in UTC).
        // OLD resolution: in the microsecond range :
        // ex: sprintf(cp, " ut=%018.7lf", pChunkInfo->ldblUnixDateAndTime );
        // According to Borland's documentation on 'printf', "%Lf" is for 'long double',
        // Borland's "long double" (with 15 bit exponent and 64 bit mantissa)
        // is good enough for sub-nanosecond resolution even for the "Unix date and time".
        // Thus, since 2015-01-07 :
        sprintf(cp, " ut=%020.9Lf", (long double)pChunkInfo->ldblUnixDateAndTime );
        // Test result (2015-01-07) : cp = " ut=1420668411.271888124"
        cp += strlen(cp);
      }

     if( pChunkInfo->gps.dblLat_deg > C_CHUNK_INFO_LAT_INVALID )  // geographic position valid ?
      {
        sprintf(cp, " glat=%.5lf glon=%.5lf masl=%.1lf kmh=%.1lf",
           pChunkInfo->gps.dblLat_deg, // geographic latitude in degrees, positive = NORTH
           pChunkInfo->gps.dblLon_deg, // geographic longitude in degrees, positive = EAST
           pChunkInfo->gps.dbl_mASL ,  // meters above sea level (from GPS receiver)
           pChunkInfo->gps.dblVel_kmh); // GPS velocity in km/h
        cp += strlen(cp);
      }
     UNREFERENCED( cp );  
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end ChunkInfoToString()


//---------------------------------------------------------------------------
BOOL StringToChunkInfo( char **ppszSource,
             T_ChunkInfo *pChunkInfo ) // [out] see \cbproj\SoundUtl\ChunkInfo.h
  // More or less "inverse function" (parser) for ChunkInfoToString() .
  // IMPORTANT: Fields which are NOT specified (in the string)
  //            are NOT modified in *pChunkInfo !
  //  For details on the string format, see ChunkInfoToString() .
{
  BOOL fContinue = TRUE;
  // double d;
  int token_index;
  char *cp;

  if( (ppszSource!=NULL) && (*ppszSource!=NULL) && (pChunkInfo!=NULL) )
   { cp = *ppszSource;
     while( fContinue )
      {
        token_index = QFile_SkipStringFromList( &cp,
           // token:   #0   #1   #2   #3     #4     #5     #6
                      "sr=\0rf=\0ut=\0glat=\0glon=\0masl=\0kmh=\0\0" );
        switch(token_index)
         { case 0: // sr= precise sampling rate
              pChunkInfo->dblPrecSamplingRate = QFile_ParseDFloat( &cp,
                      /*default:*/ pChunkInfo->dblPrecSamplingRate );
              // Added 2015-03 : If the sampling rate was parsed (possible from the "inf1" header in a wave file),
              //                 assume it's 'precise' (otherwise it wouldn't be there):
              if( pChunkInfo->dblPrecSamplingRate > 0.0 )
               {  pChunkInfo->dwValidityFlags |= (CHUNK_INFO_SAMPLE_RATE_VALID|CHUNK_INFO_SAMPLE_RATE_CALIBRATED);
               }
              break;
           case 1: // rf= radio frequency (added to the "baseband" frequency somewhere)
              pChunkInfo->dblRadioFrequency = QFile_ParseDFloat( &cp,
                      /*default:*/ pChunkInfo->dblRadioFrequency );
              break;
           case 2: // ut= unix time (and date, seconds since 1970)
              pChunkInfo->ldblUnixDateAndTime = QFile_ParseLongDoubleFloat( &cp,
                      /*default:*/ pChunkInfo->ldblUnixDateAndTime );
              // Added 2015-03 : If the timestamp was parsed (possible from the "inf1" header in a wave file),
              //                 assume it's 'precise' (otherwise it wouldn't be there):
              if( pChunkInfo->ldblUnixDateAndTime > 0.0 )
               {  pChunkInfo->dwValidityFlags |= (CHUNK_INFO_TIMESTAMPS_VALID|CHUNK_INFO_TIMESTAMPS_PRECISE);
               }
              break;
           case 3: // glat= geographic latitude in degrees
              pChunkInfo->gps.dblLat_deg = QFile_ParseDFloat( &cp,
                      /*default:*/ pChunkInfo->gps.dblLat_deg );
              break;
           case 4: // glon= geographic longitude in degrees
              pChunkInfo->gps.dblLon_deg = QFile_ParseDFloat( &cp,
                      /*default:*/ pChunkInfo->gps.dblLon_deg );
              break;
           case 5: // masl= meters above sea level
              pChunkInfo->gps.dbl_mASL = QFile_ParseDFloat( &cp,
                      /*default:*/ pChunkInfo->gps.dbl_mASL );
              break;
           case 6: // kmh= velocity in kilometers per hour
              pChunkInfo->gps.dblVel_kmh = QFile_ParseDFloat( &cp,
                      /*default:*/ pChunkInfo->gps.dblVel_kmh );
              break;
           default: // MAY be the end of the "chunk info" (as string)  ...
              while( *cp>='a' && *cp<='z' )
               { ++cp;
               }
              if( *cp != '=' )
               { fContinue = FALSE;
                 break;
               }
              else // skip the unknown, hopefully NUMERIC, value :
               { ++cp;
                 (void)QFile_ParseDFloat( &cp, 0 );
               }
              break;
         } // end switch(token_index)
      } // end while
     *ppszSource = cp;
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end StringToChunkInfo()



/* EOF < ChunkInfo2.c > */
