/***************************************************************************/
/*  File:   c:\cbproj\SoundUtl\ChunkInfo.h                                 */
/*  Copies: c:\cbproj\WSQ2\ChunkInfo.h (DL4YHF's adaption of ZL2AFP's WSQ) */
/*   Definition of a data structure which describes an "AUDIO CHUNK".      */
/*   The CHUNK itself is an array of floating point values, used in        */
/*   Spectrum Lab and some external modules and DLLs .                     */
/*  The T_ChunkInfo defined below contains additional information about    */
/*   the processed chunk, including number of samples, sampling rate, etc. */
/*                                                                         */
/*   - to be compiled with Borland C++Builder V4.0  for Win95/98/ME/XP     */
/*   - by DL4YHF 2004 ... 2014                                             */
/*                                                                         */
/*  Caution: The data types (structures) defined in this file are used     */
/*     in a few DLLs (libraries), so be careful to RECOMPILE THE LIBRARIES */
/*     after modifying these structs. Ideally, new components are APPENDED */
/*     to the structures defined below .                                   */
/*  Since 2011-06, the T_ChunkInfo structure is ALSO used in the           */
/*     Audio-I/O-DLLs, so be EXTREMELY CAREFUL when modifying it,          */
/*     and use ChunkInfo_CopyFromTo() where applicable (see ChunkInfo.c) ! */
/***************************************************************************/

// #error dobini    // <- got here when compiling SpecMain.cpp ..

#ifndef _CHUNK_INFO_H_
#define _CHUNK_INFO_H_

// #error dobini    // <- .. but did NOT get here when compiling SpecMain.cpp !

#ifndef  DWORD
// ex: # define DWORD unsigned long // unsigned 32 bit integer, formerly defined in "windef.h"
# include <windows.h>
#endif

#ifndef  LONGLONG
# define LONGLONG __int64
#endif

#ifndef  BOOL
# define BOOL int
#endif

#ifndef  SHORT
# define SHORT short int
#endif

#ifndef  FALSE
# define FALSE 0
#endif

#ifndef  TRUE
# define TRUE  1
#endif



/*----------------------- Data Types --------------------------------------*/

typedef double TRecTime;  // RELATIVE recording time .
    // "Count of SECONDS since the start of the current analysis"
    //         (not an "absolute" time in contrast to the "UNIX"-timestamp !!!)
    // Based strictly on number of samples (derived from the input device's sampling
    // rate, which may be VERY accurate if an external converter is used).
    // Unit is SECONDS.  Starts at zero when the analysis starts.
    // Used as a common timebase for trigger events in the real-time
    // audio processing thread.
    // All variables not using this system must not be declared as TRecTime !
    // Can be converted to a calendar time via SndThd_RecTimeToRealTime() .
    // Can also be converted into the Pentium CPU's "performance timer" .
    // The meaning of "TRecTime" doesn't depend on real-time mode / file analysis,
    // the only difference is the conversion from TRecTime into an absolute time,
    // which makes SndThd_RecTimeToRealTime() more complicated than you think ;-)
typedef struct // T_ChunkInfoGPS : GPS-related data within a T_ChunkInfo struct
{
  int    iUpdateCounter; // counter for 'new' GPS positions, used by some FILE WRITERS
                         // to determine whether a new position shall be 'emitted' or not
  int    iAlignmentDummy;
  double dblLat_deg;  // geographic latitude in degrees, positive = EAST, <-999 = INVALID
#        define  C_CHUNK_INFO_LAT_INVALID -999.999999 /* 'invalid' value for dblLat_deg */
  double dblLon_deg;  // geographic longitude in degrees, positive = NORTH
#        define  C_CHUNK_INFO_LON_INVALID -99.999999  /* 'invalid' value for dblLon_deg */
                      // (These 'invalid' values still suit the default display format)
  double dbl_mASL;    // meters above sea level (from GPS receiver)
  double dblVel_kmh;  // GPS velocity in km/h
  double dblDateAndTime; // time for which the above GPS position is valid (also UNIX format)

} T_ChunkInfoGPS;  // sizeof (T_ChunkInfoGPS) should be 6 * 8  =  48 [bytes]

typedef struct // T_ChunkInfo : describes a block of audio samples + recording parameters
{
  // DO NOT MODIFY THE LAYOUT OF THE FIRST 256 BYTES IN THIS STRUCTURE !

  DWORD dwSizeOfStruct; // added 2011-06-21 avoid problems with DLLs (etc) in all FUTURE versions
       // of the T_ChunkInfo structure, used in Spectrum Lab, the Audio-I/O-DLLs, etc .
       // T_ChunkInfo.dwSizeOfStruct is now used in ChunkInfo_CopyFromTo(),
       // when the SOURCE and DESTINATION structure are not compatible.
       // The 'struct version' (compatibility) can be told by comparing
       //      sizeof(T_ChunkInfo) with TChunkInfo.dwSizeOfStruct :
       //      Ideally, both are equal, which means the DLL and the host (SL)
       //      were compiled using the same (compatible) T_ChunkInfo .
       // NOTE: T_ChunkInfo.dwSizeOfStruct hasn't got anything to do
       //       with the size of an audio sample block !
  DWORD dwValidityFlags;    // Bit combination of the flags defined below.
       // Also used as 64-bit alignment dummy for the next 'LONGLONG' member.
       // These bits should be set by the AUDIO WRITER;
       //  the audio-I/O-libs (like in_AudioIO.dll) will not change them !
// ex: #define CHUNK_INFO_SAMPLE_FORMAT_VALID   0x00000001 /* removed; the 'sample format' must ALWAYS be valid */
#      define CHUNK_INFO_RESERVED_BIT           0x00000001
#      define CHUNK_INFO_SAMPLE_RATE_VALID      0x00000002
#      define CHUNK_INFO_SAMPLE_RATE_CALIBRATED 0x00000004 /* .. against GPS or at least an OCXO */
#      define CHUNK_INFO_TIMESTAMPS_VALID       0x00000008 /* applies to .rtRecTime, .ldblUnixDateAndTime */
#      define CHUNK_INFO_TIMESTAMPS_PRECISE     0x00000010 /* .ldblUnixDateAndTime "precise" to ONE AUDIO SAMPLE (or better) */
#      define CHUNK_INFO_PRECISION_NOT_AVAILABLE  0x00000020 /* set if CHUNK_INFO_SAMPLE_RATE_CALIBRATED and CHUNK_INFO_TIMESTAMPS_PRECISE will NEVER be set, because..*/
                // .. for example, because no GPS is connected at all */
#      define CHUNK_INFO_TIMING_BREAK_DETECTED  0x00000040
            // When set, SL's "input pre-processor" has detected a timing break.
            // Will be cleared again when a sufficent number of samples
            // has been 'injected' (by the 'timed resampler' between L0 and L1)
            // to fill the gap, without making the audio chunks larger
            // than T_ChunkInfo.dwMaxNrOfSamplePoints .
            // (that's one of the reasons why under "normal conditions"
            //  dwNrOfSamplePoints should be 25% less than dwMaxNrOfSamplePoints)


  DWORD dwNrOfSamplePoints; // Number of sample points IN THE CHUNK (not the 'total number of samples processed so far').
       // Caution, there may be MORE THAN ONE "T_Float" per sample point (in a multi-channel chunk).
       // Automatically set in AIO_ReadInputSamplePoints() to the number of samples read.
       // May be modified inside the processing chain, for example when decimating/interpolating.
       // The MAXIMUM(!) number of samples which any 'writer' can place
       // in the chunk is specified in dwMaxNrOfSamplePoints (further below).
       //

  DWORD nChannelsPerSample; // aka "components per sample point"; for example:
       //  '1' for REAL data,
       //  '2' for COMPLEX data or "stereo" (which ? Depends on dwFlags.CHUNK_INFO_FLAG_I_Q)
       //  '4' for TWO SDR streams with COMPLEX data, etc .
       //
       // Definition of 'sample point'
       //  (and why variables like "nSamples" should better renamed to "nSamplePoints")
       //
       // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       // | A 'sample point' is considered a group of samples, with   |
       // | possibly multiple channels, all sampled at the same time. |
       // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
       //
       // Thus, a 'sample point' is what microsoft (quite arbitrary) call "block"
       //       in a wave file's  'format' chunk :
       //       "wBlockAlign" -> "number of bytes per sample-point".
       //
       // If a subroutine receives TWO SEPARATE BLOCKS of samples,
       //  and ONE T_ChunkInfo describes the format for BOTH,
       //  T_ChunkInfo.nChannelsPerSample will still be TWO (even though each
       //  of the two SEPARATE blocks only contains samples for ONE channel.
       //  At least, that's the case in cbproj/SpecLab/SoundThd.cpp
       //  when calling SOUND_DecimatingSaveBuffer.EnterSamples() .
       //  Quoted from there:
       // > Even if SOUND_DecimatingSaveBuffer.EnterSamples() receives the input
       // > in TWO SEPARATE BLOCKS (with each block only containing ONE channel),
       // > T_ChunkInfo.nChannelsPerSample is TWO when SL runs in 'stereo' mode.
       //  The same applies to other ("old") signal processing functions which
       //  require input in two SEPARATE blocks. In the interest of flawless
       //  'multi channel capability', new signal processing functions
       //  should take only ONE T_Float* as input, and multiple channels
       //  grouped (interleaved) in a single 'sample point' as defined above.

  // size of the above "DWORD" members: 4*4 = 16 bytes .
  // At this point, we're 8-byte aligned.

  LONGLONG i64TotalSampleCounter;  // 64-bit TOTAL sample counter / index of the 1st sample in this chunk.
       // Starts counting SAMPLE POINTS at zero when the analysis
       //                             of a file or stream starts .
       // Added 2010-04-26; used as function parameter for the
       // sampling rate correction. Any subroutine can "ask"
       // for the *corrected* sampling rate of a particular sample
       //  if it needs to - see SndThd_GetCorrectedSampleRateForSampleIndex() .
       // The next member - dblPrecSamplingRate - strictly
       // applies to the sample in the CENTER of the chunk;
       // i.e. at sample index < i64TotalSampleCounter + dwNrOfSamplePoints/2 > .
       // (but for most if not all applications, it's ok
       //  to treat all samples in the chunk as if they were
       //  sampled at the same sampling rate; since the typical
       //  chunk size is between 4096 and 16384 ).
       // This counter will NEVER overflow during the lifetime of a computer :
       // 2^64 / (192000 * 60 * 60 * 24 * 365 ) =: 3046575 years .

  double dblPrecSamplingRate; // Precise sampling rate (corrected or "calibrated"), unit: HERTZ.
             // Often something like 11024.567,  44098.268 , etc. MAY CHANGE FROM CHUNK TO CHUNK !
  double dblRadioFrequency;   // frequency to which the "radio" (or similar equipment)
                              // was tuned, when this block of samples was recorded .
  double dblSampleValueRange; // originally 32767.0 (for historic reasons; 16-bit input).
             // Later (2011) changed to 1.0, i.e. the sample blocks
             // use floating point values normalized to -1.0 ... +1.0 .
             // Caution, some older modules may still use +/- 32k value range !
             // If the audio writer decides NOT to use the normalized range,
             // he MUST provide a T_ChunkInfo (for AIO_WriteOutputSamplePoints)
             // with at least a valid sample-value-range !
  double dblTotalGroupDelay; // group delay, added from each stage in the processing chain
  double dblSamplingRateDrift; // [Hz per second], usually 0.0 (added 2012-02)

  double dblCurrentLocalUnixDateAndTime; // value returned by TIM_GetCurrentUnixDateAndTime(), used for IO_AddTimedAction() .
           // To synchronize the switching of digital outputs (RS232)
           // with the modulation generated by the 'digimode terminal'
           //  (example: c:\cbproj\Digimodes\ask_fsk_mod.cpp : C_ASK_FSK_Mod::SetSymbol() ),
           // the timestamps passed to IO_AddTimedAction() must not be much older
           // than the current *REAL TIME* (read by TIM_GetCurrentUnixDateAndTime() ).
           // To avoid timing breaks in the important ("jumpless") T_ChunkInfo.ldblUnixDateAndTime,
           // the T_ChunkInfo now *also* contains a field with the real-time value:
           //     T_ChunkInfo.dblCurrentLocalUnixDateAndTime, set in SoundThd.cpp
           //  immediately after acquiring a new block of samples for processing.
           // This time dblCurrentLocalUnixDateAndTime may be subject to severe
           // jitter, bad resolution, but it must be precise enough
           // to generate Morse code, or slow BPSK on the serial port outputs.

  double dblOutputLatency_sec; // time [seconds] which will elapse between
           // a sample's T_ChunkInfo.dblCurrentLocalUnixDateAndTime' (see above)
           // and the time when THAT audio sample will appear at the output
           //              of the soundcard (or D/A converter, or similar).
           // May have already been 'corrected' [in SoundThd.cpp]
           // by adding UConfig.iAudioOutputLatencyCompensation_ms .
           // First used in combination with IO_AddTimedAction()
           // to switch a digital output (on RS-232) at the correct time.
           // Example: Also in c:\cbproj\Digimodes\ask_fsk_mod.cpp : C_ASK_FSK_Mod::SetSymbol() .

  // size of the above "LONGLONG" and "double" members: 8*8 = 64 bytes
  // At this point, we're 8-byte aligned.

  DWORD  dwInfoFlags;    // info flags for 'processing'; a bit combination of the following:
#        define CHUNK_INFO_FLAG_NONE                           0
#        define CHUNK_INFO_FLAG_I_Q                   0x00000002 // "I+Q", aka complex signal
#        define CHUNK_INFO_FLAG_LOCK_PHASES_TO_TIME   0x00000010
            // When set, use .lblUnixDateAndTime instead of the 'sample counter'
            //  to calculate sinewave, local oscillator signals, and similar .


  DWORD  dwMaxNrOfSamplePoints;  // MAXIMUM number of sample points which can
            // be placed in a chunk of samples ( = "allocated" size ).
            // This is important for certain RESAMPLERS, which may slightly
            // INCREASE the number of samples per chunk (since 2012-01-01) .
            // For compatibilitiy: If dwMaxNrOfSamplePoints is zero,
            // a processing stage may NOT change the number of samples
            // in a chunk (i.e. dwNrOfSamplePoints is 'read only' then) .

  char   sz80Command[88];   // a 'command' for certain processing stages (text)
  // size of the above 3 fields: 4+4+88 = 96 bytes


  TRecTime rtRecTime; // Recording time (relative to start of recording) .
                      // Applies to the *FIRST* sample in the chunk .
                      // Unit: SECONDS.  TRecTime is in fact 'double' (64 bit float).

  long double ldblUnixDateAndTime; // 'absolute' recording-time and -date of 1st sample in chunk,
          // Unit : "seconds passed since January 1st, 1970, 00:00:00.0" (in UTC).
          // Resolution: with 'double' (52-bit mantissa) this would be in the microsecond range,
          //    1200000000 (seconds in 38 years since Unix birthdate) / (2^52)
          //     = 2.6e-7 = better than a microsecond -> sufficient for most,
          //     but not all cases (e.g. not ok for absolute phase measurements).
          // After changing to Borland's "long double", which has 64 bit MANTISSA
          //     and a 15 bit exponent, the resolution is roughly 70 ps (!) :
          //     40 [years] * 365 * 24 * 60 * 60 / (2^64)  =  6.8e-11 [seconds]
          //   IN AN ALIGNED STRUCT, a "long double" will occupy more than
          //     80 bits .
          // Note: During file-analysis, dblUnixDateAndTime is not related
          //       to the PC's "local clock" or similar !!
          //    This member is POSSIBLY TIED TO THE OUTPUT FROM A GPS RECEIVER.
  // ex: double dblSecondsOfDay; // 'SECONDS' of the current day, for more resolution.  Removed (since long double ldblUnixDateAndTime has 70 ps resolution, see above)
          // Value range: 0 to 24*3600-epsilon.  Resolution: better than 1ns .
          //   ( 24*60*60/(2^52)  =: 1.91847e-11 )
          // Details (and usage) see c:\CBproj\SoundUtl\audiomsg.h .
          // Like dblUnixDateAndTime, this timestamp refers to
          // the first "audio" sample in the chunk (block) of audio samples.
          // If the bit 'CHUNK_INFO_FLAG_LOCK_PHASES_TO_TIME'
          // is set in .dwInfoFlags, some 'oscillator outputs' may be phase-locked
          // to this value (see SoundThd.cpp::SOUND_RunSignalGenerators)

          // ex: double dblHRTime_sec; // value read by "TIM_QueryHRTimer_sec()" when acquiring the 1st sample in chunk.
               // (dblHRTime_sec can be replaced by .ldblUnixDateAndTime if really needed,
               //  see notes on 'TIM_dblPerformanceCounterToTimeOffset' in C:\cbproj\SpecLab\Timers.cpp )
          LONGLONG i64Reserve; // planned: i64UnixDateAndTime_nsec;
          // same purpose as ldblUnixDateAndTime,
          // but this is the number of NANOSECONDS elapsed since 1970-01-01 00:00:00.0 UTC,
          // which may be better suited for compilers which don't support Borland's
          //


          // Returns the current count from some "high-resolution timer" in SECONDS.
          // May be NEGATIVE or ZERO when not supported by the audio source !
          // Totally useless during FILE ANALYSIS (only during real-time processing).
          // Intended to measure SHORT INTERVALS (relative), but not as an absolute value.
          // Originally QueryPerformanceCounter() was used internally but may be replaced
          // by something "better" as soon as the author knows more !
          // Beware of the weaknesses of QueryPerformanceCounter() on multi-processor machines !
          // Only use i64HRTime if absolutely necessary, for example in the DCF77-decoder when trying to
          //  "calibrate" the frequency of the high-resolution timer with a longwave time signal .

  // size of the above field of 'double' precision floats / times : 4*8 = 32 bytes
  // Byte offset at this point (with all above fields) :
  //             16 + 64 + 96 + 32 = 208
  // ( can be checked in BCB with the following WATCH-expression in the debugger:
  //     (int)(&MyChunkInfo.gps) - (int)&MyChunkInfo  )

  T_ChunkInfoGPS gps; // GPS related part of the T_ChunkInfo struct (48 bytes)

} T_ChunkInfo; // sizeof (T_ChunkInfo) should be  208 + 48 = 256 [2^8 bytes] .
    // This is extremely important because T_ChunkInfo is used in some DLLs,
    // and in an application which uses that DLL, and both may be compiled with
    // DIFFERENT C compilers ..Borland + GCC.. thus the byte-counting exercise .

// Size of T_ChunkInfo in different versions :
#define CHUNK_INFO_STRUCT_SIZE_V1 256
#if(sizeof(T_ChunkInfo) != CHUNK_INFO_STRUCT_SIZE_V1 )
#  error "Someting fishy with T_ChunkInfo !"
#endif



// Array of T_ChunkInfo for interpolation :
#define CHUNK_ARRAY_MAX_ENTRIES 32
typedef struct
{ T_ChunkInfo chunk_info[CHUNK_ARRAY_MAX_ENTRIES];
  int         iNewestEntry;
  int         nUsedEntries;
  LONGLONG    i64PrevSampleCounter;
} T_ChunkArray;

   // 2015-09-23, after copying "everything" from the dead Z61m :
   // > E2303 Type Name expected (i.e. T_ChunkArray not found) IN SEVERAL PLACES .
   // T_ChunkArray should have been defined in c:\cbproj\SoundUtl\ChunkInfo.h .
   // Reason: On the X61s, an outdated copy of ChunkInfo.h still existed
   //         in C:\cbproj\WSQ2 . ChunkInfo.h MUST NOT EXIST THERE (as a copy) !
   //    -> deleted all evil twins of c:\cbproj\SoundUtl\ChunkInfo.h ,
   //       and the project could flawlessly be compiled again .
// #error dobini

#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
# ifdef __cplusplus
#   define CPROT extern "C"
# else
#   define CPROT
# endif  /* ! "cplusplus"  */
#endif  /* ! defined CPROT */


CPROT void ChunkInfo_Init( T_ChunkInfo *pChunkInfo );
CPROT void ChunkInfo_CopyFromTo( T_ChunkInfo *pSrcChunkInfo,     // [in]
              T_ChunkInfo *pDstChunkInfo, int iDstSizeInBytes ); // [out]
CPROT void ChunkInfo_AdjustInfoForSampleIndexDiff(
              T_ChunkInfo *pStoredChunkInfo, double dblSampleIndexDiff, // [in]
              T_ChunkInfo *pAdjustedChunkInfo); // [out]

CPROT BOOL ChunkInfo_Interpolate( // Added 2011-12 for timestamped VLF streams
              T_ChunkInfo *pInChunkInfo1,  // [in] 'older' chunk info than the one requested by the caller
              T_ChunkInfo *pInChunkInfo2,  // [in] 'newer' chunk info than the one requested by the caller
              double dblRelativeIndex,     // [in] explained below
              T_ChunkInfo *pInterpolated); // [out] interpolated T_ChunkInfo
   // dblRelativeIndex :
   //   0.0 would return *pInChunkInfo1;
   //   1.0 would return *pInChunkInfo2;
   //      everything else should be "in between" (0..1 = interpolate) .
   //   dblRelativeIndex < 0 would extrapolate "into the past",
   //   dblRelativeIndex > 0 would extrapolate "into the future".

CPROT void         ChunkInfoArray_Init(T_ChunkArray *pChunkArray);
CPROT T_ChunkInfo *ChunkInfoArray_Append(T_ChunkArray *pChunkArray, T_ChunkInfo *pChunkInfo, double dblHistoryLen_s);
CPROT BOOL         ChunkInfoArray_GetInterpolatedEntry(T_ChunkArray *pChunkArray, T_ChunkInfo *pChunkInfo, LONGLONG i64WantedSampleIndex);

CPROT BOOL ChunkInfoToString(char *pszDest, int iMaxLen, T_ChunkInfo *pChunkInfo); // implemented in ChunkInfo2.c !
CPROT BOOL StringToChunkInfo( char **ppszSource, T_ChunkInfo *pChunkInfo );        // implemented in ChunkInfo2.c !



#endif // _CHUNK_INFO_H_
