/***************************************************************************/
/*  CWaveIO.h :                                                            */
/*   Class to import and export *.wav - files.                             */
/*   - by DL4YHF August '2000                                              */
/*   - based on the description of the WAVE-file format                    */
/*     found at http://www.wotsit.org  (excellent site, all file formats!  */
/*-------------------------------------------------------------------------*/

#define WAVEIO_USE_QFILE 1

#ifndef _WAVE_IO_H_
#define _WAVE_IO_H_

#if( WAVEIO_USE_QFILE )
  #include "QFile.h"  // DL4YHF's "quick file" routine (read+write text files, too)
#endif

#ifndef  _AUDIO_FILE_DEFS_H_
# include "AudioFileDefs.h"
#endif

#ifndef _CHUNK_INFO_H_
# include "ChunkInfo.h" // T_ChunkInfo with sample rate, date+time, 'radio frequency', GPS position, ...
  // originally located in c:\cbproj\SoundUtl\ChunkInfo.h . 
#endif

/*----------- Constant Definitions ----------------------------------------*/
 // -> most moved to AudioFileDefs.h !




/*----------- Data Types --------------------------------------------------*/


#ifndef  SWI_FLOAT_PRECISION   // should be defined under Project..Options..Conditionals
 #define SWI_FLOAT_PRECISION 1 /* 1=single precision, 2=double precision */
#endif

#ifndef T_Float /* T_Float defined in SoundTab.h, SoundMaths.h, Sound.h, WaveIO.h, ... */
#if SWI_FLOAT_PRECISION==1
 #define T_Float float
 // 2003-07-07: "float" instead of "double" saves CPU power on stoneage notebooks
#else
 #define T_Float double
#endif
#endif // ndef T_Float

// A RIFF file has an 8-byte RIFF header, identifying the file,
// and giving the residual length after the header (i.e. file_length - 8):
typedef struct
{
  char  id_riff[4];   // identifier string = "RIFF"
  DWORD len;          // remaining length after this header
  // The riff_hdr is immediately followed by a 4-byte data type identifier.
  // For .WAV files this is "WAVE" as follows:
  char wave_id[4];    // WAVE file identifier = "WAVE"
} T_WAV_RIFF_HDR;


// RIFF Chunks
//   The entire remainder of the RIFF file is "chunks".
//   Each chunk has an 8-byte chunk header identifying the type of chunk,
//   and giving the length in bytes of the data following the chunk header,
//   as follows:

typedef struct {      // CHUNK 8-byte header
  char  id[4];        // identifier, e.g. "fmt " or "data" , sometimes "inf1" (proprietary)
  DWORD dwChunkSize;  // remaining chunk/junk:) length after header
                      // data bytes follow chunk header .
     // Note: 32 bit unsigned for the "chunk size" limit the size of a wave file
     //       to 4 GBytes (or GiB, if you're a dedicated follower of fashion) !
} T_WAV_CHUNK_HDR;


// The WAVE form is defined as follows. Programs must expect(and ignore)
// any unknown chunks encountered, as with all RIFF forms.
// However, <fmt-ck> must always occur before <wave-data>, and both of these
// chunks are mandatory in a WAVE file.
//
// WAVE-form> ->
//	RIFF( 'WAVE'
//	<fmt-ck>     		// Format
//	[<fact-ck>]  		// Fact chunk
//	[<cue-ck>]  		// Cue points
//	[<playlist-ck>] 		// Playlist
//	[<assoc-data-list>] 	// Associated data list
//	<wave-data>   ) 		// Wave data
//

// WAVE Format Chunk
//  The WAVE format chunk <fmt-ck> specifies the format of the <wave-data>.
//  Since I only work with PCM format, this structure is as follows
//   (combination of <commond-fields> and <format-specific-fields>):
typedef struct {
  // The first 16 bytes form the classic 'WAVEFORMAT' structure :
  WORD  wFormatTag;         // Format category,    1=Microsoft PCM (=INTEGER!), etc:
#    define WAVE_FORMAT_PCM        1 // "pulse code modulation", always INTEGER (!), stupid standard.. (*)
#    define WAVE_FORMAT_IEEE_FLOAT 3 // also an uncompressed PCM, but this is IEEE floating point, and according to Microsoft it's "NOT PCM" ! (*)
#    define WAVE_FORMAT_ALAW       6 // 8-bit ITU-T G.711 A-law (not supported here)
#    define WAVE_FORMAT_MULAW      7 // 8-bit ITU-T G.711 -law (not supported here)
#    define WAVE_FORMAT_EXTENSIBLE 0xFFFE // format determined by SubFormat
     // (*) About Non-PCM Formats, from www-mmsp.ece.mcgill.ca/Documents/AudioFormats/WAVE/WAVE.html :
     // > An extended Format chunk is used for non-PCM data.
     // > The cbSize field gives the size of the extension.
     // > For all formats other than PCM, the Format chunk must have an extended portion.
     // > The extension can be of zero length, but the size field (with value 0) must be present.
     // > For float data, full scale is 1. The bits/sample would normally be 32 or 64.
     // > For the log-PCM formats (-law and A-law), the Rev. 3 documentation
     // > indicates that the bits/sample field (wBitsPerSample) should be set to 8 bits.
     // > The non-PCM formats must have a Fact chunk.  (!!!!)
     // ( in fact, the stupid windows-something-player refused to play wave files
     //   with floating-point samples if there was no "Fact" chunk.
     //   Details about that unnecessary annoyance in CWaveIO.cpp )
  WORD  wChannels;          // Number of channels, 1=mono, 2=stereo
  DWORD dwSamplesPerSec;    // Sampling rate     , samples per second
  DWORD dwAvgBytesPerSec;   // For buffer estimation
  WORD  wBlockAlign;        // Data block size ?? Bytes per Sample ????
              // (a misnomer, like many others. Actually 'wBlockAlign'
              //  is the NUMBER OF BYTES FOR A "SAMPLE POINT", digitized
              //  at ONE sampling step, with <wChannels> channels PER 'BLOCK'.
              //  See calculation of 'wBlockAlign' in C_WaveIO::OutOpen() )
  WORD  wBitsPerSample;     // Sample size, usually 8 or 16 bits per sample,
              // some M$ article says "Rigidly Defined as Container Size" (?)
              // (note that their 'SAMPLE' only contains a single channel,
              //  not what WB preferred to call "SAMPLE POINT")

  // After the classic 'WAVEFORMAT', there is another 16-bit WORD called cbSize:
  WORD   cbSize;   // added 2010-05-27 (to read 24-bit wave audio files) .....
  // <<<<  this is the end of what microsoft calls 'WAVEFORMATEX'  .
  //       Note that 'WAVEFORMATEX' includes the 'cbSize' member .
  //   If wFormatTag is 1 (WAVE_FORMAT_PCM),     the format is the classic "Microsoft PCM".
  //   If wFormatTag is 65534 (WAVE_FORMAT_EXTENSIBLE), the format is 'WAVEFORMATEXTENSIBLE';
  //      in that case, the WAVEFORMATEX (above) is followed by these OPTIONAL
  //      fields, which we must know to read exotic files with 24 bits/sample, etc.
  //   Note that due to the bastardized 'wave file' standard,
  //   the WAVEFORMATEXTENSIBLE-thingy cannot be used with FLOATING POINT samples
  //   because FLOATING POINT means wFormatTag = WAVE_FORMAT_IEEE_FLOAT,
  //                       and with wFormatTag == WAVE_FORMAT_IEEE_FLOAT,
  //     wFormatTag can neither be WAVE_FORMAT_PCM nor WAVE_FORMAT_EXTENSIBLE,
  //     and all this because some egghead at microsoft didn't understand
  //     what 'PCM' means : pulse code modulation, which shouldn't depend
  //     on the DATA TYPE (integer versus floating point), but that's the
  //     way things are.    See also: CWaveIO.cpp about the "Fact"-thing.
  union {     // specific to the WAVEFORMATEXTENSIBLE-thingy.. only for INTEGER samples, reasons explained above !
    WORD wValidBitsPerSample;  /* bits of precision            */
    WORD wSamplesPerBlock;     /* "valid if wBitsPerSample==0" */
    WORD wReserved;            /* If neither applies, set to zero. */
   } Samples;
  DWORD dwChannelMask;     /* which channels are present in stream */
  /*GUID*/ BYTE b16GUID[16]; // you really don't want to fool around with this :)
     // > GUIDs identify objects such as interfaces,
     // > manager entry-point vectors (EPVs), and class objects.
     // > A GUID is a 128-bit value consisting of one group
     // > of 8 hexadecimal digits, followed by three groups of 4 hexadecimal
     // > digits each, followed by one group of 12 hexadecimal digits.
   // The total size of the "WAVE_FORMAT_EXTENSIBLE" structure should be 40
   //  - at least that's what the 'RIFF chunk header' indicated in a wave
   //    file with 24 bits/sample -> WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE .
} T_WAV_CHUNK_FORMAT_EX_TENSIBLE;
  // the above funny-named monster can be treated like
  // a WAVFORMAT, WAVFORMATEX, or even WAVFORMATEXTENSIBLE (thus the name):
#define WAVEIO_SIZEOF_STANDARD_WAVEFORMAT_EX 16 /* found in old 16-bit stereo 'PCM' */
#define WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE 40 /* found in a 24-bit wave file  */

// The wBitsPerSample field specifies the number of bits of data used
// to represent each sample of each channel. If there are multiple channels,
// the sample size is the same for each channel.
// For PCM data, the wAvgBytesPerSec field of the `fmt' chunk should be equal
//  to the following formula rounded up to the next whole number:
//
//	                             wBitsPerSample
//	wChannels x wBitsPerSecond x --------------
//	                                   8
//
// The wBlockAlign field should be equal to the following formula,
// rounded to the next whole number:
//
//	            wBitsPerSample
//	wChannels x --------------
//	                  8
//
// Data Packing for PCM WAVE Files
//  In a single-channel WAVE file, samples are stored consecutively.
//  For stereo WAVE files, channel 0 represents the left channel,
//  and channel 1 represents the right channel. The speaker position
//  mapping for more than two channels is currently undefined.
//  In multiple-channel WAVE files, samples are interleaved.
//  The following diagrams show the data packing for a 8-bit mono
//  and stereo WAVE files:
// Data Packing for 8-Bit Mono PCM:
//
//		Sample 1	Sample 2	Sample 3	Sample 4
//		---------	---------	---------	---------
//		Channel 0	Channel 0	Channel 0	Channel 0
//
// Data Packing for 8-Bit Stereo PCM:
//
//                   Sample 1                 Sample 2
//		---------------------   ---------------------
//              Channel 0   Channel 1   Channel 0   Channel 0
//               (left)      (right)     (left)      (right)
//
// The following diagrams show the data packing for 16-bit mono
// and stereo WAVE files:
//
// Data Packing for 16-Bit Mono PCM:
//
//			Sample 1			Sample 2
//		----------------------	----------------------
//		Channel 0	Channel 0	Channel 0	Channel 0
//		low-order	high-order	low-order	high-order
//		byte 		byte		byte		byte
//
// Data Packing for 16-Bit Stereo PCM:
//
//                                 Sample 1
//               ---------------------------------------------
//               Channel 0   Channel 0    Channel 1  Channel 1
//               (left)      (left)      (right)     (right)
//               low-order   high-order  low-order   high-order
//               byte        byte        byte        byte
//
//
//
// Data Chunk
//  The Data chunk contains the actual sample frames (ie, all channels
//  of waveform data).

typedef struct {
    char   id[4];   // "data" (4 bytes)
    DWORD  chunkSize;
 // BYTE   waveformData[chunkSize-8];
} T_WAV_DATA_CHUNK;

// The ID is always data. chunkSize is the number of bytes in the chunk,
// not counting the 8 bytes used by ID and Size fields nor any possible
// pad byte needed to make the chunk an even size (ie, chunkSize is the
// number of remaining bytes in the chunk after the chunkSize field,
// not counting any trailing pad byte).
// The waveformData array contains the actual waveform data.
// The data is arranged into what are called sample frames.
// You can determine how many bytes of actual waveform data there is
// from the Data chunk's chunkSize field. The number of sample frames
// in waveformData is determined by dividing this chunkSize by the Format
// chunk's wBlockAlign.
// The Data Chunk is required. One, and only one, Data Chunk may appear
// in a WAVE.

typedef struct t_WAV_FACT_CHUNK
{ // "fact" header, rarely used, but seems unavoidable for FLOATING POINT WAVs !
  DWORD  dwSampleLength;  // "number of sampes per channel" (in the entire recording)
  // From  www-mmsp.ece.mcgill.ca/documents/AudioFormats/WAVE/WAVE.html :
  // > All (compressed) non-PCM formats must have a Fact chunk (Rev. 3 documentation).
  // > The chunk contains at least one value, the number of samples in the file.
  // > The Rev. 3 documentation states that the Fact chunk "is required
  // > for all new new WAVE formats", but "is not required  for the standard
  // > WAVE_FORMAT_PCM files". One presumes that files with IEEE float data
  // > (introduced after the Rev. 3 documention) need a Fact chunk.
  // WB: Technically, the "fact" chunk is utterly useless for NON-COMPRESSED
  //     recordings because there's nothing in this "fact"-thing which we
  //     don't already know (from the stoneage 'WAVEFORMATEX').
} T_WAV_FACT_CHUNK;


/*----------- Variables with "C"-style single-source principle ------------*/

#undef EXTERN
#ifdef _I_AM_WAVE_IO_
 #define EXTERN
#else
 #define EXTERN extern
#endif


/*----------- Definition of the C_WaveIO  class ---------------------------*/
class C_WaveIO
{
public:
   C_WaveIO();
   virtual ~C_WaveIO();

   // Public member functions for object's users.
   /************************************************************************/
   BOOL InOpen( char *pszFileName,     // [in] filename to open (for reading)
                int iAudioFileFormat,  // [in] AudioFileDefs.h: AUDIO_FILE_FORMAT_..
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength); // [in] size of pszExtraInfo
     /* Opens a wave file for READING audio samples.
      * Returns TRUE on success and FALSE on any error.
      */

   /************************************************************************/
   long   GetParameterByID( int iAudioFileParam ); // Replaces several older GET-functions.
            // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h .
            // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
            // for example GetVTFlags() -> GetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS) .
   BOOL   SetParameterByID( int iAudioFileParam, long i32NewValue ); // Replaces several older SET-functions.
            // Returns TRUE if the new value could be set, otherwise FALSE.
            // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
            // for example SetVTFlags(iVTFlags) -> SetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS,..) .
   double GetParameterByID_Double(int iAudioFileParam); // similar as above, for 64-bit "double" ..
   BOOL   SetParameterByID_Double(int iAudioFileParam, double dblNewValue);



   /************************************************************************/
   LONG ReadSamples( BYTE* pData, LONG Length );
     /* Reads some samples from a wave file which has been opened for READING.
      * "Length" is the size of the caller's buffer capacity in BYTES.
      * Return value: number of SAMPLE POINTS (intervals), not BYTES !!
      * More info: see WaveIO.cpp !
      */


   /************************************************************************/
   LONG ReadSampleBlocks(
         int channel_nr,    // channel_nr for 1st destination block
         int nr_samples,    // nr_samples (per destination block, length of destination arrays)
         T_Float *pFltDest1, // 1st destination block (~"LEFT")
         T_Float *pFltDest2, // 2nd destination block (~"RIGHT"), may be NULL
         T_ChunkInfo *pOutChunkInfo, // [out,optional] additional results, see ChunkInfo.h, may be a NULL pointer (!)
         char *pszExtraInfo, int iMaxStrlen ); // [out,optional] extra string with user data for wave.rd_info
     /* Reads some samples from a wave file which has been opened for READING
      * and converts them to double (floats) in the range of +- 32768.0 .
      * More info: see WaveIO.cpp !
      */



   /************************************************************************/
   BOOL WriteHeader( T_ChunkInfo *pChunkInfo ); 
   BOOL OutOpen( char *file_name,
           int iAudioFileFormat, // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPT_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int bits_per_sample,  // 8,16,24=integer, 32 or 64=floating point ! (for a single channel, since 2015-01-08)
           int channels,
           T_ChunkInfo *pInChunkInfo); // sample rate, date+time (here: NOT optional, ptr must not be NULL)
     /* Creates and Opens an audio file for WRITING audio samples.
      *         Returns TRUE on success and FALSE on any error.
      *         Detailed description of arguments in WaveIO.cpp only.
      */


   /************************************************************************/
   // ex: (not pivate) LONG WriteSamples( BYTE* pData, LONG Length); // low-level output, try to avoid
   LONG WriteSamples_Float( // app-layer output, using NORMALIZED floats
      T_Float *pfltSrc, T_ChunkInfo *pChunkInfo, char *pszExtraInfo );

   /************************************************************************/
   // Some more 'Get' - and 'Set' - functions for the AUDIO FILE class ..
   void   GetFileName(char *pszDest, int iMaxLen);

   double GetSampleRate(void);
   void   SetSampleRate(double dblSampleRate);

   int    GetNrChannels(void);
   void   SetNrChannels(int iNrChannels);       // important to read RAW files

   void   SetDataType(int iSizeOfDataType,   int iDataTypeFlags); // important to read RAW files
   void   GetDataType(int *piSizeOfDataType, int *piDataTypeFlags);
   int    GetBitsPerSample(void);  // per "single" sample, regardless of number of channels
   void   SetGain( double dblGainFactor );  // .. for raw files with FLOATING POINT numbers !
   double GetGain( void );

   double GetFrequencyOffset(void); // difference between "radio-" minus "baseband-" frequency (internal NCO + other internal frequency shift + external VFO )
   BOOL   SetFrequencyOffset(double dblFrequencyOffset);

   double GetCurrentRecordingTime(void); // see specification in c:\cbproj\SoundUtl\AudioFileIO.h (t=0 is the RECORDING START TIME, aka "track time")
   double GetRecordingStartTime(void);   // seconds elapsed since 00:00:00 GMT, January 1, 1970.
   void   SetRecordingStartTime(double dblStartTime);

   bool   AllParamsIncluded(void);

   /************************************************************************/
   BOOL CloseFile( void );
     /* Closes the WAV-file (if opened for READING exor WRITING.
      * Returns TRUE on success and FALSE on any error.
      */


   // Public properties for simplicity .. we know what we're doing, right ?
   T_WAV_RIFF_HDR     m_RiffHdr;
   T_WAV_CHUNK_HDR    m_ChunkHdr;
   T_WAV_CHUNK_FORMAT_EX_TENSIBLE m_ChunkFormat;  // WAVFORMAT, WAVFORMATEX, or even WAVFORMATEXTENSIBLE
   T_ChunkInfo MyChunkInfo; // DL4YHF's own 'chunk info' describes a chunk of audio samples throughout Spectrum Lab.
          // Used as destination for StringToChunkInfo() to parse the "inf1" header.
          // Contains *precise* sampling rate,
          //          *precise* center frequency,
          //          *precise* timestamp of 1st sample.
          //  (something that all those other RIFF-chunks could NOT provide)



   double m_dblStartTime;  // seconds elapsed since 00:00:00 GMT, January 1, 1970.
   int    m_iFileFormat;
   int    m_iFileMode;
   int    m_iAudioFileOptions; // AUDIO_FILE_OPTION_NORMAL etc; see c:\cbproj\SoundUtl\AudioFileDefs.h
   char   m_cErrorString[84];
   BOOL   m_OpenedForReading;
   BOOL   m_OpenedForWriting;
   double m_dbl_SampleRate;     // will be private sooner or later..
   double m_dblFrequencyOffset; // "radio-" minus "baseband"-frequency

   // Some more properties for reading RAW FILES . Should be set before opening
   // a RAW file (extension *.raw or *.dat)
   int    m_iSizeOfDataType; // sizof(BYTE), sizeof(short int), sizeof(float), sizeof(double) [for 'size calculations']
   int    m_iDataTypeFlags;  // AUDIO_FILE_DATA_TYPE_SIGNED_I/UNSIGNED/FLOAT (from cbproj\SoundUtl\AudioFileDefs.h)
   int    m_iBitsPerSample;  // since 2015-01-08 : 8,16,24=integer, 32 or 64=floating point ! (for a single channel)
   int    m_iNumChannels;
   WORD   m_wRawFileNumberOfChannels;
   double m_dblGainFactor;


private:
#if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
   T_QFile m_QFile;        // data structures and buffer for "quick file access"
#else  // traditional file access, using the standard runtime library :
   INT   m_FileHandle;
#endif
   BOOL  m_fMustWriteHeader;
   DWORD m_dwFilePos_Fact;  // file-internal position of the "fact"-chunk; 0="there's no fact chunk".
   DWORD m_dwFilePos_Data;  // file-internal position of 1st audio sample
   DWORD m_dwCurrFilePos;   // current file access position, measured in BYTES, not SAMPLES !
   DWORD m_dwFileSizeInBytes; // contains what the name says (even after closing a file)
   DWORD m_dwFileDataSize;  // netto size of the sample-array in BYTES. Sometimes found in the RIFF "data" chunk (thus the name).
   char  m_sz255FileName[256];
   BOOL InOpenInternal( char *pszFileName, BOOL fReOpenForWriting,
                        char *pszExtraInfo,int iExtraInfoLength );
   LONG WriteSamples( BYTE* pData, LONG Length); // low-level output, try to avoid
   void UpdateRawFileParams(void);


}; /* end class C_WaveIO */


#endif // _WAVE_IO_H_

/* EOF <WaveIO.h> */