/***************************************************************************/
/*  c:\cproj\SoundUtl\CWaveIO.cpp :                                        */
/*   Class unit to import and export *.wav - files and similar..           */
/*  - by DL4YHF August '2000  ...  January 2015                            */
/*  - based on the description of the WAVE-file format                     */
/*    found at http://www.wotsit.org  (excellent site with file formats!)  */
/*  - does NOT use any fancy windows gimmics, just standard file I/O.      */
/*  - has also been converted into a plain-vanilla "C" source in June 2003 */
/*     which compiles happily under Borland C for DOS, see \cproj\WaveTool */
/*  Info about wave files :                                                */
/*  http://www-mmsp.ece.mcgill.ca/documents/audioformats/wave/Samples.html */
/*    contained the following samples (used for testing THIS module) :     */
/*  -   M1F1-int24-AFsp.wav (138 kB)   :  WAVE file, stereo 24-bit data,   */
/*                  doesn't play in the stupid windows media player,       */
/*                  but works properly when played back in Spectrum Lab .  */
/*                  See notes in Read24BitIntegerAsFloat() further below.  */
/*  -   M1F1-int24WE-AFsp.wav (138 kB) :  WAVE (WAVE_FORMAT_EXTENSIBLE),   */
/*                                        stereo 24-bit data.              */
/*                  Plays in windoze media player and SL;                  */
/*                  confirms the usual 'Intel' byte order (lo/mid/hi) .    */
/*                (windoze media player seems to insist on th EXTENSIBLE   */
/*                 thingy, even if a simpler WAVEFORMATEX would do the job)*/
/*                                                                         */
/*  Used in (at least) the following projects :                            */
/*     - DL4YHF's Spectrum Lab                                             */
/*      (DL4YHF's Curve Editor  uses a 'downsized' variant of this module, */
/*         see \cbproj\CurveEdit\LoadSaveWave.c )                          */
/*                                                                         */
/* Latest changes :                                                        */
/*   2015-01-10: Added support for 32-bit IEEE floating point.             */
/*               From bad experience with 24-bit INTEGER wave files,       */
/*               they use the WAVEFORMATEXTENSIBLE thing, and a new GUID.  */
/*               Getting this to work with ALL players turned into a night-*/
/*               mare, see list of player tests below the version history. */
/*   2014-02-18: Converted from C++ to plain C for WSQ2,                   */
/*               see C:\cbproj\WSQ2\WaveIO.c .  Downsized a bit there.     */
/*   2011-02-13: Added WriteSamples_Float(), and changed the sequence      */
/*               of the "inf1" chunk (now AFTER the "fmt " chunk) because  */
/*           wave-files with the "inf1" chunk 1st didn't play under Linux. */
/*-------------------------------------------------------------------------*/


  // Player compatibility tests ....
  // 2015-01-10: Problem: Wave files with 'IEEE_FLOAT' *and* a 'WAVEFORMATEXTENSIBLE'
  //          could only be played with SOME, but not ALL, media players :
  //   * Total Commander's built-in "file viewer" :
  //            - no problem with 'FLOAT' + 'fact' + 'WAVEFORMATEXTENSIBLE';
  //            - no problem to skip the proprietary 'inf1' chunk;
  //   * Microsoft's stoneage "sndrec32.exe" ('Zubehr for Audiorecorder' from Win XP):
  //            - no problem with 'FLOAT' + 'fact' + 'WAVEFORMATEXTENSIBLE';
  //            - no problem with 'FLOAT' + 'fact' + 'WAVEFORMATEX';
  //            - no problem to skip the proprietary 'inf1' chunk;
  //   * Spectrum Lab :
  //            - no problem with 'FLOAT' + 'fact' + 'WAVEFORMATEXTENSIBLE';
  //            - no problem with 'FLOAT' + 'fact' + old 'WAVEFORMATEX';
  //            - no problem with 'FLOAT' WITHOUT 'fact', any 'WAVEFORMATxy';
  //            - no problem to skip the proprietary 'inf1' chunk (of course).
  //   * IRON browser (google-less 'Chrome' clone), using the internal player:
  //            - no problem with 'FLOAT' + 'fact' + 'WAVEFORMATEXTENSIBLE';
  //            - no problem to skip the proprietary 'inf1' chunk;
  //   * VLC media player 2.0.8 "Twoflower" :
  //            - refused to play 'FLOAT'-wav with 'fact' + 'WAVEFORMATEXTENSIBLE';
  //               > "wav error: unsupported codec (undf)"
  //               > "ps warning: garbage at input, trying to resync..."
  //   * VLC media player 2.1.8 "Rincewind" :
  //            - no problem with 'FLOAT' + 'fact' + 'WAVEFORMATEXTENSIBLE' + 'inf1';
  //


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h> // definition of data types like LONGLONG (wtypes.h) etc
#include <stdlib.h>
#include <stdio.h>
#include <io.h>          /* _rtl_open and other file functions       */
#include <fcntl.h>       /* O_RDONLY  and other file flags          */
#include <time.h>
#include "utility1.h"    // some helper functions by DL4YHF
#include "QFile.h" // DL4YHF's "Quick File Functions" (tried this 2008-03)
                   // File not found ? Should be in C:\cbproj\YHF_Tools\QFile.h !
#ifndef C_SND_MATH_INVALID_FLOAT_VALUE
# include "SoundMaths.h"
#endif
// #include <exception>

#pragma hdrstop
#pragma warn -8004 // <var is a assigned a value that is never used> - so what

#define _I_AM_WAVE_IO_ 1
#include "CWaveIO.h"  // header for this file with "single source"-variables

 // Found this GUID in an 'existing' wave file (pls55.wav), here as a BYTE array:
const BYTE WaveIO_b16GUID_PCM[16] = /* taken from an 'existing' wave file */
 //  [0]  [1]  [2]  [3]  [4]  [5]  [6]  [7]  [8]  [9]  [10] [11] [12] [13] [14] [15]
  { 0x01,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71 };
 // The MS site says it should be this "GUID" :
 //  00000001 - 0000 - 0010 - 8000 - 00aa00389b71
 // Note the strange different byte orders in bytes[6/7] and [8/9] !

 // Another "GUID" for working with the WAVEFORMATEXTENSIBLE struct (for FLOATING POINT samples):
const BYTE WaveIO_b16GUID_IEEE_FLOAT[16] = // similar as "PCM", but here for FlOATING POINT
 //  [0]  [1]  [2]  [3]  [4]  [5]  [6]  [7]  [8]  [9]  [10] [11] [12] [13] [14] [15]
  { 0x03,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71 };
 // GUID from the MS docu (*) : 00000003 - 0000 - 0010 - 8000 - 00aa00389b71
 // (*) C:\literatur\Signal_processing_and_filters\Multi_Channel_Wave_Audio_Files_2001.pdf



extern double GuessExactFileSampleRate(double dblIndicatedSRate);

// Helper functions (no class methods, to make them easy to use...)

/***************************************************************************/
static T_Float ReadUns16Intel( short *sp )
{ return ((T_Float)(*(unsigned short*)(sp)) - 32767.0)  * ( 1.0 / 32678.0 );
} // end ReadUns16Intel()

/***************************************************************************/
static T_Float ReadSigned16Motorola( short *sp )
{ union { BYTE b[2];  short i16; } t16;
  t16.b[0] = ((BYTE*)sp)[1];
  t16.b[1] = ((BYTE*)sp)[0];
  return (T_Float)t16.i16 * ( 1.0 / 32678.0 );
} // end ReadSigned16Motorola()

/***************************************************************************/
static T_Float ReadUns16Motorola( short *sp )
{ union { BYTE b[2];  unsigned short u16; } t16;
  t16.b[0] = ((BYTE*)sp)[1];
  t16.b[1] = ((BYTE*)sp)[0];
  return ((T_Float)t16.u16 - 32767.0) * ( 1.0 / 32678.0 );
} // end ReadUns16Motorola()


/***************************************************************************/
T_Float Read24BitIntegerAsFloat( BYTE *pbSource )
{
  union { BYTE b[4];  long i32; } t;

  // We'll always run on a machine with 'INTEL BYTE ORDER', so : b[0] = LSB  .
  // The following worked properly with a test file [ M1F1-int24-AFsp.wav ]  !
  t.b[0] = *pbSource++;   // bits 7..0
  t.b[1] = *pbSource++;   // bits 15..8
  t.b[2] = *pbSource  ;   // bits 23..16
  if( t.b[2] & 0x80 )     // expand the sign-bit to bits 31..24 :
       t.b[3] = 0xFF;
  else t.b[3] = 0x00;
  return (T_Float)t.i32 * ( 1.0 / (256.0*32768.0) );  // normalize to +/- 1.0 now

} // end Read24BitIntegerAsFloat()


/***************************************************************************/
void Conv24BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 24-bit signed integers, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        T_Float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        T_Float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  DWORD dwAddressIncrementPerSample = 3 * nSrcChannelsPerSample;
  //ex: if(   ( (nChannelsPerSample==2) && (pFltDest2==NULL) )
  //       || ( (nChannelsPerSample==4) && (pFltDest2!=NULL) )  )
  if( nDstChannelsPerSample>1 )
   { // two channels (usually I+Q) per destination block,
     // which means TWO destination array indices (in pFltDest*) per sample point
     for(i=0; i<nSamplePoints; i++ )
      { // up to 8 bits per sample, TWO SEPARATE destination blocks:
        *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // 1st channel ("I")
        *pFltDest1++ = Read24BitIntegerAsFloat( pbSource+3 );  // 2nd channel ("Q")
        if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
         { *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+6 ); // 3rd channel ("I")
           *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+9 ); // 4th channel ("Q")
         }
        pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .
      }
   }
  else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
   { // (in other words, ONE destination array index per sample point)
     if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
      { // only ONE channel to be copied...
        for(i=0; i<nSamplePoints; i++ )
         { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
           *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // one DESTINATION channel
           pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .

         }
      }
     else // (m_ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
      {  // TWO channels must be copied (and "split" into two destination blocks) :
        for(i=0; i<nSamplePoints; i++ )
         { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
           *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // 1st DESTINATION channel
           *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+3 );  // 2nd DESTINATION channel
           pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .
         }
      }
   } // end else < not TWO channels per destination block >
} // end Conv24BitSamplesToFloat()

/***************************************************************************/
void Conv32BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 32-bit floating point, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        double dblGainFactor,    // [in] linear gain factor, often the reciprocal of the file's full-scale sample value
        T_Float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        T_Float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  float *pfltSource;
  // The source data must be aligned to 4-byte boundaries, thus:
  if( ((int)pbSource&3)==0)
   { pfltSource = (float*)pbSource;
     if( nDstChannelsPerSample>1 )
      { // two channels (usually I+Q) per destination block,
        // which means TWO destination array indices (in pFltDest*) per sample point
        for(i=0; i<nSamplePoints; i++ )
         { *pFltDest1++ = pfltSource[0] * dblGainFactor;    // 1st channel ("I")
           *pFltDest1++ = pfltSource[1] * dblGainFactor;    // 2nd channel ("Q")
           if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
            { *pFltDest2++ = pfltSource[2] * dblGainFactor; // 3rd channel ("I")
              *pFltDest2++ = pfltSource[2] * dblGainFactor; // 4th channel ("Q")
            }
           pfltSource += nSrcChannelsPerSample;
         }
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      { // (in other words, ONE destination array index per sample point)
        if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pfltSource[0] * dblGainFactor;  // one DESTINATION channel
              pfltSource += nSrcChannelsPerSample;
            }
         }
        else // (m_ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pfltSource[0] * dblGainFactor;  // 1st DESTINATION channel
              *pFltDest2++ = pfltSource[1] * dblGainFactor;  // 2nd DESTINATION channel
              pfltSource += nSrcChannelsPerSample;
            }
         }
      } // end else < not TWO channels per destination block >
   } // end if < source block properly aligned ? >
} // end Conv32BitSamplesToFloat()

/***************************************************************************/
void Conv64BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 32-bit floating point, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        double dblGainFactor,    // [in] linear gain factor, often the reciprocal of the file's full-scale sample value
        T_Float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        T_Float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  double *pdblSource;
  // The source data must be aligned to 8-byte boundaries, thus:
  if( ((int)pbSource&3)==0)
   { pdblSource = (double*)pbSource;
     if( nDstChannelsPerSample>1 )
      { // two channels (usually I+Q) per destination block,
        // which means TWO destination array indices (in pFltDest*) per sample point
        for(i=0; i<nSamplePoints; i++ )
         { *pFltDest1++ = pdblSource[0] * dblGainFactor;    // 1st channel ("I")
           *pFltDest1++ = pdblSource[1] * dblGainFactor;    // 2nd channel ("Q")
           if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
            { *pFltDest2++ = pdblSource[2] * dblGainFactor; // 3rd channel ("I")
              *pFltDest2++ = pdblSource[2] * dblGainFactor; // 4th channel ("Q")
            }
           pdblSource += nSrcChannelsPerSample;
         }
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      { // (in other words, ONE destination array index per sample point)
        if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pdblSource[0] * dblGainFactor;  // one DESTINATION channel
              pdblSource += nSrcChannelsPerSample;
            }
         }
        else // (m_ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pdblSource[0] * dblGainFactor;  // 1st DESTINATION channel
              *pFltDest2++ = pdblSource[1] * dblGainFactor;  // 2nd DESTINATION channel
              pdblSource += nSrcChannelsPerSample;
            }
         }
      } // end else < not TWO channels per destination block >
   } // end if < source block properly aligned ? >
} // end Conv64BitSamplesToFloat()


/*------------- Implementation of the class  C_WaveIO ---------------------*/

/***************************************************************************/
C_WaveIO::C_WaveIO()
  /* Constructor of the Wave-IO Class.
   * Initializes all properties etc.
   */
{
   memset( &m_RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
   memset( &m_ChunkHdr,    0 , sizeof(T_WAV_CHUNK_HDR)    );
   memset( &m_ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX_TENSIBLE) );

# if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
   m_QFile.iHandle    = -1;
# else
   m_FileHandle       = -1;
# endif
   m_OpenedForReading = FALSE;
   m_OpenedForWriting = FALSE;
   m_dblStartTime     = 0.0;
   m_dwCurrFilePos    = 0;
   m_dwFileDataSize   = 0;
   m_iFileFormat      = AUDIO_FILE_FORMAT_UNKNOWN; // RAW (*.dat, *.raw), RIFF (*.wav), or SETI's "raw" (*.dat) ?
   m_iAudioFileOptions= AUDIO_FILE_OPTION_NORMAL;  // see c:\cbproj\SoundUtl\AudioFileDefs.h
   m_iBitsPerSample   = 16;         // just a "meaningful default" !
   m_cErrorString[0]  = '\0';

   m_dbl_SampleRate = 11025.0;      // initial guess, only important for RAW files
   m_iSizeOfDataType = 1;     // just a GUESS for reading RAW files,
   m_wRawFileNumberOfChannels = 1;  // should be properly set before opening such files
   m_iNumChannels    = 1;
   m_fMustWriteHeader = FALSE;

} // end of C_WaveIO's constructor

/***************************************************************************/
C_WaveIO::~C_WaveIO()
  /* Destructor of the Wave-IO Class.
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   */
{


} // end C_WaveIO's destructor


/***************************************************************************/
BOOL C_WaveIO::InOpenInternal( char * pszFileName,
                BOOL fReOpenForWriting,
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength ) // [in] size of pszExtraInfo
  /* Opens a wave file for READING, or an (existing) file for WRITING .
   * Returns TRUE on success and FALSE on any error .
   */
{
 int  i, n_bytes_to_read;
 T_WAV_CHUNK_HDR  chunk_hdr;  // contains only .id + .dwChunkSize;  used for ANY chunk
 T_WAV_FACT_CHUNK fact_chunk; // "fact" header, rarely used, but seems unavoidable for FLOATING POINT WAVs !
 struct ftime f_time;
 char *cp;
 BOOL fExtensionIsWav;
 BOOL fSubtractRecLengthFromTime = FALSE;
 BOOL ok = TRUE;
 BOOL fSampleRateSet = FALSE;
 BOOL found_data_chunk;
 char sz1kExtraInfo[1024];

  CloseFile();                   // Close "old" file if it was open
  ChunkInfo_Init(&MyChunkInfo);  // forget old "chunk info" (DL4YHF's T_ChunkInfo)

  // Look at the file extension..
  cp = strrchr(pszFileName, '.');
  if(cp)
       fExtensionIsWav = (strnicmp(cp,".wav",4)==0);
  else fExtensionIsWav = FALSE;
  // ex: m_iFileFormat = AUDIO_FILE_FORMAT_UNKNOWN;  // not sure whether RAW, RIFF, or SETI ...


  // Erase old chunks:
  memset( &m_RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
  memset( &m_ChunkHdr,    0 , sizeof(T_WAV_CHUNK_HDR)    );
  memset( &m_ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX_TENSIBLE) );

  if( fReOpenForWriting )
   {
#    if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
      ok = QFile_Open(&m_QFile, pszFileName, QFILE_O_RDWR );
#    else
      m_FileHandle = _rtl_open( pszFileName, O_RDWR );
      ok = (m_FileHandle>0);
#    endif  // ! WAVEIO_USE_QFILE
   }
  else  // open for READING (only) :
   {
#    if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
      ok = QFile_Open(&m_QFile, pszFileName, QFILE_O_RDONLY );
#    else
      m_FileHandle = _rtl_open( pszFileName, O_RDONLY);
      ok = (m_FileHandle>0);
#    endif  // ! WAVEIO_USE_QFILE
   }
  if(ok)
   {
    strncpy(m_sz255FileName, pszFileName, 255);
    // Get the "file time" as a first coarse value when the file has been
    // recorded. Note that "getftime" returns its own time structure
    // which is very different from our internal "time()"-like format.
    // "getftime" returns the date relative to 1980,
    // but SpecLab uses the Unix-style format which is
    //    "Seconds elapsed since 00:00:00 GMT(=UTC), January 1, 1970"
    // how many SECONDS between the UNIX-time(est'd 1970) and 1980 ?
    // T_Float d1,d2; BYTE *pSource;
    // pSource = "1970-01-01 00:00:00";
    // UTL_ParseFormattedDateAndTime( "YYYY-MM-DD hh:mm:ss", &d1, &pSource); // -> 0 (!)
    // pSource = "1980-01-01 00:00:00";
    // UTL_ParseFormattedDateAndTime( "YYYY-MM-DD hh:mm:ss", &d2, &pSource);
    // d2-=d1;
     // (should be something like 365*24*60*60 = 31536000, in fact it's 315532800)

    if( getftime(
#       if (WAVEIO_USE_QFILE)
         m_QFile.iHandle,
#       else
         m_FileHandle,
#       endif
        &f_time)==0) // "0" = "no error"
     {
       m_dblStartTime =  // INITIAL GUESS, hopefully there's an "inf1" / T_ChunkInfo with a more precise timestamp
         UTL_ConvertYMDhmsToUnix( // parameters for this routine:
         f_time.ft_year+1980,  // int year  (FULL year, like 2002, not crap like "year-1980",
         f_time.ft_month,      // int month (1..12)
         f_time.ft_day,        // int day   (1..31)
         f_time.ft_hour,    // int hour  (0..23) - not the annoying "AM PM" stuff :-)
         f_time.ft_min,     // int minute (0..59)
         2*f_time.ft_tsec); // T_Float dblSecond  (0.000...59.999)
       // By order of the Prophet, "dblStartTime" should really contain the RECORDING start time,
       //  in UTC, or -if a GPS receiver is connected- the current GPS time .
       // The FILE TIME is usually the time of the last write access, which will be
       // the LAST sample. So we must subtract the duration of the recording
       // as soon as we know it [but we don't know it here because the file header
       // has not been analysed yet].
       fSubtractRecLengthFromTime = TRUE;
     }
    else m_dblStartTime=0;


    // analyse the header of the WAV-file (which if POSSIBLY A RIFF-format)
#  if (WAVEIO_USE_QFILE)
    i = QFile_Read( &m_QFile,   &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#  else
    i = _rtl_read(m_FileHandle, &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#  endif
    if( i==sizeof(T_WAV_RIFF_HDR) )
     { // RIFF header successful read, check the signature :
      if(strncmp(m_RiffHdr.id_riff, "RIFF", 4) == 0)
        {
          m_iFileFormat = AUDIO_FILE_FORMAT_RIFF;  // bingo, it's the common windoze - *.WAV-format
        }
      else
        { // 'raw' audio samples without header.. usually *.DAT, very ambiguous !!
          if(   (m_iFileFormat != AUDIO_FILE_FORMAT_RAW)
             && (m_iFileFormat != AUDIO_FILE_FORMAT_SETI_1) // added 2010-05-10
            )
           { m_iFileFormat = AUDIO_FILE_FORMAT_RAW;
           }
          if( m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1 ) // added 2010-05-10
           { // Info about the SETI raw file format, by Ignacio Nacho, 2010 :
             // > Currently the sample rate is 8738133 samples/sec and the recording is
             // > in I/Q format, 8 bit signed per sample. So the recorded raw file is:
             // > 1st byte: In-phase of first sample.
             // > 2nd byte: Quadrature of first sample.
             // > 3rd byte:: In-phase of second sample.
             // > 4th byte: Quadrature of second sample
             m_dbl_SampleRate = 8738133.0; // Samples/second for SETI (8.7 MSamples/sec,
                                           // this only works in FILE ANALYSIS MODE...)
             m_ChunkFormat.wChannels = m_wRawFileNumberOfChannels = 2;
             m_iSizeOfDataType = 1;   // 8 bit per ADC value ( times 2 for I/Q )
             m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I; // SIGNED INTEGER (!) ... a "SETI speciality")
             m_ChunkFormat.wBitsPerSample = (WORD)( m_iSizeOfDataType * 8 );
           }
          if( fExtensionIsWav )
            {                                // .. should NEVER have the extension "wav"
              strcpy(m_cErrorString,"no RIFF id"); // .. so this is an error !
              ok = FALSE;
            }
        } // end else < NO RIFF-header found >
      if( m_iFileFormat == AUDIO_FILE_FORMAT_RIFF )
       { // if the audio file has a RIFF header, the RIFF-ID should be "WAVE" :
        if(strncmp(m_RiffHdr.wave_id, "WAVE", 4) != 0)
         { strcpy(m_cErrorString,"no WAVE id");
           ok = FALSE;
         }
       }
     }
   } // end if <file opened successfully>
  else
   { strcpy(m_cErrorString,"_open failed");
     // ex: ok = FALSE;
     return FALSE;
   }


  // For RIFF WAVE files (*.wav), analyse all CHUNKS, until the first "data" chunk has been found
  if( m_iFileFormat == AUDIO_FILE_FORMAT_RIFF )
   {
     memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );  // .. just in case NOTHING is read below
     found_data_chunk = FALSE;
     while( (ok) && (! found_data_chunk) )
      {
        // Read the next chunk HEADER from the input file:
#      if (WAVEIO_USE_QFILE)
        i = QFile_Read( &m_QFile,   &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
#      else
        i = _rtl_read(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
#      endif
        // 2008-04-26: Got here with chunk_hdr.id="fmt ", chunk_hdr.dwChunkSize=18 ,
        //   with a 32kHz stereo 'PCM' wave file saved by "sndrec32" from windoze XP .
        // 2015-01-09: Got here with chunk_hdr.id="fmt ", chunk_hdr.dwChunkSize=40 ,
        //   with a self-created 32 bit FLOATING POINT wave file
        //   which neither VLC- nor windoze media player liked to play
        //   (but the simple 'file viewer' in Total Commander and Spectrum Lab played it)
        //
        if( i!=sizeof(T_WAV_CHUNK_HDR) )
         { strcpy(m_cErrorString,"bad header size");
           ok = FALSE;
         }
        else // reading the chunk header was ok. Analyse it:
         {
           i = 0; // number of bytes read from a "known" chunk.
           n_bytes_to_read = chunk_hdr.dwChunkSize; // "ideally", may be limited below
           if(   (strncmp(chunk_hdr.id, "fmt ", 4) == 0) // <- usually the 1st chunk
              && (chunk_hdr.dwChunkSize >= WAVEIO_SIZEOF_STANDARD_WAVEFORMAT_EX )
              && (chunk_hdr.dwChunkSize <= WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE ) )
            { // It's *AT LEAST* a standard T_WAV_CHUNK_FORMAT - structure,
              //  so read a T_WAV_CHUNK_FORMAT_xyz from the input file .
              // BUT: The 'fmt ' chunk may be a simple 'WAVEFORMATEX'
              //      (wFormatTag=1 for integer or wFormatTag=3 for float),
              //   or a larger 'WAVEFORMATEXTENSIBLE' (wFormatTag=65534) !
              //   Thus, don't assume anything.. especially not that
              //   chunk_hdr.dwChunkSize is always 16 [bytes] ....
#            if (WAVEIO_USE_QFILE)
              i = QFile_Read( &m_QFile,   &m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
#            else
              i = _rtl_read(m_FileHandle, &m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
#            endif
              if ( (DWORD)i != chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ )
               { ok = FALSE;
               }
              // Note that the actual chunk size may differ from what we
              // just read, even though it's a "fmt "-chunk !
            } // end if "fmt "
           else if (strncmp(chunk_hdr.id, "fact", 4) == 0) // IF there is a "fact" at all, it's usually the 2nd chunk in a WAV file
            { // The strangely named 'fact' chunk, is required(?) for FLOATING POINT chunks.
              //  (only Spectrum Lab happily played back wave files with floating point samples,
              //   but [almost] no other player... including VLC... sigh.
              // 2015-01-09 : Got here with the "organfile" sample (32-bit float),
              //              from http://www.kozco.com/tech/soundtests.html,
              // locally saved as WaveArchive/32bit_float_test_OrganFinale.wav .
              //   The 'fact' followed immediately after the 'fmt '-chunk,
              //   with ...
              //        m_ChunkFormat.wFormatTag     = 3 = WAVE_FORMAT_IEEE_FLOAT,
              //        m_ChunkFormat.wChannels      = 2,
              //        m_ChunkFormat.dwSamplesPerSec = 44100,
              //        m_ChunkFormat.dwAvgBytesPerSec= 352800 (=44100*2*4),
              //        m_ChunkFormat.wBlockAlign    = 8, ( ["bytes per block" aka "multi-channel sample point"] )
              //        m_ChunkFormat.wBitsPerSample = 32,  [ applies to a SINGLE CHANNEL ]
              //        m_ChunkFormat.cbSize         = 0 (!)
              //         (which means it's NOT a 'WAVEFORMATEXTENSIBLE',
              //          and the contents of m_ChunkFormat.Samples etc are invalid)
              //        chunk_hdr.dwChunkSize = 4  [bytes for the 'netto data' in the "fact]
              // 2015-01-10 : Got here with a self-created FLOATING POINT file,
              //              same parameters as above except :
              //        m_ChunkFormat.cbSize = 22 ('cos it's a 'WAVEFORMATEXTENSIBLE'),
              //        m_ChunkFormat.Samples.wValidBitsPerSample = 32 (same as m_ChunkFormat.wBitsPerSample);
              //      [ m_ChunkFormat.Samples.wSamplesPerBlock : does NOT apply here because m_ChunkFormat.wBitsPerSample!=0 ];
              //
              if( n_bytes_to_read >  sizeof(fact_chunk) )
               {  n_bytes_to_read =  sizeof(fact_chunk);
               }
              // Note: If the "fact"-junk, oops, chunk, is even larger than expected,
              //       the 'unexpected bytes' will be skipped further below.
              if( n_bytes_to_read > 0 )
               {
#               if (WAVEIO_USE_QFILE)
                 i = QFile_Read( &m_QFile,   &fact_chunk, n_bytes_to_read );
#               else
                 i = _rtl_read(m_FileHandle, &fact_chunk, n_bytes_to_read );
#               endif
               }
              else
               { i = 0;
               }
              // 2015-01-09 : Got here with the "organfinal" sample (32-bit float),
              //              from http://www.kozco.com/tech/soundtests.html,
              //   with fact_chunk.dwSampleLength = 573378 (= circa 13 s * 44100 Hz, ok).
              //   Next chunk seen in the "organfinal" sample : "PEAK" (ignored).
              //   LATER, in the "data" chunk: dwChunkSize = 4587024 ( = 573378 * 2 * 4, ok).
              //   Conclusion: We really don't need this "fact" if the wave-file
              //               contains NON-COMPRESSED floating point samples,
              //               but unfortunately few programs play such files!
              //               Thus since 2015-01-09, CWaveIO also WRITES the
              //               "fact" file when the samples shall be saved as
              //               floating point numbers.
              // 2015-01-10 : With a self-created mono 'FLOAT' wave file,
              //               with a 'WAVEFORMATEXTENSIBLE' *and* 'fact'-chunk,
              //               got here with
              //           fact_chunk.dwSampleLength = 376832 .
              //   LATER, in the "data" chunk: dwChunkSize = 1507328 ( = 376832 * 4, ok),
              //          .. still no idea why SOME players won't play that file !
              //            ( -> more in WriteHeader ) .
            } // end if "fact"
           else if (strncmp(chunk_hdr.id, "data", 4) == 0)
            { // bingo, it's the one and only DATA-Chunk:
              m_dwFileDataSize = chunk_hdr.dwChunkSize;  // netto size of the samples in BYTES
              found_data_chunk = TRUE;
            } // end if "data"
           else if (strncmp(chunk_hdr.id, "inf1", 4) == 0)
            { // it's DL4YHF's prorietary 'Info Chunk, type 1' (since 2010-07) :
              if( sizeof(sz1kExtraInfo) > chunk_hdr.dwChunkSize )
               { //  Copy the characters, and make sure they are a properly terminated
                 //  "C" string:
#            if (WAVEIO_USE_QFILE)
                 i = QFile_Read( &m_QFile,   sz1kExtraInfo, chunk_hdr.dwChunkSize );
#            else
                 i = _rtl_read(m_FileHandle, sz1kExtraInfo, chunk_hdr.dwChunkSize );
#            endif
                 sz1kExtraInfo[chunk_hdr.dwChunkSize] = '\0'; // terminate C-string
                 // WaveIO.cpp originally didn't care about the contents of this string,
                 //      only AudioFileIO.cpp parsed some of the contents
                 //      after returning from C_WaveIO::InOpenInternal() .
                 //  Modified 2015-02-20 (even though it renders the "inheritance" useless),
                 //      because CWaveIO.cpp may need to know those parameters
                 //      BEFORE returning from InOpen() / InOpenInternal() :
                 if( sz1kExtraInfo[0]>0 )
                  { // This 'extra info string' may tell us the precise sampling rate, etc etc:
                    // >
                    // > sr=249.99821757 rf=77490.000000000 ut=1424466850.448920284
                    // > glat=52.13518 glon=8.44058 masl=113.4 kmh=0.0
                    // >
                    cp = sz1kExtraInfo;
                    if( StringToChunkInfo( &cp, &MyChunkInfo ) )
                     { // Copy some of this 'EXTRA INFO' into the sound analysis params,
                       //  but only if they are obviously VALID :
                       if( MyChunkInfo.dblPrecSamplingRate > 0.0 )
                        { m_dbl_SampleRate = MyChunkInfo.dblPrecSamplingRate;
                          fSampleRateSet = TRUE;
                          // Added 2015-03 : If the sampling rate was copied from the "inf1" header,
                          //     assume it's 'precise'. Modified in c:\cbproj\SoundUtl\ChunkInfo2.c:
                          //     StringToChunkInfo() sets dwValidityFlags for all successfully parsed fields.
                        }
                       if( MyChunkInfo.ldblUnixDateAndTime > 0.0 )
                        { m_dblStartTime = MyChunkInfo.ldblUnixDateAndTime;
                          fSubtractRecLengthFromTime = FALSE;
                        }
                       if( MyChunkInfo.dblRadioFrequency != 0.0 )
                        { m_dblFrequencyOffset = MyChunkInfo.dblRadioFrequency; // "radio-" minus "baseband"-frequency
                        }
                     } // end if < StringToChunkInfo() successful >
                  } // end if < if( sz1kExtraInfo[0]>0 )
                 if( (pszExtraInfo!=NULL) && ((DWORD)iExtraInfoLength>chunk_hdr.dwChunkSize) )
                  { // the caller also wants to retrieve this string.. copy it
                    strncpy( pszExtraInfo, sz1kExtraInfo, iExtraInfoLength );
                  }
               } // end if < sz1kExtraInfo "long enough" ? >
            } // end if "inf1"
           else // unknown chunk type, ignore it, it's ok !
            {
              i = 0;  // no "extra" bytes read from unknown chunk. Skip it (below).
              // 2011-02-09: got here with a chunk type "PAD " in a file
              // recorded with a ZOOM H-1, 96 kSamples/sec, 24 bit/s, 2 channels,
              // which could NOT be properly read by SL .
            }
           // Skip the chunk, no matter if it was unknown...
           //  (except for the "data" chunk)
           // Reason: also the known chunks may be LONGER
           // in the file than we expected in our structure !
           // For example, the size of the "fmt" header may be indicated
           //  as 16, 18, 40 bytes, or whatever, but we only read a maximum of
           //  40 bytes above (in QFile_Read), so we must possibly SKIP
           //  a few bytes in the seek-command below :
           if( (!found_data_chunk) && ok && ( chunk_hdr.dwChunkSize > (DWORD)i) )
            {  // skip unexpected junk in this chunk ;o) ...
#           if (WAVEIO_USE_QFILE)
             if (QFile_Seek( &m_QFile, chunk_hdr.dwChunkSize-i, QFILE_SEEK_CUR ) < 0)
#           else
             if (lseek( m_FileHandle, chunk_hdr.dwChunkSize-i, SEEK_CUR ) < 0)
#           endif
              { strcpy(m_cErrorString,"lseek failed");
                ok = FALSE;  // seeking the end of the unknown chunk failed !
              }
            }
         } // end if <reading a chunk header ok>
      } // end while <ok AND "data"-chunk not found>

     if (!found_data_chunk) ok=FALSE;
     // Arrived here with ok=TRUE, we should have skipped all (header-)chunks,
     // and the next byte read from the file would be the first sample.
     //
     // Test the "fmt "-chunk's contents if the format is supported:
     if (ok)
      {
        m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I; // initial guess...
        if( m_ChunkFormat.wBitsPerSample==8 )
         { // "PCM" wave audio files with EIGHT BITS PER (single) SAMPLE
           // are an exception: they use UNSIGNED integers (range 0..255),
           // not SIGNED integers as one would except (-128...+127) !
           m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_UNSIGNED;
         }
        else // in most (but not all) other cases, the samples are SIGNED INTEGERS:
         { m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_SIGNED_I;
         }
        switch( m_ChunkFormat.wFormatTag )
         { case WAVE_FORMAT_PCM :  // standard Microsoft PCM (integer samples)
              // m_iDataTypeFlags already set, signed or unsigned INTEGERS
              break;
           case WAVE_FORMAT_IEEE_FLOAT: // samples in IEEE floating point format
              m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_FLOAT;
              break;
           case WAVE_FORMAT_EXTENSIBLE: // 'WAVE_FORMAT_EXTENSIBLE'
              // (warum einfach, wenn's auch kompliziert geht..)
              //  Direkt nach dem einfachen 'WAVEFORMATEX' folgt hier
              //  noch weiteres undurchschaubares Zeug, u.A.
              //  ein 16-bit Union namens 'Samples', der wahlweise
              //  'wValidBitsPerSample', 'wSamplesPerBlock', oder 'wReserved'
              //  genannt wird. Dann gibt's noch ein 'SubFormat' mit dem
              //  grsslichen Datentyp 'GUID', anhand dessen die Unterscheidung
              //  zwischen INTEGER- und FLIESSKOMMA-Samples erfolgen muss
              //  (statt m_ChunkFormat.wFormatTag, denn darin steht ja
              //   in diesem Fall nur das nichtssagende 'WAVE_FORMAT_EXTENSIBLE'.
              // >  For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize
              // >  field must be set to 22 and the SubFormat field must be set
              // >  to KSDATAFORMAT_SUBTYPE_PCM or KSDATAFORMAT_SUBTYPE_IEEE_FLOAT.
              if( m_ChunkFormat.cbSize != 22 ) // important for the 'WAVE_FORMAT_EXTENSIBLE'
               {
                 sprintf(m_cErrorString,"wrong 'cbSize' (%d) for 'WAVEFORMATEXTENSIBLE'",
                                      (int)m_ChunkFormat.cbSize );
                 ok = FALSE;
               }
              else
              if( memcmp( m_ChunkFormat.b16GUID, WaveIO_b16GUID_IEEE_FLOAT, 16) == 0 )
               { // it's what MS call 'KSDATAFORMAT_SUBTYPE_IEEE_FLOAT'
                 // in one of their monster-headerfiles (which are NOT used here):
                 m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_FLOAT;
               }
              else
              if( memcmp( m_ChunkFormat.b16GUID, WaveIO_b16GUID_PCM, 16) == 0 )
               { // nicht 'float' sondern 'integer'
                 m_iDataTypeFlags = ( m_ChunkFormat.wBitsPerSample==8 )
                                  ? AUDIO_FILE_DATA_TYPE_UNSIGNED
                                  : AUDIO_FILE_DATA_TYPE_SIGNED_I;
               }
              else // neither KSDATAFORMAT_SUBTYPE_IEEE_FLOAT
               {   //     nor KSDATAFORMAT_SUBTYPE_PCM : cannot handle this !
                 strcpy(m_cErrorString,"unsupported GUID, need 'FLOAT' or 'PCM' !" );
                 ok = FALSE;    // unsupported Format category (uLaw,aLaw,..?)
               }
              break; // end case WAVE_FORMAT_EXTENSIBLE
           default:
              sprintf(m_cErrorString,"bad FormatTag:%d",(int)m_ChunkFormat.wFormatTag );
              ok = FALSE;    // unsupported Format category (uLaw,aLaw,..?)
              break;
         } // end switch( m_ChunkFormat.wFormatTag )
        if( (m_ChunkFormat.wChannels<1) && (m_ChunkFormat.wChannels>8) )
         {
          sprintf(m_cErrorString,"bad NrChannels:%d",(int)m_ChunkFormat.wChannels );
          ok = FALSE;    // Number of channels, 1=mono, 2=stereo
         }
        else // Added 2015-12 : If the number of channels in the 'wave chunk' is valid,
         {   //                 it should overwrite MyChunkInfo.nChannelsPerSample:
           MyChunkInfo.nChannelsPerSample = m_ChunkFormat.wChannels;
         }
        if(   (m_ChunkFormat.dwSamplesPerSec < 1)
           || (m_ChunkFormat.dwSamplesPerSec > AUDIO_FILE_MAX_SAMPLE_RATE) ) // -> c:\cbproj\SoundUtl\AudioFileDefs.h
         {
          sprintf(m_cErrorString,"bad SamplesRate:%d",(int)m_ChunkFormat.dwSamplesPerSec);
          ok = FALSE;    // Sampling rate not supported
         }
        else
         { // at least we know the 'nominal' sample rate..
           // Problem: RIFF WAVe header only uses integer value for sampling rate.
           // Try to be smart and reconstruct a 'precise' sample rate.
           // A guess (true in 90% of all cases):
           // Sampling rate is a standard soundcard sampling rate
           // or some "decimated" value from that.
           // Decimation factors can be any combination of 2^n * 3^m , 0..n..8, 0..m..8 .
           if( ! fSampleRateSet )
            { // m_dbl_SampleRate obviously NOT parsed from the "inf1" string yet :
              m_dbl_SampleRate = GuessExactFileSampleRate( m_ChunkFormat.dwSamplesPerSec );
              fSampleRateSet   = TRUE;
            }
         }
        if( (m_ChunkFormat.wBlockAlign < 1) || (m_ChunkFormat.wBlockAlign > 64) )
         {
          sprintf(m_cErrorString,"bad BlockAlign:%d",(int)m_ChunkFormat.wBlockAlign );
          ok = FALSE;    // Data block size ?? Bytes per Sample ???
         }
        if(  (m_ChunkFormat.wBitsPerSample==8)   // can we handle this ?...
          || (m_ChunkFormat.wBitsPerSample==16)
          || (m_ChunkFormat.wBitsPerSample==24)  // supported since 2010-05-27
          || (m_ChunkFormat.wBitsPerSample==32) )
         {
           m_iSizeOfDataType = m_ChunkFormat.wBitsPerSample / 8;
         }
        else  // neither 8, 16, 24, or 32 bits/sample -> too exotic for me :o)
         {
          sprintf(m_cErrorString,"bad BitsPerSample:%d",(int)m_ChunkFormat.wBitsPerSample );
          ok = FALSE;    // Sample size, usually 8 or 16 bits per sample
         }
        if( (m_ChunkFormat.wBitsPerSample==16) && (m_ChunkFormat.wBlockAlign==1) )
         {  // 16 bits per sample and "block_align=1" is impossible (see formula)
            // but it occurred in a wave file sample from G4JNT. Fix that bug HERE:
            m_ChunkFormat.wBlockAlign = (WORD)
               (m_ChunkFormat.wChannels * (m_ChunkFormat.wBitsPerSample+7) / 8);
         }
      } // end if <ok to check the "fmt "-chunk ?>

     if (ok)
      {
        // save the file-internal position of the 1st audio sample
        //    (just in case someone wants to "Rewind" later) :
#      if (WAVEIO_USE_QFILE)
        m_dwFilePos_Data = QFile_Seek( &m_QFile, 0, QFILE_SEEK_CUR );
#      else
        m_dwFilePos_Data = tell( m_FileHandle );
          // caution: tell returns a "SIGNED long" value, which is stupid
          //           because disk files may be up to 4 GBytes !
#      endif
        if( m_dwFilePos_Data == 0xFFFFFFFF )
         { strcpy(m_cErrorString,"ftell failed");
           ok = FALSE;
         }
        m_dwCurrFilePos  = m_dwFilePos_Data;
        // Note: This is not the position (file offset) of the "data chunk",
        //       but the position of the FIRST SAMPLE in the data chunk !
        if( fReOpenForWriting )
         { // if the file is 're-opened' for writing, set the current writing position
           // to the END of the file:
#         if (WAVEIO_USE_QFILE)
           m_dwCurrFilePos = QFile_Seek( &m_QFile, 0, QFILE_SEEK_END );
#         else
           m_dwCurrFilePos = lseek( m_FileHandle, 0, SEEK_END ); // set current position to the file's end
#         endif
         } // end if < re-opened for writing >
      } // end if < header analysis 'ok' >

   } // end if( m_iFileFormat == AUDIO_FILE_FORMAT_RIFF )
  else   // not AUDIO_FILE_FORMAT_RIFF,  but AUDIO_FILE_FORMAT_RAW = 'raw' audio samples without header
   { //------------------------------------------------------------------------------------

#   if (WAVEIO_USE_QFILE)
     m_dwFileDataSize = QFile_Seek( &m_QFile, 0, QFILE_SEEK_END );
#   else
     m_dwFileDataSize = lseek( m_FileHandle, 0, SEEK_END ); // determine the netto size in BYTES
#   endif

     m_dwFilePos_Data = m_dwCurrFilePos = 0;  // no header to skip, jump back to the 1st byte
#   if (WAVEIO_USE_QFILE)
     QFile_Seek( &m_QFile, m_dwCurrFilePos, QFILE_SEEK_SET );
#   else
     lseek( m_FileHandle, m_dwCurrFilePos, SEEK_SET );
#   endif

     // Only a few properties should be set for successfully reading a RAW file.
     // The caller should set them before or after opening the file for reading .
     // (preferred: Tell us everything we need to know BEFORE opening the file)
     UpdateRawFileParams();  // -> m_ChunkFormat.wBlockAlign, etc
   } // end else < not 'RIFF WAVE' but raw (headerless) audio file >

  // Arrived here, the file-pointer is set to the first audio sample in the file.

  if( fSubtractRecLengthFromTime
     && (m_dbl_SampleRate>0) && (m_ChunkFormat.wBlockAlign>0) )
   { // subtract recording duration from "recording start time"
     // (because up to now, only the DOS FILE TIME was known without an "inf1" header)
     m_dblStartTime -= m_dwFileDataSize/*bytes*/
        / ( m_dbl_SampleRate * m_ChunkFormat.wBlockAlign );
   } // end if( fSubtractRecLengthFromTime )

  if (!ok)
   {
    if(m_cErrorString[0]=='\0')
      strcpy(m_cErrorString,"other 'open' error");
    CloseFile();  // close input file if opened but bad header
    return FALSE;
   }
  else
   {
    m_OpenedForReading = ok;
    return TRUE;
   }
} // C_WaveIO::InOpenInternal(..)

/***************************************************************************/
BOOL C_WaveIO::InOpen(
                char *pszFileName,
                int iAudioFileFormat,  // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength ) // [in] size of pszExtraInfo
  /* Opens a wave file for READING .
   * Returns TRUE on success and FALSE on any error .
   */
{
  // Because we're not sure if *.DAT is 'any RAW file format'
  //         or SETI's "raw file format", let the caller pass
  //         the (assumed) audio file format to us :
  m_iFileFormat = iAudioFileFormat;
  return InOpenInternal( pszFileName, FALSE/*!fReOpenForWriting*/,
                         pszExtraInfo, iExtraInfoLength );
} // C_WaveIO::InOpen(..)


/************************************************************************/
void C_WaveIO::UpdateRawFileParams(void)
{
 if(   (m_iFileFormat==AUDIO_FILE_FORMAT_RAW )
    || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) // added 2010-05-10
   )
  { // If the most important parameters of the RAW file have been specified
    // *after* opening the file, calculate the missing parameters now :
    if( m_iSizeOfDataType < 1 )
        m_iSizeOfDataType = 1;
    if( m_wRawFileNumberOfChannels<1 )
        m_wRawFileNumberOfChannels=1;
    m_ChunkFormat.wBitsPerSample = (WORD)( 8 * m_iSizeOfDataType );
    m_ChunkFormat.wChannels   = (WORD)m_wRawFileNumberOfChannels;
    m_ChunkFormat.wBlockAlign = (WORD)(m_iSizeOfDataType * m_wRawFileNumberOfChannels);
  }
} // end C_WaveIO::UpdateRawFileParams()



/************************************************************************/
long C_WaveIO::GetParameterByID( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h .
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example GetVTFlags() -> GetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS) .
{
  DWORD dw;
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM__FILE_FORMAT :
        // Replacement for   int GetFileFormat(void) .
        //  The parameter (iAudioFileFormat) must be one of the
        //  AUDIO_FILE_FORMAT_...-constants defined in AudioFileDefs.
        return m_iFileFormat;

     case AUDIO_FILE_PARAM__FILE_SIZE_KBYTE :
        // Returns the current file size (if there is an opened file) in KBYTE.
        // Useful especially when logging audio samples  to keep track of the used disk space.
        // Returns a negative value on any error.
        // ex: if (m_OpenedForWriting || m_OpenedForReading) .. modified 2016, we want to know the filesize AFTER CLOSING !
        return (m_dwFileSizeInBytes + 1023) / 1024;

     case AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT :
        // Returns the NUMBER OF CHANNELS PER SAMPLE. Ususally 1 or 2.
        return GetNrChannels();

     case AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX :
        // Read the current sample index that will be used on the next
        // READ- or WRITE- access to the audio samples in the opened file.
        // Returns a negative value on any error.
        // Return value : sample index; 0xFFFFFFFF = ERROR ("4 Gig minus one")
        dw = 0xFFFFFFFF;
        if (
#        if (WAVEIO_USE_QFILE)
           (m_QFile.iHandle>=0)
#        else
           (m_FileHandle>=0)
#        endif
        && (m_ChunkFormat.wBlockAlign>0) )
         {
           // ex: dw = tell( m_FileHandle );  // check this : will "tell" work beyond 2 GByte ?
           // better use our local copy of the "current file position", it's faster :
           dw = m_dwCurrFilePos;
           if( dw >= m_dwFilePos_Data )
                dw -= m_dwFilePos_Data;
           else dw = 0;
           if(m_ChunkFormat.wBlockAlign>0)
                dw /= (DWORD)m_ChunkFormat.wBlockAlign;
         }
        return dw;


     case AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES :
        // Returns the total number of sampling points in the file.
        // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .
        if( m_ChunkFormat.wBlockAlign>0 )
         { // subtract recording duration from "recording start time"
           // (because up to now, only the DOS FILE TIME is known)
           return m_dwFileDataSize/*bytes*/ / m_ChunkFormat.wBlockAlign;
         } // end if( fSubtractRecLengthFromTime )
        else
         { return 0;
         }

     case AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT :
        // A "sample point" contains all channels, sampled at the very same time.
        // Example: Stereo Recording, 16 bits per (single) sample
        //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
        // Used to convert a "sample index" into a "file offset" .
        return m_ChunkFormat.wBlockAlign;

     case AUDIO_FILE_PARAM__CONNECTION_LOST        :
        // Returns TRUE when a previously opened network connection was lost,
        //  and the application should take action (like wait a few seconds,
        //  and then try to reconnect if configured that way) .
        // Used to check the "connected" status for web streams.
        // In a few (rare) cases, can also be used to SET the "connect"-status
        // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )
        return FALSE;

     case AUDIO_FILE_PARAM__NUM_TIMESTAMPS         :
        // Returns the number of timestamps received (or sent), since the stream was started
        return 0;

     case AUDIO_FILE_PARAM__VLF_TOOL_FLAGS         :
        // VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8, for non-compressed streams,
        //   compatible with Paul Nicholson's VLFRX tools.
        //   The VT-Flags themselves are defined in vlfrx-tools/vtlib.h ,
        //    or -alternatively-  in c:\cbproj\SoundUtl\AudioFileDefs.h .
        if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
         { switch( m_iSizeOfDataType )
            { case 4:                   // four-byte-float
                 return VTFLAG_FLOAT4;
              case 8:                   // eight-byte-float
                 return VTFLAG_FLOAT8;
              default:
                 break;
            }
         }
        else // not FLOAT but INTEGER :
         { switch( m_iSizeOfDataType )
            { case 1:                   // single-byte-integer
                 return VTFLAG_INT1;
              case 2:                   // two-byte-integer
                 return VTFLAG_INT2;
              case 4:                   // four-byte-integer
                 return VTFLAG_INT4;
              default:
                 break;
            }
         }
        return 0; // ?

     default:
        break;
   } // end switch( iAudioFileParam )

  // Arrived here ? The requested parameter is not supported.
  // KISS, return a negative value, because ZERO is usually a valid value for future parameters.
  return -1;

} // end C_WaveIO::GetParameterByID()

//---------------------------------------------------------------------------
double C_WaveIO::GetParameterByID_Double( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..-numbers for which INTEGER
  //         values are not sufficient, e.g. AUDIO_FILE_PARAM_LAST_TIMESTAMP .
  // May return C_SND_MATH_INVALID_FLOAT_VALUE for unsupported parameters .
  // Full specs in cbproj/SoundUtl/AudioFileIO.cpp : GetParameterByID_Double().
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM_LAST_TIMESTAMP :
        // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
        // Value doesn't change unless NEW timestamps arrive !
        break;
     case AUDIO_FILE_PARAM_TIMESTAMP_LATENCY :
        // Stream latency, determined by comparing the local time and the received one,
        // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
        break;
     default:
        break;
   }
  return C_SND_MATH_INVALID_FLOAT_VALUE; // parameter not supported by this incarnation of GetParameterByID_Double() !

} // end C_WaveIO::GetParameterByID_Double()


/************************************************************************/
BOOL C_WaveIO::SetParameterByID( int iAudioFileParam, long i32NewValue )
  // Returns TRUE if the new value could be set, otherwise FALSE.
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example SetVTFlags(iVTFlags) -> SetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS,..) .
{
  DWORD dw;
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {

     case AUDIO_FILE_PARAM__FILE_FORMAT :
        // Replacement for SetFileFormat() .
        //  The parameter (i32NewValue) must be one of the
        //  AUDIO_FILE_FORMAT_...-constants defined in AudioFileDefs.
        // This is only important BEFORE opening an audio file,
        // so the 'wrapper' knows which module to call..
        // .. especially for RAW audio files .
        if(  (i32NewValue!=AUDIO_FILE_FORMAT_RAW)
          && (i32NewValue!=AUDIO_FILE_FORMAT_RIFF)
          && (i32NewValue!=AUDIO_FILE_FORMAT_SETI_1) )
         { return FALSE;  // format not supported by this class
         }
        if( m_iFileFormat != i32NewValue )
         { // close any OLD file before switching to the new format:
           CloseFile();
         } // if( m_iAudioFileFormat = iAudioFileFormat )
        m_iFileFormat = i32NewValue;
        return TRUE;

     case AUDIO_FILE_PARAM__FILE_SIZE_KBYTE :
        return FALSE;  // this parameter will always be READ-ONLY !

     case AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT :
        // Sets the NUMBER OF CHANNELS PER SAMPLE. Ususally 1 or 2.
        // Important to read RAW files, where the number of channels isn't in the header.
        SetNrChannels( i32NewValue/*iNrChannels*/ );
        return TRUE;  // this parameter is writeable... but "not always" !

     case AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX :
        // Sets the current sample index that will be used on the next
        //  READ- or WRITE- access to the audio samples an opened file.
        //  Returns TRUE if ok ,  otherwise FALSE.
        dw = 0xFFFFFFFF;
        if (   (m_ChunkFormat.wBlockAlign>0)
            && (i32NewValue>=0)
#        if ( WAVEIO_USE_QFILE )
            && (m_QFile.iHandle >= 0)
#        else
            && (m_FileHandle >= 0)
#        endif
           )
         {
           dw = (DWORD)i32NewValue * m_ChunkFormat.wBlockAlign;
           if ( (dw>=0) && (dw<m_dwFileDataSize) )
            {
              m_dwCurrFilePos = m_dwFilePos_Data + dw;
              if( m_dwCurrFilePos > m_dwFileSizeInBytes )
               {  m_dwFileSizeInBytes = m_dwCurrFilePos;
               }
#            if (WAVEIO_USE_QFILE)
              return( QFile_Seek( &m_QFile, m_dwCurrFilePos , SEEK_SET ) > 0);
#            else
              return( lseek( m_FileHandle, m_dwCurrFilePos , SEEK_SET ) > 0);
#            endif
            }
         }
        return FALSE;

     case AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES :
        // The total number of sampling points in the file.
        // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .
        return FALSE; // "parameter is not writeable"

     case AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT :
        // A "sample point" contains all channels, sampled at the very same time.
        // Example: Stereo Recording, 16 bits per (single) sample
        //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
        // Used to convert a "sample index" into a "file offset" .
        return FALSE; // "parameter is not writeable"

     case AUDIO_FILE_PARAM__CONNECTION_LOST        :
        // Used to check the "connected" status for web streams.
        // In a few (rare) cases, can also be used to SET the "connect"-status
        // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )
        return FALSE; // "parameter cannot be set"

     case AUDIO_FILE_PARAM__NUM_TIMESTAMPS         :
        // Returns the number of timestamps received, since stream started
        return FALSE; // "parameter cannot be set"

     case AUDIO_FILE_PARAM__VLF_TOOL_FLAGS         :
        // VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8, for non-compressed streams,
        //   compatible with Paul Nicholson's VLFRX tools.
        //   The VT-Flags themselves are defined in vlfrx-tools/vtlib.h .
        switch( i32NewValue & VTFLAG_FMTMASK )
         {
           case VTFLAG_FLOAT4 : // 4 byte floats
            //   m_iSizeOfDataType = 4;
            //   m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_FLOAT;
                 break;
           case VTFLAG_FLOAT8 : // 8 byte floats
            //   m_iSizeOfDataType = 8;
            //   m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_FLOAT;
                 break;
           case VTFLAG_INT1   : // 1 byte signed integers
                 // (not 0..255 as in ancient wave files, thus 0x80 can be avoided.. fb)
             //  m_iSizeOfDataType = 1;
             //  m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
                 break;
           case VTFLAG_INT2   : // 2 byte signed integers
             //  m_iSizeOfDataType = 2;
             //  m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
                 break;
           case VTFLAG_INT4   : // 4 byte signed integers
             //  m_iSizeOfDataType = 4;
             //  m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
                 break;
           default :
             //  fOk = FALSE;
                 break;
         }
        return FALSE; // "parameter cannot be set"

     default:
        break;
   } // end switch( iAudioFileParam )

  // Arrived here ? The requested parameter is not supported. Life can be so simple.
  return FALSE;

} // end C_WaveIO::SetParameterByID()

//---------------------------------------------------------------------------
BOOL C_WaveIO::SetParameterByID_Double( int iAudioFileParam, double dblNewValue )
  // Almost the same as SetParameterByID(), but for 64-bit "double".
  // Returns TRUE if the specified parameter could be set, otherwise FALSE .
  // Full specs in cbproj/SoundUtl/AudioFileIO.cpp : SetParameterByID_Double().
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM_LAST_TIMESTAMP :
        // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
        // Value doesn't change unless NEW timestamps arrive !
        break;
     case AUDIO_FILE_PARAM_TIMESTAMP_LATENCY :
        // Stream latency, determined by comparing the local time and the received one,
        // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
        break;
     default:
        break;
   }
  return FALSE; // parameter not supported by this incarnation of GetParameterByID_Double() !

} // end C_WaveIO::SetParameterByID_Double()


/***************************************************************************/
LONG C_WaveIO::ReadSamples( BYTE* pData, LONG Length ) // ex: "InRead()"
  /* Reads some samples from a wave file which has been opened for READING,
   *            !!! WITHOUT !!! conversion of the sample format in memory .
   * Returns the NUMBER OF AUDIO SAMPLE POINTS if successful,
   *     (not the number of bytes; one sample point may contain up to 8 bytes)
   *   or a negative value if errors occurred.
   * Notes:
   *  - "Length" is the size of the caller's buffer capacity in BYTES.
   *     The actual number of AUDIO SAMPLES will depend on the sample
   *     format. For 16-bit mono recordings, the number of samples
   *     will only be half the "Length".
   *     Generally, the   number of audio samples read   is:
   *          n_samples = Length / m_ChunkFormat.wBlockAlign .
   *  - since 11/2005, up to FOUR channels with SIXTEEN BITS each are possible
   */
{
 int  i;

 if( (m_ChunkFormat.wBlockAlign==0) // <- can't be a WAV file, must be a headerless 'raw' file
    &&  (   (m_iFileFormat==AUDIO_FILE_FORMAT_RAW)
         || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) ) // added 2010-05-10
   )
  { // If the most important parameters of the RAW file have been specified
    // *after* opening the file, calculate the missing parameters now :
    UpdateRawFileParams();   // -> m_ChunkFormat.wBlockAlign, etc
  }

 if ( m_OpenedForReading && (m_ChunkFormat.wBlockAlign>0)
#   if ( WAVEIO_USE_QFILE )
     && (m_QFile.iHandle >= 0)
#   else
     && (m_FileHandle >= 0)
#   endif
    )
  {
# if ( WAVEIO_USE_QFILE )
   i = QFile_Read( &m_QFile,   pData, Length);
# else
   i = _rtl_read(m_FileHandle, pData, Length);
# endif
   if (i>0)
    {
      m_dwCurrFilePos += i;
      if( m_dwCurrFilePos > m_dwFileSizeInBytes )
       {  m_dwFileSizeInBytes = m_dwCurrFilePos;
       }
      if( m_ChunkFormat.wBlockAlign > 0 )
       { i /= m_ChunkFormat.wBlockAlign;  // number of bytes -> number of sample points (with N channels per sampling point)
       }
      return i;
    }
  }
 return -1;
} // C_WaveIO::ReadSamples(..)

/***************************************************************************/
LONG C_WaveIO::ReadSampleBlocks( // preferred API (used by Spectrum Lab)
     int channel_nr,    // channel number for 1st destination block (other channels in the source file may be skipped when reading);
                        // possibly bitwise ORed with AUDIO_FILE_TWO_CHANNELS_PER_BLOCK = 0x200,
                        //   defined and specified in C:\cbproj\SoundUtl\AudioFileDefs.h .
     int nr_samples,    // number of samples (or, more precisely, length of destination blocks:  pFltDest1[0..nr_samples-1] )
     T_Float *pFltDest1, // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
     T_Float *pFltDest2, // 2nd destination block (~"RIGHT"), may be NULL (!)
     T_ChunkInfo *pOutChunkInfo, // [out,optional] additional results, see ChunkInfo.h, may be a NULL pointer (!)
     char *pszExtraInfo, int iMaxStrlen ) // [out,optional] extra string with user data for wave.rd_info
  /* Reads some samples from a wave file and converts them to numbers (*)
   *                    in the range of +- 1.0 (not +/- 32767 anymore !).
   * If CWaveIO.cpp is used as a 'sub-module' (not a real subclass)
   *    of AudioFileIO.cpp, then C_WaveIO::ReadSampleBlocks()
   *    must only be called from C_AnyAudioFileIO::ReadSampleBlocks()
   *    because of the treatment (interpolation, etc) of the T_ChunkInfo there.
   *
   * Input parameters:
   *   channel_nr: 0= begin with the first channel (LEFT if stereo recording)
   *               1= begin with the second channel (RIGHT if stereo, or Q if complex)
   *     The UPPER BYTE of 'channel_nr' can optionally define
   *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
   *    (channel_nr>>8) : 0x00 or 0x01 = only ONE channel per destination block
   *                      0x02         = TWO channels per destination block
   *                                    (often used for complex values aka I/Q)
   *
   *   nr_samples: Max number of FLOATs which can be stored in T_Float dest[0..nr_samples-1]
   *               (this is NOT always the same as the number of sampling points,
   *                for example if the output are I/Q samples, all placed in
   *                pFltDest1, there are only nr_samples/2 complex sampling points).
   *   *pFltDestX = Pointers to destination arrays (one block per channel, sometimes "paired")
   * Return value:
   *      the NUMBER OF FLOATING-POINT VALUES placed in each destination block
   *                 (not necessarily the number of sample points!) ,
   *      or a negative value if an error occurred.
   * Note that returning less samples than you may expect is not an error
   * but indicates reaching the file's end !
   *   ( This convension later caused trouble when the input is not a FILE
   *     but a STREAM, see WPH_IO_Plugin.cpp ) .
   * (*) T_Float may be either SINGLE- or DOUBLE-precision float,
   *     defined in some header file, depending on SWI_FLOAT_PRECISION .
   */
{
  BYTE *temp_array;
  int  i;
  int  asize;
  int  n_samples_read;
  int  iResult;
  BOOL ok;
  BOOL fTwoChannelsPerBlock = FALSE;
  T_Float flt;

   fTwoChannelsPerBlock =  ((channel_nr & AUDIO_FILE_TWO_CHANNELS_PER_BLOCK) != 0)
                        && (m_ChunkFormat.wChannels>1);
   channel_nr &= 0x000F;  // remove all 'flags', leave only the start-channel number

   if ( (!m_OpenedForReading)
#     if ( WAVEIO_USE_QFILE )
      || (m_QFile.iHandle<0)
#     else
      || (m_FileHandle<0)
#     endif
      || (m_ChunkFormat.wBlockAlign<=0)
      || (m_ChunkFormat.wBitsPerSample <= 1)  )
    { return -1;
    }
   if(fTwoChannelsPerBlock)
    { nr_samples /= 2;    // here: TWO destination array indices per sample point
      // so if the caller can accept 4096 floating-point values in his array (pFltDest1),
      // he will only read 2048 sampling points in one call !
    }

   asize = nr_samples * m_ChunkFormat.wBlockAlign/* strange name for "bytes per sample-point"*/;
   if (asize<=0)
     return -1;
   temp_array = new BYTE[asize];  // allocate temporary array, C++ style
   if(temp_array==NULL)
     return -1;

   if( pOutChunkInfo != NULL )
    {  pOutChunkInfo->dblSampleValueRange = 1.0;  // since 2012-02-06
       pOutChunkInfo->dwValidityFlags = MyChunkInfo.dwValidityFlags;
       pOutChunkInfo->rtRecTime = GetCurrentRecordingTime();
       if( MyChunkInfo.dwValidityFlags & CHUNK_INFO_TIMESTAMPS_VALID )
        { pOutChunkInfo->ldblUnixDateAndTime = m_dblStartTime + pOutChunkInfo->rtRecTime;
        }
       if( m_dbl_SampleRate > 0.0 )
        { pOutChunkInfo->dblPrecSamplingRate = m_dbl_SampleRate;
        }
       pOutChunkInfo->dblRadioFrequency    = MyChunkInfo.dblRadioFrequency;
       // not set here: pOutChunkInfo->i64TotalSampleCounter
       // not set here: pOutChunkInfo->dblTotalGroupDelay; // group delay, added from each stage in the processing chain
       pOutChunkInfo->dblSamplingRateDrift = MyChunkInfo.dblSamplingRateDrift;
       pOutChunkInfo->gps = MyChunkInfo.gps; // GPS related part of the T_ChunkInfo struct (48 bytes)
    } // end if( pOutChunkInfo != NULL )

   n_samples_read = ReadSamples( // Returns the NUMBER OF AUDIO SAMPLE POINTS!
            temp_array, /* temporary BYTE-array for conversion     */
            asize );    /* size in BYTE, not "number of samples" !! */
   if (n_samples_read > nr_samples)
       n_samples_read = nr_samples;  // emergency limit, should never happen
   ok = FALSE;
   if  (m_ChunkFormat.wBitsPerSample <= 8)
    { /* up to 8 bits per sample : use a BYTE-pointer for addressing... */
     ok = TRUE;
     BYTE *bp = temp_array;
     if( (m_ChunkFormat.wChannels>1)  // Number of channels, 1=mono, 2=stereo
      && (channel_nr < m_ChunkFormat.wChannels) )
      { // channel_nr: 0=left, 1=right, ..
        bp += channel_nr;             // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;   // TWO destination array indices per sample point
        if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_UNSIGNED )
         { // here for UNSIGNED 8-bit samples ( raw value range 0..255 ) :
           for(i=0; i<n_samples_read/*!*/; i++ )
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              flt = (unsigned char)bp[0];
              *(pFltDest1++) = (flt-127.0) * (1.0 / 127.0);  // 1st channel in file ("I")
              flt = (unsigned char)bp[1];
              *(pFltDest1++) = (flt-127.0) * (1.0 / 127.0);  // 2nd channel in file ("Q")
              if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
               { flt = (unsigned char)bp[2];
                 *(pFltDest2++) = (flt-127.0) * (1.0 / 127.0); // 3rd channel in file ("I")
                 flt = (unsigned char)bp[3];
                 *(pFltDest2++) = (flt-127.0) * (1.0 / 127.0); // 4th channel in file ("Q")
               }
              bp += m_ChunkFormat.wBlockAlign; /* "block align" ~~ "bytes per sample point" */
            } // end for < 8-bit SIGNED samples >
         }
        else  // 8-bit SIGNED ( raw value range -128 ... + 127 )  :
         {
           for(i=0; i<n_samples_read/*!*/; i++ )
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              flt = (signed char)bp[0];
              *(pFltDest1++) = flt * (1.0 / 127.0);  // 1st channel in file ("I")
              flt = (signed char)bp[1];
              *(pFltDest1++) = flt * (1.0 / 127.0);  // 2nd channel in file ("Q")
              if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
               { flt = (signed char)bp[2];
                 *(pFltDest2++) = flt * (1.0 / 127.0); // 3rd channel in file ("I")
                 flt = (signed char)bp[3];
                 *(pFltDest2++) = flt * (1.0 / 127.0); // 4th channel in file ("Q")
               }
              bp += m_ChunkFormat.wBlockAlign; /* "block align" ~~ "bytes per sample point" */
            } // end for < 8-bit SIGNED samples >
         }
      }
     else // still 8 bits per value, but not "two channels per output block" ...
      { iResult = n_samples_read;  // ONE destination array index per sample point
        if( (m_ChunkFormat.wChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<n_samples_read; ++i)
            { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
              *pFltDest1++ = ( (T_Float)(*bp) - 127.0 ) * (1.0 / 127.0);
              bp += m_ChunkFormat.wBlockAlign;
            }
         }
        else
         {  // TWO or more channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<n_samples_read; ++i)
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              *pFltDest1++ = ( (T_Float)(*bp)   - 127.0 ) * (1.0 / 127.0);
              *pFltDest2++ = ( (T_Float)(*(bp+1))-127.0 ) * (1.0 / 127.0);
              bp += m_ChunkFormat.wBlockAlign;
            }
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
   } // end if (8-bit-samples)
  else
  if( (m_ChunkFormat.wBitsPerSample <= 16) && (m_ChunkFormat.wBlockAlign>1) )
   { /* 9..16 bits per sample, we use a 16bit-int-pointer for addressing. */
     ok = TRUE;
     short *sp = (short*)temp_array; // should be even address, so what ??
     if( (m_ChunkFormat.wChannels>1) // Number of channels, 1=mono, 2=stereo
      && (channel_nr < m_ChunkFormat.wChannels) )
      { // channel_nr: 0=left, 1=right, ..
        sp += channel_nr;     // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;  // TWO destination array indices per sample point
        switch( m_iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                 |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
              ) )
         {
           case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = (T_Float)( * sp  ) * (1.0/32768.0); // 1st channel ("I")
                *pFltDest1++ = (T_Float)(*(sp+1)) * (1.0/32768.0); // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 { *pFltDest2++ = (T_Float)(*(sp+2)) * (1.0/32768.0); // 3rd channel ("I")
                   *pFltDest2++ = (T_Float)(*(sp+3)) * (1.0/32768.0); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
           case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadUns16Intel(sp);   // 1st channel ("I")
                *pFltDest1++ = ReadUns16Intel(sp+1); // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 { *pFltDest2++ = ReadUns16Intel(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadUns16Intel(sp+3); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
           case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadSigned16Motorola(sp);    // 1st channel ("I")
                *pFltDest1++ = ReadSigned16Motorola(sp+1);  // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 {
                   *pFltDest2++ = ReadSigned16Motorola(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadSigned16Motorola(sp+1); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
           case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
              | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadUns16Motorola(sp);    // 1st channel ("I")
                *pFltDest1++ = ReadUns16Motorola(sp+1);  // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 {
                   *pFltDest2++ = ReadUns16Motorola(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadUns16Motorola(sp+3); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
         } // end switch( m_iDataTypeFlags & ... )
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_samples_read;  // ONE destination array index per sample point
        if( (m_ChunkFormat.wChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           switch( m_iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                   ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                    |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
                 ) )
            {
              case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
               { for(i=0; i<n_samples_read; ++i)
                  { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
                    *pFltDest1++ = (T_Float)(*sp) * (1.0 / 32768.0);
                    sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break;
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Intel(sp);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
              case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadSigned16Motorola(sp);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_MOTOROLA (signed)
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
                 | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Motorola(sp);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED,MOTOROLA
            } // end switch( m_iDataTypeFlags &  ... )
         }
        else
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           switch( m_iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                   ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                    |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
                 ) )
            {
              case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
               { for(i=0; i<n_samples_read; ++i)
                  { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
                    *pFltDest1++ = (T_Float)(  *sp  ) * (1.0 / 32768.0);
                    *pFltDest2++ = (T_Float)(*(sp+1)) * (1.0 / 32768.0);
                    sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Intel(sp);
                   *pFltDest2++ = ReadUns16Intel(sp+1);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
              case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadSigned16Motorola(sp);
                   *pFltDest2++ = ReadSigned16Motorola(sp+1);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_MOTOROLA (signed)
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
                 | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Motorola(sp);
                   *pFltDest2++ = ReadUns16Motorola(sp+1);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED,MOTOROLA
            } // end switch( m_iDataTypeFlags &  ... )
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
    } // end if (16-bit-samples)
   else if( m_ChunkFormat.wBitsPerSample == 24 ) // added 2010-05-27 ...
    {   // and modified 2011-02, because some stupid asshole decided
        // to REVERSE the byte order in a 24-bit-per-sample recording.
        // In M1F1-int24-AFsp.wav, considered a 'valid' PCM wave file,
        //   the byte order seemed to be LOW1-MID1-HIGH1 LOW2-MID2-HIGH2
        //   (1 = left channel,   2 = right channel) .
        //
     Conv24BitSamplesToFloat( temp_array, n_samples_read,
                 m_ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;   // TWO destination array indices per sample point
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_samples_read;  // ONE destination array index per sample point
      }
    }
   else if( m_iSizeOfDataType == 4 ) // actually 32-bit FLOATING POINT (?)
    { if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
       { // must be 32-bit 'IEEE single precision float'; only INTEL byte order is supported here:
         Conv32BitSamplesToFloat( temp_array, n_samples_read,
                 m_ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 m_dblGainFactor,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
         if( fTwoChannelsPerBlock )
          { // two channels (usually I+Q) per destination block (since 2005-11-01) :
            iResult = 2*n_samples_read;   // TWO destination array indices per sample point
          }
         else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
          {
            iResult = n_samples_read;  // ONE destination array index per sample point
          }
       } // end if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
    }
   else if( m_iSizeOfDataType == 8 ) // actually 64-bit FLOATING POINT (never tested, pretty useless for 'audio' signals)
    { if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
       { // must be 64-bit 'IEEE double precision float'; only INTEL byte order is supported here:
         Conv64BitSamplesToFloat( temp_array, n_samples_read,
                 m_ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 m_dblGainFactor,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
         if( fTwoChannelsPerBlock )
          { // two channels (usually I+Q) per destination block (since 2005-11-01) :
            iResult = 2*n_samples_read;   // TWO destination array indices per sample point
          }
         else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
          {
            iResult = n_samples_read;  // ONE destination array index per sample point
          }
       } // end if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
    }
   else
    { // cannot handle this sample format !!
     ok = FALSE;
     iResult = -1;  // error
    }

   delete[] temp_array;
   return iResult;

} // C_WaveIO::ReadSampleBlocks(...)

/***************************************************************************/
BOOL C_WaveIO::WriteHeader(
           T_ChunkInfo *pChunkInfo) // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
  // Writes the 'header' for the audio file (usually a RIFF WAVE file).
  //  This was once done immediately in OutOpen(), but since 2011-02
  //  writing the header is postponed until the first chunk of samples
  //  is available, and a PRECISE(!) timestamp is present in pChunkInfo .
  //
  // All relevant information has already been set in OutOpen(),
  //  but not written to the file yet (by the time we get here).
  //
{
  T_WAV_CHUNK_HDR  chunk_hdr;
  T_WAV_FACT_CHUNK fact_chunk;
  BOOL ok = TRUE;
  BOOL fWriteHeader = TRUE;
  char sz1kInfo[1024];

  // prepare a 'format chunk' ("fmt ") in memory :
  memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
  if( (m_iBitsPerSample <= 16) && (m_iNumChannels<=2) )
   { m_ChunkFormat.wFormatTag = WAVE_FORMAT_PCM;     // Format category:  1=Microsoft PCM (i.e. "integer" samples)
     chunk_hdr.dwChunkSize = WAVEIO_SIZEOF_STANDARD_WAVEFORMAT_EX; /* 16 byte.. */
     m_ChunkFormat.cbSize  = 16; // for 'WAVEFORMATEX' (not 'WAVEFORMATEXTENSIBLE')
   }
  else // more than 16 bits per sample in a single channel, or more than two channels ?
   { m_ChunkFormat.wFormatTag = WAVE_FORMAT_EXTENSIBLE; // Format category:  'WAVE_FORMAT_EXTENSIBLE'
     chunk_hdr.dwChunkSize = WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE; /* 40 byte */
     m_ChunkFormat.cbSize = 22; // for 'WAVEFORMATEXTENSIBLE' (not just the bare-bone 'WAVEFORMATEX')
     // > For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize  field
     // > must be set to 22 and the SubFormat field ("GUID") must be set
     // > to KSDATAFORMAT_SUBTYPE_PCM         (for integer samples - WB)
     // > or KSDATAFORMAT_SUBTYPE_IEEE_FLOAT  (for floating point  - WB) .
     // The 'contents' of the Waveformatdingenskirchen have already been set
     // in C_WaveIO::OutOpen(), including one of the "GUID"s mentioned above.
   } // end else < more than 16 bits per sample, or more than two channels >


  if( m_OpenedForWriting )  // See OutOpen() ...
   {
    // Update the "start time" of this recording from the audio-chunk-info,
    //        and some other details which may be important for post processing.
    //  (because it will be more accurate than the timestamp when we CREATED the file):
    if( pChunkInfo != NULL )
     { m_dblStartTime = pChunkInfo->ldblUnixDateAndTime; // ex: TIM_GetCurrentUnixDateAndTime();
       // By order of the Prophet, "dblStartTime" shall be a time in UTC,
       //    or -if a GPS receiver is connected- the current GPS time .
     }

    if(m_iFileFormat==AUDIO_FILE_FORMAT_RIFF)
     {
      // Fill and write the 'RIFF' header of the WAV-file:
      memset( &m_RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
      strncpy(m_RiffHdr.id_riff, "RIFF", 4);
      strncpy(m_RiffHdr.wave_id, "WAVE", 4);
      m_RiffHdr.len =   sizeof(T_WAV_RIFF_HDR)
                    + 2*sizeof(T_WAV_CHUNK_HDR)
                    + WAVEIO_SIZEOF_STANDARD_WAVEFORMAT_EX/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/;
       // (The length of the audio samples ("data" chunk) is still unknown,
       //  thus the length of the 'payload' will be ADDED later before closing.
       //  The same applies to the "fact" chunk: It must be updated
       //  when CLOSING the file, because we don't know in advance
       //  how many samples will be stuffed into the goddamned WAV file. )
      if( fWriteHeader )
       {
#       if ( WAVEIO_USE_QFILE )
         ok = QFile_Write( &m_QFile, (BYTE*)&m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#       else
         ok = (_rtl_write(m_FileHandle, &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) ) == sizeof(T_WAV_RIFF_HDR) );
#       endif
       }
     } // end if <RIFF file format>
   } // end if <file opened successfully>
  else
   { ok = FALSE;
   }


  // After the "RIFF WAVE" header, write various "chunks" .
  //  Each of those "chunks" has its own header, with a 4-character ID.
  if( (fWriteHeader) && (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF) && (ok) )
   {
     strncpy(chunk_hdr.id, "fmt ", 4); // Note: chunk_hdr.dwChunkSize already set above !

     // Write the T_WAV_CHUNK_FORMAT to the output RIFF wave file:
     //   (see also: analysis of a wave file with FLOATING POINT samples,
     //              the "fmt "-chunk must be written BEFORE the "fact" )
#   if ( WAVEIO_USE_QFILE )
     ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
     ok &= QFile_Write( &m_QFile, (BYTE*)&m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
#   else   // not DL4YHF's 'QFile' but this strange, ever-changing file API:
     i = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
     if (i != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
     i = _rtl_write(m_FileHandle, &m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
     if (i != chunk_hdr.dwChunkSize )   ok = FALSE; // file-write-error ?
#   endif // WAVEIO_USE_QFILE ?

     // 2015-01-09 : For reasons only Microsoft will ever know,
     //   when writing files with FLOATING POINT samples,
     //   there must (?) be a "fact"-chunk after the "fmt "-chunk,
     //   even if (for NON-compressed samples) it doesn't give the
     //   file reader any info which he cannot calculate from the "fmt "-chunk.
     if( m_iDataTypeFlags == AUDIO_FILE_DATA_TYPE_FLOAT ) // <- info from C_WaveIO::OutOpen()
      {
        memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
        strncpy(chunk_hdr.id, "fact", 4);
        chunk_hdr.dwChunkSize = sizeof( T_WAV_FACT_CHUNK ); // 4 [bytes] ?
        memset(&fact_chunk,0, sizeof( fact_chunk ) );
#      if ( WAVEIO_USE_QFILE )
        // Write the *header* for the 'fact'-chunk :
        ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr,  sizeof(T_WAV_CHUNK_HDR) );
        // Save file-internal position of the "fact" chunk. We need this LATER,
        // when closing the file, to update THIS chunk (like the "data"-chunk).
        m_dwFilePos_Fact = QFile_Seek( &m_QFile, 0, QFILE_SEEK_CUR );
        // Write the *placeholder* for the 'fact'-chunk :
        ok &= QFile_Write( &m_QFile, (BYTE*)&fact_chunk, chunk_hdr.dwChunkSize );
#      else   // not DL4YHF's 'QFile' but... this ugly "_rtl_"-stuff ?
        i = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
        if (i <= 0 ) ok = FALSE;
        m_dwFilePos_Fact = tell( m_FileHandle );
        i = _rtl_write(m_FileHandle, &fact_chunk, chunk_hdr.dwChunkSize );
        if (i <= 0 ) ok = FALSE;
#      endif // WAVEIO_USE_QFILE ?
      } // end if < FLOATING POINT samples -> also write a "fact"-chunk >

   } // end if( (fWriteHeader) && ... AUDIO_FILE_FORMAT_RIFF ... )


  // Fill and write DL4YHF's proprietary "inf1"-chunk to the output file ?
  //  ( Moved the "inf1" header to this place,
  //          between the "fmt " / "fact" and the "data" chunk,
  //          because Linux didn't like files with "inf1" at an earlier position)
  if( (fWriteHeader) && (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF) && (ok)
      && (m_iAudioFileOptions & AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER) )
   { if( ChunkInfoToString( sz1kInfo, sizeof(sz1kInfo)-1/*maxlen*/, pChunkInfo ) )
      { // Modified in 2011-02-19 to improve the timestamp accuracy:
        //  The "chunk info text" is now prepared here, when writing
        //  the first audio sample - because only at this time, we know
        //  the PRECISE timestamp of the next sample written to the file
        // (accurate enough for TOA/ATD RDF, etc) .
        memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
        strncpy(chunk_hdr.id, "inf1", 4);
        chunk_hdr.dwChunkSize = (strlen(sz1kInfo) + 4/*!*/) & 0x000FFFC;
#      if ( WAVEIO_USE_QFILE )
        ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
        ok &= QFile_Write( &m_QFile, (BYTE*)sz1kInfo, chunk_hdr.dwChunkSize );
#      else   // not DL4YHF's 'QFile' but...
        i = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
        if (i <= 0 ) ok = FALSE;
        i = _rtl_write(m_FileHandle, sz1kInfo, chunk_hdr.dwChunkSize );
        if (i <= 0 ) ok = FALSE;
#      endif // WAVEIO_USE_QFILE ?
      }
   } // end if < write an extra chunk with "additional info" ? >


  if( (fWriteHeader) && (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF) && (ok) )
   {
    // Fill and write a DATA-chunk to the output file:
    memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
    strncpy(chunk_hdr.id, "data", 4);
    chunk_hdr.dwChunkSize = 0; // still unknown, will be updated before closing.
#  if ( WAVEIO_USE_QFILE )
    ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
#  else
    i = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
    if (i != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
#  endif
    // Save file-internal position of 1st audio sample, which will also be
    // used to write the correct "data"-chunk size into the header LATER,
    // when closing the file.
#  if (WAVEIO_USE_QFILE)
    m_dwFilePos_Data = QFile_Seek( &m_QFile, 0, QFILE_SEEK_CUR );
#  else
    m_dwFilePos_Data = tell( m_FileHandle );
#  endif
    if (m_dwFilePos_Data==0xFFFFFFFF)    // m_dwFilePos_Data should be 44 now
     { ok = FALSE;
     }
    m_dwCurrFilePos = m_dwFilePos_Data;
    if( m_dwCurrFilePos > m_dwFileSizeInBytes )
     {  m_dwFileSizeInBytes = m_dwCurrFilePos;
     }
   }

  m_fMustWriteHeader = FALSE;  // "done" (at least, we tried)

  if (!ok)
   {
    CloseFile();  // close output file, could not generate a header.
    return FALSE;
   }
  else
   {
    return TRUE;
   }
} // end C_WaveIO::WriteHeader()

/***************************************************************************/
BOOL C_WaveIO::OutOpen(
           char *file_name,
           int iAudioFileFormat, // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE, AUDIO_FILE_MODE_APPEND, etc
           int iAudioFileOptions, // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int bits_per_sample,  // 8,16,24=integer, >=32 : floating point ! (for a single channel, since 2015-01-08)
           int channels,
           T_ChunkInfo *pInChunkInfo ) // initial sample rate, date+time (here: NOT optional, ptr must not be NULL)
  /* Creates and Opens an audio file for WRITING audio samples.
   *         Returns TRUE on success and FALSE on any error.
   * Input parameters..
   *   iAudioFileFormat: required to solve the ambiguity of different "raw"
   *                     file formats, which all have the extension "*.dat",
   *                     which is rather meaningless but that's the way it is.
   *   additional_info:  a null-terminated text string with additional info
   *                     which will be included in the exported file.
   *                     NULL if this feature is not used.
   *                     SpecLab uses this to save (GPS-)date, (GPS-)time,
   *                     precise sample rate, possibly the name of a (GPS-)log,
   *                     and other receiver settings when the recording started.
   *                     In Spectrum Lab, this string is 'produced'
   *                     using ChunkInfo.c::ChunkInfoToString() .
   *   bits_per_sample:  8,16,24=integer, 32=floating point ! (for a single channel, since 2015-01-08)
   *   channels:         1=mono, 2=stereo recording etc
   *   sample_rate:      number of samples per second.
   *                     If the output format is AUDIO_FILE_FORMAT_RIFF,
   *                     the sample rate should be above 8000 samples/second.
   */
{
#if ( ! WAVEIO_USE_QFILE )
 int i;
#endif
 BOOL ok = TRUE;
 int iFileFormat;

  CloseFile();   // Close "old" file if it was open

  if( pInChunkInfo==NULL )
   { return FALSE;
   }

  // round the sample rate to a 32-bit INTEGER value (for the RIFF header)
  m_dbl_SampleRate = pInChunkInfo->dblPrecSamplingRate;

  // coarse check of the arguments..
  if(  (iAudioFileFormat!=AUDIO_FILE_FORMAT_RAW)
    && (iAudioFileFormat!=AUDIO_FILE_FORMAT_SETI_1) // also "*.dat" but other file format !
    && (iAudioFileFormat!=AUDIO_FILE_FORMAT_RIFF) )
      return FALSE;
  // bits_per_sample: 8,16,24=integer, 32=floating point ! (for a single channel, since 2015-01-08)
  if( (bits_per_sample < 8) || (bits_per_sample > 32) )
      return FALSE;
  m_iBitsPerSample = bits_per_sample;
  if( (channels<1) || (channels>2) )
      return FALSE;    // only mono or stereo recording supported
                       // ..sorry for those 'quattrophiles' ;-)
  m_iNumChannels = channels;
  // 2010-05-10: now accepting 10 MSample/sec ... SETI may use this,
  //             of course this will NEVER work for real-time processing !
  if( (m_dbl_SampleRate < 1) || (m_dbl_SampleRate > AUDIO_FILE_MAX_SAMPLE_RATE) )
      return FALSE;

  m_iAudioFileOptions = iAudioFileOptions;
  m_dwCurrFilePos = m_dwFilePos_Data = m_dwFilePos_Fact = 0;  // various file positions are still unknown
  m_dwFileDataSize= 0;               // "netto" size of audio samples still zero
  m_iFileFormat = iAudioFileFormat;  // remember the file format when closing!
  m_iFileMode   = file_mode;  // used in WriteHeader() [postponed since 2011-02]
  strncpy(m_sz255FileName, file_name, 255); // save the filename for the "Get"-function,
                                            // and for the postponed WriteHeader call .

  // prepare a 'format chunk' ("fmt ", doesn't have anything to do with SL's T_ChunkInfo)
  //         in memory even if not writing to RIFF wave:
  memset( &m_ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX_TENSIBLE) );
  // m_iBitsPerSample, since 2015-01-08 : 8,16,24=integer, 32=floating point ! (for a single channel)
  m_iSizeOfDataType = m_iBitsPerSample/8; // ex: sizof(BYTE), sizeof(short int), sizeof(float), sizeof(double) [for 'size calculations']
  if( (m_iBitsPerSample <= 16) && (m_iNumChannels<=2) )
   { m_ChunkFormat.wFormatTag = WAVE_FORMAT_PCM;     // Format category:  1=Microsoft PCM
     m_ChunkFormat.cbSize = 16;   // size of the bare-bone 'WAVEFORMATEX' (not 'WAVEFORMATEXTENSIBLE')
     if( m_iBitsPerSample <= 8)   // yet another absurdity...
      { // 8-bit WAVes use UNSIGNED integer..
        m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_UNSIGNED;
      }
     else
      { // 16-bit WAVes use SIGNED integer !
        m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_SIGNED_I;
      }
   }
  else if( m_iBitsPerSample==32 )
   { // here:  32 bit IEEE *FLOATING POINT NUMBERS* .
     m_ChunkFormat.wFormatTag = WAVE_FORMAT_IEEE_FLOAT; // Format category:  3=not "PCM" but floating point, ha-ha...
     // (another way to have FLOATING POINT samples *and* the 'waveformatextensible' was discovered later,
     // see C:\literatur\Signal_processing_and_filters\Multi_Channel_Wave_Audio_Files_2001.pdf :
     // > wFormatTag is WAVE_FORMAT_EXTENSIBLE
     // > In the new structure WAVEFORMATEXTENSIBLE, the wFormatTag field
     // > must be set to WAVE_FORMAT_EXTENSIBLE (defined in MMREG.H).
     // > KSDATAFORMAT_SUBTYPE_PCM and KSDATAFORMAT_SUBTYPE_IEEE_FLOAT are
     // > the sub-format itself, not the actual tag.
     //
     m_ChunkFormat.cbSize = 16;   // doesn't matter, not written to file
     m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_FLOAT;
     // Technically, this is also "PCM" but Micro$oft say it's not,
     //              because "their" 'PCM' seems to be integer only,
     // and everything which is not "their PCM" needs a funny new chunk
     // which they call "Fact". But the "Fact" chunk is utterly useless
     // for non-compressed 'pulse code modulated' (sampled) wave files.
     // Full story from  www-mmsp.ece.mcgill.ca/Documents/AudioFormats/WAVE/WAVE.html :
     //  > * For all formats other than PCM, the Format chunk must have
     //  >   an extended portion. The extension can be of zero length,
     //  >   but the size field (with value 0) must be present.
     //  > * For float data, full scale is 1. The bits/sample would normally
     //  >   be 32 or 64.
     //     (WB: that's m_ChunkFormat.wFormatTag = WAVE_FORMAT_IEEE_FLOAT, != WAVE_FORMAT_PCM )
   }
  else // more than 16 bit/single-channel-sample, or more than 2 channels/sample point:
   {   // The 'spec' (if such a thing exists)
     m_ChunkFormat.wFormatTag = WAVE_FORMAT_EXTENSIBLE; // Format category:  'WAVE_FORMAT_EXTENSIBLE'
     // > For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize  field
     // > must be set to 22 and the SubFormat field ("GUID") must be set to
     // > KSDATAFORMAT_SUBTYPE_PCM .
     m_ChunkFormat.cbSize = 22;  // important for the 'WAVE_FORMAT_EXTENSIBLE', written to the file !
   }
  m_ChunkFormat.wChannels  = (WORD)m_iNumChannels; //  1=mono, 2=stereo
  m_ChunkFormat.dwSamplesPerSec     // Sampling rate     , samples per second
                 = (long)(m_dbl_SampleRate+0.5);   // ROUNDED TO 32bit-INTEGER VALUE!
    // ToDo: Should we use the 'nominal' sampling rate here, even if we know better ?
    // Some audio processing software may not be comfortable with 47999 samples/second.
  m_ChunkFormat.wBitsPerSample    // Sample size, usually 8 or 16 bits per sample
                 = (WORD)m_iBitsPerSample;
  m_ChunkFormat.wBlockAlign       // Data block size , Bytes per Sample-Point (*) :
                 = (WORD)( (long)m_ChunkFormat.wChannels
                 *  ( ( m_ChunkFormat.wBitsPerSample + 7 ) / 8 ) );
                 // (*) A "sample point" may contain multiple channels.
                 //     This is what the WAV file spec calls "block" sometimes.
                 // Example: WAV with 8-bit mono    : wBlockAlign = 1 [byte],
                 //          WAV with 16-bit stereo : wBlockAlign = 4 [bytes],
                 //          4-channel 32-bit float : wBlockAlign = 16 [bytes].
  m_ChunkFormat.dwAvgBytesPerSec  // For buffer estimation
                 = m_ChunkFormat.wBlockAlign
                 * m_ChunkFormat.dwSamplesPerSec;
  // The above was the mandatory 16-byte 'standard WAVEFORMAT' part .
  // Next: Fill the (optional) rest of the WAVE_FORMAT_EXTENSIBLE in memory,
  //       regardless of chunk_hdr.dwChunkSize :
  m_ChunkFormat.Samples.wValidBitsPerSample = m_ChunkFormat.wBitsPerSample;

  // Integer or floating point samples ? Micro$oft uses different GUIDs for this:
  if( m_iDataTypeFlags == AUDIO_FILE_DATA_TYPE_FLOAT )
   { // using FLOATING POINT samples in the data chunk:
     memcpy( m_ChunkFormat.b16GUID, WaveIO_b16GUID_IEEE_FLOAT, 16); // floating point = "not PCM" ? Stupid !
     // (It doesn't hurt to set the correct GUID here,
     //  even if only a WAVEFORMATEX but not a WAVEFORMATEXTENSIBLE
     //  is written to the file ... depending on m_ChunkFormat.cbSize) .
   }
  else
   { // using INTEGER samples in the data chunk:
     memcpy( m_ChunkFormat.b16GUID, WaveIO_b16GUID_PCM, 16 ); // "PCM"="integer" ? Apples and Oranges ?!
   }


  //
  // Note: Since 2011-02, writing of the WAVE FILE HEADERS may be postponed
  //       until the first call of WriteSamples(), called through AudioFileIO.cpp (!)
  //   But we already CREATE the file here, to avoid wasting time later.
  // Create new file, truncate existing file to length "zero", or APPEND data ?
  m_fMustWriteHeader = TRUE;
  if( m_iFileMode == AUDIO_FILE_MODE_APPEND )
   { // if the file already exists, the new data shall be APPENDED :
     iFileFormat = m_iFileFormat; // save this info, if InOpenInternal() fails
     if( InOpenInternal( m_sz255FileName, TRUE/*fReOpenForWriting*/,
                         NULL,0/*don't care for the old info*/) )
      { m_fMustWriteHeader = FALSE;  // do NOT write an "all-new" header in this case !
      } // end if < "APPEND"-mode, and file successfully opened >
     else  // InOpenInternal() failed when trying to re-open for writing ?! [this happened 2014-04-12]
      { m_iFileFormat = iFileFormat;  // set the file format again (after InOpenInternal failed)
        m_dwFileDataSize= 0;
      }
   }
#if ( WAVEIO_USE_QFILE )
  if(m_QFile.iHandle<=0)
   { ok = QFile_Create( &m_QFile, m_sz255FileName, 0/*attr="normal"*/ );
   }
  else ok = TRUE;  // already created/opened before ? Anyway.. ok .
#else
  if(m_FileHandle<=0)
   { m_FileHandle = _rtl_creat( m_sz255FileName, 0);
   }
  ok = (m_FileHandle>0);
#endif

  m_OpenedForWriting = ok;  // required for some polling functions,
                            // "it's ok to call WriteSamples_Float() now" .

  return ok;   // .. "ok" if the file was successfully CREATED, even with no data written so far

} // C_WaveIO::OutOpen(..)


/***************************************************************************/
LONG C_WaveIO::WriteSamples( BYTE* pData, LONG Length/*nBytes*/ )
  /* Writes some samples in 'raw' form (without any conversion)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLES written,
   *       or a NEGATIVE value if there was an error.
   * Note: "Length" is the size of the caller's buffer capacity in BYTES.
   *       The actual number of AUDIO SAMPLES will depend on the sample
   *       format.
   *       For 16-bit mono recordings (which is the only supported format
   *       up 'til now), the number of samples will only be half the "Length"
   *       etc.
   */
{
  int n_written=0;

  /* Ensure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!m_OpenedForWriting)
      || (m_ChunkFormat.wBlockAlign<=0)
      || (m_ChunkFormat.wBitsPerSample <= 1)
      || (Length <= 0)
#    if (WAVEIO_USE_QFILE)
      || (m_QFile.iHandle < 0)
#    else
      || (m_FileHandle < 0)
#    endif
        )
     return -1;

   // Try to append the new samples to the wave file :
#  if ( WAVEIO_USE_QFILE )
   if( QFile_Write( &m_QFile, pData, Length ) )  // -> QFile_Write returns TRUE or FALSE only !
    { n_written = Length;
    }
#  else
   n_written = _rtl_write(m_FileHandle, pData, Length );
#  endif
   if (n_written != Length)
    {
     // Writing to the output file failed .  We're in trouble now.
     CloseFile();
     return -1;  // <- set a breakpoint here !
    }

  /* Keep the "current file position" updated so we don't have to call
   *    the "tell()"-function too often.
   * This parameter will also be READ from the spectrum analyser main window
   * to display the actual file size.
   */
  m_dwCurrFilePos += n_written;
  if( m_dwCurrFilePos > m_dwFileSizeInBytes )
   {  m_dwFileSizeInBytes = m_dwCurrFilePos;
   }
  if( m_dwCurrFilePos-m_dwFilePos_Data >= m_dwFileDataSize)
   {  m_dwFileDataSize = m_dwCurrFilePos-m_dwFilePos_Data;
   }
  return n_written / m_ChunkFormat.wBlockAlign;
} // end C_WaveIO::WriteSamples(..)

/***************************************************************************/
LONG C_WaveIO::WriteSamples_Float( // app-layer output, using NORMALIZED floats
       T_Float *pfltSrc,     // [in] caller's source buffer, 32-bit floats
       T_ChunkInfo *pChunkInfo, // [in] number of samples, channels, precise sample rate, 'radio frequency', date and time, etc
          // btw, T_ChunkInfo is defined in c:\cbproj\SoundUtl\ChunkInfo.h .
       char * pszExtraInfo ) // [in] extra 'info' (usually 'data') added to info chunk
  /* Writes some samples in the floating point format (usually normalized to -1.0 ... +1.0)
   *       to an audio file which has been opened for WRITING.
   *    This is the preferred method for Spectrum Lab since 2011 (better suited to 24-bit precision).
   *    The actually used 'peak' value range should be specified in pChunkInfo->dblSampleValueRange.
   *
   * Returns the number of SAMPLE POINTS written,
   *       or a NEGATIVE value if there was an error.
   */
{
  int iChannel, nChannels, nSamplePoints, nBytesToSend, nSourceChannelsPerSample, nDestChannelsPerSample, iRest;
  T_Float flt, fltScalingFactor;
  double dblSampleValueRange;
  union { BYTE b[4];  long i32; } t;
  LONG nSamplesWritten = 0;
  BYTE *pbTempBuffer, *pb;
  SHORT *pI16;
  float *pfltDest;  

  if( pChunkInfo == NULL )
   { return -1;
   }
  nSamplePoints            = (int)pChunkInfo->dwNrOfSamplePoints;
  nSourceChannelsPerSample = pChunkInfo->nChannelsPerSample;
  nDestChannelsPerSample   = m_ChunkFormat.wChannels;
  nChannels = (nSourceChannelsPerSample < nDestChannelsPerSample)
             ? nSourceChannelsPerSample : nDestChannelsPerSample;
  if(  pChunkInfo->dblSampleValueRange > 0.0)
        dblSampleValueRange = pChunkInfo->dblSampleValueRange;
  else  dblSampleValueRange = 1.0;  // wild guess (!) .. see notes in \cbproj\SoundUtl\ChunkInfo.h

  /* Make sure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!m_OpenedForWriting)
      || (nSamplePoints <= 0 )
      || (nSourceChannelsPerSample <= 0)
      || (m_ChunkFormat.wBlockAlign<= 0)
      || (m_ChunkFormat.wBitsPerSample <= 1)
      || (pChunkInfo->dblSampleValueRange == 0.0)
#    if (WAVEIO_USE_QFILE)
      || (m_QFile.iHandle < 0)
#    else
      || (m_FileHandle < 0)
#    endif
        )
    { return -1;
    }

  /* If the header still needs to be written, do that NOW (we know the timestamp..) */
  if( m_fMustWriteHeader )
   { WriteHeader(
        pChunkInfo); // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
     m_fMustWriteHeader = FALSE;  // "done"
   }



#if(0)
  // Also emit a new GPS position to the file ? (since 2010-07-07)
  if(  (m_iPosUpdateCounter_WR != pChunkInfo->gps.iUpdateCounter )
        &&( m_dblInfoWriteInterval_sec != 0 )
        &&( qfAuxFile.fOpened )    )
   {
     WriteChunkInfo( m_dwSamplePointsWritten/*index*/, pChunkInfo, pszExtraInfo );
     m_iPosUpdateCounter_WR = pChunkInfo->gps.iUpdateCounter;
   } // end if < periodically emit 'position info' (with GPS data, etc) ?
#endif

  // Allocate a temporary buffer, in which the 8-, 16-, or 24-bit values
  // can be built, before flushing them to the wave file :
  nBytesToSend = (DWORD)m_ChunkFormat.wBlockAlign * nSamplePoints;
  pbTempBuffer = new BYTE[ nBytesToSend ];
  if( pbTempBuffer == NULL )
   { sprintf(m_cErrorString,"out of memory in WriteSamples");
     return -1;
   }
  // NO LAZY RETURN AFTER THIS POINT ! Don't forget to free that buffer....
  memset( pbTempBuffer, 0, nBytesToSend );
  pb = pbTempBuffer;               // buffer used for 8-bit unsigned integer
  pI16 = (SHORT*)pbTempBuffer;     // same buffer used for 16-bit integer
  pfltDest = (float*)pbTempBuffer; // same buffer used for 32-bit float

  // For best speed, use DIFFERENT LOOPS for each of the supported
  // "bits per sample" (strictly: bits per integer in each channel of a sample point)
  switch(m_ChunkFormat.wBitsPerSample)  // which sample-resolution, 8, 16 or 24 bit ?
   {
     case  8:   // input: normalized floats (-1.0 .... +1.0) ,
                // output: UNSIGNED 8-bit   (  0  .... 255 ) .
        fltScalingFactor = 127.0/* 2^7 - 1 */  /  dblSampleValueRange; // works with +/- 1.0 but also with +/-32767 sample value range !
        nSamplesWritten = nSamplePoints;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // up to 8 bits per sample:
              flt = 127.0 + pfltSrc[iChannel] * fltScalingFactor;
              if( flt< 0.0  ) flt= 0.0;
              if( flt> 255.0) flt= 255.0;
              pb[iChannel] = (BYTE)flt;
            } // end for < 8-bit UNSIGNED samples >
           pb      += nDestChannelsPerSample;   // next sample group in destination (unused channels already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // next sample group in SOURCE
         }
        break;

     case 16:   // input: normalized floats (-1.0   .... +1.0) ,
                // output: SIGNED 16-bit    (-32767 .... +32767 ) .
        fltScalingFactor = 32767.0/* 2^15 - 1 */
          /  pChunkInfo->dblSampleValueRange; // works with +/- 1.0 but also with +/-32767 sample value range !
        nSamplesWritten = nSamplePoints;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for signed 16-bit samples  (output range +/- 32k) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // scale and copy 16 bits per sample:
              flt = pfltSrc[iChannel] * fltScalingFactor;
              if( flt<-32767.0 ) flt=-32767.0;
              if( flt> 32767.0 ) flt= 32767.0;
              pI16[iChannel] = (SHORT)flt;
            } // end for < 16-bit SIGNED samples >
           pI16 += nDestChannelsPerSample; // next sample group in destination (unused channels already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // next sample group in SOURCE
         }
        break;

     case 24:   // input: normalized floats (-1.0   .... +1.0) ,
                // output: SIGNED 24-bit integer  (-(2^23-1).. 2^23-1 ) .
        fltScalingFactor = 8388607.0/* 2^23 - 1 */
          /  pChunkInfo->dblSampleValueRange; // works with +/- 1.0 but also with +/-32767 sample value range !
        nSamplesWritten = nSamplePoints;
        iRest = nDestChannelsPerSample - nSourceChannelsPerSample;
        if( iRest<0 )  iRest=0;
        iRest *= 3; // ... bytes per sample for ONE channel
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for 24-bit signed integer :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // scale and copy 24 bits per sample:
              flt = pfltSrc[iChannel] * fltScalingFactor;
              if( flt<-8388607.0 ) flt=-8388607.0;
              if( flt> 8388607.0 ) flt= 8388607.0;
              t.i32 = (long)flt;
              *pb++ = t.b[0];   // least significant byte (bits  7 ... 0)
              *pb++ = t.b[1];   // middle byte            (bits 15 ... 8)
              *pb++ = t.b[2];   // most significant byte  (bits 23 ...16)
            } // end for < 16-bit SIGNED samples >
           pb += iRest; // skip unused channels in destination (already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // inc for next sample in the SOURCE block
         } // end while
        // Arrived here, pb (pointer into destination buffer) should point here:
        if( pb != (pbTempBuffer + nBytesToSend ) )
         { iRest = pb-pbTempBuffer;  // <<< set breakpoint here !
         }
        break; // end case < 24 bit integer >

     case 32:   // input: normalized floats   (-1.0   .... +1.0) ,
                // output: 32-bit IEEE floats (-1.0   .... +1.0) .
        // Borland C++ (and most likely any C compiler for an Intel architecture)
        // DOES use 'IEEE floats' as its normal 'float' type,
        // thus it's not necessary to scale or convert anything here.
        nSamplesWritten = nSamplePoints;
        iRest = nDestChannelsPerSample - nSourceChannelsPerSample;
        if( iRest<0 )  iRest=0;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { *pfltDest++ = pfltSrc[iChannel];
            } // end for < 16-bit SIGNED samples >
           for(iChannel=0; iChannel<iRest; ++iChannel)
            { *pfltDest++ = 0.0;
            }
           pfltSrc  += nSourceChannelsPerSample; // inc for next sample in the SOURCE block
         } // end while
        break; // end case 32 bit (here, only FLOAT but not INTEGER)

     default:  // non supported sample resolution ...
        sprintf(m_cErrorString,"bad BitsPerSample:%d",(int)m_ChunkFormat.wBitsPerSample );
        nSamplesWritten = -1;  // this is a 'permanent' error
        nBytesToSend = 0;
        break;
   } // end switch(m_ChunkFormat.wBitsPerSample)

  if( nBytesToSend>0 )
   { WriteSamples( pbTempBuffer, nBytesToSend );
   }

  delete[] pbTempBuffer;
  return nSamplesWritten;

} // end C_WaveIO::WriteSamples_Float(...)



/************************************************************************/
  // Some 'Get' - and 'Set' - functions for the AUDIO FILE class ..
double C_WaveIO::GetSampleRate(void)  // this is the FILE's sampling rate !
{ return m_dbl_SampleRate;
}
void   C_WaveIO::SetSampleRate(double dblSampleRate)
{ m_dbl_SampleRate = dblSampleRate;
}

//---------------------------------------------------------------------------
int    C_WaveIO::GetNrChannels(void)
{ return m_ChunkFormat.wChannels;
}

//---------------------------------------------------------------------------
void   C_WaveIO::SetNrChannels(int iNrChannels)
{ // Only applies to RAW FILES !  For RIFF WAVE files,
  //  the number of channels will be read from the file header .
  m_wRawFileNumberOfChannels = (WORD)iNrChannels;
  if(  (m_iFileFormat==AUDIO_FILE_FORMAT_RAW)
    || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) ) // added 2010-05-10
   { UpdateRawFileParams();
   }
}

//---------------------------------------------------------------------------
int    C_WaveIO::GetBitsPerSample(void)
{ return m_ChunkFormat.wBitsPerSample;
}

//---------------------------------------------------------------------------
void   C_WaveIO::SetDataType(int iSizeOfDataType, int iDataTypeFlags)
{ // Only applies to RAW FILES !  For RIFF WAVE files,
  //  the number of bits per (single) sample will be read from the file header .
  // In Spectrum Lab, this function is called after the user has possibly
  //    modified some audio file format parameters in the
  //    'Audio File Properties' dialog (see AFilePrp.cpp) .
  m_iSizeOfDataType = iSizeOfDataType;
  m_iDataTypeFlags  = iDataTypeFlags;  // AUDIO_FILE_DATA_TYPE_SIGNED_I .. etc
  if(  (m_iFileFormat==AUDIO_FILE_FORMAT_RAW)
    || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) ) // added 2010-05-10
   { UpdateRawFileParams();       // -> m_ChunkFormat
   }
}

//---------------------------------------------------------------------------
void C_WaveIO::GetDataType(
        int *piSizeOfDataType, // [out] 0=unknown, 1=BYTEs, 2=short INTs, 4=floats, 8=double's
        int *piDataTypeFlags)  // [out] data type flags, see AudioFileDefs.h : AUDIO_FILE_DATA_TYPE_SIGNED_I / ..UNSIGNED / ..FLOAT / ..MOTOROLA
{
  if( piSizeOfDataType != NULL )
   { *piSizeOfDataType = m_iSizeOfDataType;
   }
  if( piDataTypeFlags  != NULL )
   { *piDataTypeFlags  = m_iDataTypeFlags;
   }
} // end C_WaveIO::GetDataType()

//---------------------------------------------------------------------------
void C_WaveIO::SetGain( double dblGainFactor )
  // Actually only used to analyse 'raw' files with FLOATING POINT NUMBERS,
  // because not all of those files use a scaling range of -1.0 ... +1.0  !
  // Example: To analyse a 'raw' file with 32-bit floating point numbers
  //          ranging from -1000.0 to +1000.0, set the gain to 1.0/1000
  //          *BEFORE* starting to read samples from it .
{
  m_dblGainFactor = dblGainFactor;
} // end C_WaveIO::SetGain()

//---------------------------------------------------------------------------
double C_WaveIO::GetGain( void )
{
  return m_dblGainFactor;
} // end C_WaveIO::GetGain()

//---------------------------------------------------------------------------
double C_WaveIO::GetFrequencyOffset(void) // "radio-" minus "baseband"-frequency
{
  // This parameter may contain the frequency shift caused by an internal NCO
  //      (when decimating the input, and logging a file with I/Q samples),
  // but also the 'radio frequency offset' (VFO frequency) of an external radio.
  // Both (added) are important to find out the original 'signal frequency'.
  // The old 'GetNcoFrequency' / 'SetNcoFrequency' were removed in 2015-01-10,
  // because the file itself only contains one field (in the "inf1" chunk, "rf" token)
  // for this purpose (="difference between baseband- and radio-frequency") .
  return m_dblFrequencyOffset;
} // end C_WaveIO::GetFrequencyOffset()

BOOL   C_WaveIO::SetFrequencyOffset(double dblFrequencyOffset)
{ // Details in C_WaveIO::GetFrequencyOffset() !
  m_dblFrequencyOffset = dblFrequencyOffset;
  return TRUE;
} // end C_WaveIO::SetFrequencyOffset()


double C_WaveIO::GetCurrentRecordingTime(void) // see specification in c:\cbproj\SoundUtl\AudioFileIO.h (t=0 is the RECORDING START TIME, aka "track time")
{ double d;   // no "T_Float" here, we need 64 bit floats to encode the time !
  d = (double)GetParameterByID(AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX);
  if(m_dbl_SampleRate>0)
     d /= m_dbl_SampleRate;
  return d;    // **NOT** "+ Start Time" ! ! !
}
double C_WaveIO::GetRecordingStartTime(void)
{ return m_dblStartTime; // seconds elapsed since 00:00:00 GMT, January 1, 1970.
}
void   C_WaveIO::SetRecordingStartTime(double dblStartTime)
{ m_dblStartTime = dblStartTime;

  MyChunkInfo.ldblUnixDateAndTime = dblStartTime;
  // Added 2015-03 : If the timestamp was set this way,
  //                 assume it's 'valid' and even 'precise' :
  if( dblStartTime > 0.0 )
   { MyChunkInfo.dwValidityFlags |= (CHUNK_INFO_TIMESTAMPS_VALID|CHUNK_INFO_TIMESTAMPS_PRECISE);
   }
}


void C_WaveIO::GetFileName(char *pszDest, int iMaxLen)
{ strncpy( pszDest, m_sz255FileName, iMaxLen );
}


bool   C_WaveIO::AllParamsIncluded(void)
{ return FALSE;
}



/***************************************************************************/
BOOL C_WaveIO::CloseFile( void )
  /* Closes the WAV-file (if opened for READING exor WRITING.
   * Returns TRUE when a file HAS BEEN CLOSED, of FALSE on error or 'nothing to close'.
   */
{
 BOOL ok=TRUE, closed_something=FALSE;
#if (!WAVEIO_USE_QFILE)
 int  n_written;
#endif
 T_WAV_CHUNK_HDR  chunk_hdr;
 T_WAV_FACT_CHUNK fact_chunk;

#    if (WAVEIO_USE_QFILE)
  if (m_OpenedForWriting && (m_QFile.iHandle>=0) )
#    else
  if (m_OpenedForWriting && (m_FileHandle>=0) )
#    endif
   {
     // If the file has been opened for WRITING, we may have to correct
     // some header entries (depending on the file type & size, because we did
     //  not know how long the file was going to get when writing the header).
     if(  (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF)
       && (m_dwFilePos_Data > (long)sizeof(T_WAV_CHUNK_HDR) ) )
      { // It's a RIFF WAVE file, and the position
        // of the "data" chunk structure is known:
        //   wind the file position back to the 'data' chunk,
        //   and update (overwrite) the 'dwChunkSize' in it:
#      if (WAVEIO_USE_QFILE)
        if( QFile_Seek( &m_QFile, m_dwFilePos_Data-sizeof(T_WAV_CHUNK_HDR), SEEK_SET ) > 0)
#      else
        if( lseek( m_FileHandle, m_dwFilePos_Data-sizeof(T_WAV_CHUNK_HDR), SEEK_SET ) > 0)
#      endif
         {
           strncpy(chunk_hdr.id, "data", 4);
           // The size of the "data" chunk is known now, write it to the file.
           // m_dwFileDataSize is the netto size of the sample-array in BYTES.
           // (2015-01-09 : Almost the same info is duplicated in the "fact"-chunk,
           //    the only difference is the unit: here "bytes", there "samples" )
           chunk_hdr.dwChunkSize = m_dwFileDataSize;
#         if ( WAVEIO_USE_QFILE )
           ok = QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR));
#         else
           n_written = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR));
           if (n_written != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
#         endif
         } /* end if <seeking the "data"-chunk length entry ok> */
        if (ok)
         { // If all accesses up to now were ok, also update the "total" file
           // size in the RIFF header (at the very BEGIN of the file) :
           m_RiffHdr.len = m_dwFilePos_Data + m_dwFileDataSize;
           // Rewind to the beginning of the file
           //     (where the RIFF header should be :)
#         if (WAVEIO_USE_QFILE)
           if( QFile_Seek( &m_QFile, 0L, QFILE_SEEK_SET ) == 0)
#         else
           if( lseek( m_FileHandle, 0L, SEEK_SET ) == 0)
#         endif
            {
#            if ( WAVEIO_USE_QFILE )
              ok &= QFile_Write( &m_QFile,  (BYTE*)&m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#            else
              n_written = _rtl_write(m_FileHandle, &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
              if( n_written != sizeof(T_WAV_RIFF_HDR) )
               { // writing the size-corrected RIFF header failed:
                 ok = FALSE;
               }
#            endif
            }
         } /* end if <ok to correct the file size in the RIFF header> */

        // Added 2015-01-09 : If a 'fact' chunk had been writting,
        //  then update the "dwSampleLength" member within that chunk.
        //  (otherwise, various media players refused to play files
        //   with FLOATING POINT samples)
        // m_dwFilePos_Fact : file-internal position of the "fact"-chunk; 0="there's no fact chunk".
        if( ok && (m_dwFilePos_Fact!=0) && (m_dwFilePos_Fact!=0xFFFFFFFF) )
         {
           // Fill the 'fact' chunk (which so far only exists as a PLACEHOLDER in the file).
           memset(&fact_chunk,0, sizeof( fact_chunk ) );
           // To avoid redundant info (like an extra "sample counter" for this purpose),
           // determine the 'number of sample points' aka 'number of blocks'
           // which have been written into the "data" chunk :
           //   m_dwFileDataSize is the netto size of the sample-array in BYTES,
           //   m_iBitsPerSample are the number of bits for a SINGLE channel,
           //   m_iNumChannels is the number of channels per "sample point"
           //    (this is what the spec sometimes calls a "block", too ambiguous) .
           //  Each sample point (with N channels) contains one or more BYTES,
           //  thus the calculation is ridiculously simple:
           fact_chunk.dwSampleLength = m_dwFileDataSize / ( m_iNumChannels * m_iBitsPerSample/8 );
           // Rewind to the file position where the 'fact'(.dwSampleLength)
           //  should be, and overwrite the 'dummy' placed there in ..WriteHeader.
           //  2015-01-09 : m_dwFilePos_Fact was 60, but don't rely on that.
#         if (WAVEIO_USE_QFILE)
           if( QFile_Seek( &m_QFile, m_dwFilePos_Fact, QFILE_SEEK_SET ) > 0)
            {
              ok &= QFile_Write( &m_QFile,  (BYTE*)&fact_chunk, sizeof(fact_chunk) );
            }
#         else
           if( lseek( m_FileHandle,  m_dwFilePos_Fact, SEEK_SET ) > 0)
            { n_written = _rtl_write(m_FileHandle, &fact_chunk, sizeof(fact_chunk) );
              if( n_written != sizeof(fact_chunk) )
               { ok = FALSE;
               }
            }
#         endif // WAVEIO_USE_QFILE ?
         } // end if < update "fact"-chunk shortly before closing ? >
      } /// end if  <RIFF WAVE  and  m_dwFilePos_Data > sizeof(..)  >
     closed_something = TRUE;
   } // end if (..OpenedForWriting)
  else
   { closed_something = FALSE;  // if there was NOTHING TO CLOSE, don't say "successfully closed" !
   }


  // Close the file, no matter if it was open for READING or WRITING:
  //
#if (WAVEIO_USE_QFILE)
  QFile_Close( &m_QFile ); // sets the handle invalid by itself
#else
  _rtl_close(m_FileHandle);
  m_FileHandle = -1;       // must forget this handle now !
#endif

  m_OpenedForReading = FALSE;
  m_OpenedForWriting = FALSE;

  return closed_something;
} // C_WaveIO::CloseFile(..)



/* EOF <WaveIO.cpp> */



