/***************************************************************************/
/* AudioWrt.cpp =  Audio File Writer .                                     */
/*   - by DL4YHF,  May 2002  ... October 2020                              */
/*   - no VCL, no special runtime libs, only Win32 API used.               */
/*                                                                         */
/* Used by: "SndInput", "SerInput", "SpecLab" .                            */
/*                                                                         */
/***************************************************************************/

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h>

#pragma hdrstop

#include "ErrorCodes.h"
#include "utility1.h"  // uses UTL_NamedMalloc instead of malloc (for debugging), and to *GUARANTEE* 8-byte alignment
#include "AudioWrt.h"

//***************************************************************************
//  Implementation of methods for the CAudioFileWriter         class
//***************************************************************************


//***************************************************************************
CAudioFileWriter::CAudioFileWriter()   // constructor
{
   m_hOutputFile = INVALID_HANDLE_VALUE;
} // end CAudioFileWriter::CAudioFileWriter()    [constructor]

//***************************************************************************
CAudioFileWriter::~CAudioFileWriter()   // destructor
{
  CAudioFileWriter::CloseFile();  // ... if necessary (!)
} // end CAudioFileWriter::~CAudioFileWriter()    [destructor]


//***************************************************************************
BOOL CAudioFileWriter::CreateFile(
                  char *pszDestFile,    // name of produced output file
                   int iExportDataType, // usually DATA_TYPE_INT16
             WAVEFORMATEX *pWaveFormat) // if a wave file header shall be produced
{
  CAudioFileWriter::CloseFile();  // ... if necessary (!)

  m_iExportDataType    = iExportDataType;
  m_lCountBytesWritten = 0;
  try
   {
    m_hOutputFile = ::CreateFile(
        pszDestFile,   // LPCTSTR lpFileName,    pointer to name of the file
        GENERIC_WRITE, // DWORD dwDesiredAccess, access (read-write) mode
        0,             // DWORD dwShareMode,     share mode
        NULL,          // LPSECURITY_ATTRIBUTES lpSecurityAttributes
        CREATE_ALWAYS, // DWORD dwCreationDistribution,  how to create
        FILE_ATTRIBUTE_NORMAL, // DWORD dwFlagsAndAttributes, file attributes
        NULL);         // HANDLE hTemplateFile,  handle to file with attributes to copy
   }
  catch(...)
   {
     m_dwErrorCode = GetLastError();
     m_hOutputFile = INVALID_HANDLE_VALUE;
     return FALSE;
   }
  if(m_hOutputFile==INVALID_HANDLE_VALUE)
   {
     m_dwErrorCode = GetLastError();
     if(m_dwErrorCode == NO_ERRORS) // action failed but 'no error' ?!?
      { // (yes, this happens !)
        m_dwErrorCode = ERROR_CREATE_FILE_FAILED;
      }
   }

  return m_hOutputFile!=INVALID_HANDLE_VALUE;

} // end CAudioFileWriter::CreateFile()


//***************************************************************************
BOOL CAudioFileWriter::AppendToFile(
                 T_Float *pfltSource, // source data, either single or double precision
                    long lNrSamples)  // number of 'T_Float' values (!)
  // Note: the audio writer does not care for the buffer structure.
  //       If you want to write 4 pairs of I/Q samples,
  //       lNrSamples must be 4 * 2 = 8 'values' .
{
 double d;
 BOOL   fResult;
 long   l;
 DWORD  dwBytesWritten;

  if( (m_hOutputFile==INVALID_HANDLE_VALUE) || (lNrSamples<=0) )
      return FALSE;  // oh, shut up

  l = lNrSamples;
  switch(m_iExportDataType)    // what data type for the output file ?
   {
    case DATA_TYPE_INT8     :  // write short 8 bit integer values ("short")
       { BYTE *pi8Array = (BYTE*)UTL_NamedMalloc( "AudioWr1", lNrSamples); // allocate temporary array
         BYTE *pi8 = pi8Array;
         if(!pi8Array) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try
          {
           while(l--)          // Note: the nominal input range is +-32767 (!)
            { d = ( (*pfltSource++) * (1.0 / 256.0)) + 128.0;
              if(d<0)   d= 0;
              if(d>255) d= 255;
              *pi8++ = (BYTE)d; // Note: the nominal output range is 0..255(!)
            }

           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                   (LPCVOID)pi8Array, // pointer to data to write to file
                          lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           UTL_free(pi8Array);     // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           UTL_free(pi8Array);     // free temporary array
           return FALSE;
          } }
    case DATA_TYPE_INT16     :  // write short 16 bit integer values ("short")
       { SHORT *pi16Array = (SHORT *)UTL_NamedMalloc( "AudioWr2", lNrSamples*sizeof(SHORT));
         SHORT *pi16 = pi16Array;
         if(!pi16Array) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try{
           while(l--)
            { d = 32767.0 * (*pfltSource++);
              if(d<-32768) d=-32768;
              if(d> 32767) d= 32767;
              *pi16++ = (SHORT)d;
            }
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                  (LPCVOID)pi16Array, // pointer to data to write to file
            sizeof(SHORT)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           UTL_free(pi16Array);  // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           UTL_free(pi16Array);  // free temporary array
           return FALSE;
          } }
    case DATA_TYPE_FLOAT32 :  // write single precision floating point ("float",  32 bit)
        {
#if SWI_FLOAT_PRECISION==1
         // no need to convert here, the source block is already "single precision float" format..
         try{
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                 (LPCVOID)pfltSource,   // pointer to data to write to file
           sizeof(float)*lNrSamples,  // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           return FALSE;
          }
#else /* .. SWI_FLOAT_PRECISION ..*/
         // internal processing single precision, but double precision for the outside world:
         double *pdblArray = new double[lNrSamples]; // allocate temporary array
         double *pdbl = pdblArray;
         if(!pdblArray) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try{
           while(l--)
            { *pdbl++ = (float)*pfltSource++;
            }
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                  (LPCVOID)pdblArray, // pointer to data to write to file
            sizeof(double)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           UTL_free(pdblArray);    // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           UTL_free(pdblArray);    // free temporary array
           return FALSE;
          }
#endif /* .. SWI_FLOAT_PRECISION ..*/
        }
    case DATA_TYPE_FLOAT64 :  // write double precision floating point ("double", 64 bit)
        {
#if SWI_FLOAT_PRECISION==2
         // no need to convert here, the source block is already "double" format..
         try{
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                 (LPCVOID)pfltSource,   // pointer to data to write to file
           sizeof(double)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           return FALSE;
          }
#else /* .. SWI_FLOAT_PRECISION ..*/
         // internal processing double precision, but single precision for the outside world:
         float *pfltArray = new float[lNrSamples]; // allocate temporary array
         float *pflt = pfltArray;
         if(!pfltArray) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try{
           while(l--)
            { *pflt++ = (float)*pfltSource++;
            }
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                  (LPCVOID)pfltArray, // pointer to data to write to file
            sizeof(float)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           UTL_free(pfltArray);    // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           UTL_free(pfltArray);    // free temporary array
           return FALSE;
          }
#endif /* .. SWI_FLOAT_PRECISION */
        }
    default:  // unknown data type
         return FALSE;
   } // end switch(iExportDataType)

} // end CAudioFileWriter::AppendToFile()

//***************************************************************************
BOOL CAudioFileWriter::CloseFile(void)
{
  if(m_hOutputFile != INVALID_HANDLE_VALUE)
   {
    try
     {
       CloseHandle( m_hOutputFile );
       m_hOutputFile = INVALID_HANDLE_VALUE;
       return TRUE;
     }
    catch(...)
     {
       m_dwErrorCode = GetLastError();
       m_hOutputFile = INVALID_HANDLE_VALUE;
       return FALSE;
     }
   }
  else  // there was no valid opened file..
   {   return FALSE; // don't say "successfully closed" if there WAS nothing to close !
   }
} // end CAudioFileWriter::CloseFile()


