//---------------------------------------------------------------------------
//
// File:    C:\cproj\SoundUtl\AudioViaTCP.C
// Date:    2008-01-20
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Sending uncompressed audio streams through TCP/IP connections .
// Status:  Most likely, non-functional because not used for "ages" !
//          2014-04-16 : Will soon be replaced by a TCP/IP based
//                       client / server compatible with Paul N's 'vlfrx' tools.
//          2020-10-25 : Decided to add the VLF-RX-tools-compatible SERVER
//                       for timestamped, non-compressed streams
//                       in C:\cbproj\SoundUtl\VorbisStreamServer.cpp ,
//                   NOT in C:\cproj\SoundUtl\AudioViaTCP.C ,
//                   because all the VLF-TX-tool-compatible handling of
//                   timestamps (in VT_BLOCKs) was already there, for
//                   *compressed* streams (via HTTP: "_audiostream.ogg" ).
//
//  The audio-via-TCP client/server communicates through (windows-)messages
//           with the application.
//  Look for 'case WM_WINSOCK_HTTP_APPMSG' and 'case HTTP_MSG_FILE_REQUEST'
//           in the message handler of the application's main window
//           to find the application-specific part of the HTTP server .
//
// Original file ("master") : c:\cbproj\SoundUtl\AudioViaTCP.c
//---------------------------------------------------------------------------

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#pragma hdrstop  // no precompiled headers after this ..

//#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "wsock.h"  // old wrapper for some "winsock" services, located in YHF_Tools

#pragma hdrstop  // no precompiled headers after this ..

#include "audiomsg.h"    // \CBproj\SoundUtl\audiomsg.c : "audio messages" (sent through TCP/IP etc)
#include "AudioViaTCP.h" // included by surrounding application too


// Global variables (shake your head about them, but carefully used they make life easier)
BOOL AVTCP_fServerEnabled = FALSE;


// Configuration of the audio-via-TCP client side :
BOOL AVTCP_receive_UDP;  // FALSE=off,  TRUE=enabled
BOOL AVTCP_receive_TCP;
BOOL AVTCP_send_UDP;
BOOL AVTCP_send_TCP;
char AVTCP_sz20RcvFromIP[22];
WORD AVTCP_wRcvFromPort;
char AVTCP_sz20SendToIP[22];
WORD AVTCP_wSendToPort;


#define AVTCP_RX_BUFFER_SIZE 65535  /* Number of 16-bit entries. Must be a power of two ! */
static short AVTCP_i16RxAudioBuffer[ AVTCP_RX_BUFFER_SIZE ]; // RX-buffer for uncompressed audio data


// Global "output" buffer, filled from real-time audio processing thread,
//   read by the audio-via-TCP-server to send this data :
#define AVTCP_TX_BUFFER_SIZE 65535  /* Number of 16-bit entries. Must be a power of two ! */
static short AVTCP_i16TxAudioBuffer[ AVTCP_TX_BUFFER_SIZE ]; // TX-buffer for uncompressed audio data
static DWORD AVTCP_dwTxBufferHeadIdx;   // head index for the TX-buffer (used by multiple readers)
static double AVTCP_dblTxBufSampleRate; // precise sampling rate of the data in AVTCP_i16TxAudioBuffer
static double AVTCP_dblUnixDateAndTime; // timestamp for the LATEST (newest) sample in "


// CONNECTION structure (used by the audio-via-TCP SERVER)
typedef struct tAVTCPServerConnection
{
  SOCKET  Socket;
  enum  { PS_CONNECTING, PS_SENDING_AUDIO, PS_RECEIVING_AUDIO } eProtState;
  enum  { RS_WAIT_HEADER,RS_HEADER, RS_AUDIO_SAMPLES          } eRxState;
  T_AudioMsgAnyHeader hdr;       // any kind of audio-message-header
  long  i32RxByteCount;   // number of bytes received for the present eRxState
  long  i32RxBytesWanted; // number of bytes to receive (usually audio samples)
  SHORT *pi16RxAudioBuffer;   // buffer for received audio samples, usually points to ACTCP_i16RxAudioBuffer[]
  BOOL  fWaitingToContinueWrite; // added 2008-01-02, see long story in OnWrite() !
        // Also used for outbound audio streams when there was "nothing to send at the moment".
  DWORD dwConnectTime;        // value of GetTickCount() when the connection was created
  DWORD dwRecvd; // statistics: number of bytes received for this connection (from server's point of view)
  DWORD dwSent;  // statistics: number of bytes sent for this connection (from server's point of view)
} T_AVTCP_ServerConnection;

#define AVTCP_SRV_MAX_CONNECTIONS 10 /* max number of "audio clients" to which the server may be connected */
static T_AVTCP_ServerConnection AVTCP_server_connections[AVTCP_SRV_MAX_CONNECTIONS];

static AVTCP_RX_CALLBACK  AVTCP_pvRxCallback=NULL;  // callback for received audio message blocks
static AVTCP_LOG_CALLBACK AVTCP_pvLogCallback=NULL; // callback to show & log debug messages

// Internal linkage
static SOCKET AVTCP_listenSocket = INVALID_SOCKET;  // socket to listen for incoming connects
static HWND   AVTCP_hMainWindow;   // handle of the window with the message loop
static int    AVTCP_iWindowsMsg;   // the message to be received when a network event occurs

static int AVTCP_iHttpHdlrBusy = 0; // prevents calling the periodic handler while we're processing async msgs


// Forward declarations (prototypes for "internal" functions, for what it's worth)
static void AVTCP_SRV_OnContinueWrite( T_AVTCP_ServerConnection *pSrvConn );
static void AVTCP_SRV_OnAccept( SOCKET socket, int nErrorCode);
static void AVTCP_SRV_OnRead(SOCKET socket, int nErrorCode);
static void AVTCP_SRV_SendHttpString( T_AVTCP_ServerConnection * pSrvConn, char *pszMsg );
static void AVTCP_SRV_OnWrite(SOCKET socket, int nErrorCode);
static void AVTCP_SRV_OnContinueWrite( T_AVTCP_ServerConnection *pSrvConn );
static void AVTCP_SRV_OnClose(SOCKET socket, int nErrorCode);
static void AVTCP_SRV_BeginToSendData( T_AVTCP_ServerConnection * pSrvConn );
static void AVTCP_SRV_SendData( T_AVTCP_ServerConnection * pSrvConn );
static void AVTCP_SRV_CloseConnection( T_AVTCP_ServerConnection * pSrvConn );



//===========================================================================
//  Common functions (for both SERVER- and CLIENT functions)
//===========================================================================

//---------------------------------------------------------------------------
void AVTCP_Init( void )   // should be called ONCE to initialize this module
{
} // end AVTCP_Init()


//---------------------------------------------------------------------------
void AVTCP_LogEvent( char *pszFormat, ...)
{
  va_list arglist;   // really arg listig, this arglist-thingy :o)
  char szTemp[384];

  // Write text to string and append to event list
  va_start(arglist, pszFormat);
  vsprintf( szTemp, pszFormat, arglist);
  va_end(arglist);
  if( AVTCP_pvLogCallback )
   { (*AVTCP_pvLogCallback)( szTemp );
   }
}


//---------------------------------------------------------------------------
void AVTCP_LogWSockError( LPCSTR lpText, int nErrorCode)
{
  AVTCP_LogEvent( "%s : %s", lpText, WSOCK_ErrorCodeToString( nErrorCode ) );
}


//---------------------------------------------------------------------------
void AVTCP_CopyAudioChunkForTransmission_FLT32(
       double dblSampleRate,      // precise sampling rate
       double dblUnixDateAndTime, // timestamp for the 1st sample in this block
       int    iNrOfChannels,      // number of channels (per sampling point)
       int    iNrOfSamplePoints,  // number of sample points (with 1..2 channels each)
       float* pfltSamplesChnA,    // 1st source channel, or both channels combined
       float* pfltSamplesChnB )   // 2nd source channel, optional (MAY BE NULL)
  // Puts the next block of audio samples (floats) into a CIRCULAR buffer,
  // from where it will be SENT to everyone who wants to receive these samples .
  // Caller (in SpectrumLab) : Real-time audio processing thread :
  //     TSoundThread::Execute() -> AVTCP_CopyAudioChunkForTransmission() .
  // The rest (=the actual "transmission" via the network) happens
  //
{
  int nValues = iNrOfSamplePoints * iNrOfChannels;
  DWORD dwBufIdxMask = (AVTCP_TX_BUFFER_SIZE-1);
  DWORD dwBufIdx = AVTCP_dwTxBufferHeadIdx & dwBufIdxMask;

  if( dblSampleRate <= 0 )
     return;  // avoid div-by-zero if the input parameters are invalid

  if( (pfltSamplesChnB!=NULL) && (iNrOfChannels>1) )
   { // two channels in separate source blocks:
     while( iNrOfSamplePoints-- )
      { AVTCP_i16TxAudioBuffer[dwBufIdx] = (short)*(pfltSamplesChnA++);
        dwBufIdx = (dwBufIdx+1) & dwBufIdxMask;
        AVTCP_i16TxAudioBuffer[dwBufIdx] = (short)*(pfltSamplesChnB++);
        dwBufIdx = (dwBufIdx+1) & dwBufIdxMask;
      }
   }
  else // any number of channels, but only ONE source block:
   {
     while( nValues-- )
      { AVTCP_i16TxAudioBuffer[dwBufIdx] = (short)*(pfltSamplesChnA++);
        dwBufIdx = (dwBufIdx+1) & dwBufIdxMask;
      }
   }
  // The next three instructions should be an atomic sequence (but we don't care)
  AVTCP_dwTxBufferHeadIdx = dwBufIdx;
  AVTCP_dblTxBufSampleRate= dblSampleRate;
  AVTCP_dblUnixDateAndTime= dblUnixDateAndTime + (double)iNrOfSamplePoints / dblSampleRate; // timestamp for the LATEST (!) sample

} // end AVTCP_CopyAudioChunkForTransmission()

//---------------------------------------------------------------------------
void AVTCP_CopyAudioChunkForTransmission_FLT64(
       double dblSampleRate,      // precise sampling rate
       double dblUnixDateAndTime, // timestamp for the 1st sample in this block
       int    iNrOfChannels,      // number of channels (per sampling point)
       int    iNrOfSamplePoints,  // number of sample points (with 1..2 channels each)
      double* pfltSamplesChnA,    // 1st source channel, or both channels combined
      double* pfltSamplesChnB )   // 2nd source channel, optional (MAY BE NULL)
  // Puts the next block of audio samples (floats) into a CIRCULAR buffer,
  // from where it will be SENT to everyone who wants to receive these samples .
  // Caller (in SpectrumLab) : Real-time audio processing thread :
  //     TSoundThread::Execute() -> AVTCP_CopyAudioChunkForTransmission() .
  // The rest (=the actual "transmission" via the network) happens
  //
{
  int nValues = iNrOfSamplePoints * iNrOfChannels;
  DWORD dwBufIdxMask = (AVTCP_TX_BUFFER_SIZE-1);
  DWORD dwBufIdx = AVTCP_dwTxBufferHeadIdx & dwBufIdxMask;

  if( dblSampleRate <= 0 )
     return;  // avoid div-by-zero if the input parameters are invalid

  if( (pfltSamplesChnB!=NULL) && (iNrOfChannels>1) )
   { // two channels in separate source blocks:
     while( iNrOfSamplePoints-- )
      { AVTCP_i16TxAudioBuffer[dwBufIdx] = (short)*(pfltSamplesChnA++);
        dwBufIdx = (dwBufIdx+1) & dwBufIdxMask;
        AVTCP_i16TxAudioBuffer[dwBufIdx] = (short)*(pfltSamplesChnB++);
        dwBufIdx = (dwBufIdx+1) & dwBufIdxMask;
      }
   }
  else // any number of channels, but only ONE source block:
   {
     while( nValues-- )
      { AVTCP_i16TxAudioBuffer[dwBufIdx] = (short)*(pfltSamplesChnA++);
        dwBufIdx = (dwBufIdx+1) & dwBufIdxMask;
      }
   }
  // The next three instructions should be an atomic sequence (but we don't care)
  AVTCP_dwTxBufferHeadIdx = dwBufIdx;
  AVTCP_dblTxBufSampleRate= dblSampleRate;
  AVTCP_dblUnixDateAndTime= dblUnixDateAndTime + (double)iNrOfSamplePoints / dblSampleRate; // timestamp for the LATEST (!) sample

} // end AVTCP_CopyAudioChunkForTransmission_FLT64()


//---------------------------------------------------------------------------
void AVTCP_DoNetworkStuff(void)
  // Check for timeouts, continues transmission with "limited bandwidth", etc .
  // Periodically called (every 50ms) from the application's "window procedure",
  //  possibly using a WM_TIMER event or something similar .
{
  int i;
  T_AVTCP_ServerConnection *pSrvConn;


  if( AVTCP_iHttpHdlrBusy > 0 )
     return;  // don't enter while we're working in AVTCP_HandleAsyncMsg(),
              // and prevent recursive calls while we're still "in here" .

  ++AVTCP_iHttpHdlrBusy;        // no "lazy return" after this !

  // Loop through all "connections" (in the HTTP server), and check if any of them
  // needs our special attention.
  // This feature was added 2008-01-02 for "slow transfers" in Spectrum Lab .
  // On this occasion, we can also update some statistics and housekeeping .
  for(i=0; i<AVTCP_SRV_MAX_CONNECTIONS ; ++i)
   { pSrvConn = &AVTCP_server_connections[i];

     // Is a client connected here, and waiting for something ?
     if( (pSrvConn->Socket!=INVALID_SOCKET) && (pSrvConn->fWaitingToContinueWrite) )
      { AVTCP_SRV_OnContinueWrite( pSrvConn );
      }

   } // end for

  if( AVTCP_iHttpHdlrBusy>0 )
    --AVTCP_iHttpHdlrBusy;

} // end AVTCP_DoNetworkStuff()



//===========================================================================
//  SERVER side of the audio-via-TCP/IP message protocol
//===========================================================================

//---------------------------------------------------------------------------
BOOL AVTCP_StartServer(
   int iAudioServerPort, // port number to connect the audio server via TCP/IP
   int iOptions,        // combination of AVTCP_OPTION_..-flags
   HWND hMainWindow,    // handle of the main window (required for winsock notifications)
   int  iWindowsMsgId,  // message ID for notifications from winsock to application
   AVTCP_RX_CALLBACK pvRxCallback,    // callback function for received audio
   AVTCP_LOG_CALLBACK pvLogCallback)  // callback function to show/log text messages
{
  SOCKADDR_IN  saServer;
  char  szBuf[256];
  int   i,nRet;

  // Init a few internal variables..
  memset( AVTCP_server_connections, 0, sizeof(AVTCP_server_connections) );
  for(i=0; i<AVTCP_SRV_MAX_CONNECTIONS; ++i)
   { AVTCP_server_connections[i].Socket = INVALID_SOCKET;  // marks an UNUSED ENTRY
   }

  // Save pointers to the callback functions ..
  AVTCP_pvRxCallback = pvRxCallback;  // callback function to handle received audio blocks
  AVTCP_pvLogCallback= pvLogCallback; // callback function to show/log text messages (AVTCP_LogEvent)

  // Save the Window handle and windows message ID for further use
  AVTCP_hMainWindow = hMainWindow;
  AVTCP_iWindowsMsg = iWindowsMsgId;
  AVTCP_iHttpHdlrBusy = 0;

  // Create a TCP/IP stream socket
  AVTCP_listenSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP );
  if (AVTCP_listenSocket == INVALID_SOCKET)
   {
     AVTCP_LogWSockError( "Could not create listen socket", WSAGetLastError());
     return FALSE;
   }

  // Request async notification
  // (or, less geek speak, prepare to receive windows messages
  //  when "something happens" on the socket)
  nRet = WSAAsyncSelect(AVTCP_listenSocket, AVTCP_hMainWindow, AVTCP_iWindowsMsg,
                         FD_ACCEPT | FD_READ | FD_WRITE | FD_CLOSE );
  if (nRet == SOCKET_ERROR)
   {
     AVTCP_LogWSockError( "WSAAsyncSelect() error", WSAGetLastError() );
     closesocket(AVTCP_listenSocket);
     AVTCP_listenSocket = INVALID_SOCKET;
     return FALSE;
   }

  // Always use the specified port number.
  //  Since this is not a standard service like HTTP, getservbyname doesn't work.
  saServer.sin_port = htons( (u_short)iAudioServerPort);

  // Fill in the rest of the address structure
  saServer.sin_family = AF_INET;
  saServer.sin_addr.s_addr = INADDR_ANY;

  // Bind our name to the socket
  nRet = bind(AVTCP_listenSocket, (LPSOCKADDR)&saServer, sizeof(struct sockaddr));
  if (nRet == SOCKET_ERROR)
   {
     AVTCP_LogWSockError( "bind() error", WSAGetLastError());
     closesocket(AVTCP_listenSocket);
     AVTCP_listenSocket = INVALID_SOCKET;
     return FALSE;
   }

  // Set the socket to listen
  nRet = listen(AVTCP_listenSocket, SOMAXCONN);
  if (nRet == SOCKET_ERROR)
   {
    AVTCP_LogWSockError( "listen() error", WSAGetLastError());
    closesocket(AVTCP_listenSocket);
    AVTCP_listenSocket = INVALID_SOCKET;
    return FALSE;
   }

  // Display the host name and address
  gethostname(szBuf, sizeof(szBuf));
  AVTCP_LogEvent( "Started Audio-via-TCP Server on %s:%d",  // computer_name:port
                   szBuf, htons(saServer.sin_port));
  return TRUE;
} // end AVTCP_StartServer()

//---------------------------------------------------------------------------
void AVTCP_StopServer(void)
{
  int i;
  T_AVTCP_ServerConnection *pSrvConn;

  // Forget callback vectors
  AVTCP_pvRxCallback = NULL;

  // Close the listening socket
  closesocket(AVTCP_listenSocket);
  AVTCP_listenSocket = INVALID_SOCKET;

  // Close all open connections
  for(i=0; i<AVTCP_SRV_MAX_CONNECTIONS ; ++i)
   { pSrvConn = &AVTCP_server_connections[i];
     if( pSrvConn->Socket != INVALID_SOCKET )
      { closesocket(pSrvConn->Socket);
        pSrvConn->Socket = INVALID_SOCKET;
      }
   }

  AVTCP_LogEvent("Audio-via-TCP Server Stopped");
  AVTCP_pvLogCallback = NULL;  // now AVTCP_LogEvent will be passive, too

} // end AVTCP_StopServer()

//---------------------------------------------------------------------------
BOOL AVTCP_IsServerRunning(void)
{
  return AVTCP_listenSocket != INVALID_SOCKET;

} // end AVTCP_IsServerRunning()

//---------------------------------------------------------------------------
T_AVTCP_ServerConnection * AVTCP_SocketToServerConnection( SOCKET sock )
  // Looks for a certain entry in the server's table of connections .
  // With sock=INVALID_SOCKET , this function can be used to find a free entry.
  // Return NULL means "not found" in the table .
{ int i;
  T_AVTCP_ServerConnection *pSrvConn;

  for(i=0; i<AVTCP_SRV_MAX_CONNECTIONS ; ++i)
   { pSrvConn = &AVTCP_server_connections[i];
     if( pSrvConn->Socket == sock )
      { return pSrvConn;
      }
   }
  return NULL;
} // end AVTCP_SocketToServerConnection()

//---------------------------------------------------------------------------
void AVTCP_SRV_HandleAsyncMsg( WPARAM wParam, LPARAM lParam)
  // Called from the application's "window procedure" after reception
  //  of the windows message for the Audio-via-TCP server
  //  (sent to the application from the windows socket service).
{
  int nErrorCode;

  ++AVTCP_iHttpHdlrBusy;        // no "lazy return" after this !

  nErrorCode = WSAGETSELECTERROR(lParam);

  switch(WSAGETSELECTEVENT(lParam))
   {
     case FD_ACCEPT:
        // Winsock received the connect() request and wants us
        //  to send back an accept() .
        // 2008-01-04: Only accepted a maximum of THREE simultaneous connections ?!
        AVTCP_SRV_OnAccept( (SOCKET)wParam, nErrorCode);
        break;

     case FD_READ:
        // Winsock notified us that the network buffer has more data for us
        // to collect.  Internally, we will call recv() . But caution, there are
        // many traps and pitfalls (even with this simple function, grrrr).
        // > The windows socket services will not continually flood an application
        // > with messages for a particular network event. Having successfully
        // > posted notification of a particular event to an application window,
        // > no further message(s) for that network event will be posted
        // > to the application window until the application makes the function
        // > call that implicitly reenables notification of that network event.
        // BUT (!) ...
        // > Any call to the reenabling routine, even one that fails, results
        // > in reenabling of message posting for the relevant event.
        // > For FD_READ, FD_OOB, and FD_ACCEPT events, message posting is
        // > level-triggered. This means that if the reenabling routine is called
        // > and the relevant condition is still met after the call,
        // > a WSAAsyncSelect message is posted to the application. This allows
        // > an application to be event-driven and not be concerned
        // > with the amount of data that arrives at any one time.
        // >         Consider the following sequence:
        // > - Network transport stack receives 100 bytes of data on socket s
        // >         and causes Windows Sockets 2 to post an FD_READ message.
        // > - The application issues recv( s, buffptr, 50, 0) to read 50 bytes.
        // > - Another FD_READ message is posted because there is still data to be read.
        // > With these semantics, an application need not read all available data
        // > in response to an FD_READ message  a single recv in response
        // >          to each FD_READ message is appropriate.
        AVTCP_SRV_OnRead( (SOCKET)wParam, nErrorCode);
        break;

     case FD_WRITE:
        // Winsock tells us that there is room in the buffer to send,
        //   and we must/should/may(?) send using the send() function .
        // 2008-01-02: Caution ! There may be a bug in microsoft's Winsock ...
        //  titled "BUG: Winsock--Lost FD_WRITE Notifications". From Q129974:
        // > SYMPTOMS
        // > In both of the following situations, the initial FD_WRITE notifications are never received:
        // > * Create a datagram socket and call WSAAsyncSelect() with interest
        // >   in FD_WRITE notification.
        // >
        // > -or-
        // > * Create a stream socket, call connect() to connect to a TCP server
        // >   and call WSAAsyncSelect() with interest in FD_WRITE notification.
        // >
        // For a better explanation of the FD_WRITE event, see AVTCP_SRV_OnWrite().
        AVTCP_SRV_OnWrite((SOCKET)wParam, nErrorCode);
        break;

     case FD_CLOSE:
        // Winsock tells us that the other sides socket has shut down.
        AVTCP_SRV_OnClose((SOCKET)wParam, nErrorCode);
        break;
   }

  if( AVTCP_iHttpHdlrBusy>0 )
    --AVTCP_iHttpHdlrBusy;

} // end AVTCP_SRV_HandleAsyncMsg()

//---------------------------------------------------------------------------
void AVTCP_SRV_OnAccept( SOCKET socket, int nErrorCode)
  // Called (indirectly) from Winsock service
  //  when a connection to our "audio-via-TCP-server" was requested.
{
  SOCKADDR_IN SockAddr;
  T_AVTCP_ServerConnection *pSrvConn;
  SOCKET      peerSocket;
  int         nRet;
  int         nLen;

  (void)nErrorCode;  // make BCB happy (parameter never used)
  (void)socket;

  // accept the new socket descriptor
  nLen = sizeof(SOCKADDR_IN);
  peerSocket = accept(AVTCP_listenSocket/*in*/, (LPSOCKADDR)&SockAddr/*out*/, &nLen/*out*/);
  if (peerSocket == SOCKET_ERROR)
   {
     nRet = WSAGetLastError();
     if (nRet != WSAEWOULDBLOCK)
      { // Just log the error and return
        AVTCP_LogWSockError( "accept() error", WSAGetLastError());
        return;
      }
   }

  // Make sure we get async notices for this socket
  nRet = WSAAsyncSelect(peerSocket, AVTCP_hMainWindow, AVTCP_iWindowsMsg, FD_READ | FD_WRITE | FD_CLOSE);
  if( nRet==SOCKET_ERROR )
   {
     nRet = WSAGetLastError();

     // Just log the error and return
     AVTCP_LogWSockError( "accept() error", nRet );
     return;
   }


  // Too many connections, or will we REALLY accept this request
  //  (and add it to the list) ?
  pSrvConn = AVTCP_SocketToServerConnection( INVALID_SOCKET );  // search for an "empty" server-entry
  if( pSrvConn==NULL )
   { // number of "acceptable" connections already exceeded !
     closesocket(peerSocket);
     AVTCP_LogEvent("Connection rejected from %s (too many connections)",
                       inet_ntoa(SockAddr.sin_addr) );
   }
  else
   {
     // Fill in the list of connected clients (THIS is the server)
     pSrvConn->Socket = peerSocket;
     pSrvConn->dwConnectTime = GetTickCount();
     pSrvConn->fWaitingToContinueWrite = FALSE;  // added 2008-01-02 for "slow sending" (bandwidth limited)
     pSrvConn->dwRecvd = 0;
     pSrvConn->dwSent  = 0;
     pSrvConn->eProtState = PS_CONNECTING;  // see state machine in AVTCP_SRV_OnRead()...
     pSrvConn->eRxState = RS_WAIT_HEADER;
     pSrvConn->i32RxByteCount = 0;
     AVTCP_LogEvent("Connection accepted on socket %d from %s",
                 (int)peerSocket, inet_ntoa(SockAddr.sin_addr) );
     // Shortly after this, we should get an "OnRead"-event for this socket !
   }
} // end AVTCP_SRV_OnAccept()


//---------------------------------------------------------------------------
void AVTCP_HandleRcvdBytes(T_AVTCP_ServerConnection *pSrvConn,
            BYTE *pbSource, int nBytesToProcess )
  // Handle the data, in the audio server,
  //  depending on what we are receiving at the moment :
  //  - "garbage/waiting for a header",  or
  //  - "header",  or
  //  - "audio samples" .
  // Fasten seat belt, we're using a tricky state machine...
{
  BOOL fHeaderOk;
  int  nBytesToProcessInBlock;
  BYTE *pbDst;

  while( nBytesToProcess > 0 )
   {
     // What are we receiving at the moment; an audio-message-header,
     // or digitized samples, or are we just waiting for the arrival of
     // the next recognizeable pattern ?
     switch( pSrvConn->eRxState )
      { case RS_WAIT_HEADER: // waiting for the begin of a header
           // To tell the begin of a header from "garbage",
           // the first 32-bit value in EACH header must be 0x80008000 .
           // Details in c:\CBproj\SoundUtl\audiomsg.h .
           // We don't want to make ANY assumption about the packet size here,
           // so -instead of blindly copying arrays- we use a 4-byte queue
           // to look at the header pattern .
           if( pSrvConn->i32RxByteCount < 3 )  // not enough bytes received yet..
            { // Bytes 0..2 copied into the header, treated as a BYTE-ARRAY:
              pSrvConn->hdr.b[pSrvConn->i32RxByteCount++] = *pbSource++;
              --nBytesToProcess;
            }
           else  // pSrvConn->i32RxByteCount == 3 :
            { // 4th byte received.. scroll up and look for valid pattern
              pSrvConn->hdr.b[3] = *pbSource++;
              --nBytesToProcess;
              pSrvConn->i32RxByteCount = 4;  // keep the number of received bytes limited to 4..
              if( pSrvConn->hdr.dw[0] == 0x80008000 )
               { // this MAY(!) be the begin of one of the audio-message-headers..
                 // enter the next state ("receive header").
                 // FOUR BYTES are already in there !
                 pSrvConn->eRxState = RS_HEADER;
                 pSrvConn->hdr.ami.i32SizeOfStruct = 0; // forget old struct size
               }
              else  // not the "magic number" for an audio message header... scroll up:
               { pSrvConn->hdr.dw[0] >>= 8; // this is an INTEL CPU, "low byte first", so:
                 // b[1] copied to b[0], b[2] copied to b[1],  b[3] to b[2],
                 // and the "OLDEST" byte (b[0]) falls out .
                 // The next received byte replaces b[3], see above .
               }
             }
           break; // end case RS_WAIT_HEADER
        case RS_HEADER :     // receiving an AUDIO MESSAGE HEADER
           if( pSrvConn->i32RxByteCount >= C_AUDIOMSG_MAX_HEADER_SIZE )
            { // oops .. max number of received bytes exceeded ?!
              pSrvConn->eRxState = RS_WAIT_HEADER; // wait for next header
              pSrvConn->i32RxByteCount = 0;
            }
           else // RX-byte-counter valid for the RS_HEADER state ...
            { pSrvConn->hdr.b[pSrvConn->i32RxByteCount++] = *pbSource++;
              --nBytesToProcess;
              // Time to IDENTIFY the header type ?
              if( pSrvConn->i32RxByteCount == C_AUDIOMSG_HEADER_IDENTIFIER_PART_SIZE )
               { // After the reception of a few (16?) bytes for the header,
                 // it's time to check if the header can be positively identified.
                 // If not, consider the received data as "garbage",
                 //         and jump back to the state "waiting for header".
                 // Is it really a valid "audio message header" ? See audiomsg.h !
                 fHeaderOk = ( pSrvConn->hdr.ami.i32Magic0x80008000 == AUDIOMSG_MAGIC_STRUCT_HEADER );
                 switch( pSrvConn->hdr.ami.i32StructCode )
                  { case AUDIOMSG_STRUCT_CODE_AudioMsgInfo:     /* T_AudioMsgInfo ?     */
                       fHeaderOk &= pSrvConn->hdr.ami.i32SizeOfStruct == (long)sizeof(T_AudioMsgInfo);
                       break;
                    case AUDIOMSG_STRUCT_CODE_AudioShortHeader: /* T_AudioShortHeader ? */
                       fHeaderOk &= pSrvConn->hdr.ash.i32SizeOfStruct == (long)sizeof(T_AudioShortHeader);
                       break;
                    case AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt:  /* T_AudioMsgInfoExt ? */
                       fHeaderOk &= pSrvConn->hdr.ami2.i32SizeOfStruct == (long)sizeof(T_AudioMsgInfoExt);
                       break;
                    case AUDIOMSG_STRUCT_CODE_AudioMsgQuery : // received a T_AudioMsgQuery : NOT SUPPORTED HERE YET !
                       fHeaderOk = FALSE;
                       break; // end case AUDIOMSG_STRUCT_CODE_AudioMsgQuery

                    default:  // unknown header type; discard it and wait for the next !
                       fHeaderOk = FALSE;
                       break;
                  }
                 if( ! fHeaderOk )  // what we received was garbage ->
                  { pSrvConn->eRxState = RS_WAIT_HEADER; // wait for next header
                    pSrvConn->i32RxByteCount = 0;
                  }
               } // end if < time to IDENTIFY the header type ? >
              if( pSrvConn->i32RxByteCount >= C_AUDIOMSG_MIN_HEADER_SIZE )
               { if( pSrvConn->i32RxByteCount >= pSrvConn->hdr.ami.i32SizeOfStruct )
                  { // FINISHED reception of a header.
                    // Calculate the number of bytes in the (uncompressed)
                    // audio data block which will follow directly after the header,
                    // and switch to the next RX-state .
                    pSrvConn->i32RxBytesWanted = AudioMsgBuf_CalcSampleBlockSizeInBytes( &pSrvConn->hdr );
                    if( pSrvConn->i32RxBytesWanted > (long)sizeof(AVTCP_i16RxAudioBuffer) )
                     {  // oops... cannot handle this amount of samples !
                        pSrvConn->i32RxBytesWanted = (long)sizeof(AVTCP_i16RxAudioBuffer);
                     }
                    if( pSrvConn->i32RxBytesWanted > 0 )
                     {
                       pSrvConn->eRxState = RS_AUDIO_SAMPLES; // after the header, receive the "samples"
                       pSrvConn->i32RxByteCount = 0;
                       pSrvConn->pi16RxAudioBuffer = AVTCP_i16RxAudioBuffer;
                       // ToDo: modifiy this when we have to receive audio from
                       // MORE THAN ONE CLIENT (which may connect to this server)
                     }
                    else // we're not expecting ANY data after this header..
                     {   // just inform the application, and wait for the next header.
                       if( AVTCP_pvRxCallback != NULL )
                        { (*AVTCP_pvRxCallback)( &pSrvConn->hdr, NULL/*buf*/, 0/*sample bytes*/ );
                        }
                       pSrvConn->eRxState = RS_WAIT_HEADER; // wait for next header
                       pSrvConn->i32RxByteCount = 0;
                     }
                  } // end if < FINISHED reception of a header >
               } // end if < minimum header size reached ? >
             } // end if < valid byte count for state RS_HEADER >
           break; // end case RS_HEADER
        case RS_AUDIO_SAMPLES: // receiving AUDIO SAMPLES : Pack them into a buffer,
           // and, when the RX-buffer is complete, let the application
           //  ( like SoundThd.cpp::SOUND_HandleRcvdAudioMessage) process the data.
           nBytesToProcessInBlock = pSrvConn->i32RxBytesWanted - pSrvConn->i32RxByteCount;
           if( nBytesToProcessInBlock > nBytesToProcess )
            {  nBytesToProcessInBlock = nBytesToProcess;
            }
           if( nBytesToProcessInBlock > 0 )
            {
              pbDst = (BYTE*)pSrvConn->pi16RxAudioBuffer;
              pbDst += pSrvConn->i32RxByteCount;     // number of bytes already received (for this audio block)
              pSrvConn->i32RxByteCount += nBytesToProcessInBlock;
              while( nBytesToProcessInBlock > 0 )  // copy bytes in a short loop to speed things up...
               { *pbDst++ = *pbSource++;
                 --nBytesToProcessInBlock;
                 --nBytesToProcess;
               }
              if( pSrvConn->i32RxByteCount >= pSrvConn->i32RxBytesWanted )
               { // another block of (uncompressed) audio samples is complete..
                 if( AVTCP_pvRxCallback != NULL )
                  { (*AVTCP_pvRxCallback)(&pSrvConn->hdr, (BYTE*)pSrvConn->pi16RxAudioBuffer, pSrvConn->i32RxBytesWanted );
                    // In SpecLab, the callback function is SOUND_HandleRcvdAudioMessage(),
                    // implemented in C:\cbproj\SpecLab\SoundThd.cpp .
                  }
                 pSrvConn->eRxState = RS_WAIT_HEADER; // wait for next header
                 pSrvConn->i32RxByteCount = 0;
               }
            } // end if( nBytesToProcessInBlock > 0 )
           break; // end case RS_AUDIO_SAMPLES
        default:  // oops.. switch back to the default state !
           pSrvConn->eRxState = RS_WAIT_HEADER;
           pSrvConn->i32RxByteCount = 0;
           break;
      } // end switch( pSrvConn->eRxState )
   }// end while( nBytesToProcess > 0 )
} // end AVTCP_HandleRcvdBytes()

//---------------------------------------------------------------------------
void AVTCP_SRV_OnRead(SOCKET socket, int nErrorCode)
  // Called (indirectly) from Winsock service
  //  when something shall be 'read' from the audio-via-TCP server (message FD_READ).
  //  This is usually the 2nd step after a connection was requested.
{
 BYTE  buf[2048];          // ex: "static", but we don't want static stuff in here
 char *cp;
 T_AVTCP_ServerConnection *pSrvConn;

 int          nRet;

#ifdef __BORLANDC__
 (void)nErrorCode;        // make BCB happy (parameter never used)
#else
 nErrorCode = nErrorCode; // make other compilers happy
#endif


  // Find this socket in the linked list
  pSrvConn = AVTCP_SocketToServerConnection(socket);
  if (pSrvConn == NULL)
   {
     // Not in the list. Log the error, read the data to clear the buffers,
     // and close the connection
     nRet = 0;
     while(nRet != SOCKET_ERROR)
      {    nRet = recv(socket, (char*)buf, sizeof((char*)buf)-1, 0);
      }
     closesocket(socket);
     return;
   }

  if( pSrvConn->eProtState != PS_RECEIVING_AUDIO ) // only if NOT "receiving audio" already..
   {
     // Recv the data
     //   For the FD_READ event, this is usually a string which tells us
     //   a lot about the visitor... including his OS and the browser he's using !
     //   A typical "request"-string easily exceeds 500 characters .
     // Zero the buffer so the recv is null-terminated
     memset(buf, 0, sizeof(buf));
     nRet = recv(socket, (char*)buf, sizeof( (char*)buf)-1, 0);
     if (nRet == SOCKET_ERROR)
      {
        if (WSAGetLastError() == WSAEWOULDBLOCK)
           return;
        AVTCP_LogWSockError( "recv()", WSAGetLastError() );
        AVTCP_SRV_CloseConnection(pSrvConn);
        return;
      }

     // Keep statistics
     pSrvConn->dwRecvd += nRet;
   }
  else // pSrvConn->eProtState == PS_RECEIVING_AUDIO :
   { // In this case, we will let the revc() put the data directly
     // in an audio sample buffer, to avoid unnecessary copy-operations .
     // Don't call recv() another time in this state .
   }

  // Are we just CONNECTING, RECEIVING AUDIO, or what ?
  switch( pSrvConn->eProtState )
   { case PS_CONNECTING:   // we're just beginning to connect, "talking in text mode" (!)
        // "Let him in"  ?
        //   If the FD_READ event was caused by a web browser (HTTP-request),
        //   the string in <buf> will tells us a lot about the visitor,
        //   including his OS and the browser he's using !  For example:
        // buf = "GET /HTTP/1.1\r\nHost: hf-test2:1024\r\nUser-Agent: Mozilla/5.0..."
        // Here, in the AUDIO-VIA-TCP/IP-SERVER, we don't deal with HTTP .
        //   A typical "request"-string easily exceeds 500 characters .
        cp = (char*)buf;
        if( strnicmp( cp, "GET ", 4) == 0 )   // HTTP-"GET" not supported !
         { // No, send "bad request" error
           AVTCP_SRV_SendHttpString( pSrvConn, "400 Bad Request - this is not an HTTP server !");
           AVTCP_LogEvent("AudioViaTCP rejected GET-request");
           AVTCP_SRV_CloseConnection( pSrvConn );
           return;
         }
        if( strnicmp( cp, "GET_AUDIO", 9) == 0 )
         { // client wants to receive an audio stream from this server .
           // let him join in (truly "accept" the connection) :
           pSrvConn->eProtState = PS_SENDING_AUDIO;
           AVTCP_LogEvent("AudioViaTCP accepted GET_AUDIO request");
           AVTCP_SRV_BeginToSendData( pSrvConn );
         }
        else
         { // No idea what to do with this request. Helplessly send an HTTP-like error:
           AVTCP_SRV_SendHttpString( pSrvConn, "501 Not Implemented" );
           AVTCP_LogEvent("AudioViaTCP rejected unknown request");
           AVTCP_SRV_CloseConnection( pSrvConn );
           return;
         }
        break; // end case PS_CONNECTING
     case PS_SENDING_AUDIO:   // we're sending audio data (BINARY data blocks)
        break; // end case PS_SENDING_AUDIO
     case PS_RECEIVING_AUDIO: // we're receiving audio data (BINARY data blocks)
        do
         {
           //   BTW no need to call "memset" in this case...
           nRet = recv(socket, (char*)buf, sizeof( (char*)buf)-1, 0);
           if (nRet == SOCKET_ERROR)
            { // Is it really an ERROR, or is it just "zero bytes received" ?!
              if (WSAGetLastError() != WSAEWOULDBLOCK)
               { AVTCP_LogWSockError( "recv()", WSAGetLastError() );
                 AVTCP_SRV_CloseConnection(pSrvConn);
               }
              return;
            }
           // Got here: Something received. Keep the statistics ..
           pSrvConn->dwRecvd += nRet;
           // .. and handle the received bytes, depending on the decoder state:
           AVTCP_HandleRcvdBytes( pSrvConn, buf, nRet );
         }while(nRet>0); // repeat this loop until the network buffer is empty (!)
        break; // end case PS_RECEIVING_AUDIO
     default:
        break;
   } // end switch (eProtState)

} // end AVTCP_SRV_OnRead()

//---------------------------------------------------------------------------
void AVTCP_SRV_OnWrite(SOCKET socket, int nErrorCode)
   // Called from .._HandleAsyncMsg() on the FD_WRITE event.
   //
   // As always, this function is more complicated than you may think !
   // From www.gamedev.net/reference/programming/features/asyncsock/page5.asp,
   //  written by Drew Sikora :
   //
   // > Now the FD_WRITE event is a bit more complicated. Mainly because
   // > of the way it is called. First of all, an FD_WRITE event is always
   // > generated when you first establish a connection with another socket.
   // > So you think all you have to do is plunk in a send() function
   // >              with whatever else youll need and youll be fine .
   // > Oh if that were only true. Listen carefully: the FD_WRITE event
   // > is triggered only when there is more room in the buffer with which
   // > to write data. Huh? Allow me to explain.
   // >
   // > First off, the buffer, as I use it here, is the network buffer,
   // > which you write information to. That information is then sent
   // > over the network (intranet or internet) to the receiving socket.
   // > So youd think that since there will always be space left to write data
   // > if you dont fill it up all at once, that FD_WRITE events will just
   // > keep coming and coming right? Wrong! Read the rules again.
   // > It says when there is more room. Not enough room, more room.
   // > This means you have to fill it up first! How do you do that?
   // >
   // > The general idea is to create an infinite while loop in which you will
   // > continuously send data until you max out the buffer. When is happens,
   // > send() will return the error WSAWOULDBLOCK. This means that if it were
   // > a blocking socket, it would wait (stop execution) for more room
   // > in the buffer and then send. But since it isnt, you get the error.
   // > So now that youve filled up the buffer, you just have to wait until
   // > more room becomes available so you can write again. And bingo!
   // > Up pops another FD_WRITE event.
   // > Do you have any idea how much trouble I had to go through to figure this out?
   // > You people are so darned lucky! Heres an example of an FD_WRITE event handler:
   // > (..)
   // > The implementation isnt so hard after all! It was only the concept
   // > that was messing with your head. If you use this technique, the loop
   // > will end when the network buffer becomes full.  (COMPLETELY FULL, that is..)
   // > Then, when more space is available - another FD_WRITE event will
   // > be triggered, and you can send more data.
   // > Before you jump to make use of your newfound knowledge, allow me
   // > to clarify the use of the FD_WRITE event. Do not expect to be using
   // > this event at all if you are not going to be sending large pieces of data
   // > at once. The reason for this is simple - if you base all you information
   // > transactions on FD_WRITE and never send enough to fill up the buffer,
   // > after that first trial FD_WRITE event generated at the beginning
   // > of the connection - no other FD_WRITE event will be triggered !
   // > Therefore for games, where you send as little data as possible,
   // > just forget about setting a socket for the FD_WRITE event and use send()
   // > wherever you need it.
   // > FD_WRITE is best used for transferring files, in my experience.
   //
   // DL4YHF: What a nice article, it contains everything which is MISSING
   //         (or at least hard to find) in the original winsock documentation.
{
  T_AVTCP_ServerConnection *pSrvConn;
  BYTE buf[1024];
  int nRet;

#ifdef __BORLANDC__
 (void)nErrorCode;        // make BCB happy (parameter never used)
#else
 nErrorCode = nErrorCode; // make other compilers happy
#endif


  pSrvConn = AVTCP_SocketToServerConnection(socket);
  if ( pSrvConn == NULL)
   {
     //
     // Not in our list ?  Dump the received data, and bark out .
     nRet = 0;
     while(nRet != SOCKET_ERROR)
           nRet = recv(socket, (char*)buf, sizeof( (char*)buf )-1, 0);
     closesocket(socket);
     return;
   }

  // Continue sending the stream (or whatever it is)
  AVTCP_SRV_SendData( pSrvConn );
} // end AVTCP_SRV_OnWrite()

//---------------------------------------------------------------------------
void AVTCP_SRV_OnContinueWrite( T_AVTCP_ServerConnection *pSrvConn )
   // Called periodically for a socket opened for SENDING DATA,
   //   for example for a file transfer which hasn't been sent completely.
   //   The reason why we need this function (in addition to HttpSrv_OnWrite)
   //   is explained in AVTCP_SRV_OnWrite() .
   // Caller: periodically, usually from some fancy (and jittery) windows timer .
{
   pSrvConn->fWaitingToContinueWrite = FALSE; // "done"
} // end AVTCP_SRV_OnContinueWrite()

//---------------------------------------------------------------------------
void AVTCP_SRV_OnClose(SOCKET socket, int nErrorCode)
{
  T_AVTCP_ServerConnection * pSrvConn;

#ifdef __BORLANDC__
 (void)nErrorCode;        // make BCB happy (parameter never used)
#else
 nErrorCode = nErrorCode; // make other compilers happy
#endif

  // Have we already deleted this entry?
  pSrvConn = AVTCP_SocketToServerConnection(socket);
  if (pSrvConn == NULL)
     return;

  // It's still in our list: The client must have reset the connection.

  // Clean up.
  AVTCP_SRV_CloseConnection( pSrvConn );
}


//---------------------------------------------------------------------------
void AVTCP_SRV_CloseConnection( T_AVTCP_ServerConnection * pSrvConn )
{
  DWORD dwElapsedTime_ms;

  dwElapsedTime_ms = (GetTickCount() - pSrvConn->dwConnectTime);

  // Log the event ...
  AVTCP_LogEvent("Closing socket %d, ti=%.1f s, rcvd=%ld, sent=%ld byte.",
           (int)(pSrvConn->Socket),
          (float)dwElapsedTime_ms * 1e-3,
          (long)(pSrvConn->dwRecvd),
          (long)(pSrvConn->dwSent)  );

  // Don't call HttpSrv_OnContinueWrite periodically (connection being closed !)
  pSrvConn->fWaitingToContinueWrite = FALSE;

  // Close the socket
  closesocket(pSrvConn->Socket);

  pSrvConn->Socket = INVALID_SOCKET; // this marks the list entry as "unused" !


} // end AVTCP_SRV_CloseConnection()


//---------------------------------------------------------------------------
void AVTCP_SRV_BeginToSendData( T_AVTCP_ServerConnection * pSrvConn )
  // Begin to send a file (requested from this HTTP server) .
  // Call Tree :
  //    AVTCP_SRV_HandleAsyncMsg()
  //      -> AVTCP_SRV_OnRead()        ("someone wants to read a file from us")
  //              -> AVTCP_SRV_BeginToSendData()
  //                   -> AVTCP_SRV_SendData()
{

  // Open the file for reading
  // if( 1 )
   {
     AVTCP_SRV_SendHttpString( pSrvConn, "404 Not Found" );
     AVTCP_SRV_CloseConnection( pSrvConn );
     return;
   }

  // Send as much of the file as we can
  // AVTCP_SRV_SendData( pSrvConn );
  // ( this is only the FIRST part of the audio stream ! )

} // end AVTCP_SRV_BeginToSendData()

//---------------------------------------------------------------------------
void AVTCP_SRV_SendHttpString( T_AVTCP_ServerConnection * pSrvConn, char *pszMsg )
{
  int nRet;
  static char szMsg[512];
  wsprintf(szMsg, "<body><h1>%s</h1></body>", pszMsg );
  nRet = send( pSrvConn->Socket, szMsg, strlen(szMsg), 0);
  if (nRet == SOCKET_ERROR)
   {
     if (WSAGetLastError() != WSAEWOULDBLOCK)
      {
        AVTCP_LogWSockError( "send()@SendError()", WSAGetLastError() );
        nRet = 0;
      }
   }
  pSrvConn->dwSent += nRet;
} // end AVTCP_SRV_SendHttpString()

//---------------------------------------------------------------------------
void AVTCP_SRV_SendData( T_AVTCP_ServerConnection * pSrvConn )
  // Server side: Sends "as much as possible" of the audio stream,
  //              which has already been prepared for sending.
{
  int nBytesToSend, nBytesSent;
  BYTE buf[4096];

  do
   {
     nBytesToSend = 0;

     // Send this buffer to the client
     nBytesSent = send( pSrvConn->Socket, (char*)buf, nBytesToSend, 0);
     if (nBytesSent == SOCKET_ERROR)
      { // This sometimes happens if the client has closed the connection,
        //  usually while debugging because the web browser "times out".
        if (WSAGetLastError() != WSAEWOULDBLOCK)
         {
           AVTCP_LogWSockError( "send()", WSAGetLastError() );

           AVTCP_SRV_CloseConnection( pSrvConn );
           return; // return immediately because pSrvConn is not valid any longer
         }
        nBytesSent = 0;
      }

     // Keep track of what has actually been sent (for the statistics)
     pSrvConn->dwSent += nBytesSent;

     // Are the protocol stack buffers full?
     if (nBytesSent < (int)nBytesToSend)
      {
        // We'll have to finish later .  *ppReq remains valid in this case .
        return;
      }
   }while(nBytesToSend>0);


} // end AVTCP_SRV_SendData()





//===========================================================================
//  CLIENT side of the audio-via-TCP/IP message protocol : NOT IMPLEMENTED YET
//===========================================================================

//---------------------------------------------------------------------------
void AVTCP_CLI_HandleAsyncMsg( WPARAM wParam, LPARAM lParam)
  // Must be called by the application (from the windows message loop)
  // when winsock sends a notification for the Audio-via-TCP port, SERVER side.
  // The message ID must be passed to AVTCP_StartServer( .. iWindowsMsgId .. ),
  // to configure winsock to send such messages (which geeks call "async") .
{


} // end AVTCP_CLI_HandleAsyncMsg()


// EOF < AudioViaTCP.c >