/***************************************************************************/
/*  WaveIO.h :                                                             */
/*   Unit to import and export *.wav - files.                              */
/*   - by DL4YHF August '2000                                              */
/*   - based on the description of the WAVE-file format                    */
/*     found at http://www.wotsit.org  (excellent site, all file formats!  */
/*-------------------------------------------------------------------------*/

#ifndef _AUDIO_TEXT_IO_H_
#define _AUDIO_TEXT_IO_H_


#include "QFile.h"  // DL4YHF's "quick file" routine (read+write text files)

#ifndef  _AUDIO_FILE_DEFS_H_
# include "AudioFileDefs.h"
#endif


/*----------- Constant Definitions ----------------------------------------*/

#define ATIO_FILE_FORMAT_PLAIN      0
#define ATIO_FILE_FORMAT_COOL_EDIT  1



/*----------- Data Types --------------------------------------------------*/


#ifndef  SWI_FLOAT_PRECISION   // should be defined under Project..Options..Conditionals
 #define SWI_FLOAT_PRECISION 1 /* 1=single precision, 2=double precision */
#endif

#ifndef T_Float
#if SWI_FLOAT_PRECISION==1
 #define T_Float float
 // 2003-07-07: "float" instead of "double" saves CPU power on stoneage notebooks
#else
 #define T_Float double
#endif
#endif // ndef T_Float



/*----------- Variables with "C"-style single-source principle ------------*/

#undef EXTERN
#ifdef _I_AM_AUDIO_TEXT_IO_
 #define EXTERN
#else
 #define EXTERN extern
#endif


/*----------- Definition of the C_AudioTextIO  class -----------------------*/
class C_AudioTextIO
{
public:
   C_AudioTextIO();
   virtual ~C_AudioTextIO();

   // Public member functions for object's users.
   /************************************************************************/
   BOOL InOpen( char *pszFileName );
     /* Opens a wave file for READING it's samples.
      * Returns TRUE on success and FALSE on any error.
      */

   /************************************************************************/
   DWORD GetTotalCountOfSamples(void);
     /* Returns the total number of sampling points in the file .
      */

   /************************************************************************/
   DWORD GetCurrentSampleIndex(void);
     /* Reads the current sample index that will be used on the next
      * READ- or WRITE- access to the audio samples in the opened file.
      * Returns 0xFFFFFFFF on any error.
      */


   /************************************************************************/
   BOOL WindToSampleIndex(DWORD dwNewSampleIndex);
     /* Sets the current sample index that will be used on the next
      * READ- or WRITE- access to the audio samples an opened file.
      * Returns TRUE if ok ,  otherwise FALSE.
      * As long as you don't use this routine/method,
      * the audio file will be played in a strictly "sequential" way.
      */


   /************************************************************************/
   LONG ReadSampleBlocks(
         int channel_nr,    // channel_nr for 1st destination block
         int nr_samples,    // nr_samples
         T_Float *pFltDest1, // 1st destination block (~"LEFT")
         T_Float *pFltDest2); // 2nd destination block (~"RIGHT")
     /* Reads some samples from a wave file which has been opened for READING
      * and converts them to double (floats) in the range of +- 32768.0
      * Input parameters:
      *    channel_nr: 0=first channel (LEFT if stereo recording)
      *    nr_samples: Max number of samples which can be stored in T_Float dest[0..nr_samples-1]
      *    *pFltDestX = Pointers to destination arrays (one block per channel, not "paired")
      * Returns the NUMBER OF AUDIO SAMPLES if successful,
      *         or a negative value if errors occurred.
      * Note that returning less samples than you may expect is not an error
      * but indicates reaching the file's end !
      */



   /************************************************************************/
   BOOL WriteHeader( T_ChunkInfo *pChunkInfo ); 
   BOOL OutOpen( char *file_name,
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int iHeaderOptions,
           int bits_per_sample,   // 8,16,24=integer, 32 or 64=floating point !
           int channels,
           T_Float dbl_sample_rate);
     /* Creates and Opens an audio file for WRITING audio samples.
      *         Returns TRUE on success and FALSE on any error.
      *         Detailed description of arguments in WaveIO.cpp only.
      */


   /************************************************************************/
   LONG WriteSamples( SHORT* pi16Data, LONG Length); // low-level output, try to avoid
   LONG WriteSamples_Float( // app-layer output, using NORMALIZED floats
      T_Float *pfltSrc, T_ChunkInfo *pChunkInfo, char *pszExtraInfo );

   /************************************************************************/      
   long   GetParameterByID( int iAudioFileParam );
            // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h .
            // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015.
   BOOL   SetParameterByID( int iAudioFileParam, long i32NewValue );
            // Returns TRUE if the new value could be set, otherwise FALSE.
            // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015.
   double GetParameterByID_Double(int iAudioFileParam); // similar as above, for 64-bit "double" ..
   BOOL   SetParameterByID_Double(int iAudioFileParam, double dblNewValue);



   /************************************************************************/
   // Some more 'Get' - and 'Set' - functions for the AUDIO FILE class ..
   void   GetFileName(char *pszDest, int iMaxLen);
   double GetSampleRate(void);
   void   SetSampleRate(double dblSampleRate);

   int    GetNrChannels(void);
   int    GetBitsPerSample(void);

   double GetFrequencyOffset(void); // difference between "radio-" minus "baseband-" frequency (internal NCO + other internal frequency shift + external VFO )
   BOOL   SetFrequencyOffset(double dblFrequencyOffset);

   double GetCurrentRecordingTime(void); // see specification in c:\cbproj\SoundUtl\AudioFileIO.h (t=0 is the RECORDING START TIME, aka "track time")
   double GetRecordingStartTime(void);   // seconds elapsed since 00:00:00 GMT, January 1, 1970.
   void   SetRecordingStartTime(double dblStartTime);

   bool   AllParamsIncluded(void);

   /************************************************************************/
   BOOL CloseFile( void );
     /* Closes the WAV-file (if opened for READING exor WRITING.
      * Returns TRUE on success and FALSE on any error.
      */


   // Public properties for simplicity .. we know what we're doing, right ?

   double m_dblStartTime;      // seconds since 00:00:00 GMT, January 1, 1970.
   int    m_iATIOFileFormat;   // ATIO_FILE_FORMAT_xxxx
   char   m_cSeparatorChar;
   char   m_cErrorString[81];
   BOOL   m_OpenedForReading;
   BOOL   m_OpenedForWriting;
   double m_dbl_SampleRate;     // will be private sooner or later..
   double m_dblFrequencyOffset; // "radio-" minus "baseband"-frequency
   int    m_iChannelsPerSample;
   int    m_iBitsPerSample;

private:
   T_QFile m_QFile;        // data structures and buffer for "quick file access"
   DWORD  m_dwFilePos_Data;    // file-internal position of 1st audio sample
   DWORD  m_dwCurrFilePos;     // current file access position
   DWORD  m_dwCurrSampleIndex; // audio sample index, connected to lCurrFilePos
   DWORD  m_dwFileDataSize;    // on READ: taken from "data" chunk
   DWORD  m_dwFileNrOfSamples; // total number of samples in file (may be taken from header)


}; /* end class C_AudioTextIO */


#endif // _AUDIO_TEXT_IO_H_

/* EOF <AudioTextIO.h> */