/***************************************************************************/
/*  AudioTextIO.cpp:                                                       */
/*   Unit to import and export (short) audio samples as TEXT files .       */
/*  - by DL4YHF November '2004                                             */
/*  - based on the WaveIO class                                            */
/*  - used in Spectrum Lab to analyse files produced with PC voltmeters    */
/*-------------------------------------------------------------------------*/

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h> // definition of data types like LONGLONG (wtypes.h) etc
#include <stdlib.h>
#include <stdio.h>
#include <io.h>          /* _rtl_open and other file functions       */
#include <fcntl.h>       /* O_RDONLY  and other file flags          */
#include <time.h>
#include "Timers.h"    // high-resolution 'current time'-routine
#include "utility1.h"  // some helper functions by DL4YHF (\CBproj\SoundUtl\utility1.cpp)
#ifndef C_SND_MATH_INVALID_FLOAT_VALUE
# include "SoundMaths.h"
#endif
// #include <exception>

#pragma hdrstop
#pragma warn -8004 // <var is a assigned a value that is never used> - so what

#define _I_AM_AUDIO_TEXT_IO_ 1
#include "AudioTextIO.h"  // header for this file with "single source"-variables


/*------------- Implementation of the class  C_AudioTextIO ----------------*/

/***************************************************************************/
C_AudioTextIO::C_AudioTextIO()
  /* Constructor of the Wave-IO Class.
   * Initializes all properties etc.
   */
{
   m_QFile.iHandle    = -1;
   m_OpenedForReading = FALSE;
   m_OpenedForWriting = FALSE;
   m_dblStartTime     = 0.0;
   m_dwCurrFilePos     = 0;
   m_dwFileDataSize    = 0;
   m_cErrorString[0] = '\0';
   m_cSeparatorChar  = ' ';
   m_iATIOFileFormat = ATIO_FILE_FORMAT_PLAIN;
   m_iChannelsPerSample = 1;
   m_iBitsPerSample  = 16;
} // end of C_AudioTextIO's constructor

/***************************************************************************/
C_AudioTextIO::~C_AudioTextIO()
  /* Destructor of the Wave-IO Class.
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   */
{


} // end C_AudioTextIO's destructor


/***************************************************************************/
BOOL C_AudioTextIO::InOpen( char *pszFileName )
  /* Opens a text file for READING it's samples.
   * Returns TRUE on success and FALSE on any error.
   */
{
 int i;
 struct ftime f_time;
 char  sz1023Temp[1024];
 BYTE  *pb;
 long  i32LineStartPos;

 BOOL ok = TRUE;

  CloseFile();   // Close "old" file if it was open

  sz1023Temp[1023] = '\0';   // ensure string is terminated
  m_QFile.sz512PathAndName[0] = '\0';;
  m_dwCurrSampleIndex = 0;  // reading starts at audio sample index 0 .
  m_dwFileNrOfSamples = 0xFFFFFFFF; // don't know how many samples are in the file

  if( QFile_Open(&m_QFile, pszFileName, O_RDONLY ) )
   {
    // Get the "file time" as a first coarse value when the file has been
    // recorded. Note that "getftime" returns its own time structure
    // which is very different from our internal "time()"-like format.
    // "getftime" returns the date relative to 1980,
    // but SpecLab uses the Unix-style format which is
    //    "Seconds elapsed since 00:00:00 GMT(=UTC), January 1, 1970"
    // how many SECONDS between the UNIX-time(est'd 1970) and 1980 ?
    // T_Float d1,d2; BYTE *pSource;
    // pSource = "1970-01-01 00:00:00";
    // UTL_ParseFormattedDateAndTime( "YYYY-MM-DD hh:mm:ss", &d1, &pSource); // -> 0 (!)
    // pSource = "1980-01-01 00:00:00";
    // UTL_ParseFormattedDateAndTime( "YYYY-MM-DD hh:mm:ss", &d2, &pSource);
    // d2-=d1;
     // (should be something like 365*24*60*60 = 31536000, in fact it's 315532800)

    if( getftime(m_QFile.iHandle, &f_time)==0) // "0" = "no error"
     {
       m_dblStartTime =
         UTL_ConvertYMDhmsToUnix( // parameters for this routine:
         f_time.ft_year+1980,  // int year  (FULL year, like 2002, not crap like "year-1980",
         f_time.ft_month,      // int month (1..12)
         f_time.ft_day,        // int day   (1..31)
         f_time.ft_hour,    // int hour  (0..23) !! not the nerving "AM PM" stuff please... :-)
         f_time.ft_min,     // int minute (0..59)
         2*f_time.ft_tsec); // T_Float dblSecond  (0.000...59.999)
       // By order of the Prophet, "dblStartTime" should really contain the start time.
       // The FILE TIME is usually the time of the last write access, which will be
       // the LAST sample. So we must subtract the duration of the recording
       // as soon as we know it [but we don't know it here because the file header
       // has not been analysed yet].
     }
    else m_dblStartTime=0;


    // Analyse the (possible) header of the TEXTFILE .
    //   Note: There may not be a header at all ! !
    while(1)
     {
       i32LineStartPos = QFile_Seek(  &m_QFile, 0, SEEK_CUR );
       i = QFile_ReadLine( &m_QFile,  sz1023Temp, 1023 );
       if(i<=0)
        { // back to begin of last line, and finish reading the header !
          QFile_Seek(  &m_QFile, i32LineStartPos, SEEK_SET );
          break;
        }
       else
        {  // look at the textline. Does it look like a HEADER, or like DATA ?
         pb = (BYTE*)sz1023Temp;
         while(*pb==' ' || *pb=='\t') ++pb;  // skip leading spaces (or tabs ?)
         if(  (*pb>='0' && *pb<='9') || (*pb=='+') || (*pb=='-') )
          {  // This sure is no HEADER LINE, but a textline with DATA !
             //  Stop reading the "header" (if it can be called a header at all)
            QFile_Seek(  &m_QFile, i32LineStartPos, SEEK_SET ); // back to begin of line
            break;
          }
         else // no data line, but .. a HEADER ?
          {  // Here is a sample header from Cool Edit (logsample_by_renato.txt) :
             // SAMPLES:	5999
             // BITSPERSAMPLE:	16
             // CHANNELS:	2
             // SAMPLERATE:	6000
             // NORMALIZED:	FALSE
             // -3514	7148
             // -1922	-8069
             //  ....
            if( UTL_CheckAndSkipToken( &pb, "SAMPLES:" ) )
             { m_dwFileNrOfSamples = UTL_ParseInteger( &pb, 8/*max digits*/ );
             } else
            if( UTL_CheckAndSkipToken( &pb, "BITSPERSAMPLE:" ) )
             { m_iBitsPerSample = UTL_ParseInteger( &pb, 8/*max digits*/ );
             } else
            if( UTL_CheckAndSkipToken( &pb, "CHANNELS:" ) )
             { m_iChannelsPerSample = UTL_ParseInteger( &pb, 8/*max digits*/ );
             } else
            if( UTL_CheckAndSkipToken( &pb, "SAMPLERATE:" ) )
             { m_dbl_SampleRate = UTL_ParseInteger( &pb, 8/*max digits*/ );
             } else
            if( UTL_CheckAndSkipToken( &pb, "NORMALIZED:" ) )
             { // Ignore, no idea what CoolEdit needs this information for
             }

          } // end else < no DATA line >
        } // end if <non-empty line read from file - HEADER ? >
     } // end while(1)
   } // end if <file opened successfully>
  else
   { strcpy(m_cErrorString,"_open failed");
     ok = FALSE;
   }

  if (ok)
   {
     // save file-internal position of 1st audio sample
     // (just in case someone wants to "Rewind" etc).
     i32LineStartPos = QFile_Seek(  &m_QFile, 0, SEEK_CUR );
     if( i32LineStartPos >= 0)
      { m_dwFilePos_Data = i32LineStartPos;
        m_dwCurrFilePos  = i32LineStartPos;
      }
     else // problem with QFile_Seek ?
      { strcpy(m_cErrorString,"QFile_Seek failed");
        m_dwCurrFilePos = 0;
        ok = FALSE;
      }
   }

  if (!ok)
   {
    if(m_cErrorString[0]=='\0')
      strcpy(m_cErrorString,"other 'open' error");
    CloseFile();  // close input file if opened but bad header
    return FALSE;
   }
  else
   {
    m_OpenedForReading = ok;
    return TRUE;
   }
} // C_AudioTextIO::InOpen(..)

/************************************************************************/
long C_AudioTextIO::GetParameterByID( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h.
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example GetVTFlags() -> GetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS) .
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM__FILE_FORMAT :
        // Replacement for   int GetFileFormat(void) .
        //  The parameter (iAudioFileFormat) must be one of the
        //  AUDIO_FILE_FORMAT_...-constants defined in AudioFileDefs.
        return m_iATIOFileFormat;

     case AUDIO_FILE_PARAM__FILE_SIZE_KBYTE :
        // Returns the current file size (if there is an opened file) in KBYTE.
        // Useful especially when logging audio samples  to keep track of the used disk space.
        // Returns a negative value on any error.
        // ex: if (m_OpenedForWriting || m_OpenedForReading) .. modified 2016, we want to know the filesize AFTER CLOSING !
        return m_dwCurrFilePos  / 1024;

     case AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT :
        // Returns the NUMBER OF CHANNELS PER SAMPLE. Ususally 1 or 2.
        return GetNrChannels();

     case AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX :
        // Read the current sample index that will be used on the next
        // READ- or WRITE- access to the audio samples in the opened file.
        // Returns a negative value on any error.
        // Return value : sample index; 0xFFFFFFFF = ERROR ("4 Gig minus one")
        return GetCurrentSampleIndex();

     case AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES :
        // Returns the total number of sampling points in the file.
        // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .
        return GetTotalCountOfSamples();

     case AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT :
        // A "sample point" contains all channels, sampled at the very same time.
        // Example: Stereo Recording, 16 bits per (single) sample
        //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
        // Used to convert a "sample index" into a "file offset" .
        return ( GetNrChannels() * GetBitsPerSample() ) / 8;

     case AUDIO_FILE_PARAM__CONNECTION_LOST        :
        // Returns TRUE when a previously opened network connection was lost,
        //  and the application should take action (like wait a few seconds,
        //  and then try to reconnect if configured that way) .
        // Used to check the "connected" status for web streams.
        // In a few (rare) cases, can also be used to SET the "connect"-status
        // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )
        return FALSE;

     case AUDIO_FILE_PARAM__NUM_TIMESTAMPS         :
        // Returns the number of timestamps received (or sent), since the stream was started
        return 0;

     case AUDIO_FILE_PARAM__VLF_TOOL_FLAGS         :
        // VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8, for non-compressed streams,
        //   compatible with Paul Nicholson's VLFRX tools.
        //   The VT-Flags themselves are defined in vlfrx-tools/vtlib.h .
        return 0;

     default:
        break;
   } // end switch( iAudioFileParam )

  // Arrived here ? The requested parameter is not supported.
  // KISS, return a negative value, because ZERO is usually a valid value for future parameters.
  return -1;

} // end C_AudioTextIO::GetParameterByID()


//---------------------------------------------------------------------------
double C_AudioTextIO::GetParameterByID_Double( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..-numbers for which INTEGER
  //         values are not sufficient, e.g. AUDIO_FILE_PARAM_LAST_TIMESTAMP .
  // May return C_SND_MATH_INVALID_FLOAT_VALUE for unsupported parameters .
  // Full specs in cbproj/SoundUtl/AudioFileIO.cpp : GetParameterByID_Double().
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM_LAST_TIMESTAMP :
        // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
        // Value doesn't change unless NEW timestamps arrive !
        break;
     case AUDIO_FILE_PARAM_TIMESTAMP_LATENCY :
        // Stream latency, determined by comparing the local time and the received one,
        // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
        break;
     default:
        break;
   }
  return C_SND_MATH_INVALID_FLOAT_VALUE; // parameter not supported by this incarnation of GetParameterByID_Double() !

} // end C_AudioTextIO::GetParameterByID_Double()


/************************************************************************/
BOOL C_AudioTextIO::SetParameterByID( int iAudioFileParam, long i32NewValue )
  // Returns TRUE if the new value could be set, otherwise FALSE.
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example SetVTFlags(iVTFlags) -> SetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS,..) .
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM__FILE_FORMAT :
        // Replacement for SetFileFormat() .
        return FALSE;

     case AUDIO_FILE_PARAM__FILE_SIZE_KBYTE :
        return FALSE;  // this parameter will always be READ-ONLY !

     case AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT :
        // Sets the NUMBER OF CHANNELS PER SAMPLE. Ususally 1 or 2.
        // Important to read RAW files, where the number of channels isn't in the header.
        return FALSE;  // this parameter is not writeable... yet ?

     case AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX :
        // Sets the current sample index that will be used on the next
        //  READ- or WRITE- access to the audio samples an opened file.
        //  Returns TRUE if ok ,  otherwise FALSE.
        return FALSE;

     case AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES :
        // The total number of sampling points in the file.
        // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .
        return FALSE; // "parameter is not writeable"

     case AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT :
        // A "sample point" contains all channels, sampled at the very same time.
        // Example: Stereo Recording, 16 bits per (single) sample
        //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
        // Used to convert a "sample index" into a "file offset" .
        return FALSE; // "parameter is not writeable"

     case AUDIO_FILE_PARAM__CONNECTION_LOST        :
        // Used to check the "connected" status for web streams.
        // In a few (rare) cases, can also be used to SET the "connect"-status
        // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )
        return FALSE; // "parameter cannot be set"

     case AUDIO_FILE_PARAM__NUM_TIMESTAMPS         :
        // Returns the number of timestamps received, since stream started
        return FALSE; // "parameter cannot be set"

     case AUDIO_FILE_PARAM__VLF_TOOL_FLAGS         :
        return FALSE; // "parameter cannot be set"

     default:
        break;
   } // end switch( iAudioFileParam )

  // Arrived here ? The requested parameter is not supported. Life can be so simple.
  return FALSE;

} // end C_AudioTextIO::SetParameterByID()

//---------------------------------------------------------------------------
BOOL C_AudioTextIO::SetParameterByID_Double( int iAudioFileParam, double dblNewValue )
  // Almost the same as SetParameterByID(), but for 64-bit "double".
  // Returns TRUE if the specified parameter could be set, otherwise FALSE .
  // Full specs in cbproj/SoundUtl/AudioFileIO.cpp : SetParameterByID_Double().
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM_LAST_TIMESTAMP :
        // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
        // Value doesn't change unless NEW timestamps arrive !
        break;
     case AUDIO_FILE_PARAM_TIMESTAMP_LATENCY :
        // Stream latency, determined by comparing the local time and the received one,
        // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
        break;
     default:
        break;
   }
  return FALSE; // parameter not supported by this incarnation of GetParameterByID_Double() !

} // end C_AudioTextIO::SetParameterByID_Double()



/************************************************************************/
DWORD C_AudioTextIO::GetTotalCountOfSamples(void)
  /* Returns the total number of sampling points in the file .
   *  CAUTION: THE RESULT MAY BE NEGATIVE IF WE DON'T "KNOW" THIS YET .
   *   Reason: The number of samples in the file is usually not saved
   *           in the header .
   */
{
  return m_dwFileNrOfSamples;
} // C_AudioTextIO::GetTotalCountOfSamples()

/***************************************************************************/
DWORD C_AudioTextIO::GetCurrentSampleIndex(void)
  /* Reads the current sample index that will be used on the next
   * READ- or WRITE- access to the audio samples in the opened file.
   * Returns a negative value on any error.
   */
{
 return m_dwCurrSampleIndex;
} // C_AudioTextIO::GetCurrentSampleIndex()


/***************************************************************************/
BOOL C_AudioTextIO::WindToSampleIndex(DWORD dwNewSampleIndex)
  /* Sets the current sample index that will be used on the next
   * READ- or WRITE- access to the audio samples an opened file.
   * Returns TRUE if ok ,  otherwise FALSE.
   * As long as you don't use this routine/method,
   * the audio file will be played in a strictly "sequential" way.
   *
   *  NOTE: NOT FULLY IMPLEMENTED ON TEXT FILES YET,
   *        BECAUSE THIS MAY BE UTTERLY COMPLICATED  !
   *        (  and wasn't worth the effort yet  )
   *
   */
{
  char  sz1023Temp[1024];
  char  c;
  int   nChars;

  if( (m_QFile.iHandle>=0) && (m_dwFilePos_Data>=0) )
   {
    if( QFile_Seek(  &m_QFile, m_dwFilePos_Data, SEEK_SET ) >= 0)
     {
      m_dwCurrSampleIndex = 0;
      while( m_dwCurrSampleIndex < dwNewSampleIndex)  // only here if NOT winding to the "start" !
       {
        nChars = QFile_ReadLine( &m_QFile,  sz1023Temp, 1023 );
        if(nChars<0) // END-OF-FILE ?
          return FALSE;
        c = sz1023Temp[0];
        if( (c>='0' && c<='9') || c=='-')
          ++m_dwCurrSampleIndex;
       }
      return TRUE;
     }
   }

 return FALSE;
} // C_AudioTextIO::WindToSampleIndex()


/***************************************************************************/
LONG C_AudioTextIO::ReadSampleBlocks(
                 int channel_nr,    // channel_nr for 1st destination block
                 int nr_samples,    // nr_samples
                 T_Float *pFltDest1, // 1st destination block (~"LEFT")
                 T_Float *pFltDest2) // 2nd destination block (~"RIGHT")
  /* Reads some samples from a wave file which has been opened for READING
   * and converts them to T_Float-precision floats in the range of +- 1.0
   * Input parameters:
   *    channel_nr: 0=first channel (LEFT if stereo recording)
   *    nr_samples: Max number of samples which can be stored in T_Float dest[0..nr_samples-1]
   *    *pFltDestX = Pointers to destination arrays (one block per channel, not "paired")
   * Returns the NUMBER OF AUDIO SAMPLES if successful,
   *         or a negative value if errors occurred.
   * Note: Returning less samples than requested is not an error
   *       but indicates reaching the file's end !
   */
{
  int  nChars;
  int  n_samples_read;
  char  sz1023Temp[1024];
  BYTE  *bp;

   if ( (!m_OpenedForReading) || (m_QFile.iHandle<0) || (nr_samples<1) )
     return -1;

   n_samples_read=0;
   while(n_samples_read<nr_samples)
    {
     nChars = QFile_ReadLine( &m_QFile,  sz1023Temp, 1023 );
     if(nChars<0) // END-OF-FILE ?
       return n_samples_read;
     sz1023Temp[nChars] = '\0';
     bp = (BYTE*)sz1023Temp;
     if( (*bp>='0' && *bp<='9') || *bp=='-')
      { if(pFltDest1)  *pFltDest1++ = UTL_ParseInteger( &bp, 8/*ndigits*/ );
        if(pFltDest2)  *pFltDest2++ = UTL_ParseInteger( &bp, 8/*ndigits*/ );
        ++n_samples_read;
      }
     else // everything else will be SKIPPED
      {   // (it may be a HEADER LINE in between the data lines)

      }
    }

   return n_samples_read;

} // C_AudioTextIO::ReadSampleBlocks(...)


/***************************************************************************/
BOOL C_AudioTextIO::WriteHeader(
           T_ChunkInfo *pChunkInfo) // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
  /* Writes the 'header' for the audio file (usually a RIFF WAVE file).
   *  This was once done immediately in OutOpen(), but since 2011-02
   *  writing the header is postponed until the first chunk of samples
   *  is available, and a PRECISE(!) timestamp is present in pChunkInfo .
   *  All relevant information has already been set in OutOpen(),
   *  but not written to the file yet (by the time we get here).
   */
{
   return TRUE;
} // end C_AudioTextIO::WriteHeader()


/***************************************************************************/
BOOL C_AudioTextIO::OutOpen( char *file_name,
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int iATIOFileFormat,
           int bits_per_sample,   // 8,16,24=integer, 32 or 64=floating point ! (ignored for ASCII files)
           int channels,
           T_Float dbl_sample_rate)
  /* Creates and Opens a text file for WRITING audio samples.
   *         Returns TRUE on success and FALSE on any error.
   * Input parameters..
   *   additional_info:  a null-terminated text string with additional info
   *                     which will be included in the exported file.
   *                     NULL if this feature is not used.
   *                     SpecLab uses this to save date, time, precise
   *                     sample rate, and some other receiver settings
   *                     when the recording started.
   *   bits_per_sample:  8,16,24=integer, 32 or 64=floating point !
   *   channels:         1=mono, 2=stereo recording etc
   *   sample_rate:      number of samples per second.
   */
{
 char sz255Temp[256];

 BOOL ok = TRUE;

  CloseFile();   // Close "old" file if it was open

  // round the sample rate to a 32-bit INTEGER value (for the RIFF header)
  m_dbl_SampleRate = dbl_sample_rate;
  m_iChannelsPerSample = channels;

  // coarse check of the arguments..
  if( (bits_per_sample < 8) || (bits_per_sample > 64) )
      return FALSE;
  if( (channels<1) || (channels>2) )
      return FALSE;    // only mono or stereo recording supported
                       // ..sorry for those 'quattrophiles' ;-)
  if( (dbl_sample_rate < 1e-3) || (dbl_sample_rate > 1e6) )
      return FALSE;

  m_dwCurrFilePos = m_dwFilePos_Data = 0;  // default file position if no header
  m_dwFileDataSize= 0;  // "netto" size of audio samples still zero
  m_cSeparatorChar = ' ';
  m_iATIOFileFormat = iATIOFileFormat;

  // Try to create new file or truncate existing file to length "zero":
  if( QFile_Create( &m_QFile, file_name, 0/*attrib*/ ) )
   {
     // Write header (similar to CoolEdit) ?
     switch( m_iATIOFileFormat )  // ATIO_FILE_FORMAT_xxxx
      {
        case ATIO_FILE_FORMAT_COOL_EDIT:
             // sprintf(sz255Temp,"SAMPLES:\t%d",(int)9999);
             sprintf(sz255Temp,"BITSPERSAMPLE:\t%d\r\n",(int)16);
             QFile_WriteString( &m_QFile, sz255Temp );
             sprintf(sz255Temp,"CHANNELS:\t%d\r\n",(int)channels);
             QFile_WriteString( &m_QFile, sz255Temp );
             sprintf(sz255Temp,"SAMPLERATE:\t%d\r\n",(int)(dbl_sample_rate+0.5) );
             QFile_WriteString( &m_QFile, sz255Temp );
             sprintf(sz255Temp,"NORMALIZED:\tFALSE\r\n");
             QFile_WriteString( &m_QFile, sz255Temp );
             m_cSeparatorChar = '\t';  // TAB = chr(0x09) used by CoolEdit
             break;
        default:
             break;
      } // end switch( iHeaderOptions )

     ok = TRUE;
   } // end if <file opened successfully>
  else  // couldn't create the text file
   { ok = FALSE;
   }


  if (!ok)
   {
    CloseFile();  // close output file, could not generate a header.
    return FALSE;
   }
  else
   {
    m_OpenedForWriting = ok;
    return TRUE;
   }
} // C_AudioTextIO::OutOpen(..)


/***************************************************************************/
LONG C_AudioTextIO::WriteSamples( SHORT* pi16Data, LONG iLength )
  /* Writes some samples to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLES written,
   *  or a NEGATIVE value on any error.
   * Note: "Length" is the size of the caller's buffer capacity in 16-bit INTEGERS(!) .
   *       The actual number of AUDIO SAMPLES will depend on the number
   *       of channels (!) .
   *       For STEREO recordings, the number of sample points is only
   *       half the "Length", etc.
   */
{
  int iChannel;
  int n_written;
  char sz255Temp[256];
  char *cp;

  /* Ensure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!m_OpenedForWriting) || (m_QFile.iHandle<0) || (iLength <= 0) || (m_iChannelsPerSample<1) )
     return -1;
   n_written = 0;
   while(n_written < iLength)
    {
     cp = sz255Temp;
     for (iChannel=0; iChannel<m_iChannelsPerSample; ++iChannel)
      { sprintf(cp,"%d", *(pi16Data++) );
        ++n_written;
        cp+=strlen(cp);
        if( (iChannel+1)<m_iChannelsPerSample)
         { *cp++=m_cSeparatorChar;
         }
      }
     *cp++ = '\r'; *cp++='\n'; *cp='\0';
     if( ! QFile_WriteString( &m_QFile, sz255Temp ) )
      { // Writing to the output file failed  !!!
        // May be in trouble now, because the reason may be a full
        // harddisk and Windooze is going to crash in a few moments :-(
        CloseFile();
        return -1;  // also set a breakpoint here !
      }
    }
  return n_written;
} // end C_AudioTextIO::WriteSamples(..)

/***************************************************************************/
LONG C_AudioTextIO::WriteSamples_Float( // app-layer output, using NORMALIZED floats
       T_Float *pfltSrc,     // [in] caller's source buffer, 32-bit floats
       T_ChunkInfo *pChunkInfo, // [in] number of samples, channels, precise sample rate, 'radio frequency', date and time, etc
          // btw, T_ChunkInfo is defined in c:\cbproj\SoundUtl\ChunkInfo.h .
       char * pszExtraInfo ) // [in] extra 'info' (usually 'data') added to info chunk
  /* Writes some samples in the 'normalized' floating point format (-1.0 ... +1.0)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLE POINTS written,
   *       or a NEGATIVE value if there was an error.
   */
{
  int n_written;
  char sz255Temp[256];
  char *cp;

  int iChannel, nChannels, nSamplePoints, nSourceChannelsPerSample, iRest;

  if( pChunkInfo == NULL )
   { return -1;
   }
  nSamplePoints            = (int)pChunkInfo->dwNrOfSamplePoints;
  nSourceChannelsPerSample = pChunkInfo->nChannelsPerSample;
  nChannels = (nSourceChannelsPerSample < m_iChannelsPerSample)
             ? nSourceChannelsPerSample : m_iChannelsPerSample;
  iRest = m_iChannelsPerSample - nSourceChannelsPerSample; // number of unused 'dummy' channels
  if( iRest<0 )  iRest=0;



  /* Ensure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!m_OpenedForWriting) || (m_QFile.iHandle<0)
         ||(nSamplePoints<=0) || (nSourceChannelsPerSample<1) )
    { return -1;
    }
   n_written = 0;
   while(n_written < nSamplePoints)
    {
     cp = sz255Temp;
     for (iChannel=0; iChannel<nChannels; ++iChannel)
      { sprintf(cp,"%d", (int)(32767.0 * (*pfltSrc++) ) );
        cp+=strlen(cp);
        if( (iChannel+1)<m_iChannelsPerSample)
         { *cp++=m_cSeparatorChar;
         }
      }
     for (iChannel=0; iChannel<iRest; ++iChannel)
      { sprintf(cp,"0" );          // "filler" for currently unavailable channels
        cp+=strlen(cp);
        if( (iChannel+1)<m_iChannelsPerSample)
         { *cp++=m_cSeparatorChar;
         }
      }
     *cp++ = '\r'; *cp++='\n'; *cp='\0';
     if( ! QFile_WriteString( &m_QFile, sz255Temp ) )
      { // Writing to the output file failed  !!!
        // May be in trouble now, because the reason may be a full
        // harddisk and Windooze is going to crash in a few moments :-(
        CloseFile();
        return -1;  // also set a breakpoint here !
      }
     ++n_written;
    }
  return n_written;

} // end C_AudioTextIO::WriteSamples_Float(...)


/************************************************************************/
  // Some 'Get' - and 'Set' - functions for the AUDIO FILE class ..
double C_AudioTextIO::GetSampleRate(void)
{ return m_dbl_SampleRate;             }
void   C_AudioTextIO::SetSampleRate(double dblSampleRate)
{ m_dbl_SampleRate = dblSampleRate;    }
int    C_AudioTextIO::GetNrChannels(void)
{ return m_iChannelsPerSample;         }
int    C_AudioTextIO::GetBitsPerSample(void)
{ return m_iBitsPerSample;           
}

//---------------------------------------------------------------------------
double C_AudioTextIO::GetFrequencyOffset(void)  // "radio-" minus "baseband"-frequency
{ // Details in C_WaveIO::GetFrequencyOffset() !
  return m_dblFrequencyOffset;
} // end C_AudioTextIO::GetFrequencyOffset()

BOOL   C_AudioTextIO::SetFrequencyOffset(double dblFrequencyOffset)
{ // Details in C_WaveIO::GetFrequencyOffset() !
  m_dblFrequencyOffset = dblFrequencyOffset;
  return TRUE;
} // end C_AudioTextIO::SetFrequencyOffset()


double C_AudioTextIO::GetCurrentRecordingTime(void) // see specification in c:\cbproj\SoundUtl\AudioFileIO.h (t=0 is the RECORDING START TIME, aka "track time")
{ double d;   // no "T_Float" here, we need 64 bit floats to code the time !
  d = (double)GetCurrentSampleIndex();
  if(m_dbl_SampleRate>0)
     d /= m_dbl_SampleRate;
  return d;    // **NOT** "+ Start Time" ! ! !
}
double C_AudioTextIO::GetRecordingStartTime(void)
{ return m_dblStartTime;               }
void   C_AudioTextIO::SetRecordingStartTime(double dblStartTime)
{ m_dblStartTime = dblStartTime;       }


void C_AudioTextIO::GetFileName(char *pszDest, int iMaxLen)
{ strncpy( pszDest, m_QFile.sz512PathAndName, iMaxLen );
}


bool   C_AudioTextIO::AllParamsIncluded(void)
{ return FALSE;                        }



/***************************************************************************/
BOOL C_AudioTextIO::CloseFile( void )
  /* Closes the WAV-file (if opened for READING exor WRITING.
   * Returns TRUE when a file HAS BEEN CLOSED, of FALSE on error or 'nothing to close'.
   */
{
 BOOL closed_something=FALSE;

  if( m_QFile.iHandle >= 0)
   {
     QFile_Close( &m_QFile );
     closed_something = TRUE;
   }

  m_OpenedForReading = FALSE;
  m_OpenedForWriting = FALSE;

  return closed_something;
} // C_AudioTextIO::CloseFile(..)



/* EOF <AudioTextIO.cpp> */

