/***************************************************************************/
/* AudioSnd.cpp =  Audio Message Sender .                                  */
/*  Sends audio samples to other applications through WM_COPYDATA messages */
/*   - by DL4YHF,  June 2004                                               */
/*   - no VCL, no special runtime libs, only Win32 API used.               */
/*                                                                         */
/* Used by: "SndInput", "SerInput", "SpecLab" .                            */
/*                                                                         */
/***************************************************************************/

#include <windows.h>

#pragma hdrstop

#include "ErrorCodes.h"
#include "AudioSnd.h"

//***************************************************************************
//  Implementation of methods for the CAudioMessageSender class
//***************************************************************************


//***************************************************************************
CAudioMessageSender::CAudioMessageSender()   // constructor
{
   m_hOutputFile = INVALID_HANDLE_VALUE;
} // end CAudioMessageSender::CAudioMessageSender()    [constructor]

//***************************************************************************
CAudioMessageSender::~CAudioMessageSender()   // destructor
{
  CAudioMessageSender::Disconnect();  // ... if necessary (!)
} // end CAudioMessageSender::~CAudioMessageSender()    [destructor]


//***************************************************************************
BOOL CAudioMessageSender::ConnectRcvr(
                  HWND hRcvrWindow )    // receiver's 'window handle'
   // connect to an "audio receiver"
{
  CAudioMessageSender::Disconnect(hRcvrWindow);  // ... if necessary (!)

  return TRUE;
} // end CAudioMessageSender::ConnectRcvr()


//***************************************************************************
BOOL CAudioMessageSender::DisconnectRcvr(
                  HWND hRcvrWindow )    // receiver's 'window handle'
   // disconnect from "audio receiver"
{
  return TRUE;
} // end CAudioMessageSender::DisconnectRcvr()



//***************************************************************************
BOOL CAudioMessageSender::SendAudio(
                 T_Float *pfltSource, // source data, either single or double precision
                    long lNrSamples)  // count of '64-bit-float' values (!)
  // Note: the audio writer does not care for the buffer structure.
  //       If you want to write 4 pairs of I/Q samples,
  //       lNrSamples must be 4 * 2 = 8 'values' .
{
 double d;
 BOOL   fResult;
 long   l;
 DWORD  dwBytesWritten;

  if( (m_hOutputFile==INVALID_HANDLE_VALUE) || (lNrSamples<=0) )
      return FALSE;  // oh, shut up

  l = lNrSamples;
  switch(m_iExportDataType)    // what data type for this receiver ?
   {
    case DATA_TYPE_INT8     :  // write short 8 bit integer values ("short")
       { BYTE *pi8Array = (BYTE*)malloc(lNrSamples); // allocate temporary array
         BYTE *pi8 = pi8Array;
         if(!pi8Array) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try
          {
           while(l--)          // Note: the nominal input range is +-32767 (!)
            { d = ( (*pfltSource++) * (1.0 / 256.0)) + 128.0;
              if(d<0)   d= 0;
              if(d>255) d= 255;
              *pi8++ = (BYTE)d; // Note: the nominal output range is 0..255(!)
            }

           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                   (LPCVOID)pi8Array, // pointer to data to write to file
                          lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           free(pi8Array);     // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           free(pi8Array);     // free temporary array
           return FALSE;
          } }
    case DATA_TYPE_INT16     :  // write short 16 bit integer values ("short")
       { SHORT *pi16Array = (SHORT *)malloc(lNrSamples*sizeof(SHORT));
         SHORT *pi16 = pi16Array;
         if(!pi16Array) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try{
           while(l--)
            { d=*pfltSource++;
              if(d<-32768) d=-32768;
              if(d> 32767) d= 32767;
              *pi16++ = (SHORT)d;
            }
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                  (LPCVOID)pi16Array, // pointer to data to write to file
            sizeof(SHORT)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           free(pi16Array);  // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           free(pi16Array);  // free temporary array
           return FALSE;
          } }
    case DATA_TYPE_FLOAT32 :  // write single precision floating point ("float",  32 bit)
        {
#if SWI_FLOAT_PRECISION==1
         // no need to convert here, the source block is already "single precision float" format..
         try{
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                 (LPCVOID)pfltSource,   // pointer to data to write to file
           sizeof(float)*lNrSamples,  // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           return FALSE;
          }
#else /* .. SWI_FLOAT_PRECISION ..*/
         // internal processing single precision, but double precision for the outside world:
         double *pdblArray = new double[lNrSamples]; // allocate temporary array
         double *pdbl = pdblArray;
         if(!pdblArray) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try{
           while(l--)
            { *pdbl++ = (float)*pfltSource++;
            }
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                  (LPCVOID)pdblArray, // pointer to data to write to file
            sizeof(double)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           free(pdblArray);    // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           free(pdblArray);    // free temporary array
           return FALSE;
          }
#endif /* .. SWI_FLOAT_PRECISION ..*/
        }
    case DATA_TYPE_FLOAT64 :  // write double precision floating point ("double", 64 bit)
        {
#if SWI_FLOAT_PRECISION==2
         // no need to convert here, the source block is already "double" format..
         try{
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                 (LPCVOID)pfltSource,   // pointer to data to write to file
           sizeof(double)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           return FALSE;
          }
#else /* .. SWI_FLOAT_PRECISION ..*/
         // internal processing double precision, but single precision for the outside world:
         float *pfltArray = new float[lNrSamples]; // allocate temporary array
         float *pflt = pfltArray;
         if(!pfltArray) { m_dwErrorCode=MEMORY_ERROR; return FALSE; }
         try{
           while(l--)
            { *pflt++ = (float)*pfltSource++;
            }
           fResult = WriteFile(
                         m_hOutputFile, // handle to file to write to
                  (LPCVOID)pfltArray, // pointer to data to write to file
            sizeof(float)*lNrSamples, // number of bytes to write
                     &dwBytesWritten, // pointer to number of bytes written
                              NULL ); // pointer to structure for overlapped I/O
           if(fResult)
              m_lCountBytesWritten += dwBytesWritten;
           free(pfltArray);    // free temporary array
           return fResult;
          }catch(...){
           m_dwErrorCode = GetLastError();
           free(pfltArray);    // free temporary array
           return FALSE;
          }
#endif /* .. SWI_FLOAT_PRECISION */
        }
    default:  // unknown data type
         return FALSE;
   } // end switch(iExportDataType)

} // end CAudioMessageSender::AppendToFile()

//***************************************************************************
BOOL CAudioMessageSender::Disconnect(void)
{
  if(m_hOutputFile != INVALID_HANDLE_VALUE)
   {
    try
     {
       CloseHandle( m_hOutputFile );
       m_hOutputFile = INVALID_HANDLE_VALUE;
       return TRUE;
     }
    catch(...)
     {
       m_dwErrorCode = GetLastError();
       m_hOutputFile = INVALID_HANDLE_VALUE;
       return FALSE;
     }
   }
  else  // there was no valid opened file..
   {   return FALSE;
   }
} // end CAudioMessageSender::Disconnect()


