/***************************************************************************/
/* AudioMsgHdlr.cpp =  Audio Message Handler .                             */
/*  Sends and receives audio samples to/from other applications            */
/*                     through WM_COPYDATA messages                        */
/*   - by DL4YHF,  June 2004                                               */
/*   - no VCL, no special runtime libs, only Win32 API used.               */
/*                                                                         */
/* Status:  **UNDER CONSTRUCTION**                                         */
/* Used by: "SndInput", "SerInput", "SpecLab" .                            */
/*                                                                         */
/***************************************************************************/

#include <windows.h>

#pragma hdrstop

#include "ErrorCodes.h"
#include "AudioMsgHdlr.h"


// Macro to turn 4 characters into a DOUBLEWORD (Intel byte order, LSB first) .
// Used for DL4YHF's WM_COPYDATA-handler:
//    ch0..ch2 describe the TYPE of a WM_COPYDATA message (here: "aud" = audio),
//    ch3 is the "struct version" whereever applicable .
#ifndef  FOUR_CHARS_TO_DW
 #define FOUR_CHARS_TO_DW(ch0,ch1,ch2,ch3) ((DWORD)(ch0)|((DWORD)(ch1)<<8)|((DWORD)(ch2)<<16)|((DWORD)(ch3)<<24))
#endif

//***************************************************************************
//  Outside the   CAudioMessageHandler class  ..    helper routines
//***************************************************************************

//***************************************************************************
void IncreaseRxBufferSize( T_AudioMsg_Buffer *pABuffer, DWORD dwMinRequiredBufferSize )
   // Increases the allocated (RX-) buffer size ..
   //   - WITHOUT THROWING AWAY THE SAMPLES CURRENTLY IN THERE
   //   - but keeping the max buffer size in bounds (AUDIO_HDLR_MAX_BUFFER_SIZE)
{
 T_AudioMsg_Buffer NewBuffer;
 DWORD dwSourceIndex;

  if( dwMinRequiredBufferSize > AUDIO_HDLR_MAX_BUFFER_SIZE) // keep memory consumption limited !
      dwMinRequiredBufferSize = AUDIO_HDLR_MAX_BUFFER_SIZE;

  if( dwMinRequiredBufferSize <= pABuffer->dwAllocatedBufferSize )
     return;   // no need to change anything, the buffer already has this size

  // Initialize and allocate the new buffer:
  memset( &NewBuffer, 0, sizeof( T_AudioMsg_Buffer ) );
  NewBuffer.dwAllocatedBufferSize = dwMinRequiredBufferSize;
  NewBuffer.pFltBuffer = (T_Float*)malloc( sizeof(T_Float) * NewBuffer.dwAllocatedBufferSize);
  if( NewBuffer.pFltBuffer == NULL )
     return;  // bad luck, couldn't allocate a new buffer, good luck with the old one !

  // Arrived here: A new buffer with a larger size has been allocated.
  // If there is something in the "old" buffer, copy it into the new,
  //  and fill the rest with ZEROES (as valid floating point numbers).
  if( // is there something in the "old" buffer ?
        (pABuffer->dwAllocatedBufferSize > 0)
     && (pABuffer->pFltBuffer != NULL)
     && (pABuffer->dwBufIndexIn != pABuffer->dwBufIndexOut)
    )
   { dwSourceIndex = pABuffer->dwBufIndexOut;
     while( dwSourceIndex != pABuffer->dwBufIndexOut )
      { if( dwSourceIndex >= pABuffer->dwAllocatedBufferSize)
            dwSourceIndex = 0;
        if( NewBuffer.dwBufIndexIn >= NewBuffer.dwAllocatedBufferSize)
            NewBuffer.dwBufIndexIn = 0;  // should never happen but who knows .. safety first !

        NewBuffer.pFltBuffer[NewBuffer.dwBufIndexIn++] = pABuffer->pFltBuffer[dwSourceIndex++];
      }
     // Fill the rest of the "new" buffer with ZEROES to avoid illegal float formats in it :
     while( NewBuffer.dwBufIndexIn < NewBuffer.dwAllocatedBufferSize )
      {
        NewBuffer.pFltBuffer[NewBuffer.dwBufIndexIn++] = 0.0;
      }
   } // end if < is there something in the "old" buffer ? >

  // Now free the "old" buffer, and replace it with the NEW buffer (!)
  if(pABuffer->pFltBuffer != NULL)
     free( pABuffer->pFltBuffer );
  *pABuffer = NewBuffer;  // BLOCK-COPY of a normal "C"-struct !
    // this only copies a few bytes, including the POINTER to the buffer,
    // but not the dynamically allocated memory block itself !
} // end CAudioMessageHandler::IncreaseRxBufferSize()



//***************************************************************************
//  Implementation of methods for the CAudioMessageHandler class
//***************************************************************************



//***************************************************************************
int CAudioMessageHandler::GetRcvrIndex( HWND hwndRcvr )
{
  for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
    if( m_Receiver[i].hwnd == hwndRcvr )
      return i;
  return -1;
} // end CAudioMessageHandler::GetRcvrIndex()

//***************************************************************************
int CAudioMessageHandler::GetSenderIndex( HWND hwndSender )
{
  for(int i=0; i<AUDIO_HDLR_MAX_SENDERS; ++i)
    if( m_Sender[i].hwnd == hwndSender )
      return i;
  return -1;
} // end CAudioMessageHandler::GetSenderIndex()



//***************************************************************************
BOOL CAudioMessageHandler::ConnectDestination(
        HWND hwndRcvr    ,    // connect to a "receiver" ( WINDOW HANDLE ! )
        int  iRcvdDataType)   // type of data I want to send
  // connect to an "audio receiver"
{
  int i;

  if( (hwndRcvr == NULL) || (hwndRcvr == HWND_BROADCAST) )
   { // cannot or MUST NOT connect to this receiver :
     return FALSE;
   }

  i = GetRcvrIndex( hwndRcvr );
  if(i >= 0)
   { Disconnect(hwndRcvr);  // ... if necessary (!)
     m_Receiver[i].hwnd = NULL;
   }

  i = GetRcvrIndex( NULL );     // find an empty table entry
  if(i >= 0)
   { // found an unused entry, occupy it :
     m_Receiver[i].hwnd = hwndRcvr;
     m_Receiver[i].iDataType = iRcvdDataType;
   }

  return TRUE;
} // end CAudioMessageHandler::ConnectDestination()


//***************************************************************************
BOOL CAudioMessageHandler::ConnectSource(
        HWND hwndSender  ,    // connect to an "audio source" ( WINDOW HANDLE ! )
        int  iSentDataType)   // type of data I want to receive later
  // Connects to an "audio receiver" by means of WoBu's WM_COPYDATA-based
  // local audio server.
{
  int i;

  if( (hwndSender == NULL) || (hwndSender == HWND_BROADCAST) )
   { // cannot or MUST NOT connect this sender (illegal handle)
     return FALSE;
   }

  i = GetSenderIndex( hwndSender );
  if(i >= 0)
   { Disconnect(hwndSender);  // ... if necessary (!)
     m_Sender[i].hwnd = NULL;
   }

  i = GetSenderIndex( NULL ); // find an empty table entry
  if(i >= 0)
   { // found an unused entry, occupy it :
     m_Sender[i].hwnd = hwndSender;
     m_Sender[i].iDataType = iSentDataType;
   }

  return TRUE;
} // end CAudioMessageHandler::ConnectSource()


//***************************************************************************
LONG CAudioMessageHandler::HandleWmCopydata(
                 HWND hwndDestWnd,       // WINDOW HANDLE of destination (should be "mine")
                 HWND hwndSendingWnd,    // WINDOW HANDLE of the sender of this message
                 COPYDATASTRUCT *pCds )  // pointer to COPYDATASTRUCT, passed in message
{
 int i;
  //
  //  About the members of the COPYDATASTRUCT:
  //
  //  dwData
  //    Specifies up to 32 bits of data to be passed to the receiving application.
  //    Here, dwData is used to distinguish AUDIO DATA BLOCKS from COMMANDS .
  //
  //  cbData
  //    Specifies the size, in bytes, of the data pointed to
  //    by the lpData member.
  //
  //  lpData
  //    Points to data to be passed to the receiving application.
  //    This member can be NULL.
  if(pCds->dwData == FOUR_CHARS_TO_DW('a','u','d','c') )
   { // AUDio Command (to audio message handler)
     return 1L;  // return value = 1   means "I did handle this message"
   } // end if (dwData = "audc")
  else if(pCds->dwData == FOUR_CHARS_TO_DW('a','u','d','d') )
   { // AUDio DATA (lpData is a block of AUDIO SAMPLES)
     // Check if they have been sent from one of the registered SENDERS,
     // if so, COPY THEM INTO A BUFFER AND RETURN A.S.A.P,
     // otherwise ignore the message .
     // Note that the data MUST be copied into an own circular buffer here,
     // because the data in the WM_COPYDATA message must be treated as
     // "read-only" .
     i = GetSenderIndex( hwndSendingWnd );
     if(i >= 0)  // accept samples from this sender ...
      {
        m_Sender[i].hwnd = NULL;
        return 1L;  // return value = 1   means "I did handle this message"
      }
     return 0L;  // return value = 0   means "I didn't handle this message"
   } // end if (dwData = "audd")
  else // don't know what this is, ignore it silently ..
   {
     return 0L;  // return value = 0   means "I didn't handle this message"
   }
} // end CAudioMessageHandler::HandleWmCopydata()


//***************************************************************************
CAudioMessageHandler::CAudioMessageHandler()   // constructor
{
  int i;
   for(i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
    { memset( &m_Receiver[i], 0, sizeof(T_AudioMsgClient) );
      m_Receiver[i].hwnd = NULL;
    }
   for(i=0; i<AUDIO_HDLR_MAX_SENDERS; ++i)
    { memset( &m_Sender[i], 0, sizeof(T_AudioMsgClient) );
      m_Sender[i].hwnd = NULL;
      memset( &m_RxBuffer[i], 0, sizeof(T_AudioMsg_Buffer) );
      m_RxBuffer[i].pFltBuffer = NULL;
    }
   m_hwndMyWindow = NULL;      // my own window handle is still unknown

   m_dwTotalSamplesSent = 0;
} // end CAudioMessageHandler::CAudioMessageHandler()    [constructor]

//***************************************************************************
CAudioMessageHandler::~CAudioMessageHandler()   // destructor
{
  int i;

   // Cleanup dynamically allocated stuff. From header:
   //  T_AudioMsgClient  m_Receiver[AUDIO_HDLR_MAX_RECEIVERS];
   //  T_AudioMsgClient  m_Sender  [AUDIO_HDLR_MAX_SENDERS];
   //  T_AudioMsg_Buffer m_RxBuffer[AUDIO_HDLR_MAX_SENDERS];

   for(i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
    if(  m_Receiver[i].hwnd != NULL )
     {
       Disconnect( m_Receiver[i].hwnd );
     }

   for(i=0; i<AUDIO_HDLR_MAX_SENDERS; ++i)
    { if( m_RxBuffer[i].pFltBuffer!=NULL)
       { free( m_RxBuffer[i].pFltBuffer );
         m_RxBuffer[i].pFltBuffer = NULL;
       }
    }


} // end CAudioMessageHandler::~CAudioMessageHandler()    [destructor]



//***************************************************************************
void CAudioMessageHandler::SetWindowHandle( HWND hwndMyWindow )
{
  m_hwndMyWindow = hwndMyWindow;
} // end CAudioMessageHandler::Init()



//***************************************************************************
BOOL CAudioMessageHandler::Disconnect(
                  HWND hRcvrWindow )    // receiver's 'window handle'
   // disconnect from "audio receiver" or "audio sender"
{
  return TRUE;
} // end CAudioMessageHandler::DisconnectRcvr()


//***************************************************************************
BOOL CAudioMessageHandler::SendAudio( // Sends an audio block to all connected recipients (in different formats)
                 T_Float *pfltSource, // source data, either single or double precision
                long i32NrSamplePoints,  // count of SAMPLE POINTS, not "single float values" (!)
                int  iChannelsPerSample) // info about the SOURCE: 1=mono, 2=stereo
  // Note: The audio message handler cares for the buffer structure.
  //       Because there may be more than one 'receiver',
  //         with different numbers of channels for each,
  //         the count of samples is SAMPLE POINTS here,
  //         a sample point may consist of more than one T_Float value !
  //   Example: If the source stream is "STEREO audio" (two channels),
  //       and lNrSamplePoints is 1024, there must be 2048 T_Float values
  //       at pfltSource !
{
 double d;
 BOOL   fResult = FALSE;
 long   l;
 long   i32NrSamplesMultChannels;
 //DWORD  dwBytesWritten;
 int    iAudioRcvr;
 COPYDATASTRUCT cds;
 LRESULT result;



  i32NrSamplesMultChannels = i32NrSamplePoints * iChannelsPerSample;
  if( i32NrSamplesMultChannels <= 0 )
      return FALSE;  // umm, why should I send "nothing" ?

  // Prepare sending some WM_COPYDATA messages to other applications:
  //   Fill out a COPYDATASTRUCT ...
  cds.dwData = FOUR_CHARS_TO_DW('a','u','d','0');  // "aud0" = magic number for "WM_COPYDATA containing AUDIO DATA"
  cds.cbData = 0;
  cds.lpData = (PVOID)NULL;


  // To avoid converting the data format more than once for the same type,
  // run through all possible receivers to find equal "destination types" :
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_INT8) )
     {
      cds.cbData = i32NrSamplesMultChannels * sizeof(BYTE);
      BYTE *pi8Array = (BYTE*)malloc(cds.cbData); // allocate temporary array
      BYTE *pi8 = pi8Array;
      if(pi8Array)
       {
        try
         {
          l = i32NrSamplesMultChannels;
          while(l--)          // Note: the nominal input range is +-32767 (!)
           { d = ( (*pfltSource++) * (1.0 / 256.0)) + 128.0;
             if(d<0)   d= 0;
             if(d>255) d= 255;
             *pi8++ = (BYTE)d; // Note: the nominal output range is 0..255(!)
           }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pi8Array;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_INT8) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pi8Array);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pi8Array);     // free temporary array
         }
       } // end if < array could be allocated >

       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_INT8 >
   } // end for(iAudioRcvr=0; ....

  // The same again for all who want to receive 16-bit integers ....
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_INT16) )
     {
      cds.cbData = i32NrSamplesMultChannels * sizeof(SHORT);
      SHORT *pi16Array = (SHORT *)malloc(cds.cbData);
      SHORT *pi16 = pi16Array;

      if(pi16Array)
       {
        try
         {
          l = i32NrSamplesMultChannels;
          while(l--)
            { d=*pfltSource++;
              if(d<-32768) d=-32768;
              if(d> 32767) d= 32767;
              *pi16++ = (SHORT)d;
            }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pi16Array;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_INT16) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pi16Array);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pi16Array);     // free temporary array
         }
       } // end if < array could be allocated >
       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_INT16 >
   } // end for(iAudioRcvr=0; ....


  // The same again for all who want to receive 32-bit FLOAT's ....
#if SWI_FLOAT_PRECISION==1
  // no need to convert here, the source block is already "single precision float" format..
  cds.cbData = i32NrSamplesMultChannels * sizeof(float);
  cds.lpData = (PVOID)pfltSource;
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL) && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT32) )
     {
      result=SendMessage( m_Receiver[iAudioRcvr].hwnd, // handle of destination window
                          WM_COPYDATA,        // message to send (type of)
                  (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                              (LPARAM)&cds);
      fResult |= (result==TRUE);
     } // end if < found another receiver for this data type >
   } // end for < all possible receivers >
#else /* .. SWI_FLOAT_PRECISION ..*/
  // internal processing double precision, but single precision for the outside world:
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT32) )
     {
      cds.cbData = lNrSamples * sizeof(float);
      float *pfltArray = (float*)malloc(cds.cbData);
      float *pflt = pfltArray;
      if(pfltArray)
       {
        try
         {
          l = lNrSamples;
          while(l--)
            { *pflt++=*pfltSource++;
            }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pfltArray;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_FLOAT32) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);                                  
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pfltArray);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pfltArray);    // free temporary array
         }
       } // end if < array could be allocated >
       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_FLOAT32 >
   } // end for(iAudioRcvr=0; ....
#endif /* .. SWI_FLOAT_PRECISION ..*/


  // The same again for all who want to receive 64-bit FLOAT's ....
#if SWI_FLOAT_PRECISION==2
  // no need to convert here, the source block is already "double precision float" format..
  cds.cbData = lNrSamples * sizeof(double);
  cds.lpData = (PVOID)pfltSource;
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL) && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT64) )
     {
      result=SendMessage( m_Receiver[iAudioRcvr].hwnd, // handle of destination window
                          WM_COPYDATA,        // message to send (type of)
                  (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                              (LPARAM)&cds);
      fResult |= (result==TRUE);                              
     } // end if < found another receiver for this data type >
   } // end for < all possible receivers >
#else /* .. SWI_FLOAT_PRECISION ..*/
  // internal processing single precision, but double precision for the outside world:
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT64) )
     {
      cds.cbData = i32NrSamplesMultChannels * sizeof(double);
      double *pdblArray = (double*)malloc(cds.cbData);
      double *pdbl = pdblArray;
      if(pdblArray)
       {
        try
         {
          l = i32NrSamplesMultChannels;
          while(l--)
            { *pdbl++=*pfltSource++;
            }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pdblArray;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_FLOAT32) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pdblArray);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pdblArray);    // free temporary array
         }
       } // end if < array could be allocated >
       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_FLOAT32 >
   } // end for(iAudioRcvr=0; ....
#endif /* .. SWI_FLOAT_PRECISION ..*/

  if(fResult==TRUE)
     m_dwTotalSamplesSent += (DWORD)i32NrSamplePoints;

  return fResult;   // returns TRUE if data were sent to at least one receiver

} // end CAudioMessageHandler::SendAudio()



//***************************************************************************
BOOL CAudioMessageHandler::GetReceivedAudio(
        T_Float *pfltDest,       // pointer to destination block, usually single precision float
        long i32NrSamplePoints,  // count of SAMPLE POINTS the caller may read from the buffer
        int  iChannelsPerSample) // info about the caller's DESTINATION: 1=mono, 2=stereo
  // Gets some received audio samples from an internal buffer,
  //     which have previously been received through a WM_COPYDATA message
  //     (possibly in another thread, that's one reason to use a buffer).
  // Returns TRUE if the required count of samples could be delivered,
  //           FALSE otherwise .
  // Note: The audio message handler cares for the buffer structure.
  //       'i32NrSamplePoints' is the count of samples is SAMPLE POINTS,
  //       a sample point may consist of more than one T_Float value !
  //   Example: If the source stream is "STEREO audio" (two channels),
  //       and lNrSamplePoints is 1024, there must be 2048 T_Float values
  //       at pfltSource !
{
 T_Float fltTemp;
 T_Float *pfltSource = NULL;
 BOOL   fResult = FALSE;
 long   l;
 long   i32NrSamplesMultChannels;
 // DWORD  dwBytesWritten;
 int    iAudioRcvr;
 COPYDATASTRUCT cds;
 LRESULT result;



  i32NrSamplesMultChannels = i32NrSamplePoints * iChannelsPerSample;
  if( i32NrSamplesMultChannels <= 0 )
      return FALSE;  // umm, why should I send "nothing" ?

  // Prepare sending some WM_COPYDATA messages to other applications:
  //   Fill out a COPYDATASTRUCT ...
  cds.dwData = FOUR_CHARS_TO_DW('a','u','d','0');  // "aud0" = magic number for "WM_COPYDATA containing AUDIO DATA"
  cds.cbData = 0;
  cds.lpData = (PVOID)NULL;


  // To avoid converting the data format more than once for the same type,
  // run through all possible receivers to find equal "destination types" :
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (pfltSource != NULL)
      && (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_INT8) )
     {

      cds.cbData = i32NrSamplesMultChannels * sizeof(BYTE);
      BYTE *pi8Array = (BYTE*)malloc(cds.cbData); // allocate temporary array
      BYTE *pi8 = pi8Array;
      if(pi8Array)
       {
        try
         {
          l = i32NrSamplesMultChannels;
          while(l--)          // Note: the nominal input range is +-32767 (!)
           { fltTemp = ( (*pfltSource++) * (1.0 / 256.0)) + 128.0;
             if(fltTemp<0)   fltTemp= 0;
             if(fltTemp>255) fltTemp= 255;
             *pi8++ = (BYTE)fltTemp; // Note: the nominal output range is 0..255(!)
           }
          // Now, after converting the block into the required data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pi8Array;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_INT8) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pi8Array);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pi8Array);     // free temporary array
         }
       } // end if < array could be allocated >

       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_INT8 >
   } // end for(iAudioRcvr=0; ....

  // The same again for all who want to receive 16-bit integers ....
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (pfltSource != NULL)
      && (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_INT16) )
     {
      cds.cbData = i32NrSamplesMultChannels * sizeof(SHORT);
      SHORT *pi16Array = (SHORT *)malloc(cds.cbData);
      SHORT *pi16 = pi16Array;

      if(pi16Array)
       {
        try
         {
          l = i32NrSamplesMultChannels;
          while(l--)
            { fltTemp=*pfltSource++;
              if(fltTemp<-32768) fltTemp=-32768;
              if(fltTemp> 32767) fltTemp= 32767;
              *pi16++ = (SHORT)fltTemp;
            }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pi16Array;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_INT16) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pi16Array);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pi16Array);     // free temporary array
         }
       } // end if < array could be allocated >
       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_INT16 >
   } // end for(iAudioRcvr=0; ....


  // The same again for all who want to receive 32-bit FLOAT's ....
#if SWI_FLOAT_PRECISION==1
  // no need to convert here, the source block is already "single precision float" format..
  cds.cbData = i32NrSamplesMultChannels * sizeof(float);
  cds.lpData = (PVOID)pfltSource;
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL) && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT32) )
     {
      result=SendMessage( m_Receiver[iAudioRcvr].hwnd, // handle of destination window
                          WM_COPYDATA,        // message to send (type of)
                  (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                              (LPARAM)&cds);
      fResult |= (result==TRUE);
     } // end if < found another receiver for this data type >
   } // end for < all possible receivers >
#else /* .. SWI_FLOAT_PRECISION ..*/
  // internal processing double precision, but single precision for the outside world:
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (pfltSource != NULL)
      && (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT32) )
     {
      cds.cbData = lNrSamples * sizeof(float);
      float *pfltArray = (float*)malloc(cds.cbData);
      float *pflt = pfltArray;
      if(pfltArray)
       {
        try
         {
          l = lNrSamples;
          while(l--)
            { *pflt++=*pfltSource++;
            }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pfltArray;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_FLOAT32) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);                                  
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pfltArray);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pfltArray);    // free temporary array
         }
       } // end if < array could be allocated >
       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_FLOAT32 >
   } // end for(iAudioRcvr=0; ....
#endif /* .. SWI_FLOAT_PRECISION ..*/


  // The same again for all who want to receive 64-bit FLOAT's ....
#if SWI_FLOAT_PRECISION==2
  // no need to convert here, the source block is already "double precision float" format..
  cds.cbData = lNrSamples * sizeof(double);
  cds.lpData = (PVOID)pfltSource;
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL) && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT64) )
     {
      result=SendMessage( m_Receiver[iAudioRcvr].hwnd, // handle of destination window
                          WM_COPYDATA,        // message to send (type of)
                  (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                              (LPARAM)&cds);
      fResult |= (result==TRUE);
     } // end if < found another receiver for this data type >
   } // end for < all possible receivers >
#else /* .. SWI_FLOAT_PRECISION ..*/
  // internal processing single precision, but double precision for the outside world:
  for(iAudioRcvr=0; iAudioRcvr<AUDIO_HDLR_MAX_RECEIVERS; ++iAudioRcvr)
   {
    if(  (m_Receiver[iAudioRcvr].hwnd!=NULL)
      && (m_Receiver[iAudioRcvr].iDataType==DATA_TYPE_FLOAT64) )
     {
      cds.cbData = i32NrSamplesMultChannels * sizeof(double);
      double *pdblArray = (double*)malloc(cds.cbData);
      double *pdbl = pdblArray;
      if(pdblArray)
       {
        try
         {
          l = i32NrSamplesMultChannels;
          while(l--)
            { *pdbl++=*pfltSource++;
            }
          // Now, after converting into this data type ONCE,
          //  send the data to all receivers which require this format :
          cds.lpData = (PVOID)pdblArray;
          for(int i=0; i<AUDIO_HDLR_MAX_RECEIVERS; ++i)
           {
            if(  (m_Receiver[i].hwnd!=NULL) && (m_Receiver[i].iDataType==DATA_TYPE_FLOAT32) )
             {
              // An application must use the SendMessage function to send this message,
              // not the PostMessage function ...
              result=SendMessage( m_Receiver[i].hwnd, // handle of destination window
                                  WM_COPYDATA,        // message to send (type of)
                          (WPARAM)m_hwndMyWindow,     // wParam (1st message param)
                                  (LPARAM)&cds);
              fResult |= (result==TRUE);
             } // end if < found another receiver for this data type >
           } // end for < all possible receivers >
          free(pdblArray);     // free temporary array
         }catch(...){
           m_dwErrorCode = GetLastError();
           free(pdblArray);    // free temporary array
         }
       } // end if < array could be allocated >
       break;  // don't search any futher, all receivers for this type have been fed.
     } // end if < found a receiver for DATA_TYPE_FLOAT32 >
   } // end for(iAudioRcvr=0; ....
#endif /* .. SWI_FLOAT_PRECISION ..*/

  if(fResult==TRUE)
     m_dwTotalSamplesSent += (DWORD)i32NrSamplePoints;

  return fResult;   // returns TRUE if data were sent to at least one receiver

} // end CAudioMessageHandler::GetReceivedAudio()



//***************************************************************************
HWND CAudioMessageHandler::GetReceiverHandle( int iRcvrIndex )
{
   if( iRcvrIndex>=0 && iRcvrIndex<AUDIO_HDLR_MAX_RECEIVERS)
     return m_Receiver[iRcvrIndex].hwnd;
   else
     return NULL;
} // end CAudioMessageHandler::GetRcvrHandle()

//***************************************************************************
DWORD CAudioMessageHandler::GetNrOfSamplesSent(void)
{
  return m_dwTotalSamplesSent;
} // end CAudioMessageHandler::GetNrOfSamplesSent()


// EOF < AudioMsgHdlr >
