/*************************************************************************/
/*  Project : Audio Message Tester,  written in Dev-C++ .                */
/*  Purpose : Sends a sine wave to spectrum lab,                         */
/*            using WM_COPYDATA messages for the audio data .            */
/*  Requires: audiomsg.c ,  utility1.cpp (or utility2.cpp) ,  wsock.c  . */
/*  Compiler: Dev-C++ / MinGW / GNU, but any C-compiler capable of       */
/*            producing a windows executable should do . There is no     */
/*            Borland- or even M$-Visual-C-specific stuff in here .      */
/*  Project files:                                                       */
/*           AudioMsgTester_BCB6.prj (for Borland C++ V6; recommended);  */
/*           AudioMessageTest.dev (for Dev-C++ or wxDevC++);             */
/*************************************************************************/
#include <windows.h>
#include <stdio.h>     /* no standard-I/O, but sprintf used here           */
#include <math.h>
#include <wsock.h>     /* DL4YHF's wrapper for the windows socket services */
#include <utility1.h>  /* DL4YHF's "utility" with date- and time functions */

#include "audiomsg.h"  /* audio message buffer for sending data to SpecLab */

  // The Audio Message Test project contained the following modules (anno 2012):
  //  - ??\SoundUtl\AudioMessageTest_main.c
  //  - ??\SoundUtl\audiomsg.c
  //  - ??\swi_audio_message_test\switches.h
  //  - ??\SoundUtl\utility2.cpp
  //
  // Notes about compiling this project ("AudioMessageTest") :
  //  - Forget about Dev-C++ - it's very bugged, and the debugger is a nightmare
  //
  //  - If you have, use Borland C++Builder (V6) or later versions .
  //     For Borland C++Builder, use the project 'AudioMsg_Tester_BCB6.prj .
  //
  //  - If [wx]Dev-C++ complains about a gazillion of "undefined references",
  //         try adding the following library somewhere under
  //         'Project'..'Project Options'..'Parameters'(!!!!)...'Linker' :
  //            ../../../Dev-Cpp/lib/libgdi32.a
  //            ../../Dev-Cpp/lib/libwinmm.a
  //            ../../Dev-Cpp/lib/libwsock32.a
  //
  //  - If [wx]Dev-C++ complains
  //      [Linker Error] undefined reference to `_imp__UTL_NamedMalloc'
  //
  //  - If the program compiled with [wx]Dev-C++ opens up a stupid CONSOLE window
  //      (even though it should be a windows GUI application),
  //      get rid of [wx]Dev-C++ and get a professional development system.
  //


/*  Declare Windows procedure  */
LRESULT CALLBACK WindowProcedure (HWND, UINT, WPARAM, LPARAM);

UINT_PTR g_MyTimer;

BOOL g_fPaused = FALSE;
int  g_iDataType_Bits = 16; // data type for transmission, settable via commandline:
    // 8 = 8 bit unsigned, 16=16 bit signed integer, 32= 32-bit floating point .

int  g_iFixedNumChannels = -1; // fixed number of channels (from commandline),
                            // or -1 to poll this value from the remote end

int  g_iFixedSamplingRate = -1;  // samples per second (from commandline),
                            // or -1 to poll this value from the remote end




/*  Make the class name into a global variable  */
char szClassName[ ] = "AudioMessageTest";

/* Audio Message buffer structure (used to send audio to spectrum lab): */
T_AudioMsgBuffer * g_sAudioMsgBuffer;  /* defined in audiomsg.h */


/* Info-strings (displayed in the small control window) */
#define NUM_INFO_STRINGS 4
char g_sz80InfoStrings[NUM_INFO_STRINGS][84];


/* Windows main function (program entry point for the executable file) */
int WINAPI WinMain (HINSTANCE hThisInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpszArgument,
                    int nCmdShow ) // initial 'show' state of window

{
  HWND hwnd;               /* This is the handle for our window */
  MSG messages;            /* Here messages to the application are saved */
  WNDCLASSEX wincl;        /* Data structure for the windowclass */
  UINT result;
  SYSTEMTIME stm;
  char *cp;

  UTL_Init();   // initialize DL4YHF's "utility" (library with miscellaneous
                //  stuff, including portable date- and time conversions)
  memset( g_sz80InfoStrings, 0, sizeof(g_sz80InfoStrings) );

  // Parse the command line (still possible, even in a windoze GUI application):
  // Example:  AudioMessageTest.exe /dt=16 /ch=2 /sr=22050
  // ( use 16 bit signed integer;  2 channels per sample, 22050 samples/second )
  if( lpszArgument != NULL )
   { cp = lpszArgument;
     while( *cp != 0 )
      { while( *cp==' ' )
         { ++cp;
         }
        if(strnicmp( cp, "/dt=", 4) == 0 )
         { cp+=4;
           g_iDataType_Bits = strtod( cp, &cp );
         }
        else if(strnicmp( cp, "/ch=", 4) == 0 )
         { cp+=4;
           g_iFixedNumChannels = strtod( cp, &cp );
         }
        else if(strnicmp( cp, "/sr=", 4) == 0 )
         { cp+=4;
           g_iFixedSamplingRate = strtod( cp, &cp );
         }
        else // unrecognized command line argument -> skip it
         { while( *cp!=' ' && *cp!=0)
            { ++cp;
            }
         }
      }
   } // end if < parse command line > ?


  // Init the Window structure ("class", hasn't got anything to do with C++):
  wincl.hInstance = hThisInstance;
  wincl.lpszClassName = szClassName;
  wincl.lpfnWndProc = WindowProcedure;      /* This function is called by windows */
  wincl.style = CS_DBLCLKS;                 /* Catch double-clicks */
  wincl.cbSize = sizeof (WNDCLASSEX);
  // Use default icon and mouse-pointer
  wincl.hIcon = LoadIcon (NULL, IDI_APPLICATION);
  wincl.hIconSm = LoadIcon (NULL, IDI_APPLICATION);
  wincl.hCursor = LoadCursor (NULL, IDC_ARROW);
  wincl.lpszMenuName = NULL;                 /* No menu */
  wincl.cbClsExtra = 0;                      /* No extra bytes after the window class */
  wincl.cbWndExtra = 0;                      /* structure or the window instance */
  /* Use Windows's default color as the background of the window */
  wincl.hbrBackground = (HBRUSH) COLOR_BACKGROUND;

  // Register the window class, and if it fails quit the program
  if (!RegisterClassEx (&wincl))
      return 0;

  // now create main window
  hwnd = CreateWindow(
       szClassName,    // lpClassName,    pointer to registered class name
       "Audio Message Test", // "lpWindowName", pointer to window name, effectively the TITLE
       // DWORD dwStyle,        window style. Tried a lot ...
       WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_OVERLAPPED,  // normal window
       // WS_POPUP | WS_DLGFRAME ,   // window with no title, quite unusual as main window
       // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_SIZEBOX, // sizeable
       // WS_CAPTION,   // with title, but no icon and no system menu nor close symbol
       10, // int x
       10, // int y
       320,     // int nWidth , window width
       120,     // int nHeight, window height
       NULL,    // HWND hWndParent, handle to parent or owner window
       NULL,    // HMENU hMenu,     handle to menu or child-window identifier
       hThisInstance, // HANDLE hInstance, handle to application instance
       NULL);     // LPVOID lpParam,   pointer to window-creation data

  /* Make the window visible on the screen */
  ShowWindow (hwnd, nCmdShow );

  /* Initialize the audio message buffer and prepare a timer to produce data */
  //  Both the sampling rate and the number of channels may be changed later
  //  'on the fly', which is not a REQUIREMENT for the sender but may be nice to have.
  AudioMsgBuf_Init( &g_sAudioMsgBuffer, 22050.0/*dblSampleRate*/, 2/*channels*/ );

  /* Set the INITIAL timestamp of the sent data to the current date & time */
  GetSystemTime( &stm ); // -> date and time in UTC, windoze-specific format
      // Convert date & time from the ugly Windows format
      //  into the easily 'calculatable' (incrementable) Unix format,
      //  which is "number of seconds since 1970-01-01 00:00:00 UTC" :
  g_sAudioMsgBuffer->info.ldblUnixDateAndTime = UTL_ConvertYMDhmsToUnix(
         stm.wYear, stm.wMonth, stm.wDay,  stm.wHour, stm.wMinute,
         (double)stm.wSecond + (double)stm.wMilliseconds * 1e-3 );

  g_MyTimer = SetTimer( // creates a timer with the specified time-out value
           hwnd,    // handle of window for timer messages
           0,       // timer identifier
           50,      // UINT uElapse = time-out value, in milliseconds
           NULL );  // lpTimerFunc  = address of timer procedure
  // Note: if your win32 programmer's reference says SetTimer returns a UINT,
  //       not a UINT_PTR, the damned thing is too old .
  if( g_MyTimer==0 ) // if "SetTimer" fails, it returns zero, or what ?
   { g_MyTimer = g_MyTimer;
   }

  /* Run the message loop. It will run until GetMessage() returns 0 */
  while (GetMessage (&messages, NULL, 0, 0))
   {
        /* Translate virtual-key messages into character messages */
        TranslateMessage(&messages);
        /* Send message to WindowProcedure */
        DispatchMessage(&messages);
   }

  /* The program return-value is 0 - The value that PostQuitMessage() gave */
  return messages.wParam;
}

void OnWmTimer(HWND hWnd, WORD wTimerID ) // called on WM_TIMER
{
  int  iSample, nSamples;
  long i, iSamplingRate, iNumChannels;
  long free_buffer_space_at_receiver;
  DWORD dwBufIdx;
  HWND hwndSpecLab;
  HDC  hdc;
  float signal_frequency;
  T_AudioMsgBuffer *pBuf = g_sAudioMsgBuffer; // audio message buffer for output
  char *cp;
  static long  i32MessagesSent = 0;
  static double sdblFreqModPhaseAccu = 0.0;
  static double sdblPhaseAccu = 0.0;
  static BOOL fAlreadyHere = FALSE;

  if ( fAlreadyHere )
    return;

  if ( g_fPaused )
    return;

  fAlreadyHere = TRUE;
  g_sz80InfoStrings[0][0] = '\0';
  g_sz80InfoStrings[1][0] = '\0';

  // Try to find the window handle of SpectrumLab (which must be running):
  hwndSpecLab = FindWindow( "TSpectrumLab", NULL );

  // Only if Spectrum Lab's window was found, generate audio data and send them.
  // Since 2012-03 : POLL Spectrum Lab how many samples it can accept.
  if( hwndSpecLab!=NULL && g_sAudioMsgBuffer!=NULL )
   {
     nSamples = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf ); // free space in the LOCAL buffer

     // Unless the command line 'commanded' fixed parameters,
     // kindly ask the 'remote end' (usually Spectrum Lab) about
     // the currently used sample rate, and the number of channels.
     // Here, in the Audio-Message-TESTER, this is polled periodically,
     //       so SL can be restarted with different parameters without the need
     //       to restart the audio message tester.
     // In your application, you have other (simpler) choices:
     //       - Don't poll the parameters at all (because you know the parameters
     //         are correctly set in SL's user configuration already); or
     //       - Poll the current parameters (number of channels + sampling rate)
     //         only ONCE, before initialising your audio-message-buffer.
     //         (btw, that's the reason why we don't need a T_AudioMsgBuffer object
     //          as parameter for AudioMsg_PollRemoteXYZ.. )
     cp = g_sz80InfoStrings[1];
     if(  g_iFixedNumChannels > 0 ) // no FIXED number of channels (from commandline) ->
      { iNumChannels = g_iFixedNumChannels;
        sprintf(cp, "%d channels [fix], ", (int)iNumChannels );
      }
     else // use the same "number of channels" as currently used by Spectrum Lab:
      { iNumChannels = AudioMsg_PollRemoteNumChannels_WMSG( hWnd/*hwndMyWindow*/,  hwndSpecLab/*hwndHisWindow*/ );
        sprintf(cp, "%d channels [var], ", (int)iNumChannels );
      }
     if( iNumChannels > 0 )
      { // Update the 'current' parameter for the audio buffer instance:
        AudioMsgBuf_SetNumberOfChannels(  pBuf, iNumChannels  );
      }
     cp += strlen(cp);
     if( g_iFixedSamplingRate > 0 )  // samples per second (from commandline),
      { iSamplingRate = g_iFixedSamplingRate;
        sprintf(cp, "%d samples/s [fix]  ", (int)iSamplingRate );
      }
     else // use the same sampling rate as currently used by Spectrum Lab:
      { iSamplingRate = AudioMsg_PollRemoteSamplingRate_WMSG( hWnd/*hwndMyWindow*/,  hwndSpecLab/*hwndHisWindow*/ );
        sprintf(cp, "%d samples/s [var]  ", (int)iSamplingRate );
      }
     if( iSamplingRate > 0 )
      { // Update the 'current' parameter for the audio buffer instance:
        AudioMsgBuf_SetCurrentSamplingRate( pBuf, iSamplingRate );
      }


     free_buffer_space_at_receiver = // free space in the REMOTE receiver's buffer
       AudioMsg_PollRemoteBufferSpace_WMSG( hWnd/*hwndMyWindow*/,  hwndSpecLab/*hwndHisWindow*/ )
       / g_sAudioMsgBuffer->info.i32NumChannels;
     // If we ignored the above buffer-check and kept sending audio data
     //  faster than 'the other side' can accept, AudioMsgBuf_SendAudioViaWmCopydata()
     //  would be blocked for a few hundred milliseconds, which may be evil .
     if( nSamples > free_buffer_space_at_receiver )
      {  nSamples = free_buffer_space_at_receiver; // don't produce more samples than the other side can accept
      }
     if( nSamples > 16384 ) // keep the size of a single WM_COPYDATA message low
      {  nSamples = 16384;  // (16384 samples per 50 milliseconds are sufficient
                            //  even for 192kSamples/sec in 'real time')
      }
     // Only produce new data if the receiver (and the audio-message-buffer)
     //   is ready to accept a reasonable number of samples :
     if( nSamples >= 1024 )
      { dwBufIdx = (DWORD)pBuf->m_i64SampleCountIn & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
        for(iSample=0; iSample<nSamples; ++iSample )
         {
           // Generate a low-frequency sinewave for the MODULATION:
           sdblFreqModPhaseAccu += 0.2/*Hz*/ * 2.0 * 3.14159265 / pBuf->info.dblSampleRate;
           if( sdblFreqModPhaseAccu > 2.0 * 3.14159265 )
               sdblFreqModPhaseAccu -= 2.0 * 3.14159265;
           signal_frequency = 1000.0/*Hz*/ + 900.0 * sin(sdblFreqModPhaseAccu);

           // Generate the sinewave, with 'signal_frequency' = momentary frequency in Hz
           sdblPhaseAccu += signal_frequency/*Hz*/ * 2.0 * 3.14159265 / pBuf->info.dblSampleRate;
           if( sdblPhaseAccu > 2.0 * 3.14159265 )
               sdblPhaseAccu -= 2.0 * 3.14159265;
           pBuf->fltSampleBuffer[dwBufIdx][0] = sin(sdblPhaseAccu);
           pBuf->fltSampleBuffer[dwBufIdx][1] = cos(sdblPhaseAccu);
           dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
         } // end while
        pBuf->m_i64SampleCountIn += nSamples;  // also used as FIFO HEAD INDEX (!)
        pBuf->info.ldblUnixDateAndTime += (long double)nSamples / pBuf->info.dblSampleRate;
        i = AudioMsgBuf_SendAudioViaWmCopydata( pBuf, hWnd/*MyWindow*/, hwndSpecLab/*HisWindow*/,
              g_iDataType_Bits); // [in] data type for transmission: 8 = 8 bit unsigned,
                                 // 16=16 bit signed integer, 32= 32-bit floating point
        // AudioMsgBuf_SendAudioViaWmCopydata() may not be able to send
        //     the ENTIRE buffer content, because it will only send ONE
        //     WM_COPYDATA message, with a size resticted to approx. 64 kByte !
        ++i32MessagesSent;
        sprintf(g_sz80InfoStrings[0], "BufSpace=%06d  TxCount=%d     ",
             (int)i, (int)i32MessagesSent );
      } // end if < free_buffer_space_at_receiver large enough >
     else // not worth to 'produce' new data, but there may still be samples
      {   // still waiting for transmission in the T_AudioMsgBuffer object !
        nSamples = nSamples; // << potential breakpoint
          // Those 'remaining' samples will remain in the FIFO, until the next
          // 50-millisecond-timer interval .
        // sprintf(g_sz80InfoStrings[0], " Out of buffer space " ); // this is NOT an error !
      }
   } // end if( W2S_hwndDestWindow!=NULL )
  else
   {
     strcpy( g_sz80InfoStrings[0], "Can't find window \"TSpectrumLab\" ! ");
   }

  hdc = GetDC( hWnd );
   {
     for(i=0; i<NUM_INFO_STRINGS; ++i)
      {
        TextOut( hdc, 0, 16*i, g_sz80InfoStrings[i], strlen(g_sz80InfoStrings[i]) );
      }
   }
  ReleaseDC( hWnd, hdc );

  fAlreadyHere = FALSE;

} // end OnWmTimer()



/*  This function is called by the Windows function DispatchMessage()  */

LRESULT CALLBACK WindowProcedure (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  switch (message)                  /* handle the messages */
   {

     case WM_DESTROY:
        PostQuitMessage (0);       /* send a WM_QUIT to the message queue */
        break;
 
     //-------------- optional message handlers ----------------------------
     case WM_TIMER:
        OnWmTimer(hwnd, (WORD)wParam/*wTimerID*/ ); 
        return 0;

     case WM_KEYUP:
        switch( wParam )
         { case 'P': 
              g_fPaused = !g_fPaused;
              break;
           default:
              break;
         }
        return 0; // "An application should return zero if it processes this message"

  /* case WM_COPYDATA:
        return OnWmCopydata( hWnd,  // handle of THIS window
                     (HWND)wParam,  // handle of SENDING window
         (PCOPYDATASTRUCT)lParam);  // pointer to a COPYDATASTRUCT (read-only!)
   */

     default:                      /* for messages that we don't deal with */
        return DefWindowProc (hwnd, message, wParam, lParam);
   }

  return 0;
}


//---------------------------------------------------------------------------
//  Functions called from the audio message module (audiomsg.c),
//      and other modules which use "utility1.h" ...
//---------------------------------------------------------------------------

void DEBUG_EnterErrorHistory(  
        int  debug_level,   // importance of the message (to be entered or not)
        int  options,       // reserved for future extensions
     double time_of_occurrence,  // fractional GMT SECONDS since Jan.1970
        char *message_fmt, // pointer to null-terminated format string (source)
                   ... )   // optional argument list (printf-style)
{
} // end DEBUG_EnterErrorHistory()

//---------------------------------------------------------------------------
//  Functions called from the WSOCK module ...
//---------------------------------------------------------------------------

      // Error- and event log...
void WSOCK_LogError( char * pszText, int nErrorCode)
{
}
void WSOCK_LogEvent( char * pszFormat, ... )
{
}

      // Client...
void WSOCK_CLI_OnConnect( T_WSOCK_Client *pClient )
{
}
void WSOCK_CLI_OnDisconnect( T_WSOCK_Client *pClient, int iErrorCode )
{
}
void WSOCK_CLI_OnWrite( T_WSOCK_Client *pClient )
{ // notification of readiness for writing ("may send now")
}
void WSOCK_CLI_OnRead ( T_WSOCK_Client *pClient )
{
}

      // Server...
BOOLEAN WSOCK_SRV_OnConnect( T_WSOCK_Server *pServer )
{
  return FALSE;
}
void WSOCK_SRV_OnDisconnect( T_WSOCK_Server *pServer, int iErrorCode )
{
}
void WSOCK_SRV_OnWrite( T_WSOCK_Server *pServer )
{
}
void WSOCK_SRV_OnRead ( T_WSOCK_Server *pServer )
{
}

/* EOF < AudioMessageTest_main.c >  . Leave an empty line after this for certain compilers ! */
