/***************************************************************************/
/* File:   c:\cbproj\SoundUtl\AudioFileIO.h                                */
/* Author: Wolfgang Buescher (DL4YHF)                                      */
/* Date:   2008-11-15                                                      */
/* Purpose:                                                                */
/*   Unit to import and export *.wav - files, etc.                         */
/*   - by DL4YHF August '2002 as a common wrapper for RIFF WAVE and others */
/*-------------------------------------------------------------------------*/

#ifndef _AUDIO_FILE_IO_H_
#define _AUDIO_FILE_IO_H_

#include "SWITCHES.H"

// Are we going to use some WINAMP PLUGINS to support exotic audio file formats ?
#ifndef  SWI_WINAMP_PLUGINS_SUPPORTED
# define SWI_WINAMP_PLUGINS_SUPPORTED 0 /* 0 = do NOT use winamp plugins */
#endif

#include "ChunkInfo.h"
#include "CWaveIO.h"      // Wave-file-I/O implemented as a C++ class
#include "VorbisFileIO.h" // class C_VorbisFileIO (not poisoned by software patents)
#include "AudioTextIO.h"
#include "QFile.h"        // DL4YHF's file abstraction layer ("Quick File I/O")
#ifndef C_SND_MATH_INVALID_FLOAT_VALUE
# include "SoundMaths.h"
#endif


#if( SWI_WINAMP_PLUGINS_SUPPORTED )
 #include "WPH_Core.h"      // "core" to load winamp plugins dynamically ("Winamp Plugin Host")
 #include "WPH_IO_Plugin.h" // C_WPH_IO_Plugin : class for wrapping winamp input/output plugins
#endif // SWI_WINAMP_PLUGINS_SUPPORTED

/*----------- Constant Definitions ----------------------------------------*/
    // .. moved to c:\cbproj\SoundUtl\AudioFileDefs.h !
#ifndef  _AUDIO_FILE_DEFS_H_
 #include "AudioFileDefs.h" //  contains AUDIO_FILE_FORMAT_UNKNOWN,
             // and type definitions like T_AudioFileInfoCallback .
#endif


/*----------- Data Types --------------------------------------------------*/


/*----------- Prototypes ( no class methods )  ----------------------------*/

extern "C" char *AudioSampleDataTypeToString(int iSizeOfDataType, int iDataTypeFlags); // -> e.g. "float32", etc^10
extern "C" void AudioFileIO_GetSaveDialogFilterAndExtensionForAudioFileFormat(
                 int iAudioFileType, char **ppszFilter, char **ppszExtension );




/*----------- Definition of the C_AnyAudioFileIO  class ------------------*/
class C_AnyAudioFileIO
{
public:
   C_AnyAudioFileIO();
   virtual ~C_AnyAudioFileIO();

   //
   // Public functions for object's users.
   void Exit(void);  // semi-destructor, call *before* UTL_ExitFunction() !

   /***************************************************************************/
   void SetInfoCallback(
           T_AudioFileInfoCallback pInfoCallback, // [in, optional] progress indicator callback (or similar)
           void *pvUserCallbackData);             // [in, optional] user callback data


   /***************************************************************************/
   BOOL SetStreamLogfile(char *pszLogfileName);
    /* Prepares writing an optional logfile with compressed audio data.
     * Must be called BEFORE opening a webstream, otherwise the logfile
     *     would be unusable in most cases, without the initial headers.
     * [in] : pszLogfileName = filename of an optional (audio-)logfile,
     *                         or NULL when *no* logfile shall be written.
     */
   BOOL CloseLogfile( void );
    /* Closes the *LOGFILE* (without closing the *STREAM*).
     */
   BOOL GetLogfileInfo(T_StreamLogInfo *pStreamLogInfo);
    /* Retrieves info about the current STREAM LOGFILE (not the stream itself).
     * Type T_StreamLogInfo is defined in AudioFileDefs.h .
     */


   /************************************************************************/
   BOOL InOpen( char *pszFileName,     // [in] filename to open (for reading)
                int  iAudioFileType,   // [in] to solve "*.dat"-ambiguity !
                int  options,          // [in] AUDIO_FILE_OPTION_TIMESTAMPS , ... (bitflags)
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength); // [in] size of pszExtraInfo
     /* Opens any of the SUPPORTED audio files for READING .
      * Returns TRUE on success and FALSE on any error.
      * If there WAS an error, GetErrorString() may provide the reason.
      */


   /************************************************************************/
   int GetState(void); // Returns one of the values defined as AUDIO_FILE_IO_STATE_... in AudioFileDefs.h
   long GetCurrentBitrate(void);  // measures the compressed bitrate [bit/sec]
   LONGLONG GetNumSamplesWritten(void);
   LONGLONG GetNumSamplesRead(void);


   /************************************************************************/
   LONG ReadSamples( BYTE* pData, LONG Length,
         T_ChunkInfo *pOutChunkInfo, // additional results, see ChunkInfo.h
         char *pszExtraInfo, int iMaxStrlen );
     /* Reads some samples from a file without format conversion .
      * "Length" is the size of the caller's buffer capacity in BYTES.
      * Return value: number of SAMPLE POINTS (intervals), not BYTES !!
      */


   /************************************************************************/
   LONG ReadSampleBlocks(
         int first_channel_idx, // channel_nr for 1st destination block
         int nr_samples,      // nr_samples
         T_Float *pfltDest1,  // 1st destination block (~"LEFT")
         T_Float *pfltDest2,  // 2nd destination block (~"RIGHT")
         T_ChunkInfo *pOutChunkInfo, // [out,optional] see ChunkInfo.h, may be a NULL pointer (!)
         char *pszExtraInfo, int iMaxStrlen); // [out,optional] extra string with user data for wave.rd_info
     /* Reads some samples from a wave file (or, in some cases, stream)
      *   which has been opened for READING (or, for a stream, reception)
      *   and converts them to T_Float (floating points).
      * Input parameters:
      *    first_channel_idx: 0= begin with the first channel (LEFT if stereo recording)
      *                1= begin with the second channel (RIGHT if stereo, or Q if complex)
      *     The UPPER BYTE of 'first_channel_idx' can optionally define
      *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
      *    (first_channel_idx>>8) : 0x00 or 0x01 = only ONE channel per destination block
      *                      0x02         = TWO channels per destination block
      *                                    (often used for complex values aka I/Q)
      *
      *   nr_samples: Max number of samples which can be stored in T_Float dest[0..nr_samples-1]
      *   *dest: Pointer to destination array.
      *   *pOutChunkInfo: Additional results, see ChunkInfo.h,
      *                   may be a NULL pointer (!) if the caller doesn't need it.
      * Returns the NUMBER OF AUDIO SAMPLES if successful,
      *         or a negative value if errors occurred.
      * Note: Returning less samples than you may expect is not an error
      *       but indicates reaching the file's end !
      * Depending on C_AnyAudioFileIO.m_iAudioFileOptions & AUDIO_FILE_OPTION_NON_BLOCKING_CALLS,
      *   if the "file" is in fact a STREAM, this function MAY or MAY NOT wait
      *   until a sufficient amount of data (or 'Ogg pages') has arrived from
      *   the network to deliver the requested number of samples.
      *   Full details (and the first implementation of a blocking or non-blocking
      *   ReadSampleBlocks() )  in C_VorbisFileIO::ReadSampleBlocks() .
      */

   LONG ReadSampleBlocks2( // only here for compatibility
         int first_channel_idx, int nr_samples,
         T_Float *pfltDest1,  T_Float *pfltDest2,
         T_ChunkInfo *pOutChunkInfo, char *pszExtraInfo, int iMaxStrlen);


   /************************************************************************/
   void SetExtraHeaderStringForAuxFile( char *pszExtraColumnHeader );
     /* Call before OutOpen, if the 'auxiliary file' shall have
      * a user-defined 'header string' for the extra DATA COLUMNS .
      * This should (!) match the 'pszExtraInfo' in WriteSamples() later.
      */

   /************************************************************************/
   BOOL OutOpen( char *pszFileName,
           int iAudioFileFormat, // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int bits_per_sample,   // 8,16,24=integer, 32 or 64=floating point ! (ignored for compressed files)
           int channels,
           T_ChunkInfo *pInChunkInfo, // additional input, see ChunkInfo.h
           double dblInfoWriteInterval_sec, // interval, in seconds, to add 'chunk info' blocks in the output. 0=off.
                       //  (can be configured in SpecLab under "Options".."GPS Receiver" )
           int    iColumnSeparatorChar );   // ' ' ',' ';' or '\t'
  /* Creates and Opens an audio file for WRITING audio samples.
   *         Returns TRUE on success and FALSE on any error.
   * Input parameters..
   *         see detailed description of arguments in XXXX.cpp
   */


   /************************************************************************/
   LONG WriteSamples( BYTE* pData, LONG Length, // low-level output, try to avoid
                      T_ChunkInfo *pInChunkInfo, char *pszExtraInfo );
   LONG WriteSamples_Float( // app-layer output, using NORMALIZED floats
      T_Float *pfltSrc, T_ChunkInfo *pChunkInfo, char *pszExtraInfo );


   /************************************************************************/      
   long   GetParameterByID( int iAudioFileParam );
            // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h.
            // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015.
   BOOL   SetParameterByID( int iAudioFileParam, long i32NewValue );
            // Returns TRUE if the new value could be set, otherwise FALSE.
            // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015.
   double GetParameterByID_Double(int iAudioFileParam); // similar as above, for 64-bit "double" ..
   BOOL   SetParameterByID_Double(int iAudioFileParam, double dblNewValue);


   /************************************************************************/
   // Some more 'Get' - and 'Set' - functions for the AUDIO FILE class ..

   void   GetFileName(char *pszDest, int iMaxLen);
   char  *GetFileName2(void);
   BOOL   IsStream(void);   // -> TRUE when the "file" is in fact an endless, non-seekable stream from the internet, or similar

   double GetSampleRate(void); // this is THE FILE's sampling rate ! 
   void   SetSampleRate(double dblSampleRate);

   int    GetNrChannels(void);
   void   SetNrChannels(int iNrChannels);       // important to read RAW files

   int    GetBitsPerSample(void);  // per "single" sample, regardless of number of channels
   void   SetDataType(int iSizeOfDataType, int iDataTypeFlags); // for RAW files
   void   GetDataType(int *piSizeOfDataType, int *piDataTypeFlags); // partly deprecated !
            // iSizeOfDataType : 0=unknown, 1=8 bit, 2=16 bit, 3=24 bit, 4=32 bit
            // iDataTypeFlags  : AUDIO_FILE_DATA_TYPE_UNSIGNED (-> AudioFileDefs.h)

   void   SetGain( double dblGainFactor );  // .. for raw files with FLOATING POINT numbers !
   double GetGain( void );

   double GetFrequencyOffset(void); // difference between "radio-" minus "baseband-" frequency (internal NCO + other internal frequency shift + external VFO )
   BOOL   SetFrequencyOffset(double dblFrequencyOffset);

   double GetCurrentRecordingTime(void); // Number of seconds since begin of RECORDING (t=0),
                                         //    depending on current sample index,
                                         //    does *NOT* include Recording Start Time (see below).
   double GetRecordingStartTime(void);  // seconds elapsed since 00:00:00 GMT, January 1, 1970
   void   SetRecordingStartTime(
            double dblStartTime); // [in] "Unix" date-and-time (seconds since 00:00:00 GMT, January 1, 1970)

   BOOL   WriteHeader( T_ChunkInfo *pChunkInfo );
   BOOL   CloseFile( char *pszWhy );

   void   GetErrorString(char *pszDest, int iMaxLen);
   BOOL   IsOpenedForReading(void);
   BOOL   IsOpenedForWriting(void);
   BOOL   AllParamsIncluded(void);    // concerns the HEADER of a WAVE RIFF file

   // should be private but must be public, because these members are used in callbacks:
   T_AudioFileInfoCallback m_pInfoCallback; // [optional] progress indicator callback (or similar)
   void  *m_pvUserCallbackData;             // [optional] user callback data


private:
   int    m_iAudioFileFormat;         // WAVE_FORMAT_xxx
   int    m_iChannelsPerSample;       // usually 1 (mono) or 2 (stereo)
   int    m_iBytesPerSampleGroup;     // (#channels * #bits/ADC_value) / 8
   LONGLONG m_i64SamplePointsWritten; // used as key into the extra 'info' file (while WRITING a file)
   LONGLONG m_i64SamplePointsRead;    // used as key into the extra 'info' file (while WRITING a file)
   int    m_iPosUpdateCounter_WR;     // to detect 'new' GPS position, here: for WRITING files
   int    m_iPosUpdateCounter_RD;     // similar GPS-position-update-counter, here: for READING files
   double m_dblInitialSampleRate;     // initial sampling rate (samples/sec)
   double m_dblCurrSampleRate;        // current sampling rate (from last chunk written to or read from file)
   double m_dblInfoWriteInterval_sec; // interval, in seconds, to add 'chunk info' blocks in the output. 0=off.
   long double m_ldblCurrUnixDateAndTime; // current 'absolute' date and time (for READING files) - 80 bit float !
   double m_dblCurrTimeSinceRecStart; // current 'relative' time since the file began
   char   m_cErrorString[84];         // only contains errors detected by this wrapper,
                                      // not from the low-level audio file driver !
   char   m_sz80ExtraColumnHeaderForAuxFile[84];
   char   m_sz511CurrFileName[512];   // Name of the current file (from OutOpen for WriteHeader)
   char   m_sz255StreamLogfile[256];  // Name of the optional "webstream logfile"
   BOOL   m_fWriteLogfile;            // TRUE=yes, FALSE=no
   BOOL   m_fMustWriteHeader;         // flag for the 1st call of C_AnyAudioFileIO::WriteSamples()

   C_WaveIO       WaveIO;    // class for handling WAVE- and RAW files
   C_AudioTextIO  TextIO;    // class for storing audio samples in text files
   C_VorbisFileIO VorbisIO;  // class for handling Ogg/Vorbis *audio* files (*.ogg)
   T_QFile        qfAuxFile;  // 'auxiliary' File I/O object for the GPS- and 'Info'- log
   T_QFile        qfWebstreamLog;       // logfile for web-stream data (usually Ogg/Vorbis format)
   int          m_iColumnSeparatorChar; // column separator character in 'auxiliary' files, for WRITING
    // (for reading, any of the characters SPACE, COMMA, SEMICOLON, or TAB is accepted)
   int          m_iAudioFileOptions; // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> AudioFileDefs.h]
   BOOL         m_fReadAuxFile; // flag from the main file header, set if an auxiliary file should be present
   typedef struct
    { T_ChunkInfo  chunk_info;
      char         sz255ExtraInfo[256];
      DWORD        dwSampleIndex;
      BOOL         fValid;
    } tInfo2;
   tInfo2 m_CurrInfo, m_NextInfo; // 'current' and 'next' audio chunk info

# if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
          // C_WPH_IO_Plugin WPH_Plugin is just a wrapper between the
          // C_AnyAudioFileIO class and the 'experimental' winamp-plugin-host.
          // Implemented in c:\cbproj\WinampPluginHost\WPH_IO_Plugin.cpp .
          // The CORE of the winamp plugin (-host) is written in C, not C++ :
          // Heavily commented code in c:\cbproj\WinampPluginHost\WPH_Core.c .
   C_WPH_IO_Plugin WPH_Plugin; // class for wrapping winamp input/output plugins
# endif

   void WriteChunkInfo( unsigned long dwSampleIndex, T_ChunkInfo *pChunkInfo, char *pszExtraInfo );
   BOOL LoadChunkInfoForSampleIndex(unsigned long dwSampleIndex); // -> m_CurrChunkInfo
   void SetTimesInReturnedChunkInfo(T_ChunkInfo *pOutChunkInfo);


}; /* end class C_AnyAudioFileIO */


#endif // _AUDIO_FILE_IO_H_

/* EOF <AudioFileIO.h> */