/***************************************************************************/
/*  \cbproj\SoundUtl\AudioFileIO.cpp :                                     */
/*   Unit to import and export *.wav - files, etc.                         */
/*   - by DL4YHF August '2002 as a common wrapper for RIFF WAVE and others */
/*  Recent changes :                                                       */
/*   2020-10-11  : Added AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS .     */
/*   2008-11-15  : Optional support for WINAMP INPUT PLUGINS .             */
/*   2004-11-04  : Added the option to read simple TEXT files,             */
/*                 with one sampling point per line .                      */
/*                                                                         */
/*  Used in / by :                                                         */
/*   - Spectrum Lab by WoBu, to "analyse and play WAVE files" (etc) .      */
/*-------------------------------------------------------------------------*/

#include <windows.h> // definition of data types like LONGLONG (wtypes.h) etc
#include <stdio.h>   // no 'standard I/O' but sprintf()
#include <math.h>

#pragma hdrstop      // for Borland: no precompiled headers after this point.

#include "Timers.h"    // high-resolution 'current time'-routine

#include "ChunkInfo.h"   // \cbproj\SoundUtl\ChunkInfo.h describes a 'chunk' of audio samples
#include "ErrorCodes.h"  // Combination of WINDOWS error codes + DL4YHF's own
#include "YHF_Inet.h"    // DL4YHF's 'internet' helper functions (uses sockets)
#include "AudioFileIO.h" // header for this file



/*------------- mother's little helpers -----------------------------------*/


/****************************************************************************/
BOOL AudioFileSupportsInfoChannel(
            int iAudioFileType) // [in] audio file type like AUDIO_FILE_FORMAT_RIFF
  // Return value : 1 = this type of audio files supports an 'info channel'
  //                    INTERNALLY (i.e. GPS log within the same file as the audio)
  //                0 = this type of audio file can NOT embed an 'info channel'
  //                    in the same file as the audio; thus AudioFile must
  //                    create (or open) a SECOND file for the 'info channel' .
{
  switch( iAudioFileType ) // which of these files supports an internal 'info' channel ?
   {
     case AUDIO_FILE_FORMAT_UNKNOWN: // for opening a file with auto-detection
     case AUDIO_FILE_FORMAT_RAW    : // headerless 'raw' audio samples (no *.WAV - file !)
     case AUDIO_FILE_FORMAT_RIFF   : // the common windoze - *.WAV-format
     case AUDIO_FILE_FORMAT_OGG    : // Ogg/Vorbis audio files (ogg=container, vorbis=codec)
     case AUDIO_FILE_FORMAT_TEXT   : // simple TEXT-file format, one sampling point per line
     case AUDIO_FILE_FORMAT_PLUGIN : // any other format, requiring a PLUGIN
     case AUDIO_FILE_FORMAT_SETI_1 : // SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
     case AUDIO_FILE_FORMAT_MP3    : // not supported by SpecLab (requires plugin)
     default:
        return 0;  // all these files do NOT support an internal 'info' channel



   // ex:  case AUDIO_FILE_FORMAT_S_DAT  : // Vorbis-based VLF sreaming format with timestamps, etc
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS:  // uncompressed stream, identified by T_StreamHeader structs
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
        return 1;
   }
} // end AudioFileSupportsInfoChannel()

//---------------------------------------------------------------------------
void AudioFileIO_GetSaveDialogFilterAndExtensionForAudioFileFormat( // what a name - but leaves no doubt about the purpose :o)
       int iAudioFileType,   // [in]  e.g. AUDIO_FILE_FORMAT_OGG
       char **ppszFilter,    // [out] e.g. "Ogg Vorbis audio (*.ogg)|*.ogg"
       char **ppszExtension) // [out] e.g. "ogg" (not "*,ogg" ! )
  // See also (file selector with ALL filters supported by Spectrum Lab) :
  //   C:\cbproj\SpecLab\SpecMain.cpp : TSpectrumLab::SelectAudioFiles() !
{
  char *pszFilter    = "Unknown / binary (*.bin)|*.bin";
  char *pszExtension = "bin";
  switch( iAudioFileType ) // which of these files supports an internal 'info' channel ?
   {
     case AUDIO_FILE_FORMAT_RAW    : // headerless 'raw' audio samples (no *.WAV - file !)
          pszFilter    = "Raw audio files (*.raw)|*.raw";
          pszExtension = "bin";
          break;
     case AUDIO_FILE_FORMAT_RIFF   : // the common windoze - *.WAV-format
          pszFilter    = "Wave Files (*.wav)|*.wav";
          pszExtension = "wav";
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg/Vorbis audio files (ogg=container, vorbis=codec)
          pszFilter    = "Ogg Vorbis audio (*.ogg)|*.ogg";
          pszExtension = "ogg";
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // simple TEXT-file format, one sampling point per line
          pszFilter    = "Audio samples as text (*.txt)|*.txt";
          pszExtension = "txt";
          break;
     case AUDIO_FILE_FORMAT_SETI_1 : // SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          pszFilter    = "SETI 8-bit raw I/Q (*.dat)|*.dat";
          pszExtension = "dat";
          break;
     case AUDIO_FILE_FORMAT_MP3    : // not supported by SpecLab (requires plugin)
          pszFilter    = "MP3 compressed audio (*.mp3)|*.mp3";
          pszExtension = "mp3";
          break;
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS:  // uncompressed stream, identified by T_StreamHeader structs
          pszFilter    = "Non-compressed stream logs (*.dat)|*.dat";
          pszExtension = "dat";
          break;
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          pszFilter    = "VLF-RX streams with VT_BLOCKs (*.vlfrx)|*.vlfrx";
          pszExtension = "vlfrx";
          break;
     default:
          break; // leave pszFilter and pszExtension as-is ("bin")
   } // end switch( iAudioFileType )

  if( ppszFilter != NULL )
   { *ppszFilter = pszFilter;  // [out] pointer to a const-string
   }
  if( ppszExtension != NULL )
   { *ppszExtension = pszExtension;
   }

} // end AudioFileIO_GetSaveDialogFilterAndExtensionForAudioFileFormat()


//---------------------------------------------------------------------------
char *AudioFileToAuxFileName( char *pszAudioFileName )
{
  static char sz255InfoFileName[256];
  char *cp;
  strncpy(sz255InfoFileName, pszAudioFileName, 255 );
  sz255InfoFileName[255] = '\0';
  cp = strrchr(sz255InfoFileName, '.' );
  if( cp )
   { strcpy(cp, ".aux" );   // *.aux for "AUXILIARY" file (simple text file)
   }
  return sz255InfoFileName;
} // end AudioFileToAuxFileName()


/****************************************************************************/
long GetNominalSampleRate( double dblSR ) // precise or calibrated SR -> nominal SR
{                                         // (sometimes required for the file header)
  if( dblSR>5400 && dblSR<5600 )
   { return 5512;   // theoretically 5512.5 samples/seccond
   }
  if( dblSR>7000 && dblSR<9000 )
   { return 8000;
   }
  if( dblSR>10000 && dblSR<11500 )
   { return 11025;
   }
  if( dblSR>11500 && dblSR<13000 )
   { return 12000;
   }
  if( dblSR>15000 && dblSR<17000 )
   { return 16000;
   }
  if( dblSR>21000 && dblSR<23000 )
   { return 22050;
   }
  if( dblSR>31000 && dblSR<33000 )
   { return 32000;
   }
  if( dblSR>43000 && dblSR<45000 )
   { return 44100;
   }
  if( dblSR>46000 && dblSR<50000 )
   { return 48000;
   }
  if( dblSR>94000 && dblSR<98000 )
   { return 96000;
   }
  if( dblSR>=19000 && dblSR<194000 )
   { return 192000;
   }
  return (long)(dblSR+.5);  // non-standard SR, leave this as-is
} // end GetNominalSampleRate()


/***************************************************************************/
static void ReplaceCommasWithColumnSeparators( char *pszString, char cColumnSeparatorChar )
{
  while( *pszString != '\0' )
   {
     if( *pszString == ',' )
      {  *pszString = cColumnSeparatorChar;
      }
     ++pszString;
   }

} // end ReplaceCommasWithColumnSeparators()

/***************************************************************************/
char *AudioSampleDataTypeToString( // returns data type names like "float64" (*)
         int iSizeOfDataType, // [in] 1,2,4, or even 8 bytes per single-channel sample
         int iDataTypeFlags)  // [in] AUDIO_FILE_DATA_TYPE_FLOAT,
                              //      AUDIO_FILE_DATA_TYPE_SIGNED_I,
                              //      AUDIO_FILE_DATA_TYPE_UNSIGNED, ..
  // (*) Names (strings) returned by VFIO_VTFlagsToString()
  //                      and AudioSampleDataTypeToString() should be COMPATIBLE !
  // Btw, if the IDE is too stupid to find the definitions of AUDIO_FILE_DATA_TYPE_..:
  //      They are defined in c:\cbproj\SoundUtl\AudioFileDefs.h .
{
  switch( (iDataTypeFlags << 4) | iSizeOfDataType )
   { case (AUDIO_FILE_DATA_TYPE_SIGNED_I << 4) | 1  : return "int8";
     case (AUDIO_FILE_DATA_TYPE_SIGNED_I << 4) | 2  : return "int16";
     case (AUDIO_FILE_DATA_TYPE_SIGNED_I << 4) | 3  : return "int24";
     case (AUDIO_FILE_DATA_TYPE_SIGNED_I << 4) | 4  : return "int32";
     case (AUDIO_FILE_DATA_TYPE_SIGNED_I << 4) | 8  : return "int64";
     case (AUDIO_FILE_DATA_TYPE_UNSIGNED << 4) | 1  : return "uns8";
     case (AUDIO_FILE_DATA_TYPE_UNSIGNED << 4) | 2  : return "uns16";
     case (AUDIO_FILE_DATA_TYPE_UNSIGNED << 4) | 3  : return "uns24";
     case (AUDIO_FILE_DATA_TYPE_UNSIGNED << 4) | 4  : return "uns32";
     case (AUDIO_FILE_DATA_TYPE_UNSIGNED << 4) | 8  : return "uns64";
     case (AUDIO_FILE_DATA_TYPE_FLOAT    << 4) | 4  : return "float32";
     case (AUDIO_FILE_DATA_TYPE_FLOAT    << 4) | 8  : return "float64";
     default: return "???";
   }
} // end AudioSampleDataTypeToString()

/*------------- Implementation of the class  C_AnyAudioFileIO -------------*/

//---------------------------------------------------------------------------
extern "C" int AudioFileIO_InetCallback(
          void *pvUserCallbackData,
          struct tInetClient *pInetClient,
          int iEventType,
          char *pszInfo,
          int  iProgressPercent)
  // Callback function for INET_OpenAndReadFromServer() .
  // Used as a progress indicator AND to analyse (parse) the HTTP GET RESPONSE
  // (which may tell us a bit more about the type of the file) .
{
  C_AnyAudioFileIO *pAIO = (C_AnyAudioFileIO*)pvUserCallbackData;
  int iAudioFileIOEvent;

  if( pAIO->m_pInfoCallback )  // pass this info on to an upper layer ?
   { switch( iEventType & (~INET_EVENT_FLAG_SHOW_TIMESTAMP) )
      { case INET_EVENT_TYPE_INFO :
           iAudioFileIOEvent = AUDIO_FILE_IO_EVENT_TYPE_INFO;
           break;
        case INET_EVENT_TYPE_ERROR :
           iAudioFileIOEvent = AUDIO_FILE_IO_EVENT_TYPE_ERROR;
           break;
        case INET_EVENT_TYPE_PARSE_RESPONSE :
           iAudioFileIOEvent = AUDIO_FILE_IO_EVENT_TYPE_PARSE_RESPONSE;
           break;
        default:  // no equivalent for this inAudioFileIO.h ...
           iAudioFileIOEvent = 0;
           break;
      }
     if( iAudioFileIOEvent )
      { pAIO->m_pInfoCallback( pAIO->m_pvUserCallbackData,
                  iAudioFileIOEvent, pszInfo, iProgressPercent);
      }
   } // end if( pAIO->m_pInfoCallback )

  return NO_ERRORS;
} // end AudioFileIO_InetCallback()


/***************************************************************************/
C_AnyAudioFileIO::C_AnyAudioFileIO()
  /* Constructor of the Wave-IO Class.
   * Initializes all properties etc.
   */
{
  m_iAudioFileFormat = AUDIO_FILE_FORMAT_UNKNOWN;
  m_iPosUpdateCounter_WR=0; // to detect 'new' GPS position, etc
  m_iPosUpdateCounter_RD=0; // similar GPS-position-update-counter, here: for READING files
  m_i64SamplePointsRead = m_i64SamplePointsWritten = 0;
  m_dblInitialSampleRate = m_dblCurrSampleRate = 0.0; // completely unknown (here in the constructor)
  memset( &qfAuxFile, 0, sizeof(qfAuxFile) );  // File I/O object for the GPS log
  memset( &qfWebstreamLog,0, sizeof(qfAuxFile) );   // logfile for web-stream data
  memset( m_sz255StreamLogfile, 0, sizeof( m_sz255StreamLogfile ) );
  m_fWriteLogfile = FALSE;
  memset( m_cErrorString, 0, sizeof(m_cErrorString) );
  memset( &m_CurrInfo, 0, sizeof(m_CurrInfo) );
  memset( &m_NextInfo, 0, sizeof(m_NextInfo) );
  m_fMustWriteHeader = FALSE;
  ChunkInfo_Init( &m_CurrInfo.chunk_info );  // see \cbproj\SoundUtl\ChunkInfo.c
  ChunkInfo_Init( &m_NextInfo.chunk_info );
} // end of C_AnyAudioFileIO's constructor

/***************************************************************************/
C_AnyAudioFileIO::~C_AnyAudioFileIO()
  /* Destructor of the Wave-IO Class.
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   *
   * Note: To avoid the following in the 'debug run log' ...
   *   > 21:38:28.7 WARNING: Application didn't free 1 memory block  when quitting.
   *   > 21:38:28.7 WARNING: Name of 1st non-freed block = "VorbisIO" .
   *  ... call C_VorbisFileIO::Exit() **before** UTL_ExitFunction() .
   */
{
   Exit();
} // end C_AnyAudioFileIO's destructor


/***************************************************************************/
void C_AnyAudioFileIO::Exit(void)
  /* "Semi-Destructor" of the Wave-IO Class.
   * Can be called BY THE APPLICATION to prevent (false) warnings
   * about non-freed memory blocks - see C_VorbisFileIO::~C_VorbisFileIO() .
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   */
{

   CloseFile(NULL/*pszWhy*/);    // close old file (if still opened)
   // if( qfWebstreamLog.fOpened )
   //  { QFile_Close( &qfWebstreamLog );
   //  }


   VorbisIO.Exit();  // avoid mis-diagnosed warnings about 'non-freed blocks'   

} // end C_AnyAudioFileIO::Exit()


/***************************************************************************/
void C_AnyAudioFileIO::SetInfoCallback(
     T_AudioFileInfoCallback pInfoCallback, // [in, optional] user callback function
     void *pvUserCallbackData)              // [in, optional] user callback data
{
  m_pInfoCallback      = pInfoCallback;
  m_pvUserCallbackData = pvUserCallbackData;
  VorbisIO.SetInfoCallback( m_pInfoCallback, pvUserCallbackData );
} // end C_AnyAudioFileIO::SetInfoCallback()

/***************************************************************************/
BOOL C_AnyAudioFileIO::SetStreamLogfile(
        char * pszLogfileName) // [in] filename of an optional (audio-)logfile;
                               //      NULL = we don't want to log incoming or outgoing stream
  // This method must be called BEFORE opening a webstream,
  // because otherwise we'd miss the signature ("OggS") in the log,
  // and the result wouldn't be a valid Ogg/Vorbis file anymore !
{
  if( pszLogfileName == NULL )
   { m_fWriteLogfile = FALSE;
   }
  else
   { m_fWriteLogfile = TRUE;
     strncpy( m_sz255StreamLogfile, pszLogfileName, 255 );
   }
  VorbisIO.SetStreamLogfile( pszLogfileName );
  return TRUE;
} // end C_AnyAudioFileIO::SetStreamLogfile()

/***************************************************************************/
BOOL C_AnyAudioFileIO::CloseLogfile( void )
  // Closes the *LOGFILE*, without closing the *STREAM* itself.
{
  BOOL fResult = VorbisIO.CloseLogfile();
  // "Anyone else" who supports stream logfiles ?   Call them from here !
  return fResult;

} // end C_AnyAudioFileIO::CloseLogfile()


/***************************************************************************/
BOOL C_AnyAudioFileIO::GetLogfileInfo(
       T_StreamLogInfo *pStreamLogInfo) // [out] 'audio stream log info', defined in c:\cbproj\SoundUtl\AudioFileDefs.h
  // Retrieves info about the current STREAM LOGFILE (not the stream itself).
  // Returns TRUE if the info in the T_StreamLogInfo is *valid*,
  //         which doesn't mean the stream is CURRENTLY BEING LOGGED !
  // ( check flag T_StreamLogInfo.iLogging, as specified in AudioFileDefs.h )
{
  BOOL fResult = VorbisIO.GetLogfileInfo(pStreamLogInfo);
  // "Anyone else" who supports stream logfiles ?   Call them from here !
  return fResult;
} // end C_AnyAudioFileIO::GetLogfileInfo()


/***************************************************************************/
BOOL C_AnyAudioFileIO::InOpen(
                char * pszFileName,   // [in] filename or URL(*) to open (for reading / stream-in)
                int  iAudioFileType,  // [in] to solve "*.dat"-ambiguity !
                int  options,         // AUDIO_FILE_OPTION_TIMESTAMPS , ... (bitflags)
                char *pszExtraInfo,   // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength) // [in] size of pszExtraInfo
  /* Opens a local audio file or an internet audio stream
   *       for READING, and (if necessary) start uncompressing samples.
   * Returns TRUE on success and FALSE on any error.
   * On return, all information which the caller needs to know
   *            must be available *before* the first call of ReadSamples().
   *           (for example, SL's file processing dialog presents
   *            the precise sampling rate, number of channels,
   *            "radio" center frequency, etc, and lets the user confirm
   *            or even modify these parameters before processing the data.
   *            See  TSpectrumLab::OpenInputFileAnalysis() .
   *            Applies to (at least) .GetFrequencyOffset(),
   *                   .GetTotalCountOfSamples(), .GetRecordingStartTime(),
   *                   .GetNrChannels(), .GetDataType(),  .GetFileFormat(),
   *                   and possibly a bunch of other member functions.
   *            Thus, C_AnyAudioFileIO::InOpen() may have to WAIT for a few
   *            seconds (AUDIO_FILE_IO_MAX_TIMEOUT_MS) if the "file"
   *            is in fact a WEB AUDIO STREAM !
   *
   * (*) URL of the 'REAL' audio stream - please no stupid M3U .
   *     To open a "stream" which in fact is just an M3U ("playlist"),
   *     THE CALLER is responsible to open/download that M3U first,
   *     and pick one of the names from that playlist.
   */
{
  BOOL fOpened = FALSE;
  char sz1kExtraInfo[1024];
  char *cp;
  T_INET_Client InetClient;  // Simple HTTP client, implemented in YHF_Inet.c
  int i;

  CloseFile("before re-opening");    // close old file (if still opened)
  // if( qfWebstreamLog.fOpened )
  //  { QFile_Close( &qfWebstreamLog );
  //  }
  memset( m_cErrorString, 0, sizeof(m_cErrorString) );
  ChunkInfo_Init( &m_CurrInfo.chunk_info );
  m_fReadAuxFile = FALSE;
  m_i64SamplePointsRead = 0;
  m_iPosUpdateCounter_RD = 0;
  m_dblCurrTimeSinceRecStart = 0.0; // current 'relative' time since the file began
  m_CurrInfo.fValid = m_NextInfo.fValid = FALSE;
  strncpy( m_sz511CurrFileName, pszFileName, 511); // since 2020-10, value returned by GetFileName2() ... KISS !
  m_iAudioFileFormat = INET_GuessAudioFileTypeFromName(pszFileName, iAudioFileType);
  if( m_iAudioFileFormat == AUDIO_FILE_FORMAT_OTHER_WEB_STREAM )
   { // Can't tell the format from the filename, i.e. "http://67.207.143.181:80/vlf1" :
     // Must connect the URL, and download the first part of the file
     // to find out what it REALLY is. After that, we hopefully know if
     // the stream is MP3-encoded, or Ogg/Vorbis, or whatever. Examples:
     // The content-type from http://67.207.143.181:80/vlf1
     //                  was  "application/ogg" (Todmorden, using Ogg/Vorbis) .
     // The content-type from http://67.207.143.181:80/vlf6
     //                  was  "application/mpeg" (Spenge, using MP3 compression) .
     // Since there is no internal codec for it, MP3-streams are not supported !
     INET_InitClient( &InetClient,
              AudioFileIO_InetCallback, (void*)this ); // [in] optional callback
     i = INET_OpenAndReadFromServer(
              &InetClient,      // [in,out] T_INET_Client struct
              pszFileName,      // [in] filename or URL
              AUDIO_FILE_IO_MAX_TIMEOUT_MS, // [in] max. timeout in milliseconds (ex: 5000, but that's too long)
              INET_RD_OPTION_RETURN_AFTER_RESPONSE, // [in] options
              NULL, 0 ); // [out] NETTO file data ("content", no headers) : not needed here
     if( i>=0 )  // got a response from the remote server -> check it !
      { // Look at the "Content-Type", maybe it's one of the known ones..
        // 2011-12-27 :  Got here with InetClient.sz80ContentType == "text/html"
        //               when trying to open the stream at http://67.207.143.182:80/testwb .
        m_iAudioFileFormat = INET_ContentTypeToAudioFileFormat( InetClient.sz80ContentType,
                              AUDIO_FILE_FORMAT_UNKNOWN/*Default*/ );
        if( m_iAudioFileFormat == AUDIO_FILE_FORMAT_UNKNOWN )
         { sprintf( m_cErrorString, "Cannot guess file type from \"%s\"",
                                     InetClient.sz80ContentType );
         }
      }
     else // 'i' must be an error code "made negative" by YHF_Inet.c :
      {
      }
     // Regardless of what it was, close the HTTP client again.
     // This is a bit clumsy - it would be more elegant to keep the stream open,
     // but since we already read the first 1024 bytes we'd have to "put them back"
     // into the stream buffer... too complicated !
     sprintf( sz1kExtraInfo, "after type detection ('%s')",
              INET_AudioFileFormatToContentType(m_iAudioFileFormat) );
     INET_CloseClient( &InetClient, sz1kExtraInfo/*pszWhy*/ );
   }
  sz1kExtraInfo[0] = '\0';
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard *.wav (RIFF wave audio)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          fOpened = WaveIO.InOpen(pszFileName, m_iAudioFileFormat, sz1kExtraInfo, 1023 );
          // At this point, sz1kExtraInfo may(!) contain a long string from a wave header,
          // for example (from a wave file with decimated and frequency-converted samples):
          // sk1kExtraInfo = "sr=98.7654321 rf=8270.000000 ut=1420930608.6249 glat=52.1 glon=8.4 masl=108.8 kmh=0.0" .
          //                    |             |              |                    |
          //     precise sampling rate  radio freq offset  universal time     GPS data (lat, lon, height, speed)
          //
          // CWaveIO.cpp also parses the string, with result in its own 'chunk info' because
          //   it needs to have those info before returning from its own "InOpen").
          m_CurrInfo.chunk_info = WaveIO.MyChunkInfo;
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          VorbisIO.SetParameterByID( AUDIO_FILE_PARAM__FILE_FORMAT, m_iAudioFileFormat );
          if( m_fWriteLogfile )
           { VorbisIO.SetStreamLogfile( m_sz255StreamLogfile );
           }
          else // don't produce a 'logfile' for the input stream :
           { VorbisIO.SetStreamLogfile( NULL );
           }
          fOpened = VorbisIO.InOpen(pszFileName, options/*AUDIO_FILE_OPTION_TIMESTAMPS,etc*/ );
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          fOpened = TextIO.InOpen(pszFileName );
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          // C_WPH_IO_Plugin WPH_Plugin is just a wrapper between the
          // C_AnyAudioFileIO class and the 'experimental' winamp-plugin-host.
          // Implemented in c:\cbproj\WinampPluginHost\WPH_IO_Plugin.cpp .
          // The CORE of the winamp plugin (-host) is written in C, not C++ :
          // Heavily commented code in c:\cbproj\WinampPluginHost\WPH_Core.c .
          fOpened = WPH_Plugin.InOpen(pszFileName );
          if( ! fOpened )
           { strcpy( m_cErrorString, "Winamp-plugin host failed to open MP3" );
           }
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          if( m_cErrorString[0] == 0 )
           { strcpy( m_cErrorString, "unsupported audio file format" );
           }
          break;
   }
  if( pszExtraInfo!=NULL && iExtraInfoLength>0 )
   { strncpy( pszExtraInfo, sz1kExtraInfo, iExtraInfoLength );
   }
  if( fOpened )
   { m_dblInitialSampleRate = m_dblCurrSampleRate = GetSampleRate(); // COARSE VALUE (1 Hz resolution from standard wave header)
     m_ldblCurrUnixDateAndTime = GetRecordingStartTime();  // just an initial guess for the recording start time (UNIX format)
     // If there's no better source for the first sample's timestamp,
     //  the file's CREATION DATE AND TIME will also be used for the 'chunk info'.
     // (this may be overwritten immediately below, in StringToChunkInfo() )
     m_CurrInfo.chunk_info.ldblUnixDateAndTime = m_ldblCurrUnixDateAndTime;
   }
  if( fOpened && sz1kExtraInfo[0]>0 ) // also open an 'auxiliary' file (with GPS data, etc) ?
   { // This 'extra info string' may tell us the precise sampling rate, etc etc:
     cp = sz1kExtraInfo;
     if( StringToChunkInfo( &cp, &m_CurrInfo.chunk_info ) )
      { // Copy some of this 'EXTRA INFO' into the sound analysis params,
        //  but only if they are obviously VALID :
        if( m_CurrInfo.chunk_info.dblPrecSamplingRate > 0.0 )
         { m_dblInitialSampleRate = m_dblCurrSampleRate = m_CurrInfo.chunk_info.dblPrecSamplingRate;
         }
        if( m_CurrInfo.chunk_info.ldblUnixDateAndTime > 0.0 )
         { m_ldblCurrUnixDateAndTime = m_CurrInfo.chunk_info.ldblUnixDateAndTime;
         }
        // Pass the above results (from StringToChunkInfo(), in m_CurrInfo.chunk_info)
        // on to the other file codecs, for example AudioFileIO.cpp may also
        // need to the PRECISE sampling rate, NCO offset, because it may be asked for them:
        SetFrequencyOffset( m_CurrInfo.chunk_info.dblRadioFrequency );
      }
     // Regardles of the contents of the 'extra info string',
     //  start counting the number of samples at ZERO at the begin of each track.
     //  This is important for the adjustment of the timestamps later:
     m_CurrInfo.chunk_info.i64TotalSampleCounter = 0;

     // In this case, the 'INFO STRING' (somewhere in the audio file header)
     // may also define the EXISTANCE(! - not the name and path) of an AUXILIARY FILE .
     if( (cp=strstr( sz1kExtraInfo, " auxfile=")) != NULL )
      { // Ok, there seems to be a FLAG for the presence of an 'auxiliary' file.
        // Note: We do NOT save the name of the auxiliary file in the main file,
        //       because the user may RENAME BOTH FILES for his own convenience.
        //  All we need to know is the PRESENCE of an auxiliary file (a 'flag').
        //  auxfile=0 means "there is no auxiliary file" .
        //  auxfile=1 means "there should be an auxiliary file" .
        switch( cp[9/*strlen(" auxfile=")*/] )
         { case '1':  // for "myaudiofile.wav", there should be a "myaudiofile.txt" :
              m_fReadAuxFile = TRUE;
           default:
              break;
         } // end switch < auxfile-flag >
      } // end if < info string contains pattern " auxfile=" >
   }
  if( fOpened && m_fReadAuxFile ) // main file opened, aux file SHOULD exist:
   { if( QFile_Open( &qfAuxFile, AudioFileToAuxFileName(pszFileName), QFILE_O_RDONLY ) )
      {
      }
     else // main file opened, but auxiliary file is missing ..
      { m_fReadAuxFile = FALSE;
        // this is no reason NOT to play the main file,
        // but at least let the caller (the application) know
        // there's something not 100% ok :
        sprintf( m_cErrorString, "missing aux file: %s", AudioFileToAuxFileName(pszFileName) );
      }
   }
  // On this occasion, update *some*(!) infos in the two 'chunk info' structs:
  m_CurrInfo.chunk_info.dblPrecSamplingRate = m_dblInitialSampleRate;
  // ex: m_NextInfo.chunk_info.dblPrecSamplingRate = m_dblInitialSampleRate;
  m_NextInfo.chunk_info = m_CurrInfo.chunk_info; // modif. 2015-01-10

  return fOpened;
} // C_AnyAudioFileIO::InOpen(..)


/************************************************************************/
int C_AnyAudioFileIO::GetState(void)
  // Returns one of the values defined as
  // AUDIO_FILE_IO_STATE_... in AudioFileDefs.h, for example:
  //   AUDIO_FILE_IO_STATE_PASSIVE/RECORDING/SENDING/WAIT_CONN/NEED_TIMESTAMP .
{
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetState();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  return AUDIO_FILE_IO_STATE_PASSIVE; // unknown !
} // end C_AnyAudioFileIO::GetState()

/***************************************************************************/
long C_AnyAudioFileIO::GetCurrentBitrate(void)
  // measures the compressed bitrate [bit/sec]
{
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetCurrentBitrate();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  return 0; // unknown !
} // end C_AnyAudioFileIO::GetCurrentBitrate()

//---------------------------------------------------------------------------
LONGLONG C_AnyAudioFileIO::GetNumSamplesWritten(void)
{
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetNumSamplesWritten();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  return m_i64SamplePointsWritten; // use "our own" counter

} // end C_AnyAudioFileIO::GetNumSamplesWritten()

//---------------------------------------------------------------------------
LONGLONG C_AnyAudioFileIO::GetNumSamplesRead(void)
{
  return m_i64SamplePointsRead; // use "our own" counter
} // end C_AnyAudioFileIO::GetNumSamplesRead()

/***************************************************************************/
void C_AnyAudioFileIO::SetTimesInReturnedChunkInfo(T_ChunkInfo *pOutChunkInfo)
  // Makes sure the various 'times' in pOutChunkInfo are valid :
  // pOutChunkInfo->rtRecTime = recording time (seconds since recording started)
  // pOutChunkInfo->ldblUnixDateAndTime = seconds since Januar 1st, 1970, 00:00:00.0 UTC
  // pOutChunkInfo->dblHRTime_sec = value read by "TIM_QueryHRTimer_sec()" when the sample was acquired;
  //                               completely useless in a recorded file (only for real-time processing)
  // Details about these 'timers' in \cbproj\SoundUtl\ChunkInfo.h   .
{
  if( pOutChunkInfo != NULL )
   {
     pOutChunkInfo->rtRecTime       = m_dblCurrTimeSinceRecStart;
     pOutChunkInfo->ldblUnixDateAndTime = m_ldblCurrUnixDateAndTime;
     // ex: pOutChunkInfo->dblHRTime_sec = m_dblCurrTimeSinceRecStart;
   }
} // end C_AnyAudioFileIO::SetTimesInReturnedChunkInfo()


/***************************************************************************/
LONG C_AnyAudioFileIO::ReadSamples(
        BYTE* pData,   // [out] destination block with audio samples
        LONG Length,   // [in] size of pData (in BYTES)
        T_ChunkInfo *pOutChunkInfo, // [out,optional] see ChunkInfo.h, may be a NULL pointer (!)
        char *pszExtraInfo, int iMaxStrlen ) // [out,optional] extra string with user data for wave.rd_info
  /* Reads some samples from a wave file which has been opened for READING,
   *                    WITHOUT conversion of the sample format in memory .
   * Returns the NUMBER OF AUDIO SAMPLE POINTS if successful,
   *     (not the number of bytes; one sample point may contain up to 8 bytes)
   *   or a negative value if errors occurred.
   * Notes:
   *  - "Length" is the size of the caller's buffer capacity in BYTES.
   *     The actual number of AUDIO SAMPLES will depend on the sample
   *     format. For 16-bit mono recordings, the number of samples
   *     will only be half the "Length".
   */
{
  long nSamplePointsRead = 0; // must not be unsigned 'cos some readers return negative values
                              // if an ERROR has been detected (not just "temporarily no data")
  double dblTsample;

  // if necessay (and possible), read the next INFO BLOCK from a separate file:
  if( LoadChunkInfoForSampleIndex( m_i64SamplePointsRead ) )
   {
     if( pOutChunkInfo!=NULL)
      { *pOutChunkInfo = m_CurrInfo.chunk_info;   // return chunk info from 'auxiliary' file
        // Make sure the various 'times' in pOutChunkInfo are valid:
        SetTimesInReturnedChunkInfo( pOutChunkInfo );
      }
     if( (pszExtraInfo!=NULL) && (iMaxStrlen>0) )
      { strncpy( pszExtraInfo, m_CurrInfo.sz255ExtraInfo, iMaxStrlen );
      }
   }

  // Now read the audio samples, depending on the file type:
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          nSamplePointsRead = WaveIO.ReadSamples( pData, Length );
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          // doesn't exist (Vorbis only delivers single-precision floats) :
          // nSamplePointsRead = VorbisIO.ReadSamples(pData, Length ); // only for FILES, not STREAMS
          nSamplePointsRead = -1;
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          nSamplePointsRead = -1;
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          nSamplePointsRead = WPH_Plugin.ReadSamples( pData, Length );
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  if( nSamplePointsRead > 0 )  // keep track of the AUDIO SAMPLE INDEX
   {  m_i64SamplePointsRead += nSamplePointsRead;
      if( m_dblCurrSampleRate > 0 )
       { dblTsample = (double)nSamplePointsRead / m_dblCurrSampleRate;
       }
      else
       { dblTsample = 0.0;
       }
      m_dblCurrTimeSinceRecStart += dblTsample;
      m_ldblCurrUnixDateAndTime   += dblTsample;
   }
  return nSamplePointsRead;
} // end C_AnyAudioFileIO::ReadSamples()


/***************************************************************************/
LONG C_AnyAudioFileIO::ReadSampleBlocks2(
     int first_channel_idx,  // channel_nr for 1st destination block
     int iMaxDestLen,    // length of destination blocks:  pFltDest1[0..iMaxDestLen-1]
     T_Float *pFltDest1, // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
     T_Float *pFltDest2, // 2nd destination block (~"RIGHT")
     T_ChunkInfo *pOutChunkInfo, // [out,optional] see ChunkInfo.h, may be a NULL pointer (!)
     char *pszExtraInfo, int iMaxStrlen ) // [out,optional] extra string with user data for wave.rd_info
  /* Reads some samples from a wave file and converts them to numbers (*)
   *                                         in the range of +- 1.0 .
   *  Used for all file-readers which don't have their own ReadSampleBlocks()-method.
   *
   * Input parameters:
   *   first_channel_idx: 0= begin with the first channel (LEFT if stereo recording)
   *               1= begin with the second channel (RIGHT if stereo, or Q if complex)
   *     The UPPER BYTE of 'first_channel_idx' can optionally define
   *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
   *    (first_channel_idx>>8) : 0x00 or 0x01 = only ONE channel per destination block
   *                      0x02         = TWO channels per destination block
   *                                    (often used for complex values aka I/Q)
   *
   *   iMaxDestLen: Max number of FLOATs which can be stored in T_Float dest[0..iMaxDestLen-1]
   *               (this is NOT ALWAYS the same as the number of sampling points,
   *                for example if the output are I/Q samples, all placed in
   *                pFltDest1, there are only iMaxDestLen/2 complex sampling points).
   *   *pFltDestX = Pointers to destination arrays (one block per channel, sometimes "paired")
   * Return value:
   *      the NUMBER OF SINGLE FLOATING-POINT VALUES placed in each destination block,
   *                 (not necessarily the number of sample points!) ,
   *      or a negative value if an error occurred.
   * Note that returning less samples than you may expect is not an error
   * but indicates reaching the file's end !
   *   ( This convension later caused trouble when the input is not a FILE
   *     but a STREAM, see WPH_IO_Plugin.cpp ) .
   * (*) T_Float may be either SINGLE- or DOUBLE-precision float,
   *     defined in various headers, depending on SWI_FLOAT_PRECISION .
   */
{
  BYTE *temp_array;
  int  i;
  int  asize;
  int  n_sample_points_max, n_sample_points_read;
  int  iResult;
  int  iNumChannels;
  int  iBitsPerSample;
  int  iBytesPerSamplePoint;
  BOOL fTwoChannelsPerBlock;
  T_Float fltFactor;

   iNumChannels   = GetNrChannels();
   iBitsPerSample = GetBitsPerSample();
   fTwoChannelsPerBlock =  ((first_channel_idx & AUDIO_FILE_TWO_CHANNELS_PER_BLOCK) != 0)
                        && (iNumChannels>1);
   first_channel_idx &= 0x000F;  // remove all 'flags', leave only the start-channel number

   // ex: iBytesPerSamplePoint = (iNumChannels * iBitsPerSample +  7) / 8;
   // Not sure about this: would a 2-channel, 20 bit, stream use (2*20+7)/8=5
   //   or   2 * (int)((20+7)/8) = 2 * 3 = 6 bytes per sample group ??
   // In other words, will some bits 'share' one byte, and if, how ?
   iBytesPerSamplePoint = (iNumChannels * iBitsPerSample +  7) / 8;

   // Note: a "Point" consists of one or two channels (i.e. 4 bytes for STEREO 16-bit recording) .
   // Someone else decided to call this "wBlockAlign" somewhere.  We don't .
   if( iBytesPerSamplePoint < 1 )
    { return -1;
    }
   if(fTwoChannelsPerBlock)
    { n_sample_points_max = iMaxDestLen / 2; // here: TWO destination array indices per sample point
      // so if the caller can accept 4096 floating-point values in his array (pFltDest1),
      // he will only read 2048 sampling points in one call of this subroutine !
    }
   else
    { n_sample_points_max = iMaxDestLen;
    }

   asize = n_sample_points_max * iBytesPerSamplePoint;
   if (asize<=0)
     return -1;
   temp_array = new BYTE[asize];  // allocate temporary array
   if(temp_array==NULL)
     return -1;                   // NO LAZY RETURN AFTER THIS POINT !

   n_sample_points_read = ReadSamples( // Returns the NUMBER OF AUDIO SAMPLE POINTS!
            temp_array, /* temporary BYTE-array for conversion     */
            asize ,     /* size in BYTE, not "number of samples" !! */
            pOutChunkInfo, pszExtraInfo, iMaxStrlen );
   if (n_sample_points_read > n_sample_points_max)
       n_sample_points_read = n_sample_points_max;  // emergency limit, should never happen
   if  (iBitsPerSample <= 8)
    { /* up to 8 bits per sample, we use a BYTE-pointer for addressing. */
     BYTE *bp = temp_array;
     fltFactor = 1.0 / 127.0;
     if( (iNumChannels>1)  // Number of channels, 1=mono, 2=stereo
      && (first_channel_idx < iNumChannels) )
      { // first_channel_idx: 0=left, 1=right, ..
        bp += first_channel_idx;             // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_sample_points_read;   // TWO destination array indices per sample point
        for(i=0; i<n_sample_points_read/*!*/; i++ )
         { // up to 8 bits per sample, TWO SEPARATE destination blocks:
           *pFltDest1++ = ( (T_Float)(*bp)   - 127.0 ) * fltFactor;  // 1st channel in file ("I")
           *pFltDest1++ = ( (T_Float)(*bp+1) - 127.0 ) * fltFactor;  // 2nd channel in file ("Q")
           if( (pFltDest2!=NULL) && (iNumChannels>2) )
            { *pFltDest2++ = ( (T_Float)(*(bp+2))-127.0 ) * fltFactor; // 3rd channel in file ("I")
              *pFltDest2++ = ( (T_Float)(*(bp+3))-127.0 ) * fltFactor; // 4th channel in file ("Q")
            }
           bp += iBytesPerSamplePoint;
         }
      }
     else // ! fTwoChannelsPerBlock ..
      { iResult = n_sample_points_read;  // ONE destination array index per sample point
        if( (iNumChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<n_sample_points_read; ++i)
            { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
              *pFltDest1++ = ( (T_Float)(*bp) - 127.0 ) * fltFactor;
              bp += iBytesPerSamplePoint;
            }
         }
        else
         {  // TWO or more channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<n_sample_points_read; ++i)
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              *pFltDest1++ = ( (T_Float)(*bp)   - 127.0 ) * fltFactor;
              *pFltDest2++ = ( (T_Float)(*(bp+1))-127.0 ) * fltFactor;
              bp += iBytesPerSamplePoint;
            }
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
   } // end if (8-bit-samples)
  else
  if( (iBitsPerSample <= 16) && (iBytesPerSamplePoint>1) )
   { /* most likely 16 bits per sample, we use a 16bit-int-pointer for addressing. */
     short *pi16 = (short*)temp_array; // should be even address, so what ??
     fltFactor = 1.0 / 32767.0;
     if( (iNumChannels>1) // Number of channels, 1=mono, 2=stereo
      && (first_channel_idx < iNumChannels) )
      { // first_channel_idx: 0=left, 1=right, ..
        pi16 += first_channel_idx;     // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_sample_points_read;  // TWO destination array indices per sample point
        for(i=0; i<n_sample_points_read/*!*/; i++ )
         { // up to 8 bits per sample, TWO SEPARATE destination blocks:
             *pFltDest1++ = fltFactor * (T_Float)(*pi16);        // 1st channel ("I")
             *pFltDest1++ = fltFactor * (T_Float)(*(pi16+1));    // 2nd channel ("Q")
             if( (pFltDest2!=NULL) && (iNumChannels>2) )
              { *pFltDest2++ = fltFactor * (T_Float)(*(pi16+2)); // 3rd channel ("I")
                *pFltDest2++ = fltFactor * (T_Float)(*(pi16+3)); // 4th channel ("Q")
              }
             pi16 += (iBytesPerSamplePoint / 2); // pointer to 16-bit-values!!
         }
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_sample_points_read;  // ONE destination array index per sample point
        if( (iNumChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<n_sample_points_read; ++i)
            { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
              *pFltDest1++ = fltFactor * (T_Float)(*pi16);
              pi16 += (iBytesPerSamplePoint / 2); // pointer to 16-bit-values!!
            }
         }
        else
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<n_sample_points_read; ++i)
            { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
              *pFltDest1++ = fltFactor * (T_Float)(*pi16);
              *pFltDest2++ = fltFactor * (T_Float)(*(pi16+1));
              pi16 += (iBytesPerSamplePoint / 2); // pointer to 16-bit-values!!
            }
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
    } // end if (16-bit-samples)
   else
    { // cannot handle this sample format !!
     iResult = -1;  // error
    }

   delete[] temp_array;
   return iResult;   // Note: the result may be NEGATIVE if an error occurs .

} // C_AnyAudioFileIO::ReadSampleBlocks2(...)


/***************************************************************************/
LONG C_AnyAudioFileIO::ReadSampleBlocks(
        int first_channel_idx,    // channel_nr for 1st destination block,
         // plus flags from AudioFileDefs.h (like AUDIO_FILE_TWO_CHANNELS_PER_BLOCK)
        int iMaxDestLen,    // number of floats in pdblDest[0..iMaxDestLen-1]
        T_Float *pdblDest1, // 1st destination block (~"LEFT")
        T_Float *pdblDest2, // 2nd destination block (~"RIGHT") , may be NULL
        T_ChunkInfo *pChunkInfo, // additional results, see ChunkInfo.h, may be a NULL pointer (!)
        char *pszExtraInfo, int iMaxStrlen )
  /* Reads some samples from a wave file which has been opened for READING
   * and converts them to floating-point numbers .
   * Input parameters:
   *    first_channel_idx: 0= begin with the first channel (LEFT if stereo recording)
   *                1= begin with the second channel (RIGHT if stereo, or Q if complex)
   *       + AUDIO_FILE_TWO_CHANNELS_PER_BLOCK (bitflag in first_channel_idx)
   *                 = "read TWO channels per destination block" :
   *         The UPPER BYTE of 'first_channel_idx' can optionally define
   *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
   *    (first_channel_idx>>8) : 0x00 or 0x01 = only ONE channel per destination block
   *                      0x02         = TWO channels per destination block
   *                                    (often used for complex values aka I/Q)
   *
   *    iMaxDestLen: Max number of entries which can be stored in T_Float dest[0..iMaxDestLen-1]
   *    *dest: Pointer to destination array.
   * Returns the NUMBER OF AUDIO SAMPLES if successful,
   *         or a negative value if errors occurred.
   *
   * Note1: Returning less samples than you may expect is not an error
   *        but indicates that the file's end has been reached.
   */
{
  long nSamplePointsRead = 0; // the result may be NEGATIVE if an error occurs !
  double dblDiff, dblTsample;

  // if necessary and possible, read the next CHUNK INFO BLOCK
  //  (possibly from a separate file, or whatever) :
  if( LoadChunkInfoForSampleIndex( m_i64SamplePointsRead ) )
   {
     if( pChunkInfo!=NULL)
      { *pChunkInfo = m_CurrInfo.chunk_info;   // return chunk info from 'auxiliary' file
        // Make sure the various 'times' in pChunkInfo are valid:
        SetTimesInReturnedChunkInfo( pChunkInfo );
      }
     if( (pszExtraInfo!=NULL) && (iMaxStrlen>0) )
      { strncpy( pszExtraInfo, m_CurrInfo.sz255ExtraInfo, iMaxStrlen );
      }
   }
  else  // could not 'load' the chunk-info from the file (or aux-file),
   {    // so try to GENERATE a chunk-info as good as we can :
     if( pChunkInfo!=NULL)
      { ChunkInfo_CopyFromTo(  // Copies a T_ChunkInfo structure, version-safe (added 2011-06-2011)
           &m_CurrInfo.chunk_info, // [in] source chunk, MUST HAVE BEEN INITIALIZED when opening the file
            pChunkInfo,         // [out] destination chunk info, with the following 'expected' size:
            pChunkInfo->dwSizeOfStruct); // [in] expected size of pDstChunkInfo, measured in BYTES
        // 2015-12-09: Got here with m_CurrInfo.chunk_info.nChannelsPerSample = 1
        //             when reading a STEREO wave file. That's wrong. Fixed in C_AnyAudioFileIO::InOpen().
        // Because m_CurrInfo.chunk_info may be "much older"
        //  than the first audio sample here, adjust AT LEAST the timestamps
        //  in the chunk-info returned to the caller (don't modify m_CurrInfo.chunk_info here) :
        dblDiff = (double)m_i64SamplePointsRead - (double)m_CurrInfo.chunk_info.i64TotalSampleCounter;
        ChunkInfo_AdjustInfoForSampleIndexDiff( // adjust the TIMESTAMPS (most of all)
             &m_CurrInfo.chunk_info,     // [in] 'stored' chunk info
             dblDiff, // [in] sample index difference
                      // between the index requested by the caller
                      //        (for the first sample he has retrieved in this call)
                      //   minus the older *stored* chunk-info-block (above)
                      //                          read from a file, etc.
                      // See notes in the implementation why
                      // a FLOATING POINT number is used for this argument.
             pChunkInfo); // [out] 'adjusted' chunk info
      }
   } // end else < no 'chunk info' read from the input file >

  // Now read the AUDIO SAMPLES :
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          // 2015-02-20: Crashed in WaveIO.ReadSampleBlocks() when reading
          //  a decimated wave file with 250 floating point samples per second.
          // 2015-12-14: Got here with first_channel_idx = 512 (!) .
          //   That's the flag AUDIO_FILE_TWO_CHANNELS_PER_BLOCK = 0x200,
          //   defined and specified in C:\cbproj\SoundUtl\AudioFileDefs.h .
          nSamplePointsRead = WaveIO.ReadSampleBlocks( first_channel_idx, iMaxDestLen, pdblDest1, pdblDest2,
                                          pChunkInfo, pszExtraInfo, iMaxStrlen );
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          nSamplePointsRead = VorbisIO.ReadSampleBlocks(first_channel_idx, iMaxDestLen,
                                     pdblDest1, pdblDest2, pChunkInfo );
          // Note: VorbisFileIO.cpp MAY have much better timestamps
          //       (from an extra logical stream in the physical Ogg stream)
          //       which overwrites certain fields in pChunkInfo !
          //  But it may not have any valid timestamps at all,
          //       in that case the flag CHUNK_INFO_TIMESTAMPS_VALID
          //       will now be CLEARED in pChunkInfo->->dwValidityFlags .
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          nSamplePointsRead = TextIO.ReadSampleBlocks( first_channel_idx, iMaxDestLen, pdblDest1, pdblDest2);
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          nSamplePointsRead = ReadSampleBlocks2( first_channel_idx, iMaxDestLen,  pdblDest1, pdblDest2,
                                    pChunkInfo,pszExtraInfo,iMaxStrlen);
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  if( pChunkInfo != NULL )  // since 2010-12-26: Fill out some fields which MAY still be missing...
   { //    .. but only if not set by the specific reader (like CWaveIO.cpp) .
     // See definition of T_ChunkInfo in c:\cbproj\SoundUtl\ChunkInfo.h .
     //  Added 2010-12-26, because with some fields left 'to fate',
     //  the 'fast audio file analysis' in Spectrum Lab failed [sometimes] .
     if( m_dblCurrSampleRate > 0.0 )
      {  if(  (pChunkInfo->dblPrecSamplingRate < (0.9*m_dblCurrSampleRate) )
           || (pChunkInfo->dblPrecSamplingRate > (1.1*m_dblCurrSampleRate) ) )
          {  // The 'chunk info' has obviously not been initialized by the caller
             // -> set at least some of the struct members to meaningful values
             pChunkInfo->dblPrecSamplingRate = m_dblCurrSampleRate;
          }
      }
   } // end if( pChunkInfo != NULL )
  if( nSamplePointsRead > 0 )  // keep track of the AUDIO SAMPLE INDEX (for the aux file)
   {  m_i64SamplePointsRead += nSamplePointsRead;
      if( m_dblCurrSampleRate > 0 )
       { dblTsample = (double)nSamplePointsRead / m_dblCurrSampleRate;
       }
      else
       { dblTsample = 0.0;
       }
      m_dblCurrTimeSinceRecStart += dblTsample;
      m_ldblCurrUnixDateAndTime  += dblTsample;
   }

  return nSamplePointsRead;
} // C_AnyAudioFileIO::ReadSampleBlocks(...)


/***************************************************************************/
void C_AnyAudioFileIO::SetExtraHeaderStringForAuxFile( char *pszExtraColumnHeader )
  /* Call before OutOpen, if the 'auxiliary file' shall have
   * a user-defined 'header string' for the extra DATA COLUMNS .
   * This should (!) match the 'pszExtraInfo' in WriteSamples() later.
   */
{
  if( pszExtraColumnHeader != NULL )
   { strncpy( m_sz80ExtraColumnHeaderForAuxFile, pszExtraColumnHeader, 80 );
     m_sz80ExtraColumnHeaderForAuxFile[80] = '\0';
   }
  else
   { m_sz80ExtraColumnHeaderForAuxFile[0] = '\0';
   }
} // end C_AnyAudioFileIO::SetExtraHeaderStringForAuxFile()


/***************************************************************************/
BOOL C_AnyAudioFileIO::WriteHeader(
           T_ChunkInfo *pChunkInfo) // [in] precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
  /* Writes the 'header' for the audio file (usually a RIFF WAVE file).
   *  This was once done immediately in OutOpen(), but since 2011-02
   *  writing the header is postponed until the first chunk of samples
   *  is available, and a PRECISE(!) timestamp is present in pChunkInfo .
   *  All relevant information has already been set in OutOpen(),
   *  but not written to the file yet (by the time we get here).
   */
{
  // long i32NominalSR;
  char sz1kExtraInfo[1024];
  char sz255Temp[256];
  // char *pszExtraInfo = NULL;
  char *cp;


  if( pChunkInfo==NULL ) // chunk info is mandatory here (contains sample rate)
   { strcpy( m_cErrorString, "missing chunk info" );
     return FALSE;
   }
  ChunkInfoToString( sz1kExtraInfo, sizeof(sz1kExtraInfo)-1/*maxlen*/, pChunkInfo );
  if(    (m_dblInfoWriteInterval_sec>0)
      && ((m_iAudioFileOptions & AUDIO_FILE_OPTION_SAVE_EXTRA_INFO_IN_AUX_FILE) != 0 )
      && (!AudioFileSupportsInfoChannel(m_iAudioFileFormat)) )
   { // Must create an EXTRA file for the 'info channel' (GPS data, rx freq, ..)
     // Write TWO HEADER LINES describing the file, enough for experienced developers
     // to implement their own parsers for this files without further info :
     if( QFile_Create( &qfAuxFile, AudioFileToAuxFileName(m_sz511CurrFileName), 0 ) )
      { QFile_WriteString(&qfAuxFile, "; Spectrum Lab auxiliary data file for \"" );
        QFile_WriteString(&qfAuxFile, QFile_GetFilenameWithoutPath(m_sz511CurrFileName) );
        QFile_WriteString(&qfAuxFile, "\"\r\n");
        sprintf( sz255Temp,
           // 10 dig.  --13 digits-- ----10---- ----11----- ---7--- --6---
           ";sample_nr,unix_date_sec,latitude_d,longitude_d,h_m_asl,sp_kmh" );
        ReplaceCommasWithColumnSeparators( sz255Temp, (char)m_iColumnSeparatorChar );
        strcat( sz255Temp, m_sz80ExtraColumnHeaderForAuxFile ); // added 2010-07-28
        strcat( sz255Temp, "\r\n" );
        QFile_WriteString(&qfAuxFile, sz255Temp );
        // pszExtraInfo = sz1kExtraInfo;
        cp = sz1kExtraInfo + strlen(sz1kExtraInfo);
        if( cp < (sz1kExtraInfo + 1000) )
         { strcpy(cp," auxfile=1");  // mark the presence of an 'auxiliary file'
         }
      }
   } // end if < create an extra file for the 'info channel' ?
 switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.WriteHeader( pChunkInfo );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.WriteHeader( pChunkInfo );
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.WriteHeader( pChunkInfo );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not directly supported by SpecLab (requires plugin)
          return WPH_Plugin.WriteHeader( pChunkInfo );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }

 return FALSE;
} // C_AnyAudioFileIO::WriteHeader(..)


/***************************************************************************/
BOOL C_AnyAudioFileIO::OutOpen(
                char * pszFileName,    // [in] name of the audio file (to be written)
                int iAudioFileFormat,  // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
                int file_mode,         // AUDIO_FILE_MODE_OVERWRITE, AUDIO_FILE_MODE_APPEND,  etc
                int iAudioFileOptions, // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
                int bits_per_sample,   //  8,16,24=integer, 32 or 64=floating point ! (since 2015-01-08)
                int channels,
        // ex:  T_Float dblFileSampleRate, T_Float dblFeedingSampleRate, T_Float dblNcoFrequency
                T_ChunkInfo *pChunkInfo, // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
                double dblInfoWriteInterval_sec, // interval, in seconds, to add 'chunk info' blocks in the output. 0=off.
                       //  (can be configured in SpecLab under "Options".."GPS Receiver" )
                int    iColumnSeparatorChar )    // ' ' ',' ';' or '\t' (tab)
  /* Creates and Opens an audio file for WRITING audio samples.
   *         Returns TRUE on success and FALSE on any error.
   * Input parameters..
   *   iAudioFileFormat: required to solve the ambiguity of different "raw"
   *                     file formats, which all have the extension "*.dat",
   *                     which is rather meaningless but that's the way it is.
   *   file_mode:        defines whether to APPEND or OVERWRITE, if the file
   *                     with the specified name already exists .
   *                     Allowed (so far): AUDIO_FILE_MODE_APPEND
   *                                    or AUDIO_FILE_MODE_OVERWRITE .
   *   additional_info:  a null-terminated text string with additional info
   *                     which will be included in the exported file.
   *                     NULL if this feature is not used.
   *                     SpecLab uses this to save (GPS-)date, (GPS-)time,
   *                     precise sample rate, possibly the name of a (GPS-)log,
   *                     and other receiver settings when the recording started.
   *   bits_per_sample:  8,16,24=integer, 32 or 64=floating point ! (since 2015-01-08)
   *   channels:         1=mono, 2=stereo recording etc
   *   xx SampleRate:    number of samples per second.
   */
{
  long i32NominalSR;
  // char sz255Temp[256];
  // char *cp;

  CloseFile(NULL);  // close old file even if we cannot detect the new file type
  memset( m_cErrorString, 0, sizeof(m_cErrorString) );

  if( pChunkInfo==NULL ) // chunk info is mandatory here (contains sample rate)
   { strcpy( m_cErrorString, "missing chunk info" );
     return FALSE;
   }
  // Modified 2015-01-08:  bits_per_sample:  8,16,24=integer, 32 or 64=floating point !
  if( channels<1 || channels>8 || bits_per_sample<8 || bits_per_sample>64 || ((bits_per_sample&7)!=0) )
   { strcpy( m_cErrorString, "output-options not supported" );
     return FALSE;
   }
  m_fMustWriteHeader     = TRUE;     // flag for the 1st call of C_AnyAudioFileIO::WriteSamplesXYZ()
  m_iChannelsPerSample   = channels; // usually 1 (mono) or 2 (stereo)
  m_iBytesPerSampleGroup = channels * ((bits_per_sample+7)/8); // (#channels * #bits/ADC_value) / 8
  // ex: m_iBytesPerSampleGroup = channels * (bits_per_sample+7)/8;
  // replaced 2011-02-10, result may be wrong for exotic formats !
  m_dblInitialSampleRate = m_dblCurrSampleRate = pChunkInfo->dblPrecSamplingRate;
  i32NominalSR = GetNominalSampleRate( m_dblInitialSampleRate );  // pretty useless for DECIMATED files !

  m_iPosUpdateCounter_WR  = -1; // notequal pChunkInfo->iPosUpdateCounter
  m_i64SamplePointsWritten = 0;
  m_dblInfoWriteInterval_sec = dblInfoWriteInterval_sec;
  m_iColumnSeparatorChar  = iColumnSeparatorChar;   // ' ' ',' ';' or '\t' as column separator in 'auxiliary' files
  if(   m_iColumnSeparatorChar!=' ' && m_iColumnSeparatorChar!=','
     && m_iColumnSeparatorChar!=';' && m_iColumnSeparatorChar!='\t' )
   { m_iColumnSeparatorChar = ' ';
   }
  m_iAudioFileOptions = iAudioFileOptions; // special options like AUDIO_FILE_OPTION_SAVE_EXTRA_INFO_IN_AUX_FILE, etc
  if( bits_per_sample < 8 )
   {  bits_per_sample = 16;   // default value (if not 'properly' configured)
   }
  strncpy( m_sz511CurrFileName, pszFileName, 511); // since 2020-10, value returned by GetFileName2() ... KISS !
  m_iAudioFileFormat = INET_GuessAudioFileTypeFromName(pszFileName,iAudioFileFormat);
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.OutOpen(pszFileName,m_iAudioFileFormat,
                         file_mode,
                         iAudioFileOptions, // AUDIO_FILE_OPTION_NORMAL etc [AudioFileDefs.h]
                         bits_per_sample,   // 8,16,24=integer, 32 or 64=floating point !
                         channels,
                         pChunkInfo );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          VorbisIO.SetParameterByID( AUDIO_FILE_PARAM__FILE_FORMAT, m_iAudioFileFormat );
          return VorbisIO.OutOpen( // Opens an output file, or audio stream .
                      pszFileName, // char *file_name
                iAudioFileOptions, // int  options,
                         channels, // int channels,
                     i32NominalSR, // int nominal_sample_rate like 44100 or 48000 [Hz]
                              0.5, // float base_quality,  -0.1 to 1.0 (lo to hi); 0.5 = good start
            TIM_GetCurrentUnixDateAndTime()); // double dblStartTime; start of recording for the file header

     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.OutOpen(pszFileName,file_mode,iAudioFileOptions,
                         ATIO_FILE_FORMAT_COOL_EDIT,
                         bits_per_sample,   // 8,16,24=integer, 32 or 64=floating point ! (here, most likely ignored)
                         channels,
                         pChunkInfo->dblPrecSamplingRate );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.OutOpen(pszFileName,
                         bits_per_sample, // 8,16,24=integer, 32 or 64=floating point !
                         channels,
                         pChunkInfo->dblPrecSamplingRate );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
 return FALSE;
} // C_AnyAudioFileIO::OutOpen(..)


/***************************************************************************/
LONG C_AnyAudioFileIO::WriteSamples(  // low-level "raw" output ... avoid !
       BYTE* pData,                   // (use WriteSamples_Float() instead)
       LONG Length, // [in] size of pData in BYTES (see Note below)
       T_ChunkInfo *pChunkInfo, // << precise sample rate, 'radio frequency', date and time, etc
       char * pszExtraInfo ) // extra 'info' (usually 'data') added to info chunk
  /* Writes some samples in 'raw' form (without any conversion)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLES written,
   *       or a NEGATIVE value if there was an error.
   * Note: "Length" is the size of the caller's buffer capacity in BYTES.
   *       The actual number of AUDIO SAMPLES will depend on the sample
   *       format.
   *       For 16-bit mono recordings (which is the only supported format
   *       up 'til now), the number of samples will only be half the "Length"
   *       ( actually sizeof(SHORT) * NrOfSamplePoints * NrOfChannelsPerPoint )
   */
{
  unsigned long dwSamplePoints = Length / m_iBytesPerSampleGroup;


  // Also emit a new GPS position to the file ? (since 2010-07-07)
  if( pChunkInfo != NULL )
   {
     if( m_fMustWriteHeader ) // flag for the 1st call of C_AnyAudioFileIO::WriteSamplesXYZ()
      {
        WriteHeader( pChunkInfo );  // 'postponed' writing of header (for precise timestamp, etc)
        m_fMustWriteHeader = FALSE; // "done" (regardless if successful or not)
      }
     if(  (m_iPosUpdateCounter_WR != pChunkInfo->gps.iUpdateCounter )
        &&( m_dblInfoWriteInterval_sec != 0 )
        &&( qfAuxFile.fOpened )    )
      {
        WriteChunkInfo( m_i64SamplePointsWritten/*index*/, pChunkInfo, pszExtraInfo );
        m_iPosUpdateCounter_WR = pChunkInfo->gps.iUpdateCounter;
      } // end if < periodically emit 'position info' (with GPS data, etc) ?
   } // end if( pChunkInfo != NULL )


  m_i64SamplePointsWritten += dwSamplePoints; // used as key into the extra 'info' file

  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          // discarded / now private: return WaveIO.WriteSamples(pData, Length );
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          // discarded / impossible : the Ogg/Vorbis encoder requires floating point values
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.WriteSamples((SHORT*)pData, Length / sizeof(SHORT) );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.WriteSamples( pData, Length );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
 return -1;

} // end C_AnyAudioFileIO::WriteSamples(..)

/***************************************************************************/
LONG C_AnyAudioFileIO::WriteSamples_Float( // app-layer output, using NORMALIZED floats
       T_Float *pfltSrc,     // [in] caller's source buffer, 32-bit floats
       T_ChunkInfo *pChunkInfo, // [in] number of samples, channels, precise sample rate, 'radio frequency', date and time, etc
          // btw, T_ChunkInfo is defined in c:\cbproj\SoundUtl\ChunkInfo.h .
       char * pszExtraInfo ) // [in] extra 'info' (usually 'data') added to info chunk
  /* Writes some samples in the 'normalized' floating point format (-1.0 ... +1.0)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLE POINTS written,
   *       or a NEGATIVE value if there was an error.
   */
{
  if( pChunkInfo == NULL )
   { return 0;
   }
  unsigned long dwSamplePoints = pChunkInfo->dwNrOfSamplePoints;

  if( m_fMustWriteHeader ) // flag for the 1st call of C_AnyAudioFileIO::WriteSamplesXYZ()
   {
     WriteHeader( pChunkInfo );  // 'postponed' writing of header (for precise timestamp, etc)
     m_fMustWriteHeader = FALSE; // "done" (regardless if successful or not)
   }

  // Also emit a new GPS position to the file ? (since 2010-07-07)
  if(  (m_iPosUpdateCounter_WR != pChunkInfo->gps.iUpdateCounter )
        &&( m_dblInfoWriteInterval_sec != 0 )
        &&( qfAuxFile.fOpened )    )
   {
     WriteChunkInfo( m_i64SamplePointsWritten/*index*/, pChunkInfo, pszExtraInfo );
     m_iPosUpdateCounter_WR = pChunkInfo->gps.iUpdateCounter;
   } // end if < periodically emit 'position info' (with GPS data, etc) ?


  m_i64SamplePointsWritten += dwSamplePoints; // used as key into the extra 'info' file

  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.WriteSamples_Float( pfltSrc, pChunkInfo, pszExtraInfo );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.WriteSamples_Float( pfltSrc, pChunkInfo, pszExtraInfo );
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.WriteSamples_Float(  pfltSrc, pChunkInfo, pszExtraInfo );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.WriteSamples_Float( pfltSrc, pChunkInfo, pszExtraInfo );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
 return -1;

} // end C_AnyAudioFileIO::WriteSamples_Float(...)


/***************************************************************************/
void C_AnyAudioFileIO::WriteChunkInfo(  // ... usually into an extra file for GPS & Co
        unsigned long dwSampleIndex,  // [in] audio sample index (to precisely sync with audio file)
        T_ChunkInfo *pChunkInfo,      // [in] date and time, GPS info, ..
        char * pszExtraInfo )         // [in] extra info from user, see "wave.info"
{
  char sz255[256];
  char *cp;
  cp = sz255;
  cp[255] = 0;

  if( ! qfAuxFile.fOpened )
   { return;
   }


  // printf format strings tested with DL4YHF's CalcEd ... use fixed widths below .
  // Test file saved as chunk_info_format_test.txt in the SL directory .

  // 1st column : audio sample index (for easy sync with the AUDIO FILE)
  sprintf(cp, "%010lu", (unsigned long)dwSampleIndex );    // 10 digits ok for two hours at 192kSamples/sec
  cp += strlen(cp);

  // 2nd column : "GPS time and date in UNIX format" (from GPS receiver, NOT ADJUSTED by anything)
  sprintf(cp, ",%013.2lf", pChunkInfo->gps.dblDateAndTime ); // 13 digits total, ok for > 100 years,
                                                            //  2 digits after the comma, ok for 10 ms .
  cp += strlen(cp);

  // 3rd, 4th, 5th, 6th column : geographic position (latitude, longitude, meters above sea leavel, GPS velocity)
  sprintf(cp, ",%+010.6lf,%+011.6lf,%+07.1lf,%+06.1lf",
       pChunkInfo->gps.dblLat_deg, // geographic latitude in degrees, positive = NORTH
                                   // sign+2 digits before the decimal, 6 digits after the decimal
                // ( max 10000 km ~~ 90 ,   resolution = 1 * 0.000001 ~~ 0.11 meters )
       pChunkInfo->gps.dblLon_deg, // geographic longitude in degrees, positive = EAST
                                   // sign+2 digits before the decimal, 6 digits after the decimal
                // ( max 20000 km ~~ 180,   resolution in the same region as lat..)
       pChunkInfo->gps.dbl_mASL  , // meters above sea level (from GPS receiver)
       pChunkInfo->gps.dblVel_kmh); // GPS velocity in kilometers per hour
  cp += strlen(cp);

  // 7th, optional, column : precise, possibly 'continuously calibrated' sampling rate
  if( (pChunkInfo->dblPrecSamplingRate>0) && (pChunkInfo->dblPrecSamplingRate != m_dblInitialSampleRate ) )
   { sprintf(cp, ",sr=%.12lg",  pChunkInfo->dblPrecSamplingRate );
     cp += strlen(cp);
   }

  // 8th, optional, column: 'Radio Frequency' (HF tuner frequency offset or "VFO")
  if( pChunkInfo->dblRadioFrequency != 0.0)
   { sprintf(cp, ",rf=%.12lf", pChunkInfo->dblRadioFrequency );
     cp += strlen(cp);
   }

  if( (pszExtraInfo!=NULL) && (cp-sz255)<200 )
   { if( pszExtraInfo[0] > 0 )
      { // *cp++ = ',';  // removed 2010-07-28, because "wr_info_header" ALSO contains the separator 
        strncpy( cp, pszExtraInfo, 250-(cp-sz255) );
        cp += strlen(cp);
      }
   }

  sprintf( cp, "\r\n" );

  ReplaceCommasWithColumnSeparators( sz255, (char)m_iColumnSeparatorChar );

  QFile_WriteString(&qfAuxFile, sz255);

} // end C_AnyAudioFileIO::WriteChunkInfo()

/***************************************************************************/
static BOOL SkipColumnSeparator( char **ppc )
{
  switch( **ppc )
   {
     case ' ':
     case ',':
     case ';':
     case '\t':
        ++*ppc;
        return TRUE;
     default:
        return FALSE;
   }
} // end SkipColumnSeparator()

/***************************************************************************/
BOOL C_AnyAudioFileIO::LoadChunkInfoForSampleIndex(
         unsigned long dwSampleIndex ) // [in] audio-sample-index for which to retrieve the 'info'
  // Reads the 'Chunk Info' (with GPS data and others) for a given audio sample
  //       from the input file, using limited buffering in RAM .
  //   This 'Info' usually contains the precise sample rate, radio tuning frequency,
  //    GPS position data, and possibly some user-defined data string .
  // Return values:
  //    TRUE = ok, the requested info is in m_CurrInfo + m_sz255CurrExtraInfo,
  //    FALSE= problem, there is no 'info' available for this sample point
  //           (not even "close" to the requested index) .
{
  char sz255[256];
  char *cp;
  int  line_length, token_index;
  BOOL ok,done;

  if( ! qfAuxFile.fOpened ) // there is no 'auxiliary file' to read the chunk-info from !
   { return FALSE;
   }
  // Does it make sense to read the 'next' info entry at all ?
  // There may be TWO entries in RAM, already read from the file:
  //  m_CurrInfo = the 'current' item, already read from the file,
  //                    and used up to now .
  //  m_NextInfo = the 'next' item, also already loaded from the file,
  //                    to retrieve the audio sample index already .
  // These 'INFO' items are only contained in the file every few seconds .
  // We consider the CURRENT chunk info valid until the begin of the NEXT :
  //
  // Diagram #1 :
  //                 0         10        20        30
  //  Sample Index : |.........|...I.....|.........|....
  //
  //  Info in RAM :  [CurrentInfo]   [NextInfo]      ...
  //   ..valid for   |<------------>||<----------->?|
  //
  if(  (dwSampleIndex >= m_CurrInfo.dwSampleIndex )
    && (dwSampleIndex <  m_NextInfo.dwSampleIndex )
    &&  m_CurrInfo.fValid && m_NextInfo.fValid )  // 'chunk infos' valid at all ?
   { // the 'current info' is still up-to-date for this sample,
     // no need to do anything (avoid file accesses, etc) !
     return TRUE;
   }
  // Arrived here, we know that 'm_CurrInfo' doesn't match the requested
  //  audio sample index (usually it will be TOO OLD, but after 'rewinding' the file
  //  it may also be TOO NEW) .  m_NextInfo must be reloaded from the file
  //  in any case, because we need it to determine the 'valid index range' (end)
  //  of m_CurrInfo - see diagram #1 .
  // Try to read the next INFO item from the file :
  ok = FALSE;
  while( !ok )  // may have to repeat this if the line doesn't contain DATA,
   {            // or if the entry isn't the one we're looking for yet .
     // Copy 'Next' to 'Current' before reading a new 'Next' from the file:
     m_CurrInfo = m_NextInfo;
     m_NextInfo.fValid = FALSE;
     // Read the next TEXT LINE from the auxiliary file:
     line_length = QFile_ReadLine( &qfAuxFile, sz255, 255 );
     if( line_length < 0 )  // end of file ?
      {  return FALSE;      // out of luck !
      }
     // Try to parse the text line. If it's a valid "line with data",
     //  it begins with a digit (for the 1st colum, the audio sample index) .
     // Note: A compatible (but simplified) variant of this parser
     //   was copied from AudioFileIO.cpp to c:\cbproj\MapData\GpsTracks.c ,
     //   to check Spectrum Lab's "auxiliary logfiles" in DL4YHF's
     //   RDF-Calculator (which contains a very crude mapping application,
     //   too) .
     if( sz255[0]>='0' && sz255[0]<='9' )
      { cp = sz255;
        // 1st column : audio sample index (for easy sync with the AUDIO FILE)
        m_NextInfo.dwSampleIndex = (DWORD)QFile_ParseInteger( &cp, 11/*ndigits*/, 10/*radix*/, 0/*deflt*/ );
        m_NextInfo.chunk_info.i64TotalSampleCounter = m_NextInfo.dwSampleIndex;
        ok = (SkipColumnSeparator(&cp) > 0 );

        // 2nd column : "GPS time and date in UNIX format" (from GPS receiver, NOT ADJUSTED by anything)
        m_NextInfo.chunk_info.gps.dblDateAndTime = QFile_ParseDFloat( &cp, 0.0/*Default*/ );
        ok &= (SkipColumnSeparator(&cp) > 0 );

        // 3rd column : geographic position : latitude in degrees
        m_NextInfo.chunk_info.gps.dblLat_deg = QFile_ParseDFloat( &cp, C_CHUNK_INFO_LAT_INVALID );
        ok &= (SkipColumnSeparator(&cp) > 0 );

        // 4th column : geographic position : longitude in degrees
        m_NextInfo.chunk_info.gps.dblLon_deg = QFile_ParseDFloat( &cp, C_CHUNK_INFO_LON_INVALID );
        ok &= (SkipColumnSeparator(&cp) > 0 );

        // 5th column : meters above sea leavel
        m_NextInfo.chunk_info.gps.dbl_mASL = QFile_ParseDFloat( &cp, 0.0/*Default*/ );
        ok &= (SkipColumnSeparator(&cp) > 0 );

        // 6th column : GPS velocity
        m_NextInfo.chunk_info.gps.dblVel_kmh = QFile_ParseDFloat( &cp, 0.0/*Default*/ );
        // Don't skip the following space character here !
        if( ok )
         { ++m_iPosUpdateCounter_RD;
           m_NextInfo.chunk_info.gps.iUpdateCounter = m_iPosUpdateCounter_RD;
           m_NextInfo.fValid = TRUE;

           // There may be some other, OPTIONAL columns to parse:
           done = FALSE;
           while( (!done) && (SkipColumnSeparator(&cp)>0) )  // is there something else ?
            {
              token_index = QFile_SkipStringFromList( &cp, "sr=\0rf=\0\0" );
              switch(token_index)
               { case 0:  // 7th, optional, column: precise, possibly 'continuously calibrated' sampling rate
                    m_NextInfo.chunk_info.dblPrecSamplingRate = QFile_ParseDFloat( &cp, 0.0/*Default*/ );
                    break;
                 case 1:  // 8th, optional, column: 'Radio Frequency' (HF tuner frequency offset or "VFO")
                    m_NextInfo.chunk_info.dblRadioFrequency = QFile_ParseDFloat( &cp, 0.0/*Default*/ );
                    break;
                 // ... add other, new tokens here ...

                 default: // must be the user-defined 'Extra Info' string, copy it, up to the end of the line
                    strncpy( m_NextInfo.sz255ExtraInfo, cp, 254);
                    m_NextInfo.sz255ExtraInfo[254] = 0;
                    done = TRUE;
                    break;
               } // end switch(token_index)
            } // end while( ! done ... )
           if( ! m_CurrInfo.fValid )
            { ok = FALSE;  // continue reading the file
            }
           else // 'CurrentInfo' is valid, but ...
            { // Is m_CurrentInfo the item we've been looking for ?
              // Look at diagram #1 again :
              //
              //                 0         10        20        30
              //  Sample Index : |.........|.........|.........|..
              //                  (A)   (B)            (C)
              //  Info in RAM : .... [CurrentInfo]   [NextInfo]
              //   valid range       |<------------>||<----------->?|
              if( dwSampleIndex < (DWORD)m_CurrInfo.dwSampleIndex )
               { // case (A) : must obviously REWIND the file (towards the begin)
                 // .. not implemented yet !
    //           if( m_fRewoundForAuxFile ...
    //            { ....
    //            }
                 ok = TRUE;   // ok (even though the info is 'too new',
                              // or there was no entry written for index zero
                              // which may happen if the GPS had a "late start"
                              // and the file-writer decided NOT to emit anything
                              // at the begin of the recording)
               }
              else if( dwSampleIndex < (DWORD)m_NextInfo.dwSampleIndex )
               { // (B) the requested sample index indeed belongs to the 'current' entry.
                 ok = TRUE;   // success (will leave the file-reading loop)
               }
              else
               { // (C) must continue the file-reading loop
                 ok = FALSE;
               }
            } // end else < CurrInfo valid >
         } // end if < parsing ok >
      } // end if( sz255[0]>='0' && sz255[0]<='9' )
   } // end while ( ! ok )
  return ok;  // -> TRUE=ok, FALSE=not ok (end of file)
} // end C_AnyAudioFileIO::LoadChunkInfoForSampleIndex()




/************************************************************************/
// Some 'Get' - and 'Set' - functions for the AUDIO FILE class ..

/************************************************************************/
long C_AnyAudioFileIO::GetParameterByID( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h.
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example GetVTFlags() -> GetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS) .
{
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetParameterByID( iAudioFileParam );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetParameterByID( iAudioFileParam );
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetParameterByID( iAudioFileParam );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.GetParameterByID( iAudioFileParam );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  return -1;
} // end C_AnyAudioFileIO::GetParameterByID()


//---------------------------------------------------------------------------
double C_AnyAudioFileIO::GetParameterByID_Double(
         int iAudioFileParam) // [in] e.g. AUDIO_FILE_PARAM_LAST_TIMESTAMP, etc (see AudioFileDefs.h)
  // Almost the same as C_AnyAudioFileIO::GetParameterByID(), but for 64-bit "double".
  // May return C_SND_MATH_INVALID_FLOAT_VALUE when not supported.
{
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN:
          break;
     case AUDIO_FILE_FORMAT_RAW    :
     case AUDIO_FILE_FORMAT_RIFF   :
     case AUDIO_FILE_FORMAT_SETI_1 :
          return WaveIO.GetParameterByID_Double( iAudioFileParam );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetParameterByID_Double( iAudioFileParam );
     case AUDIO_FILE_FORMAT_TEXT   :
          return TextIO.GetParameterByID_Double( iAudioFileParam );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  // Do we still support Winamp plugins ? (guess not)
     case AUDIO_FILE_FORMAT_PLUGIN :
     case AUDIO_FILE_FORMAT_MP3    :
          return WPH_Plugin.GetParameterByID_Double( iAudioFileParam );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  return C_SND_MATH_INVALID_FLOAT_VALUE;

} // end C_AnyAudioFileIO::GetParameterByID_Double()


/************************************************************************/
BOOL C_AnyAudioFileIO::SetParameterByID( int iAudioFileParam, long i32NewValue )
  // Returns TRUE if the new value could be set, otherwise FALSE.
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example SetVTFlags(iVTFlags) -> SetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS,..) .
{
  if( iAudioFileParam==AUDIO_FILE_PARAM__FILE_FORMAT )  // replaces SetFileFormat()
   { // special case, sets m_iAudioFileFormat() !
     if( m_iAudioFileFormat != i32NewValue )
      { // close any OLD file before switching to the new format:
        CloseFile( "to switch format"/*pszWhy*/ );
      } // if < different audio file format >
     m_iAudioFileFormat = i32NewValue;
   }
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.SetParameterByID( iAudioFileParam, i32NewValue );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.SetParameterByID( iAudioFileParam, i32NewValue );
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.SetParameterByID( iAudioFileParam, i32NewValue );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.SetParameterByID( iAudioFileParam, i32NewValue );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }

  // Arrived here ? The requested parameter is not supported.
  return FALSE;
} // end C_AnyAudioFileIO::SetParameterByID()

//---------------------------------------------------------------------------
BOOL C_AnyAudioFileIO::SetParameterByID_Double(
         int iAudioFileParam, // [in] e.g. AUDIO_FILE_PARAM_LAST_TIMESTAMP, etc (see AudioFileDefs.h)
         double dblNewValue ) // [in] new value (to be set) as a 64-bit floating point number
  // Almost the same as C_AnyAudioFileIO::SetParameterByID(), but for 64-bit "double".
  // Returns TRUE if the specified parameter could be set, otherwise FALSE .
{
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN:
          break;
     case AUDIO_FILE_FORMAT_RAW    :
     case AUDIO_FILE_FORMAT_RIFF   :
     case AUDIO_FILE_FORMAT_SETI_1 :
          return WaveIO.SetParameterByID_Double( iAudioFileParam, dblNewValue );
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.SetParameterByID_Double( iAudioFileParam, dblNewValue );
     case AUDIO_FILE_FORMAT_TEXT   :
          return TextIO.SetParameterByID_Double( iAudioFileParam, dblNewValue );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  // Do we still support Winamp plugins ? (guess not)
     case AUDIO_FILE_FORMAT_PLUGIN :
     case AUDIO_FILE_FORMAT_MP3    :
          return WPH_Plugin.SetParameterByID_Double( iAudioFileParam, dblNewValue );
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  // Arrived here ? The requested parameter cannot be set.
  return FALSE;

} // end C_AnyAudioFileIO::SetParameterByID_Double()


/************************************************************************/
double C_AnyAudioFileIO::GetSampleRate(void)
  // Note: This is the FILE's sampling rate ,
  //       not necessarily the same as the "reader" would like to have !
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetSampleRate();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetSampleRate();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetSampleRate();
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.GetSampleRate();
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          return 0;
   }
}

//---------------------------------------------------------------------------
void   C_AnyAudioFileIO::SetSampleRate(double dblSampleRate)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          WaveIO.SetSampleRate(dblSampleRate);
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          VorbisIO.SetSampleRate(dblSampleRate);
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (4)
          TextIO.SetSampleRate(dblSampleRate);
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          WPH_Plugin.SetSampleRate(dblSampleRate);
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
}

//---------------------------------------------------------------------------
int    C_AnyAudioFileIO::GetNrChannels(void)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetNrChannels();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetNrChannels();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetNrChannels();
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.GetNrChannels();
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          return 0;
   }
}

//---------------------------------------------------------------------------
void  C_AnyAudioFileIO::SetNrChannels(int iNrChannels) // important to read RAW files
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          WaveIO.SetNrChannels(iNrChannels);
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          VorbisIO.SetNrChannels(iNrChannels);
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          WPH_Plugin.SetNrChannels(iNrChannels);
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
} // end C_AnyAudioFileIO::SetNrChannels()


//---------------------------------------------------------------------------
int    C_AnyAudioFileIO::GetBitsPerSample(void)
{  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetBitsPerSample();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetBitsPerSample(); // ex: 16 (fixed value)
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetBitsPerSample();
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.GetBitsPerSample();
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          return 0 ;
   }
}

//---------------------------------------------------------------------------
void   C_AnyAudioFileIO::SetDataType(int iSizeOfDataType, int iDataTypeFlags)
   // important to read RAW files; and similar files "without header" .
   // iSizeOfDataType : 0=unknown, 1=8 bit, 2=16 bit, 3=24 bit, 4=32 bit
   // iDataTypeFlags  : AUDIO_FILE_DATA_TYPE_UNSIGNED, AUDIO_FILE_DATA_TYPE_FLOAT, ..
   //                   (defined in AudioFileDefs.h)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          WaveIO.SetDataType(iSizeOfDataType,iDataTypeFlags);
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          break;    // trying to set the 'data type' here doesn't make sense
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          WPH_Plugin.SetDataType(iSizeOfDataType,iDataTypeFlags);
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
} // end C_AnyAudioFileIO::SetDataType()

//---------------------------------------------------------------------------
void C_AnyAudioFileIO::GetDataType(
        int *piSizeOfDataType, // [out] 0=unknown, 1=BYTEs, 2=short INTs, 4=floats, 8=double's
        int *piDataTypeFlags)  // [out] data type flags, see AudioFileDefs.h : AUDIO_FILE_DATA_TYPE_SIGNED_I / ..UNSIGNED / ..FLOAT / ..MOTOROLA
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          WaveIO.GetDataType(piSizeOfDataType, piDataTypeFlags);
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          VorbisIO.GetDataType(piSizeOfDataType, piDataTypeFlags);
          break;    // trying to retrieve the 'data type' here doesn't make sense
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          WPH_Plugin.GetDataType(piSizeOfDataType,piDataTypeFlags);
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
} // end C_AnyAudioFileIO::GetDataType()


//---------------------------------------------------------------------------
void C_AnyAudioFileIO::SetGain( double dblGainFactor )
  // Actually only used to analyse 'raw' files with FLOATING POINT NUMBERS,
  // because not all of those files use a scaling range of -1.0 ... +1.0  !
  // Example: To analyse a 'raw' file with 32-bit floating point numbers
  //          ranging from -1000.0 to +1000.0, set the gain to 1.0/1000
  //          *BEFORE* starting to read samples from it .
  //  In Spectrum Lab, this happens in TSpectrumLab::OpenInputFileAnalysis() .
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          WaveIO.SetGain( dblGainFactor );
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          break;    // trying to retrieve the 'data type' here doesn't make sense
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
} // end C_AnyAudioFileIO::SetGain()

//---------------------------------------------------------------------------
double C_AnyAudioFileIO::GetGain( void )
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetGain();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          break;    // trying to retrieve the 'gain' here doesn't make sense
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          break;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          break;
   }
  return 1.0;   // if not especially supported, assume UNITY GAIN (when reading audio file)
} // end C_AnyAudioFileIO::GetGain()


//---------------------------------------------------------------------------
double C_AnyAudioFileIO::GetFrequencyOffset(void) // "radio-" minus "baseband"-frequency
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetFrequencyOffset();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetFrequencyOffset();  // vorbis-encoded I/Q files ? maybe some day...
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetFrequencyOffset();
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return 0;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          return 0 ;
   }
}

//---------------------------------------------------------------------------
BOOL C_AnyAudioFileIO::SetFrequencyOffset(double dblFrequencyOffset)
{ // Details in C_WaveIO::GetFrequencyOffset() !
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.SetFrequencyOffset(dblFrequencyOffset);
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.SetFrequencyOffset(dblFrequencyOffset);  // vorbis-encoded I/Q files ? maybe some day...
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.SetFrequencyOffset(dblFrequencyOffset);
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return 0;
#endif // SWI_WINAMP_PLUGINS_SUPPORTED
     default:
          return 0 ;
   }
}

//---------------------------------------------------------------------------
double C_AnyAudioFileIO::GetCurrentRecordingTime(void)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetCurrentRecordingTime();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetCurrentRecordingTime();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetCurrentRecordingTime();
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.GetCurrentRecordingTime();
#endif
     default:
          return 0 ;

   }
}

//---------------------------------------------------------------------------
double C_AnyAudioFileIO::GetRecordingStartTime(void) // seconds elapsed since 00:00:00 GMT, January 1, 1970.
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return 0;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.GetRecordingStartTime();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.GetRecordingStartTime();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.GetRecordingStartTime();
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.GetRecordingStartTime();
#endif
     default:
          return 0 ;
   }
}

//---------------------------------------------------------------------------
void   C_AnyAudioFileIO::SetRecordingStartTime(
         double dblStartTime) // [in] "Unix" date-and-time (seconds since 00:00:00 GMT, January 1, 1970)
  // Also used during FILE ANALYSIS: Called from TSpectrumLab::OpenInputFileAnalysis()
{
  // Also let the 'date and time' in the 'chunk info' begin at the same date-and-time :
  m_CurrInfo.chunk_info.ldblUnixDateAndTime = dblStartTime;
  // Added 2015-03 : If the timestamp was set this way,
  //                 assume it's 'valid' and even 'precise' :
  if( m_CurrInfo.chunk_info.ldblUnixDateAndTime > 0.0 )
   {  m_CurrInfo.chunk_info.dwValidityFlags |= (CHUNK_INFO_TIMESTAMPS_VALID|CHUNK_INFO_TIMESTAMPS_PRECISE);
   }

  // Invoke the file-specific decoder; it may want to know the file's start-time, too:
  switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          WaveIO.SetRecordingStartTime(dblStartTime);
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          VorbisIO.SetRecordingStartTime(dblStartTime);
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          TextIO.SetRecordingStartTime(dblStartTime);
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          WPH_Plugin.SetRecordingStartTime(dblStartTime);
          break;
#endif
     default:
          break;
   }
}

//---------------------------------------------------------------------------
void C_AnyAudioFileIO::GetFileName(char *pszDest, int iMaxLen)
{
  if( (pszDest!=NULL) && (iMaxLen>0) )
   { pszDest[0] = '\0';   // "empty" result if there's no meaningful result
     switch(m_iAudioFileFormat)
      { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
             break;
        case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
        case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
        case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
             WaveIO.GetFileName(pszDest, iMaxLen);
             break;
        case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
        case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
        case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
             VorbisIO.GetFileName(pszDest, iMaxLen);
             break;
        case AUDIO_FILE_FORMAT_TEXT   : // (3)
             TextIO.GetFileName(pszDest, iMaxLen);
             break;
#     if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
        case AUDIO_FILE_FORMAT_PLUGIN : // (5)
        case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
             WPH_Plugin.GetFileName(pszDest, iMaxLen);
             break;
#     endif
        default:
             break;
      }
   }
}

//---------------------------------------------------------------------------
char  *C_AnyAudioFileIO::GetFileName2(void) // greatly simplified ..
{ return m_sz511CurrFileName; // <- same for "input"- and "output" files
}


//---------------------------------------------------------------------------
BOOL C_AnyAudioFileIO::IsStream( void )
  // Returns TRUE if the input is an AUDIO STREAM,
  //     or  FALSE if the input is an "ordinary audio file" .
  // Certain callers need to know this, because AUDIO STREAMS can only
  //  be read & analysed at their nominal speed, while AUDIO FILES
  //  (stored on the local harddisk) can be analysed faster .
  // Furthermore, data read from a STREAM may not be available TEMPORARILY,
  //  while retrieving "no samples" from a FILE means we reached the end .
{
  char sz255FileOrStreamName[256];
  char *cp, *cp2;

  GetFileName( sz255FileOrStreamName, 255/*iMaxLen*/ );
  cp = sz255FileOrStreamName;
  cp2 = sz255FileOrStreamName+5;   // "http://blablabla" ?
  while( (cp<cp2) && (*cp!=0) && (*cp!=':') )
   { ++cp;
   }
  if( cp[0]==':' && cp[1]=='/' && cp[2]=='/' )
   { // this may be a "protocol" (part of a URL or "URI" as some like to call it)
     return TRUE;   // this may be an audio stream, since it's definitely NOT a "local file"
   }
  else
   { // not sure what it is, but most likely NOT a "stream"  .. except:
     switch(m_iAudioFileFormat) // is it something 'special' ?
      {
        case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
        case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
        case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
           // this may be a stream, OR an local disk file :
           return VorbisIO.IsStreamButNoFile(); // TRUE=stream, FALSE=local file
        default:
           return FALSE; // not a STREAM but a FILE (which is, most likely, SEEKABLE)
      }
   }
} // end C_AnyAudioFileIO::IsStream()


//---------------------------------------------------------------------------
void C_AnyAudioFileIO::GetErrorString(char *pszDest, int iMaxLen)
{
  // If there's an error string from this wrapper, return it:
  if( m_cErrorString[0] > 0 )
   { strncpy(pszDest, m_cErrorString, iMaxLen);
   }
  else switch(m_iAudioFileFormat) // otherwise pass error from low-level audio file driver
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          strncpy(pszDest, "unknown file format", iMaxLen);
          break;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          strncpy(pszDest, WaveIO.m_cErrorString, iMaxLen);
          break;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          strncpy(pszDest, VorbisIO.GetLastErrorAsString(), iMaxLen);
          break;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          strncpy(pszDest, TextIO.m_cErrorString, iMaxLen);
          break;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          strncpy(pszDest, WPH_Plugin.m_cErrorString, iMaxLen);
          break;
#endif

     default:
          strncpy(pszDest, "unsupported file format", iMaxLen);
          break;
   }
}

BOOL   C_AnyAudioFileIO::IsOpenedForReading(void)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return FALSE;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.m_OpenedForReading;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.IsOpenedForReading();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.m_OpenedForReading;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
     case AUDIO_FILE_FORMAT_MP3    : // (8) not supported by SpecLab (requires plugin)
          return WPH_Plugin.m_OpenedForReading;
#endif
     default:
          return FALSE;
   }
}


BOOL   C_AnyAudioFileIO::IsOpenedForWriting(void)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return FALSE;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.m_OpenedForWriting;
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return VorbisIO.IsOpenedForWriting();
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TextIO.m_OpenedForWriting;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
          return WPH_Plugin.m_OpenedForWriting;
#endif
     default:
          return FALSE;
   }
}


BOOL   C_AnyAudioFileIO::AllParamsIncluded(void)
{ switch(m_iAudioFileFormat)
   { case AUDIO_FILE_FORMAT_UNKNOWN: // (0)
          return FALSE;
     case AUDIO_FILE_FORMAT_RAW    : // (1) any "raw" file format, usually *.dat
     case AUDIO_FILE_FORMAT_RIFF   : // (2) standard "RIFF wave audio" (*.wav)
     case AUDIO_FILE_FORMAT_SETI_1 : // (6) SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
          return WaveIO.AllParamsIncluded();
     case AUDIO_FILE_FORMAT_OGG    : // Ogg(container) / Vorbis(codec) (*.ogg)
     case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS: // uncompressed stream, already identified by T_StreamHeader structs. ALSO SUPPORTED BY VorbisFileIO.cpp (!)
     case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS: // uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
          return TRUE;
     case AUDIO_FILE_FORMAT_TEXT   : // (3)
          return TRUE;
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
     case AUDIO_FILE_FORMAT_PLUGIN : // (5)
          return TRUE;
#endif
     default:
          return FALSE;
   }
}



/***************************************************************************/
BOOL C_AnyAudioFileIO::CloseFile(
        char *pszWhy ) // [in, optional] reason for closing, may be NULL.
        // Added for the display on the 'Analyse / play audio stream' panel,
        // which often contained 'closing internet socket' but didn't say WHY.
  /* Closes the WAV-file (if opened for READING exor WRITING.
   * Returns TRUE when a file HAS BEEN CLOSED; FALSE on error or 'nothing to close'.
   */
{
 BOOL closed_something;
  closed_something = WaveIO.CloseFile();
  closed_something |= TextIO.CloseFile();
  closed_something |= VorbisIO.CloseFile( pszWhy );
#if( SWI_WINAMP_PLUGINS_SUPPORTED )  /* Can we use winamp plugins  ? */
  closed_something |= WPH_Plugin.CloseFile();
#endif
  if( qfAuxFile.fOpened )
   { QFile_Close( &qfAuxFile );
     closed_something = TRUE;
   }
 return closed_something;
} // C_AnyAudioFileIO::CloseFile(..)



/* EOF <AudioFileIO.cpp> */

