/***************************************************************************/
/* File:  c:\cbproj\SoundUtl\AudioFileDefs.h                               */
/* Author: Wolfgang Buescher (DL4YHF)                                      */
/* Date:   2010-05-10                                                      */
/* Purpose:                                                                */
/*   Common definitions for various audio file readers / writers .         */
/*                                                                         */
/* Latest modifications:                                                   */
/*  2010-05-10 : Added file type AUDIO_FILE_FORMAT_SETI_1 .                */
/*-------------------------------------------------------------------------*/

#ifndef  _AUDIO_FILE_DEFS_H_
# define _AUDIO_FILE_DEFS_H_


/*----------- Constant Definitions ----------------------------------------*/

#define AUDIO_FILE_FORMAT_UNKNOWN 0  // for opening a file with auto-detection
#define AUDIO_FILE_FORMAT_RAW     1  // headerless 'raw' audio samples (no *.WAV - file !), also used by T_CFG_AudioServer.iFormatAndCompression
#define AUDIO_FILE_FORMAT_RIFF    2  // the common windoze - *.WAV-format
#define AUDIO_FILE_FORMAT_OGG     3  // Ogg/Vorbis audio files (ogg=container, vorbis=codec)
#define AUDIO_FILE_FORMAT_TEXT    4  // simple TEXT-file format, one sampling point per line
#define AUDIO_FILE_FORMAT_PLUGIN  5  // any other format, requiring a PLUGIN
#define AUDIO_FILE_FORMAT_SETI_1  6  // SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
#define AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS 7  // uncompressed stream, identified by T_StreamHeader structs. SUPPORTED BY VorbisFileIO.cpp (!)
#define AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS 8 // uncompressed stream with VLF-TX-tools VT_BLOCKs but *no* stream-headers
#define AUDIO_FILE_FORMAT_MP3     9  // not supported by SpecLab (requires plugin)
#define AUDIO_FILE_FORMAT_M3U     10  // not an AUDIO FILE FORMAT but a 'playlist' (text)
#define AUDIO_FILE_FORMAT_OTHER_WEB_STREAM 11 // anything else beginning with "http://" or "tcp://"

// Optional flags for the parameter 'channel_nr' in ReadSampleBlocks() :
#define AUDIO_FILE_TWO_CHANNELS_PER_BLOCK 0x0200  // read TWO channels per destination block

// Possible values for the parameter 'file_mode' in OutOpen() :
#define AUDIO_FILE_MODE_OVERWRITE 0
#define AUDIO_FILE_MODE_APPEND    1

#define AUDIO_FILE_MAX_SAMPLE_RATE 10000000 /* [Hz].. 10 MSample/sec for SETI */

// Bitflags for .iDataTypeFlags  ( in addition to .iSizeOfDataType ) :
#define AUDIO_FILE_DATA_TYPE_NORMAL   0    /* "normal" = none of the following: */
#define AUDIO_FILE_DATA_TYPE_SIGNED_I 0
#define AUDIO_FILE_DATA_TYPE_UNSIGNED 0x01 /* if NOT set, it's a SIGNED integer */
#define AUDIO_FILE_DATA_TYPE_FLOAT    0x02
#define AUDIO_FILE_DATA_TYPE_MOTOROLA 0x04 /* extra flag, bitwise OR-able, but AVOID BIG ENDIAN ! */

// Audio file options for OPENING / CREATING an audio file or stream.
//  Bitwise combineable flags for parameter iAudioFileOptions and C_AnyAudioFileIO.m_iAudioFileOptions .
//  Added to write Ogg Vorbis files with a timestamp channel (2010-06-06) .
#define AUDIO_FILE_OPTION_NORMAL                      0x0000
#define AUDIO_FILE_OPTION_TIMESTAMPS                  0x0001
#define AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER 0x0002
#define AUDIO_FILE_OPTION_SAVE_EXTRA_INFO_IN_AUX_FILE 0x0004
#define AUDIO_FILE_OPTION_NON_BLOCKING_CALLS          0x0008 // make e.g.


// States for recorders and stream receivers/transmitters; first used in VorbisFileIO.cpp .
//  When adding new constants here, also add them in c:\cbproj\SpecLab\AudioStreamOut_GUI.cpp
//  (where these states are displayed as human readable text, in a switch/case..)
#define AUDIO_FILE_IO_STATE_PASSIVE   0
#define AUDIO_FILE_IO_STATE_RECEIVING 1
#define AUDIO_FILE_IO_STATE_SENDING   2
#define AUDIO_FILE_IO_STATE_WAIT_CONN 3
#define AUDIO_FILE_IO_STATE_LOST_CONN 4
#define AUDIO_FILE_IO_STATE_NEED_TIMESTAMP 5 /* would like to send but cannot (lacking valid timestamps) */
#define AUDIO_FILE_IO_STATE_RECORDING 6  /* added 2015-11-25 to simplify the display in the 'circuit window' */

// Possible values for iAudioFileParam, in C_AnyAudioFileIO::GetParameterByID().
//  GetParameterByID() + SetParameterByID() is used for more 'exotic' parameters,
//  without the need to add more and more Get/Set functions in the interface.
//  A few parameters (like AUDIO_FILE_PARAM_LAST_TIMESTAMP, .._TIMESTAMP_LATENCY)
//  can be accessed as 64-bit 'double' (floating point) numbers,
//  using the new API functions  GetParameterByID_Double() + SetParameterByID_Double().

#define AUDIO_FILE_PARAM__FILE_FORMAT                1
  // Replacement for  BOOL SetFileFormat(int iAudioFileFormat)
  //             and   int GetFileFormat(void) .
  //  The parameter (i32ParamValue = iAudioFileFormat) must be one of the
  //  AUDIO_FILE_FORMAT_...-constants defined in AudioFileDefs.
  //  Sometimes, the audio file format must be set BEFORE opening a file !
  //  If the caller knows that the file/stream which is about to be opened
  //     is an UNCOMPRESSED stream ("identified by T_StreamHeader structs");
  //     or to write/send uncompressed files/streams instead of Ogg/Vorbis .
  //  [in] : AUDIO_FILE_FORMAT_OGG or AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS, etc.

#define AUDIO_FILE_PARAM__FILE_SIZE_KBYTE            2
  // Returns the current file size (if there is an opened file) in KBYTE.
  // Useful especially when logging audio samples  to keep track of the used disk space.
  // Returns a negative value on any error.

#define AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT 3
  // Retrieve (or, sometimes, modify) the NUMBER OF CHANNELS PER SAMPLE (!).
  // A "sample point" contains all channels, sampled at the very same time .  

#define AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX       4
  // Get, in rare cases also set the current sample index that will be used
  // on the next READ- or WRITE- access to the audio samples in the opened file.
  // Returns a negative value on any error.

#define AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES     5
  // Used to retrieve the total number of sampling points in the file.
  // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .

#define AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT     6
  // A "sample point" contains all channels, sampled at the very same time.
  // Example: Stereo Recording, 16 bits per (single) sample
  //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
  // Used to convert a "sample index" into a "file offset" .

#define AUDIO_FILE_PARAM__CONNECTION_LOST           10
  // Used to check the "connected" status for web streams.
  // In a few (rare) cases, can also be used to SET the "connect"-status
  // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )

#define AUDIO_FILE_PARAM__NUM_TIMESTAMPS            20
  // Retrieve the number of timestamps received, or sent, since stream started.

#define AUDIO_FILE_PARAM__VLF_TOOL_FLAGS            21
  // Get/set VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8, for non-compressed streams,
  //   compatible with Paul Nicholson's VLFRX tools.
  //   The VT-Flags themselves are defined in vlfrx-tools/vtlib.h, AND further below.

#define AUDIO_FILE_PARAM_WAITED_FOR_INPUT_MS        22
  // Number of MILLISECONDS(!) spent in e.g. ReadSampleBlocks() WAITING for input .

  // Some of the following 'parameter numbers' can also be accessed as 'double'
  //  values, using GetParameterByID_Double() + SetParameterByID_Double() :
#define AUDIO_FILE_PARAM_LAST_TIMESTAMP             24
  // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
  // Value doesn't change unless NEW timestamps arrive !
#define AUDIO_FILE_PARAM_TIMESTAMP_LATENCY          25
  // Stream latency, determined by comparing the local time and the received one,
  // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
#define AUDIO_FILE_PARAM_RX_BUFFER_USAGE_SECONDS    26
  // Number of *seconds* of audio samples, currently present in an RX-stream-buffer.
  // Added 2022-03 to check if a backslash is building up, as suspected in the
  // Lot Et Garonne VLF stream (F6ETU),




// Bit definitions for VT_BLOCK and VT_BUFFER flags field. Copied from vlfrx-tools/vtlib.h :
#ifndef VTFLAG_RELT
# define VTFLAG_RELT    (1<<0)   // Timestamps are relative, not absolute
# define VTFLAG_FLOAT4  (1<<1)   // 4 byte floats  (8 byte default)
# define VTFLAG_FLOAT8  0
# define VTFLAG_INT1    (2<<1)   // 1 byte signed integers
# define VTFLAG_INT2    (3<<1)   // 2 byte signed integers
# define VTFLAG_INT4    (4<<1)   // 4 byte signed integers
# define VTFLAG_FMTMASK  (VTFLAG_FLOAT4 | VTFLAG_INT1 | VTFLAG_INT2 | VTFLAG_INT4)   // Mask for format bits

#endif



#define AUDIO_FILE_IO_MAX_TIMEOUT_MS 2000 // max. time to spend in C_AnyAudioFileIO::InOpen() if the "file" is a WEB AUDIO STREAM

//---------------------------------------------------------------------------
// Type definitions. Used by AudioFileIO, VorbisFileIO, and possibly more .
//---------------------------------------------------------------------------
typedef int(*T_AudioFileInfoCallback)(
                    void *pvUserCallbackData, // [in] address of any user-speficed data
                    int iEventType, // one of the following (or maybe a bit combination?)
#                       define AUDIO_FILE_IO_EVENT_TYPE_INFO  0
#                       define AUDIO_FILE_IO_EVENT_TYPE_ERROR 1
#                       define AUDIO_FILE_IO_EVENT_TYPE_PARSE_RESPONSE 2
#                       define AUDIO_FILE_IO_EVENT_FLAG_SHOW_TIMESTAMP 8
                    char *cp,  // info string or HTTP response
                    int  i );  // progress percentage or numeric error code

typedef struct tStreamLogInfo
{
  char sz255LogFileName[256];  // Absolute path to the LOGFILE (not the webstream)
  char sz255StreamURL[256];    // URL of the audio stream (or similar resource).
                               //  May be using Spectrum Lab's pseudo-URL-notation
                               //  like "rawtcp://78.46.38.217:4801"
                               //  if the source isn't connectable via HTTP !
  int  iLogging;               // 0=not logging,  1=logging,  negative=error
  int  iLogFileFormat;         // AUDIO_FILE_FORMAT_..., AUDIO_FILE_FORMAT_UNKNOWN
  int  iLogFileSize_kByte;     // current size of the LOGFILE(!) in kByte



} T_StreamLogInfo;

#endif /* ndef _AUDIO_FILE_DEFS_H */

/* EOF < ?\SoundUtl\AudioFileDefs.h > */
