//---------------------------------------------------------------------------
// File  :  AM_FM_DeMod.cpp
// Date  :  2002-07-21   (yyyy-mm-dd)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//   Implementation of a class for modulating and demodulating AM + FM signals
//
// Revision history (yyyy-mm-dd):
//   2002-07-21  Written for initial testing purposes.... by DL4YHF
//---------------------------------------------------------------------------

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h>
#include <math.h>

#include <ChebCasc.h>   // cascaded Chebyshev lowpass filter of N-th order

#pragma hdrstop   // no precompiled headers after this point

#include "SoundTab.h"
#include "SoundMaths.h" // some math subroutines like ..CalculateAngle()
#include "AM_FM_DeMod.h"

// #pragma warn -8017
#pragma warn -8004  // .. is assigned a value that is never used (well..)
#pragma warn -8080  // .. is declared but never used (.. so what ?)


#ifndef  C_PI
 #define C_PI  (3.1415926535897932384626)
#endif


//**************************************************************************
//    Tables used in this module
//**************************************************************************

  // see SoundTab.cpp

  

//**************************************************************************
//    Global Variables
//**************************************************************************




//**************************************************************************
//    Routines (no class methods)
//**************************************************************************


/***************************************************************************/
void SndUtl_InitNCO( T_DeMod_NCO *pNCO,
                    double dblFCenter, double dblFSample )
{
  pNCO->dblFCenter = dblFCenter;
  pNCO->dblFSample = dblFSample;
  pNCO->dblPhase   = 0.0;
  if(pNCO->dblFSample>0)
        pNCO->dblPhzInc  = (double)SOUND_COS_TABLE_LEN
                         * (double)pNCO->dblFCenter / pNCO->dblFSample;
   else pNCO->dblPhzInc  = 0.0;
} // end SndUtl_InitNCO( )


/***************************************************************************/
void SndUtl_InitIIRLowpass(   // initializes a SINGLE (but long) IIR lowpass (*)
          double dblCutoffFreq,  // cutoff freq, normalized to sampling rate
          int    iNrOfCoeffs,    // = 2*(filter_order+1),   try 2 + 2*20
          T_AM_FM_DEMOD_FILTER *pFilter)
  // (*) such lowpasses may be unstable -
  //     see notes in the implementation of CAmFmDeModulator .
{
 BOOL fResult = TRUE;  // still "everything ok"
 int  iNrOfPoles, i;
 T_AM_FM_DEMOD_FILTER NewFilter;

  if(iNrOfCoeffs>AM_FM_DEMOD_MAX_FILTER_COEFFS)  // 1st limitation: from THIS module
     iNrOfCoeffs=AM_FM_DEMOD_MAX_FILTER_COEFFS;
  if(iNrOfCoeffs>CHEB_MAX_COEFFICIENTS)        // 2nd limitation: from CASCADED CHEBYSHEV CALCULATOR(!)
     iNrOfCoeffs=CHEB_MAX_COEFFICIENTS;

  iNrOfPoles = (iNrOfCoeffs-2) / 2;  // iNrOfPoles must be 2,4,...20
  if(iNrOfPoles & 1)
   --iNrOfPoles;               // should not happen, set breakpoint here
  if(iNrOfPoles < 2)
     iNrOfPoles = 2;
  if(iNrOfPoles > 20)
     iNrOfPoles = 20;

  for(i=0; i<AM_FM_DEMOD_MAX_FILTER_COEFFS; ++i)
   {
    NewFilter.dblCoeffs[i] = 0;  // originally [2 + 2*20]
    NewFilter.dblDelayZ[i] = 0;
   }

  CHEB_CalcCascadedLowOrHighpass(
      iNrOfPoles,       //  number of poles,  2,4,...20
      dblCutoffFreq,    // cutoff freq divided by sampling rate
      FALSE,            // false=lowpass,   true=highpass
      10.0 ,            //  percent ripple,   0 to 29
      NULL ); // [out,optional] coeffs for a chain of biquad stages

  // A.S.A.P. copy the results from the filter designer into the own structure
  // because other callers may overwrite the global vars in ChebCasc.c soon .
  NewFilter.iNrOfCoeffs = 2 + 2*iNrOfPoles;
  for(i=0; i<=iNrOfPoles; ++i)
   {
    NewFilter.dblCoeffs[2*i]   = +CHEB_result_A[i];  // ALPHA coeffs
    NewFilter.dblCoeffs[2*i+1] = -CHEB_result_B[i];  // BETA  coeffs
   }
  *pFilter = NewFilter;  // rapidly copy new coeffs, the filter may still be RUNNING
} // end SndUtl_InitIIRLowpass()



/***************************************************************************/
BOOL SndUtl_RunThroughIIRFilter(
          T_Float *pdbl_Input,
          T_Float *pdbl_Output,
          int    iNumberOfSamples,
          T_AM_FM_DEMOD_FILTER *pFilter)
    /* Let a block of samples run through a digital IIR-filter.
     * Used for the frequency converter ("Weaver-method")
     *          and the phase demodulator (after multiplying).
     */
{
  int    i,j,jmax,k;
  T_DEMOD_Float d,x,y;
  T_DEMOD_Float *pCoeff, *pZMemory;
  BOOL    fResult = TRUE;

  jmax = pFilter->iNrOfCoeffs/2 - 1;

  // algorithm for all non-cascaded IIR - filters  with <N>-th order:
  for(i=0; i<iNumberOfSamples; ++i)
   {
     x = pdbl_Input[i];  // filter input = "X"
     pCoeff = pFilter->dblCoeffs;  // alpha0, beta0, alpha1, beta1, .....
     pZMemory=pFilter->dblDelayZ;
     try
      {
       y = x * (*(pCoeff++))/*alpha0*/ + pZMemory[0];
#if ( T_DEMOD_FLOAT_IS_DOUBLE ) // limiter for DOUBLE precision :
       if(y>1e100)
        { y=1e100;  // some VERY CRUDE solution to avoid overflow.. set breakpoint here !
          fResult = FALSE;
        }
#else
       if(y>1e30)
        { y=1e30;   // some VERY CRUDE solution to avoid overflow.. set breakpoint here !
          fResult = FALSE;
        }
#endif
       pCoeff++;            // skip beta0, it should be zero anyway
       for(j=1; j<jmax; ++j)
          {
           // Note: DON'T CONTINUE EXECUTION if program stops in the next line..
           d =  x * (*(pCoeff++))   /* alpha1..alpha19 */
                  + pZMemory[j];
#        if(0) && ( T_DEMOD_FLOAT_IS_DOUBLE ) // limiter for DOUBLE precision :
           if(d>1e100 || d<-1e100) // crashed so completely that a RE-BOOT was necessary !!!!!!!!
#        else
           if(d>1e30 || d<-1e30)
#        endif
            {
              d=0; // CRASH - set breakpoint here !  (try-catch doesnt trap this)
              fResult = FALSE;
              pZMemory[j] = 0.0;   // this didn't help to prevent increasing oscillation amplitudes, so:
              for(k=0; k<jmax; ++k)
                  pZMemory[k] = 0.0;
            }
           pZMemory[j-1] = d - y * (*(pCoeff++)); /* beta1 ..beta19  */
          }
        pZMemory[j-1] = x * pCoeff[0]  /*alpha20*/
                      - y * pCoeff[1]; /*beta 20*/
        pdbl_Output[i] = y;
       }catch(...)
       {
        pdbl_Output[i] = x;  // crashed filter works like a bypass
        fResult = FALSE;
       }
   } // end for <all samples in the filter buffer>

  return fResult;
} // end SndUtl_RunThroughIIRFilter()

/***************************************************************************/
void SndUtl_Run1stOrderLowpass(
          T_Float *pFlt_Input, T_Float *pFlt_Output, int iNumberOfSamples,
          T_Float *pFlt_Accu,  // Storage for the last output sample between two calls
          double dblTimeConstant_sec, double dblSampleRate_Hz )
    /* Runs a block of samples through a "simple RC lowpass" (1st order)
     *  as used for a WIDE FM DE-EMPHASIS (in FM broadcasting).
     *
     * Parameter dblTimeConstant_sec :
     *   Use 50 usec for EUROPEAN FM broadcasters,
     *    or 75 usec for North American FM broadcasters.
     */
{
  int    i;
  // Principle, from Wikipedia:Low-pass_filter :
  // y[i] = alpha * x[i] + (1-alpha)*y[i-1]
  // with alpha = dt / (RC_time_constant + dt)
  double  dt = 1.0 / dblSampleRate_Hz; // -> sample_interval
  T_Float a = dt / ( dblTimeConstant_sec + dt );
  T_Float y = *pFlt_Accu;  // previous OUTPUT sample

  for(i=0; i<iNumberOfSamples; ++i)
   {
     // ex: y = a * pFlt_Input[i] + (1.0-a) * y; // refactored:
     y += a * (pFlt_Input[i] - y);
     pFlt_Output[i] = y;
   } // end for <all samples in the filter buffer>

  *pFlt_Accu = y;  // save the last output sample for the next call

} // end SndUtl_Run1stOrderLowpass()



//***************************************************************************
//  Implementation of methods for the CAmFmDeModulator   class
//***************************************************************************


//***************************************************************************
CAmFmDeModulator::CAmFmDeModulator()
  // constructor
{
  m_FltSamplingRate     = 0;
  m_FltNco2Phase        = 0;
  m_dblAfcTuningRangeHz = 1.0;
  memset( &m_IIRchain_I, 0, sizeof(T_FILTER_DATA) );
  memset( &m_IIRchain_Q, 0, sizeof(T_FILTER_DATA) );
} // end CAmFmDeModulator::CAmFmDeModulator()


//***************************************************************************
CAmFmDeModulator::~CAmFmDeModulator()
  // destructor, clean up may be required one day..
{
} // end CAmFmDeModulator::~CAmFmDeModulator() [destructor]



//***************************************************************************
BOOL CAmFmDeModulator::Init(
                     int iDeModType,    // SOUND_DEMOD_TYPE_xxx
                     int iOptions,      // DEMOD_OPTION_xxx (like AFC, etc)
                  double dblSampleRate, // often 11025, 22050 or 44100
                  double dblFCenter,    // Modulator or Demodulator Center Frequency [Hz]
                  double dblBandwidth,  // Modulator or Demodulator Bandwidth [Hz]
                  double dblModulatorFactor,    // relative modulation amplitude (0..1)
                  double dblModulatorCarrier )  // relative carrier amplitude (0..1)

  // Prepares operation of the AM/FM modulator/demodulator.
  //
{
 double d;

   SndUtl_InitNCO( &m_NCO, dblFCenter, dblSampleRate );
   m_FltNco2Phase = 0;
   m_FltSamplingRate=dblSampleRate;
   m_iDeModType   = iDeModType;        // SOUND_DEMOD_TYPE_xxx
   m_iDeModOptions= iOptions;
   m_FltBandwidth = dblBandwidth;      // for FM modulator: "max deviation from center freq"
   m_FltModFactor = dblModulatorFactor;            // internal modulation factor
   m_FltModCarrier= 32767.0 * dblModulatorCarrier; // internal carrier amplitude
   m_dblPreviousDemodPhase = 0.0;      // for phase- and frequency demodulator
   m_dblAfcTuningOffsetHz  = 0.0;
   m_iFreqDiscrState = 0;

   m_Filter_I.iNrOfCoeffs = 0;
   for(int i=0; i<AM_FM_DEMOD_MAX_FILTER_COEFFS;  ++i)
    { m_Filter_I.dblCoeffs[i] = 0;
      m_Filter_I.dblDelayZ[i] = 0;
    }
   m_Filter_Q = m_Filter_I;
   m_fltAfcLowpass4Angle = m_fltAfcLowpass4Freqcy = 0.0;
   if(dblSampleRate<1)
      dblSampleRate=1;

   switch(m_iDeModType)
    {   // special initialisation procs for certain modulator/demodulator types ?

      case SOUND_DEMOD_TYPE_PH_DEM :   // prepare PHASE demodulator
      case SOUND_DEMOD_TYPE_FR_DEM :   // prepare FREQUENCY demodulator
        {
         // Must cut off everything above 1.5 * f_center here,
         //   because at 2*f_center is a strong harmonic from the multiplier.
         //   So the higher the 'carrier' frequency, the better the phase meter.
         d = 1.5 * dblFCenter;
         // An additional limit of demodulator bandwidth may be desirable :
         if(m_FltBandwidth>1)
          { if(d > m_FltBandwidth)
               d = m_FltBandwidth;
          }
         d = d / dblSampleRate; // cutoff freq, normalized to sampling rate
         if(d>0.5)  d=0.5;
         // ex: if(d<0.03) d=0.03; // beware: edge freq < 0.05 sometimes made the filter CRASH(!)
         if(d<0.005) d=0.005;
         // 2011-02-22: for DCF77 demodulation with a soundcard / correlation tests,
         //             a lowpass with 2000 Hz corner frequency was required,
         //             i.e. d = 2000 / 192000 = ~~0.01
         SndUtl_InitIIRLowpass(
                d,  // cutoff freq, normalized to sampling rate
                    // (The lower the normalized cutoff frequency, the higher
                    //  the required filter order.  Something left to improve !)
                2 + 2*14,      // nr_of_coeffs = 2 +  2*filter_order
                &m_Filter_I ); // T_AM_FM_DEMOD_FILTER *pFilter
         // 2011-02-22: Because a single "long" IIR filter tends to oscillate,
         //             a chain of biquad stages was tried instead:
         CHEB_CalcCascadedLowOrHighpass(  // implemented in ChebCasc.c
             10,          // [in] number of poles,  2,4,...20
             d,           // [in] cutoff frequency,  0 to 0.5 [*f_sample]
             FALSE,       // [in] is_highpass, false=lowpass,  true=highpass
             10.0,        // [in] percent ripple,   0 to 29
             &m_IIRchain_I); // [out] T_FILTER_DATA data for a chain of biquads

         // design only one branch, use a copy with same coeffs in other branch:
         m_Filter_Q   = m_Filter_I;    // .. for the OLD filter (a single, "long", IIR)
         m_IIRchain_Q = m_IIRchain_I;  // .. for the NEW filter (a cascade of IIR biquads)
       } break; // end case SOUND_DEMOD_TYPE_PH_DEM

      case SOUND_DEMOD_TYPE_FREQ_SHIFTER :  // prepare FREQUENCY SHIFTER
       {
         // Frequency converter (up and down)
         //     Uses Donald K. Weaver's method for SSB generation.
         //     Produce the coeffs for two suitable low pass filters:
         SndUtl_InitIIRLowpass(
                0.25,          // cutoff freq, normalized to sampling rate
                2 + 2*16,      // = 2 +  2*filter_order
                &m_Filter_I ); // T_AM_FM_DEMOD_FILTER *pFilter
         // design only one branch, use copy with same coeffs in other branch:
         m_Filter_Q = m_Filter_I;
       } break; // end case SOUND_DEMOD_TYPE_FREQ_SHIFTER

      case SOUND_DEMOD_TYPE_AM_DEM :   // prepare AM demodulator , and ...
      default:
         d =  dblBandwidth / dblSampleRate;
         if(d>0.5)  d=0.5;
         if(d<0.05) d=0.05;
         SndUtl_InitIIRLowpass(
                d,  // cutoff freq, normalized to sampling rate
                    // (The lower the normalized cutoff frequency, the higher
                    //  the required filter order.  Something left to improve !)
                2 + 2*12,      // nr_of_coeffs = 2 +  2*filter_order
                &m_Filter_I ); // T_AM_FM_DEMOD_FILTER *pFilter
         // design only one branch, use copy with same coeffs in other branch:
         m_Filter_Q = m_Filter_I;

    } // end switch(m_iDeModType)

   return TRUE;
} // end CAmFmDeModulator::Init()


/***************************************************************************/
BOOL CAmFmDeModulator::ProcessSamples(
                  T_Float *pfltSource,  // main input (or "I" = in-phase component)
                  T_Float *pfltSourceQ, // optional quadrature component for INPUT (*)
                  T_Float *pfltDest,    // main output (or "I" = in-phase component)
                  T_Float *pfltDestQ,   // optional quadrature component for OUTPUT
                  T_ChunkInfo *pChunkInfo) // [in] chunk size, PRECISE sampling rate, recording time, etc
  // Runs a number of audio samples through a modulator/demodulator.
  // Note: pfltSource and pfltDest may -or may not- point to the same memory locations !
  // (*) pfltSourceQ=NULL indicates that the input is REAL (not COMPLEX) .
{
  double d;
  BOOL fResult = TRUE;
  int nSamples = pChunkInfo->dwNrOfSamplePoints; // see c:\cbproj\SoundUtl\ChunkInfo.h
  double Tsample = 1.0 / pChunkInfo->dblPrecSamplingRate; // sampling interval in seconds
  // Fr den PID-Regler in manchen Demodulatoren:
  double Kp = 0.0;  // proportionaler Anteil, P-contribution : Verstrkung Kp
  double Ki = 0.0;  // Integralanteil, I-contribution
                    //   aus analogem PID-Regler:  Ki = Kp / Tn (Tn=Nachstellzeit)
  double Kd = 0.0;  // Differentialanteil, D-contribution
                    //   aus analogem PID-Regler:  Kd = Kp * Tv (Tv=Vorhaltzeit)


  // Because the sampling rate may change slighly for each chunk of samples,
  // re-calculate the NCO's phase increment (which operates like a DDS) :
  m_NCO.dblFSample = pChunkInfo->dblPrecSamplingRate;
  if( m_iDeModOptions & DEMOD_OPTION_AFC )
   { m_NCO.dblPhzInc = (double)SOUND_COS_TABLE_LEN
                     * (double)( m_NCO.dblFCenter + m_dblAfcTuningOffsetHz )
                               / m_NCO.dblFSample;
   }
  else // similar, but WITHOUT AFC (automatic frequency control) :
   { m_NCO.dblPhzInc = (double)SOUND_COS_TABLE_LEN
                     * (double) m_NCO.dblFCenter / m_NCO.dblFSample;
   }



  switch( m_iDeModType )
   {
    case SOUND_DEMOD_TYPE_AM_MOD :
         while(nSamples--)
          {
           // Run the NCO (numerical controlled oscillator)
           m_NCO.dblPhase += m_NCO.dblPhzInc;
           while(m_NCO.dblPhase >=(double)SOUND_COS_TABLE_LEN) // "while", not "if" !!
                 m_NCO.dblPhase -=(double)SOUND_COS_TABLE_LEN; // table index wrap
           d = SoundTab_fltCosTable[(int)m_NCO.dblPhase];

           // calculate AM modulator output:
           //  full input swing = full output swing = +-32767
           //  dblNCOoutput = +- 1
           *pfltDest++ = ( (*pfltSource++ * m_FltModFactor) + m_FltModCarrier) * d;
          } // end while(nSamples--)
         return fResult;

    case SOUND_DEMOD_TYPE_AM_DEM :
         // Crude AM demodulator (at least, with a lowpass AFTER the demodulator !)
         { double d;
           int    i;
           for(i=0; i<nSamples; ++i)
            {
             // Get the next audio sample from the source block, and "rectify" it
             d = pfltSource[i];
             if(d<0) d=-d;
             pfltDest[i] = d;
            } // end while(nSamples--)

           // Run the rectified samples through a lowpass filter
           fResult &= SndUtl_RunThroughIIRFilter(  // still in cache :)
                pfltDest,        // T_Float *pdbl_Input,
                pfltDest,        // T_Float *pdbl_Output,
                nSamples, // int    iNumberOfSamples,
                &m_Filter_I );   // coeffs and delay memory
         } return fResult;  // end case SOUND_DEMOD_TYPE_AM_DEM

    case SOUND_DEMOD_TYPE_FR_MOD : {  // true FREQUENCY modulator
         double dblFreqToPhzInc;
         double dblModFact2 = m_FltModFactor * m_FltBandwidth * 0.5 / 32768.0;
         if ( m_NCO.dblFSample > 0 )
            dblFreqToPhzInc = (double)SOUND_COS_TABLE_LEN / m_NCO.dblFSample;
         while(nSamples--)
          {
           // Reprogram the NCO for the instantaneous frequency (for EVERY sample)..
           // Actual freq = center frequency + (baseband_signal * ModFactor).
           // Only POSITIVE frequencies are allowed here (because of the sine TABLE).
           d = m_NCO.dblFCenter + *pfltSource++ * dblModFact2;
           if(d<0) d=0;
           // convert a FREQUENCY into a phase increment for the oscillator..
           d *= dblFreqToPhzInc;

           // Run the NCO (numerical controlled oscillator)
           m_NCO.dblPhase += d;
           while(m_NCO.dblPhase >=(double)SOUND_COS_TABLE_LEN)
                 m_NCO.dblPhase -=(double)SOUND_COS_TABLE_LEN; // table index wrap
           while(m_NCO.dblPhase < 0.0)
                 m_NCO.dblPhase +=(double)SOUND_COS_TABLE_LEN;

           *pfltDest++ = SoundTab_fltCosTable[(int)m_NCO.dblPhase] * m_FltModCarrier;
          } // end while(nSamples--)
         } return fResult;

    case SOUND_DEMOD_TYPE_PH_MOD : {    // TRUE PHASE modulator
         double dblModToPhzOffs = 0;

         // Calc conversion factor from 'modulating signal' to 'phase offset':
         if ( m_NCO.dblFSample > 0 )
           dblModToPhzOffs = (double)SOUND_COS_TABLE_LEN * m_FltModFactor * 0.5 / ( 32768.0);
         while(nSamples--)
          {
           // Run the NCO (numerical controlled oscillator)
           m_NCO.dblPhase += m_NCO.dblPhzInc;
           while(m_NCO.dblPhase >=(double)SOUND_COS_TABLE_LEN)
                 m_NCO.dblPhase -=(double)SOUND_COS_TABLE_LEN; // table index wrap
           while(m_NCO.dblPhase < 0.0)
                 m_NCO.dblPhase +=(double)SOUND_COS_TABLE_LEN;

           // Add a phase offset from the modulating signal to the carrier signal:
           d =  m_NCO.dblPhase + *pfltSource++ * dblModToPhzOffs;
           while(d >= (double)SOUND_COS_TABLE_LEN)
                 d -= (double)SOUND_COS_TABLE_LEN;
           while(d < 0)    // avoid MODULO stuff here !
                 d += (double)SOUND_COS_TABLE_LEN;

           *pfltDest++ = SoundTab_fltCosTable[(int)d] * m_FltModCarrier;
          } // end while(nSamples--)
         } return fResult;  // end case PHASE MODULATOR


    case SOUND_DEMOD_TYPE_PH_DEM :    // PHASE demodulator
    case SOUND_DEMOD_TYPE_FR_DEM :    // FREQUENCY demodulator
        {
         // Principles for phase demodulators...
         //   A - Crude phase detector with a single quadrature oscillator
         //       Mix the input with a quadrature oscillator to "zero hertz".
         //       The phase is measured at the complex output.
         //       Optionally, a low-pass filtered DC component from the output
         //                 can be used for automatic frequency control.
         //       Advantage:    fast and simple.
         //       Disadvantage: phase output is a triangle function,
         //                     amplitude=180,  frequency = 2 * f_center .
         //                     (-> a lowpass with sharp cutoff is required,
         //                         which would limit the detector's bandwidth
         //                         at the expense of higher CPU load)
         //
         // An FM demodulator is basically a phase demodulator..
         //       The frequency of the input signal is computed by taking
         //       the derivative of the I/Q phase. Taking the derivative
         //       in digital is just computing the delta between
         //       two consecutive samples, with some care for the case
         //       of these two samples are not of the same sign.
         //
         int iSineTableOffset = (int) (0.499 + 3.0 * (double)SOUND_COS_TABLE_LEN / 4.0);
         int iCosTableIndex;
         double angle, d, dblMeasuredFrequencyOffset;
         T_Float fltI, fltQ;

         // Fr den PID-Regler in manchen Demodulatoren:
         Kp;  // proportionaler Anteil, P-contribution
         Ki;  // Integralanteil, I-contribution
                     //   aus analogem PID-Regler:  Ki = Kp / Tn (Tn=Nachstellzeit)
         Kd;  // Differentialanteil, D-contribution
                     //   aus analogem PID-Regler:  Kd = Kp * Tv (Tv=Vorhaltzeit)


         while(nSamples--)
          {
           // Run the NCO (numerical controlled oscillator).
           // Note: the phase increment must always be positive in this case.
           m_NCO.dblPhase += m_NCO.dblPhzInc;   // unit: index into cosine table!
           while(m_NCO.dblPhase >=(double)SOUND_COS_TABLE_LEN)
                 m_NCO.dblPhase -=(double)SOUND_COS_TABLE_LEN; // table index wrap
           while(m_NCO.dblPhase < 0.0)
                 m_NCO.dblPhase +=(double)SOUND_COS_TABLE_LEN;
           iCosTableIndex = (int)m_NCO.dblPhase;
           fltI =  SoundTab_fltCosTable[iCosTableIndex];
           fltQ = -SoundTab_fltCosTable[(iCosTableIndex+iSineTableOffset) % SOUND_COS_TABLE_LEN];

           // multiply input signal with the complex LO signal.
           //  The input may be real (with pfltSourceQ=NULL) or complex !
           if( pfltSourceQ == NULL )
            { // REAL input :
              d = *pfltSource++;
              fltI *= d;
              fltQ *= d;
            }
           else
            { // COMPLEX input :  (a + jb) * (c + jd) = a*c-b*d + j ( a*d + b*c )
              //   here: a = *pfltSource, b=*pfltSourceQ,
              //         c = dbl_I      , d= dbl_Q  (complex oscillator signal)
              d    = *pfltSource * fltI - *pfltSourceQ * fltQ; // don't modify dbl_I yet..
              fltQ = *pfltSource * fltQ + *pfltSourceQ * fltI;
              fltI = d;
              ++pfltSource;
              ++pfltSourceQ;
            }


           // run both branches of the I/Q stream through a lowpass filters:
#if(0)  // old filter: a single, "long" IIR, which was sometimes unstable
           fResult &= SndUtl_RunThroughIIRFilter(
                &fltI,        // T_Float *pdbl_Input,
                &fltI,        // T_Float *pdbl_Output,
                1,             // int    iNumberOfSamples,
                &m_Filter_I ); // coeffs and delay memory
           fResult &= SndUtl_RunThroughIIRFilter(  // still in cache :)
                &fltQ,        // T_Float *pdbl_Input,
                &fltQ,        // T_Float *pdbl_Output,
                1,             // int    iNumberOfSamples,
                &m_Filter_Q ); // coeffs and delay memory
#else   // since 2011-02-23, use the CASCADED IIR filter designed by ChebCasc.c :
           SndMat_RunThroughFilter(
                &fltI,        // T_Float *pdbl_Input,
                &fltI,        // T_Float *pdbl_Output,
                1,             // int    iNumberOfSamples,
                &m_IIRchain_I ); // T_FILTER_DATA *filter; coeffs and delay memory
           SndMat_RunThroughFilter(
                &fltQ,        // T_Float *pdbl_Input,
                &fltQ,        // T_Float *pdbl_Output,
                1,             // int    iNumberOfSamples,
                &m_IIRchain_Q ); // T_FILTER_DATA *filter; coeffs and delay memory
#endif

           // Calculate the angle from the complex product ("analytic signal"):
           angle = SndMat_CalculateAngle( fltI, fltQ );   // -> result -pi..+pi

           // Demodulate the FREQUENCY. We may also need this for the phase
           //  demodulator with AFC to steer the NCO frequency.
           // omega = delta_phi / delta_t;   frequency = omega/(2*pi) .
           // First calculate d="delta_phi".
           d = (angle - m_dblPreviousCalculatedAngle) / (2*C_PI); // -> 1.0 = "one circle"
           m_dblPreviousCalculatedAngle = angle;
           // Handle phase wraps for the FREQUENCY measurement.
           // A phase jump of a half circle or more must be a "wrap".
           if(d>=0.5)
              d-=1;
           else if(d<=-0.5)
              d+=1;
           // Calculate the momentary frequency offset (in Hertz).
           // If the input signal's frequency is higher than the NCO's,
           // the result shall be positive (tested, ok) .
           // The following result is also used as 'error signal' for the AFC.
           dblMeasuredFrequencyOffset = d * pChunkInfo->dblPrecSamplingRate;

           // Similar for the PHASE: deal with phase-wraparounds .
           //   The output of the discriminator shall not jump by almost 360
           //   when crossing the +180 or -180 border. To achieve that,
           //   the phase is INTERNALLY registered with more than just one circle.
           // First calculate d="delta_phi" from the previously measured angle,
           // and decide if a 'full circle' must be subtracted or added.
           d = angle - m_dblPreviousDemodPhase; // -> delta phi in "circles"
           m_dblPreviousDemodPhase = angle;
           // The difference between the current and the previous calculated angle
           // must be much less than 180 (due to the lowpass filtering), so:
           if(d>=C_PI)
              d -= 2.0*C_PI;
           else if(d<=-C_PI)
              d += 2.0*C_PI;
           // Arrived here: d = small angle difference between this and the previous reading.
           m_dblAccumulatedAngle += d;  // accumulated, NON-WRAPPING, measured angle
           d = m_dblAccumulatedAngle;
           // Don't let the sum of "measured" and "accumulated" angle grow too large:
           if( d > 3.0*C_PI )
            {  m_dblAccumulatedAngle -= 2.0*C_PI;
               d = m_dblAccumulatedAngle;
            }
           else
           if( d < -3.0*C_PI )
            {  m_dblAccumulatedAngle += 2.0*C_PI;
               d = m_dblAccumulatedAngle;
            }
           angle = d;   // angle with the accumulated angles, 'rarely' wrapping

           // Crude "AFC" .  Uses a PID controller to keep the output angle
           //                at zero, by *slightly* adjusting the NCO frequency.
           // For details on the PID algorithm used here, see
           //    [RN_RT]: http://www.rn-wissen.de/index.php/Regelungstechnik .
           //  (Sehr praxisnah und lesenswert, auch ohne hohe Mathematik...)
           //  w = Sollwert
           //          hier: idealer Phasenwinkel vom Diskriminator = 0
           //  x = Istwert (Regelgre),
           //          hier: gemessener Phasenwinkel (angle)
           //  e = Regeldifferenz (nicht "Regelabweichung" !)
           //  y = Stellgre, Ausgang des Reglers, Eingang der Strecke .
           //          hier: Korrektur fr Oszillator (NCO), m_dblAfcTuningOffsetHz
           //  z = Strgre, hier z.B. Fehler in der Abtastrate,
           //          undefinierte Phasenlage beim Start (muss ausgeregelt werden)
           if( m_iDeModOptions & DEMOD_OPTION_AFC )
            { double dblE = angle;  // error signal for the controller, here: ANGLE (in radians)
              double y;

              // PID-Algorithmus nach [RN_RT] "Umsetzung in Code nach der Differenzengleichung":
              m_dblAfcPidEsum += dblE;    // Integration I-Anteil
              if( m_dblAfcPidEsum > C_PI ) // avoid wind-up...
                  m_dblAfcPidEsum = C_PI;
              if( m_dblAfcPidEsum < -C_PI )
                  m_dblAfcPidEsum = -C_PI;
              y = Kp * dblE                            // P-contribution
                + Ki * m_dblAfcPidEsum*Tsample         // I-contribution
                + Kd * (dblE-m_dblAfcPidEold)/Tsample; // D-contribution
              m_dblAfcPidEold = dblE;



              // Reprogram the NCO with the AFC'ed center frequency:
              m_NCO.dblPhzInc = (double)SOUND_COS_TABLE_LEN
                     * (double)( m_NCO.dblFCenter + m_dblAfcTuningOffsetHz )
                               / m_NCO.dblFSample;
            } // end if < AFC enabled >

           // depending on what kind of demodulator we have...
           if(m_iDeModType == SOUND_DEMOD_TYPE_PH_DEM)
            {
              d = angle * 180.0 / C_PI;
              // The result is a PHASE DIFFERENCE ranging from -180 to +180.
            }
           else // no PHASE- but FREQUENCY DEMODULATION...
            { //
              d = dblMeasuredFrequencyOffset; /* without m_dblAfcTuningOffsetHz ! */
                  // (if the input frequency doesn't change, the output of
                  //  the *AFC'ed* FM demodulator shall be ZERO after some time)
            } // end if <freq demodulator>

           if( m_iDeModOptions & DEMOD_OPTION_BYPASS_DISCR )
            {    // test mode : bypass the discriminator (to test the lowpass filter)
              *pfltDest++ = fltI;
            }
           else  // normal operation: copy the DISCRIMINATOR OUTPUT to the destination block
            {
              *pfltDest++ = d;
            }

          } // end while(nSamples--)
         } return fResult;  // end case PHASE DEMODULATOR + FREQUENCY DEMODULATOR


    case SOUND_DEMOD_TYPE_FREQ_SHIFTER : {    // frequency converter (up and down)
         // Principle
         //     Uses Donald K. Weaver's method for SSB generation,
         //     using four multipliers,
         //           two equal but simple low-pass filters,
         //           and two quadrature oscillators.
         //     The first oscillator + multiplier pair shifts the frequency
         //         UP so the center frequency is at sr/4  (sr=sampling rate).
         //     Then, the signal is filtered with a simple lowpass (f_cutoff=sr/4).
         //     The second oscillator + multiplier pair shifts the frequency
         //         DOWN to a center frequency of zero.
         //     A good explanation for the Weaver method of SSB generation is at
         //       http://www.csounds.com/ezine/summer2000/processing/ ,
         //     locally saved as
         //       \Downloads\SignalProcessing\Weaver_from_Csound_Magazine.htm .
         //
         int iCosTableIndex;
         int iSinTableOffset1, iSinTableOffset2;
         double d, fltPhzInc1, fltPhzInc2;
         T_Float fltI, fltQ;
         int    i,j,jmax,k;
         double x,y;
         const T_Float *pCoeff;
         if(m_NCO.dblFSample<1)
            m_NCO.dblFSample=1;
         SoundTab_GetNcoParams( // the first oscillator for the Weaver block runs at sr/4:
              0.25, &iSinTableOffset1, &fltPhzInc1 );
         SoundTab_GetNcoParams( // the second oscillator for the Weaver block runs at sr/4 + fShift:
             -0.25 - m_NCO.dblFCenter/m_NCO.dblFSample, &iSinTableOffset2, &fltPhzInc2 );
         jmax = sizeof(SoundTab_HalfbandIIR20Lowpass) / (2*sizeof(T_Float))  - 1;
         while(nSamples--)
          {
           // Get the next audio sample from the source block
           d = *pfltSource++;

           // Multiply the input signal with the I/Q output from the 1st quadrature oscillator.
           // Note: the phase increment must always be positive in this case.
           m_NCO.dblPhase += fltPhzInc1;
           while(m_NCO.dblPhase >=(double)SOUND_COS_TABLE_LEN)
                 m_NCO.dblPhase -=(double)SOUND_COS_TABLE_LEN; // table index wrap
           while(m_NCO.dblPhase < 0.0)
                 m_NCO.dblPhase +=(double)SOUND_COS_TABLE_LEN;
           iCosTableIndex = (int)m_NCO.dblPhase;
           fltI =   d * SoundTab_fltCosTable[iCosTableIndex];
           fltQ = - d * SoundTab_fltCosTable[(iCosTableIndex+iSinTableOffset1) % SOUND_COS_TABLE_LEN];


           // Run the output of the 1st multiplier pair through a lowpass filter
           // with a cutoff frequency near sr/4 .
           // The csound article says we don't have to care about the phase here
           // as long as both filters are equal.
           // So let's use a sharp IIR filter, which results in a nice lower edge frequency.
           // Some notes:
           //    - without the lowpass, there is no sideband rejection at all
           //    - don't be fooled by the csound article which says
           //       > it only needs an ordinary lowpass filters of moderate order,
           //       > which can be designed with any standard method.
           //      In fact, a sharp edge at sr/4 is very important for good
           //      unwanted-sideband-rejection of low input frequencies.
           //      (for audio purposes, sideband rejection of a few dB's may be ok)
           // algorithm for all non-cascaded IIR - filters  with <N>-th order:
           // run both branches of the I/Q stream through lowpass filters
           fResult &= SndUtl_RunThroughIIRFilter(
                &fltI,        // T_Float *pdbl_Input,
                &fltI,        // T_Float *pdbl_Output,
                1,             // int    iNumberOfSamples,
                &m_Filter_I ); // coeffs and delay memory
           fResult &= SndUtl_RunThroughIIRFilter(  // still in cache :)
                &fltQ,        // T_Float *pdbl_Input,
                &fltQ,        // T_Float *pdbl_Output,
                1,             // int    iNumberOfSamples,
                &m_Filter_Q ); // coeffs and delay memory

#if(0) // filter test ?
           dbl_I = 0;
#else  // no filter test
           // Multiply the output of the lowpass filter with the second quadrature oscillator.
           // Run the 1st NCO (numerical controlled oscillator).
           // Note: the phase increment must always be positive in this case.
           m_FltNco2Phase += fltPhzInc2;
           while(m_FltNco2Phase >=(double)SOUND_COS_TABLE_LEN)
                 m_FltNco2Phase -=(double)SOUND_COS_TABLE_LEN; // table index wrap
           while(m_FltNco2Phase < 0.0)
                 m_FltNco2Phase +=(double)SOUND_COS_TABLE_LEN;
           iCosTableIndex = (int)m_FltNco2Phase;
           fltI *= SoundTab_fltCosTable[iCosTableIndex];
           fltQ *= SoundTab_fltCosTable[(iCosTableIndex+iSinTableOffset2) % SOUND_COS_TABLE_LEN];
#endif  // no filter test

           *pfltDest++ = fltI + fltQ;

          } // end while(nSamples--)
         } return fResult;  // end case FREQUENCY SHIFTER


    case SOUND_DEMOD_TYPE_OFF    :
    default:  // copy input into output (if different memory range)
         if( pfltSource != pfltDest )
          {
           while(nSamples--)
             *pfltDest++ = *pfltSource++;
          }
         return fResult;
   } // end switch( m_iDeModType )
} // end CAmFmDeModulator::ProcessSamples()
