/*
 *   main.c handlings for the DCF serial clock daemon.
 *
 *   Copyright 1997        quacks@paula.owl.de (Joerg Krause)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "dcf77.h"
#include <signal.h>

static void	InitSignalHandlers(void);
static void	PutPID(void);
static int	CheckPID(void);

char *port     = "/dev/cua0";	/* Default port for the clk */
int  cmos      = FALSE; 	/* TRUE == Update CMOS Clk (requires RTC Kernel) */
int  lastday    = 0;		/* Day of the last time update */
int  loglevel   = LEV_ERR_ONLY;	/* notification level */
int  rtc        = -1;		/* File descriptor for /dev/rtc */
int  fd         = -1;		/* File descriptor for /dev/cuaX **/
int  kill_other = FALSE;	/* try to kill already running dcf77d ? */
int  pidfile    = -1;


char *version  = "dcf77d 1.01";

int makedaemon(void)
{
  pid_t pid;

  if ( ( pid = fork()) < 0)
    return(-1);
  else
  {
    if (pid != 0)
      exit(0); /* Quitting parent process */
  }
  /* Child is running */
  
  setsid();

  chdir("/");

  umask(0);

  return(0);
}


/*
 * Pick up a numeric argument.  It must be nonnegative and in the given
 * range (except that a vmax of 0 means unlimited).
 */
static long numarg(char *meaning, long vmin, long vmax)
{
	char *p;
	long val;

	val = strtol(optarg, &p, 10);
	if (*p)
		errx(1, "illegal %s -- %s", meaning, optarg);
	if (val < vmin || (vmax && val > vmax))
		errx(1, "%s must be between %ld and %ld", meaning, vmin, vmax);
	return (val);
}

static void usage(char *progname)
{
  printf("%s (c) 1997 by Joerg Krause\n\n",version);
  printf("usage: %s [-k] [-h] [-c] [-p device] [-l loglevel]\n",progname);
}

int main(int argc, char *argv[])
{
  int ch;

  /* check if we're really invoked by root
  ** or a users with root-privilegs
  */
  if (getuid() != 0)
  {
    fprintf(stderr, "dcf77d: Sorry, must be root to execute this.\n");
    exit(1);
  }

  
  /* Scan thru the commandline */
  while ((ch = getopt(argc,argv, "khcl:p:")) != -1)
  {
    switch (ch)
    {
      case 'h':
	/* print help */
	usage(argv[0]);
	exit(0);
	break;
      case 'p':
	/* tty or cua device to use */
	port = optarg;
	break;
      case 'c':
	/* Set C-MOS time aswell */
	cmos = TRUE;
	break;
      case 'l':
	/* loglevel */
	loglevel = numarg("Log level: 0=Quiet, 1=Errors, 2=All,",0L,2L);
	break;
      case 'k':
	/* kill already running daemons */
	kill_other = TRUE;
	break;	
      default:
	usage(argv[0]);
	exit(1);
	break;
    }
  }

  /* check for running daemon */
  if (CheckPID() != 0)
  {
    fprintf(stderr, "dcf77d: Already running.\n");
    exit(1);
  }

  /* install sig-handler */
  InitSignalHandlers();

  /* Create a daemon */
  if (makedaemon() != 0)
  {
    fprintf(stderr,"Failed to create daemon process");
    exit(1);
  }

  /* put PID file */
  PutPID();
  
  /* Start logging */
  openlog("dcf77d", 0, LOG_DAEMON);

  startclock();

  exit(0);
}


/*******************
** InitSignalHandler() -- 
** setup signal handlers.
*/

static void InitSignalHandlers()
{
  /* install signal handler */

  if (signal(SIGINT, SIG_IGN) != SIG_IGN)
  {
    signal(SIGINT, Done);
  }
  
  if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
  {
    /* maybe restart on HUP ?? */
    signal(SIGHUP, Done);
  }

  if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
  {
    signal(SIGQUIT, Done);
  }
  
  signal(SIGTERM, Done);

  return;
}

/******************/


/*******************
** Done --  
** sighandler and cleanup.
*/

void Done(int sig)
{

  /* cleanup and exit quiet */
  if (pidfile != -1)
  {
    close(pidfile);
    unlink(DCF_PID);
  }
  
  if (fd != -1)
    close(fd);

  /* paranoia */
  if ((cmos) && (rtc != -1))
    close(rtc);

  exit(0);
}

/******************/


/*******************
** PutPID -- 
** save PID to DCF_PID file
*/

static void PutPID()
{

  char pidbuffer[MAX_PID_LEN];
  int pidlen;

  snprintf(pidbuffer, MAX_PID_LEN, "%d", (int)getpid());
  pidlen = strlen(pidbuffer);
  
  if ((pidfile = open(DCF_PID,
		      (O_RDWR|O_CREAT|O_TRUNC),
		      (S_IRUSR|S_IWUSR))) == -1)
  {
    fprintf(stderr, "dcf77d: Couldn't open %s.\n", DCF_PID);
    exit(1);
  }

  if (write(pidfile, (void *)pidbuffer, pidlen) != pidlen)
  {
    close(pidfile);
    pidfile = -1;
    unlink(DCF_PID);
    fprintf(stderr, "dcf77d: Couldn't write PID to %s.\n", DCF_PID);
    exit(1);
  }

  return;
}

/******************/


/*******************
** CheckPID -- 
** check for another instance already running.
** Seems to be a bit paranoid sometimes, but since we
** are a root-only program it is better to be paranoid :-)
**
** NOTE that sig-handlers should be installed
** after this is called.
*/

static int CheckPID()
{
  struct stat o_pid_stat;
  int o_pid_fd = -1;
  char o_pid_string[MAX_PID_LEN];
  int o_pid_len, o_pid;

  errno = 0;
  if (stat(DCF_PID, &o_pid_stat) == -1)
  {
    /* if file does not exist,
    ** simply return 0 indicating
    ** that all is clear and we're ready to go.
    */
    if (errno == ENOENT)
      return(0);
    else
      goto bad_pidfile;
  }

  /* before accessing the pid-file do
  ** some paranoia checks
  ** (must be a regular file, belong to root, has rw for
  **  root set and is not executable for anyone and has not
  **  S_ISUID or S_ISGID bit set)
  */
  if ((o_pid_stat.st_uid == 0) &&
      (S_ISREG(o_pid_stat.st_mode)) &&
      (o_pid_stat.st_mode & (S_IRUSR|S_IWUSR)) &&
      (!(o_pid_stat.st_mode &(S_ISUID|S_ISGID|S_IXUSR|S_IXGRP|S_IXOTH))))
  {
    /* o.k., access */
    if ((o_pid_fd = open(DCF_PID, O_RDONLY)) == -1)
      goto bad_pidfile;

    if ((o_pid_len = read(o_pid_fd,o_pid_string,(MAX_PID_LEN-1))) == -1)
      goto bad_pidfile;

    close(o_pid_fd);
    o_pid_fd = -1;
        
    /* check if read buffer contains a valid process id */
    if (o_pid_len > 0)
    {
      char *endptr;
      
      /* force last char to be \0-termination*/
      o_pid_string[o_pid_len + 1] = '\0';

      /* convert */
      o_pid = strtoul(o_pid_string, &endptr, 10);

      if (((void *)endptr != (void *)(o_pid_string + o_pid_len)) ||
	  (o_pid == ULONG_MAX) ||
	  (o_pid < 2))
      {
	fprintf(stderr, "dcf77d: Wrong data in %s.\n", DCF_PID);
	goto bad_pidfile;
      }

      /* check if there is a running process with this pid */
      errno = 0;
      if (kill(o_pid, 0) == -1)
      {
	if (errno == ESRCH)
	{
	  /* no such process, maybe an old file,
	  ** try to delete it.
	  */
	  unlink(DCF_PID);

	  /* ... and go */
	  return(0);
	}
      }

      /* there is a running process,
      */
      if (kill_other == TRUE)
      {
	int i;
	
	/* try to kill it */
	kill(o_pid, SIGTERM);
	for (i = 0; i < 5; i++)
	{
	  /* wait */
	  sleep(1);
	  errno = 0;
	  if (kill(o_pid, 0) == -1)
	  {
	    /* has it gone ? */ 
	    if (errno == ESRCH)
	      return(0);
	  }
	}

	/* process has not terminated,
	** so better we exit.
	*/
	fprintf(stderr, "dcf77d: Unable to kill process %d.\n", o_pid); 
	return(1);
      }
      /* other process exists
      ** and should not be killed */
      return(1);
    }
      
  }

  fprintf(stderr, "dcf77d: Sorry Guy, wrong permissions for %s.\n", DCF_PID);
  
  /* not nice, but efficient here */
  bad_pidfile:
    if (o_pid_fd != -1)
      close(o_pid_fd);
    fprintf(stderr, "dcf77d: Error while accessing %s.\n",DCF_PID);
    exit(1);
}

/******************/






