/*
 *   getinfo.c convert the clock data and set the system time.
 *
 *   Copyright 1997        quacks@paula.owl.de (Joerg Krause)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "dcf77.h"
#include <linux/mc146818rtc.h>

static struct tm loctime;
time_t last_utc=0;        /* Last time received from the clock */

extern int  cmos;       /* TRUE == Update CMOS Clk (requires RTC Kernel) */
extern int  rtc;
extern int  loglevel;  	/* log level */

/*--------------------------------------------------
 * Check the given range of 'buf' for EVEN parity.
 * Return TRUE if parity is ok.
 *--------------------------------------------------*/
 
static int CheckParity(char *buf, int FirstByte, int ParityByte)
{
  char *parity, *tstr;
  int i, sum; 

  sum    = 0;
  parity = buf + ParityByte;
  tstr   = buf + FirstByte;
  
  for( i = 0 ; i <= (ParityByte - FirstByte) ; i++ )
  {
    if(*tstr == 1)
      sum ^= *tstr;
    tstr++; 
  }
  if(sum)
    return(FALSE);
  else
    return(TRUE);
}



void ParseDCFdata(char buf[])
{
  int i, j; 
  int parity_flg=TRUE;
  
  /*
   *  Parity check and other error check possibility's
   */

  if(!CheckParity(buf, Min1Start, MinParity))	 /* Minutes Parity */
    parity_flg = FALSE;
  
  if(!CheckParity(buf, Hour1Start, HourParity))	 /* Hour Parity */
    parity_flg = FALSE;
  
  if(!CheckParity(buf, Day1Start, DateParity))	 /* Date Parity */
    parity_flg = FALSE;
  
  if((buf[Summer] == 1) && (buf[Winter] == 1))   /* only 1 state allowed */
    parity_flg = FALSE;
  
  if(buf[DataStart] != 1 )			 /* must allways be 1	 */
    parity_flg = FALSE;
  
  if(parity_flg)
  {
    time_t loc_utc, kernel_utc;
    
    /* Start a new minute */
    loctime.tm_sec  =0;
    loctime.tm_min  =0;
    loctime.tm_hour =0;
    loctime.tm_mday =0;
    loctime.tm_mon  =0;
    loctime.tm_year =0;
    loctime.tm_isdst=0;		/* Winter time (default) */
    
    /*---- Build Minute ----*/
    for(i = Min1Start,  j = 1; i <= Min1End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_min =  loctime.tm_min +j; 
      }
      j <<= 1 ;
    }
    
    for(i = Min10Start, j = 1; i <= Min10End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_min =  loctime.tm_min + (j*10); 
      }
       j <<= 1 ;
    }

    /*---- Build Hour ----*/
    for(i = Hour1Start, j = 1; i <= Hour1End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_hour =  loctime.tm_hour +j;  
      }
      j <<= 1 ;
    }
    
    for(i = Hour10Start, j = 1; i <= Hour10End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_hour =  loctime.tm_hour +(j*10);  
      }
      j <<= 1 ;
    }

    /*---- Build Day ----*/
    for(i = Day1Start, j = 1; i <= Day1End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_mday =  loctime.tm_mday +j;  
      }
      j <<= 1 ;
    }
    
    for(i = Day10Start, j = 1; i <= Day10End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_mday =  loctime.tm_mday +(j*10);  
      }
      j <<= 1 ;
    }
    
    /*---- Build Month ----*/
    for(i = Month1Start, j = 1; i <= Month1End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_mon =  loctime.tm_mon +j;  
      }
      j <<= 1 ;
    }
    loctime.tm_mon = loctime.tm_mon + (buf[Month10] *10);
    loctime.tm_mon--;
    
    /*---- Build Year ----*/
    for(i = Year1Start, j = 1; i <= Year1End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_year =  loctime.tm_year +j;  
      }
      j <<= 1 ;
    }
		
    for(i = Year10Start, j = 1; i <= Year10End ; i++ )
    {
      if(buf[i] ==  1)
      {
	loctime.tm_year =  loctime.tm_year +(j*10); 
      }
      j <<= 1 ;
    }

    /* Set summer or winter time */

    if (!buf[Winter])
      loctime.tm_isdst = 1;

    /*  writelog(LOG_NOTICE,"-rb- new time calculated."); */
    
    /*
     *  Set the new time to kernel and cmos
     */

    if ((loc_utc = mktime (&loctime)) >= 0)
    {
      /* Initialize var */
      if (!last_utc)
	last_utc=loc_utc;
      
      if (difftime(loc_utc, last_utc) != (double) 60)
      {
	writelog(LOG_ERR,"Clock out of synch.");
	last_utc=loc_utc;
      }
      else
      {
	last_utc=loc_utc;
	/* may this one fail ?? I hope it doesn't :)*/
	kernel_utc = time(NULL);
	
	if (difftime(loc_utc, kernel_utc) != (double)0)
	{
	  if ((stime(&loc_utc))==-1)
	    writelog(LOG_ERR,"Failed to set new kernel time.");
	  else
	  {
	    if (cmos)
	    {
	      int retval;
	      
	      /* Be sure that struct rtc_time is equal with struct tm !!*/
	      if ((retval = ioctl(rtc, RTC_SET_TIME ,&loctime)) ==-1)
		writelog(LOG_ERR,"Failed to set new cmos time.");
	      else
		writelog(LOG_NOTICE, "Synchronized kernel time and cmos clock.");
	    }
	    else
	      writelog(LOG_NOTICE, "Synchronized kernel time.");
	  }
	}
      }
    }
    else
    {
      writelog(LOG_NOTICE,"Time not set... mktime ?");
    }
  }
  else
  {
    writelog(LOG_NOTICE,"Incorrect Parity Flag(s)");
  }
}
