/* SP12: A serial programmer for working with Atmel AVR uCs          */
/* Copyright (C) 1997-2003 Ken Huntington, Kevin Towers, Pitronics.  */

/* This program is free software; you can redistribute it and/or     */
/* modify it under the terms of the GNU General Public License       */
/* as published by the Free Software Foundation; either version 2    */
/* of the License, or (at your option) any later version.            */

/* This program is distributed in the hope that it will be useful,   */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/* GNU General Public License for more details.                      */

/* You should have received a copy of the GNU General Public License */
/* along with this program; if not, write to the Free Software       */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston,            */
/* MA  02111-1307, USA.                                              */

/* Pitronics can be reached by email: sbolt@xs4all.nl                */
/* Kevin Towers can be reached by email: ktowers@omnexcontrols.com   */
/* Ken Huntington can be reached by email: kenh@compmore.net         */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "dos_cpt.h"
#include "sp12.h"


/* Accepts a 16-bit address. Reads the codeWord in two steps         */ 
/* from the device connected to the Centronics port.                 */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

unsigned int readCodeWord(unsigned int address) {
    
   unsigned char readByte;
   unsigned int codeWord;
   unsigned long readCmd;

   codeWord = 0;
   readCmd = READ_PROG_HI | ((long) address << 8);
   readByte = (unsigned char) clockOutCommand(readCmd);
   codeWord = codeWord | readByte;
   readCmd = READ_PROG_LO | ((long) address << 8);
   readByte = (unsigned char) clockOutCommand(readCmd);
   codeWord = (codeWord << 8) | readByte;
   return(codeWord);
}

/* Accepts a 16-bit codeWord and a 7-bit address. Writes the code    */ 
/* in two steps with no delays into the ATMega device connected to   */
/* the Centronics port.                                              */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

void loadCodePage(unsigned int address, unsigned int codeWord) {

   unsigned char loByte;
   unsigned char hiByte;
   unsigned long writeCmd;

   loByte = (unsigned char) codeWord & 0x00FF;
   hiByte = (unsigned char) ((codeWord & 0xFF00) >> 8);

   writeCmd = LOAD_PAGE_LO | ((long) address << 8);
   writeCmd = writeCmd | loByte;
   clockOutCommand(writeCmd);
   writeCmd = LOAD_PAGE_HI | ((long) address << 8);
   writeCmd = writeCmd | hiByte;  
   clockOutCommand(writeCmd);
}

/* Accepts a 16-bit page number and programs it into the ATMega      */
/* device connected to the Centronics parallel port.                 */
/* The page is read back and compared with the buffer; if all        */
/* is well, 0 is returned, else 1.                                   */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

int writeCodePage(unsigned int flashBuf[], unsigned int address) {
   
   unsigned long writeCmd;
   unsigned int idx;

   writeCmd = WRITE_PAGE | ((long) address << 8);
   clockOutCommand(writeCmd);
   loopDelay(timeConst.pageWrite);
   for (idx = address + 1 - device.pageSize; idx <= address; idx++) {
      if (flashBuf[idx] != 0xffff) {
         if (readCodeWord(idx) != flashBuf[idx])
            return(1);
      }
   }
   return(0);
}

/* Accepts a 16-bit codeWord and a 16-bit address. Writes the code   */
/* in two steps into the device connected to the Centronics port and */
/* verifies arrival. Returns 1 if verify fails three times, else 0.  */
/* The time constant byteWrite is adjusted dynamically,              */
/* if the device is not a 1200(A) and optimizeFlag allows.           */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

int writeFlashVerified(unsigned int address, unsigned int codeWord,
                        int optimizeFlag) {
 
   unsigned char loByte;
   unsigned char hiByte;
   unsigned long readCmd, writeCmd;
   int failure;

   loByte = (unsigned char) codeWord & 0x00FF;
   hiByte = (unsigned char) ((codeWord & 0xFF00) >> 8);

   writeCmd = WRITE_PROG_HI | ((long) address << 8);
   writeCmd = writeCmd | hiByte;
   readCmd = READ_PROG_HI | ((long) address << 8);
   failure = writeByteVerified(writeCmd, readCmd, optimizeFlag);
   if (failure == 0) {
      writeCmd = WRITE_PROG_LO | ((long) address << 8);
      writeCmd = writeCmd | loByte;
      readCmd = READ_PROG_LO | ((long) address << 8);
      failure = writeByteVerified(writeCmd, readCmd, optimizeFlag);
   }
   return(failure);
}

/* Expects flashBuf[] to be filled with data.                        */
/* If a (page) write fails, 1 is returned, else 0.                   */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

int writeFlashArea(unsigned int flashBuf[], long bufLimit,
                                            int optimizeFlag) {

   long address, previous; 
   int writeFlg = 0;
   int idx = 1;

   previous = 0;
   printf("...............................................................\r");
   for (address = 0; address < bufLimit; address++) {
      if (device.pageMode) {
         /*
          * To speed things up: Skip locations containing 0xffff
          * (The device has been erased before writing started)
          */
         if (flashBuf[address] != 0xffff) {
            loadCodePage((unsigned int) address, flashBuf[address]); 
            writeFlg = 1;
         }
         if (writeFlg && (address > 0) && 
                  ((address & (device.pageSize - 1)) == device.pageSize - 1)) {
            writeFlg = 0;
            if (writeCodePage(flashBuf, (unsigned int) address) == 1) {
               if (++idx < 6) {
                  /*
                   * recalculate first address of the failed page
                   * once only. Note: The calculation below puts 
                   * address at the last address value of the previous
                   * page. But address is incremented _after_ this
                   * calculation, _before_ loadCodePage is called
                   * again.
                   */
                  if (idx == 2) address = address - device.pageSize;
                  if (timeConst.pageWriteAdjusted == 1) writeRetries++;
               } else {
                  printf("!!\n");
                  return (1);
               }
               if (optimizeFlag == 11) {
                  timeConst.pageWrite += timeConst.pageWrite / 2;
                  if (timeConst.pageWrite > 1.5 * timeConst.pageWriteDefault) {
                     printf("!!\n");
                     return (1);
                  }
                  timeConst.pageWriteAdjusted = 1;
               }
            } else {
               if (optimizeFlag == 11 && timeConst.pageWriteAdjusted == 0) {
                  timeConst.pageWrite -= timeConst.pageWrite / 2;
               }
               idx = 1;
            }
         }
      } else {
         if (flashBuf[address] != 0xffff) {
            if (writeFlashVerified((unsigned int) address, 
                         flashBuf[address], optimizeFlag) == 1) {
               printf("!!\n");
               return (1);
            }
         }
      }
      if (address >= (previous + bufLimit / 64)) {
         putchar('o');
         fflush(stdout);
         previous = address;
      } 
   }
   /*
    * In page mode, a sequence like pageWrite reducing from 64ms
    * to 32, 16, 8 and 4ms, then adjusting back upwards to 6ms 
    * and finally 9ms is valid; this would cause writeRetries
    * to be incremented once. Therefore:
    */
   if (device.pageMode && writeRetries == 1)
      writeRetries = 0; 
   printf("\n");
   return (0);
}

/* (expects flashBuf[] to be filled with 0xFFFF)                     */
/* Loops readCodeWord into flashBuf[]                                */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

void readFlashArea(unsigned int flashBuf[], long bufLimit) {

   long address, previous;

   previous = 0;
   printf("...............................................................\r");

   for (address = 0; address < bufLimit; address++) {
      flashBuf[address] = readCodeWord((unsigned int) address);
      if (address >= (previous + bufLimit / 64)) {
         putchar('o');
         fflush(stdout);
         previous = address;
      } 
   }
   printf("\n");
}
