/* SP12: A serial programmer for working with Atmel AVR uCs          */
/* Copyright (C) 1997-2003 Ken Huntington, Kevin Towers, Pitronics.  */

/* This program is free software; you can redistribute it and/or     */
/* modify it under the terms of the GNU General Public License       */
/* as published by the Free Software Foundation; either version 2    */
/* of the License, or (at your option) any later version.            */

/* This program is distributed in the hope that it will be useful,   */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/* GNU General Public License for more details.                      */

/* You should have received a copy of the GNU General Public License */
/* along with this program; if not, write to the Free Software       */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston,            */
/* MA  02111-1307, USA.                                              */

/* Pitronics can be reached by email: sbolt@xs4all.nl                */
/* Kevin Towers can be reached by email: ktowers@omnexcontrols.com   */
/* Ken Huntington can be reached by email: kenh@compmore.net         */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "dos_cpt.h"
#include "sp12.h"


/* Accepts a 16-bit address. Reads the dataByte in two steps         */ 
/* from the device connected to the Centronics port.                 */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

unsigned char readDataByte(unsigned int address) {
    
   unsigned int dataByte;
   unsigned long readCmd;

   readCmd = READ_EEPROM | ((long) address << 8);
   dataByte = (unsigned char) clockOutCommand(readCmd);
   return(dataByte);
}

/* Accepts a dataByte and a 16-bit address. Writes the code in two   */
/* steps into the device connected to the Centronics port and        */
/* verifies arrival. Returns 1 if verify fails three times, else 0.  */
/* The time constant byteWrite is adjusted dynamically,              */
/* if the device is not a 1200(A) and optimizeFlag allows.           */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

int writeEepromVerified(unsigned int address, unsigned char dataByte,
                         int optimizeFlag) {

   int failure;
   unsigned long readCmd, writeCmd;

   readCmd = READ_EEPROM | ((long) address << 8);
   writeCmd = WRITE_EEPROM | ((long) address << 8);
   writeCmd = writeCmd | dataByte;
   failure = writeByteVerified(writeCmd, readCmd, optimizeFlag);
   return(failure);
}

/* Expects eepromBuf[] to be filled with data.                       */
/* executes writeEepromVerified() address by address;                */
/* if verify fails, stops and returns 1, else returns 0.             */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

int writeEepromArea(unsigned int eepromBuf[], long bufLimit, 
                             int optimizeFlag, int overlayFlag) {

   long address, previous;

   previous = 0;
   printf("...............................................................\r");
   /*
    * If the overlayFlag is cleared, all addresses are written to
    * since all data words (including 0xFF) can be valid.
    * When the overlayFlag is set, buffer addresses containing 
    * the original 0xFF padding are skipped.
    */
   for (address = 0; address < bufLimit; address++) {
      if (address >= (previous + bufLimit / 64)) {
         putchar('o');
         fflush(stdout);
         previous = address;
      } 
      if (overlayFlag && eepromBuf[address] > 0xFF)
         continue;
      if (writeEepromVerified (address, (unsigned char) eepromBuf[address],
                                                         optimizeFlag) == 1) {
         printf("!!\n");
         return (1);
      }
   }
   printf("\n");
   return (0);
}

/* (expects eepromBuf[] to be filled with 0xFF)                      */
/* Loops readDataByte into eepromBuf[]                               */
/* Power bits prior state assumed on; left on at return.             */
/* SCK prior state assumed lo; left lo at return.                    */

void readEepromArea(unsigned int eepromBuf[], long bufLimit) {

   long address, previous;

   previous = 0;
   printf("...............................................................\r");

   for (address = 0; address < bufLimit; address++) {
      eepromBuf[address] = readDataByte(address);
      if (address >= (previous + bufLimit / 64)) {
         putchar('o');
         fflush(stdout);
         previous = address;
      } 
   }
   printf("\n");     
}

