/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"

/*------------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  static int
Load_Line( char *buff, FILE *pfile, char *mesg )
{
  int
    num_chr, /* Number of characters read, excluding lf/cr */
    chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];

  /* Prepare error message */
  snprintf( error_mesg, sizeof(error_mesg),
      _("Error reading %s\n"\
        "Premature EOF (End Of File)"), mesg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return error if chr = EOF */
  if( (chr = fgetc(pfile)) == EOF )
  {
    fprintf( stderr, "xhamlog: %s\n", error_mesg );
    fclose( pfile );
    Error_Dialog( error_mesg, QUIT );
    return( EOF );
  }

  /* Ignore commented lines, white spaces and eol/cr */
  while(
      (chr == '#') ||
      (chr == HT ) ||
      (chr == ' ') ||
      (chr == CR ) ||
      (chr == LF ) )
  {
    /* Go to the end of line (look for LF or CR) */
    while( (chr != CR) && (chr != LF) )
      /* Get next character, return error if chr = EOF */
      if( (chr = fgetc(pfile)) == EOF )
      {
        fprintf( stderr, "xhamlog: %s\n", error_mesg );
        fclose( pfile );
        Error_Dialog( error_mesg, QUIT );
        return( EOF );
      }

    /* Dump any CR/LF remaining */
    while( (chr == CR) || (chr == LF) )
      /* Get next character, return error if chr = EOF */
      if( (chr = fgetc(pfile)) == EOF )
      {
        fprintf( stderr, "xhamlog: %s\n", error_mesg );
        fclose( pfile );
        Error_Dialog( error_mesg, QUIT );
        return( EOF );
      }

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = 80 or EOF or CR/LF */
  while( num_chr < 80 )
  {
    /* If LF/CR reached before filling buffer, return line */
    if( (chr == LF) || (chr == CR) ) break;

    /* Enter new character to line buffer */
    buff[num_chr++] = (char)chr;

    /* Get next character */
    if( (chr = fgetc(pfile)) == EOF )
    {
      /* Terminate buffer as a string if chr = EOF */
      buff[num_chr] = '\0';
      return( SUCCESS );
    }

    /* Abort if end of line not reached at 80 char. */
    if( (num_chr == 80) && (chr != LF) && (chr != CR) )
    {
      /* Terminate buffer as a string */
      buff[num_chr] = '\0';
      snprintf( error_mesg, sizeof(error_mesg),
          _("Error reading %s\n"\
            "Line longer than 80 characters"), mesg );
      fprintf( stderr, "xhamlog: %s\n%s\n", error_mesg, buff );
      fclose( pfile );
      Error_Dialog( error_mesg, QUIT );
      return( ERROR );
    }

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( SUCCESS );

} /* End of Load_Line() */

/*------------------------------------------------------------------------*/

/*  Load_Config()
 *
 *  Loads the xhamlogrc configuration file
 */

  gboolean
Load_Config( gpointer idata )
{
  char
    rc_fpath[128], /* File path to xhamlogrc */
    line[81];     /* Buffer for Load_Line  */

  /* Config file pointer */
  FILE *xhamlogrc;

  GtkWidget *entry;


  /* Setup path to xhamlog home dir */
  snprintf( rc_data.home_dir, sizeof(rc_data.home_dir),
      "%s/%s", getenv("HOME"), "xhamlog" );

  /* Setup xhamlog file paths */
  snprintf( rc_fpath, sizeof(rc_fpath),
      "%s/%s", rc_data.home_dir, RC_FILE );
  snprintf( rc_data.stn_log_file, sizeof(rc_data.stn_log_file),
      "%s/%s", rc_data.home_dir, STATION_LOG_FILE );
  snprintf( rc_data.stn_qsl_file, sizeof(rc_data.stn_qsl_file),
      "%s/%s", rc_data.home_dir, STATION_QSL_FILE );
  snprintf( rc_data.stn_adif_file, sizeof(rc_data.stn_adif_file),
      "%s/%s", rc_data.home_dir, STATION_ADIF_FILE );

  /* Open xhamlogrc file */
  xhamlogrc = fopen( rc_fpath, "rw" );
  if( xhamlogrc == NULL )
  {
    perror( rc_fpath );
    Error_Dialog(
    _("Failed to open "RC_FILE" file\n"\
      "Quit xhamlog and correct"), QUIT );
    return( FALSE );
  }

  /*** Read runtime configuration data ***/
  /* Read Capitalize enable flag, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Capitalize Enable")) != SUCCESS )
    return( FALSE );

  if( strcmp(line, "yes") == 0 )
    SetFlag( CAPITALIZE );
  else if( strcmp(line, "no") != 0 )
  {
    fclose( xhamlogrc );
    Error_Dialog(
        _("Error reading "RC_FILE"\n"\
          "Unrecognized menu option\n"\
          "Quit xhamlog and correct"), QUIT );
    return( FALSE );
  }

  /* Read operator callsign, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Operator Callsign")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_call, line, sizeof(rc_data.my_call) );

  /* Read operator name, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Operator Name")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_name, line, sizeof(rc_data.my_name) );

  /* Read operator surname, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Operator Surname")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_srname, line, sizeof(rc_data.my_srname) );

  /* Read operator address line 1, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Operator Address")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_addr1, line, sizeof(rc_data.my_addr1) );

  /* Read operator address line 2, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Operator Address")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_addr2, line, sizeof(rc_data.my_addr2) );

  /* Read Zone, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("ITU Zone")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_zone, line, sizeof(rc_data.my_zone) );

  /* Read QTH, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("QTH Name")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_qth, line, sizeof(rc_data.my_qth) );

  /* Read QTH locator, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("QTH Locator")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.my_loc, line, sizeof(rc_data.my_loc) );

  /* Read Transmitter name, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Transmitter Name")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.tx, line, sizeof(qso_record.tx) );
  Strlcpy( rc_data.tx, qso_record.tx, sizeof(rc_data.tx) );

  /* Read Transmitter power, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Transmitter Power")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.tx_power, line, sizeof(qso_record.tx_power) );
  Strlcpy( rc_data.tx_power, qso_record.tx_power, sizeof(rc_data.tx_power) );

  /* Read Transmitter antenna, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Transmitter Antenna")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.tx_ant, line, sizeof(qso_record.tx_ant) );
  Strlcpy( rc_data.tx_ant, qso_record.tx_ant, sizeof(rc_data.tx_ant) );

  /* Read Receiver name, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Receiver Name")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.rx, line, sizeof(qso_record.rx) );
  Strlcpy( rc_data.rx, qso_record.rx, sizeof(rc_data.rx) );

  /* Read Receiver N.Fig, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Receiver Noise Figure")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.rx_nfig, line, sizeof(qso_record.rx_nfig) );
  Strlcpy( rc_data.rx_nfig, qso_record.rx_nfig, sizeof(rc_data.rx_nfig) );

  /* Read Receiver antenna, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Receiver Antenna")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.rx_ant, line, sizeof(qso_record.rx_ant) );
  Strlcpy( rc_data.rx_ant, qso_record.rx_ant, sizeof(rc_data.rx_ant) );

  /* Read default remarks on QSO */
  if( Load_Line(line, xhamlogrc, _("Default Remarks")) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.remarks, line, sizeof(qso_record.remarks) );

  /* Read log file viewer command */
  if( Load_Line(line, xhamlogrc, _("Log File Viewer")) != SUCCESS )
    return( FALSE );
  line[25] = '\0';
  snprintf( rc_data.log_viewer, sizeof(rc_data.log_viewer),
      "%s %s/%s", line, rc_data.home_dir, STATION_LOG_FILE );

  /* Read log file printer command */
  if( Load_Line(line, xhamlogrc, _("Log File Printer")) != SUCCESS )
    return( FALSE );
  line[25] = '\0';
  snprintf( rc_data.log_print, sizeof(rc_data.log_print),
      "%s %s/%s", line, rc_data.home_dir, STATION_LOG_FILE );

  /* Read qsl card file viewer command */
  if( Load_Line(line, xhamlogrc, _("QSL Card Viewer")) != SUCCESS )
    return( FALSE );
  line[25] = '\0';
  snprintf( rc_data.qsl_viewer, sizeof(rc_data.qsl_viewer),
      "%s %s/%s", line, rc_data.home_dir, STATION_QSL_FILE );

  /* Read qsl card file printer command */
  if( Load_Line(line, xhamlogrc, _("QSL Card Printer")) != SUCCESS )
    return( FALSE );
  line[25] = '\0';
  snprintf( rc_data.qsl_print, sizeof(rc_data.qsl_print),
      "%s %s/%s", line, rc_data.home_dir, STATION_QSL_FILE );

  /*** Setup default CAT items ***/
  /* Read serial port device, abort if EOF */
  if( Load_Line(line, xhamlogrc, _("Serial Port device")) != SUCCESS )
    return( FALSE );
  Strlcpy( rc_data.cat_serial, line, sizeof(rc_data.cat_serial) );

  /* Read transceiver type */
  if( Load_Line(line, xhamlogrc, _("Transceiver Type")) != SUCCESS )
    return( FALSE );
  if( strncmp(line, "FT847", strlen(line)) == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "ft847")), TRUE );
    rc_data.tcvr_type = FT847;
  }
  else if( strncmp(line, "FT857", strlen(line)) == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "ft857")), TRUE );
    rc_data.tcvr_type = FT857;
  }
  else if( strncmp(line, "K3", strlen(line)) == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "k3")), TRUE );
    rc_data.tcvr_type = K3;
  }
  else
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    rc_data.tcvr_type = NONE;
  }

  /* Enter default data to main window */
  entry = Builder_Get_Object( main_window_builder, "my_call" );
  gtk_entry_set_text( GTK_ENTRY(entry), rc_data.my_call );
  entry = Builder_Get_Object( main_window_builder, "my_name" );
  gtk_entry_set_text( GTK_ENTRY(entry), rc_data.my_name );
  entry = Builder_Get_Object( main_window_builder, "my_zone" );
  gtk_entry_set_text( GTK_ENTRY(entry), rc_data.my_zone );
  entry = Builder_Get_Object( main_window_builder, "my_qth" );
  gtk_entry_set_text( GTK_ENTRY(entry), rc_data.my_qth );
  entry = Builder_Get_Object( main_window_builder, "my_loc" );
  gtk_entry_set_text( GTK_ENTRY(entry), rc_data.my_loc );
  entry = Builder_Get_Object( main_window_builder, "tx" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.tx );
  entry = Builder_Get_Object( main_window_builder, "tx_power" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.tx_power );
  entry = Builder_Get_Object( main_window_builder, "tx_ant" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.tx_ant );
  entry = Builder_Get_Object( main_window_builder, "rx" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.rx );
  entry = Builder_Get_Object( main_window_builder, "rx_nfig" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.rx_nfig );
  entry = Builder_Get_Object( main_window_builder, "rx_ant" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.rx_ant );
  entry = Builder_Get_Object( main_window_builder, "remarks" );
  gtk_entry_set_text( GTK_ENTRY(entry), qso_record.remarks );

  fclose( xhamlogrc );

  return( FALSE );

} /* End of Load_Config() */

/*------------------------------------------------------------------*/

/*  Find_Line()
 *
 *  Finds a line in a text file containing a given string
 */

  static gboolean
Find_Line( char **line_idx, char *line, const char *string, FILE *fp )
{
  int err;

  /* Load lines and look for string */
  do
  {
    err = Load_Line( line, fp, _("Text file line") );
    *line_idx = strstr(line, string);
  }
  while( (err == 0) && (*line_idx == NULL) );

  /* Usually EOF is the cause of error */
  if( err )
  {
    Error_Dialog(
        _("Error searching text file\n"\
          "May be empty or corrupt"), OK );
    return( FALSE );
  }

  *line_idx += strlen(string);

  return( TRUE );

} /* Find_Line() */

/*------------------------------------------------------------------------*/

/*  Find_String()
 *
 *  Finds a string following spaces
 */

  static gboolean
Find_String( char **line_idx, char *line, const char *string )
{
  if( (*line_idx = strstr(line, string)) == NULL )
  {
    Error_Dialog(
        _("Error searching station_log.txt\n"\
          "May be empty or corrupted"), QUIT );
    return( FALSE );
  }

  *line_idx += strlen(string);

  return( TRUE );

} /* Find_String() */

/*------------------------------------------------------------------------*/

/*  File_Print()
 *
 *  Prints a string to a file
 */

  static gboolean
File_Print( FILE **fp, char *string )
{
  int sl = (int)strlen(string);
  if( fprintf(*fp, "%s", string) != sl )
  {
    perror( "xhamlog: fprintf" );
    ClearFlag( SAVE_RECORD );
    Error_Dialog(
        _("Error printing to file\n"\
          "Quit xhamlog and correct"), QUIT );
    Close_File( fp );
    return( FALSE );
  }

  return( TRUE );

} /* File_Print() */

/*------------------------------------------------------------------------*/

/*  Save_QSO_Record()
 *
 *  Saves QSO record to .txt file
 */

  gboolean
Save_QSO_Record( void )
{
  /* Buffer for reading and printing to files */
  char buff[REC_BUFF_SIZE];
  char *buff_idx;

  int
    num_records, /* Number of QSO records    */
    num_qsled,   /* Number of QSL's printed  */
    fopen_ret;   /* Open_File() return value */

  /* Create a file for station log if not already open */
  fopen_ret = Open_File(
      &rc_data.station_log_fp, rc_data.stn_log_file, "r+" );
  if( fopen_ret == FILE_NEW )
  {
    /* Print a header to log file */
    snprintf( buff, REC_BUFF_SIZE,
        _("|------------------------------------"\
        "--------------------------------------|\n"\
        "|          Station Log File - Created in"\
        " Text format by %10s        |\n"\
        "|------------------------------------"\
        "--------------------------------------|\n"),
        PACKAGE_STRING );
    if( ! File_Print(&rc_data.station_log_fp, buff) )
      return( FALSE );

    /* Print an initial (0) count of records and qsl's */
    snprintf( buff, REC_BUFF_SIZE,
        _("|  Num of QSO Records:%-6d  Records QSL'd:%-6d"\
        "  Size of Records:%-4d   |\n"\
        "|------------------------------------"\
        "--------------------------------------|\n"),
        0, 0, RECORD_SIZE );
    if( ! File_Print(&rc_data.station_log_fp, buff) )
      return( FALSE );
  }
  else if( fopen_ret == FILE_FAIL )
    return( FALSE );

  /*** Increment and print Record and QSL count ***/
  /* Go to beginning of log file */
  rewind( rc_data.station_log_fp );

  /* Find line with "Number of QSO Records:" */
  if( ! Find_Line(&buff_idx, buff, _("QSO Records:"), rc_data.station_log_fp) )
    return( FALSE );
  num_records = atoi(buff_idx);

  /* Find "Number of QSO's QSL'd:" string */
  if( ! Find_String(&buff_idx, buff, _("Records QSL'd:")) )
    return( FALSE );
  num_qsled = atoi(buff_idx);

  /* Go back to beginning of line and print new record */
  long sk = (long)(strlen(buff)+1);
  if( !fseek(rc_data.station_log_fp, -sk, SEEK_CUR) )
    return( FALSE );
  snprintf( buff, REC_BUFF_SIZE,
      _("|  Num of QSO Records:%-6d  Records QSL'd:%-6d"\
      "  Size of Records:%-4d   |\n"\
      "|------------------------------------"\
      "--------------------------------------|\n"),
      ++num_records, num_qsled, RECORD_SIZE );
  if( ! File_Print(&rc_data.station_log_fp, buff) )
    return( FALSE );

  /*** Print record in text log file ***/
  if( !fseek(rc_data.station_log_fp, 0, SEEK_END) )
    return( FALSE );
  snprintf( buff, REC_BUFF_SIZE,
      _("|DX/ CALL: %-14s NAME: %-12s QTH: %-12s  LOC: %-6s|\n"\
      "|MY/ CALL: %-14s ZONE: %-12s QTH: %-12s  LOC: %-6s|\n"\
      "|QSO DATE: %-11s    TIME: %-5s UTC   FREQ: %-13s QSL: NO    |\n"\
      "|QSO MODE: %-11s DX-REPT: %-3s      MY-REPT: %-3s     VIA: %-12s|\n"\
      "|MY TRANS: %-11s   POWER: %-9s    ANT: %-15s          |\n"\
      "|MY RECVR: %-11s   N.FIG: %-7s      ANT: %-15s          |\n"\
      "| REMARKS: %-45s                   |\n"\
      "|------------------------------------"\
      "--------------------------------------|\n"),
      qso_record.dx_call, qso_record.dx_name, qso_record.dx_qth, qso_record.dx_loc,
      qso_record.my_call, rc_data.my_zone,    qso_record.my_qth, qso_record.my_loc,
      qso_record.date, qso_record.time,       qso_record.freq,
      qso_record.mode, qso_record.dx_rst,     qso_record.my_rst, qso_record.via,
      qso_record.tx,   qso_record.tx_power,   qso_record.tx_ant,
      qso_record.rx,   qso_record.rx_nfig,    qso_record.rx_ant, qso_record.remarks );
  if( ! File_Print(&rc_data.station_log_fp, buff) )
    return( FALSE );

  ClearFlag( SAVE_RECORD );
  Close_File( &rc_data.station_log_fp );

  return( TRUE );

} /* Save_QSO_Record() */

/*------------------------------------------------------------------------*/

/*  Read_QSO_Record()
 *
 *  Reads and validates QSO record entries
 */

  gboolean
Read_QSO_Record( void )
{
  GtkWidget *entry;

  char *month[12] =
  { _("Jan"),_("Feb"),_("Mar"),_("Apr"),_("May"),_("Jun"),
    _("Jul"),_("Aug"),_("Sep"),_("Oct"),_("Nov"),_("Dec") };

  /* Enter field values to QSO record structure */
  entry = Builder_Get_Object( main_window_builder, "dx_call" );
  Strlcpy( qso_record.dx_call,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.dx_call ) );

  entry = Builder_Get_Object( main_window_builder, "dx_rst" );
  Strlcpy( qso_record.dx_rst,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.dx_rst ) );

  entry = Builder_Get_Object( main_window_builder, "my_rst" );
  Strlcpy( qso_record.my_rst,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.my_rst ) );

  entry = Builder_Get_Object( main_window_builder, "dx_name" );
  Strlcpy( qso_record.dx_name,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.dx_name ) );

  entry = Builder_Get_Object( main_window_builder, "dx_qth" );
  Strlcpy( qso_record.dx_qth,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.dx_qth ) );

  entry = Builder_Get_Object( main_window_builder, "dx_loc" );
  Strlcpy( qso_record.dx_loc,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.dx_loc ) );

  entry = Builder_Get_Object( main_window_builder, "freq" );
  Strlcpy( qso_record.freq,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.freq ) );
  Strlcat( qso_record.freq, " MHz", sizeof( qso_record.freq ) );

  entry = Builder_Get_Object( main_window_builder, "mode" );
  Strlcpy( qso_record.mode,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.mode ) );

  entry = Builder_Get_Object( main_window_builder, "via" );
  Strlcpy( qso_record.via,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.via ) );

  entry = Builder_Get_Object( main_window_builder, "tx" );
  Strlcpy( qso_record.tx,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.tx ) );

  entry = Builder_Get_Object( main_window_builder, "tx_power" );
  Strlcpy( qso_record.tx_power,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.tx_power ) );
  Strlcat( qso_record.tx_power, " Watt", sizeof( qso_record.tx_power ) );

  entry = Builder_Get_Object( main_window_builder, "tx_ant" );
  Strlcpy( qso_record.tx_ant,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.tx_ant ) );

  entry = Builder_Get_Object( main_window_builder, "rx" );
  Strlcpy( qso_record.rx,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.rx ) );

  entry = Builder_Get_Object( main_window_builder, "rx_nfig" );
  Strlcpy( qso_record.rx_nfig,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.rx_nfig ) );
  Strlcat( qso_record.rx_nfig, " dB", sizeof( qso_record.rx_nfig ) );

  entry = Builder_Get_Object( main_window_builder, "rx_ant" );
  Strlcpy( qso_record.rx_ant,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.rx_ant ) );

  entry = Builder_Get_Object( main_window_builder, "my_call" );
  Strlcpy( qso_record.my_call,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.my_call ) );

  entry = Builder_Get_Object( main_window_builder, "my_zone" );
  Strlcpy( rc_data.my_zone,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( rc_data.my_zone ) );

  entry = Builder_Get_Object( main_window_builder, "my_qth" );
  Strlcpy( qso_record.my_qth,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.my_qth ) );

  entry = Builder_Get_Object( main_window_builder, "my_loc" );
  Strlcpy( qso_record.my_loc,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.my_loc ) );

  entry = Builder_Get_Object( main_window_builder, "remarks" );
  Strlcpy( qso_record.remarks,
      gtk_entry_get_text(GTK_ENTRY(entry)),
      sizeof( qso_record.remarks ) );

  /* Enter date and time */
  {
    int temp;

    entry = Builder_Get_Object( main_window_builder, "day" );
    temp = atoi( gtk_entry_get_text(GTK_ENTRY(entry)) );
    if( temp <= 0 )
      return( FALSE );
    snprintf( qso_record.date, 4, "%02d/", temp );

    entry = Builder_Get_Object( main_window_builder, "month" );
    temp  = atoi( gtk_entry_get_text(GTK_ENTRY(entry)) );
    if( temp <= 0 )
      return( FALSE );
    snprintf( &qso_record.date[3], 5, "%3s/", month[temp-1] );

    entry = Builder_Get_Object( main_window_builder, "year" );
    temp = atoi( gtk_entry_get_text(GTK_ENTRY(entry)) );
    if( (temp <= 2000) || (temp > 3000) )
      return( FALSE );
    snprintf( &qso_record.date[7], 5, "%4d", temp );

    entry = Builder_Get_Object( main_window_builder, "hour" );
    if( strlen(gtk_entry_get_text(GTK_ENTRY(entry))) < 1 )
      return( FALSE );
    snprintf( qso_record.time, 3, "%02d",
        atoi(gtk_entry_get_text(GTK_ENTRY(entry))) );

    entry = Builder_Get_Object( main_window_builder, "minute" );
    if( strlen(gtk_entry_get_text(GTK_ENTRY(entry))) < 1 )
      return( FALSE );
    snprintf( &qso_record.time[2], 4, ":%02d",
        atoi(gtk_entry_get_text(GTK_ENTRY(entry))) );
  }

  /* Validate QSO Record */
  if( (strlen(qso_record.dx_call) > 2) &&
      (strlen(qso_record.my_call) > 2) &&
      (strlen(qso_record.dx_rst)  > 1) &&
      (strlen(qso_record.my_rst)  > 1) &&
      (strlen(qso_record.freq)    > 0) &&
      (strlen(qso_record.mode)    > 1) )
    return( TRUE );
  else
    return( FALSE );

} /* Read_QSO_Record() */

/*------------------------------------------------------------------------*/

/*  Truncate_Spaces()
 *
 *  Removes trailing spaces from strings
 */

  static void
Truncate_Spaces( char *string )
{
  int idx, sl;

  sl = (int)strlen(string);
  for( idx = 0; idx < sl; idx++ )
    if( string[idx] == ' ' )
      break;
  string[idx] = '\0';

} /* Truncate_Spaces() */

/*------------------------------------------------------------------------*/

/* Atof()
 *
 * Replaces atof() to take into account the
 * locale-dependent decimal point character
 */

  static double
Atof( const char *nptr )
{
  int idx;
  size_t len;
  double d = 0.0;
  char *s = NULL;
  static gboolean first_call = TRUE;
  static char dp = '.';

  /* Find locale-dependent decimal point character */
  if( first_call )
  {
    struct lconv *lcnv;
    lcnv = localeconv();
    dp = *lcnv->decimal_point;
    first_call = FALSE;
  }

  /* Look for a . or , decimal point character
   * in the supplied number buffer (string) */
  len = strlen( nptr );
  for( idx = 0; idx < (int)len; idx++ )
    if( (nptr[idx] == ',') || (nptr[idx] == '.') )
      break;

  /* Create temporary string to modify decimal point */
  s = malloc( len + 1 );
  if( s == NULL )
  {
    Error_Dialog(
        "Memory allocation failed for"
        "Atof() number buffer string",
        QUIT );
    return( d );
  }
  strcpy( s, nptr );
  s[ len ] = '\0';

  /* If a decimal point character is found, replace */
  if( idx < (int)len ) s[idx] = dp;
  d = atof( s );
  free( s );

  return( d );
} /* End of Atof() */

/*------------------------------------------------------------------*/

/*  Open_File()
 *
 *  Opens a file
 */

  int
Open_File( FILE **fp, char *fpath, const char *mode )
{
  /* Check if file exists */
  if( *fp != NULL ) return( FILE_OPEN );

  /* If file does not exist, it will open new */
  *fp = fopen( fpath, mode );
  if( *fp == NULL )
  {
    char mesg[MESG_SIZE];
    perror( fpath );
    snprintf( mesg, sizeof(mesg),
        _("Failed to open\n%s\n"\
          "Quit xfhell and correct"), fpath );
    Error_Dialog( mesg, QUIT );
    return( FILE_FAIL );
  }
  else
  {
    /* Get current file position */
    long offset = ftell( *fp );
    if( offset < 0 )
    {
      char mesg[MESG_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("ftell() failed for\n%s\n"\
            "Quit xfhell and correct"), fpath );
      Error_Dialog( mesg, QUIT );
      return( FILE_FAIL );
    }

    /* Test if a new file was created */
    if( fseek(*fp, 0, SEEK_END) < 0 )
    {
      char mesg[MESG_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("fseek() failed for\n%s\n"\
            "Quit xfhell and correct"), fpath );
      Error_Dialog( mesg, QUIT );
      return( FILE_FAIL );
    }

    /* Test if a new file was created */
    if( ftell(*fp) == 0 )
      return( FILE_NEW );
    else if( fseek(*fp, offset, SEEK_SET) < 0 ) /* Reset file position */
    {
      char mesg[MESG_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("fseek() failed for\n%s\n"\
            "Quit xfhell and correct"), fpath );
      Error_Dialog( mesg, QUIT );
      return( FILE_FAIL );
    }
 }

  return( FILE_OPEN );
} /* Open_File() */

/*------------------------------------------------------------------------*/

/*  Make_QSL_Cards()
 *
 *  Makes a postscript file with QSL cards for
 *  the QSO included in the station log file
 */

  gboolean
Make_QSL_Cards( void )
{
  /* For use with "stat" */
  struct stat buf;

  /* Buffer for reading and printing to files */
  char buff[QSL_BUFF_SIZE];
  size_t s = sizeof(buff);
  char *line_idx;

  int
    num_records, /* Number of QSO records in station's log file  */
    record_size, /* Size in byte of QSO records in station's log */
    num_qsled,   /* Number of QSO's in log that have been QSL'd  */
    num_cards,   /* Number of QSL "cards" in station's QSL file  */
    cards_todo,  /* Number of QSL "cards" to add to the QSL file */
    fopen_ret,   /* Return value from Open_File() */
    card_idx;    /* Index to QSL "card" in a page (10 QSL cards) */

  /* Check that Log file is available */
  if( stat( rc_data.stn_log_file, &buf) == -1 )
  {
    Error_Dialog(
        _("Station Log file not found\n"\
          "Please save QSO data first"), OK );
    return( FALSE );
  }
  else
  {
    if( buf.st_size < 1001 )
    {
      Error_Dialog(
          _("Station Log file seems corrupt\n"\
            "Please check file and correct"), OK );
      return( FALSE );
    }
  } /* if( stat( rc_data.stn_log_file, &buf) == -1 ) */

  /* If QSL file is missing, copy the QSL template in its place */
  if( stat(rc_data.stn_qsl_file, &buf) == -1 )
  {
    char sys_comm[100];

    snprintf( sys_comm, sizeof(sys_comm),
        "cp %s/%s %s/%s",
        rc_data.home_dir, "qsl_template.ps",
        rc_data.home_dir, "station_qsl.ps" );
    if( system(sys_comm) != 0 )
    {
      Error_Dialog(
          _("Failed to open new Station QSL file\n"\
            "Please quit xhamlog and correct"), OK );
      return( FALSE );
    }

  } /* if( stat(rc_data.stn_qsl_file, &buf) == -1 ) */

  /* Create a file for QSO ADIF log if not available */
  fopen_ret = Open_File(
      &rc_data.log_adif_fp, rc_data.stn_adif_file, "a" );
  if( fopen_ret == FILE_NEW )
  {
    /* Enter a header to the file */
    snprintf( buff, s,
        "QSO Log file created in ADIF v1.0 format by %s <EOH>\n\n",
        PACKAGE_STRING );
    if( ! File_Print(&rc_data.log_adif_fp, buff) )
      return( FALSE );
  }
  else if( fopen_ret == FILE_FAIL )
    return( FALSE );

  /* Open files as needed */
  if( Open_File(
        &rc_data.station_log_fp, rc_data.stn_log_file, "w+") == FILE_FAIL )
    return( FALSE );

  /* Go to beginning of log file */
  rewind( rc_data.station_log_fp );

  /* Find line with "Num of QSO Records:" */
  if( ! Find_Line(
        &line_idx, buff, _("QSO Records:"), rc_data.station_log_fp) )
    return( FALSE );
  num_records = atoi(line_idx);

  /* Find "Records QSL'd:" string */
  if( ! Find_String(&line_idx, buff, _("Records QSL'd:")) )
    return( FALSE );
  num_qsled = atoi(line_idx);

  /* Number of QSO's needing QSL cards */
  cards_todo = num_records - num_qsled;
  if( ! cards_todo )
  {
    Error_Dialog( _("Station Log shows all QSOs are QSL'd"), OK );
    return( TRUE );
  }

  /* Find "Size of Records:" string */
  if( ! Find_String(&line_idx, buff, _("Size of Records:")) )
    return( FALSE );
  record_size = atoi(line_idx);

  /* Open file as needed */
  if( Open_File(
        &rc_data.station_qsl_fp, rc_data.stn_qsl_file, "r+") == FILE_FAIL )
    return( FALSE );

  /* Go to beginning of QSL card postscript file */
  rewind( rc_data.station_qsl_fp );

  /* Find line with "Number of QSL Cards:" */
  if( ! Find_Line(&line_idx, buff, _("QSL Cards:"), rc_data.station_qsl_fp) )
    return( FALSE );
  num_cards = atoi(line_idx);

  /* Go to end of QSL card postscript file */
  rewind( rc_data.station_qsl_fp );

  /* Go to first non-QSL'd QSO record */
  if( !fseek(rc_data.station_log_fp, -(1+cards_todo*record_size), SEEK_END) )
    return( FALSE );

  /*** Make new QSL cards in QSL card postscript file ***/
  /* Make only multiples of CARDS_PER_PAGE cards */
  cards_todo = (cards_todo/CARDS_PER_PAGE) * CARDS_PER_PAGE;
  while( cards_todo-- )
  {
    char *oper = ""; /* Operating condition (/M, /P etc) */
    int idx;

    /* Index to QSL "card" in a page of the station_qsl.ps file  */
    /* There is room for CARDS_PER_PAGE (10) cards per p.s. page */
    card_idx = num_cards % CARDS_PER_PAGE;
    num_cards++;

    if( ! Get_QSO_Record(rc_data.station_log_fp, TRUE) )
      return( FALSE );

    /* Find callsign suffix if any */
    int sl = (int)strlen( qso_record.my_call );
    for( idx = sl; idx > 0; idx-- )
      if( qso_record.my_call[idx] == '/' )
      {
        oper = &qso_record.my_call[idx];
        break;
      }

    /* Print QSO data to QSL card */
    /* This is postscript code    */
    if( snprintf( buff, s,
          "/dxcall   (%s) def\n"
          "/date     (%s) def\n"
          "/utc      (%s) def\n"
          "/freq     (%8.3f) def\n"
          "/mode     (%s) def\n"
          "/rprt     (%s) def\n"
          "/via      (%s) def\n"
          "/tx       (%s) def\n"
          "/pout     (%d) def\n"
          "/txant    (%s) def\n"
          "/rx       (%s) def\n"
          "/rxnf     (%3.1f) def\n"
          "/rxant    (%s) def\n"
          "/remks    (%s) def\n"
          "/zone     (%s Lo) def\n"
          "/qthloc   (c: %s) def\n"
          "/mycall   (%s) def\n"
          "/oper     (%s) def\n"
          "/name     (%s %s) def\n"
          "/addr1    (%s) def\n"
          "/addr2    (%s) def\n",
            qso_record.dx_call,
            qso_record.date,
            qso_record.time,
            Atof(qso_record.freq),
            qso_record.mode,
            qso_record.dx_rst,
            qso_record.via,
            qso_record.tx,
            atoi(qso_record.tx_power),
            qso_record.tx_ant,
            qso_record.rx,
            Atof(qso_record.rx_nfig),
            qso_record.rx_ant,
            qso_record.remarks,
            rc_data.my_zone,
            qso_record.my_loc,
            rc_data.my_call,
            oper,
            rc_data.my_name,
            rc_data.my_srname,
            rc_data.my_addr1,
            rc_data.my_addr2 ) < 0 )
            {
              perror( "snprintf" );
              Error_Dialog(
                  _("Error printing to string\n"\
                    "Quit xhamlog and correct"), QUIT );
              return( FALSE );
            }
    if( ! File_Print(&rc_data.station_qsl_fp, buff) )
      return( FALSE );

    /* Reposition QSL card on page */
    /* This is postscript code     */
    switch( card_idx )
    {
      case 0:
        snprintf( buff, s,
            "90 rotate\n427 -574 translate\nqslcard\n\n" );
        if( ! File_Print(&rc_data.station_qsl_fp, buff) )
          return( FALSE );
        break;

      case 1:
        snprintf( buff, s,
            "-385   0  translate\nqslcard\n\n" );
        if( ! File_Print(&rc_data.station_qsl_fp, buff) )
          return( FALSE );
        break;

      case 2: case 4:
        snprintf( buff, s,
            "0   186 translate\nqslcard\n\n" );
        if( ! File_Print(&rc_data.station_qsl_fp, buff) )
          return( FALSE );
        break;

      case 3:
        snprintf( buff, s,
            "385  0 translate\nqslcard\n\n" );
        if( ! File_Print(&rc_data.station_qsl_fp, buff) )
          return( FALSE );
        break;

      case 5:
        snprintf( buff, s,
            "-385   0  translate\nqslcard\nshowpage\n\n" );
        if( ! File_Print(&rc_data.station_qsl_fp, buff) )
          return( FALSE );
        break;

      default:
        puts( _("Error making QSL cards") );
        return( FALSE );
    }

    /* Print QSO record to adif format log also */
    Truncate_Spaces( qso_record.dx_call );
    Truncate_Spaces( qso_record.freq );
    Truncate_Spaces( qso_record.mode );

    snprintf( buff, s,
        "<CALL:%d>%s<QSO_DATE:8>%s<TIME_ON:4>%s\n"
        "<FREQ:%d>%s<MODE:%d>%s<RST_SENT:%d>%s<EOR>\n\n",
        (int)strlen(qso_record.dx_call),qso_record.dx_call,
        qso_record.date_adif,           qso_record.time_adif,
        (int)strlen(qso_record.freq),   qso_record.freq,
        (int)strlen(qso_record.mode),   qso_record.mode,
        (int)strlen(qso_record.dx_rst), qso_record.dx_rst );
    if( ! File_Print(&rc_data.log_adif_fp, buff) )
      return( FALSE );

    num_qsled++;
  }

  /* Go to beginning of QSL card postscript file */
  if( !fseek(rc_data.station_qsl_fp, 3, SEEK_SET) )
    return( FALSE );

  /* Print number of QSL cards to file */
  snprintf( buff, s,
      _("%% Number of QSL Cards:%-6d"), num_cards );
  if( ! File_Print(&rc_data.station_qsl_fp, buff) )
    return( FALSE );

  /* Go back to beginning of log and print new data */
  rewind( rc_data.station_log_fp );
  if( ! Find_Line( &line_idx, buff,
        _(_("Num of QSO Records:")), rc_data.station_log_fp) )
    return( FALSE );
  long sk = (long)(strlen(buff)+1);
  if( !fseek(rc_data.station_log_fp, -sk, SEEK_CUR) )
    return( FALSE );

  /* Mark as QSL'd */
  snprintf( buff, s,
      _("|  Num of QSO Records:%-6d  Records QSL'd:%-6d"\
      "  Size of Records:%-4d   |\n"\
      "|------------------------------------"\
      "--------------------------------------|\n"),
      num_records, num_qsled, (int)s );
  if( ! File_Print(&rc_data.station_log_fp, buff) )
    return( FALSE );

  Close_File( &rc_data.station_log_fp );
  Close_File( &rc_data.station_qsl_fp );
  Close_File( &rc_data.log_adif_fp );

  return( TRUE );

} /* Make_QSL_Cards() */

/*------------------------------------------------------------------------*/

/*  Get_QSO_Record()
 *
 *  Gets a QSO record from the station log file
 */

  gboolean
Get_QSO_Record( FILE *fp, gboolean qsl )
{
  char
    line[81],
    *line_idx;


  /*** Read lines from log file and enter data to qso_record ***/
  if( ! Find_Line(&line_idx, line, _("DX/ CALL: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.dx_call, line_idx, sizeof(qso_record.dx_call) );

  if( ! Find_String(&line_idx, line, _("NAME: ")) )
    return( FALSE );
  Strlcpy( qso_record.dx_name, line_idx, sizeof(qso_record.dx_name) );

  if( ! Find_String(&line_idx, line, _("QTH: ")) )
    return( FALSE );
  Strlcpy( qso_record.dx_qth, line_idx, sizeof(qso_record.dx_qth) );

  if( ! Find_String(&line_idx, line, _("LOC: ")) )
    return( FALSE );
  Strlcpy( qso_record.dx_loc, line_idx, sizeof(qso_record.dx_loc) );

  if( ! Find_Line(&line_idx, line, _("MY/ CALL: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.my_call, line_idx, sizeof(qso_record.my_call) );

  if( ! Find_String(&line_idx, line, _("ZONE: ")) )
    return( FALSE );
  Strlcpy( rc_data.my_zone, line_idx, sizeof(rc_data.my_zone) );

  if( ! Find_String(&line_idx, line, _("QTH: ")) )
    return( FALSE );
  Strlcpy( qso_record.my_qth, line_idx, sizeof(qso_record.my_qth) );

  if( ! Find_String(&line_idx, line, _("LOC: ")) )
    return( FALSE );
  Strlcpy( qso_record.my_loc, line_idx, sizeof(qso_record.my_loc) );

  if( ! Find_Line(&line_idx, line, _("QSO DATE: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.date, line_idx, sizeof(qso_record.date) );

  if( ! Find_String(&line_idx, line, _("TIME: ")) )
    return( FALSE );
  Strlcpy( qso_record.time, line_idx, sizeof(qso_record.time) );

  if( ! Find_String(&line_idx, line, _("FREQ: ")) )
    return( FALSE );
  Strlcpy( qso_record.freq, line_idx, sizeof(qso_record.freq) );

  /* Mark record as QSL'd */
  if( qsl )
  {
    if( ! Find_String(&line_idx, line, _("QSL: ")) )
      return( FALSE );

    /* Enter QSL: YES in the log file */
    long sk = (long)(1 + strlen(line_idx));
    if( !fseek(fp, -sk, SEEK_CUR) )
      return( FALSE );

    if( ! File_Print(&fp, _("YES")) )
      return( FALSE );
  }

  if( ! Find_Line(&line_idx, line, _("QSO MODE: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.mode, line_idx, sizeof(qso_record.mode) );

  if( ! Find_String(&line_idx, line, _("DX-REPT: ")) )
    return( FALSE );
  Strlcpy( qso_record.dx_rst, line_idx, sizeof(qso_record.dx_rst) );

  if( ! Find_String(&line_idx, line, _("MY-REPT: ")) )
    return( FALSE );
  Strlcpy( qso_record.my_rst, line_idx, sizeof(qso_record.my_rst) );

  if( ! Find_String(&line_idx, line, _("VIA: ")) )
    return( FALSE );
  Strlcpy( qso_record.via, line_idx, sizeof(qso_record.via) );

  if( ! Find_Line(&line_idx, line, _("MY TRANS: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.tx, line_idx, sizeof(qso_record.tx) );

  if( ! Find_String(&line_idx, line, _("POWER: ")) )
    return( FALSE );
  Strlcpy( qso_record.tx_power, line_idx, sizeof(qso_record.tx_power) );

  if( ! Find_String(&line_idx, line, _("ANT: ")) )
    return( FALSE );
  Strlcpy( qso_record.tx_ant, line_idx, sizeof(qso_record.tx_ant) );

  if( ! Find_Line(&line_idx, line, _("MY RECVR: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.rx, line_idx, sizeof(qso_record.rx) );

  if( ! Find_String(&line_idx, line, _("N.FIG: ")) )
    return( FALSE );
  Strlcpy( qso_record.rx_nfig, line_idx, sizeof(qso_record.rx_nfig) );

  if( ! Find_String(&line_idx, line, _("ANT: ")) )
    return( FALSE );
  Strlcpy( qso_record.rx_ant, line_idx, sizeof(qso_record.rx_ant) );

  if( ! Find_Line(&line_idx, line, _("REMARKS: "), fp) )
    return( FALSE );
  Strlcpy( qso_record.remarks, line_idx, sizeof(qso_record.remarks) );

  /* Enter date and time in adif format */
  {
    int idx;
    char *month[12] =
    { _("Jan"),_("Feb"),_("Mar"),_("Apr"),_("May"),_("Jun"),
      _("Jul"),_("Aug"),_("Sep"),_("Oct"),_("Nov"),_("Dec") };

    /* Enter date in ADIF format */
    Strlcpy( qso_record.date_adif, &qso_record.date[7], 5 );
    for( idx = 0; idx < 12; idx++ )
      if( strncmp(month[idx], &qso_record.date[3], 3) == 0 )
        break;
    snprintf( &qso_record.date_adif[4], 3, "%02d", idx+1 );
    Strlcat( qso_record.date_adif, qso_record.date, 3 );

    /* Enter time in adif format */
    Strlcpy( qso_record.time_adif,  qso_record.time, 3 );
    Strlcat( qso_record.time_adif, &qso_record.time[3], 3 );
  }

  return( TRUE );

} /* Get_QSO_Record() */

/*------------------------------------------------------------------------*/

/*  Clear_QSO_Record()
 *
 *  Clears all QSO Record fields
 */

  void
Clear_QSO_Record( gboolean all )
{
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "dx_call")),    "" );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "dx_rst")), "" );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "my_rst")), "" );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "dx_name")),    "" );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "dx_qth")), "" );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "dx_loc")), "" );

  if( all )
  {
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "freq")), "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "mode")), "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "via")),  "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "day")),  "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "month")),    "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "year")), "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "hour")), "" );
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(main_window_builder, "minute")),"" );
  }

  ClearFlag( SAVE_RECORD );

} /* Clear_QSO_Record() */

/*------------------------------------------------------------------------*/

/*  Default_Rig_Data()
 *
 *  Enter default Rig data
 */

  void
Default_Rig_Data( void )
{
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "tx")),
      rc_data.tx );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "tx_power")),
      rc_data.tx_power );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "tx_ant")),
      rc_data.tx_ant );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "rx")),
      rc_data.rx );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "rx_nfig")),
      rc_data.rx_nfig );
  gtk_entry_set_text( GTK_ENTRY(
        Builder_Get_Object(main_window_builder, "rx_ant")),
      rc_data.rx_ant );

} /* Clear_Rig_Data() */

/*------------------------------------------------------------------------*/

/*  Enter_Date()
 *
 *  Enters UTC date and time to date fields
 */

  void
Enter_Date( void )
{
  /* Variables for time and date */
  time_t tp;    /* Time type     */
  struct tm dt; /* Date and time */

  gchar buff[5];

  GtkWidget *entry;

  /* Enter time and date to QSO data */
  if( isFlagSet(NEW_DATE) )
  {
    time( &tp );
    dt = *gmtime( &tp );
    strftime( qso_record.date_adif, 9, "%Y%m%d",   &dt );
    strftime( qso_record.time_adif, 5, "%H%M",     &dt );

    entry = Builder_Get_Object( main_window_builder, "day" );
    strftime( buff, 3, "%d", &dt );
    gtk_entry_set_text( GTK_ENTRY(entry), buff );
    entry = Builder_Get_Object( main_window_builder, "month" );
    strftime( buff, 3, "%m", &dt );
    gtk_entry_set_text( GTK_ENTRY(entry), buff );
    entry = Builder_Get_Object( main_window_builder, "year" );
    strftime( buff, 5, "%Y", &dt );
    gtk_entry_set_text( GTK_ENTRY(entry), buff );
    entry = Builder_Get_Object( main_window_builder, "hour" );
    strftime( buff, 3, "%H", &dt );
    gtk_entry_set_text( GTK_ENTRY(entry), buff );
    entry = Builder_Get_Object( main_window_builder, "minute" );
    strftime( buff, 3, "%M", &dt );
    gtk_entry_set_text( GTK_ENTRY(entry), buff );
  }
}

/*------------------------------------------------------------------------*/

/*  Process_RST()
 *
 *  Processes RST-out and RST-in fields
 */

  void
Process_RST( GtkEditable *editable )
{
  const gchar *rst;
  int len, idx;

  /* Validate RST: numerics only, */
  /* min 337 max 559, ignore blank field */
  rst = gtk_entry_get_text( GTK_ENTRY(editable) );
  len = (int)strlen(rst);

  /* Ignore blank fields */
  if( len == 0 )
    return;

  for( idx = 0; idx < len; idx++ )
  {
    /* Reject non-numbers */
    if( (rst[idx] < '0') ||
        (rst[idx] > '9') )
    {
      Bad_Entry_Dialog(
          _("Invalid character entered\n"\
            "Numbers only allowed") );
      return;
    }

    /* Reject RST < 337 or > 599 */
    switch( idx )
    {
      case 0:
        if( (rst[idx] < '3') ||
            (rst[idx] > '5') )
        {
          Bad_Entry_Dialog(
              _("Number out of range\n"\
                "Min = 3 and Max = 5") );
          return;
        }
        break;

      case 1:
        if( rst[idx] < '3' )
        {
          Bad_Entry_Dialog(
              _("Number out of range\n"\
                "Min = 3 and Max = 9") );
          return;
        }
        break;

      case 2:
        if( rst[idx] < '7' )
        {
          Bad_Entry_Dialog(
              _("Number out of range\n"\
                "Min = 7 and Max = 9") );
          return;
        }
    } /* switch( idx ) */

  } /* for( idx = 0; idx < len; idx++ ) */

} /* Process_RST() */

/*------------------------------------------------------------------------*/

/*  Process_Locator()
 *
 *  Processes QTH locator data
 */

  void
Process_Locator( GtkEditable *editable )
{
  int len, idx;
  char buff[7];

  /* Get entry field, ignore blank field */
  Strlcpy( buff, gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(buff) );
  len = (int)strlen(buff);
  if( len == 0 ) return;

  for( idx = 0; idx < len; idx++ )
  {
    /* Capitalize letters */
    if( (buff[idx] > 0x60) &&
        (buff[idx] < 0x7b) )
      buff[idx] -= 0x20;

    /* Validate grid locator */
    switch( idx )
    {
      case 0: case 1: /* First letters */
        if( (buff[idx] < 'A') ||
            (buff[idx] > 'S') )
        {
          Bad_Entry_Dialog(
              _("Invalid character entered\n"\
                "Letters 'A' to 'S' only") );
          return;
        }
        break;

      case 2: case 3: /* Middle numbers */
        if( (buff[idx] < '0') ||
            (buff[idx] > '9') )
        {
          Bad_Entry_Dialog(
              _("Invalid character entered\n"\
                "Numbers only allowed") );
          return;
        }
        break;

      case 4: case 5: /* Last letters */
        if( (buff[idx] < 'A') ||
            (buff[idx] > 'X') )
        {
          Bad_Entry_Dialog(
              _("Invalid character entered\n"\
                "Letters 'A' to 'X' only") );
          return;
        }

    } /* switch( idx ) */

  } /* for( idx = 0; idx < len; idx++ ) */

  gtk_entry_set_text( GTK_ENTRY(editable), buff );

} /* Process_Locator() */

/*------------------------------------------------------------------------*/

/*  Process_Callsign()
 *
 *  Processes station callsigns
 */

  void
Process_Callsign( GtkEditable *editable )
{
  int len, idx;
  char buff[15];

  /* Get entry field, ignore blank field */
  Strlcpy( buff, gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(buff) );
  len = (int)strlen( buff );
  if( len == 0 )
    return;

  for( idx = 0; idx < len; idx++ )
  {
    /* Capitalize letters */
    if( (buff[idx] > 0x60) &&
        (buff[idx] < 0x7b) )
      buff[idx] -= 0x20;

    /* Allow only alpha-numerics and '/' */
    if( ((buff[idx]  < 'A') || (buff[idx]  > 'Z')) &&
        ((buff[idx]  < '0') || (buff[idx]  > '9')) &&
        (buff[idx] != '/') )
    {
      Bad_Entry_Dialog(
          _("Invalid character entered\n"\
            "Alpha-numerics and '/' only") );
      return;
    }

  } /* for( idx = 0; idx < len; idx++ ) */

  gtk_entry_set_text( GTK_ENTRY(editable), buff );

} /* Process_Callsign() */

/*------------------------------------------------------------------------*/

/*  Capitalize_Entry()
 *
 *  Capitalizes letters in a GtkEntry
 */

  void
Capitalize_Entry( GtkEditable *editable )
{
  char buff[13];
  int len;

  /* Read entry, ignore blank field */
  Strlcpy( buff, gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(buff) );
  len = (int)strlen(buff);
  if( len == 0 ) return;

  /* Capitalize letters if enabled */
  if( isFlagSet(CAPITALIZE) )
  {
    int idx;
    for( idx = 0; idx < len; idx++ )
      if( (buff[idx] > 0x60) &&
          (buff[idx] < 0x7b) )
        buff[idx] -= 0x20;
  }

  /* Enter data to field QSO record structure */
  gtk_entry_set_text( GTK_ENTRY(editable), buff );

} /* Capitalize_Entry() */

/*------------------------------------------------------------------------*/

/*  Is_Numerical()
 *
 *  Checks if an entry is numerical
 */

  void
Is_Numerical( GtkEditable *editable )
{
  int len, idx = 0;
  const gchar *buff;

  buff = gtk_entry_get_text(GTK_ENTRY(editable));
  len = (int)strlen( buff );
  if( len == 0 )
    return;

  /* Ignore leading spaces */
  while( buff[idx++] == ' ' );

  /* Check if numerical */
  for( ; idx < len; idx++ )
    if( ((buff[idx]  < '0') || (buff[idx]  > '9')) &&
        ((buff[idx] != '.') && (buff[idx] != ',')) )
    {
      Bad_Entry_Dialog(
          _("Invalid character entered\n"\
            "Numbers and '.' or ',' only allowed") );
      break;
    }

} /* Is_Numerical() */

/*------------------------------------------------------------------------*/

/*  Close_File()
 *
 *  Closes and NULL's an open file pointer
 */

  void
Close_File( FILE **fp )
{
  if( *fp != NULL )
  {
    fclose( *fp );
    *fp = NULL;
  }
}

/*------------------------------------------------------------------------*/

/* Import_Log()
 *
 * Imports a log file from other apps
 */
  gboolean
Import_Log( FILE *fp )
{
  /* Buffer for reading and printing to files */
  char buff[REC_BUFF_SIZE];
  char *line_idx;

  int
    num_records,   /* Number of QSO records in station's log file   */
    record_size,   /* Size in byte of QSO records in station's log  */
    num_logged,    /* Number of QSO's in imported log copied to log */
    records_tolog; /* Number of QSL "cards" to add to the QSL file  */

  /* Find line with "Num of QSO Records:" */
  if( !Find_Line(&line_idx, buff, _("QSO Records:"), fp) )
    return( FALSE );
  num_records = atoi( line_idx );

  /* Find "Records Logged:" string */
  if( !Find_String(&line_idx, buff, _("Records Logged:")) )
    return( FALSE );
  num_logged = atoi( line_idx );

  /* Number of records to copy to log */
  records_tolog = num_records - num_logged;
  if( ! records_tolog )
  {
    Error_Dialog(
        _("Imported log shows all QSOs\n"\
          "already copied to Station Log"), OK );
    return( TRUE );
  }

  /* Find "Size of Records:" string */
  if( !Find_String(&line_idx, buff, _("Size of Records:")) )
    return( FALSE );
  record_size = atoi( line_idx );

  /* Go to first non-copied QSO record */
  if( !fseek(fp, -(1+records_tolog*record_size), SEEK_END) )
  /*** Copy records from imported log to station log ***/
  do
  {
    if( !Get_QSO_Record(fp, FALSE) )
      return( FALSE );
    Save_QSO_Record();
  }
  while( ++num_logged != num_records );

  /* Go back to beginning of log and print new data */
  rewind( fp );
  if( !Find_Line( &line_idx, buff, _("Num of QSO Records:"), fp) )
    return( FALSE );

  long sk = (long)(strlen(buff)+1);
  if( !fseek(fp, -sk, SEEK_CUR) )
    return( FALSE );
  snprintf( buff, REC_BUFF_SIZE,
      _("|  Num of QSO Records:%-6d  Records Logged:%-6d"\
      "  Size of Records:%-4d  |\n"\
      "|------------------------------------"\
      "--------------------------------------|\n"),
      num_records, num_logged, RECORD_SIZE );
  if( !File_Print(&fp, buff) )
    return( FALSE );

  return( TRUE );
} /* Import_Log( FILE *fp ) */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flow control flags
 *
 *  See xhamlog.h for definition of flow control flags
 */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet( int flag )
{
  return( (Flags & flag) == flag );
}

  int
isFlagClear( int flag )
{
  return( (~Flags & flag) == flag );
}

  void
SetFlag( int flag )
{
  Flags |= flag;
}

  void
ClearFlag( int flag )
{
  Flags &= ~flag;
}

  void
ToggleFlag( int flag )
{
  Flags ^= flag;
}

/*------------------------------------------------------------------------*/

/*  Cleanup()
 *
 *  Cleans up before quitting
 */

  void
Cleanup( void )
{
  /* Save outstanding records */
  if( isFlagSet(SAVE_RECORD) && Read_QSO_Record() )
    Save_QSO_Record();

  /* Close open files */
  Close_File( &rc_data.log_adif_fp );
  Close_File( &rc_data.station_log_fp );
  Close_File( &rc_data.station_qsl_fp );

  /* End CAT if enabled */
  Close_Tcvr_Serial();

} /* Cleanup( void ) */

/*------------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;

  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
    dest[idx] = src[idx];
    idx++;
    ch = src[idx];
    n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*------------------------------------------------------------------*/

/* Strlcat()
 *
 * Concatenates at most n-1 chars from src string into dest string.
 * Unlike other such library fuctions, this makes sure that the dest
 * string is null terminated by copying only n-1 chars to leave room
 * for the terminating char. n would normally be the sizeof(dest)
 * string but copying will not go beyond the terminating null of src
 */
  void
Strlcat( char *dest, const char *src, size_t n )
{
  char ch = dest[0];
  int idd = 0; /* dest index */
  int ids = 0; /* src  index */

  /* Find terminating null of dest */
  while( (n > 0) && (ch != '\0') )
  {
    idd++;
    ch = dest[idd];
    n--; /* Count remaining char's in dest */
  }

  /* Copy n-1 chars to leave room for terminating null */
  n--;
  ch = src[ids];
  while( (n > 0) && (ch != '\0') )
  {
    dest[idd] = src[ids];
    ids++;
    ch = src[ids];
    idd++;
    n--;
  }

  /* Terminate dest string */
  dest[idd] = '\0';

} /* Strlcat() */

/*------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n",
      _("Usage: xhamlog [-hv]") );

  fprintf( stderr, "%s\n",
      _("       -h: Print this usage information and exit"));

  fprintf( stderr, "%s\n",
      _("       -v: Print version number and exit"));

} /* End of Usage() */

/*------------------------------------------------------------------------*/

