/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "main.h"
#include "shared.h"

/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

int
main (int argc, char *argv[])
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;


  gtk_init (&argc, &argv);

  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /* Process command line options */
  while( (option = getopt(argc, argv, "hv") ) != -1 )
    switch( option )
    {
      case 'h': /* Print usage and exit */
        Usage();
        exit( 0 );

      case 'v': /* Print version */
        puts( PACKAGE_STRING );
        exit( 0 );

      default: /* Print usage and exit */
        Usage();
        exit( -1 );

    } /* End of switch( option ) */

  /* Initialize (clear) file pointers */
  rc_data.station_log_fp = NULL;
  rc_data.station_qsl_fp = NULL;
  rc_data.log_adif_fp    = NULL;

  /* Create file path to xhamlog glade file */
  Strlcpy( rc_data.xhamlog_glade,
      getenv("HOME"), sizeof(rc_data.xhamlog_glade) );
  Strlcat( rc_data.xhamlog_glade,
      "/xhamlog/xhamlog.glade", sizeof(rc_data.xhamlog_glade) );

  /* Check for the Glade config file */
  FILE *fp = fopen( rc_data.xhamlog_glade, "r" );
  if( fp == NULL )
  {
    fprintf( stderr, "xhamlog: cannot open xhamlog Glade GUI Description file.\n" );
    perror( rc_data.xhamlog_glade );
    fprintf( stderr, "xhamlog: trying to create xhamlog config directory "
                     "from the installation prefix file tree.\n" );

    /* Find the binary's path (location in file system) */
    char exe_path[256], file_path[288];
    
    /* Read the file path to xhamlog executable */
    size_t len = sizeof( exe_path );
    int bytes = (int)readlink( "/proc/self/exe", exe_path, len );
    if( bytes <= 0 )
    {
      fprintf( stderr, "xhamlog: cannot read xhamlog binary's location.\n" );
      perror( "/proc/self/exe" );
      exit( -1 );
    }

    /* Remove "/bin/xhamlog" from the path with room for termination */
    bytes -= sizeof( "/bin/xhamlog" ) - 1;
    if( bytes < 1 )
    {
      fprintf( stderr, "xhamlog: cannot create file path to examples/xhamlog.\n" );
      exit( -1 );
    }

    /* Create file path to xhamlog examples directory */
    exe_path[bytes] = '\0';
    Strlcpy( file_path, exe_path, sizeof(file_path) );
    Strlcat( file_path, "/share/examples/xhamlog", sizeof(file_path) );
    fprintf( stderr, "xhamlog: creating xhamlog config directory from: %s\n", file_path );

    /* Create system command to copy examples/xhamlog to ~/.xhamlog */
    char syscmnd[512];
    Strlcpy( syscmnd, "cp -r ", sizeof(syscmnd) );
    Strlcat( syscmnd, file_path, sizeof(syscmnd) );
    Strlcat( syscmnd, " ", sizeof(syscmnd) );
    Strlcat( syscmnd, getenv("HOME"), sizeof(syscmnd) );
    Strlcat( syscmnd, "/xhamlog",   sizeof(syscmnd) );
    int ret = system( syscmnd );
    if( ret == -1 )
    {
      fprintf( stderr, "xhamlog: cannot create xhamlog's working directory.\n" );
      perror( file_path );
      exit( -1 );
    }
  } /* if( fp == NULL ) */
  else fclose( fp );

  /* Create xhamlog main window */
  main_window = create_main_window( &main_window_builder );
  gtk_window_set_title( GTK_WINDOW(main_window), PACKAGE_STRING );
  gtk_widget_show (main_window);
  popup_menu = create_popup_menu( &popup_menu_builder );

  SetFlag( CAPITALIZE | NEW_DATE | NEW_FREQ | NEW_MODE );
  g_idle_add( Load_Config, main_window );

  gtk_main ();

  return( 0 );
}

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  /* Wrap up and quit */
  Cleanup();
  fprintf( stderr, "\n" );
  switch( signal )
  {
    case SIGINT :
      fprintf( stderr, "%s\n",  _("xhamlog: Exiting via User Interrupt") );
      exit( signal );

    case SIGSEGV :
      fprintf( stderr, "%s\n",  _("xhamlog: Segmentation Fault") );
      exit( signal );

    case SIGFPE :
      fprintf( stderr, "%s\n",  _("xhamlog: Floating Point Exception") );
      exit( signal );

    case SIGABRT :
      fprintf( stderr, "%s\n",  _("xhamlog: Abort Signal received") );
      exit( signal );

    case SIGTERM :
      fprintf( stderr, "%s\n",  _("xhamlog: Termination Request received") );
      exit( signal );
  }

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

