/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "cat.h"
#include "shared.h"


/* Serial port File descriptor */
static int serial_fd = 0;

/* Original serial port options */
static struct termios serial_old_options;

static char
  /* Command codes for Yaesu FT847/FT857 CAT */
  FT847_CAT_ON[]    = { 0x00, 0, 0, 0, (char)0x00 }, /*  Enable CAT */
  FT847_CAT_OFF[]   = { 0x00, 0, 0, 0, (char)0x80 }, /* Disable CAT */
  FT847_RX_STATUS[] = { 0x00, 0, 0, 0, (char)0xE7 }, /* Request RX status  */
  FT847_MAIN_VFO[]  = { 0x00, 0, 0, 0, (char)0x03 }, /* Read main VFO freq */
  FT847_PTT_OFF[]   = { 0x00, 0, 0, 0, (char)0x88 }, /* PTT Off (receive)  */

  /* Command codes for Elecraft K3 */
  K3_K31_S[]    = "K31;",           /* Set K3 CAT mode */
  K3_K31_G[]    = "K3;",            /* Request K3 CAT mode */
  K3_SMTR_G[]   = "SM;",            /* Request K3 S-meter  */
  K3_SMTR_S[]   = "SMnnnn;",        /* K3 S-meter data */
  K3_VFOA_S[]   = "FAnnnnnnnnnnn;", /* Set VFO A frequency */
  K3_VFOA_G[]   = "FA;",            /* Get VFO A frequency */
  K3_PTT_OFF[]  = "RX;",            /* PTT Off (receive) */
  K3_MODE_S[]   = "MDn;",           /* Get transceiver mode */
  K3_MODE_G[]   = "MD;";            /* Get transceiver mode */

/*-------------------------------------------------------------------------*/

/*  Read_Tcvr_Serial()
 *
 *  Reading Data from the Tcvr's Serial Port
 */

  static void
Read_Tcvr_Serial( char *data, size_t len )
{
  ssize_t ret = read( serial_fd, data, len );

  /* Error condition */
  if( (size_t)ret != len )
  {
    tcflush( serial_fd, TCIFLUSH );
    fprintf( stderr,
        "xhamlog: read() return value wrong: %d/%d\n",
        (int)ret, (int)len );
    Error_Dialog( _("Serial port read() I/O error"), OK );
    return;
  }

  return;
} /* End of Read_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Tcvr_Serial()
 *
 *  Writes a command to the Tranceiver
 */

  static void
Write_Tcvr_Serial( char *data, size_t len )
{
  tcflush( serial_fd, TCIOFLUSH );
  ssize_t ret = write( serial_fd, data, len );

  /* Error condition */
  if( (size_t)ret != len )
  {
    tcflush( serial_fd, TCOFLUSH );
    fprintf( stderr,
        "xhamlog: write() return value wrong: %d/%d\n",
        (int)ret, (int)len );
    Error_Dialog( _("Serial port write() I/O error"), OK );
    return;
  }

  /* Give time to CAT to do its thing */
  usleep(50000);

  return;
} /* End of Write_Tcvr_Command() */

/*-------------------------------------------------------------------------*/

/*  Close_Tcvr_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Tcvr_Serial( void )
{
  if( isFlagSet(CAT_SETUP) )
  {
      switch( rc_data.tcvr_type )
      {
        /* Disable CAT for FT847 */
        case FT847:
          Write_Tcvr_Serial( FT847_PTT_OFF, sizeof(FT847_PTT_OFF) );
          Write_Tcvr_Serial( FT847_CAT_OFF, sizeof(FT847_CAT_OFF) );
          break;

        case FT857:
          Write_Tcvr_Serial( FT847_PTT_OFF, sizeof(FT847_PTT_OFF) );
          break;

        /* Return K2/K3 to original status */
        case K2: case K3:
          Write_Tcvr_Serial( K3_PTT_OFF, sizeof(K3_PTT_OFF)-1 );
          break;
      }

    ClearFlag( CAT_SETUP );

    if( serial_fd > 0 )
    {
      tcsetattr( serial_fd, TCSANOW, &serial_old_options );
      close( serial_fd );
      serial_fd = 0;
    }
  } /* if( isFlagSet(CAT_SETUP) || isFlagSet(RTS_SETUP) ) */

} /* End of Close_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Open_Tcvr_Serial()
 *
 *  Opens Tcvr's Serial Port device, returns the file
 *  descriptor serial_fd on success or exits on error
 */

  gboolean
Open_Tcvr_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */
  int error = 0;

  /* Abort if serial already open */
  if( isFlagSet(CAT_SETUP) ) return( TRUE );

  /* Open CAT serial port, exit on error */
  serial_fd = open( rc_data.cat_serial, O_RDWR | O_NOCTTY );
  if( serial_fd < 0 )
  {
    char mesg[MESG_SIZE];
    perror( rc_data.cat_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    snprintf( mesg, sizeof(mesg),
        _("Failed to open %s\n"\
          "CAT not enabled"), rc_data.cat_serial );
    Error_Dialog( mesg, OK );
    return( FALSE );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  if( fcntl( serial_fd, F_SETLK, &lockinfo ) == -1 )
  {
    char mesg[MESG_SIZE];
    perror( rc_data.cat_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    if( fcntl(serial_fd, F_GETLK, &lockinfo) != -1 )
    {
      snprintf( mesg, sizeof(mesg),
          _("Failed to lock %s\n"\
            "Device locked by pid %d\n"\
            "CAT not enabled"),
          rc_data.cat_serial, lockinfo.l_pid );
      Error_Dialog( mesg, OK );
    }
    return( FALSE );
  }

  /* Save the current serial port options */
  error |= tcgetattr( serial_fd, &serial_old_options );

  /* Read the current serial port options */
  error |= tcgetattr( serial_fd, &new_options );

  /* Set the i/o baud rates */
  switch( rc_data.tcvr_type )
  {
    case FT847:
      error |= cfsetspeed( &new_options, B57600 );
      break;

    case FT857:
      error |= cfsetspeed( &new_options, B38400 );
      break;

    case K2: case K3:
      error |= cfsetspeed( &new_options, B38400 );
      break;
  }

  /* Set options for 'raw' I/O mode */
  cfmakeraw( &new_options );

  /* Setup read() timeout to .2 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 2;

  /* Setup the new options for the port */
  error |= tcsetattr( serial_fd, TCSAFLUSH, &new_options );
  if( error )
  {
    if( serial_fd > 0 ) Close_Tcvr_Serial();
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    Error_Dialog( _("Failed to Enable CAT"), OK );
    return( FALSE );
  }

  /* Enable CAT by testing comms with transceiver */
  switch( rc_data.tcvr_type )
  {
    case FT847:
      {
        char test;
        Write_Tcvr_Serial( FT847_CAT_ON, sizeof(FT847_CAT_ON) );
        Write_Tcvr_Serial( FT847_RX_STATUS, sizeof(FT847_RX_STATUS) );
        Read_Tcvr_Serial( &test, 1 );
        SetFlag( CAT_SETUP );
      }
      break;

    case FT857:
      {
        char test;
        Write_Tcvr_Serial( FT847_RX_STATUS, sizeof(FT847_RX_STATUS) );
        Read_Tcvr_Serial( &test, 1 );
        SetFlag( CAT_SETUP );
      }
      break;

    case K2:
      /* Get K2 status as a test of serial port */
      Write_Tcvr_Serial( K3_VFOA_G, sizeof(K3_VFOA_G)-1 );
      Read_Tcvr_Serial(K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      SetFlag( CAT_SETUP );
      break;

    case K3:
      /* Set K3 to mode K31 as a test of serial port */
      Write_Tcvr_Serial( K3_K31_S, sizeof(K3_K31_S)-1 );
      Write_Tcvr_Serial( K3_K31_G, sizeof(K3_K31_G)-1 );
      Read_Tcvr_Serial( K3_K31_S, sizeof(K3_K31_S)-1 );
      if( strncmp(K3_K31_S, "K31;", sizeof(K3_K31_S)-1 ) == 0 )
        SetFlag( CAT_SETUP );
      else
      {
        tcsetattr( serial_fd, TCSANOW, &serial_old_options );
        close( serial_fd );
        serial_fd = 0;
        rc_data.tcvr_type = NONE;
        gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
              Builder_Get_Object(popup_menu_builder, "none")), TRUE );
        return( FALSE );
      }

      break;
  } /* switch( rc_data.tcvr_type ) */

  return( TRUE );
} /* End of Open_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Rx_Status()
 *
 *  Reads the transceiver's status (Tx/Rx freq,
 *  mode, po/s-meter etc) from the Transceiver.
 */

  gboolean
Read_Rx_Status( char *mode, char *freq, char *rst )
{
  /* A 5-char string for sending and receiving  */
  /* 5-byte strings to and from the transceiver */
  char cmnd_parm[5];
  int frq = 0, smt = 0, k3_md = 0, idx;

  char *mode_names[NUM_MODE_CODES+1] = { TCVR_MODE_NAMES };

  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  /* Read Rx status */
  switch( rc_data.tcvr_type )
  {
    case FT847: case FT857:
      /* Read Rx status (S-meter) */
      bzero( cmnd_parm, sizeof(cmnd_parm) );
      Write_Tcvr_Serial( FT847_RX_STATUS, sizeof(FT847_RX_STATUS) );
      Read_Tcvr_Serial( cmnd_parm, 1 );

      /* Mask out S-meter */
      if( rc_data.tcvr_type == FT847 )
      {
        smt = cmnd_parm[0] & FT847_SMETER;
        smt = smt/4 + 3;
      }
      else
      {
        smt = cmnd_parm[0] & FT857_SMETER;
        smt = smt/2 + 3;
      }

      /* Keep S-meter report in range 3-9 */
      if( smt > 9 ) smt = 9;

      /* Read Main VFO frequency */
      Write_Tcvr_Serial( FT847_MAIN_VFO, sizeof(FT847_MAIN_VFO) );
      Read_Tcvr_Serial( cmnd_parm, sizeof(cmnd_parm) );

      /* Enter operating mode */
      idx = NUM_MODE_CODES;
      if( rc_data.tcvr_type == FT847 )
      {
        char mode_codes[NUM_MODE_CODES] = { FT847_MODE_CODES };
        for( idx = 0; idx < NUM_MODE_CODES; idx++ )
          if( cmnd_parm[4] == mode_codes[idx] )
            break;
      }
      else if( rc_data.tcvr_type == FT857 )
      {
        char mode_codes[NUM_MODE_CODES] = { FT857_MODE_CODES };
        for( idx = 0; idx < NUM_MODE_CODES; idx++ )
          if( cmnd_parm[4] == mode_codes[idx] )
            break;
      }
      Strlcpy( mode, mode_names[idx], 9 ); /* For null */

      /* Decode frequency data to 1 kHz */
      frq = 0;
      frq += (cmnd_parm[0] & 0xF0) >> 4;
      frq *= 10;
      frq += cmnd_parm[0] & 0x0F;
      frq *= 10;
      frq += (cmnd_parm[1] & 0xF0) >> 4;
      frq *= 10;
      frq += cmnd_parm[1] & 0x0F;
      frq *= 10;
      frq += (cmnd_parm[2] & 0xF0) >> 4;
      frq *= 10;
      frq += cmnd_parm[2] & 0x0F;
      break;

    case K2: case K3:
      {
        char mode_codes[NUM_MODE_CODES] = { K3_MODE_CODES };

        /* Read Main VFO frequency */
        Write_Tcvr_Serial( K3_VFOA_G, sizeof(K3_VFOA_G)-1 );
        Read_Tcvr_Serial(  K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
        frq = atoi( K3_VFOA_S+2 );
        frq += 500;
        frq /= 1000;

        /* Get K3 mode */
        Write_Tcvr_Serial( K3_MODE_G, sizeof(K3_MODE_G)-1 );
        Read_Tcvr_Serial(  K3_MODE_S, sizeof(K3_MODE_S)-1 );
        k3_md = atoi( K3_MODE_S+2 );
        for( idx = 0; idx < NUM_MODE_CODES; idx++ )
          if( k3_md == mode_codes[idx] )
            break;
        Strlcpy( mode, mode_names[idx], 9 );

        /* Read S-meter */
        Write_Tcvr_Serial( K3_SMTR_G, sizeof(K3_SMTR_G)-1 );
        Read_Tcvr_Serial(  K3_SMTR_S, sizeof(K3_SMTR_S)-1 );
        smt = atoi( K3_SMTR_S+2 );
        if( rc_data.tcvr_type == K3 )
        {
          smt += 9;
          smt = smt / 3;
        }
        else
        {
          smt += 6;
          smt /= 2;
        }
        break;
      }
  } /* switch( rc_data.tcvr_type ) */

  /* Enter freq and S-meter data */
  snprintf( freq, 8, "%4.3f", (double)frq / 1000.0 );
  if( smt > 9 ) smt = 9;
  smt = 500 + 10 * smt + 9;
  if( smt < 0 ) smt = 0;
  snprintf( rst, 4, "%3d", smt );

  return( TRUE );
} /* End of Read_Rx_Status() */

/*-------------------------------------------------------------------------*/

