/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "shared.h"

/* File variables */
static GtkWidget
  *quit_dialog = NULL,
  *save_dialog = NULL,
  *error_dialog = NULL,
  *query_dialog = NULL,
  *bad_entry_dialog = NULL,
  *file_dialog = NULL;

static int query_ok_action;

/*  Error_Dialog()
 *
 *  Opens an error dialog box
 */

  void
Error_Dialog( char *message, gboolean hide )
{
  GtkWidget *label = NULL;
  GtkBuilder *builder = NULL;

  if( !error_dialog )
  {
    error_dialog = create_error_dialog( &builder );
    label = Builder_Get_Object( builder, "error_message" );
    gtk_label_set_text( GTK_LABEL(label), message );
    if( hide ) gtk_widget_hide( Builder_Get_Object(builder, "error_ok_button") );
    gtk_widget_show( error_dialog );
    g_object_unref( builder );
  }
}


  void
Quit_Dialog( char *message )
{
  GtkWidget *label = NULL;
  GtkBuilder *builder = NULL;
  if( !quit_dialog )
  {
    quit_dialog = create_quit_dialog( &builder );
    label = Builder_Get_Object( builder, "quit_message" );
    gtk_label_set_text( GTK_LABEL(label), message );
    gtk_widget_show( quit_dialog );
    g_object_unref( builder );
  }
}


  void
Save_Dialog( char *message )
{
  GtkWidget *label;
  GtkBuilder *builder = NULL;
  if( !save_dialog )
  {
    save_dialog = create_save_dialog( &builder );
    label = Builder_Get_Object( builder, "save_message" );
    gtk_label_set_text( GTK_LABEL(label), message );
    gtk_widget_show( save_dialog );
    g_object_unref( builder );
  }
}

  void
Query_Dialog( char *message )
{
  GtkWidget *label;
  GtkBuilder *builder = NULL;
  if( !query_dialog )
  {
    query_dialog = create_query_dialog( &builder );
    label = Builder_Get_Object( builder, "query_message" );
    gtk_label_set_text( GTK_LABEL(label), message );
    gtk_widget_show( query_dialog );
    g_object_unref( builder );
  }
}


  void
Bad_Entry_Dialog( char *message )
{
  GtkWidget *label;
  GtkBuilder *builder = NULL;
  if( !bad_entry_dialog )
  {
    bad_entry_dialog = create_bad_entry_dialog( &builder );
    label = Builder_Get_Object( builder, "bad_entry" );
    gtk_label_set_text( GTK_LABEL(label), message );
    gtk_widget_show( bad_entry_dialog );
    g_object_unref( builder );
  }
}


  gboolean
on_main_window_button_press_event(
    GtkWidget      *widget,
    GdkEventButton *event,
    gpointer        user_data)
{
  if( event->button == 3 )
    gtk_menu_popup_at_pointer( GTK_MENU(popup_menu), NULL );

  return( TRUE );
}


  gboolean
on_main_window_delete_event(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  Quit_Dialog(_("Quit xhamlog?") );
  return( TRUE );
}


  void
on_main_window_destroy(
    GObject  *object,
    gpointer  user_data)
{
  Cleanup();
  gtk_main_quit();
}


  void
on_dx_call_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_Callsign( editable );
}


  void
on_dx_rst_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
}


  void
on_my_rst_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
}


  void
on_dx_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Capitalize_Entry( editable );
}


  void
on_dx_qth_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Capitalize_Entry( editable );
}


  void
on_dx_loc_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_Locator( editable );
}

  void
on_day_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const gchar *data = gtk_entry_get_text(GTK_ENTRY(editable));
  int day = atoi( data );

  if( strlen(data) == 0 )
    return;

  if( (day < 1) || (day > 31) )
  {
    Bad_Entry_Dialog( _("Day number out of range") );
    return;
  }
}


  void
on_month_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const gchar *data = gtk_entry_get_text(GTK_ENTRY(editable));
  int mon = atoi( data );

  if( strlen(data) == 0 )
    return;

  if( (mon < 1) || (mon > 12) )
  {
    Bad_Entry_Dialog( _("Month number out of range") );
    return;
  }
}

  void
on_year_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const gchar *data = gtk_entry_get_text(GTK_ENTRY(editable));
  if( strlen(data) < 4 )
    return;

  int year = atoi( data );
  if( year < 2000 )
  {
    Bad_Entry_Dialog( _("Bad year number") );
    return;
  }
}


  void
on_hour_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const gchar *data = gtk_entry_get_text(GTK_ENTRY(editable));
  int hrs = atoi( data );

  if( strlen(data) == 0 )
    return;

  if( (hrs < 0) || (hrs > 23) )
  {
    Bad_Entry_Dialog( _("Hour number out of range") );
    return;
  }
}


  void
on_minute_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const gchar *data = gtk_entry_get_text(GTK_ENTRY(editable));
  int min = atoi( data );

  if( strlen(data) == 0 )
    return;

  if( (min < 0) || (min > 59) )
  {
    Bad_Entry_Dialog( _("Minute number out of range") );
    return;
  }
}


  void
on_freq_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Is_Numerical( editable );
}


  void
on_tx_power_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Is_Numerical( editable );
}


  void
on_rx_nfig_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Is_Numerical( editable );
}


  void
on_my_call_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_Callsign( editable );
}


  void
on_my_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Capitalize_Entry( editable );
}


  void
on_my_zone_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
}


  void
on_my_qth_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Capitalize_Entry( editable );
}


  void
on_my_loc_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_Locator( editable );
}


  void
on_new_qso_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  GtkWidget *entry;

  /* For reading Tcvr status */
  char
    mode[9],
    freq[8],
    rst[4];

    /* Save record if needed and completed */
    if( isFlagSet(SAVE_RECORD) )
    {
      if( Read_QSO_Record() )
        Save_QSO_Record();
      else
      {
        Save_Dialog(
            _("QSO Record is not complete\n"\
              "Do you still want to save?") );
        return;
      }
    }

    /* Initialize for new entry */
    Clear_QSO_Record( FALSE );
    SetFlag( SAVE_RECORD );

    /* Read Receiver status if CAT enabled */
    if( Open_Tcvr_Serial() )
    {
      Read_Rx_Status( mode, freq, rst );
      if( isFlagSet(SERIAL_IO_ERROR) ) return;

      entry = Builder_Get_Object( main_window_builder, "dx_rst" );
      gtk_entry_set_text( GTK_ENTRY(entry), rst );
      if( isFlagSet(NEW_FREQ) )
      {
        entry = Builder_Get_Object( main_window_builder, "freq" );
        gtk_entry_set_text( GTK_ENTRY(entry), freq );
      }
      if( isFlagSet(NEW_MODE) )
      {
        entry = Builder_Get_Object( main_window_builder, "mode" );
        gtk_entry_set_text( GTK_ENTRY(entry), mode );
      }
    }

    Enter_Date();

    entry = Builder_Get_Object( main_window_builder, "dx_call" );
    gtk_widget_grab_focus( entry );
}


  void
on_save_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  /* Save record if needed (and valid) */
  if( Read_QSO_Record() )
    Save_Dialog( _("Save QSO Record to file?") );
  else Save_Dialog(
      _("QSO Record is incomplete\n"\
        "Do you still want to save?") );
}


  void
on_default_rig_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Default_Rig_Data();
}


  void
on_clear_qso_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Clear_QSO_Record( FALSE );
}


  void
on_clear_all_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Clear_QSO_Record( TRUE );
  SetFlag( NEW_DATE | NEW_FREQ | NEW_MODE );
}


  void
on_preview_log_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( system(rc_data.log_viewer) == -1 )
    fprintf( stderr, "xhamlog: system() function call failed\n" );
}


  void
on_preview_qsl_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( Make_QSL_Cards() )
    if( system(rc_data.qsl_viewer) == -1 )
      fprintf( stderr, "xhamlog: system() function call failed\n" );
}


  void
on_quit_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Quit_Dialog( _("Quit xhamlog?") );
}


  void
on_error_quit_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Cleanup();
  gtk_main_quit();
}


  void
on_error_ok_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( error_dialog );
}


  void
on_cancel_save_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( save_dialog );
}


  void
on_save_dialog_destroy(
    GObject *object,
    gpointer user_data)
{
  save_dialog = NULL;
}


  void
on_quit_dialog_destroy(
    GObject *object,
    gpointer user_data)
{
  quit_dialog = NULL;
}


  void
on_save_ok_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Save_QSO_Record();
  gtk_widget_destroy( save_dialog );
}


  void
on_bad_entry_ok_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( bad_entry_dialog );
}


  void
on_bad_entry_dialog_destroy(
    GObject *object,
    gpointer user_data)
{
  bad_entry_dialog = NULL;
}


  void
on_cancel_quit_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( quit_dialog );
}


  void
on_quit_ok_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( main_window );
}


  gboolean
on_entry_button_press_event(
    GtkWidget      *widget,
    GdkEventButton *event,
    gpointer        user_data)
{
  /* Clear entry widget on button 3 */
  if( event->button == 3 )
  {
    gtk_entry_set_text( GTK_ENTRY(widget), "" );
    return( TRUE );
  }

  return( FALSE );
}


  void
on_query_cancel_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( query_dialog );
}


  void
on_query_dialog_destroy(
    GObject *object,
    gpointer user_data)
{
  query_dialog = NULL;
}


  void
on_query_ok_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  char file_name[64];

  gtk_widget_destroy( query_dialog );

  size_t s = sizeof( file_name );
  Strlcpy( file_name, getenv("HOME"), s );

  switch( query_ok_action )
  {
    case DELETE_QSL_FILE:
      Strlcat ( file_name, "/xhamlog/STATION_QSL_FILE", s );
      break;

    case DELETE_ADIF_FILE:
      Strlcat ( file_name, "/xhamlog/STATION_ADIF_FILE", s );
      break;

    default: return;
  }

  unlink( (const char *)file_name );
}


  gboolean
on_date_entry_button_press_event(
    GtkWidget      *widget,
    GdkEventButton *event,
    gpointer        user_data)
{
  /* Enter date on button 3 */
  if( event->button == 3 )
  {
    SetFlag( NEW_DATE );
    Enter_Date();
    return( TRUE );
  }

  ClearFlag( NEW_DATE );

  return( FALSE );
}


  gboolean
on_freq_entry_button_press_event(
    GtkWidget      *widget,
    GdkEventButton *event,
    gpointer        user_data)
{
  GtkWidget *entry;

  /* For reading Tcvr status */
  char
    mode[9],
    freq[8],
    rst[4];

    /* Enter mode from CAT on button 3 */
    if( event->button == 3 )
    {
      /* If above is successful */
      if( isFlagSet(CAT_SETUP) )
      {
        Read_Rx_Status( mode, freq, rst );
        if( isFlagSet(SERIAL_IO_ERROR) ) return( FALSE );

        entry = Builder_Get_Object( main_window_builder, "freq" );
        gtk_entry_set_text( GTK_ENTRY(entry), freq );
      }
      return( TRUE );
    }

    ClearFlag( NEW_FREQ );

    return( FALSE );
}


  gboolean
on_mode_entry_button_press_event(
    GtkWidget      *widget,
    GdkEventButton *event,
    gpointer        user_data)
{
  GtkWidget *entry;

  /* For reading Tcvr status */
  char
    mode[9],
    freq[8],
    rst[4];

    /* Enter mode from CAT on button 3 */
    if( event->button == 3 )
    {
      /* If above is successful */
      if( isFlagSet(CAT_SETUP) )
      {
        Read_Rx_Status( mode, freq, rst );
        if( isFlagSet(SERIAL_IO_ERROR) ) return( FALSE );

        entry = Builder_Get_Object( main_window_builder, "mode" );
        gtk_entry_set_text( GTK_ENTRY(entry), mode );
      }
      return( TRUE );
    }

    ClearFlag( NEW_MODE );

    return( FALSE );
}


  void
on_print_qsl_cards_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  /* For use with "stat" */
  struct stat buf;

  /* Check that Log file is available */
  if( stat( rc_data.stn_qsl_file, &buf) == -1 )
  {
    Error_Dialog(
        _("Station QSL file not found\n"\
          "Please Preview QSL cards first"), OK );
    return;
  }

  if( system(rc_data.qsl_print) == -1 )
    fprintf( stderr, "xhamlog: system() function call failed\n" );

  query_ok_action = DELETE_QSL_FILE;
  Query_Dialog( _("Delete QSL cards file?") );
}


  void
on_print_station_log_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( system(rc_data.log_print) == -1 )
    fprintf( stderr, "xhamlog: system() function call failed\n" );
}


  void
on_ft847_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rc_data.tcvr_type = FT847;
    Close_Tcvr_Serial();
    Open_Tcvr_Serial();
  }
}


  void
on_ft857_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rc_data.tcvr_type = FT857;
    Close_Tcvr_Serial();
    Open_Tcvr_Serial();
  }
}


  void
on_k3_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rc_data.tcvr_type = K3;
    Close_Tcvr_Serial();
    Open_Tcvr_Serial();
  }
}


  void
on_none_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Close_Tcvr_Serial();
    rc_data.tcvr_type = NONE;
  }
}


  void
on_import_log_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  GtkBuilder *builder = NULL;
  if( !file_dialog )
  {
    file_dialog = create_filechooserdialog( &builder );
    gtk_file_chooser_set_current_folder(
        GTK_FILE_CHOOSER(file_dialog), getenv("HOME") );
    gtk_widget_show( file_dialog );
    g_object_unref( builder );
  }
}


  void
on_filechooser_cancel_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( file_dialog );
}


  void
on_filechooserdialog_response(
    GtkDialog *dialog,
    gint       response_id,
    gpointer   user_data)
{
  if( response_id == GTK_RESPONSE_OK )
  {
    gchar *file_name;

    file_name =
      gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(file_dialog) );
    if( file_name == NULL ) return;

    if( Open_File(&ext_log_fp, file_name, "r+") == FILE_OPEN )
    {
      if( Import_Log(ext_log_fp) )
        Close_File( &ext_log_fp );
    }

    gtk_widget_destroy( file_dialog );
  }
}


  void
on_filechooserdialog_destroy(
    GObject *object,
    gpointer user_data)
{
  file_dialog = NULL;
}


  void
on_delete_qsl_file_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  query_ok_action = DELETE_QSL_FILE;
  Query_Dialog( _("Delete QSL cards file?") );
}


  void
on_delete_adif_log_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  query_ok_action = DELETE_ADIF_FILE;
  Query_Dialog( _("Delete ADIF log file?") );
}


  void
on_error_dialog_destroy(
    GObject *object,
    gpointer user_data)
{
  error_dialog = NULL;
}


  gboolean
on_error_dialog_delete_event(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  return( TRUE );
}

