/*  xhamlog: A logging program for radio amateurs.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_H
#define COMMON_H 1

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <fcntl.h>
#include <termios.h>
#include <string.h>
#include <sys/stat.h>

/* Control Flags */
#define	SAVE_RECORD			0x00000001	/* Save QSO record */
#define	CAT_SETUP			0x00000002	/* Enable Transceiver CAT */
#define SERIAL_IO_ERROR		0x00000004	/* Serial port I/O error occured */
#define CAPITALIZE			0x00000008	/* Capitalize letters in entries */
#define NEW_DATE			0x00000010	/* Enter new date from sys clock */
#define NEW_MODE			0x00000020	/* Enter new oper. mode from CAT */
#define NEW_FREQ			0x00000040	/* Enter new frequency from CAT  */

/* Action codes for query_ok_action */
#define DELETE_ADIF_FILE	1
#define DELETE_QSL_FILE		2

/* Return values */
#define ERROR		1
#define SUCCESS		0

/* xhamlog runtime config file */
#define RC_FILE	"xhamlogrc"

/* Size of char arrays (strings) for error messages etc */
#define MESG_SIZE	128

/* Transceiver type */
enum
{
  NONE = 0,
  FT847,
  FT857,
  K2,
  K3
};

/* Configuration parameters structure */
typedef struct
{
  FILE
    *log_adif_fp,    /* File for ADIF QSO record */
    *station_qsl_fp, /* File for QSL cards */
    *station_log_fp; /* File for station's log */

  /* Serial ports for CAT */
  char
    tcvr_type,
    cat_serial[15];

  /* My info */
  char
    my_call[15],
    my_name[13],
    my_srname[13],
    my_addr1[31],
    my_addr2[31],
    my_zone[15],
    my_qth[13],
    my_loc[7];

  /* Tx/Rx power, N.Fig, antennas */
  char
    tx[12],
    tx_power[10],
    tx_ant[16],
    rx[12],
    rx_nfig[8],
    rx_ant[16];

  /* Viewer and printer commands */
  char
    log_viewer[51],
    log_print[51],
    qsl_viewer[51],
    qsl_print[51];

  /* xhamlog "home" directory */
  char
    home_dir[31],
    stn_log_file[51],
    stn_adif_file[51],
    stn_qsl_file[51];

} rc_data_t;

/* QSO record type definition */
typedef struct
{
  /* QSO data */
  char
    dx_call[15],  /* Remote (dx) station callsign */
    dx_rst[4],    /* My RST report to Dx station  */
    dx_name[13],  /* Remote station operator name */
    dx_qth[13],   /* Remote station QTH loc. name */
    dx_loc[7],    /* Remote station QTH locator   */
    date[12],     /* Normal format QSO date  */
    date_adif[9], /* QSO date in ADIF format */
    time[6],      /* Normal format QSO time  */
    time_adif[5], /* QSO time in ADIF format */
    freq[14],     /* QSO operating frequency MHz */
    mode[12],     /* Mode of operation */
    via[13];      /* Via which path or relay Sat */

  /* Tx/Rx power, N.Fig, antennas */
  char
    tx[12],
    tx_power[10],
    tx_ant[16],
    rx[12],
    rx_nfig[8],
    rx_ant[16];

  /* My call, rst, qth, locator */
  char
    my_call[15],
    my_rst[4],
    my_qth[13],
    my_loc[7];

  /* Remarks on QSO */
  char remarks[46];

} qso_record_t;

/*-------------------------------------------------------------------*/

/* Function prototypes produced by cproto */
/* callbacks.c */
void Error_Dialog(char *message, gboolean hide);
void Quit_Dialog(char *message);
void Save_Dialog(char *message);
void Query_Dialog(char *message);
void Bad_Entry_Dialog(char *message);
gboolean on_main_window_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
gboolean on_main_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_main_window_destroy(GtkObject *object, gpointer user_data);
void on_dx_call_changed(GtkEditable *editable, gpointer user_data);
void on_dx_rst_changed(GtkEditable *editable, gpointer user_data);
void on_my_rst_changed(GtkEditable *editable, gpointer user_data);
void on_dx_name_changed(GtkEditable *editable, gpointer user_data);
void on_dx_qth_changed(GtkEditable *editable, gpointer user_data);
void on_dx_loc_changed(GtkEditable *editable, gpointer user_data);
void on_day_changed(GtkEditable *editable, gpointer user_data);
void on_month_changed(GtkEditable *editable, gpointer user_data);
void on_year_changed(GtkEditable *editable, gpointer user_data);
void on_hour_changed(GtkEditable *editable, gpointer user_data);
void on_minute_changed(GtkEditable *editable, gpointer user_data);
void on_freq_changed(GtkEditable *editable, gpointer user_data);
void on_tx_power_changed(GtkEditable *editable, gpointer user_data);
void on_rx_nfig_changed(GtkEditable *editable, gpointer user_data);
void on_my_call_changed(GtkEditable *editable, gpointer user_data);
void on_my_name_changed(GtkEditable *editable, gpointer user_data);
void on_my_zone_changed(GtkEditable *editable, gpointer user_data);
void on_my_qth_changed(GtkEditable *editable, gpointer user_data);
void on_my_loc_changed(GtkEditable *editable, gpointer user_data);
void on_new_qso_clicked(GtkButton *button, gpointer user_data);
void on_save_clicked(GtkButton *button, gpointer user_data);
void on_default_rig_clicked(GtkButton *button, gpointer user_data);
void on_clear_qso_clicked(GtkButton *button, gpointer user_data);
void on_clear_all_clicked(GtkButton *button, gpointer user_data);
void on_preview_log_clicked(GtkButton *button, gpointer user_data);
void on_preview_qsl_clicked(GtkButton *button, gpointer user_data);
void on_quit_clicked(GtkButton *button, gpointer user_data);
void on_error_quit_button_clicked(GtkButton *button, gpointer user_data);
void on_error_ok_button_clicked(GtkButton *button, gpointer user_data);
void on_cancel_save_clicked(GtkButton *button, gpointer user_data);
void on_save_ok_clicked(GtkButton *button, gpointer user_data);
void on_bad_entry_ok_clicked(GtkButton *button, gpointer user_data);
void on_cancel_quit_clicked(GtkButton *button, gpointer user_data);
void on_quit_ok_clicked(GtkButton *button, gpointer user_data);
gboolean on_entry_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_query_cancel_button_clicked(GtkButton *button, gpointer user_data);
void on_query_ok_button_clicked(GtkButton *button, gpointer user_data);
gboolean on_date_entry_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
gboolean on_freq_entry_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
gboolean on_mode_entry_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_print_qsl_cards_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_print_station_log_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_ft847_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_ft857_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_k3_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_none_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_import_log_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_filechooser_cancel_clicked(GtkButton *button, gpointer user_data);
void on_filechooserdialog_response(GtkDialog *dialog, gint response_id, gpointer user_data);
void on_delete_qsl_file_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_delete_adif_log_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_error_dialog_destroy(GtkObject *object, gpointer user_data);
gboolean on_error_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
/* cat.c */
gboolean Open_Tcvr_Serial(void);
void Close_Tcvr_Serial(void);
gboolean Read_Tcvr_Serial(char *data, size_t len);
gboolean Write_Tcvr_Serial(char *data, size_t len);
gboolean Read_Rx_Status(char *mode, char *freq, char *rst);
/* interface.c */
GtkWidget *create_main_window(void);
GtkWidget *create_error_dialog(void);
GtkWidget *create_save_dialog(void);
GtkWidget *create_bad_entry_dialog(void);
GtkWidget *create_quit_dialog(void);
GtkWidget *create_query_dialog(void);
GtkWidget *create_main_menu(void);
GtkWidget *create_filechooserdialog(void);
/* main.c */
int main(int argc, char *argv[]);
gboolean Load_Config(gpointer idata);
/* shared.c */
/* support.c */
GtkWidget *lookup_widget(GtkWidget *widget, const gchar *widget_name);
void add_pixmap_directory(const gchar *directory);
GtkWidget *create_pixmap(GtkWidget *widget, const gchar *filename);
GdkPixbuf *create_pixbuf(const gchar *filename);
void glade_set_atk_action_description(AtkAction *action, const gchar *action_name, const gchar *description);
/* utils.c */
int Load_Line(char *buff, FILE *pfile, char *messg);
gboolean Save_QSO_Record(void);
gboolean Read_QSO_Record(void);
gboolean Make_QSL_Cards(void);
gboolean Get_QSO_Record(FILE *fp, gboolean qsl);
void Clear_QSO_Record(gboolean all);
void Default_Rig_Data(void);
void Enter_Date(void);
gboolean Find_Line(char **line_idx, char *line, const char *string, FILE *fp);
gboolean Find_String(char **line_idx, char *line, const char *string);
void Process_RST(GtkEditable *editable);
void Process_Locator(GtkEditable *editable);
void Process_Callsign(GtkEditable *editable);
void Capitalize_Entry(GtkEditable *editable);
void Is_Numerical(GtkEditable *editable);
int Open_File(FILE **fp, char *fpath, const char *mode);
void Close_File(FILE **fp);
gboolean File_Print(FILE **fp, char *string);
void Trunc_Spaces(char *string);
gboolean Import_Log(FILE *fp);
int isFlagSet(int flag);
int isFlagClear(int flag);
void SetFlag(int flag);
void ClearFlag(int flag);
void ToggleFlag(int flag);
void Cleanup(void);
void Strlcpy(char *dest, const char *src, size_t n);
void Strlcat(char *dest, const char *src, size_t n);
void Usage(void);
double Atof(const char *nptr);

#endif
