/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "cat.h"
#include "shared.h"

/* Serial port File descriptor */
static int serial_fd = 0;

/* Original serial port options */
static struct termios serial_old_options;

/* Original transceiver status */
static tcvr_status_t orig_tcvr_status;

static char
  /* Command codes for Yaesu FT847/FT857 CAT */
  FT847_CAT_ON[]    = { 0x00, 0, 0, 0, (char)0x00 }, /*  Enable CAT */
  FT847_CAT_OFF[]   = { 0x00, 0, 0, 0, (char)0x80 }, /* Disable CAT */
  FT847_RX_STATUS[] = { 0x00, 0, 0, 0, (char)0xE7 }, /* Request RX status  */
  FT847_MAIN_VFO[]  = { 0x00, 0, 0, 0, (char)0x03 }, /* Read main VFO freq */
  FT847_SET_VFO[]   = { 0x00, 0, 0, 0, (char)0x01 }, /* Set main VFO freq  */
  FT847_PTT_ON[]    = { 0x00, 0, 0, 0, (char)0x08 }, /* PTT On (transmit)  */
  FT847_PTT_OFF[]   = { 0x00, 0, 0, 0, (char)0x88 }, /* PTT Off (receive)  */
  FT847_MODE_USB[]  = { (char)0x01, 0, 0, 0, (char)0x07 }, /* Set mode to USB */
  FT847_MODE_CW[]   = { (char)0x02, 0, 0, 0, (char)0x07 }, /* Set mode to CW  */
  FT847_MODE_CWN[]  = { (char)0x82, 0, 0, 0, (char)0x07 }, /* Set mode to CWN */

  /* Command codes for Elecraft K3 */
  K3_K31_S[]    = "K31;",           /* Set K3 CAT mode */
  K3_K31_G[]    = "K3;",            /* Request K3 CAT mode */
  K3_SMTR_G[]   = "SM;",            /* Request K3 S-meter  */
  K3_SMTR_S[]   = "SMnnnn;",        /* K3 S-meter data */
  K3_VFOA_S[]   = "FAnnnnnnnnnnn;", /* Set VFO A frequency */
  K3_VFOA_G[]   = "FA;",            /* Get VFO A frequency */
  K3_COMP_S[]   = "CPnnn;",         /* Set Microphone Gain */
  K3_COMP_G[]   = "CP;",            /* Set Microphone Gain */
  K3_POWER_S[]  = "PCnnn;",         /* Set TX power output */
  K3_POWER_G[]  = "PC;",            /* Get TX power output */
  K3_PTT_ON[]   = "TX;",            /* PTT On (transmit) */
  K3_PTT_OFF[]  = "RX;",            /* PTT Off (receive) */
  K3_MODE_G[]   = "MD;",            /* Get transc mode  */
  K3_MODE_S[]   = "MDn;",           /* Transceiver mode */
  K3_MODE_USB[] = "MD2;",           /* Set mode to USB */
  //K3_MODE_CW[]  = "MD3;",         /* Set mode to CW  */
  K3_MODE_CWR[] = "MD7;";           /* Set mode to CWR */

/*------------------------------------------------------------------------*/

/* Cancel_CAT()
 *
 * Cancels Transceiver CAT (on error)
 */

  static void
Cancel_CAT( void )
{
  close( serial_fd );
  serial_fd = 0;
  rc_data.tcvr_type = NONE;
  gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
        Builder_Get_Object(popup_menu_builder, "none")), TRUE );
  ClearFlag( CAT_SETUP );
  ClearFlag( TCVR_SERIAL_TEST );
  Error_Dialog( "Failed to Establish Transceiver CAT", OK  );
}

/*-------------------------------------------------------------------------*/

/*  Read_Tcvr_Serial()
 *
 *  Reading Data from the Tcvr's Serial Port
 */

  static void
Read_Tcvr_Serial( char *data, size_t len )
{
  int nbytes = 0, cnt = 0;

  /* Check for data to be available */
  if( isFlagClear(TCVR_SERIAL_TEST) &&
      isFlagClear(CAT_SETUP) )
    return;
  while( ++cnt < MAX_SERIAL_RETRIES )
  {
    ioctl( serial_fd, FIONREAD, &nbytes);
    if( nbytes == (int)len ) break;
    fprintf( stderr,
        "xfhell: read(): incorrect number of bytes available:"
        " %d/%d\n", nbytes, (int)len );
    usleep(100000);
  }

  /* Error condition */
  if( cnt >= MAX_SERIAL_RETRIES )
  {
    tcflush( serial_fd, TCIOFLUSH );
    fprintf( stderr,
        "xfhell: read(): incorrect number of bytes available:"
        " %d/%d after %d tries\n", nbytes, (int)len, cnt );
    data[0] = '\0';
    Cancel_CAT();
    return;
  }

  /* Clear data buffer and read from serial */
  size_t ret = (size_t)read( serial_fd, data, len );

  /* Error condition */
  if( ret != len )
  {
    tcflush( serial_fd, TCIOFLUSH );
    fprintf( stderr,
        "xfhell: read() return value wrong: %d/%d\n",
        (int)ret, (int)len );
    data[0] = '\0';
    Cancel_CAT();
    return;
  }

  return;
} /* End of Read_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Tcvr_Serial()
 *
 *  Writes a command to the Tranceiver
 */

  static void
Write_Tcvr_Serial( char *data, size_t len )
{
  int cnt = 0;
  size_t ret = 0;

  /* Flush serial and write command, retrying if needed */
  if( isFlagClear(TCVR_SERIAL_TEST) &&
      isFlagClear(CAT_SETUP) )
    return;

  tcflush( serial_fd, TCIOFLUSH );
  while( cnt++ < MAX_SERIAL_RETRIES )
  {
    ret = (size_t)write( serial_fd, data, len );
    if( ret == len ) break;
    fprintf( stderr,
        "xfhell: write(): incorrect number of bytes written:"
        " %d/%d\n", (int)ret, (int)len );
    usleep(100000);
  }

  /* Error condition */
   if( cnt >= MAX_SERIAL_RETRIES )
  {
    tcflush( serial_fd, TCIOFLUSH );
    fprintf( stderr,
        "xfhell: write(): incorrect number of bytes written:"
        " %d/%d after %d tries\n", (int)ret, (int)len, cnt );
    Cancel_CAT();
    return;
  }

  /* Give time to CAT to do its thing */
  usleep(100000);
  return;

} /* End of Write_Tcvr_Command() */

/*-------------------------------------------------------------------------*/

/* Tcvr_Status()
 *
 * Gets or Sets Transceiver status (only for the K2/K3 currently)
 * Gets or Sets the Tx Power O/P, Mic Gain and Compressor level
 */
  static gboolean
Tcvr_Status( int action, tcvr_status_t *status )
{
  /* For K2 or K3 */
  if( (rc_data.tcvr_type == K2) || (rc_data.tcvr_type == K3) )
  {
    /* Get transceiver status */
    if( action == GET_TCVR_STATUS )
    {
      Write_Tcvr_Serial( K3_POWER_G, sizeof(K3_POWER_G)-1 );
      Read_Tcvr_Serial(  K3_POWER_S, sizeof(K3_POWER_S)-1 );
      if( rc_data.tcvr_type == K3 )
      {
        Write_Tcvr_Serial( K3_COMP_G, sizeof(K3_COMP_G)-1 );
        Read_Tcvr_Serial(  K3_COMP_S, sizeof(K3_COMP_S)-1 );
      }
      Write_Tcvr_Serial( K3_VFOA_G, sizeof(K3_VFOA_G)-1 );
      Read_Tcvr_Serial(  K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      Write_Tcvr_Serial( K3_MODE_G, sizeof(K3_MODE_G)-1 );
      Read_Tcvr_Serial(  K3_MODE_S, sizeof(K3_MODE_S)-1 );
      status->tx_power = atoi( K3_POWER_S+2 );
      status->comp_lev = atoi(  K3_COMP_S+2 );
      status->rig_mode = atoi(  K3_MODE_S+2 );
      status->vfo_freq = atoi(  K3_VFOA_S+2 );
      return( TRUE );
    } /* if( action == GET_TCVR_STATUS ) */

    /* Set transceiver status */
    if( action == SET_TCVR_STATUS )
    {
      snprintf( K3_POWER_S+2, sizeof(K3_POWER_S)-2,  "%03d;", status->tx_power );
      snprintf(  K3_COMP_S+2, sizeof(K3_COMP_S)-2,  "%03d;", status->comp_lev );
      snprintf(  K3_MODE_S+2, sizeof(K3_MODE_S)-2,    "%d;", status->rig_mode );
      snprintf(  K3_VFOA_S+2, sizeof(K3_VFOA_S)-2, "%011d;", status->vfo_freq );

      Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      sleep( 1 );
      Write_Tcvr_Serial( K3_MODE_S, sizeof(K3_MODE_S)-1 );
      Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      if( rc_data.tcvr_type == K3 )
        Write_Tcvr_Serial( K3_COMP_S, sizeof(K3_COMP_S)-1 );
      Write_Tcvr_Serial(K3_POWER_S, sizeof(K3_POWER_S)-1);
      return( TRUE );

    } /* if( action == SET_TCVR_STATUS ) */
  } /* if( (rc_data.tcvr_type == K2) || (rc_data.tcvr_type == K2) ) */

  return( FALSE );
} /* Tcvr_Status() */

/*-------------------------------------------------------------------------*/

/*  Open_Tcvr_Serial()
 *
 *  Opens Tcvr's Serial Port device, returns the file
 *  descriptor serial_fd on success or exits on error
 */

  gboolean
Open_Tcvr_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */
  int error = 0;
  char test[5];


  /* Abort if serial already open */
  if( isFlagSet(CAT_SETUP) ) return( TRUE );

  /* Open CAT serial port, exit on error */
  serial_fd = open( rc_data.cat_serial, O_RDWR | O_NOCTTY );
  if( serial_fd < 0 )
  {
    char mesg[MESG_SIZE];

    perror( rc_data.cat_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    snprintf( mesg, sizeof(mesg),
        _("Failed to open %s\n"\
          "CAT not enabled"), rc_data.cat_serial );
    Error_Dialog( mesg, OK );
    return( FALSE );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  error = fcntl( serial_fd, F_SETLK, &lockinfo );
  if( error < 0 )
  {
    char mesg[MESG_SIZE];

    perror( rc_data.cat_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    if( fcntl(serial_fd, F_GETLK, &lockinfo) != -1 )
    {
      snprintf( mesg, sizeof(mesg),
          _("Failed to lock %s\n"\
            "Device locked by pid %d\n"\
            "CAT not enabled"),
          rc_data.cat_serial, lockinfo.l_pid );
      Error_Dialog( mesg, OK );
    }
    return( FALSE );
  }

  /* Save the current serial port options */
  error |= tcgetattr( serial_fd, &serial_old_options );

  /* Read the current serial port options */
  error |= tcgetattr( serial_fd, &new_options );

  /* Set the i/o baud rates */
  switch( rc_data.tcvr_type )
  {
    case FT847:
      error |= cfsetspeed( &new_options, B57600 );
      break;

    case FT857:
      error |= cfsetspeed( &new_options, B38400 );
      break;

    case K2: case K3:
      error |= cfsetspeed( &new_options, B38400 );
      break;
  }

   /* Set options for 'raw' I/O mode */
  cfmakeraw( &new_options );

  /* Setup read() timeout to .2 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 2;

  /* Setup the new options for the port */
  error |= tcsetattr( serial_fd, TCSAFLUSH, &new_options );
  if( error )
  {
    if( serial_fd > 0 )
    {
      tcsetattr( serial_fd, TCSANOW, &serial_old_options );
      close( serial_fd );
      serial_fd = 0;
    }
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    Error_Dialog( _("Failed to Enable CAT"), OK );
    return( FALSE );
  }

  /* Enable CAT by testing comms with transceiver */
  SetFlag( TCVR_SERIAL_TEST );
  switch( rc_data.tcvr_type )
  {
    case FT847:
      Write_Tcvr_Serial( FT847_CAT_ON, sizeof(FT847_CAT_ON) );
      Write_Tcvr_Serial( FT847_MAIN_VFO, sizeof(FT847_MAIN_VFO) );
      Read_Tcvr_Serial( test, 5 );
      break;

    case FT857:
      Write_Tcvr_Serial( FT847_MAIN_VFO, sizeof(FT847_MAIN_VFO) );
      Read_Tcvr_Serial( test, 5);
      break;

    case K2:
      /* Get K2 VFO A as a test of serial port */
      Write_Tcvr_Serial( K3_VFOA_G, sizeof(K3_VFOA_G)-1 );
      Read_Tcvr_Serial(  K3_VFOA_S, sizeof(K3_VFOA_S)-1 );

      /* Get original tcvr status */
      Tcvr_Status( GET_TCVR_STATUS, &orig_tcvr_status );

      /* Set tcvr power and compressor level */
      if( !Set_Tx_Power(rc_data.tx_output, 0) )
        return( FALSE );
      break;

    case K3:
      /* Set K3 to mode K31 as a test of serial port */
      Strlcpy( K3_K31_S, "K31;", sizeof(K3_K31_S) );
      Write_Tcvr_Serial( K3_K31_S, sizeof(K3_K31_S)-1 );
      Write_Tcvr_Serial( K3_K31_G, sizeof(K3_K31_G)-1 );
      Read_Tcvr_Serial(  K3_K31_S, sizeof(K3_K31_S)-1 );
      if( strcmp(K3_K31_S, "K31;") != 0 )
      {
        fprintf( stderr, "xfhell: failed to enable K3 CAT\n" );
        Cancel_CAT();
        return( FALSE );
      }

      /* Get original tcvr status */
      Tcvr_Status( GET_TCVR_STATUS, &orig_tcvr_status );

      /* Set tcvr power and compressor level */
      if( !Set_Tx_Power(rc_data.tx_output, 0) )
        return( FALSE );
      break;

  } /* switch( rc_data.tcvr_type ) */

  ClearFlag( TCVR_SERIAL_TEST );
  SetFlag( CAT_SETUP );
  return( TRUE );

} /* End of Open_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/* RS232_RTS_DTR_Set()
 *
 * Sets the Serial Port RTS/DTR line UP/DOWN
 */
  static gboolean
RS232_RTS_DTR_Set( gboolean set )
{
  int modem_lines;

  /* Abort if CAT disabled */
  if( isFlagClear(RTS_DTR_SETUP) ) return( TRUE );

  /* Read current modem state */
  if( ioctl(serial_fd, TIOCMGET, &modem_lines) == -1 )
  {
    perror( rc_data.rts_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    Error_Dialog(
        _("Failed to set RTS/DTR\n"\
          "RTS/DTR Control disabled"), OK );
    ClearFlag( RTS_DTR_SETUP );
    return( FALSE );
  }

  /* Change the RTS/DTR state */
  if( set )
    modem_lines |=  (TIOCM_RTS | TIOCM_DTR);
  else
    modem_lines &= ~(TIOCM_RTS | TIOCM_DTR);

  /* Write the new setting */
  if( ioctl(serial_fd, TIOCMSET, &modem_lines) == -1 )
  {
    perror( rc_data.rts_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    Error_Dialog(
        _("Failed to set RTS/DTR\n"\
          "RTS/DTR Control disabled"), OK );
    ClearFlag( RTS_DTR_SETUP );
    return( FALSE );
  }

  return( TRUE );
} /* RS232_RTS_DTR_Set() */

/*-------------------------------------------------------------------------*/

/*  Close_Tcvr_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Tcvr_Serial( void )
{
  if( isFlagSet(CAT_SETUP) || isFlagSet(RTS_DTR_SETUP) )
  {
    if( isFlagSet(CAT_SETUP) )
    {
      switch( rc_data.tcvr_type )
      {
        /* Disable CAT for FT847 */
        case FT847:
          Write_Tcvr_Serial( FT847_PTT_OFF, sizeof(FT847_PTT_OFF) );
          Write_Tcvr_Serial( FT847_CAT_OFF, sizeof(FT847_CAT_OFF) );
          break;

        case FT857:
          Write_Tcvr_Serial( FT847_PTT_OFF, sizeof(FT847_PTT_OFF) );
          break;

        /* Return K2/K3 to original status */
        case K2: case K3:
          Write_Tcvr_Serial(K3_PTT_OFF, sizeof(K3_PTT_OFF)-1 );
          Tcvr_Status(SET_TCVR_STATUS, &orig_tcvr_status );
          break;
      }
    } /* if( isFlagSet(CAT_SETUP) ) */
    else RS232_RTS_DTR_Set( FALSE );

    if( serial_fd > 0 )
    {
      if( isFlagSet(CAT_SETUP) )
        tcsetattr( serial_fd, TCSADRAIN, &serial_old_options );
      close( serial_fd );
      serial_fd = 0;
    }

    ClearFlag( CAT_SETUP );
    ClearFlag( RTS_DTR_SETUP );

  } /* if( isFlagSet(CAT_SETUP) || isFlagSet(RTS_DTR_SETUP) ) */

} /* End of Close_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/* Open_RTS_DTR_Serial()
 *
 * Opens the Serial Port to be used for RTS/DTR control of PTT
 */
  gboolean
Open_RTS_DTR_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */
  int error = 0;

  /* Abort if serial already open */
  if( isFlagSet(RTS_DTR_SETUP) ) return( TRUE );

  /* Open Serial device, exit on error */
  serial_fd = open( rc_data.rts_serial, O_RDWR | O_NOCTTY );
  if( serial_fd < 0 )
  {
    char mesg[MESG_SIZE];

    perror( rc_data.rts_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    snprintf( mesg, sizeof(mesg),
        _("Failed to open %s\n"\
          "RTS/DTR Control not enabled"), rc_data.rts_serial );
    Error_Dialog( mesg, OK );
    return( FALSE );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  error = fcntl( serial_fd, F_SETLK, &lockinfo );
  if( error < 0 )
  {
    char mesg[MESG_SIZE];

    perror( rc_data.rts_serial );
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    if( fcntl(serial_fd, F_GETLK, &lockinfo) == 0 )
    {
      snprintf( mesg, sizeof(mesg),
          _("Failed to lock %s\n"\
            "Device locked by pid %d\n"\
            "RTS/DTR Control not enabled"),
          rc_data.rts_serial, lockinfo.l_pid );
      Error_Dialog( mesg, OK );
    }
    return( FALSE );
  }

  /* Save the current serial port options */
  error |= tcgetattr( serial_fd, &serial_old_options );

  /* Read the current serial port options */
  error |= tcgetattr( serial_fd, &new_options );

  /* Set options for RTS/DTR control */
  new_options.c_cflag |= ( CRTSCTS | HUPCL );
  if( isFlagSet(ENABLE_DTR) )
    new_options.c_cflag &= CLOCAL;

  /* Setup the new options for the port */
  error |= tcsetattr(serial_fd, TCSAFLUSH, &new_options);
  if( error )
  {
    if( serial_fd > 0 )
    {
      tcsetattr( serial_fd, TCSANOW, &serial_old_options );
      close( serial_fd );
      serial_fd = 0;
    }
    rc_data.tcvr_type = NONE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(popup_menu_builder, "none")), TRUE );
    Error_Dialog( _("Failed to Enable RTS/DTR Control"), OK );
    return( FALSE );
  }
  else SetFlag( RTS_DTR_SETUP );

  return( TRUE );
} /* End of Open_RTS_DTR_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Rx_Freq()
 *
 *  Reads the Rx freq of the Yaesu transceiver.
 */

  static gboolean
Read_Rx_Freq( int *freq )
{
  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  /* A char string for sending and receiving  */
  /* data strings to and from the transceiver */
  char cmnd_parm[5];

  switch( rc_data.tcvr_type )
  {
    case FT847: case FT857:
      /* Read Rx frequency */
      bzero( cmnd_parm, sizeof(cmnd_parm) );
      Write_Tcvr_Serial( FT847_MAIN_VFO, sizeof(FT847_MAIN_VFO) );
      Read_Tcvr_Serial( cmnd_parm, sizeof(cmnd_parm) );

      /* Decode frequency data to 10Hz */
      *freq = 0;
      *freq += (cmnd_parm[0] & 0xF0) >> 4;
      *freq *= 10;
      *freq += cmnd_parm[0] & 0x0F;
      *freq *= 10;
      *freq += (cmnd_parm[1] & 0xF0) >> 4;
      *freq *= 10;
      *freq += cmnd_parm[1] & 0x0F;
      *freq *= 10;
      *freq += (cmnd_parm[2] & 0xF0) >> 4;
      *freq *= 10;
      *freq += cmnd_parm[2] & 0x0F;
      *freq *= 10;
      *freq += (cmnd_parm[3] & 0xF0) >> 4;
      *freq *= 10;
      *freq += cmnd_parm[3] & 0x0F;
      *freq *= 10;
      break;

    case K2: case K3:
      /* Read Rx frequency */
      Write_Tcvr_Serial( K3_VFOA_G, sizeof(K3_VFOA_G)-1 );
      Read_Tcvr_Serial(  K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      *freq = atoi( K3_VFOA_S+2 );
      break;

  } /* switch( rc_data.tcvr_type ) */

  return( TRUE );
} /* End of Read_Rx_Freq() */

/*------------------------------------------------------------------------*/

/*  Write_Rx_Freq()
 *
 *  Writes the Rx freq to the Yaesu transceiver.
 */

  static gboolean
Write_Rx_Freq( int freq )
{
  /* Buffer used for converting freq. to string */
  char freq_buff[9];
  int idx; /* Index for loops etc */


  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  switch( rc_data.tcvr_type )
  {
    case FT847: case FT857:
      /* Set Rx frequency */
      bzero( FT847_SET_VFO, 4 );
      if( freq > 470000000 ) freq = 470000000;
      if( freq < 0 ) freq = 0;
      snprintf( freq_buff, sizeof(freq_buff), "%08d", freq / 10 );
      for( idx = 0; idx < 4; idx++ )
      {
        FT847_SET_VFO[idx]  = (char)((freq_buff[2*idx]   - '0') << 4);
        FT847_SET_VFO[idx] |= (freq_buff[2*idx+1] - '0');
      }
      Write_Tcvr_Serial( FT847_SET_VFO, sizeof(FT847_SET_VFO) );
      break;

    case K2: case K3:
      /* Set Rx frequency */
      snprintf( K3_VFOA_S+2, sizeof(K3_VFOA_S)-2, "%011d;", freq );
      Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      break;

  } /* switch( rc_data.tcvr_type ) */

  return( TRUE );
} /* Write_Rx_Freq() */

/*-------------------------------------------------------------------------*/

/*  Read_Rx_Status()
 *
 *  Reads the transceiver's status (Tx/Rx freq,
 *  mode, po/s-meter etc) from the Transceiver.
 */

  gboolean
Read_Rx_Status( char *freq, char *rst )
{
  /* A 5-char string for sending and receiving  */
  /* 5-byte strings to and from the transceiver */
  char tcvr_status[5];
  int frq = 0, smt = 0;


  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  /* Read Rx status */
  switch( rc_data.tcvr_type )
  {
    case FT847: case FT857:
      /* Read Rx status (S-meter) */
      bzero( tcvr_status, sizeof(tcvr_status) );
      Write_Tcvr_Serial( FT847_RX_STATUS, sizeof(FT847_RX_STATUS) );
      Read_Tcvr_Serial(  tcvr_status, 1 );

      /* Mask out S-meter */
      if( rc_data.tcvr_type == FT847 )
      {
        smt = tcvr_status[0] & FT847_SMETER;
        smt = smt / 4 + 3;
      }
      else
      {
        smt = tcvr_status[0] & FT857_SMETER;
        smt = smt/2 + 3;
      }

      /* Keep S-meter report in range 3-9 */
      if( smt > 9 ) smt = 9;

      /* Read Rx mode and frequency */
      Write_Tcvr_Serial( FT847_MAIN_VFO, sizeof(FT847_MAIN_VFO) );
      Read_Tcvr_Serial(  tcvr_status, sizeof(tcvr_status) );

      /* Decode frequency data to 1 kHz */
      frq = 0;
      frq += (tcvr_status[0] & 0xF0) >> 4;
      frq *= 10;
      frq += tcvr_status[0] & 0x0F;
      frq *= 10;
      frq += (tcvr_status[1] & 0xF0) >> 4;
      frq *= 10;
      frq += tcvr_status[1] & 0x0F;
      frq *= 10;
      frq += (tcvr_status[2] & 0xF0) >> 4;
      frq *= 10;
      frq += tcvr_status[2] & 0x0F;
      break;

    case K2: case K3:
      /* Read Rx frequency */
      Read_Rx_Freq( &frq );
      frq += 500;
      frq /= 1000;

      /* Read S-meter */
      Write_Tcvr_Serial( K3_SMTR_G, sizeof(K3_SMTR_G)-1 );
      Read_Tcvr_Serial(  K3_SMTR_S, sizeof(K3_SMTR_S)-1 );

      /* Following makes S-meter reading
       * more like the amateur RST code */
      smt = atoi( K3_SMTR_S+2 );
      if( rc_data.tcvr_type == K3 )
      {
        smt += 9;
        smt = smt / 3;
      }
      else
      {
        smt += 6;
        smt /= 2;
      }
      break;

  } /* switch( rc_data.tcvr_type ) */

  /* Enter frequency and S-meter data */
  snprintf( freq, 8, "%-7.3f", (double)frq/1000.0 );
  if( smt > 9 ) smt = 9;
  smt = 500 + 10 * smt + 9;
  if( smt < 0 ) smt = 0;
  snprintf( rst, 4, "%3d", smt );

  return( TRUE );
} /* End of Read_Rx_Status() */

/*-------------------------------------------------------------------------*/

/*  Set_Tcvr_Mode()
 *
 *  Sets the Transceiver operating mode
 */

  gboolean
Set_Tcvr_Mode( void )
{
  /* Set appropriate tcvr mode if CAT enabled */
  if( isFlagSet(CAT_SETUP) )
  {
    if( isFlagSet(TRANSMIT_MODE) )
    {
      /* Set tcvr power and compressor level */
      if( !Set_Tx_Power(rc_data.tx_output, 0) )
        return( FALSE );

      switch( rc_data.tcvr_type )
      {
        case FT847: case FT857:
          Write_Tcvr_Serial( FT847_MODE_USB, sizeof(FT847_MODE_USB) );
          Write_Tcvr_Serial( FT847_PTT_ON, sizeof(FT847_PTT_ON) );
          break;

        case K2: case K3:
          Write_Tcvr_Serial( K3_MODE_USB, sizeof(K3_MODE_USB)-1 );
          Write_Tcvr_Serial( K3_PTT_ON, sizeof(K3_PTT_ON)-1 );
          break;

      } /* switch( rc_data.tcvr_type ) */

      sleep( 1 );
    }/*  if( isFlagSet(TRANSMIT_MODE) ) */

    if( isFlagSet(RECEIVE_MODE) )
    {
      switch( rc_data.tcvr_type )
      {
        case FT847:
          Write_Tcvr_Serial( FT847_PTT_OFF, sizeof(FT847_PTT_OFF) );
          Write_Tcvr_Serial( FT847_MODE_CWN, sizeof(FT847_MODE_CWN) );
          break;

        case FT857:
          Write_Tcvr_Serial( FT847_PTT_OFF, sizeof(FT847_PTT_OFF) );
          Write_Tcvr_Serial( FT847_MODE_CW, sizeof(FT847_MODE_CW) );
          break;

        case K2: case K3:
          Write_Tcvr_Serial( K3_PTT_OFF, sizeof(K3_PTT_OFF)-1 );
          Write_Tcvr_Serial( K3_MODE_CWR, sizeof(K3_MODE_CWR)-1 );
          break;

      } /* switch( rc_data.tcvr_type ) */
    } /* if( isFlagSet(RECEIVE_MODE) ) */

  } /* if( isFlagSet(CAT_SETUP) ) */

  /* Rig control via RS232 RTS/DTR */
  if( isFlagSet(RTS_DTR_SETUP) )
  {
    if( isFlagSet(TRANSMIT_MODE) )
    {
      if( !RS232_RTS_DTR_Set(TRUE) ) return( FALSE );
    }
    else
    {
      if( !RS232_RTS_DTR_Set(FALSE) ) return( FALSE );
    }
  } /* if( isFlagSet(RTS_DTR_SETUP) ) */

  return( TRUE );
} /* Set_Tcvr_Mode() */

/*-------------------------------------------------------------------------*/

/* Tune_Tcvr()
 *
 * Tunes the transceiver to the frequency of the strongest
 * signal near a mouse click in the waterfall window
 */

  gboolean
Tune_Tcvr( double x )
{
  int
    from, to,   /* Range to scan for a max bin value  */
    bin_max,    /* Max bin value found in this range  */
    max_idx,    /* ifft idx at which the max is found */
    ifft_idx,   /* Idx used to search ifft bin values */
    tcvr_freq,  /* Transceiver's Rx frequency */
    audio_freq; /* Audio frequency in waterfal window */

  /* Frequency correction */
  int freq_correction;


  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  /* Calculate ifft index corresponding to pointer x
   * in waterfall window. This is +1 over the pointer
   * position since the first FFT bin is not used */
  ifft_idx = (int)(x + 1.5);

  /* Look above and below click point for max bin val */
  from = ifft_idx - CLICK_TUNE_RANGE;
  if( from < 0 ) from = 0;
  to = ifft_idx + CLICK_TUNE_RANGE;
  if( to >= wfall_pixbuf.width )
    to = wfall_pixbuf.width - 1;

  /* Find max bin value around click point */
  bin_max = 0;
  max_idx = ifft_idx;
  for( ifft_idx = from; ifft_idx < to; ifft_idx++ )
    if( bin_max < bin_ave[ifft_idx] )
    {
      bin_max = bin_ave[ifft_idx];
      max_idx = ifft_idx;
    }

  /* Audio frequency corresponding to ifft index. Since the
   * waterfall width has to be an odd number so that a center
   * line may be present, and since the FFT has to be a power
   * of 2 (e.g. even), the first (DC) output bin of the FFT is
   * not used and hence max_idx has to be increased by 1 and
   * the waterfall width also increased by one to correspond
   * with the FFT width. Only this way the audio frequency that
   * corresponds to FFT index can be calculated accurately. */
  audio_freq =
    ( 2 * (max_idx + 1) * rc_data.tone_freq ) / ( wfall_pixbuf.width + 1 );

  /* Read current Rx frequency */
  tcvr_freq = 0;
  Read_Rx_Freq( &tcvr_freq );
  freq_correction = audio_freq - rc_data.tone_freq;

  /* Calculate and write receiver
   * frequency to center audio on */
  switch( rc_data.tcvr_type )
  {
    case FT847: case FT857:
      tcvr_freq += freq_correction;
      break;

    case K2: case K3:
      tcvr_freq += freq_correction; /* Now using CW-REV */
      break;

  } /* switch( rc_data.tcvr_type ) */
  Write_Rx_Freq( tcvr_freq );

  return( TRUE );
} /* Tune_Tcvr() */

/*-------------------------------------------------------------------------*/

/* Set_Tx_Power()
 *
 * Sets the Tx power output and speech
 * compression level of Elecraft K2 or K3
 */
  gboolean
Set_Tx_Power( int tx_power, int comp_lev )
{
  int power, comp;

  if( (rc_data.tcvr_type == K2) || (rc_data.tcvr_type == K3) )
  {
    /* Read transceiver Tx power setting */
    Write_Tcvr_Serial( K3_POWER_G, sizeof(K3_POWER_G)-1 );
    Read_Tcvr_Serial(  K3_POWER_S, sizeof(K3_POWER_S)-1 );
    power = atoi( K3_POWER_S+2 );

    /* Change power setting if request is different */
    if( power != tx_power )
    {
      snprintf( K3_POWER_S+2, sizeof(K3_POWER_S)-2, "%03d;", tx_power );
      Write_Tcvr_Serial( K3_POWER_S, sizeof(K3_POWER_S)-1 );
    }

    /* Read transceiver compressor setting */
    Write_Tcvr_Serial( K3_COMP_G, sizeof(K3_COMP_G)-1 );
    Read_Tcvr_Serial(  K3_COMP_S, sizeof(K3_COMP_S)-1 );
    comp = atoi( K3_COMP_S+2 );

    /* Change compressor setting if new request is different */
    if( (comp != comp_lev) && (rc_data.tcvr_type == K3) )
    {
      snprintf( K3_COMP_S+2, sizeof(K3_COMP_S)-2, "%03d;", comp_lev );
      Write_Tcvr_Serial( K3_COMP_S, sizeof(K3_COMP_S)-1 );
    }

  } /* if( (rc_data.tcvr_type == K2) || (rc_data.tcvr_type == K3) ) */

  return( TRUE );
} /* Set_Tx_Power() */

/*-------------------------------------------------------------------------*/

