/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sound.h"
#include "shared.h"
#include <math.h>

/* ALSA pcm capture and mixer handles */
static snd_pcm_t *capture_handle  = NULL;
static snd_pcm_t *playback_handle = NULL;
static snd_mixer_t *mixer_handle  = NULL;
static snd_pcm_hw_params_t *hw_params = NULL;

/* Simple mixer elements for setting up
 * capture amd playback sources and volume */
static snd_mixer_elem_t *cap_elem = NULL;
static snd_mixer_elem_t *pcm_elem = NULL;
static snd_mixer_elem_t *mst_elem = NULL;

/* Index to signal samples buffer for Rx */
static int rx_buff_idx;
static short *recv_buffer = NULL;

/*------------------------------------------------------------------------*/

/* Open_PCM()
 *
 * Opens a pcm device for a given handle
 */
  static gboolean
Open_PCM(
	snd_pcm_t **handle,
	snd_pcm_stream_t stream,
	char *mesg, int *error )
{
  /* Open pcm */
  *error = snd_pcm_open(
	  handle, rc_data.pcm_dev, stream, SND_PCM_ASYNC );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot open sound device "), MESG_SIZE );
	Strlcat( mesg, rc_data.pcm_dev, MESG_SIZE );
	return( FALSE );
  }

  /* Allocate memory to hardware parameters structure */
  *error = snd_pcm_hw_params_malloc( &hw_params );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot allocate hw_params struct"), MESG_SIZE );
	return( FALSE );
  }

  /* Initialize hardware parameter structure */
  *error = snd_pcm_hw_params_any( *handle, hw_params );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot initialize hw_params struct"), MESG_SIZE );
	return( FALSE );
  }

  /* Set access type */
  *error = snd_pcm_hw_params_set_access(
	  *handle, hw_params, PCM_ACCESS );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot set PCM access type"), MESG_SIZE );
	return( FALSE );
  }

  /* Set sample format */
  *error = snd_pcm_hw_params_set_format(
	  *handle, hw_params, PCM_FORMAT );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot set sample format"), MESG_SIZE );
	return( FALSE );
  }

  /* Set sample rate */
  *error = snd_pcm_hw_params_set_rate(
	  *handle, hw_params, (unsigned int)rc_data.dsp_rate, EXACT_VAL );
  if( *error < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set sample rate to %d"), rc_data.dsp_rate );
	return( FALSE );
  }

  /* Set channel count */
  *error = snd_pcm_hw_params_set_channels(
	  *handle, hw_params, (unsigned int)rc_data.num_chn );
  if( *error < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set channel count to %d"), rc_data.num_chn );
	return( FALSE );
  }

  /* Set number of periods */
  *error = snd_pcm_hw_params_set_periods(
	  *handle, hw_params, NUM_PERIODS, EXACT_VAL );
  if( *error < 0)
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set number periods to %d"), NUM_PERIODS );
	return( FALSE );
  }

  /* Set period size */
  *error = snd_pcm_hw_params_set_period_size(
	  *handle, hw_params, PERIOD_SIZE, EXACT_VAL);
  if( *error < 0)
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set period size to %d"), PERIOD_SIZE );
	return( FALSE );
  }

  /* Set parameters */
  *error = snd_pcm_hw_params( *handle, hw_params );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot set capture parameters"), MESG_SIZE );
	return( FALSE );
  }
  snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  /* Prepare audio interface for use */
  *error = snd_pcm_prepare( *handle );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot prepare audio interface"), MESG_SIZE );
	return( FALSE );
  }

  return( TRUE );
} /* Open_PCM() */

/*------------------------------------------------------------------------*/

/* Open_Mixer()
 *
 * Opens mixer interface
 */
  static gboolean
Open_Mixer( char *mesg, int *error )
{
  snd_mixer_elem_t *elem;
  snd_mixer_selem_id_t *sid;

  /* Abort if mixer already setup */
  if( isFlagSet(MIXER_SETUP) ) return( TRUE );

  /* Open mixer handle */
  *error = snd_mixer_open( &mixer_handle, 0 );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot open mixer handle"), MESG_SIZE );
	return( FALSE );
  }

  /* Attach mixer */
  *error = snd_mixer_attach( mixer_handle, rc_data.snd_card );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot attach mixer to "), MESG_SIZE );
	Strlcat( mesg, rc_data.snd_card,  MESG_SIZE );
	return( FALSE );
  }

  /* Register mixer */
  *error = snd_mixer_selem_register( mixer_handle, NULL, NULL );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot register mixer"), MESG_SIZE );
	return *error;
  }

  /* Load mixer */
  *error = snd_mixer_load( mixer_handle );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot load mixer"), MESG_SIZE );
	return( FALSE );
  }

  /* Allocate selem_id structure */
  *error = snd_mixer_selem_id_malloc( &sid );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot allocate selem_id struct"), MESG_SIZE );
	return( FALSE );
  }

  /* Find capture source selem */
  snd_mixer_selem_id_set_index( sid, 0 );
  snd_mixer_selem_id_set_name( sid, rc_data.cap_src );
  elem = snd_mixer_find_selem( mixer_handle, sid );
  if( elem == NULL )
  {
	Strlcat( mesg, _("Cannot find capture source element "), MESG_SIZE );
	Strlcat( mesg, rc_data.cap_src,  MESG_SIZE );
	snd_mixer_selem_id_free(sid);
	*error = 0;
	return( FALSE );
  }

  /* Set capture switch for capture source */
  if( snd_mixer_selem_has_capture_switch(elem) )
  {
	*error = snd_mixer_selem_set_capture_switch(
		elem, rc_data.channel, 1 );
	if( *error < 0 )
	{
	  Strlcat( mesg, _("Cannot set capture device "), MESG_SIZE );
	  Strlcat( mesg, rc_data.cap_src, MESG_SIZE );
	  snd_mixer_selem_id_free(sid);
	  return( FALSE );
	}
  }
  else
  {
	snprintf( mesg, MESG_SIZE,
		_("Device %s does not have Capture capability"),
		rc_data.cap_src );
	snd_mixer_selem_id_free(sid);
	*error = 0;
	return( FALSE );
  }

  /* Find capture volume selem if not -- */
  if( strcmp(rc_data.cap_vol, "--") != 0 )
  {
	snd_mixer_selem_id_set_index( sid, 0 );
	snd_mixer_selem_id_set_name( sid, rc_data.cap_vol );
	cap_elem = snd_mixer_find_selem( mixer_handle, sid );
	if( cap_elem == NULL )
	{
	  snprintf( mesg, MESG_SIZE,
		  _("Cannot find volume element %s"),
		  rc_data.cap_vol );
	  Error_Dialog( mesg, OK );
	}
  }

  /* Find Master playback volume selem if not -- */
  if( strcmp(rc_data.mst_vol, "--") != 0 )
  {
	snd_mixer_selem_id_set_index( sid, 0 );
	snd_mixer_selem_id_set_name( sid, rc_data.mst_vol );
	mst_elem = snd_mixer_find_selem( mixer_handle, sid );
	if( mst_elem == NULL )
	{
	  snprintf( mesg, MESG_SIZE,
		  _("Cannot find volume element %s"),
		  rc_data.mst_vol );
	  Error_Dialog( mesg, OK );
	}
  }

  /* Find PCM playback volume selem if not -- */
  if( strcmp(rc_data.pcm_vol, "--") != 0 )
  {
	snd_mixer_selem_id_set_index( sid, 0 );
	snd_mixer_selem_id_set_name( sid, rc_data.pcm_vol );
	pcm_elem = snd_mixer_find_selem( mixer_handle, sid );
	if( pcm_elem == NULL )
	{
	  snprintf( mesg, MESG_SIZE,
		  _("Cannot find volume element %s"),
		  rc_data.pcm_vol );
	  Error_Dialog( mesg, OK );
	}

	/* Set PCM playback switch for playback source */
	if( pcm_elem != NULL )
	{
	  if( snd_mixer_selem_has_playback_switch(pcm_elem) )
	  {
		*error = snd_mixer_selem_set_playback_switch(
			pcm_elem, rc_data.channel, 1 );
		if( *error < 0 )
		{
		  snprintf( mesg, MESG_SIZE,
			  _("Cannot set playback source %s\n"\
				"Error: %s"), rc_data.pcm_vol, snd_strerror(*error) );
		  Error_Dialog( mesg, QUIT );
		}
	  }
	  else
	  {
		snprintf( mesg, MESG_SIZE,
			_("Device %s does not have Playback capability"),
			rc_data.pcm_vol );
		snd_mixer_selem_id_free(sid);
		*error = 0;
		return( FALSE );
	  }
	}
  } /* if( pcm_elem != NULL ) */
  snd_mixer_selem_id_free(sid);

  SetFlag( MIXER_SETUP );
  return( TRUE );
} /* Open_Mixer() */

/*------------------------------------------------------------------------*/

/* Open_Capture()
 *
 * Opens sound card for Capture
 */
  gboolean
Open_Capture( char *mesg, int *error )
{
  /* Return if Capture is setup */
  if( isFlagSet(CAPTURE_SETUP) ) return( TRUE );

  /* Open & setup pcm for Capture */
  if( !Open_PCM(
		&capture_handle,
		SND_PCM_STREAM_CAPTURE,
		mesg, error) )
	return( FALSE );

  /* Size of samples buffer depends on stereo/mono mode */
  rc_data.rxbuff_len = PERIOD_SIZE * rc_data.num_chn;

  /* Index to rx samples buffer (set to end) */
  rx_buff_idx = rc_data.rxbuff_len;

  /* Allocate memory to receive samples buffer */
  if( recv_buffer == NULL )
	if( !mem_alloc((void **)&recv_buffer,
		(size_t)rc_data.rxbuff_len * sizeof(short)) )
	return( FALSE );
  memset( recv_buffer, 0,
	  (size_t)rc_data.rxbuff_len * sizeof(short) );

  /* Open mixer & set playback voulume, abort on failure.
   * Failure to set Volume levels is not taken as fatal */
  if( !Open_Mixer(mesg, error) ) return( FALSE );
  Set_Capture_Level( rc_data.cap_lev, mesg, error );

  SetFlag( CAPTURE_SETUP );
  return( TRUE );
} /* Open_Capture() */

/*------------------------------------------------------------------------*/

/* Open_Playback()
 *
 * Opens sound card for Playback
 */
  gboolean
Open_Playback( char *mesg, int *error )
{
  /* Return if Playback is setup */
  if( isFlagSet(PLAYBACK_SETUP) ) return( TRUE );

  /* Open & setup pcm for Playback */
  if( !Open_PCM(
		&playback_handle,
		SND_PCM_STREAM_PLAYBACK,
		mesg, error) )
	return( FALSE );

  /* Open mixer & set playback volume, abort on failure
   * Failure to set Volume levels is not taken as fatal */
  if( !Open_Mixer(mesg, error) ) return( FALSE );
  Set_Playback_Level( rc_data.pbk_lev, mesg, error );

  SetFlag( PLAYBACK_SETUP );
  return( TRUE );
} /* Open_Playback() */

/*------------------------------------------------------------------------*/

/* Set_Playback_Level()
 *
 * Sets Playback Control level
 */
  gboolean
Set_Playback_Level( int level, char *mesg, int *error )
{
  long cmin, cmax;

  /* Return with no error if Mixer not setup */
  if( mixer_handle == NULL ) return( TRUE );

  /* Set PCM playback volume */
  if( pcm_elem != NULL )
  {
	if( snd_mixer_selem_has_playback_volume(pcm_elem) )
	{
	  /* Change from % volume to sound card value */
	  long lev;
	  snd_mixer_selem_get_playback_volume_range( pcm_elem, &cmin, &cmax );
	  lev = cmin + ((cmax - cmin) * level) / 100;

	  /* Set PCM playback volume */
	  *error = snd_mixer_selem_set_playback_volume(
		  pcm_elem, rc_data.channel, lev );
	  if( *error < 0 )
	  {
		snprintf( mesg, MESG_SIZE,
			_("Cannot set playback volume to %d\n"\
			  "Error: %s"),
			level, snd_strerror(*error) );
		Error_Dialog( mesg, OK );
		return( FALSE );
	  }
	}
  } /* if( pcm_elem != NULL ) */

  /* Set Master playback volume */
  if( mst_elem != NULL )
  {
	if( snd_mixer_selem_has_playback_volume(mst_elem) )
	{
	  /* Change from % volume to sound card value */
	  long lev;
	  snd_mixer_selem_get_playback_volume_range(
		  mst_elem, &cmin, &cmax );
	  lev  = cmin + ((cmax - cmin) * level) / 100;

	  /* Set playback volume */
	  *error = snd_mixer_selem_set_playback_volume(
		  mst_elem, rc_data.channel, lev );
	  if( *error < 0 )
	  {
		snprintf( mesg, MESG_SIZE,
			_("Cannot set playback volume to %d\n"\
			  "Error: %s"),
			level, snd_strerror(*error) );
		Error_Dialog( mesg, OK );
		return( FALSE );
	  }
	}
  } /* if( mst_elem != NULL ) */

  return( TRUE );
} /* Set_Playback_Level() */

/*------------------------------------------------------------------------*/

/* Set_Capture_Level()
 *
 * Sets Capture Control level
 */
  gboolean
Set_Capture_Level( int level, char *mesg, int *error )
{
  long cmin, cmax;

  /* Return with no error if Mixer not setup */
  if( mixer_handle == NULL ) return( TRUE );

  /* Set capture volume */
  if( cap_elem != NULL )
  {
	if( snd_mixer_selem_has_capture_volume(cap_elem) )
	{
	  /* Change from % volume to sound card value */
	  long lev;
	  snd_mixer_selem_get_capture_volume_range(
		  cap_elem, &cmin, &cmax );
	  lev = cmin + ((cmax - cmin) * level) / 100;

	  /* Set capture volume */
	  *error = snd_mixer_selem_set_capture_volume(
		  cap_elem, rc_data.channel, lev );
	  if( *error < 0 )
	  {
		snprintf( mesg, MESG_SIZE,
			_("Cannot set capture volume to %d\n"\
			  "Error: %s"), level, snd_strerror(*error) );
		Error_Dialog( mesg, OK );
		return( FALSE );
	  }
	}
  } /* if( cap_elem != NULL ) */

  return( TRUE );
} /* Set_Capture_Level() */

/*------------------------------------------------------------------------*/

  void
Close_Playback( void )
{
  if( playback_handle != NULL )
	snd_pcm_close( playback_handle );
  playback_handle = NULL;

  if( hw_params != NULL )
	snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  ClearFlag(PLAYBACK_SETUP);
}

/*------------------------------------------------------------------------*/

  void
Close_Capture( void )
{
  if( capture_handle != NULL )
	snd_pcm_close( capture_handle );
  capture_handle = NULL;

  if( hw_params != NULL )
	snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  ClearFlag(CAPTURE_SETUP);
}

/*------------------------------------------------------------------------*/

  void
Close_Mixer( void )
{
  if( mixer_handle != NULL )
	snd_mixer_close( mixer_handle );
  mixer_handle = NULL;

  ClearFlag(MIXER_SETUP);
}

/*------------------------------------------------------------------------*/

/* Xrun_Recovery()
 *
 * Recover from underrrun (broken pipe) and suspend
 */
  static gboolean
Xrun_Recovery( snd_pcm_t *handle, int error )
{
  if( error == -EPIPE )
  {
	error = snd_pcm_prepare( handle );
	if( error < 0 )
	{
	  char mesg[MESG_SIZE];
	  snprintf( mesg, sizeof(mesg),
		  _("Cannot recover from underrun, prepare failed\n"
			"Error: %s\n"), snd_strerror(error) );
	  Error_Dialog( mesg, QUIT );
	  return( FALSE );
	}
  }
  else if( error == -ESTRPIPE )
  {
	while( (error = snd_pcm_resume(handle)) == -EAGAIN )
	  sleep(1);
	if( error < 0 )
	{
	  error = snd_pcm_prepare( handle );
	  if( error < 0 )
	  {
		char mesg[MESG_SIZE];
		snprintf( mesg, sizeof(mesg),
			_("Cannot recover from suspend, prepare failed\n"\
			  "Error: %s\n"), snd_strerror(error) );
		Error_Dialog( mesg, QUIT );
		return( FALSE );
	  }
	}
  }

  return( TRUE );
} /* Xrun_Recovery() */

/*------------------------------------------------------------------------*/

/*  Signal_Sample()
 *
 *  Gets the next DSP sample of the signal input.
 */

  gboolean
Signal_Sample( short *sample_val )
{
  snd_pcm_sframes_t error;

  /* Read samples from DSP if in Receive mode */
  if( isFlagSet(RECEIVE_MODE) && isFlagClear(KEYBD_BUSY) )
  {
	/* Refill DSP samples buffer when needed */
	if( rx_buff_idx >= rc_data.rxbuff_len )
	{
	  /* Read audio samples from DSP, abort on error */
	  error = snd_pcm_readi( capture_handle, recv_buffer, PERIOD_SIZE );
	  if( error != PERIOD_SIZE )
	  {
		fprintf( stderr, "xfhell: Signal_Sample(): %s\n",
			snd_strerror((int)error) );

		/* Try to recover from error */
		if( !Xrun_Recovery(capture_handle, (int)error) )
		  return( FALSE );
	  } /* if( error != RX_BUFFER_SIZE ) */

	  /* Start buffer index according to stereo/mono mode */
	  rx_buff_idx = rc_data.use_chn;

	} /* if( rx_buff_idx >= buffer_size ) */

	/* Get a sample from buffer */
	*sample_val = recv_buffer[rx_buff_idx];

	/* Increment according to mono/stereo mode */
	rx_buff_idx += rc_data.num_chn;

  } /* if( isFlagSet(RECEIVE_MODE) && isFlagClear(KEYBD_BUSY) ) */
  else if( isFlagSet(TX2RX_LOOPBACK) )
  {
	*sample_val = xmit_buffer[tx_buff_idx];
	tx_buff_idx += rc_data.num_chn;
  } /* if( isFlagSet(RECEIVE_MODE) ) */

  /* Display waterfall when input buffer full.
   * ifft_stride samples are added for each input element */
  if( isFlagClear(ENABLE_SCOPE) )
	IFFT_Data( *sample_val );

  return( TRUE );
} /* End of Signal_Sample() */

/*------------------------------------------------------------------------*/

/*  DSP_Write()
 *
 *  Write a samples buffer to DSP
 */

  gboolean
DSP_Write( short *buffer, int buff_size )
{
  snd_pcm_sframes_t error;
  int num_writes, idx;
  short *buff_ptr;
  snd_pcm_uframes_t num_frames;

  if( playback_handle == NULL ) return( FALSE );

  /* Number of writes needed to write all data in buffer */
  num_writes = buff_size / PERIOD_SIZE + 1;

  /* Index (poiner) to samples buffer */
  buff_ptr = buffer;

  /* Write samples buffer to DSP in chunks of PERIOD_SIZE */
  for( idx = 0; idx < num_writes; idx++ )
  {
	/* Calculate number of frames to write. It should be = PERIOD_SIZE
	 * if we have >= PERIOD_SIZE samples left to write, or =
	 * buff_size - (num_writes-1) * PERIOD_SIZE otherwise */
	num_frames = (snd_pcm_uframes_t)(buff_size - idx * PERIOD_SIZE);
	if( num_frames > PERIOD_SIZE ) num_frames = PERIOD_SIZE;

	  /* Write samples buffer to DSP */
	  error = snd_pcm_writei(
		  playback_handle, buff_ptr, num_frames );
	  if( error != (snd_pcm_sframes_t)num_frames )
	  {
		if( error < 0 )
		  fprintf( stderr, "xfhell: DSP_Write(): %s\n",
			  snd_strerror((int)error) );
		else fprintf( stderr,
			"xfhell: DSP_Write(): Written only %d frames out of %d\n",
			(int)error, (int)num_frames );

		/* Try to recover from error */
	  if( !Xrun_Recovery(playback_handle, (int)error) )
		return( FALSE );

	} /* if( error != (snd_pcm_sframes_t)buff_size ) */

	buff_ptr += PERIOD_SIZE * rc_data.num_chn;
  } /* for( idx = 0; idx < num_writes; idx++ ) */

  return( TRUE );
} /* DSP_Write() */

/*------------------------------------------------------------------------*/

