/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "operation.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Initialize_Transmit()
 *
 *  Initializes the Transmit mode
 */

  static gboolean
Initialize_Transmit(gpointer data)
{
  int error;
  char mesg[MESG_SIZE];

  /* Setup sound card, abort on error */
  mesg[0]= '\0';
  if( !Open_Playback(mesg, &error) )
  {
	if( error )
	{
	  Strlcat( mesg, _("\nError: "), sizeof(mesg) );
	  Strlcat( mesg, snd_strerror(error), sizeof(mesg) );
	}
	Close_Capture();

	ClearFlag(
		TRANSMIT_KEYBD |
		TRANSMIT_MACRO |
		TRANSMIT_MODE );
	Set_TxRx_Labels();
	Error_Dialog( mesg, QUIT );
	return( FALSE );
  }

  /* Clear buffer */
  memset( xmit_buffer, 0,
	  (size_t)rc_data.txbuff_len *
	  (size_t)rc_data.num_chn * sizeof(short) );

  /* Switch to USB and PTT ON, enable pixbuf save */
  SetFlag( TRANSMIT_MODE );
  if( isFlagSet(ENABLE_CAT) )
  {
	if( !Open_Tcvr_Serial() || !Set_Tcvr_Mode() )
	{
	  ClearFlag(
		  TRANSMIT_KEYBD |
		  TRANSMIT_MACRO |
		  TRANSMIT_MODE );
	  Set_TxRx_Labels();
	  Close_Playback();
	  return( FALSE );
	}
  } /* if( isFlagSet(ENABLE_CAT) ) */
  else if( isFlagSet(ENABLE_RTS_DTR) )
  {
	/* Setup RTS/DTR control */
	if( !Open_RTS_DTR_Serial() || !Set_Tcvr_Mode() )
	{
	  ClearFlag(
		  TRANSMIT_KEYBD |
		  TRANSMIT_MACRO |
		  TRANSMIT_MODE );
	  Set_TxRx_Labels();
	  Close_Playback();
	  return( FALSE );
	}
  }
  Set_TxRx_Labels();

  /* Open qso record file */
  if( !Open_Record_File() )
  {
	ClearFlag(
		TRANSMIT_KEYBD |
		TRANSMIT_MACRO |
		TRANSMIT_MODE );
	Set_TxRx_Labels();
	Close_Playback();
	return( FALSE );
  }

  return( TRUE );
} /* Initialize_Transmit() */

/*------------------------------------------------------------------------*/

  gboolean
Transmit_Keybd( gpointer data )
{
  /* First call of function flag */
  static gboolean first_call = TRUE;


  /*** Cancel Transmit if Tx flag cleared ***/
  if( isFlagClear(TRANSMIT_KEYBD) &&
	  isFlagClear(KEYBD_BUSY) )
  {
	if( isFlagClear(TRANSMIT_MACRO) )
	{
	  ClearFlag( TRANSMIT_MODE );
	  Close_Playback();
	}
	first_call = TRUE;
	return( FALSE );
  } /* if( isFlagClear(TRANSMIT_KEYBD) ) */

  /*** Initialize if needed ***/
  if( first_call )
  {
	first_call = FALSE;
	gtk_widget_grab_focus(
		Builder_Get_Object(main_window_builder, "tx_textview") );

	if( isFlagClear(TRANSMIT_MODE) )
	  if( !Initialize_Transmit(data) )
	  {
		first_call = TRUE;
		return( FALSE );
	  }

	return( TRUE );
  } /* if( first_call ) */

  /* Transmit key buffer */
  if( space_cnt > 0 )
  {
	/* Xmit "smiley" etc glyphs with (shift) F1-F12 */
	if( (keybd_buff[txch_ptr] >= GDK_KEY_F1) &&
		(keybd_buff[txch_ptr] <= GDK_KEY_F12) )
	{
	  if( !Transmit_Character(&rc_data.font_data,
			(int)(16 + keybd_buff[txch_ptr] - GDK_KEY_F1)) )
		return( FALSE );
	}
	else
	{
	  /* Illegal chars are ignored */
	  if( ((int)keybd_buff[txch_ptr] >= rc_data.font_data.first_glyph) &&
		  ((int)keybd_buff[txch_ptr] <= rc_data.font_data.last_glyph) )
	  {
		if( !Transmit_Character(&rc_data.font_data, (int)keybd_buff[txch_ptr]) )
		  return( FALSE );
	  }
	}

	/* Count down spaces */
	if( (keybd_buff[txch_ptr] == ' ') ||
		(keybd_buff[txch_ptr] == '\0') )
	  space_cnt--;

	/* Reset key buffer pointer. It must go back one count
	 * to align with the keystroke buffer pointer due
	 * to the extra space inserted after each character */
	if( !space_cnt ) txch_ptr--;
	if( ++txch_ptr == KEY_BUFF_SIZE )
	  txch_ptr = 0;

  } /* if( space_cnt > 0 ) */
  else	/* Transmit idle character */
  {
	if( !Transmit_Character(&rc_data.font_data, IDLE_CHAR) )
	  return( FALSE );

	ClearFlag( KEYBD_BUSY );

	if( isFlagSet(TRANSMIT_MACRO) )
	  ClearFlag( TRANSMIT_KEYBD );
	else if( isFlagClear(TRANSMIT_KEYBD) )
	  ClearFlag( TRANSMIT_MODE );
  }

  return( TRUE );
} /* End of Transmit_Keybd() */

/*------------------------------------------------------------------------*/

/*  Transmit_Macro()
 *
 *  Transmits a pre-recorded Macro
 */

  gboolean
Transmit_Macro( gpointer data )
{
  /* Text buffer marker */
  static GtkTextIter iter;

  static int
	macro_cnt = 0, /* Count macro characters tx'd */
	tag_idx   = 0; /* Index for transmitting tags */

  /* First call of function flag */
  static gboolean first_call = TRUE;

  /* Tags for entering fields in macros */
  char *tags[NUM_OF_TAGS] = TAG_TABLE;

  /* String within a tag <> */
  char tag_str[13];

  /* Tag replacement string */
  static char tag_rep[22];


  /* Delay macro till keybd not busy */
  if( isFlagSet(KEYBD_BUSY) ) return( TRUE );

  /* Abort macro if user disabled, remain in Tx mode */
  /* Stop macro on '~' character, remain in Tx mode  */
  if( isFlagClear(TRANSMIT_MACRO) ||
	  (macro[macro_idx][macro_cnt] == '~') )
  {
	if( !Transmit_Character(&rc_data.font_data, IDLE_CHAR) )
	  return( FALSE );

	if( isFlagSet(RECORD_QSO) )
	  fflush( rc_data.qso_record_fp );

	macro_cnt = tag_idx = 0;
	first_call = TRUE;
	ClearFlag( TRANSMIT_MACRO |	TRANSMIT_TAG );
	SetFlag( TRANSMIT_KEYBD );
	g_idle_add( Transmit_Keybd, NULL );

	return( FALSE );
  }

  /* Stop macro at string terminator  */
  /* or CW id char (%), go to Rx mode */
  if( (macro[macro_idx][macro_cnt] == '\0') ||
	  (macro[macro_idx][macro_cnt] == '*') )
  {
	if( isFlagSet(RECORD_QSO) )
	  fflush( rc_data.qso_record_fp );

	/* Transmit Morse ID */
	if( macro[macro_idx][macro_cnt] == '*' )
	{
	  if( !Morse_Transmit(rc_data.cw_mesg) )
	  {
		Close_Playback();
		return( FALSE );
	  }
	}

	macro_cnt = tag_idx = 0;
	first_call = TRUE;
	ClearFlag(
		TRANSMIT_MACRO |
		TRANSMIT_KEYBD |
		TRANSMIT_MODE  |
		TRANSMIT_TAG );
	SetFlag( RECEIVE_MODE );
	Close_Playback();
	g_idle_add( Receive_Mode, NULL );
	return( FALSE );
  }

  /*** Initialize if needed ***/
  if( first_call )
  {
	first_call = FALSE;
	if( isFlagClear(TRANSMIT_MODE) )
	  if( !Initialize_Transmit(data) )
	  {
		first_call = TRUE;
		return( FALSE );
	  }

	return( TRUE );
  } /* if( first_call ) */

  /* Set up TRANSMIT text view */
  gtk_text_buffer_get_iter_at_offset(
	  tx_text_buffer, &iter,
	  gtk_text_buffer_get_char_count
	  (tx_text_buffer) );

  /*** Look for tags and substitude accordingly ***/
  if( (macro[macro_idx][macro_cnt] == '<') &&
	  isFlagClear(TRANSMIT_TAG) )
  {
	SetFlag( TRANSMIT_TAG );

	/*** Copy tag string to buffer and identify ***/
	macro_cnt++; /* Move into tag field */

	/* Copy line to tag, max 12 chars or EOS */
	int idx = 0;
	size_t siz = sizeof( tag_str ) - 1;
	while( (idx < (int)siz) &&
		(macro[macro_idx][macro_cnt] != '>') )
	{
	  tag_str[idx] = macro[macro_idx][macro_cnt];
	  idx++;
	  macro_cnt++;
	}

	/* Terminate tag buffer and advance macro buffer */
	tag_str[idx] = '\0';
	macro_cnt++;

	/* Identify tag and substitude accordingly */
	idx = 0;
	while( idx < NUM_OF_TAGS )
	{
	  if( strcmp(tag_str, tags[idx]) == 0 )	break;
	  idx++;
	}

	/* Abort if tag unidentified */
	if( idx == NUM_OF_TAGS )
	{
	  /* If tag is not a number */
	  if( (tag_str[0] < 0x30) && (tag_str[0] > 0x39) )
	  {
		Error_Dialog(
			_("Error reading xfhellrc file\n"\
			  "A tag in a macro is invalid\n"\
			  "Quit xfhell and correct"), QUIT );
		Close_Playback();
		return( FALSE );
	  }
	  else
	  {
		/* Take number as glyph encoding and xmit */
		if( !Transmit_Character(&rc_data.font_data, atoi(tag_str)) )
		  return( FALSE );
		ClearFlag( TRANSMIT_TAG );
		return( TRUE );
	  }
	}

	/* Replace tags */
	siz = sizeof( tag_rep );
	switch( idx )
	{
	  case 0: /* own-call */
		Strlcpy( tag_rep, rc_data.call, siz );
		break;

	  case 1: /* own-name */
		Strlcpy( tag_rep, rc_data.name, siz );
		break;

	  case 2: /* own-qth */
		Strlcpy( tag_rep, rc_data.qth, siz );
		break;

	  case 3: /* own-loc */
		Strlcpy( tag_rep, rc_data.loc, siz );
		break;

	  case 4: /* own-rst */
		Strlcpy( tag_rep, qso_record.my_rst, siz );
		break;

	  case 5: /* rem-call */
		Strlcpy( tag_rep, qso_record.dx_call, siz );
		break;

	  case 6: /* rem-name */
		Strlcpy( tag_rep, qso_record.dx_name, siz );
		break;

	  case 7: /* rem-qth */
		Strlcpy( tag_rep, qso_record.dx_qth, siz );
		break;

	  case 8: /* rem-loc */
		Strlcpy( tag_rep, qso_record.dx_loc, siz );
		break;

	  case 9: /* rem-rst */
		Strlcpy( tag_rep, qso_record.dx_rst, siz );
		break;

	  case 10: /* date-time */
		Strlcpy( tag_rep, qso_record.date, siz );
		Strlcat ( tag_rep, " ", siz );
		Strlcat( tag_rep, qso_record.time, siz );
		break;

	  case 11: /* op-freq, strip trailing spaces */
		{
		  int i;

		  Strlcpy( tag_rep, qso_record.freq, siz );
		  i = (int)strlen( tag_rep ) - 1;
		  for( ; i >= 0; i-- )
			if( tag_rep[i] == ' ' )
			  tag_rep[i] = '\0';
		}
		break;

	  case 12: /* app-version */
		snprintf( tag_rep, siz, "%s", PACKAGE_STRING );

	} /* switch( idx ) */

  } /* if( (macro[macro_idx][macro_cnt] == '<') && ) */

  /*** Transmit tag replacement ***/
  if( isFlagSet(TRANSMIT_TAG) )
  {
	if( tag_rep[tag_idx] != '\0' )
	{
	  /* Capitalize letters if enabled */
	  if( isFlagSet(CAPITALIZE) &&
		  (tag_rep[tag_idx] > 0x60)  &&
		  (tag_rep[tag_idx] < 0x7b) )
		tag_rep[tag_idx] -= 0x20;

	  if( !Transmit_Character(
			&rc_data.font_data, tag_rep[tag_idx]) )
		return( FALSE );
	  Print_Character( tag_rep[tag_idx], tx_text_buffer,
		  tx_scrolledwindow, &iter );
	  tag_idx++;

	} /* if( (tag_rep[tag_idx] != '\0') ) */
	else
	{
	  ClearFlag( TRANSMIT_TAG );
	  tag_idx = 0;
	}

	return( TRUE );

  } /* if( isFlagSet(TRANSMIT_TAG) && */

  /* Capitalize letters if enabled */
  if( isFlagSet(CAPITALIZE) &&
	  (macro[macro_idx][macro_cnt] > 0x60) &&
	  (macro[macro_idx][macro_cnt] < 0x7b) )
	macro[macro_idx][macro_cnt] -= 0x20;

  /* Transmit a char from Macro */
  if( (macro[macro_idx][macro_cnt] == CR) ||
	  (macro[macro_idx][macro_cnt] == LF) )
  {
	if( !Transmit_Character(&rc_data.font_data, ' ') )
	  return( FALSE );
  }
  else if( !Transmit_Character(
		&rc_data.font_data, macro[macro_idx][macro_cnt]) )
	return( FALSE );

  Print_Character( macro[macro_idx][macro_cnt],
	  tx_text_buffer, tx_scrolledwindow, &iter );
  macro_cnt++;

  return( TRUE );
} /* Transmit_Macro() */

/*------------------------------------------------------------------------*/

/* Receive_Mode()
 *
 * Controls receiving and decoding Hell signals
 */
  gboolean
Receive_Mode( gpointer data )
{
  /* First call of function flag */
  static gboolean first_call = TRUE;

  /* Character column data ("element" values) */
  static unsigned char *column = NULL;
  static int bm_height = 0;


  /* Cancel Receive if Rx flag cleared */
  if( isFlagClear(RECEIVE_MODE) )
  {
	Close_Capture();
	first_call = TRUE;
	return( FALSE );
  }

  /* Delay receive till keyboard not busy */
  if( isFlagSet(KEYBD_BUSY) ) return( TRUE );

  /* Allocate memory to column */
  if( bm_height != rc_data.bitmap_height )
  {
	bm_height = rc_data.bitmap_height;
	if( !mem_realloc((void **)&column,
		  sizeof(unsigned char) * (size_t)bm_height) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
  }

  /*** Initialize ***/
  if( first_call )
  {
	char mesg[MESG_SIZE];
	int error;

	first_call = FALSE;

	/* Setup sound card, abort on error */
	mesg[0]= '\0';
	if( !Open_Capture(mesg, &error) )
	{
	  if( error )
	  {
		Strlcat( mesg, _("\nError: "), sizeof(mesg) );
		Strlcat( mesg, snd_strerror(error), sizeof(mesg) );
	  }
	  Close_Capture();
	  ClearFlag( RECEIVE_MODE );
	  Set_TxRx_Labels();
	  first_call = TRUE;
	  Error_Dialog( mesg, QUIT );
	  return( FALSE );
	}

	/* Set tcvr mode, abort on error */
	if( isFlagSet(ENABLE_CAT) )
	{
	  if( !Open_Tcvr_Serial() || !Set_Tcvr_Mode() )
	  {
		ClearFlag( RECEIVE_MODE );
		Set_TxRx_Labels();
		Close_Capture();
		first_call = TRUE;
		return( FALSE );
	  }
	} /* if( isFlagSet(ENABLE_CAT) ) */
	else if( isFlagSet(ENABLE_RTS_DTR) )
	{
	  /* Setup RTS/DTR control */
	  if( !Open_RTS_DTR_Serial() ||
		  !Set_Tcvr_Mode() )
	  {
		ClearFlag( RECEIVE_MODE );
		Set_TxRx_Labels();
		Close_Capture();
		first_call = TRUE;
		return( FALSE );
	  }
	}

	/* Abort if QSO record file fails to open */
	if( !Open_Record_File() )
	{
	  Close_Capture();
	  first_call = TRUE;
	  return( FALSE );
	}

	Set_TxRx_Labels();
	return( TRUE );
  } /* if( first_call ) */

  /*** Draw characters to Receive window ***/

  /* Make and draw a column, abort on dsp error */
  if( !Assemble_Column(column) )
  {
	Close_Capture();
	return( FALSE );
  }

  if( !Draw_Column(column) )
  {
	Close_Capture();
	return( FALSE );
  }

  return( TRUE );
} /* End of Receive_Mode() */

/*------------------------------------------------------------------------*/

