/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "main.h"
#include "shared.h"

/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

  int
main (int argc, char *argv[])
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;


  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  gtk_init (&argc, &argv);

  /* Process command line options */
  while( (option = getopt(argc, argv, "hv") ) != -1 )
	switch( option )
	{
	  case 'h': /* Print usage and exit */
		Usage();
		exit( 0 );

	  case 'v': /* Print version */
		puts( PACKAGE_STRING );
		exit( 0 );

	  default: /* Print usage and exit */
		Usage();
		exit( -1 );

	} /* End of switch( option ) */

  /* Null pixbuf pointers */
  receive_pixbuf.pixbuf = NULL;
  wfall_pixbuf.pixbuf   = NULL;

  /* Create main window */
  Strlcpy( rc_data.xfhell_glade,
	  getenv("HOME"), sizeof(rc_data.xfhell_glade) );
  Strlcat( rc_data.xfhell_glade,
	  "/xfhell/xfhell.glade", sizeof(rc_data.xfhell_glade) );
  main_window = create_main_window( &main_window_builder );
  gtk_window_set_title( GTK_WINDOW(main_window), PACKAGE_STRING );
  popup_menu = create_popup_menu( &popup_menu_builder );

  /* Get drawing area widget */
  drawingarea = Builder_Get_Object( main_window_builder, "rx_drawingarea" );

  /* Get Tx text buffer and scroller */
  tx_text_buffer = gtk_text_view_get_buffer
	( GTK_TEXT_VIEW(Builder_Get_Object(main_window_builder, "tx_textview")) );
  tx_scrolledwindow = Builder_Get_Object( main_window_builder, "tx_scrolledwindow" );

  /* Get scope widget */
  scope = Builder_Get_Object( main_window_builder, "scope" );
  scope_label = Builder_Get_Object( main_window_builder, "scope_label" );

  /* Get and enter contrast value */
  GtkRange *range =
	GTK_RANGE( Builder_Get_Object(main_window_builder, "contrast") );
  GtkAdjustment *adjust =
	GTK_ADJUSTMENT( gtk_range_get_adjustment(range) );
  int val = (int)( gtk_adjustment_get_value(adjust) );
  contrast = ELEMENT_MAX - (ELEMENT_MAX / 10) * val;

  /* Set labels to Xmit and Rcve buttons */
  {
	GtkWidget *button;

	button = Builder_Get_Object( main_window_builder, "xmit" );
	xmit_status = gtk_label_new( _("TRANSMIT") );
	gtk_widget_show( xmit_status );
	gtk_container_add( GTK_CONTAINER(button), xmit_status );
	g_object_set_data_full( G_OBJECT(main_window), "xmit_status",
		g_object_ref(xmit_status), (GDestroyNotify)g_object_unref );

	button = Builder_Get_Object( main_window_builder, "rcve" );
	rcve_status = gtk_label_new( _("RECEIVE") );
	gtk_widget_show( rcve_status );
	gtk_container_add( GTK_CONTAINER(button), rcve_status );
	g_object_set_data_full( G_OBJECT(main_window), "rcve_status",
		g_object_ref( rcve_status ), (GDestroyNotify)g_object_unref );
  }

  /* Set default flags */
  SetFlag( TX2RX_LOOPBACK );
  SetFlag( RECORD_QSO );
  SetFlag( SAVE_PIXBUF );
  rc_data.num_rows  = 2;
  rc_data.rev_video = FALSE;
  rc_data.log_adif_fp    = NULL;
  rc_data.station_log_fp = NULL;

  /* Show main window */
  gtk_widget_show (main_window);

  /* Get sizes of displays */
  GtkAllocation alloc;
  gtk_widget_get_allocation( drawingarea, &alloc );
  rc_data.rxarea_width  = alloc.width;
  rc_data.rxarea_height = alloc.height;
  if( !Create_Rx_Pixbuf() )
	Error_Dialog( _("Failed to create pixbuf for Rx window"), QUIT );
  SetFlag( CLEAR_RX_WINDOW );

  gtk_widget_get_allocation( scope, &alloc );
  if( !Create_Scope_Pixbuf(alloc.width, alloc.height) )
	Error_Dialog( _("Failed to create pixbuf for waterfall"), QUIT );

  /* Load runtime config file, abort on error */
  g_idle_add( Load_Config, main_window );

  gtk_main ();

  return 0;
}

/*-------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  /* Wrap up and quit */
  Cleanup();
  fprintf( stderr, "\n" );
  switch( signal )
  {
	case SIGINT :
	  fprintf( stderr, "%s\n",  _("xfhell: Exiting via User Interrupt") );
	  exit( signal );

	case SIGSEGV :
	  fprintf( stderr, "%s\n",  _("xfhell: Segmentation Fault") );
	  exit( signal );

	case SIGFPE :
	  fprintf( stderr, "%s\n",  _("xfhell: Floating Point Exception") );
	  exit( signal );

	case SIGABRT :
	  fprintf( stderr, "%s\n",  _("xfhell: Abort Signal received") );
	  exit( signal );

	case SIGTERM :
	  fprintf( stderr, "%s\n",  _("xfhell: Termination Request received") );
	  exit( signal );
  }

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

