/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "codec.h"
#include "shared.h"

/* sine and cosine lookup tables */
static short int
  *sine_table   = NULL,
  *cosine_table = NULL;

/*------------------------------------------------------------------------*/

/*  Assemble_Column()
 *
 *  Assembles a character column ( 14 or 12 dots) from incoming
 *  Hellschreiber pulse train. The column is stored as an array
 *  of signal intensity values for the elements that comprise
 *  each dot, depending on dot size. For a 2x4 pixel/dot font size,
 *  there would be 4 elements/dot e.g. 1 elements/pixel vertically.
 */

  gboolean
Assemble_Column( unsigned char *column )
{
 /* Element dot values for a column */
  static int *dots = NULL;

  int
	elem_lev, 	/* Signal level during a element  */
	elem_max,	/* Max value of column's elements */
	row_min,	/* Minimum value of row averages  */
	row_max,	/* Maximum value of row averages  */
	col_ave,	/* Average value of row averages  */
	min_idx,	/* Index to row with minimum value */
	max_idx,	/* Index to row with maximum value */
	temp, idx, idc, idd;

  /* Average value of element max */
  static int
	*row_ave = NULL, 	/* Average element value per row */
	fbb_h  = 0,			/* Font bbx height flag */
	fbb_h2 = 0,			/* Font bbx height flag / 2 */
	deskew_idx  = 0,	/* Index used to calc resampling */
	deskew_intg = 0,	/* Integral of deskew factor */
	ave_max  = 0;		/* Average of element max values */


  /* Allocate memory to row_average */
  if( !fbb_h || (fbb_h != rc_data.font_data.fbb_h) )
  {
	fbb_h  = rc_data.font_data.fbb_h;
	fbb_h2 = fbb_h / 2;
	if( !mem_realloc((void **)&row_ave,
		  sizeof(int) * (size_t)fbb_h) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
	bzero( row_ave, sizeof(int) * (size_t)fbb_h );

	/* Allocate the element dots buffer */
	if( !mem_realloc((void **)&dots,
		  sizeof(int) * (size_t)fbb_h) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
  } /* if( fbb_h != rc_data.font_data.fbb_h ) */

  /* Assemble a column's dots. The entries in the elem */
  /* array represent "elements" of dots. There is one  */
  /* element/pixel in a dot, according to its height.  */
  elem_max = min_idx = max_idx = row_max = col_ave = 0;
  row_min = 100000;
  for( idx = 0; idx < fbb_h; idx++ )
  {
	/* Get the sig level of an Element dot and clamp */
	if( !Get_Element(&elem_lev) ) return( FALSE );
	if( elem_lev < 0 )   elem_lev = 0;
	if( elem_lev > 255 ) elem_lev = 255;
	dots[ idx ] = elem_lev;

	/* Find average value of each row of a character.
	 * Needed for automatic deskewing of the display. */
	if( isFlagSet(AUTO_DESKEW) )
	{
	  row_ave[idx] *= ( ROW_AVE_WIN - 1 );
	  row_ave[idx] += elem_lev;
	  row_ave[idx] /= ROW_AVE_WIN;
	  col_ave += row_ave[idx];

	  /* Find row with the lowest value */
	  if( row_ave[idx] < row_min )
	  {
		row_min = row_ave[idx];
		min_idx = idx;
	  }

	  /* Find row with the highest value */
	  if( row_ave[idx] > row_max )
	  {
		row_max = row_ave[idx];
		max_idx = idx;
	  }

	} /* if( isFlagSet(AUTO_DESKEW) ) */

	/* Find max of element levels */
	if( elem_lev > elem_max )
	  elem_max = elem_lev;
  } /* for( idx = 0; idx < fbb_h; idx++ ) */

  /* Try to correct skew by adjusting sampling freq */
  if( isFlagSet(AUTO_DESKEW) )
  {
	/* Select either index to min level row (usually
	 * when characters are being sent) or index to max
	 * level row (usually when idle dots are being sent)
	 * On reverse video the test has to be reversed */
	col_ave /= fbb_h;
	gboolean test = row_max < (3 * col_ave);
	if( rc_data.rev_video ) test = !test;

	if( test )	/* Receiving characters (normal video) */
	  deskew_idx = min_idx;
	else		/* Receiving idle dots (normal video) */
	  deskew_idx = max_idx;

	/* Change the direction of skew
	 * correction around mid column */
	if( deskew_idx > fbb_h2 )
	  deskew = -deskew_idx;
	else
	  deskew = deskew_idx;

	/* Integrate error if in limits */
	if( (deskew > 0) && (deskew_intg < fbb_h) )
	  deskew_intg++;
	if( (deskew < 0) && (deskew_intg > -fbb_h) )
	  deskew_intg--;

	/* Make it proportional + integral control */
	deskew += deskew_intg;
	if( deskew > fbb_h )  deskew = fbb_h;
	if( deskew < -fbb_h ) deskew = -fbb_h;

  } /* if( isFlagSet(AUTO_DESKEW) ) */
  else deskew = 0;

  /* Sliding window average of element max */
  ave_max *= ( MAXSEG_AVE_WIN - 1 );
  ave_max += elem_max;
  ave_max /= MAXSEG_AVE_WIN;

  /*** Scale element values to a max of 255 ***/
  /* Limits up-scaling of noise */
  temp = contrast * MAXSEG_AVE_WIN;
  if( ave_max < temp ) ave_max = temp;
  if( !ave_max ) ave_max = 1;

  /* Scale element values to a max of 255 */
  idc = 0;
  for( idx = 0; idx < fbb_h; idx++ )
  {
	if( ++min_idx >= fbb_h ) min_idx = 0;

	elem_lev = 255 * MAXSEG_AVE_WIN * dots[idx] / ave_max;
	if( elem_lev > 255 ) elem_lev = 255;

	/* Fill in pixels of element dot */
	for( idd = 0; idd < rc_data.dot_size; idd++ )
	  column[idc++] = (unsigned char)elem_lev;
  }

  return( TRUE );
} /* Assemble_Column() */

/*------------------------------------------------------------------------*/

/* Tx_Feld_Character()
 *
 * Transmits one Feld Hell character pointed by "glx"
 */

  static gboolean
Tx_Feld_Character( font_data_t *font_data, int glx )
{
  int
	buff_idx,	/* Index to audio samples buffer  */
	cosine_idx,	/* Index to cos wave lookup table */
	bmx,		/* Bitmap x index */
	bmy,		/* Bitmap y index */
	half_dot,	/* Duration in DSP samples of half a dot */
	idx;

  static int
	sine_idx  = 0, /* Index to sin wave lookup table */
	tone_ctrl = 0; /* Tone raise/lower control indx */

  unsigned int
	last_dot, /* State of last dot transmitted */
	new_dot;  /* State of new dot to transmit  */

  /* Column buffer for drawing chars */
  static unsigned char *column = NULL;
  static int bm_height = 0;

  /* Allocate memory to column. bitmap_height is in pixels */
  if( bm_height != rc_data.bitmap_height )
  {
	bm_height = rc_data.bitmap_height;
	if( !mem_realloc((void **)&column,
		  sizeof(unsigned char) * (size_t)bm_height) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
  }

  /* Adjust glyph index */
  glx -= font_data->first_glyph;

  /* Transmit columns */
  half_dot = rc_data.samples_per_dot / 2;
  for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ )
  {
	/* Initialize buffer idx according to stereo/mono mode */
	buff_idx = rc_data.use_chn;

	/* Reverse video if activated */
	if( rc_data.rev_video )
	  last_dot = 1;
	else
	  last_dot = 0;

	/* Go up in bitmap buffer row by row */
	for( bmy = font_data->fbb_h - 1; bmy >= 0; bmy-- )
	{
	  /* Set tone control according to bitmap value. */
	  new_dot =
		font_data->glyph_data[glx].bitmap[bmy] & (1 << (31-bmx));

	  /* Reverse video if activated */
	  if( rc_data.rev_video )
		new_dot  = !new_dot;

	  /* If two consecutive 1's raise tone, else drop */
	  if( !last_dot && new_dot )
		tone_ctrl = RAISE_TONE;
	  else if( last_dot && !new_dot )
		tone_ctrl = LOWER_TONE;
	  else if( last_dot && new_dot )
		tone_ctrl = KEEP_TONE;
	  else
		tone_ctrl = SILENT_TONE;
	  last_dot = new_dot;

	  /* Raise/Lower audio tone as requested */
	  switch( tone_ctrl )
	  {
		case SILENT_TONE: /*** No tone (carrier down) ***/

		  /* Fill buffer with zero level samples */
		  for( idx = 0; idx < rc_data.samples_per_dot; idx++ )
		  {
			xmit_buffer[buff_idx] = 0;
			buff_idx += rc_data.num_chn;
		  }
		  break;

		case RAISE_TONE: /*** Raise tone (carrier up) ***/

		  /* Fill buffer with cosine shaped edge for half dot */
		  for( cosine_idx = 0; cosine_idx < half_dot; cosine_idx++ )
		  {
			xmit_buffer[buff_idx] =
			  (cosine_table[cosine_idx] * sine_table[sine_idx]) >> 15;
			buff_idx += rc_data.num_chn;
			if( ++sine_idx >= rc_data.samples_per_cycle )
			  sine_idx = 0;
		  }

		  /* Fill buffer with steady tone for half dot */
		  for( idx = 0; idx < half_dot; idx++ )
		  {
			xmit_buffer[buff_idx] = sine_table[sine_idx];
			buff_idx += rc_data.num_chn;
			if( ++sine_idx >= rc_data.samples_per_cycle )
			  sine_idx = 0;
		  }
		  break;

		case LOWER_TONE: /*** Lower tone (Carrier down) ***/

		  /* Fill buffer with cosine falling edge for half dot */
		  for( cosine_idx = half_dot-1; cosine_idx >= 0; cosine_idx-- )
		  {
			xmit_buffer[buff_idx] =
			  (cosine_table[cosine_idx] * sine_table[sine_idx]) >> 15;
			buff_idx += rc_data.num_chn;
			if( ++sine_idx >= rc_data.samples_per_cycle )
			  sine_idx = 0;
		  }

		  /* Fill buffer with zero level samples for half dot */
		  for( idx = 0; idx < half_dot; idx++ )
		  {
			xmit_buffer[buff_idx] = 0;
			buff_idx += rc_data.num_chn;
		  }
		  break;

		case KEEP_TONE: /*** Keep tone raised (carrier up) ***/

		  /* Fill buffer with steady tone */
		  for( idx = 0; idx < rc_data.samples_per_dot; idx++ )
		  {
			xmit_buffer[buff_idx] = sine_table[sine_idx];
			buff_idx += rc_data.num_chn;
			if( ++sine_idx >= rc_data.samples_per_cycle )
			  sine_idx = 0;
		  }

	  } /* switch( tone_ctrl ) */

	} /* for( bmy = font_data->fbb_h-1; bmy >= 0; bmy-- ) */

	/*** Write samples buffer to DSP ***/
	if( !DSP_Write(xmit_buffer, rc_data.txbuff_len) )
	  return( FALSE );

	/* Loopback Tx to Rx */
	if( isFlagSet(TX2RX_LOOPBACK) )
	{
	  /* Reset tx buffer idx */
	  tx_buff_idx = rc_data.use_chn;

	  /* Draw characters as they are xmitted */
	  if( !Assemble_Column(column) ||
		  !Draw_Column(column) )
		return( FALSE );

	} /* if( isFlagSet(TX2RX_LOOPBACK) ) */

  } /* for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ ) */

  return( TRUE );
} /* Tx_Feld_Character() */

/*------------------------------------------------------------------------*/

/* Tx_FMHell_Character()
 *
 * Transmits one FM Hell character pointed by "glx"
 */

  static gboolean
Tx_FMHell_Character( font_data_t *font_data, int glx )
{
  int
	buff_idx, /* Index to audio samples buffer */
	bmx,	  /* Bitmap x index */
	bmy,	  /* Bitmap y index */
	idx;

  unsigned int dot; /* State of dot to transmit  */

  static double
	twopi = 2.0 * M_PI,
	tone_phase,		/* Phase of audio tone */
	white_phinc,	/* Phase increment for white tone */
	black_phinc,	/* Phase increment for black tone */
	baud_rate = 0.0;

  /* Column buffer for drawing chars */
  static unsigned char *column = NULL;
  static int bm_height = 0;

  /* Allocate memory to column. bitmap_height is in pixels */
  if( bm_height != rc_data.bitmap_height )
  {
	bm_height = rc_data.bitmap_height;
	if( !mem_realloc((void **)&column,
		  sizeof(unsigned char) * (size_t)bm_height) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
  }

  /* Initialize on new parameters or first call */
  if( (baud_rate < rc_data.baud_rate) ||
	  (baud_rate > rc_data.baud_rate) )
  {
	baud_rate = rc_data.baud_rate;

	white_phinc = twopi / (double)rc_data.dsp_rate;
	black_phinc = white_phinc;

	white_phinc *= ( (double)rc_data.tone_freq + baud_rate );
	black_phinc *= ( (double)rc_data.tone_freq - baud_rate );
	tone_phase = 0.0;
  }

  /* Adjust glyph index */
  glx -= font_data->first_glyph;

  /* Transmit columns */
  for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ )
  {
	/* Initialize buffer idx according to stereo/mono mode */
	buff_idx = rc_data.use_chn;

	/* Go up in bitmap buffer row by row */
	for( bmy = font_data->fbb_h - 1; bmy >= 0; bmy-- )
	{
	  /* Set tone control according to bitmap (dot) value */
	  dot = font_data->glyph_data[glx].bitmap[bmy] & ( 1 << (31-bmx) );

	  /* Reverse video if selected */
	  if( rc_data.rev_video ) dot = !dot;

	  /* Black dot, send black tone */
	  if( dot )
	  {
		/* Fill buffer with samples at black tone rate */
		for( idx = 0; idx < rc_data.samples_per_dot; idx++ )
		{
		  xmit_buffer[buff_idx] = (short)( 32767.0 * sin(tone_phase) );
		  buff_idx   += rc_data.num_chn;
		  tone_phase += black_phinc;
		  if( tone_phase >= twopi ) tone_phase -= twopi;
		}
	  } /* if( dot ) */
	  else
	  {
		/* Blank dot, fill buffer with samples at white tone rate */
		for( idx = 0; idx < rc_data.samples_per_dot; idx++ )
		{
		  xmit_buffer[buff_idx] = (short)( 32767.0 * sin(tone_phase) );
		  buff_idx   += rc_data.num_chn;
		  tone_phase += white_phinc;
		  if( tone_phase >= twopi ) tone_phase -= twopi;
		}
	  }
	} /* for( bmy = temp; bmy >= 0; bmy-- ) */

	/*** Write samples buffer to DSP ***/
	if( !DSP_Write(xmit_buffer, rc_data.txbuff_len) )
	  return( FALSE );

	/* Loopback Tx to Rx */
	if( isFlagSet(TX2RX_LOOPBACK) )
	{
	  /* Reset tx buffer idx */
	  tx_buff_idx = rc_data.use_chn;

	  /* Draw characters as they are xmitted */
	  if( !Assemble_Column(column) ||
		  !Draw_Column(column) )
		return( FALSE );
	} /* if( isFlagSet(TX2RX_LOOPBACK) ) */

  } /* for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ ) */

  return( TRUE );
} /* Tx_FMHell_Character() */

/*------------------------------------------------------------------------*/

/* Transmit_Character()
 *
 * Transmits one Hell character pointed by "glx"
 */

  gboolean
Transmit_Character( font_data_t *font_data, int glx )
{
  gboolean ret;

  if( isFlagSet(MODE_FELDHELL) )
	ret = Tx_Feld_Character(font_data, glx);
  else
	ret = Tx_FMHell_Character(font_data, glx);

  if( !ret ) Close_Playback();

  return( ret );
} /* Transmit_Character() */

/*------------------------------------------------------------------------*/

/* Morse_Transmit()
 *
 * Transmits a string Morse code
 */

  gboolean
Morse_Transmit( char *mesg )
{
  /* DSP samples buffer.
   * Size is (samples/cycle)*(audio freq)*(dash duration)
   * Dash duration is abt 3/wpm seconds.
   * In stereo mode the buffer size must be doubled
   */
  static short *samples_buff = NULL;
  static int buf_size = 0;

  int
	mesg_len,   /* String length of message to send  */
	mesg_idx,   /* Index into the message string     */
	morse_chr,  /* Morse code character in hex equiv */
	morse_idx,  /* Index into the Morse Hex code     */
	elem_dur,   /* Duration in samples of Morse elem */
	dash_dur,   /* Duration in samples of Morse dash */
	tone_dur,   /* Duration in samples of trans tone */
	buff_idx,   /* Index to audio samples buffer     */
	buff_size,  /* Buffer size to be written to DSP  */
	cosine_idx, /* Index to cosine wave lookup table */
	sine_idx,   /* Index to sine wave lookup table   */
	per_dot2,	/* 2 * samples_per_dot */
	idx;

  /* Tables of Morse and ascii characters */
  int
	ascii_char[] = ASCII_CHAR,
	morse_code[] = MORSE_CODE;


  /* Allocate memory to samples buffer */
  int tmp = 3 * rc_data.num_chn * rc_data.dsp_rate / rc_data.wpm;
  if( !buf_size || (buf_size != tmp) )
  {
	buf_size = tmp;
	if( !mem_realloc((void **)&samples_buff,
		  sizeof(short) * (size_t)buf_size) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
  }

  /* Duration in samples of Morse element */
  elem_dur = rc_data.dsp_rate / rc_data.wpm;
  dash_dur = 3 * elem_dur;
  buff_size = dash_dur * rc_data.num_chn;
  per_dot2 = 2 * rc_data.samples_per_dot;

  /* Clean up buffer */
  memset( samples_buff, 0, (size_t)buff_size * sizeof(short) );

  /* Convert ascii characters to Morse hex code */
  mesg_len = (int)strlen( mesg );
  for( mesg_idx = 0; mesg_idx < mesg_len; mesg_idx++ )
  {
	/* Find equivalent Hex morse code */
	for( idx = 0; idx < NUM_MORSE_CHARS; idx++ )
	  if( mesg[mesg_idx] == ascii_char[idx] )
		break;
	if( idx < NUM_MORSE_CHARS )
	  morse_chr = morse_code[idx];
	else morse_chr = 1;

	/* Find beginning of Morse code (1st '1') */
	for( idx = 8; idx >= 0; idx-- )
	  if( morse_chr & (1 << idx) )
		break;

	morse_idx = idx - 1;
	/* Transmit dots (1's) or dashes (0's) */
	while( morse_idx >= 0 )
	{
	  /* If element is dot ('1') tone duration in samples - edges */
	  if( morse_chr & (1 << morse_idx) )
		tone_dur = elem_dur - per_dot2;
	  else /* If element is dash ('0') tone duration in samples is triple */
		tone_dur = dash_dur - per_dot2;

	  /* Total samples to be generated */
	  buff_size = tone_dur + per_dot2;

	  /* Initialize buffer idx according to stereo/mono mode */
	  buff_idx = rc_data.use_chn;

	  /* Fill buffer with cosine shaped rising edge */
	  sine_idx = 0;
	  for( cosine_idx = 0; cosine_idx < rc_data.samples_per_dot; cosine_idx++ )
	  {
		samples_buff[buff_idx] =
		  (cosine_table[cosine_idx] * sine_table[sine_idx]) >> 15;
		buff_idx += rc_data.num_chn;
		if( ++sine_idx == rc_data.samples_per_cycle )
		  sine_idx = 0;
	  }

	  /* Fill buffer with the steady tone samples */
	  for( idx = 0; idx < tone_dur; idx++ )
	  {
		samples_buff[buff_idx] = sine_table[sine_idx];
		buff_idx += rc_data.num_chn;
		if( ++sine_idx == rc_data.samples_per_cycle )
		  sine_idx = 0;
	  }

	  /* Fill buffer with cosine shaped falling edge */
	  for( cosine_idx = rc_data.samples_per_dot-1; cosine_idx >= 0; cosine_idx-- )
	  {
		samples_buff[buff_idx] =
		  (cosine_table[cosine_idx] * sine_table[sine_idx]) >> 15;
		buff_idx += rc_data.num_chn;
		if( ++sine_idx == rc_data.samples_per_cycle )
		  sine_idx = 0;
	  }

	  /*** Write samples buffer to DSP, abort on error ***/
	  if( !DSP_Write(samples_buff, buff_size) )
		return( FALSE );

	  /* Initialize buffer according to stereo/mono mode */
	  if( rc_data.num_chn == 2 )
		buff_idx = rc_data.use_chn;
	  else buff_idx = 0;

	  /* Fill buffer with 0 level for space */
	  /* If inter-word space add 3 extra space elements */
	  if( morse_chr == 0x01 )
	  {
		for( idx = 0; idx < dash_dur; idx++ )
		{
		  samples_buff[buff_idx] = 0;
		  buff_idx += rc_data.num_chn;
		}
		buff_size = dash_dur;

		/*** Writebuffer to DSP, abort on error ***/
		if( !DSP_Write(samples_buff, buff_size) )
		  return( FALSE );

	  }/*  if( morse_chr == 0x01 ) */

	  /* Add inter-element space */
	  for( idx = 0; idx < elem_dur; idx++ )
	  {
		samples_buff[buff_idx] = 0;
		buff_idx += rc_data.num_chn;
	  }
	  buff_size = elem_dur;

	  /*** Write buffer to DSP, abort on error ***/
	  if( !DSP_Write(samples_buff, buff_size) )
		return( FALSE );

	  morse_idx--;

	} /* while( morse_idx >= 0 ) */

	/* Send inter-character space */
	/* Initialize buffer according to stereo/mono mode */
	if( rc_data.num_chn == 2 )
	  buff_idx = rc_data.use_chn;
	else buff_idx = 0;

	for( idx = 0; idx < 2 * elem_dur; idx++ )
	{
	  samples_buff[buff_idx] = 0;
	  buff_idx += rc_data.num_chn;
	}

	buff_size = 2 * elem_dur;

	/*** Write samples buffer to DSP ***/
	if( !DSP_Write(samples_buff, buff_size) )
	  return( FALSE );

  } /* for( mesg_idx = 0; mesg_idx < mesg_len; mesg_idx++ ) */

  return( TRUE );
} /* Morse_Transmit() */

/*------------------------------------------------------------------------*/

/* Make_Sin_Wavetable()
 *
 * Makes a sine look-up wavetable 0-2pi
 */

  gboolean
Make_Sin_Wavetable()
{
  double sine, angle;
  int idx;

  /* Allocate buffer */
  if( !mem_alloc((void **)&sine_table,
		sizeof(short) * (size_t)rc_data.samples_per_cycle) )
	return( FALSE );

  angle = 2.0 * M_PI / (double)rc_data.samples_per_cycle;
  for( idx = 0; idx < rc_data.samples_per_cycle; idx++ )
  {
	sine = sin( angle * (double)idx );
	sine_table[idx] = (short)(sine * 32767.0);
  }

  return( TRUE );
} /* Make_Sin_Wavetable() */

/*------------------------------------------------------------------------*/

/* Make_Cos_Wavetable()
 *
 * Makes a cosine look-up wavetable 0-pi/2
 */

  gboolean
Make_Cos_Wavetable()
{
  double angle, cosine;
  int half_dot, idx;

  /* Allocate buffer */
  half_dot = rc_data.samples_per_dot / 2;
  if( !mem_realloc( (void **)&cosine_table,
	  sizeof(short) * (size_t)half_dot ) )
	return( FALSE );

  angle = M_PI_2 / (double)half_dot;
  for( idx = 0; idx < half_dot; idx++ )
  {
	/* Make a raised cos shape */
	cosine = sin( angle * (double)idx );
	cosine_table[idx] = (short)( cosine * 32767.0 );
  }

  return( TRUE );
} /* Make_Cos_Wavetable() */

/*------------------------------------------------------------------------*/

