/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "shared.h"

/*----------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile, char *mesg )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];

  /* Prepare error message */
  snprintf( error_mesg, sizeof(error_mesg),
	  _("Error reading %s\n"\
		"Premature EOF (End Of File)"), mesg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return if chr = EOF */
  chr = fgetc( pfile );
  if( chr == EOF ) return( EOF );

  /* Ignore commented lines, lines starting
   * with white spaces and newline/carriage ret */
  while(
	  (chr == '#') ||
	  (chr == ' ') ||
	  (chr == HT ) ||
	  (chr == CR ) ||
	  (chr == LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) && (chr != LF) )
	{
	  /* Get next character, return error if chr = EOF */
	  chr = fgetc( pfile );
	  if( chr == EOF )
	  {
		fprintf( stderr, "xwefax: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg, HIDE_OK );
		return( ERROR );
	  }
	}

	/* Dump any CR/LF remaining */
	while( (chr == CR) || (chr == LF) )
	{
	  /* Get next character, return error if chr = EOF */
	  chr = fgetc( pfile );
	  if( chr == EOF )
	  {
		fprintf( stderr, "xwefax: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg, HIDE_OK );
		return( ERROR );
	  }
	}

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = 80 or EOF or CR/LF */
  while( num_chr < LINE_BUF_LEN )
  {
	/* If LF/CR reached before filling buffer, return line */
	if( (chr == LF) || (chr == CR) )
	{
	  /* Get next character */
	  chr = fgetc( pfile );
	  if( chr == EOF )
	  {
		/* Terminate buffer as a string if chr = EOF */
		buff[num_chr] = '\0';
		return( EOF );
	  }

	  /* Restore char in not EOF */
	  ungetc( chr, pfile );
	  break;
	}

	/* Enter new character to line buffer */
	buff[num_chr++] = (char)chr;

	/* Get next character */
	chr = fgetc( pfile );
	if( chr == EOF )
	{
	  /* Terminate buffer as a string if chr = EOF */
	  buff[num_chr] = '\0';
	  return( EOF );
	}

	/* Abort if end of line not reached at 80 char. */
	if( (num_chr == LINE_BUF_LEN) && (chr != LF) && (chr != CR) )
	{
	  /* Terminate buffer as a string */
	  buff[num_chr] = '\0';
	  snprintf( error_mesg, sizeof(error_mesg),
		  _("Error reading %s\n"\
			"Line longer than %d characters"), mesg, LINE_BUF_LEN );
	  fprintf( stderr, "xwefax: %s\n%s\n", error_mesg, buff );
		Error_Dialog( error_mesg, HIDE_OK );
	  fclose( pfile );
	  return( ERROR );
	}

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( SUCCESS );
} /* End of Load_Line() */

/*----------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  static void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;

  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
	dest[idx] = src[idx];
	idx++;
	ch = src[idx];
	n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*----------------------------------------------------------------------*/

/*  Load_Config()
 *
 *  Loads the sdrxrc configuration file
 */

  gboolean
Load_Config( gpointer data )
{
  char
	rc_fpath[64], /* File path to sdrxrc */
	line[81];     /* Buffer for Load_Line  */

  /* Config file pointer */
  FILE *sdrxrc;

  int mode_idx;


  /* Setup file path to sdrxrc in working dir */
  snprintf( rc_fpath, sizeof(rc_fpath),
	  "%s/.sdrx/sdrxrc", getenv("HOME") );

  /* Open sdrxrc file */
  if( !Open_File( &sdrxrc, rc_fpath, "r") )
  {
	Error_Dialog( _("Failed to open sdrxrc file"), HIDE_OK );
	return( FALSE );
  }

  /*** Read runtime configuration data ***/

  /* Read SDR Dongle Type, abort if EOF */
  if( Load_Line(line, sdrxrc, _("SDR Dongle Type")) != SUCCESS )
	return( FALSE );
  if( strcmp(line, "RTLSDR") == 0 )
	rc_data.dongle_type = DONGLE_TYPE_RTLSDR;
  else if( strcmp(line, "SDRPLAY") == 0 )
	rc_data.dongle_type = DONGLE_TYPE_SDRPLAY;
  else if( strcmp(line, "PERSEUS") == 0 )
	rc_data.dongle_type = DONGLE_TYPE_PERSEUS;
  else
  {
	Error_Dialog(
		_("Invalid Dongle Type\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /* Hide Perseus related frame */
	if( rc_data.dongle_type != DONGLE_TYPE_PERSEUS )
	  gtk_widget_hide( lookup_widget(sdrx_main_window, "perseus_frame") );

  /* Read Tuner Center Frequency, abort if EOF */
  if( Load_Line(line, sdrxrc,
		_("Tuner Center Frequency")) != SUCCESS )
	return( FALSE );
  int center_freq = atoi( line );
  if( center_freq <= 0 )
  {
	Error_Dialog(
		_("Invalid Tuner Center Frequency\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }
  rc_data.sdrx_center_freq = (uint32_t)center_freq;

  /* Read Carrier Modulation Mode, abort if EOF */
  if( Load_Line(line, sdrxrc,
		_("Carrier Modulation Mode")) != SUCCESS )
	return( FALSE );
  mode_idx = atoi( line );
  if( (mode_idx < 0) || (mode_idx >= MODULATION_MODES_NUM) )
  {
	Error_Dialog(
		_("Invalid Carrier Modulation Mode\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /* Read librtlsdr Device Index, abort if EOF */
  if( Load_Line(line, sdrxrc, _("RtlSdr Device Index")) != SUCCESS )
	return( FALSE );
  rc_data.rtlsdr_device_index = atoi( line );
  if( rc_data.rtlsdr_device_index > 8 )
  {
	Error_Dialog(
		_("Invalid librtlsdr Device Index\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /* Read RTLSDR Frequency Correction Factor, abort if EOF */
  if( Load_Line(line, sdrxrc,
		_("RTLSDR Frequency Correction Factor")) != SUCCESS )
	return( FALSE );
  rc_data.rtlsdr_freq_correction = atoi( line );
  if( abs(rc_data.rtlsdr_freq_correction) > 100 )
  {
	Error_Dialog(
		_("Invalid RTLSDR Frequency Correction\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /* Read SDRPlay Frequency Correction Factor, abort if EOF */
  if( Load_Line(line, sdrxrc,
		_("SDRPlay Frequency Correction Factor")) != SUCCESS )
	return( FALSE );
  rc_data.rsp1_freq_correction = atof( line );
  if( fabs(rc_data.rsp1_freq_correction) > 100.0 )
  {
	Error_Dialog(
		_("Invalid SDRPlay Frequency Correction\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /* Read Perseus Device Index, abort if EOF */
  if( Load_Line(line, sdrxrc, _("Perseus Device Index")) != SUCCESS )
	return( FALSE );
  rc_data.perseus_device_index = atoi( line );
  if( (rc_data.perseus_device_index > 8) ||
	  (rc_data.perseus_device_index < 0) )
  {
	Error_Dialog(
		_("Invalid Perseus Device Index\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /* Read Perseus Frequency Correction Factor, abort if EOF */
  if( Load_Line(line, sdrxrc,
		_("Perseus Frequency Correction Factor")) != SUCCESS )
	return( FALSE );
  rc_data.perseus_freq_correction = atof( line );
  if( fabs(rc_data.perseus_freq_correction) > 100.0 )
  {
	Error_Dialog(
		_("Invalid Perseus Frequency Correction\n"\
		  "Quit and correct .sdrxrc"), HIDE_OK );
	return( FALSE );
  }

  /*** Read Sound Card configuration data ***/
  /* Read card name, abort if EOF */
  if( Load_Line(line, sdrxrc, _("Sound Card Name")) != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.sound_card, line, sizeof(rc_data.sound_card) );

  /* Read pcm device name, abort if EOF */
  if( Load_Line(line, sdrxrc, _("PCM Device Name")) != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.pcm_device, line, sizeof(rc_data.pcm_device) );

  /* Set text entries to combos */
  Combobox_Append_Text();
  SetFlag( RCCONFIG_SETUP );
  Mode_Combobox_Changed( mode_idx );

  /* Display center frequency */
  Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );

  /* Resize top window (shrink) */
  gtk_window_resize( GTK_WINDOW(sdrx_main_window), 10, 10 );

  /* Default settings */
  rc_data.adagc_scale = 1.0;

  /* Set audio derived agc for Perseus */
  if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS )
  {
	GtkWidget *btn = lookup_widget(
		sdrx_main_window, "audio_agc_radiobtn" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(btn), TRUE );
  }

  /* ADAGC decay setting */
  GtkWidget *temp = lookup_widget( sdrx_main_window, "agc_hscale" );
  double val = gtk_range_get_value( GTK_RANGE(temp) );
  rc_data.adagc_decay = 1.0 - val / ADAGC_DIVISOR;

  /* Squelch setting */
  temp = lookup_widget( sdrx_main_window, "squelch_hscale" );
  val  = gtk_range_get_value( GTK_RANGE(temp) );
  rc_data.squelch_value  = SQUELCH_THRESHOLD * SQUELCH_RANGE;
  rc_data.squelch_value -= SQUELCH_THRESHOLD * val;

  /* Init semaphore */
  sem_init( &pback_semaphore,  0, 0 );

  /* Enable Rx front end presector */
  SetFlag( PRESELECT_ENABLE );
  SetFlag( PERSEUS_ATTEN_AUTO );

  fclose( sdrxrc );

  return( FALSE );
} /* End of Load_Config() */

/*----------------------------------------------------------------------*/

/***  Memory allocation/freeing utils ***/
  void
mem_alloc( void **ptr, size_t req )
{
  free_ptr( ptr );
  *ptr = malloc( req );
  if( *ptr == NULL )
  {
	perror( _("sdrx: A memory allocation request failed") );
	exit( -1 );
  }

  memset( *ptr, 0, req );
} /* End of void mem_alloc() */

/*----------------------------------------------------------------------*/

  void
mem_realloc( void **ptr, size_t req )
{
  *ptr = realloc( *ptr, req );
  if( *ptr == NULL )
  {
	perror( _("sdrx: A memory re-allocation request failed") );
	exit( -1 );
  }

  memset( *ptr, 0, req );
} /* End of void mem_realloc() */

/*----------------------------------------------------------------------*/

  void
free_ptr( void **ptr )
{
  if( *ptr != NULL ) free( *ptr );
  *ptr = NULL;

} /* End of void free_ptr() */

/*----------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flags */

/* An int variable holding the single-bit flags */
int Flags = 0;

  inline int
isFlagSet( int flag )
{
  return( Flags & flag );
}

  inline int
isFlagClear( int flag )
{
  return( ~Flags & flag );
}

  void
SetFlag( int flag )
{
  Flags |= flag;
}

  void
ClearFlag( int flag )
{
  Flags &= ~flag;
}

  void
ToggleFlag( int flag )
{
  Flags ^= flag;
}

/*----------------------------------------------------------------------*/

/* Sdrx_New_Buffer()
 *
 * Makes changes needed when Sdrx samples buffer changes
 */
  void
Sdrx_New_Buffer( GtkComboBox *combobox )
{
  /* Get Buffer combobox active entry */
  if( combobox != NULL )
  {
	SetFlag( PLAYBACK_SUSPEND );
	int idx = gtk_combo_box_get_active( combobox );
	char *buf[] = { SDRX_BUF_SIZE };
	rc_data.sdrx_buffer_len = atoi( buf[idx] ) * 1024;
  }
  else return;

  /* Reset sound buffer indices */
  demod_buf_num = 0;
  sound_buf_num = 0;

  /* If playback buffers already set up */
  if( rc_data.playback_buffs && playback_buf )
  {
	/* Free existing playback buffers */
	for( int idx = 0; idx < rc_data.playback_buffs; idx++ )
	  free_ptr( (void **)&playback_buf[idx] );
  }

  /* New number of playback ring buffers */
  rc_data.playback_buffs = rc_data.sdrx_buffer_len / PLAYBACK_BUF_LEN;

  /* Allocate memory to Playback ring buffers */
  mem_realloc( (void **)&playback_buf, sizeof(short *) * rc_data.playback_buffs );
  size_t buf_size  = SND_NUM_CHANNELS * PLAYBACK_BUF_LEN * sizeof(short);
  for( int idx = 0; idx < rc_data.playback_buffs; idx++ )
  {
	/* Allocate playback buffer */
	playback_buf[idx] = NULL;
	mem_alloc( (void **)&playback_buf[idx], buf_size );
  }

  /* Allocate Sdrx I/O buffer */
  size_t mreq = (size_t)rc_data.sdrx_buffer_len * sizeof(double);
  mem_realloc( (void *)&sdrx_buf_i, mreq );
  mem_realloc( (void *)&sdrx_buf_q, mreq );

  /* Initialize filter data */
  fft_filter_data_i.samples_buf     = sdrx_buf_i;
  fft_filter_data_i.samples_buf_len = rc_data.sdrx_buffer_len;
  fft_filter_data_q.samples_buf     = sdrx_buf_q;
  fft_filter_data_q.samples_buf_len = rc_data.sdrx_buffer_len;

  SetFlag( SDRX_NEW_BUFFER );
  ClearFlag( PLAYBACK_SUSPEND );

} /* Sdrx_New_Buffer() */

/*----------------------------------------------------------------------*/

/* Set_FFT_Bandwidth_Combobox()
 *
 * Appends entries of available FFT bandwidths to the
 * FFT Bandwidth Combo Box according to sample rate
 */
  void
Set_FFT_Bandwidth_Combobox( uint32_t sample_rate )
{
  double rate_ref  = (double)sample_rate / 1000.0;
  double rate      = rate_ref;
  double fft_decim = (double)rc_data.sdrx_buffer_len / (double)FFT_INPUT_LENGTH;
  gchar txt[16];


  SetFlag( OPER_PENDING );

  /* Clear combo box tree model */
  GtkTreeModel *model =
	gtk_combo_box_get_model( GTK_COMBO_BOX(fft_bw_combobox) );
  gtk_list_store_clear( GTK_LIST_STORE(model) );

  /* Enter a sequence of FFT B/W starting from the ADC sample
   * rate down to a minimum of rate/FFT_MIN_BANDWIDTH kHz */
  while( rate_ref / rate <= fft_decim )
  {
	if( rate < FFT_MIN_BANDWIDTH ) break;
	snprintf( txt, sizeof(txt), "%0.1f", rate );
	gtk_combo_box_prepend_text( GTK_COMBO_BOX(fft_bw_combobox), txt );
	rate /= 2.0;
  }

  ClearFlag( OPER_PENDING );

  /* Make the FFT bandwidth at least equal to demod bandwidth */
  int idx = 0;
  double bndw = (double)rc_data.demod_bandwidth;
  gtk_combo_box_set_active( GTK_COMBO_BOX(fft_bw_combobox), idx );
  while( bndw > rc_data.fft_bandwidth )
  {
	static double bw = 0.0;

	/* Gone past last combobox entry but still bndw > rc_data.fft_bandwidth */
	gtk_combo_box_set_active( GTK_COMBO_BOX(fft_bw_combobox), ++idx );
	if( bw == rc_data.fft_bandwidth )
	{
	  /* Set the last (largest) FFT bandwidth available */
	  gtk_combo_box_set_active( GTK_COMBO_BOX(fft_bw_combobox), --idx );
	  break;
	}
	bw = rc_data.fft_bandwidth;
  } /* while( bndw > rc_data.fft_bandwidth ) */

} /* Set_FFT_Bandwidth_Combobox() */

/*----------------------------------------------------------------------*/

/* Open_File()
 *
 * Opens a file, aborts on error
 */
  gboolean
Open_File( FILE **fp, char *fname, const char *mode )
{
  /* Open Channel A image file */
  *fp = fopen( fname, mode );
  if( *fp == NULL )
  {
	perror( fname );
	return( FALSE );
  }

  return( TRUE );
} /* End of Open_File() */

/*----------------------------------------------------------------------*/

