/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "stations.h"

/*----------------------------------------------------------------------*/

/* cell_edited_callback()
 *
 * Text cell edited callback
 */
  static void
cell_edited_callback(
	GtkCellRendererText *cell,
	gchar				*path,
	gchar               *new_text,
	gpointer             user_data )
{
  /* Stations treeview selection objects */
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  guint column;

  column = GPOINTER_TO_UINT(
	  g_object_get_data(G_OBJECT(cell), "column") );
  selection = gtk_tree_view_get_selection(
	  GTK_TREE_VIEW(user_data) );
  gtk_tree_selection_get_selected(
	  selection, &model, &iter );
  gtk_list_store_set( GTK_LIST_STORE(model),
	  &iter, column, new_text, -1 );

  SetFlag( SAVE_STATIONS );

} /* cell_edited_callback() */

/*----------------------------------------------------------------------*/

/* Helper function
  static gboolean
gtk_tree_model_iter_previous(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
  GtkTreePath *path;
  gboolean ret;

  path = gtk_tree_model_get_path (tree_model, iter);
  ret = gtk_tree_path_prev (path);
  if (ret == TRUE)
	gtk_tree_model_get_iter (tree_model, iter, path);
  gtk_tree_path_free (path);
  return ret;
} */

/*----------------------------------------------------------------------*/

/* Insert_Columns()
 *
 * Inserts columns in a list store
 */
  static void
Insert_Columns(	GtkListStore* store, int ncols, char *colname[] )
{
  int idx;
  GtkTreeModel *model;
  GtkCellRenderer *renderer;

  /* Create renderer and insert columns to treeview */
  for( idx = 0; idx < ncols; idx++ )
  {
	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "editable", TRUE, NULL);
	g_signal_connect( renderer, "edited",
		(GCallback)cell_edited_callback, stations_treeview );
	g_object_set_data( G_OBJECT(renderer),
		"column", GUINT_TO_POINTER(idx) );
	gtk_tree_view_insert_column_with_attributes(
		stations_treeview, -1, colname[idx],
		renderer, "text", idx, NULL );
  }

  /* Create a model to insert list store */
  model = GTK_TREE_MODEL( store );
  gtk_tree_view_set_model( stations_treeview, model );

  /* Destroy model automatically with view */
  g_object_unref( model );

} /* Insert_Columns() */

/*----------------------------------------------------------------------*/

/* Create_List_Store()
 *
 * Create stores needed for the treeview
 */
  static void
Create_List_Store( void )
{
  /* Station list column names */
  char *list_col_name[LIST_NUM_COLS] =
  {
	_("STATION NAME"),
	_(" FREQUENCY "),
	_(" MODE     ")  };

  /* Create list stores only if needed */
  if( stations_list_store != NULL ) return;

  /* Create stations list store */
  stations_list_store = gtk_list_store_new(
	  LIST_NUM_COLS,
	  G_TYPE_STRING,
	  G_TYPE_STRING,
	  G_TYPE_STRING );

  /* Insert station list columns */
  Insert_Columns( stations_list_store, LIST_NUM_COLS, list_col_name );

} /* Create_List_Store() */

/*----------------------------------------------------------------------*/

/* List_Stations()
 *
 * Reads stations data from file and lists in tree view
 */
  void
List_Stations( void )
{
  gboolean ret;			 /* Return status variable */
  FILE *fp;				 /* Stations file pointer */
  int eof;				 /* End of File flag */
  GtkTreeIter iter;		 /* Treeview list iteration */
  GtkTreeSelection *sel; /* Selected treeview row */
  GtkTreePath *path;	 /* Path to ead data from treeview */
  int start, end;		 /* Start and end character of strings in treeview */
  char line_buf[LINE_BUF_LEN + 1]; /* Buffers one line of station data */
  char stations_fpath[64];

  /* Create treeview pointer */
  stations_treeview = GTK_TREE_VIEW(
	  lookup_widget(stations_window, "stations_treeview") );

  /* Create the list store */
  Create_List_Store();

  /* Setup file path to sdrxrc in working dir */
  snprintf( stations_fpath, sizeof(stations_fpath),
	  "%s/.sdrx/stations", getenv("HOME") );

  /* Open stations file */
  if( !Open_File(&fp, stations_fpath, "r") )
  {
	Error_Dialog( "Failed to open Stations File in ~/sdrx", SHOW_OK );
	return;
  }

  /* Clear all tree view */
  gtk_list_store_clear( stations_list_store );

  /* Get new row if available */
  ret = gtk_tree_model_get_iter_first(
	  GTK_TREE_MODEL(stations_list_store), &iter);

  /* Fill station list */
  while( !ret )
  {
	/* Read a line and separate entries */
	eof = Load_Line( line_buf, fp, "Station File" );
	if( eof == ERROR )
	{
	  Error_Dialog( "Error reading Stations File in ~/sdrx", SHOW_OK );
	  return;
	}

	/* Append a row and fill in station data */
	gtk_list_store_append( stations_list_store, &iter );

	/* Separate entries in the line by string
	 * terminator and set data to list store */
	/* Station Name */
	start = 0;
	end = STATIONS_NAME_WIDTH;
	line_buf[ end ] = '\0';
	gtk_list_store_set( stations_list_store,
		&iter, NAME_COL, &line_buf[start], -1 );

	/* Frequency - Hz */
	end += STATIONS_FREQ_WIDTH + 1;
	line_buf[ end ] = '\0';
	start += STATIONS_NAME_WIDTH + 1;
	gtk_list_store_set( stations_list_store,
		&iter, FREQ_COL, &line_buf[start], -1 );

	/* Modulation Mode */
	end += STATIONS_MODE_WIDTH + 1;
	line_buf[ end ] = '\0';
	start += STATIONS_FREQ_WIDTH + 1;
	gtk_list_store_set( stations_list_store,
		&iter, MODE_COL, &line_buf[start], -1 );

	/* Stop at end of file */
	if( eof == EOF ) break;

	/* Get new row if available */
	ret = gtk_tree_model_iter_next(
		GTK_TREE_MODEL(stations_list_store), &iter);

  } /* while() */

  /* Select the first entry */
  sel = gtk_tree_view_get_selection( stations_treeview );
  path = gtk_tree_path_new_from_string( "0" );
  gtk_tree_selection_select_path( sel, path );
  gtk_tree_path_free( path );

  ClearFlag( SAVE_STATIONS );
  fclose( fp );

} /* List_Stations() */

/*----------------------------------------------------------------------*/

/* Save_Stations_File()
 *
 * Saves the data in stations treeview to a given filename
 */
  gboolean
Save_Stations_File( void )
{
  FILE *fp = NULL;		/* Stations file pointer */
  GtkTreeIter iter;		/* Iteration to station treeview */
  char file_line[STATIONS_LINE_LEN]; /* Buffer for one file line */
  gboolean ret = FALSE;	/* Return value of functions */
  char *string;			/* String created from Wefax parameters */
  size_t len, limit;	/* Length and its limit of above string */
  int idx, value;
  char str[STATIONS_FREQ_WIDTH + 1];
  char stations_fpath[64];


  /* Do not save if treeview is not
   * valid or save flag not set */
  if( (stations_treeview == NULL) || isFlagClear(SAVE_STATIONS) )
	return( FALSE );

  /* Stations treeview model */
  GtkTreeModel *model = GTK_TREE_MODEL(stations_list_store);

  /* Get the first iteration */
  ret = gtk_tree_model_get_iter_first( model, &iter);

  /* Setup file path to sdrxrc in working dir */
  snprintf( stations_fpath, sizeof(stations_fpath),
	  "%s/.sdrx/stations", getenv("HOME") );

  /* Open stations file for writing */
  if( ret && !Open_File(&fp, stations_fpath, "w") )
  {
	Error_Dialog( "Failed to open Stations File in ~/sdrx", SHOW_OK );
	return( FALSE );
  }

  /* Write treeview data */
  while( ret )
  {
	/* Clear file line */
	for( idx = 0; idx < STATIONS_LINE_LEN; idx++ )
	  file_line[idx] = ' ';

	/* Get the station name */
	gtk_tree_model_get( model, &iter, NAME_COL, &string, -1 );
	len = strlen( string );
	limit = STATIONS_NAME_WIDTH;
	if( len > limit ) len = limit;

	/* Remove leading spaces */
	idx = 0;
	while( (string[idx] == ' ') && (idx < (int)len) )
	{
	  idx++;
	  len--;
	}
	strncpy( file_line, &string[idx], len );
	idx = STATIONS_NAME_WIDTH + 1;
	g_free( string );

	/* Get the station frequency */
	gtk_tree_model_get( model, &iter, FREQ_COL, &string, -1 );
	value = atoi( string );

	/* Format value before writing to file */
	snprintf( str, STATIONS_FREQ_WIDTH + 1,
		"%*d", STATIONS_FREQ_WIDTH, value );
	strncpy( &file_line[idx], str, STATIONS_FREQ_WIDTH );
	g_free( string );
	idx += STATIONS_FREQ_WIDTH + 1;

	/* Get and write the station Mode */
	gtk_tree_model_get( model, &iter, MODE_COL, &string, -1 );
	snprintf( &file_line[idx], STATIONS_MODE_WIDTH + 1,
		"%*s", STATIONS_MODE_WIDTH, string );
	g_free( string );

	/* Write line to stations file */
	file_line[STATIONS_LINE_LEN - 1] = '\0';
	fprintf( fp, "%s\n", file_line );

	ret = gtk_tree_model_iter_next( model, &iter);
  } /* while( ret ) */

  ClearFlag( SAVE_STATIONS );
  fclose( fp );

  return( FALSE );
} /* Save_Stations_File() */

/*----------------------------------------------------------------------*/

/* Select_Treeview_Row()
 *
 * Selects a stations treeview row on user click
 * on up or down buttons in stations window
 */
  void
Select_Treeview_Row( int direction )
{
  static int row_num = 0;
  char path_str[3];				/* String to create a path from */
  GtkTreeSelection *selection;	/* Selected row in treeview */
  GtkTreePath *path;			/* A path into the selected row */

  /* Increment or decrement row number
   * and clamp between 0 and 99 */
  if( direction == SELECT_ROW_DOWN )
  {
	row_num ++;
	if( row_num > 99 ) row_num = 99;
  }
  else if( direction == SELECT_ROW_UP )
  {
	row_num--;
	if( row_num < 0 ) row_num = 0;
  }

  /* Create a selection object */
  selection = gtk_tree_view_get_selection( stations_treeview );

  /* Create a string for new path and select row */
  snprintf( path_str, sizeof(path_str), "%d", row_num );
  path = gtk_tree_path_new_from_string( path_str );
  gtk_tree_selection_select_path( selection, path );
  gtk_tree_path_free( path );

  /* Simulate a button press on treeview */
  Stations_Cursor_Changed( stations_treeview );

} /* Select_Treeview_Row() */

/*----------------------------------------------------------------------*/

/* Stations_Cursor_Changed()
 *
 * Handles cursor changed event on stations treeview.
 * Reads the selected row and sets up receiver center
 * frequency and modulation mode if they are changed
 */
  void
Stations_Cursor_Changed( GtkTreeView *treeview )
{
  /* Stations treeview selection objects */
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  gchar *name;	/* Station name */
  gchar *value;	/* Value from treeview strings */
  int idx, len;
  char *modulation_modes[MODULATION_MODES_NUM] = { MODULATION_MODES };
  gchar title[STATIONS_NAME_WIDTH + 16];

  uint32_t center_freq;
  static int modulation_mode = -1;


  /* Get the selected row */
  selection = gtk_tree_view_get_selection( treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter );

  /* Get the station name */
  gtk_tree_model_get( model, &iter, NAME_COL, &name, -1 );

  /* Remove trailing spaces */
  len = (int)strlen( name ) - 1;
  for( idx = len; idx >= 0; idx-- )
	if( name[idx] != ' ' ) break;
  name[ idx + 1 ] = '\0';

  /* Set the title of the main window */
  snprintf( title, sizeof(title), "%s %s", PACKAGE_STRING, name );
  gtk_window_set_title( GTK_WINDOW(sdrx_main_window), title );
  g_free( name );

  /* Get the RX frequency */
  gtk_tree_model_get( model, &iter, FREQ_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  center_freq = (uint32_t)atoi( value );
  g_free( value );

  /* Get the station Mode */
  gtk_tree_model_get( model, &iter, MODE_COL, &value, -1 );
  if( strstr(value, "--") ) return;

  /* Bypass any leading spaces */
  gchar *tmp = value;
  while( tmp[0] == ' ' ) tmp++;

  /* Match list value to modes array */
  for( idx = 0; idx < MODULATION_MODES_NUM; idx++ )
	if( strcmp(modulation_modes[idx], tmp) == 0 )
	  break;
  g_free( value );

  /* Show error message once if mode not recognized */
  if( (idx == MODULATION_MODES_NUM) && (modulation_mode != idx) )
	Error_Dialog( "Unknown Modulation Mode in Stations List", SHOW_OK );
  modulation_mode = idx;

  /* Set up new new frequency on change */
  if( rc_data.sdrx_center_freq != center_freq )
  {
	rc_data.sdrx_center_freq = center_freq;
	Display_Center_Frequency( rc_data.sdrx_center_freq );
	Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );
  }

  /* Set up new mode if valid and new */
  if( (rc_data.modulation_mode != modulation_mode) &&
	  ( modulation_mode != MODULATION_MODES_NUM) )
  {
	Mode_Combobox_Changed( modulation_mode );
  }

} /* Stations_Cursor_Changed() */

/*----------------------------------------------------------------------*/

/* Treeview_Button_Press()
 *
 * Handles the stations treeview button press callback
 */
  void
Treeview_Button_Press( void )
{
  /* Stations treeview selection objects */
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  char *modulation_modes[MODULATION_MODES_NUM] = { MODULATION_MODES };

  /* Data from stations treeview */
  gchar
	freq[STATIONS_FREQ_WIDTH+1],
	mode[STATIONS_MODE_WIDTH+1];

  /* Get the selected row */
  selection = gtk_tree_view_get_selection( stations_treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter );

  /* Create data to insert to treeview */
  snprintf( freq, sizeof(freq), "%*d",
	  STATIONS_FREQ_WIDTH, rc_data.sdrx_center_freq );

  snprintf( mode, sizeof(mode), "%*s",
	  STATIONS_MODE_WIDTH, modulation_modes[rc_data.modulation_mode] );

  /* Set data to list store */
  gtk_list_store_set(
	  stations_list_store, &iter,
	  FREQ_COL, freq,
	  MODE_COL, mode,
	  -1 );

} /* Treeview_Button_Press() */

/*----------------------------------------------------------------------*/

/* New_Station_Row()
 *
 * Adds a new row to the stations treeview
 */
  void
New_Station_Row( void )
{
  /* Stations treeview selection objects */
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter, sibling;
  int ncols, idx;

  if( stations_treeview == NULL )
	return;

  /* Find selected row and add new after */
  selection = gtk_tree_view_get_selection( stations_treeview );
  if( !gtk_tree_selection_get_selected(selection, &model, &sibling) )
  {
	/* Empty tree view case */
	model = gtk_tree_view_get_model( stations_treeview );
	gtk_list_store_insert( GTK_LIST_STORE(model), &iter, 0 );
  }
  else gtk_list_store_insert_after(
	  GTK_LIST_STORE(model), &iter, &sibling);

  /* Prime columns of new row */
  gtk_tree_selection_select_iter( selection, &iter);
  ncols = gtk_tree_model_get_n_columns( model );
  for( idx = 0; idx < ncols; idx++ )
	gtk_list_store_set(
		GTK_LIST_STORE(model), &iter, idx, "--", -1 );

} /* New_Station_Row() */

/*----------------------------------------------------------------------*/

/* Delete_Station_Row()
 *
 * Deletes a selected row in the stations treeview
 */
  void
Delete_Station_Row( void )
{
  /* Stations treeview selection objects */
  GtkTreeModel *model;
  GtkTreeIter   iter;
  GtkTreeSelection *selection;

  if( stations_treeview == NULL )
    return;

  selection = gtk_tree_view_get_selection( stations_treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter);
  gtk_list_store_remove( GTK_LIST_STORE(model), &iter );

  SetFlag( SAVE_STATIONS );

} /* Delete_Station_Row() */

/*----------------------------------------------------------------------*/

