/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sound.h"
#include "shared.h"

/* ALSA pcm capture and mixer handles */
static snd_pcm_t *handle = NULL;
static snd_pcm_hw_params_t *hw_params = NULL;
static int playback_num_writes;

/*----------------------------------------------------------------------*/

/* Open_PCM()
 *
 * Opens a pcm device for a given handle
 */
  static gboolean
Open_PCM( void )
{
  int ret;
  gchar mesg[ MESG_SIZE ];

  /* Open pcm */
  ret = snd_pcm_open( &handle, rc_data.pcm_device,
	  SND_PCM_STREAM_PLAYBACK, SND_PCM_MODE );
  if( ret < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot open sound device %s"), rc_data.pcm_device );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }

  /* Allocate memory to hardware parameters structure */
  ret = snd_pcm_hw_params_malloc( &hw_params );
  if( ret < 0 )
  {
	Error_Dialog( _("Cannot allocate hw_params struct"), HIDE_OK );
	return( FALSE );
  }

  /* Initialize hardware parameter structure */
  ret = snd_pcm_hw_params_any( handle, hw_params );
  if( ret < 0 )
  {
	Error_Dialog( _("Cannot initialize hw_params struct"), HIDE_OK );
	return( FALSE );
  }

  /* Set access type */
  ret = snd_pcm_hw_params_set_access(
	  handle, hw_params, SND_PCM_ACCESS );
  if( ret < 0 )
  {
	Error_Dialog( _("Cannot set PCM access type"), HIDE_OK );
	return( FALSE );
  }

  /* Set sample format */
  ret = snd_pcm_hw_params_set_format(
	  handle, hw_params, SND_PCM_FORMAT );
  if( ret < 0 )
  {
	Error_Dialog( _("Cannot set sample format"), HIDE_OK );
	return( FALSE );
  }

  /* Set sample rate */
  ret = snd_pcm_hw_params_set_rate(
	  handle, hw_params, SND_DSP_RATE, SND_EXACT_VALUE );
  if( ret < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set sample rate to %d"), SND_DSP_RATE );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }

  /* Set channel count */
  ret = snd_pcm_hw_params_set_channels(
	  handle, hw_params, SND_NUM_CHANNELS );
  if( ret < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set channel count to %d"), SND_NUM_CHANNELS );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }

  /* Set number of periods */
  ret = snd_pcm_hw_params_set_periods(
	  handle, hw_params, SND_NUM_PERIODS, SND_EXACT_VALUE );
  if( ret < 0)
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set number periods to %d"), SND_NUM_PERIODS );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }

  /* Set period size */
  ret = snd_pcm_hw_params_set_period_size(
	  handle, hw_params, SND_PERIOD_SIZE, SND_EXACT_VALUE );
  if( ret < 0)
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set period size to %d"), SND_PERIOD_SIZE );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }

  /* Set parameters */
  ret = snd_pcm_hw_params( handle, hw_params );
  if( ret < 0 )
  {
	snprintf( mesg, MESG_SIZE, _("Cannot set capture parameters") );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }
  snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  /* Prepare audio interface for use */
  ret = snd_pcm_prepare( handle );
  if( ret < 0 )
  {
	snprintf( mesg, MESG_SIZE, _("Cannot prepare audio interface") );
	Error_Dialog( mesg, HIDE_OK );
	return( FALSE );
  }

  return( TRUE );
} /* Open_PCM() */

/*----------------------------------------------------------------------*/

/* Open_Playback()
 *
 * Opens sound card for Playback
 */
  gboolean
Open_Playback( void )
{
  /* Open & setup pcm for Playback */
  if( !Open_PCM() )
  {
	ClearFlag( PLAYBACK_START );
	return( FALSE );
  }

  playback_num_writes = PLAYBACK_BUF_LEN / SND_PERIOD_SIZE;

  /* Allocate memory to Playback ring buffers */
  if( !playback_buf && rc_data.playback_buffs )
  {
	mem_alloc( (void **)&playback_buf, sizeof(short *) * rc_data.playback_buffs );
	size_t buf_size  = SND_NUM_CHANNELS * PLAYBACK_BUF_LEN * sizeof(short);
	for( int idx = 0; idx < rc_data.playback_buffs; idx++ )
	{
	  /* Allocate playback buffer */
	  playback_buf[idx] = NULL;
	  mem_alloc( (void **)&playback_buf[idx], buf_size );
	}
  }

  SetFlag( PLAYBACK_STOP );
  SetFlag( PLAYBACK_START );

  return( TRUE );
} /* Open_Playback() */

/*----------------------------------------------------------------------*/

/* Xrun_Recovery()
 *
 * Recover from underrrun (broken pipe) and suspend
 */
  static gboolean
Xrun_Recovery( snd_pcm_t *handle, int error )
{
  if( error == -EPIPE )
  {
	error = snd_pcm_prepare( handle );
	if( error < 0 )
	{
	  fprintf( stderr,
		  _("Cannot recover from underrun, prepare failed\n"\
			"Error: %s\n"), snd_strerror(error) );
	  return( FALSE );
	}
   }
  else if( error == -ESTRPIPE )
  {
	while( (error = snd_pcm_resume(handle)) == -EAGAIN )
	  sleep(1);
	if( error < 0 )
	{
	  error = snd_pcm_prepare( handle );
	  if( error < 0 )
	  {
		fprintf( stderr,
			_("Cannot recover from suspend, prepare failed\n"\
			  "Error: %s\n"), snd_strerror(error) );
		return( FALSE );
	  }
	}
  }

  return( TRUE );
} /* Xrun_Recovery() */

/*----------------------------------------------------------------------*/

/*  DSP_Write()
 *
 *  Write a samples buffer to DSP
 */
  gboolean
DSP_Write( short *buffer, int buf_len )
{
  snd_pcm_sframes_t error;
  int idx, buf_idx;

  /* Abort if sound card not ready */
  if( handle == NULL ) return( FALSE );

  /* Write samples buffer to DSP in chunks of PERIOD_SIZE */
  buf_idx = 0;
  for( idx = 0; idx < playback_num_writes; idx++ )
  {
	/* Write samples buffer to DSP */
	error = snd_pcm_writei( handle, &buffer[buf_idx], SND_PERIOD_SIZE );
	if( error != SND_PERIOD_SIZE )
	{
	  if( error < 0 )
		fprintf( stderr, _("sdrx: DSP_Write(): %s\n"),
			snd_strerror((int)error) );
	  else
		fprintf( stderr,
			_("sdrx: DSP_Write(): Written only %d frames out of %d\n"),
			(int)error, buf_len );

	  /* Try to recover from error */
	  if( !Xrun_Recovery(handle, (int)error) ) return( FALSE );
	} /* if( error != PERIOD_SIZE ) */

	 buf_idx += SND_BUF_STRIDE;
  } /* for( idx = 0; idx < playback_num_writes; idx++ ) */

  return( TRUE );
} /* DSP_Write() */

/*----------------------------------------------------------------------*/

/* Close_Playback()
 *
 * Close playback handle
 */
  void
Close_Playback( void )
{
  /* Wait for async write to terminate */
  while( isFlagSet(PLAYBACK_STOP) )
	usleep( 10000 );

  ClearFlag( PLAYBACK_START );

  if( handle != NULL )
  {
	snd_pcm_close( handle );
	handle = NULL;
  }

  if( hw_params != NULL )
  {
	snd_pcm_hw_params_free( hw_params );
	hw_params = NULL;
  }

  /* Free playback buffer */
  if( playback_buf )
  {
	for( int idx = 0; idx < rc_data.playback_buffs; idx++ )
	  free_ptr( (void **)&playback_buf[idx] );
	free_ptr( (void **)&playback_buf );
  }

} /* Close_Playback() */

/*----------------------------------------------------------------------*/

