/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sdrplay.h"
#include "shared.h"


/* File variable for the center
 * frequency used in file functions */
static double rsp1_center_freq;

static pthread_t main_thread, rsp1_thread;

/*----------------------------------------------------------------------*/

/* SDRPlay_Data_Cb()
 *
 * Callback function of SdrPlay SDRPlay dongle
 * returning sampling data from the device.
 */
  static void
SDRPlay_Data_Cb(
	short *xi,
	short *xq,
	unsigned int firstSampleNum,
	int grChanged,
	int rfChanged,
	int fsChanged,
	unsigned int numSamples,
	unsigned int reset,
	void *cbContext )
{
  /* Local buffers for xi and xq */
  static short *xi_buf = NULL, *xq_buf = NULL;
  static int xiq_buf_len, xiq_cpy_idx = 0, xiq_out_idx = 0;

  static int sdrx_buffer_len = 0, count = 0;
  int sdrx_buf_idx, idx;


  /* Initialize on first call or change of params */
  if( sdrx_buffer_len != rc_data.sdrx_buffer_len )
  {
	sdrx_buffer_len = rc_data.sdrx_buffer_len;
	xiq_buf_len  = sdrx_buffer_len * 2;
	xiq_cpy_idx  = 0;
	xiq_out_idx  = 0;
	count = 0;

	size_t siz = (size_t)xiq_buf_len * sizeof(short);
	mem_realloc( (void *)&xi_buf, siz );
	mem_realloc( (void *)&xq_buf, siz );
  }

  /* Copy I/Q sample data to local buffers */
  for( idx = 0; idx < numSamples; idx++ )
  {
	xi_buf[xiq_cpy_idx] = xi[idx];
	xq_buf[xiq_cpy_idx] = xq[idx];
	xiq_cpy_idx++;
	if( xiq_cpy_idx >= xiq_buf_len )
	  xiq_cpy_idx -= xiq_buf_len;
  }

  /* Count number of samples buffered */
  count += (int)numSamples;

  /* Copy local buffers to sdrx buffers in double form */
  if( count >= sdrx_buffer_len )
  {
	for( sdrx_buf_idx = 0; sdrx_buf_idx < sdrx_buffer_len; sdrx_buf_idx++ )
	{
	  sdrx_buf_i[sdrx_buf_idx] = (double)xi_buf[xiq_out_idx];
	  sdrx_buf_q[sdrx_buf_idx] = (double)xq_buf[xiq_out_idx];
	  xiq_out_idx++;
	  if( xiq_out_idx >= xiq_buf_len )
		xiq_out_idx = 0;
	}
	count -= sdrx_buffer_len;
  }
  else return;

  /* Post to semaphore that DSP data is ready */
  int sval;
  sem_getvalue( &pback_semaphore, &sval );
  if( !sval ) sem_post( &pback_semaphore );

} /* SDRPlay_Data_Cb() */

/*----------------------------------------------------------------------*/

/* SDRPlay_GC()
 *
 * SDRPlay Gain Control callback function
 */
  static void
SDRPlay_GC( unsigned int gRdB, unsigned int lnaGRdB, void *cbContext )
{
  gRdB += lnaGRdB;
  if( gRdB > SDRPLAY_MAX_GRDB )
	gRdB = SDRPLAY_MAX_GRDB;

  /* This formula converts gain reductions to S-meter fraction
   * and the S-meter display function converts it to dBm. It is
   * not likely to be linear or accurate, it was worked out from
   * readings taken using a signal generator, experimentally */
  rc_data.S_meter    = 1.0 - ( 95.0 - 1.1 * (gRdB - 19.0) ) / SMETER_RANGE_DBM;
  rc_data.tuner_gain = ( 1.0 - rc_data.S_meter ) * MAN_GAIN_RANGE;

} /* SDRPlay_GC() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Set_Center_Frequency()
 *
 * Sets the SDRPlay Tuner Center Frequemcy
 */
  void
SDRPlay_Set_Center_Frequency( uint32_t center_freq )
{
  mir_sdr_ErrT ret;
  int idx;


  /* Frequency ranges in Hz of the SDRPlay RF front end */
  uint32_t freq_ranges[NUM_FREQ_RANGES] = { FREQUENCY_RANGES };
  static int prev_freq_idx = 0;

  /* Check for acceptable required center frequency */
  if( (center_freq >=  freq_ranges[0]) &&
	  (center_freq < freq_ranges[NUM_FREQ_RANGES-1]) )
	rsp1_center_freq = (double)center_freq;
  else
  {
	/* Display status of frequency change */
	Sdrx_Frequency_Status( ERROR );
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
	Error_Dialog(
		_("Required Center Frequency is out of range"), SHOW_OK );
	return;
  }

  if( isFlagSet(SDRX_INITIALIZED) )
  {
	/* Look for the range index of new frequency */
	for( idx = 1; idx < NUM_FREQ_RANGES; idx++ )
	  if( center_freq < freq_ranges[idx] )
		break;

	/* If no range change, use mir_sdr_SetRf() to set new frequency */
	if( prev_freq_idx == idx )
	{
	  ret = mir_sdr_SetRf( rsp1_center_freq, ABSOLUTE_MODE, IMMEDIATE );
	  if( ret == mir_sdr_Fail )
	  {
		/* Display status of frequency change */
		Sdrx_Frequency_Status( ERROR );
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
		Error_Dialog(
			_("Failed to set new Center Frequency"), SHOW_OK );
		return;
	  }
	}
	else
	{
	  int gRdB, gRdBsystem, samplesPerPacket;
	  mir_sdr_Bw_MHzT bwType;
	  mir_sdr_If_kHzT ifType;

	  /* Bandwidth in MHz, presumably in the SDRPlay hardware? */
	  bwType = mir_sdr_BW_0_200;

	  /* IF type to use (zero IF, I/Q) */
	  ifType = mir_sdr_IF_Zero;

	  /* Initial gain reduction */
	  gRdB = 30;

	  /* Re-initialize SDRPlay Streaming */
	  mir_sdr_Reinit(
		  &gRdB,
		  0,
		  (double)rsp1_center_freq / 1.0E6,
		  bwType,
		  ifType,
		  mir_sdr_LO_Auto,
		  ENABLE,
		  &gRdBsystem,
		  ENABLE,
		  &samplesPerPacket,
		  mir_sdr_CHANGE_RF_FREQ );

	  prev_freq_idx = idx;
	}

	/* Display status of frequency change */
	Sdrx_Frequency_Status( SUCCESS );

  } /* if( isFlagSet(SDRX_INITIALIZED) ) */

} /* SDRPlay_Set_Center_Frequency() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Set_Tuner_Gain_Mode()
 *
 * Sets the Tuner Gain mode to Auto or Manual
 */
  void
SDRPlay_Set_Tuner_Gain_Mode( int mode )
{
  /* Abort if Sdrx is not initialized */
  if( isFlagClear(SDRX_INITIALIZED) ) return;

  switch( mode )
  {
	case TUNER_GAIN_AUTO:
	case TUNER_GAIN_ADAGC:
	  mir_sdr_AgcControl(
		SDRPLAY_GAIN_AUTO, -30, 0, 0, 0, IMMEDIATE, ENABLE );
	  break;

	case TUNER_GAIN_MANUAL:
	  mir_sdr_AgcControl(
		SDRPLAY_GAIN_MANL, -30, 0, 0, 0, IMMEDIATE, ENABLE );
	  break;
  }

} /* SDRPlay_Set_Tuner_Gain_Mode() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Set_Tuner_Gain()
 *
 * Set the Tuner Gain if in Manual mode
 */
  gboolean
SDRPlay_Set_Tuner_Gain( double gain )
{
  int gRdBsystem;

  /* In th RSP1 gain reduction is specified */
  int gRdB = round( SDRPLAY_MAX_GRDB - gain );
  static int xgRdB = 0;

  /* Set gain only if there is a change */
  if( xgRdB != gRdB )
  {
	xgRdB = gRdB;
	mir_sdr_SetGrAltMode(
		&gRdB, ENABLE, &gRdBsystem, ABSOLUTE_MODE, IMMEDIATE );
	rc_data.tuner_gain = SDRPLAY_MAX_GRDB - (double)gRdB;
  }
  rc_data.S_meter = (double)gRdB / SDRPLAY_MAX_GRDB;

  return( TRUE );
} /* SDRPlay_Set_Tuner_Gain() */

/*----------------------------------------------------------------------*/

/* Calc_Decimation()
 *
 * Calculates the decimation ratio required to give
 * the final sampling reate required by the application
 */
  static gboolean
Calc_Decimation( int *decim )
{
  int idc, decm;
  double ratio;

  /* Use decimation if sampling rate < MIN_SAMPLING_RATE MHz */
  if( rc_data.sdrx_sample_rate &&
	  (rc_data.sdrx_sample_rate < MIN_SMPL_RATE) )
  {
	/* Find nearest acceptable decimation ratio
	 * FIXME >= 32 only with 8MHz rate and rate should be > IF B/W */
	ratio = (double)MIN_SMPL_RATE / (double)rc_data.sdrx_sample_rate;
	ratio = ceil( ratio );
	decm  = (int)ratio;

	/* Required decimation out of range */
	if( decm > DECIMATE_64 )
	{
	  Error_Dialog( _("Required Sampling Rate out of range"), SHOW_OK );
	  return( FALSE );
	}

	/* Find nearest acceptable decimation ratio */
	for( idc = 2; idc <= DECIMATE_64; idc *= 2 )
	{
	  if( idc >= decm )
	  {
		decm = idc;
		break;
	  }
	}

  } /* if( rc_data.sdrx_sample_rate < MIN_SAMPLING_RATE ) */
  else decm = 0;

  *decim = decm;

  return( TRUE );
} /* Calc_Decimation() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Set_Sample_Rate()
 *
 * Sets the SDRPlay Sample Rate
 */
  gboolean
SDRPlay_Set_Sample_Rate( void )
{
  /* Calculate oversampling to
   * give 48000 S/s to sound card */
  rc_data.oversampling =
	(int)rc_data.sdrx_sample_rate / SND_DSP_RATE;

  if( isFlagSet(SDRX_INITIALIZED) )
  {
	int decim;
	double rfMHz, fsMHz;
	mir_sdr_Bw_MHzT bwType;
	mir_sdr_If_kHzT ifType;
	int gRdB, gRdBsystem, samplesPerPacket;

	/* Calculate the sample rate according
	 * to decimation factor found below */
	if( !Calc_Decimation(&decim) ) return( FALSE );
	fsMHz = (double)rc_data.sdrx_sample_rate / 1.0E6;
	if( decim ) fsMHz *= (double)decim;

	/* Center frequency in MHz */
	rfMHz = rsp1_center_freq / 1.0E6;

	/* Bandwidth in MHz, presumably in the SDRPlay hardware? */
	bwType = mir_sdr_BW_0_200;

	/* IF type to use (zero IF, I/Q) */
	ifType = mir_sdr_IF_Zero;

	/* Initial gain reduction */
	gRdB = 30;

	/* Re-initialize SDRPlay Streaming */
	mir_sdr_Reinit(
		&gRdB,
		fsMHz,
		rfMHz,
		bwType,
		ifType,
		mir_sdr_LO_Auto,
		ENABLE,	// Enable LNA if useGrAltMode == ENABLE
		&gRdBsystem,
		ENABLE,
		&samplesPerPacket,
		mir_sdr_CHANGE_FS_FREQ );

	/* Set decimation control */
	if( decim )
	  mir_sdr_DecimateControl( ENABLE, decim, USE_AVERAGING );
	else
	  mir_sdr_DecimateControl( DISABLE, 0, 0 );

  } /* if( isFlagSet(SDRX_INITIALIZED) ) */

  return( TRUE );
} /* SDRPlay_Set_Sample_Rate() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Change_Buffer()
 *
 * Resets and reallocates SDRPlay async buffer
 */
  gboolean
SDRPlay_Change_Buffer( GtkComboBox *combobox )
{
  /* Cancel async streaming thread */
  SDRPlay_Close_Device();

  /* Make changes needed after buffer change */
  Sdrx_New_Buffer( combobox );

  /* Initialize SDRPlay Streaming */
  if( isFlagSet(PLAYBACK_START) ||
	  isFlagSet(MONITORS_START) )
	if( !SDRPlay_Initialize() ) return( FALSE );

  return( TRUE );
} /* SDRPlay_Change_Buffer() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Close_Device()
 *
 * Closes thr SDRPlay device, if open
 */
  void
SDRPlay_Close_Device( void )
{
  if( isFlagSet(SDRX_INITIALIZED) )
  {
	pthread_cancel( rsp1_thread );
	ClearFlag( SDRX_INITIALIZED );
	mir_sdr_ErrT err = mir_sdr_StreamUninit();
	if( err == mir_sdr_Fail )
	  Error_Dialog( _("Failed to Uninit SDRPlay Streaming"), HIDE_OK );
  }

} /* SDRPlay_Close_Device() */

/*----------------------------------------------------------------------*/

/* Mir_Sdr_StreamInit()
 *
 * Initializes the API and Device
 * hardware for samples streaming
 */
  static void *
Mir_Sdr_StreamInit( void *arg )
{
  mir_sdr_ErrT err;
  int gRdB, gRdBsystem, samplesPerPacket;
  double rfMHz, fsMHz;
  mir_sdr_Bw_MHzT bwType;
  mir_sdr_If_kHzT ifType;
  int setPoint_dBfs, decim;
  mir_sdr_AgcControlT enable;


  /* Init I and Q Low Pass Filter */
  Init_Roofing_Filter();

  /* Enable synthesizer frequency correction */
  mir_sdr_SetPpm( rc_data.rsp1_freq_correction );

  /* Calculate the sample rate according
   * to decimation factor found below */
  if( !Calc_Decimation(&decim) ) return( NULL );
  fsMHz = (double)rc_data.sdrx_sample_rate / 1.0E6;
  if( decim ) fsMHz *= (double)decim;

  /* Center frequency in MHz */
  rfMHz = rsp1_center_freq / 1.0E6;

  /* Bandwidth in MHz, presumably in the SDRPlay hardware? */
  bwType = mir_sdr_BW_0_200;

  /* IF type to use (zero IF, I/Q) */
  ifType = mir_sdr_IF_Zero;

  /* Initial gain reduction */
  gRdB = 30;

  /* Initialize SDRPlay data streaming */
  err = mir_sdr_StreamInit(
	  &gRdB,
	  fsMHz,
	  rfMHz,
	  bwType,
	  ifType,
	  ENABLE,	// Enable LNA if useGrAltMode == ENABLE
	  &gRdBsystem,
	  ENABLE,	// Use mir_sdr_SetGrAltMode()
	  &samplesPerPacket,
	  SDRPlay_Data_Cb,
	  SDRPlay_GC,
	  (void *)NULL );

  if( err != mir_sdr_Success )
  {
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
	Error_Dialog( _("Failed to Initialize SDRPlay Streaming"), HIDE_OK );
	return( NULL );
  }

  /* Display status of frequency change */
  Sdrx_Frequency_Status( SUCCESS );

  /* Set decimation control */
  if( decim )
	mir_sdr_DecimateControl( ENABLE, decim, USE_AVERAGING );
  else
	mir_sdr_DecimateControl( DISABLE, 0, 0 );

  /* Enable DC offset and IQ imbalance correction */
  mir_sdr_DCoffsetIQimbalanceControl( ENABLE, ENABLE );

  /* Enable AGC and Gain set point dB */
  setPoint_dBfs = -30;
  enable = mir_sdr_AGC_100HZ;

  /* Enable AGC with a setPoint of -30dBfs */
  err = mir_sdr_AgcControl(
	  enable,        /* Specifies the AGC mode required */
	  setPoint_dBfs, /* Specifies the required set point in dBfs */
	    0,			 /* Not currently used, set to 0 */
	    0,			 /* Not currently used, set to 0 */
	    0,			 /* Not currently used, set to 0 */
	  IMMEDIATE,     /* Update control */
	  ENABLE );	     /* LNA state to use in gain updates */

  if( err != mir_sdr_Success )
  {
	Error_Dialog( _("Failed to Setup SDRPlay AGC"), HIDE_OK );
	return( NULL );
  }

  SetFlag( SDRX_INITIALIZED );
  pthread_join( main_thread, NULL );

  return( NULL );
} /* Mir_Sdr_StreamInit() */

/*----------------------------------------------------------------------*/

/* SDRPlay_Initialize()
 *
 * Initializes the SDRPlay SDRPlay SDR Receiver
 */
  gboolean
SDRPlay_Initialize( void )
{
  mir_sdr_ErrT err;
  float ver;

  /* Check function return values */
  int ret = -1;

  /* Abort if already init */
  if( isFlagSet(SDRX_INITIALIZED) ) return( TRUE );

  /* Check API version */
  err = mir_sdr_ApiVersion( &ver );
  if( err == mir_sdr_Fail )
  {
	Error_Dialog( _("Failed to check libmirsdrapi-rsp version"), HIDE_OK );
	return( FALSE );
  }
  if( ver != MIR_SDR_API_VERSION )
  {
	Error_Dialog( _("Include file mirsdrapi-rsp.h: Version does not\n"
					"match the installed version of libmirsdrapi-rsp"), HIDE_OK );
	return( FALSE );
  }

  /* Enable debug output */
  //mir_sdr_DebugEnable( ENABLE );

  /* Create a thread for async read from RTl device */
  /* Get the thread id of the main thread */
  main_thread = pthread_self();

  ret = pthread_create( &rsp1_thread, NULL, Mir_Sdr_StreamInit, NULL );
  if( ret != SUCCESS )
  {
	Error_Dialog( _("Failed to create SDRPlay Streaming thread"), HIDE_OK );
	return( FALSE );
  }

  sleep( 2 );

  return( TRUE );
} /* SDRPlay_Initialize() */

/*----------------------------------------------------------------------*/

