/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "rtlsdr.h"
#include "shared.h"

/* rtlsdr device handle */
static rtlsdr_dev_t *dev = NULL;

static int
  num_gains,	  /* Number of tuner gain values */
  *gains = NULL;  /* Tuner gains array */

static pthread_t rtlsdr_pthread_id;
static sem_t rtlsdr_semaphore;

/*----------------------------------------------------------------------*/

/* RtlSdr_Data_Cb()
 *
 * Callback function for rtlsdr_read_async()
 */
  static void
RtlSdr_Data_Cb( unsigned char *buf, uint32_t len, void *ctx )
{
  int idx, sdrx_buf_idx;

  sdrx_buf_idx = 0;
  for( idx = 0; idx < len; idx += 2 )
  {
	sdrx_buf_i[sdrx_buf_idx] = (double)( buf[idx]   - 127 );
	sdrx_buf_q[sdrx_buf_idx] = (double)( buf[idx+1] - 127 );
	sdrx_buf_idx++;
  }
  RtlSdr_Get_Tuner_Gain();

  /* Post to semaphore that DSP data is ready */
  int sval;
  sem_getvalue( &pback_semaphore, &sval );
  if( !sval ) sem_post( &pback_semaphore );

} /* RtlSdr_Data_Cb() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Read_Async()
 *
 * Pthread function for async reading of RTLSDR I/Q samples
 */
  static void *
RtlSdr_Read_Async( void *arg )
{
  int ret = rtlsdr_read_async(
	  dev, RtlSdr_Data_Cb, arg, NUM_ASYNC_BUF,
	  (uint32_t)(2 * rc_data.sdrx_buffer_len) );

  if( ret != SUCCESS )
	fprintf( stderr, _("rtlsdr_read_async() returned %d\n"), ret );

  int sval;
  sleep(1);
  sem_getvalue( &rtlsdr_semaphore, &sval );
  if( !sval ) sem_post( &rtlsdr_semaphore );

  return( NULL );
} /* RtlSdr_Read_Async() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Set_Center_Frequency()
 *
 * Sets the Center Frequency of the RTLSDR Tuner
 */
  void
RtlSdr_Set_Center_Frequency( uint32_t center_freq )
{
  int ret = 0;

  if( isFlagSet(RTLSDR_OPEN) )
  {
	/* Set the Center Frequency of the RTLSDR Device */
	ret = rtlsdr_set_center_freq( dev, center_freq );
	if( ret < 0 )
	{
	  /* Display status of frequency change */
	  Sdrx_Frequency_Status( ERROR );
	  gtk_toggle_button_set_active(
		  GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
	  Error_Dialog(
		  _("Failed to set Tuner Center Frequency"), SHOW_OK );
	  return;
	}

	/* Get the Center Frequency of the RTLSDR Device */
	ret = rtlsdr_get_center_freq( dev );

	/* Display status of frequency change */
	gboolean err = (ret != center_freq) || (ret == 0);
	if( err )
	{
	  Sdrx_Frequency_Status( ERROR );
	  gtk_toggle_button_set_active(
		  GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
	  Error_Dialog(
		  _("Failed to set Tuner Center Frequency"), SHOW_OK );
	  return;
	}

  } /* if( isFlagSet(RTLSDR_OPEN) ) */

  /* Display status of frequency change */
  Sdrx_Frequency_Status( SUCCESS );

} /* RtlSdr_Set_Center_Frequency() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Set_Tuner_Gain_Mode()
 *
 * Sets the Tuner Gain mode to Auto or Manual
 */
  void
RtlSdr_Set_Tuner_Gain_Mode( int mode )
{
  /* Abort if RTLSDR device is not open */
  if( isFlagClear(RTLSDR_OPEN) )
	return;

  switch( mode )
  {
	case TUNER_GAIN_AUTO:
	case TUNER_GAIN_ADAGC:
	  mode = 0; /* Auto AGC */
	  break;

	case TUNER_GAIN_MANUAL:
	  mode = 1; /* Manual AGC */
	  break;
  }

  /* Set Tuner Gain Mode */
  gboolean ret = rtlsdr_set_tuner_gain_mode( dev, mode );
  if( ret != SUCCESS )
  {
	Error_Dialog( _("Failed to set Tuner Gain Mode"), SHOW_OK );
	return;
  }

  /* Get and display Tuner gain also */
  RtlSdr_Get_Tuner_Gain();

} /* RtlSdr_Set_Tuner_Gain_Mode() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Set_Tuner_Gain()
 *
 * Set the Tuner Gain if in Manual mode
 */
  gboolean
RtlSdr_Set_Tuner_Gain( double gain )
{
  /* Abort if RTLSDR device is not open */
  if( isFlagClear(RTLSDR_OPEN) )
	return( TRUE );

  int
	idx,
	min,
	diff,
	igain,
	igx;

  if( gains == NULL )
  {
	/* Get the available Tuner Gains */
	num_gains = rtlsdr_get_tuner_gains( dev, NULL );
	if( num_gains <= 0 )
	{
	  Error_Dialog( _("Failed to get Tuner Number of Gains"), HIDE_OK );
	  return( FALSE );
	}

	/* Get the Gains List from the Device */
	mem_alloc( (void *)&gains, (size_t)num_gains * sizeof(int) );
	num_gains = rtlsdr_get_tuner_gains( dev, gains );
	if( num_gains <= 0 )
	{
	  Error_Dialog( _("Failed to get Tuner Gains Array"), HIDE_OK );
	  return( FALSE );
	}

  } /* if( gains == NULL ) */

  /* Find nearest available gain */
  igain = (int)( gain * 10.0 );
  min = 1000; igx = 0; /* Prime */
  for( idx = 0; idx < num_gains; idx++ )
  {
	diff = abs( gains[idx] - igain );
	if( diff < min )
	{
	  min = diff;
	  igx = idx;
	}
  }
  if( igx >= num_gains )
	igx = num_gains - 1;

  /* Set the Tuner Gain */
  int ret = rtlsdr_set_tuner_gain( dev, gains[igx] );
  if( ret != SUCCESS )
  {
	Error_Dialog( _("Failed to set Tuner Gain"), HIDE_OK );
	return( FALSE );
  }

  rc_data.tuner_gain = gains[igx] / 10.0;

  return( TRUE );
} /* RtlSdr_Set_Tuner_Gain() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Get_Tuner_Gain()
 *
 * Gets the Gain setting from the RTLSDR Tuner
 * and displays it in the relevant entry widget
 */
  void
RtlSdr_Get_Tuner_Gain( void )
{
  /* Abort if RTLSDR device is not open */
  if( isFlagClear(RTLSDR_OPEN) )
	return;

  /* Get the Tuner Gain */
  rc_data.tuner_gain = (double)rtlsdr_get_tuner_gain( dev ) / 10.0;
  rc_data.S_meter =
	( RTLSDR_MAX_GAIN - rc_data.tuner_gain ) / RTLSDR_MAX_GAIN;

} /* RtlSdr_Get_Tuner_Gain() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Set_Sample_Rate()
 *
 * Sets the RTLSDR Sample Rate
 */
  gboolean
RtlSdr_Set_Sample_Rate( void )
{
  /* Abort if RTLSDR device is not open */
  if( isFlagClear(RTLSDR_OPEN) )
	return( TRUE );

  /* rtlsdr device sample rate */
  uint32_t sample_rate;

  /* Set RTLSDR Sample Rate */
  int ret = rtlsdr_set_sample_rate(
	  dev, rc_data.sdrx_sample_rate );
  if( (ret != SUCCESS) || (ret == -EINVAL) )
  {
	Error_Dialog( _("Failed to set ADC Sample Rate"), HIDE_OK );
	return( FALSE );
  }

  /* Get RTLSDR Sample Rate */
  sample_rate = rtlsdr_get_sample_rate( dev );
  if( sample_rate == 0 )
  {
	Error_Dialog( _("Failed to get ADC Sample Rate"), HIDE_OK );
	return( FALSE );
  }

  /* Calculate oversampling to give 48000 S/s to sound card */
  rc_data.oversampling = sample_rate / SND_DSP_RATE;

  return( TRUE );
} /* RtlSdr_Set_Sample_Rate() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Change_Buffer()
 *
 * Resets and reallocates RTLSDR async buffer
 */
  gboolean
RtlSdr_Change_Buffer( GtkComboBox *combobox )
{
  int ret;


  /* Cancel async streaming thread */
  if( isFlagSet(SDRX_INITIALIZED) )
  {
	rtlsdr_cancel_async( dev );
	sem_wait( &rtlsdr_semaphore );
  }

  /* Make changes needed after buffer change */
  Sdrx_New_Buffer( combobox );

  /* Reset RTLSDR data buffer */
  if( isFlagSet(RTLSDR_OPEN) )
  {
	ret = rtlsdr_reset_buffer( dev );
	if( ret != SUCCESS )
	{
	  Error_Dialog(
		  _("Failed to Reset Sampling Buffer"), HIDE_OK );
	  return( FALSE );
	}
  }

  /* Create the thread for async read from RTL device */
  if( isFlagSet(SDRX_INITIALIZED) )
  {
	usleep( 100000 );
	ret = pthread_create(
		&rtlsdr_pthread_id, NULL, RtlSdr_Read_Async, NULL );
	if( ret != SUCCESS )
	{
	  Error_Dialog(
		  _("Failed to create async read thread"), HIDE_OK );
	  return( FALSE );
	}
  }

  return( TRUE );
} /* RtlSdr_Change_Buffer() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Initialize()
 *
 * Initialize rtlsdr device status
 */
  gboolean
RtlSdr_Initialize( void )
{
  /* Check function return values */
  int ret = -1;

  /* Abort if already init */
  if( isFlagSet(SDRX_INITIALIZED) ) return( TRUE );

  /* Open RTLSDR Device */
  ret = rtlsdr_open( &dev, rc_data.rtlsdr_device_index );
  if( ret != SUCCESS )
  {
	Error_Dialog( _("Failed to open RTL-SDR device"), HIDE_OK );
	return( FALSE );
  }
  SetFlag( RTLSDR_OPEN );

  /* Set the Center Frequency of the RTLSDR Device */
  RtlSdr_Set_Center_Frequency( rc_data.sdrx_center_freq );

  /* Set the Frequency Correction factor for the device */
  if( rc_data.rtlsdr_freq_correction )
  {
	ret = rtlsdr_set_freq_correction( dev, rc_data.rtlsdr_freq_correction );
	if( ret != SUCCESS )
	{
	  Error_Dialog(
		  _("Failed to set Frequency Correction factor"), HIDE_OK );
	  return( FALSE );
	}
  }

  /* Set Tuner Gain Mode to auto */
  RtlSdr_Set_Tuner_Gain_Mode( TUNER_GAIN_AUTO );

  /* Set the RTLSDR ADC/DSP sample rate */
  if( !RtlSdr_Set_Sample_Rate() )
	return( FALSE );

  /* Reset and reallocate async buffer */
  if( !RtlSdr_Change_Buffer( NULL ) )
	return( FALSE );

  /* Init I and Q Low Pass Filter */
  Init_Roofing_Filter();

  /* Create a thread for async read from RTl device */
  ret = pthread_create( &rtlsdr_pthread_id, NULL, RtlSdr_Read_Async, NULL );
  if( ret != SUCCESS )
  {
	Error_Dialog(
		_("Failed to create RtlSdr Async Read thread"), HIDE_OK );
	return( FALSE );
  }
  sleep( 1 );

  sem_init( &rtlsdr_semaphore,  0, 0 );
  SetFlag( SDRX_INITIALIZED );

  return( TRUE );
} /* RtlSdr_Initialize() */

/*----------------------------------------------------------------------*/

/* RtlSdr_Close_Device()
 *
 * Closes thr RTLSDR device, if open
 */
  void
RtlSdr_Close_Device( void )
{
  ClearFlag( SDRX_INITIALIZED );
  ClearFlag( RTLSDR_OPEN );
  rtlsdr_cancel_async( dev );

  if( dev != NULL )
  {
	sem_wait( &rtlsdr_semaphore );
	rtlsdr_close( dev );
	dev = NULL;
  }

  if( gains != NULL )
  {
	free( (void *)gains );
	gains = NULL;
  }

} /* RtlSdr_Close_Device() */

/*----------------------------------------------------------------------*/

