/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "perseus.h"
#include "shared.h"

/* Perseus device description */
static perseus_descr *descr = NULL;

/*----------------------------------------------------------------------*/

/* Perseus_Data_Cb()
 *
 * Callback function for perseus_start_async_input
 */
  static int
Perseus_Data_Cb( void *buf, int buf_size, void *extra )
{
  /* The buffer received contains 24-bit IQ samples
   * (6 bytes per sample). We convert the samples to
   * 32 bit (msb aligned) integer IQ samples and
   * then store them in the Sdrx's buffer as doubles */
  uint8_t *samplebuf = (uint8_t*)buf;
  int idx, nSamples  = buf_size / 6;
  iq_sample s;

  /* Local buffers for i and q */
  static int32_t *i_buf = NULL, *q_buf = NULL;
  static int iq_buf_len, iq_cpy_idx = 0, iq_out_idx = 0;

  static int sdrx_buffer_len = 0, count = 0;
  int sdrx_buf_idx;


  /* Initialize on first call or change of params */
  if( sdrx_buffer_len != rc_data.sdrx_buffer_len )
  {
	sdrx_buffer_len = rc_data.sdrx_buffer_len;
	iq_buf_len   = sdrx_buffer_len * 2;
	iq_cpy_idx   = 0;
	iq_out_idx   = 0;
	count        = 0;

	size_t siz = (size_t)iq_buf_len * sizeof(int32_t);
	mem_realloc( (void *)&i_buf, siz );
	mem_realloc( (void *)&q_buf, siz );
  }

  /* Clear top bytes */
  s.iq_data.i1 = s.iq_data.q1 = 0;

  sdrx_buf_idx = 0;
  for( idx = 0; idx < nSamples; idx++ )
  {
	/* Repack data to 32 bit (msb aligned) integers */
	s.iq_data.i2 = *samplebuf++;
	s.iq_data.i3 = *samplebuf++;
	s.iq_data.i4 = *samplebuf++;
	s.iq_data.q2 = *samplebuf++;
	s.iq_data.q3 = *samplebuf++;
	s.iq_data.q4 = *samplebuf++;

	/* Copy I/Q sample data to local buffers */
	i_buf[iq_cpy_idx] = s.iq.i;
	q_buf[iq_cpy_idx] = s.iq.q;
	iq_cpy_idx++;
	if( iq_cpy_idx >= iq_buf_len )
	  iq_cpy_idx -= iq_buf_len;

  } /* for( idx = 0; idx < nSamples; idx++ ) */

  /* Count number of samples buffered */
  count += (int)nSamples;

  /* Copy local buffers to sdrx buffers in double form */
  if( count >= sdrx_buffer_len )
  {
	for( sdrx_buf_idx = 0; sdrx_buf_idx < sdrx_buffer_len; sdrx_buf_idx++ )
	{
	  sdrx_buf_i[sdrx_buf_idx] = (double)i_buf[iq_out_idx];
	  sdrx_buf_q[sdrx_buf_idx] = (double)q_buf[iq_out_idx];
	  iq_out_idx++;
	  if( iq_out_idx >= iq_buf_len ) iq_out_idx = 0;
	}
	count -= sdrx_buffer_len;
  }
  else return( 0 );

  /* Post to semaphore that DSP data is ready */
  int sval;
  sem_getvalue( &pback_semaphore, &sval );
  if( !sval ) sem_post( &pback_semaphore );

  return( 0 );
} /* Perseus_Data_Cb() */

/*----------------------------------------------------------------------*/

/* Perseus_Read_Async()
 *
 * Pthread function to start async reading of Perseus I/Q samples
 */
  static void *
Perseus_Read_Async( void *arg )
{
  int ret = perseus_start_async_input(
	  descr, PERSEUS_ASYNC_BUF_SIZE,
	  Perseus_Data_Cb, NULL );

  if( ret < 0 )
	fprintf( stderr,
		_("perseus_start_async_input() returned %s\n"),
		perseus_errorstr() );

  return( NULL );
} /* Perseus_Read_Async() */

/*----------------------------------------------------------------------*/

/* Perseus_Set_Center_Frequency()
 *
 * Sets the Perseus FPGA DDC Center Frequency
 */
  void
Perseus_Set_Center_Frequency( uint32_t center_freq )
{
  if( isFlagSet(SDRX_INITIALIZED) )
  {
	/* Set center frequency, enable preselector filters */
	double freq = (double)center_freq;
	freq *= 1.0 + rc_data.perseus_freq_correction / 1.0E6;
	int ret = perseus_set_ddc_center_freq(
		descr, (long)freq, isFlagSet(PRESELECT_ENABLE) );

	/* Display status of frequency change */
	if( ret < 0 )
	{
	  Sdrx_Frequency_Status( ERROR );
	  gtk_toggle_button_set_active(
		  GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
	  Error_Dialog(
		  _("Failed to set DDC Center Frequency"), SHOW_OK );
	  return;
	}

	/* Display status of frequency change */
	Sdrx_Frequency_Status( SUCCESS );

  } /* if( isFlagSet(SDRX_INITIALIZED) ) */

} /* Perseus_Set_Center_Frequency() */

/*----------------------------------------------------------------------*/

/* Perseus_Set_Tuner_Gain_Mode()
 *
 * Sets the Tuner Gain mode to Auto or Manual
 */
  void
Perseus_Set_Tuner_Gain_Mode( int mode )
{
  if( mode == TUNER_GAIN_AUTO )
  {
	ClearFlag( TUNER_GAIN_AUTO );
	ClearFlag( TUNER_GAIN_MANUAL );
	SetFlag(   TUNER_GAIN_ADAGC );
  }

} /* Perseus_Set_Tuner_Gain_Mode() */

/*----------------------------------------------------------------------*/

/* Perseus_Set_Tuner_Gain()
 *
 * Set the Tuner Gain if in Manual mode
 */
  gboolean
Perseus_Set_Tuner_Gain( double gain )
{
  /* Gain is in range 0 - 100. Reverse to provide signal level */
  gain = MAN_GAIN_RANGE - gain;

  /* Display S-Meter (range is 0.0 - 1.0) */
  rc_data.S_meter = gain / MAN_GAIN_RANGE;

  /* Calculate equivalent ADAGC scale */
  rc_data.adagc_scale =
	rc_data.S_meter * SMETER_RANGE_DBM - rc_data.smeter_offset;
  rc_data.adagc_scale = pow( 10.0, rc_data.adagc_scale / 20.0 );

  return( TRUE );
} /* Perseus_Set_Tuner_Gain() */

/*----------------------------------------------------------------------*/

/* Perseus_Set_Sample_Rate()
 *
 * Sets the Perseus Sample Rate
 */
  gboolean
Perseus_Set_Sample_Rate( void )
{
  if( isFlagSet(SDRX_INITIALIZED) )
  {
	/* Stop async input */
	perseus_stop_async_input( descr );

	/* Configure the receiver sample rate */
	int ret = perseus_set_sampling_rate(
		descr, (int)rc_data.sdrx_sample_rate );
	if( ret < 0 )
	{
	  char txt[MESG_SIZE];
	  snprintf( txt, sizeof(txt),
		  _("Perseus FPGA configuration error:\n%s"),
		  perseus_errorstr() );
	  Error_Dialog( txt, HIDE_OK );
	  return( FALSE );
	}

	/* Restart async input */
	ret = perseus_start_async_input(
		descr, PERSEUS_ASYNC_BUF_SIZE,
		Perseus_Data_Cb, NULL );
	if( ret < 0 )
	  fprintf( stderr,
		  _("perseus_start_async_input() returned %s\n"),
		  perseus_errorstr() );

  } /* if( isFlagSet(SDRX_INITIALIZED) ) */

  /* Calculate oversampling to give 48000 S/s to sound card */
  rc_data.oversampling =
	(int)rc_data.sdrx_sample_rate / SND_DSP_RATE;

  return( TRUE );
} /* Perseus_Set_Sample_Rate() */

/*----------------------------------------------------------------------*/

/* Perseus_Change_Buffer()
 *
 * Resets and reallocates Perseus async buffer
 */
  gboolean
Perseus_Change_Buffer( GtkComboBox *combobox )
{
  /* Stop async input */
  if( isFlagSet(SDRX_INITIALIZED) )
	perseus_stop_async_input( descr );

  /* Make changes needed after buffer change */
  Sdrx_New_Buffer( combobox );

  if( isFlagSet(SDRX_INITIALIZED) )
  {
	/* Re-configure the receiver sample rate */
	int ret = perseus_set_sampling_rate(
		descr, (int)rc_data.sdrx_sample_rate );
	if( ret < 0 )
	{
	  char txt[MESG_SIZE];
	  snprintf( txt, sizeof(txt),
		  _("Perseus FPGA configuration error:\n%s"),
		  perseus_errorstr() );
	  Error_Dialog( txt, HIDE_OK );
	  return( FALSE );
	}

	/* Restart async read from Perseus device */
	ret = perseus_start_async_input(
		descr, PERSEUS_ASYNC_BUF_SIZE,
		Perseus_Data_Cb, NULL );
	if( ret < 0 )
	  fprintf( stderr,
		  _("perseus_start_async_input() returned %s\n"),
		  perseus_errorstr() );
  } /* if( isFlagSet(SDRX_INITIALIZED) ) */

  return( TRUE );
} /* Perseus_Change_Buffer() */

/*----------------------------------------------------------------------*/

/* Perseus_Settings()
 *
 * Sets the Preamp on/off, Attenuators on/off and ADC Dither on/off
 */
  gboolean
Perseus_Settings( uint8_t setting )
{
  char txt[MESG_SIZE];
  static uint8_t settings = PERSEUS_INITIAL_SETTINGS;
  int ret;


  if( !descr ) return( TRUE );

  /* Set or clear a setting flag bit */
  if( setting & PERSEUS_CLEAR_SETTING )
	settings &= setting;
  else
	settings |= setting;

  /* Set attenuators */
  ret = perseus_set_attenuator_n(
	  descr, settings & PERSEUS_ATTEN_SETTINGS );
  if( ret < 0 )
  {
	snprintf( txt, sizeof(txt),
		_("Perseus: perseus_set_attenuator() error:\n%s"),
		perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( FALSE );
  }
  else /* Calculate total attenuator setting */
  {
	double att_10db = 0.0, att_20db = 0.0;

	if( settings & PERSEUS_ATTEN_10DB )
	  att_10db = 10.0;
	else
	  att_10db = 0.0;

	if( settings & PERSEUS_ATTEN_20DB )
	  att_20db = 20.0;
	else
	  att_20db = 0.0;
	rc_data.perseus_attenuator = att_10db + att_20db;

	if( !(settings & PERSEUS_ADC_PREAMP) )
	  rc_data.perseus_attenuator += PERSEUS_PREAMP_GAIN;
  }

  /* Set ADC preamp and dither */
  ret = perseus_set_adc( descr,
	  settings & PERSEUS_ADC_DITHER,
	  settings & PERSEUS_ADC_PREAMP );
  if( ret < 0 )
  {
	snprintf( txt, sizeof(txt),
		_("Perseus perseus_set_adc() error:\n%s"),
		perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( FALSE );
  }

	return( TRUE );
} /* Perseus_Settings() */

/*----------------------------------------------------------------------*/

/* Perseus_Close_Device()
 *
 * Closes thr Perseus device, if open
 */
  void
Perseus_Close_Device( void )
{
  if( isFlagSet(SDRX_INITIALIZED) )
  {
	ClearFlag( SDRX_INITIALIZED );
	perseus_stop_async_input( descr );
	perseus_close( descr );
	descr = NULL;
	perseus_exit();
  }

} /* Perseus_Close_Device() */

/*----------------------------------------------------------------------*/

  static void
Perseus_Init_Error( void )
{
  perseus_close( descr );
  descr = NULL;
} /* Perseus_Init_Error() */

/*----------------------------------------------------------------------*/

/* Perseus_Initialize()
 *
 * Initializes the Perseus Perseus SDR Receiver
 */
  gboolean
Perseus_Initialize( void )
{
  int ret;
  char txt[MESG_SIZE];
  eeprom_prodid prodid;
  pthread_t pthread_id;

  /* Abort if already init */
  if( isFlagSet(SDRX_INITIALIZED) ) return( TRUE );

  /* Set debug info dumped to stderr
   * to the maximum verbose level */
  perseus_set_debug( PERSEUS_DEBUG_LEVEL );

  /* Check how many Perseus receivers
   * are connected to the system */
  ret = perseus_init();
  if( ret == 0 )
  {
	perseus_exit();
	Error_Dialog( _("No Perseus Receivers detected"), HIDE_OK );
	return( FALSE );
  }

  /* Open Perseus device */
  descr = perseus_open( rc_data.perseus_device_index );
  if( descr == NULL )
  {
	perseus_exit();
	snprintf( txt, sizeof(txt),
		_("Failed to open Perseus Receiver:\n%s"), perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( FALSE );
  }

  /* Download the standard firmware to the unit */
  ret = perseus_firmware_download( descr, NULL );
  if( ret < 0 )
  {
	Perseus_Init_Error();
	snprintf( txt, sizeof(txt),
		_("Perseus Firmware download error:\n%s"), perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( FALSE );
  }

  /* Dump some information about the receiver (S/N and HW rev) */
  if( descr->is_preserie == TRUE )
	fprintf( stderr, "The device is a preserie unit" );
  else if( perseus_get_product_id(descr, &prodid) < 0 )
	fprintf( stderr, "Perseus get_product_id error: %s",
		perseus_errorstr() );
  else
	fprintf(stderr, "Receiver S/N: "
		"%05d-%02hX%02hX-%02hX%02hX-%02hX%02hX - HW Release:%hd.%hd\n",
		(uint16_t) prodid.sn,
		(uint16_t) prodid.signature[5],
		(uint16_t) prodid.signature[4],
		(uint16_t) prodid.signature[3],
		(uint16_t) prodid.signature[2],
		(uint16_t) prodid.signature[1],
		(uint16_t) prodid.signature[0],
		(uint16_t) prodid.hwrel,
		(uint16_t) prodid.hwver);

  /* Configure the receiver sample rate */
  ret = perseus_set_sampling_rate(
	  descr, (int)rc_data.sdrx_sample_rate );
  if( ret < 0 )
  {
	Perseus_Init_Error();
	snprintf( txt, sizeof(txt),
		_("Perseus fpga configuration error:\n%s"),
		perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( FALSE );
  }

  /* Disable ADC Dither, enable ADC Preamp, disable attenuators */
  rc_data.perseus_attenuator = 0.0;
  if( !Perseus_Settings(PERSEUS_INITIAL_SETTINGS) )
  {
	Perseus_Init_Error();
	return( FALSE );
  }

  /* Set center frequency, enable preselector filters */
  double center_freq =
	(double)rc_data.sdrx_center_freq + (double)rc_data.weaver_offset;
  center_freq *= 1.0 + rc_data.perseus_freq_correction / 1.0E6;
  ret = perseus_set_ddc_center_freq(
	  descr, (long)center_freq, isFlagSet(PRESELECT_ENABLE) );
  if( ret < 0 )
  {
	Perseus_Init_Error();
	Sdrx_Frequency_Status( ERROR );
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );
	Error_Dialog(
		_("Failed to set DDC Center Frequency"), SHOW_OK );
	return( FALSE );
  }

  /* Display status of frequency change */
  Sdrx_Frequency_Status( SUCCESS );

  /* Init I and Q Low Pass Filter */
  Init_Roofing_Filter();

  /* Set Perseus Gain Mode to auto, which switches gain mode to ADAGC */
  Perseus_Set_Tuner_Gain_Mode( TUNER_GAIN_AUTO );

	/* Create a thread for async read from Perseus device */
  ret = pthread_create( &pthread_id, NULL, Perseus_Read_Async, NULL );
  if( ret != SUCCESS )
  {
	Perseus_Init_Error();
	Error_Dialog(
		_("Failed to create Perseus Async Read thread"), HIDE_OK );
	return( FALSE );
  }
  sleep( 1 );

  SetFlag( SDRX_INITIALIZED );

  return( TRUE );
} /* Perseus_Initialize() */

/*----------------------------------------------------------------------*/

/* Perseus_Sample_Rate_Idx()
 *
 * Finds the combobox index that corresponds to a sample rate
 */
  int
Perseus_Sample_Rate_Idx( int rate )
{
  int buf[PERSEUS_RATES_BUF_SIZE], idx, ret;
  char txt[MESG_SIZE];

  /* Get available sampling rates */
  ret = perseus_get_sampling_rates(
	  NULL, buf, PERSEUS_RATES_BUF_SIZE );
  if( ret < 0 )
  {
	snprintf( txt, sizeof(txt),
		_("Perseus get_sampling_rates error:\n%s"),
		perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( 0 );
  }

  /* Find which combobox entry
   * corresponds to given rate */
  idx = 0;
  while( buf[idx] )
  {
	buf[idx] /= 1000;
	if( buf[idx] == rate ) return( idx );
	idx++;
  }

  return( 0 );
} /* Perseus_Sample_Rate_Idx() */

/*----------------------------------------------------------------------*/

/* Perseus_Sample_Rate()
 *
 * Finds the sample rate that corresponds to a combobox index
 */
  int
Perseus_Sample_Rate( int index )
{
  int buf[PERSEUS_RATES_BUF_SIZE], ret, rate;
  char txt[MESG_SIZE];

  /* Get available sampling rates */
  ret = perseus_get_sampling_rates(
	  NULL, buf, PERSEUS_RATES_BUF_SIZE );
  if( ret < 0 )
  {
	snprintf( txt, sizeof(txt),
		_("Perseus get_sampling_rates error:\n%s"),
		perseus_errorstr() );
	Error_Dialog( txt, HIDE_OK );
	return( 0 );
  }

  /* Return rate in ks/s */
  rate = buf[index] / 1000;
  return( rate );

} /* Perseus_Sample_Rate() */

/*----------------------------------------------------------------------*/

/* Perseus_ADAGC_Smeter()
 *
 * Calculates S-meter indication from ADAGC scale factor
 * and also controls input attenuators in auto mode
 */
  void
Perseus_ADAGC_Smeter( void )
{
  /* ADAGC scale factor in relative dBm */
  double log_adagc_scale;
  static gboolean att_10db = FALSE, att_20db = FALSE;

  /* Set Perseus attenuators if in auto mode */
  log_adagc_scale = 20.0 * log10( rc_data.adagc_scale );
  if( isFlagSet(PERSEUS_ATTEN_AUTO) )
  {
	if( log_adagc_scale > PERSEUS_ATT_UP )
	{
	  if( !att_10db && !att_20db )
	  {
		Perseus_Settings( PERSEUS_ATTEN_10DB );
		att_10db = TRUE;
	  }
	  else if( !att_20db && att_10db )
	  {
		Perseus_Settings(  PERSEUS_ATTEN_20DB );
		Perseus_Settings( ~PERSEUS_ATTEN_10DB );
		att_20db = TRUE;
		att_10db = FALSE;
	  }
	  else if( att_20db && !att_10db )
	  {
		Perseus_Settings( PERSEUS_ATTEN_10DB );
		att_10db = TRUE;
	  }
	} /* if( log_adagc_scale > PERSEUS_ATT_UP ) */
	else if( log_adagc_scale < PERSEUS_ATT_DOWN )
	{
	  if( att_10db )
	  {
		Perseus_Settings( ~PERSEUS_ATTEN_10DB );
		att_10db = FALSE;
	  }
	  else if( att_20db && !att_10db )
	  {
		Perseus_Settings(  PERSEUS_ATTEN_10DB );
		Perseus_Settings( ~PERSEUS_ATTEN_20DB );
		att_10db = TRUE;
		att_20db = FALSE;
	  }
	} /* if( log_adagc_scale > PERSEUS_ATT_DOWN ) */

  } /* if( isFlagSet(PERSEUS_ATTEN_AUTO) ) */
  else
  {
	att_10db = FALSE;
	att_20db = FALSE;
  }

  /* Total offset value for correct S-meter indication */
  rc_data.smeter_offset = PERSEUS_SMETER_OFFSET + rc_data.perseus_attenuator;
  rc_data.S_meter  = log_adagc_scale + rc_data.smeter_offset;
  rc_data.S_meter /= SMETER_RANGE_DBM;
  rc_data.tuner_gain = ( 1.0 - rc_data.S_meter ) * MAN_GAIN_RANGE;

} /* Perseus_ADAGC_Smeter() */

/*----------------------------------------------------------------------*/

