/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "monitors.h"
#include "shared.h"

/* FFT output bin values */
static int *fft_bin_val = NULL;

/* Height of Signal Display */
static int signal_height;

/*----------------------------------------------------------------------*/

/* FFT_Initialize()
 *
 * Initializes FFT()
 */
  gboolean
FFT_Initialize( int fft_length )
{
  size_t mreq;
  static int fft_len = 0;


  /* Abort if input data not right */
  if( !fft_length )
  {
	ClearFlag( FFT_INITIALIZED );
	Error_Dialog( _("Failed to Initialize FFT() "), HIDE_OK );
	return( FALSE );
  }

  /* Reallocate on length change */
  if( fft_len != fft_length )
  {
	/* Allocate fft buffers */
	ClearFlag( FFT_INITIALIZED );
	mreq = (size_t)fft_length * sizeof(double);
	mem_realloc( (void **)&fft_in_i, mreq );
	mem_realloc( (void **)&fft_in_q, mreq );
	mem_realloc( (void **)&fft_bin_val, mreq );

	fft_len = fft_length;
  } /* if( fft_length != fft_len ) */

  /* Scale factor to keep FFT in range */
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  rc_data.fft_scale = RTLSDR_FFT_SCALE;
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  rc_data.fft_scale = SDRPLAY_FFT_SCALE;
	  break;

	case DONGLE_TYPE_PERSEUS:
	  rc_data.fft_scale = PERSEUS_FFT_SCALE;
	  break;
  }

  SetFlag( FFT_INITIALIZED );
  SetFlag( MONITORS_START );
  return( TRUE );

} /* FFT_Initialize() */

/*----------------------------------------------------------------------*/

/* FFT()
 *
 * This computes an in-place complex-to-complex FFT
 * x and y are the real and imaginary arrays of 2^m points.
 */
  void
FFT( int n, double *x, double *y )
{
  int m, i, i1, j, k, i2, l, l1, l2;
  double c1, c2, tx, ty, t1, t2, u1, u2, z;

  /* Calculate the order of fft */
  i = n;
  for( m = 0; i != 1; m++ )
	i /= 2;

  /* Do the bit reversal */
  i2 = n >> 1;
  j = 0;
  for( i = 0; i < n - 1; i++ )
  {
	if( i < j )
	{
	  tx   = x[i];
	  ty   = y[i];
	  x[i] = x[j];
	  y[i] = y[j];
	  x[j] = tx;
	  y[j] = ty;
	}

	k = i2;
	while( k <= j )
	{
	  j -= k;
	  k >>= 1;
	}

	j += k;
  }

  /* Compute the FFT */
  c1 = -1.0;
  c2 = 0.0;
  l2 = 1;

  for( l = 0; l < m; l++ )
  {
	l1 = l2;
	l2 <<= 1;
	u1 = 1.0;
	u2 = 0.0;

	for( j = 0; j < l1; j++ )
	{
	  for( i = j; i < n; i += l2 )
	  {
		i1 = i + l1;
		t1 = u1 * x[i1] - u2 * y[i1];
		t2 = u1 * y[i1] + u2 * x[i1];
		x[i1] = x[i] - t1;
		y[i1] = y[i] - t2;
		x[i] += t1;
		y[i] += t2;
	  }

	  z =  u1 * c1 - u2 * c2;
	  u2 = u1 * c2 + u2 * c1;
	  u1 = z;
	}

	c2 = -sqrt( (1.0 - c1) / 2.0 );
	c1 =  sqrt( (1.0 + c1) / 2.0 );
  }

} /* FFT() */

/*----------------------------------------------------------------------*/

/* FFT_SDR()
 *
 * FFT function. It covers positive and negative
 * frequencies either side of the bandwidth center.
 */
  void
FFT_SDR( int fft_length )
{
  int
	op_bin_idx,	 /* Index to FFT decimated ouput bin */
	fft_bin_mid, /* Index to middle of FFT's output bin */
	fft_bin_idx; /* Index to FFT's output bins */

  /* Value of FFT decimated output bins */
  static double op_bin_val = 200;

  /* Maximum value of fft bins */
  static double bin_max;

  /* Range of levels in vertical scale of spectrum display */
  static double range = (double)(LEVEL_STEP * NUM_LEVEL_STEPS);

  double val_max = -1000.0, magn;


  /* Perform FFT on input samples */
  FFT( fft_length, fft_in_i, fft_in_q );

  op_bin_idx  = 0;
  fft_bin_mid = fft_length / 2;

  /* Calculate output FFT bins for "positive" frequencies */
  for( fft_bin_idx = fft_bin_mid; fft_bin_idx < fft_length; fft_bin_idx++ )
  {
	/* Stop if reaching the end of o/p bins array */
	if( op_bin_idx >= FFT_INPUT_LENGTH ) break;

	/* Calculate signal power at each frequency (output bin) */
	fft_in_i[fft_bin_idx] /= rc_data.fft_scale;
	fft_in_q[fft_bin_idx] /= rc_data.fft_scale;
	magn = hypot( fft_in_i[fft_bin_idx], fft_in_q[fft_bin_idx] );

	/* Keep previous value if magnitude is zero */
	if( magn ) op_bin_val = 20.0 * log10( magn );

	/* Record maximum bin value */
	if( val_max < op_bin_val ) val_max = op_bin_val;

	/* Offset bin values to below max bin value (0 dBm relative) */
	op_bin_val -= bin_max;

	/* Maintain bin values in range 0 to -range (100) dBm */
	if( op_bin_val > 0.0 )
	  op_bin_val = 0.0;
	else if( op_bin_val < -range )
	  op_bin_val = -range;

	/* Convert bin values to height in spectrum display */
	fft_bin_val[op_bin_idx] =
	  (int)( (range + op_bin_val) / range * (double)signal_height );

	op_bin_idx++;

  } /* for( fft_bin_idx = fft_bin_mid; fft_bin_idx < fft_length; fft_bin_idx++ ) */

  /* Calculate output FFT bins for "negative" frequencies */
  for( fft_bin_idx = 0; fft_bin_idx < fft_bin_mid; fft_bin_idx++ )
  {
	/* Stop if reaching the end of o/p bins array */
	if( op_bin_idx >= FFT_INPUT_LENGTH ) break;

	/* Calculate signal power at each frequency (output bin) */
	fft_in_i[fft_bin_idx] /= rc_data.fft_scale;
	fft_in_q[fft_bin_idx] /= rc_data.fft_scale;
	magn = hypot( fft_in_i[fft_bin_idx], fft_in_q[fft_bin_idx] );

	/* Keep previous value if magnitude is zero */
	if( magn ) op_bin_val = 20.0 * log10( magn );

	/* Record maximum bin value */
	if( val_max < op_bin_val ) val_max = op_bin_val;

	/* Offset bin values to below max bin value (0 dBm relative) */
	op_bin_val -= bin_max;

	/* Maintain bin values in range 0 to -range (100) dBm */
	if( op_bin_val > 0.0 )
	  op_bin_val = 0.0;
	else if( op_bin_val < -range )
	  op_bin_val = -range;

	/* Convert bin values to height in spectrum display */
	fft_bin_val[op_bin_idx] =
	  (int)( (range + op_bin_val) / range * (double)signal_height );

	op_bin_idx++;

  } /* for( fft_bin_idx = 0; fft_bin_idx < fft_bin_mid; fft_bin_idx++ ) */

  /* Reset bin_max variable */
  bin_max = val_max;
  val_max = -1000.0;

} /* FFT_SDR() */

/*----------------------------------------------------------------------*/

/* Display_Spectrum()
 *
 * Displays FFT Spectrum as "waterfall"
 */
  void
Display_Spectrum( void )
{
  int
	vert_lim,  /* Limit of vertical index for copying lines */
	idh, idv,  /* Index to hor. and vert. position in spectrum display */
	pixel_val, /* Greyscale value of pixel derived from fft o/p */
	fft_idx;   /* Index to fft output array */

  /* Pointer to current pixel */
  static guchar *pix;


  /* If not initialized */
  if( !signal_height ) return;

  /* Copy each line of spectrum to next one */
  vert_lim = rc_data.spectrum_height - 2;
  for( idv = vert_lim; idv > 0; idv-- )
  {
	pix =
	  spectrum_pixels +
	  rc_data.spectrum_rowstride * idv +
	  rc_data.spectrum_n_channels;

	for( idh = 0; idh < FFT_INPUT_LENGTH; idh++ )
	{
	  pix[0] = pix[ -rc_data.spectrum_rowstride];
	  pix[1] = pix[1-rc_data.spectrum_rowstride];
	  pix[2] = pix[2-rc_data.spectrum_rowstride];
	  pix += rc_data.spectrum_n_channels;
	}
  } /* for( idv = vert_lim; idv > 0; idv-- ) */

  /* Go to top left +1 hor. +1 vert. of pixbuf */
  pix =
	spectrum_pixels +
	rc_data.spectrum_rowstride +
	rc_data.spectrum_n_channels;

  /* Do the FFT on input array */
  FFT_SDR( FFT_INPUT_LENGTH );

  /* Calculate pixel values after FFT */
  for( fft_idx = 0; fft_idx < FFT_INPUT_LENGTH; fft_idx++ )
  {
	int n;

	/* Color code signal strength (rescale to max = 255 ) */
	pixel_val = ( fft_bin_val[fft_idx] * 255 ) / signal_height;

	if( pixel_val < 85 )
	{
	  pix[0] = 0;
	  pix[1] = 0;
	  pix[2] = 3 + (guchar)pixel_val * 3;
	}
	else if( pixel_val < 170 )
	{
	  n = pixel_val - 85;
	  pix[0] = 0;
	  pix[1] = 3   + (guchar)n * 3;
	  pix[2] = 255 - (guchar)n * 3;
	}
	else
	{
	  pix[0] = (guchar)pixel_val;
	  pix[1] = 255;
	  pix[2] = 0;
	}

	pix += rc_data.spectrum_n_channels;

  } /* for( fft_idx = 0; fft_idx < FFT_INPUT_LENGTH; fft_idx++ ) */

  /* At last draw spectrum */
  gtk_widget_queue_draw( spectrum_display );

} /* Display_Spectrum() */

/*----------------------------------------------------------------------*/

/* Fit_to_Int_Scale()
 *
 * Finds a new range of values between upper_val and lower_val
 * so that the value of steps between scale points is a
 * multiple of 1, 2, 5 or 10, for easier interpolation
 */
  static void
Fit_to_Int_Scale(
	double *upper_val,
	double *lower_val,
	double *range,
	double *scale_step,
	int    *num_points )
{
  int order;

  /* Initial range of values between max and min */
  *range = *upper_val - *lower_val;
  if( *range < 10.0 ) return;

  /* Initial values of step between scale points */
  *scale_step = *range / (double)*num_points;

  /* Order (log 10) of this step */
  order = (int)log10( *scale_step );

  /* Get this step in the range 1.0 - 10.0 */
  *scale_step /= pow( 10.0, (double)order );

  /* Chose a nearby step value that is easier to interpolate
   * between scale points, e.g. scale steps of 1, 2, 5, 10 */
       if( *scale_step < 1.0 )	*scale_step = 1.0;
  else if( *scale_step < 2.0 )	*scale_step = 2.0;
  else if( *scale_step < 5.0 )	*scale_step = 5.0;
  else if( *scale_step < 10.0 )	*scale_step = 10.0;

  /* Raise this new step to its original order */
  *scale_step *= pow( 10.0, (double)order );

  /* Find a new number of steps to match new step value.
   * The new range must be smaller than the original */
  *num_points = (int)( *range * 0.95 / *scale_step );

  /* Calculate new range to be multiple of scale intervals */
  *range = *scale_step * (double)( *num_points - 1 );

} /* Fit_to_Int_Scale() */

/*-----------------------------------------------------------------------*/

/* Plot_Frequency_Scale()
 *
 * Draws out a horizontal scale, between the min
 * and max value of the variable to be plotted
 */
  static void
Plot_Frequency_Scale(
	cairo_t *cr,
	double  x,
	double  y,
	double *x_offset,
	double *range,
	int    *num_points )
{
  int idx;
  char value[16];
  PangoLayout *layout;

  static int
	pl_width,  /* Pango Layout Width  */
	pl_height; /* Pango Layout Height */

  /* Lower and Upper value of scale and scale step */
  static double lower_val, upper_val, scale_step;

  double
	khz_per_pixel,	/* Resolution of Signal Display in kHz/pixel */
	offset,			/* Offset in kHz between center freq and nearest scale point */
	fft_bw,			/* FFT bandwidth in kHz */
	fft_bw2,		/* Half the FFT bandwidth in kHz */
	center_freq,	/* Receiver center frequency in kHz */
	origin,			/* The frequency of first scale point of the freq scale */
	stride,			/* The distance between signal display scale points in kHz */
	margin;			/* Margin in kHz between first scale point and lowest freq */

  /* Center frequency in kHz */
  center_freq = (double)rc_data.sdrx_center_freq / 1000.0;

  /* Number of values to use in the frequency scale is
   * calculated according to the order of the frequency */
  *num_points = MAX_FREQ_SCALE_STEPS - (int)log10( center_freq );

  /* FFT bandwidth in kHz */
  fft_bw  = rc_data.fft_bandwidth / 1000.0;

  /* Half the FFT bandwidth in kHz */
  fft_bw2 = fft_bw / 2.0;

  /* This is the range of values
   * required for the frequency scale */
  lower_val = center_freq - fft_bw2;
  upper_val = center_freq + fft_bw2;
  *range    = upper_val - lower_val;

  /* Adjust the range and number of values in
   * frequency scale to make a legible display */
  Fit_to_Int_Scale( &upper_val, &lower_val, range, &scale_step, num_points );

  /* Abort if not enough values to plot */
  if( *num_points < 3 ) return;

  /* Frequency step per pixel of monitor (kHz per pixel) */
  khz_per_pixel = fft_bw / (double)FFT_INPUT_LENGTH;

  /* Margin in kHz between FFT B/W and Scale range, either side */
  margin = ( fft_bw - *range ) / 2.0;

  /* Frequency in kHz of first point in frequency scale */
  origin = lower_val + margin;

  /* Round origin to multiple of scale steps */
  idx    = (int)( origin / scale_step + 0.5 );
  origin = (double)idx * scale_step;

  /* Offset between Rx center freq and the center of scale */
  offset = margin + origin + *range / 2.0 - center_freq;

  /* Offset in pixels for the position of the frequency
   * scale. It makes it possible to keep the Rx center
   * frequency in the middle of the signal display */
  *x_offset = offset / khz_per_pixel;

  /* Create a pango layout and get sizes */
  snprintf( value, sizeof(value), "%1.0f",
	  (double)rc_data.sdrx_center_freq / 1000.0 );
  layout = gtk_widget_create_pango_layout( signal_display, value );
  pango_layout_get_pixel_size( layout, &pl_width, &pl_height );

  /* The clear height of the signal signal display above the frequency scale.
   * It is rounded to a multiple of the Level scales number of steps to make
   * the stride of the Level scale values and scale lines an integer number. */
  signal_height = ( rc_data.signal_height - pl_height ) / NUM_LEVEL_STEPS;
  signal_height *= NUM_LEVEL_STEPS;

  /* Clear the Frequency scale area */
  cairo_set_source_rgb( cr, SIGNAL_BACKGND );
  cairo_rectangle( cr,
	  0.0, (double)signal_height,
	  (double)FFT_INPUT_LENGTH,
	  (double)(rc_data.signal_height - signal_height) );
  cairo_fill( cr );

  /* Align with center of Pango Layout rectangle */
  x  = *x_offset;
  x -= pl_width / 2;
  y -= pl_height;

  /* Distance between printed scale values */
  stride  = (double)FFT_INPUT_LENGTH * *range / fft_bw;
  stride /= (double)(*num_points - 1);

  /* Print horizontal scale values using Pango */
  cairo_set_source_rgb( cr, SIGNAL_SCALES );
  for( idx = 0; idx < *num_points; idx++ )
  {
	snprintf( value, sizeof(value), "%1.0f", origin );
	pango_layout_set_text( layout, value, -1 );
	cairo_move_to( cr, x, y );
	pango_cairo_show_layout( cr, layout );
	origin += scale_step;
	x += stride;
  }

  g_object_unref( layout );
  ClearFlag( PLOT_SCALES );

} /* Plot_Frequency_Scale() */

/*-----------------------------------------------------------------------*/

/* Plot_Level_Scale()
 *
 * Plots a vertical scale to the left of the Signal display.
 * Normally this would be in dBm or dB relative as needed.
 */
  static void
Plot_Level_Scale( cairo_t *cr )
{
  int idx, width, height;
  double yps, stride, origin;
  char value[16];
  PangoLayout *layout;


  /* Print vertical scale values using Pango */
  yps	 = 1.0;
  stride = (double)signal_height / (double)NUM_LEVEL_STEPS;

  /* Clear the scale text's rectangle */
  layout = gtk_widget_create_pango_layout( signal_display, "-150 " );
  pango_layout_get_pixel_size( layout, &width, &height );
  cairo_set_antialias(  cr, CAIRO_ANTIALIAS_NONE );
  cairo_set_source_rgb( cr, SIGNAL_BACKGND );
  cairo_rectangle( cr,
	  0.0, 0.0, (double)width, (double)height );
  cairo_fill( cr );

  /* Print start of scale below top reticle line */
  cairo_set_source_rgb( cr, SIGNAL_SCALES );
  pango_layout_set_text( layout, " 0", -1 );
  cairo_move_to( cr, 0.0, yps );
  pango_cairo_show_layout( cr, layout );

  /* Print rest of scale values above reticle lines */
  yps    = stride - height;
  origin = 0.0;
  for( idx = 0; idx < NUM_LEVEL_STEPS; idx++ )
  {
	/* Clear the scale text's rectangle */
	cairo_set_source_rgb( cr, SIGNAL_BACKGND );
	cairo_rectangle( cr,
		0.0, yps, (double)width, (double)height );
	cairo_fill( cr );

	/* Print scale values */
	origin -= LEVEL_STEP;
	snprintf( value, sizeof(value), "%1.0f", origin );
	pango_layout_set_text( layout, value, -1 );
	cairo_set_source_rgb( cr, SIGNAL_SCALES );
	cairo_move_to( cr, 0.0, yps );
	pango_cairo_show_layout( cr, layout );
	yps += stride;
  }

  /* Draw horizontal divisions */
  cairo_set_line_width( cr, 1.0 );
  cairo_set_source_rgb( cr, SIGNAL_RETICLE );
  yps = 1.0;
  for( idx = 0; idx <= NUM_LEVEL_STEPS; idx++ )
  {
	cairo_move_to( cr, 0.0, yps );
	cairo_line_to( cr, (double)rc_data.monitor_width, yps );
	yps += stride;
  }

  /* Stroke paths */
  cairo_stroke( cr );

  g_object_unref( layout );

} /* Plot_Level_Scale() */

/*-----------------------------------------------------------------------*/

/*  Display_Signal()
 *
 *  Displays amplitude of the FFT Spectrum bins
 */

  void
Display_Signal( void )
{
  /* Points to plot */
  static GdkPoint *points = NULL;
  static int points_idx = 0, num_points;
  static cairo_t *cr = NULL;
  static double range = 10.0, x_offset;

  /* X co-ordinate and width of the Demodulator
   * bandwidth rectangle in the signal signal display */
  double
	dembw_x,	 /* X co-ordinate of Demodulator bandwidth rectangle */
	dembw_width, /* Width of Demodulator bandwidth rectangle */
	bkgnd_width, /* Width of signal background areas, outside above */
	xps, yps,	 /* X and Y co-ordinates for Cairo graphics  */
	stride;		 /* Distance in pixels between vertical scale lines */

  /* Index to fft output array */
  int fft_idx;

  int idx;


  /* Initialize on first call */
  if( points == NULL )
	  mem_alloc( (void *)&points,
		  (size_t)FFT_INPUT_LENGTH * sizeof(GdkPoint) );

  /* Re-initialize on signal height change */
  if( isFlagSet(MONITORS_INIT) )
  {
	cairo_destroy( cr );
	cr = NULL;
	ClearFlag( MONITORS_INIT );
  }

  /* Initialize cairo when needed */
  if( cr == NULL )
  {
	cr = gdk_cairo_create( signal_pixmap );
	cairo_set_line_join( cr, CAIRO_LINE_JOIN_ROUND );
	cairo_set_line_cap(  cr, CAIRO_LINE_CAP_ROUND );
  }

  /* Plot the frequency scale under the signal signal  */
  if( isFlagSet(PLOT_SCALES) )
  {
	Plot_Frequency_Scale(
		cr, 0.0, (double)rc_data.signal_height,
		&x_offset, &range, &num_points );
  }

  /* Save values to be plotted (scaled to fit display) */
  points_idx = 0;
  for( fft_idx = 0; fft_idx < FFT_INPUT_LENGTH; fft_idx++ )
  {
	points[points_idx].y = signal_height - fft_bin_val[fft_idx];
	if( points[points_idx].y < 0 )
	  points[points_idx].y = 0;
	points[points_idx].x = points_idx;
	points_idx++;
  } /* for( fft_idx = 0; fft_idx < FFT_INPUT_LENGTH; fft_idx++ ) */

  /* Draw demodulator bandwidth background */
  dembw_width  = (double)FFT_INPUT_LENGTH;
  dembw_width *= (double)rc_data.demod_bandwidth / rc_data.fft_bandwidth;
  dembw_x  = ( (double)FFT_INPUT_LENGTH - dembw_width ) / 2.0;
  cairo_set_source_rgb( cr, SIGNAL_DEMODBW );
  cairo_rectangle( cr,
	  dembw_x, 0.0,
	  dembw_width,
	  (double)signal_height );
  cairo_fill( cr );

  /* Draw signal background */
  bkgnd_width = ( (double)FFT_INPUT_LENGTH - dembw_width ) / 2.0;
  cairo_set_source_rgb( cr, SIGNAL_BACKGND );
  cairo_rectangle( cr,
	  0.0, 0.0,
	  bkgnd_width + 1.0,
	  (double)signal_height );
  cairo_fill( cr );
  cairo_rectangle( cr,
	  dembw_x + dembw_width, 0.0,
	  bkgnd_width, (double)signal_height );
  cairo_fill( cr );

  /* Plot signal graph */
  cairo_set_line_width( cr, 1.5 );
  cairo_set_antialias( cr, CAIRO_ANTIALIAS_BEST );
  cairo_set_source_rgb( cr, SIGNAL_TRACES );
  cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
  for( idx = 1; idx < points_idx; idx++ )
	cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );

  /* Stroke paths */
  cairo_stroke( cr );

  /* Draw vertical divisions */
  cairo_set_line_width( cr, 1.0 );
  cairo_set_antialias(  cr, CAIRO_ANTIALIAS_NONE );
  cairo_set_source_rgb( cr, SIGNAL_RETICLE );

  /* Distance between vertical reticle lines, in pixels */
  stride  = (double)FFT_INPUT_LENGTH * range * 1000.0;
  stride /= rc_data.fft_bandwidth * (double)(num_points - 1);

  /* Starting X position and Y position
   * of vertical reticle lines, in pixels */
  xps = x_offset;
  yps = (double)signal_height;

  /* Draw vertical reticle lines */
  for( idx = 0; idx < num_points; idx++ )
  {
	cairo_move_to( cr, xps, 0.0 );
	cairo_line_to( cr, xps, yps );
	xps += stride;
  }

  /* Stroke paths */
  cairo_stroke( cr );

  /* Draw horizontal divisions */
  Plot_Level_Scale( cr );

  /* Draw vertical centerline */
  cairo_set_source_rgb( cr, SIGNAL_CENTERLINE );
  xps = (double)FFT_INPUT_LENGTH / 2.0;
  yps = (double)signal_height;
  cairo_move_to( cr, xps, 0.0 );
  cairo_line_to( cr, xps, yps );

  /* Stroke paths */
  cairo_stroke( cr );

  /* Render pixmap to screen */
  gdk_window_set_back_pixmap(
	  signal_display->window, signal_pixmap, FALSE );
  gdk_window_clear( signal_display->window );

  /* Wait for GTK to complete its tasks */
  while( g_main_context_iteration(NULL, FALSE) );

} /* Display_Signal */

/*----------------------------------------------------------------------*/

/* Tune_to_Monitor()
 *
 * Tunes the Sdrx tuner to the frequency of the strongest
 * signal near a mouse click in the spectrum or signal window
 */

  void
Tune_to_Monitor( double x )
{
  int
	from, to,	/* Range to scan for a max bin value */
	bin_max,	/* Max bin value found in this range */
	max_idx,	/* fft idx at which the max is found */
	fft_idx,	/* Idx used to search fft bin values */
	center_freq = (int)rc_data.sdrx_center_freq;
	
  /* Frequency at click position in monitor */
  double click_freq;


  /* Abort ifSdrx not ready or FFT not running */
  if( isFlagClear(SDRX_INITIALIZED) ||
	  isFlagClear(FFT_INITIALIZED) )
	return;

  /* Calculate fft index corresponding to pointer x in spectrum window */
  fft_idx = (int)x;

  /* Look above and below click point for max bin val */
  from = fft_idx - CLICK_TUNE_RANGE;
  if( from < 0 ) from = 0;
  to = fft_idx + CLICK_TUNE_RANGE;
  if( to >= FFT_INPUT_LENGTH )
	to = FFT_INPUT_LENGTH - 1;

  /* Find max bin value around click point */
  bin_max = 0; max_idx = fft_idx;
  for( fft_idx = from; fft_idx < to; fft_idx++ )
	if( bin_max < fft_bin_val[fft_idx] )
	{
	  bin_max = fft_bin_val[fft_idx];
	  max_idx = fft_idx;
	}

  /* Offset frequency corresponding to fft index */
  click_freq  = (double)max_idx - (double)FFT_INPUT_LENGTH / 2.0 + 0.5;
  click_freq *= rc_data.fft_bandwidth / (double)(FFT_INPUT_LENGTH - 1);

  /* Set Tuner Frequency */
  center_freq += click_freq;
  if( center_freq < 0 ) center_freq = 0;
  rc_data.sdrx_center_freq = (uint32_t)center_freq;

  /* Offset center freq for SSB operation to compensate
   * later offset of Weaver phasing frequency offset */
  rc_data.sdrx_center_freq += (uint32_t)rc_data.weaver_offset;
  Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );

  return;
} /* Tune_to_Monitor() */

/*----------------------------------------------------------------------*/

/* Sdrx_Frequency_Status()
 *
 * Displays the status of center frequency setting
 */
  void
Sdrx_Frequency_Status( gboolean err )
{
  /* Indicate Sdrx Frequency status */
  if( err )
  {
	GtkWidget *image = lookup_widget( sdrx_main_window, "status_image" );
	gtk_image_set_from_icon_name(
		GTK_IMAGE(image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	GtkWidget *label = lookup_widget( sdrx_main_window, "status_label" );
	gtk_label_set_text( GTK_LABEL(label), "ERROR" );
  }
  else
  {
	GtkWidget *image = lookup_widget( sdrx_main_window, "status_image" );
	gtk_image_set_from_icon_name(
		GTK_IMAGE(image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
	GtkWidget *label = lookup_widget(
		sdrx_main_window, "status_label" );
	gtk_label_set_text( GTK_LABEL(label), "O.K." );
  }

} /* Sdrx_Frequency_Status() */

/*----------------------------------------------------------------------*/

