/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "shared.h"

/*----------------------------------------------------------------------*/

/*  Error_Dialog()
 *
 *  Opens an error dialog box
 */
  void
Error_Dialog( gchar *mesg, gboolean hide_ok )
{
  if( error_dialog == NULL )
	error_dialog = create_error_dialog();
  GtkLabel *label = GTK_LABEL( lookup_widget(error_dialog, "error_label") );
  gtk_label_set_text( label, mesg );

  /* For non-recoverable errors, stop sdrx and force quit */
  if( hide_ok )
  {
	Sdrx_Stop();
	GtkWidget *button = lookup_widget( error_dialog, "error_ok_button" );
	gtk_widget_hide( button );
  }

  gtk_widget_show( error_dialog );

} /*  Error_Dialog() */

/*----------------------------------------------------------------------*/

/* Display_Center_Frequency()
 *
 * Enters the center frequency to relevant entry widget
 */
  void
Display_Center_Frequency( uint32_t freq )
{
  gchar text[16];

  /* Set center freq value in kHz, freq value is in Hz */
  snprintf( text, sizeof(text), "%10.1f", (double)freq / 1000.0 );
  gtk_entry_set_text( GTK_ENTRY(center_freq_entry), text );
  Update_Spin_Dial( freq );

} /* Display_Center_Frequency() */

/*----------------------------------------------------------------------*/

/* Display_Frequency_Spin_Dial()
 *
 * Sets the Tuner frequency and displays it in the "spin dial"
 */
  void
Display_Frequency_Spin_Dial( GtkLabel *label, int data )
{
  const gchar *txt;
  int value, idx, freq, mult;
  gchar lbl[4], new_txt[31];
  GtkLabel *labels;


  /* Set new value to label */
  snprintf( new_txt, sizeof(new_txt),
	  "<span size=\"x-large\">%1d</span>", data );
  gtk_label_set_markup( label, new_txt );

  /* Read frequency from labels */
  mult = 1; freq = 0;
  for( idx = 1; idx <= 10; idx++ )
  {
	snprintf( lbl, sizeof(lbl), "f%d", idx );
	labels = GTK_LABEL( lookup_widget(sdrx_main_window, lbl) );
	txt = gtk_label_get_text( labels );
	value = atoi( txt );
	freq += (uint32_t)value * mult;
	mult *= 10;
  }

  /* Set tuner center frequency */
  if( freq > 0 )
  {
	rc_data.sdrx_center_freq = freq;
	Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );
  }
  else
  {
	/* Zero label if data goes negative */
	snprintf( new_txt, sizeof(new_txt),
		"<span size=\"x-large\">%d</span>", 0 );
	gtk_label_set_markup( label, new_txt );
  }

} /* Display_Frequency_Spin_Dial() */

/*----------------------------------------------------------------------*/

/* Update_Spin_Dial()
 *
 * Updates the Frequency spin dial with latest center frequency
 */
  void
Update_Spin_Dial( uint32_t freq )
{
  uint32_t mult;
  int value, idx;
  GtkLabel *labels;
  gchar lbl[4], new_txt[31];


  /* Enter new frequency to the Frequency "spin dial" */
  mult = 1000000000;
  for( idx = 10; idx > 0; idx-- )
  {
	value = freq / mult;
	snprintf( new_txt, sizeof(new_txt),
		"<span size=\"x-large\">%1d</span>", value );
	snprintf( lbl, sizeof(lbl), "f%d", idx );
	labels = GTK_LABEL( lookup_widget(sdrx_main_window, lbl) );
	gtk_label_set_markup( labels, new_txt );
	freq -= value * mult;
	mult /= 10;
  }

} /* Update_Spin_Dial() */

/*----------------------------------------------------------------------*/

/* Combobox_Append_Text()
 *
 * Appends text into the mode, fft bandwidth, sample rate,
 * data buffer, demodulator bandwidth and weaver comboboxes
 */
  void
Combobox_Append_Text( void )
{
  gchar *demod_bw[]      = { DEMOD_BAND_WIDTHS };
  gchar *sdrx_buf[]      = { SDRX_BUF_SIZE };
  gchar *rtlsdr_srate[]  = { RTLSDR_SAMPLE_RATES };
  gchar *sdrplay_srate[] = { SDRPLAY_SAMPLE_RATES };
  gchar *weaver_freqs[]  = { WEAVER_FREQUENCIES };
  int idx;

  /* Append Demodulator bandwidths */
  for( idx = 0; idx < DEMOD_BAND_WIDTHS_NUM; idx++ )
	gtk_combo_box_append_text(
		GTK_COMBO_BOX(demod_bw_combobox), demod_bw[idx] );

  /* Append sample rates */
  if( rc_data.dongle_type == DONGLE_TYPE_SDRPLAY )
	for( idx = 0; idx < SDRPLAY_SAMPLE_RATES_NUM; idx++ )
	  gtk_combo_box_append_text(
		  GTK_COMBO_BOX(smp_rate_combobox), sdrplay_srate[idx] );
  else if( rc_data.dongle_type == DONGLE_TYPE_RTLSDR )
	for( idx = 0; idx < RTLSDR_SAMPLE_RATES_NUM; idx++ )
	  gtk_combo_box_append_text(
		  GTK_COMBO_BOX(smp_rate_combobox), rtlsdr_srate[idx] );
  else if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS )
  {
	int buf[PERSEUS_RATES_BUF_SIZE], idx;
	char txt[MESG_SIZE];

	/* Get available sampling rates */
	if( perseus_get_sampling_rates( NULL, buf, PERSEUS_RATES_BUF_SIZE ) < 0 )
	{
	  snprintf( txt, sizeof(txt),
		  _("Perseus get_sampling_rates error:\n%s"), perseus_errorstr() );
	  Error_Dialog( txt, HIDE_OK );
	  return;
	}

	/* Append rates till last in list (buf[idx] = 0) */
	idx = 0;
	while( buf[idx] )
	{
	  snprintf( txt, sizeof(txt), "%d", buf[idx] / 1000 );
	  gtk_combo_box_append_text(
		  GTK_COMBO_BOX(smp_rate_combobox), txt );
	  idx++;
	}
  } /* if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS ) */

  /* Append buffer sizes */
  for( idx = 0; idx < SDRX_BUF_SIZE_NUM; idx++ )
	gtk_combo_box_append_text(
		GTK_COMBO_BOX(sdrx_buf_combobox), sdrx_buf[idx] );

  /* Append Weaver frequencies */
  for( idx = 0; idx < WEAVER_FREQUENCIES_NUM; idx++ )
	gtk_combo_box_append_text(
		GTK_COMBO_BOX(weaver_combobox), weaver_freqs[idx] );

} /* Combobox_Append_Text() */

/*----------------------------------------------------------------------*/

/* Sdrx_Display_S_meter()
 *
 * Displays the S-meter bar and signal strength
 */
  void
Sdrx_Display_S_meter( double smeter )
{
  if( smeter < 0.0 ) rc_data.S_meter = 0.0;
  if( smeter > 1.0 ) rc_data.S_meter = 1.0;

  gtk_progress_bar_set_fraction(
	  GTK_PROGRESS_BAR(smeter_progbar), (gdouble)rc_data.S_meter );

} /* Sdrx_Display_S_meter() */

/*----------------------------------------------------------------------*/

static gboolean init = TRUE;

/* Sdrx_Display_Time()
 *
 * Displays Time and Data as decoded from (VLF) time stations
 */
  void
Sdrx_Display_Time( time_data_t time_data )
{
  /* Time Display Window's widgets */
  static GtkWidget
	*time_label	 = NULL,
	*date_label	 = NULL,
	*cest_image	 = NULL,
	*cet_image 	 = NULL,
	*stwarn_image   = NULL,
	*steffect_image = NULL,
	*txstat_image   = NULL,
	*sumtim_image   = NULL,
	*leapsec_image  = NULL,
	*hour_parity_image   = NULL,
	*minute_parity_image = NULL,
	*date_parity_image   = NULL,
	*year_parity_image   = NULL,
	*month_parity_image  = NULL,
	*mday_parity_image   = NULL,
	*dut_parity_image    = NULL,
	*tjd_parity_image    = NULL,
	*dut_label		= NULL,
	*rbu_dut1_label = NULL,
	*msf_dut1_label = NULL,
	*tjd_label		= NULL;


  /* Initialize on first call */
  if( init )
  {
	time_label = lookup_widget( time_window, "time_label" );
	date_label = lookup_widget( time_window, "date_label" );
	dut_label  = lookup_widget( time_window, "dut_label" );
	rbu_dut1_label = lookup_widget( time_window, "rbu_dut1_label" );
	msf_dut1_label = lookup_widget( time_window, "msf_dut1_label" );
	tjd_label  = lookup_widget( time_window, "tjd_label" );
	cest_image = lookup_widget( time_window, "cest_image" );
	cet_image  = lookup_widget( time_window, "cet_image" );
	txstat_image   = lookup_widget( time_window, "txstat_image" );
	sumtim_image   = lookup_widget( time_window, "sumtim_image" );
	stwarn_image   = lookup_widget( time_window, "stwarn_image" );
	steffect_image = lookup_widget( time_window, "steffect_image" );
	leapsec_image  = lookup_widget( time_window, "leapsec_image" );
	hour_parity_image	= lookup_widget( time_window, "hour_parity_image" );
	minute_parity_image	= lookup_widget( time_window, "minute_parity_image" );
	date_parity_image	= lookup_widget( time_window, "date_parity_image" );
	year_parity_image	= lookup_widget( time_window, "year_parity_image" );
	month_parity_image	= lookup_widget( time_window, "month_parity_image" );
	mday_parity_image	= lookup_widget( time_window, "mday_parity_image" );
	dut_parity_image	= lookup_widget( time_window, "dut_parity_image" );
	tjd_parity_image	= lookup_widget( time_window, "tjd_parity_image" );
	date_parity_image	= lookup_widget( time_window, "date_parity_image" );

	init = FALSE;
  } /* if( init ) */

  /* Format and Display Time */
  char time_date[96];
  strftime( time_date, sizeof(time_date),
	  "<span size=\"large\" weight=\"bold\" font=\"mono\">%H:%M:%S</span>",
	  &(time_data.tim) );
  gtk_label_set_markup( GTK_LABEL(time_label), time_date );

  /* These checks and displays done on the minute */
  if( time_data.tim.tm_sec == 0 )
  {
	/* Clear non-common indicators */
	gtk_image_set_from_icon_name(
		GTK_IMAGE(year_parity_image),
		"media-playback-stop", GTK_ICON_SIZE_BUTTON );
	gtk_image_set_from_icon_name(
		GTK_IMAGE(month_parity_image),
		"media-playback-stop", GTK_ICON_SIZE_BUTTON );
	gtk_image_set_from_icon_name(
		GTK_IMAGE(mday_parity_image),
		"media-playback-stop", GTK_ICON_SIZE_BUTTON );

	/* Minutes Parity Error */
	if( time_data.status & MINUTE_ERROR )
	  gtk_image_set_from_icon_name(
		  GTK_IMAGE(minute_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	else
	  gtk_image_set_from_icon_name(
		  GTK_IMAGE(minute_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	/* Hours Parity Error */
	if( time_data.status & HOUR_ERROR )
	  gtk_image_set_from_icon_name(
		  GTK_IMAGE(hour_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	else
	  gtk_image_set_from_icon_name(
		  GTK_IMAGE(hour_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	/* General Date Parity Error */
	if( time_data.status & DATE_ERROR )
	  gtk_image_set_from_icon_name(
		  GTK_IMAGE(date_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	else
	  gtk_image_set_from_icon_name(
		  GTK_IMAGE(date_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );


	/* These are provided by DCF77 and ALL162 */
	if( (time_data.station == DCF77) ||
		(time_data.station == ALL162) )
	{
	  /* If Z1 and Z2 (CEST/CET) are same, its an error condition */
	  if( time_data.status & DCF77_Z1_Z2_ERROR )
	  {
		gtk_image_set_from_icon_name(
			GTK_IMAGE(cet_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
		gtk_image_set_from_icon_name(
			GTK_IMAGE(cest_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  }
	  else /* Display Summer Time status */
	  {
		/* CEST Summer Time */
		if( time_data.status & DCF77_Z1_CEST )
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(cest_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
		else
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(cest_image), "media-playback-stop", GTK_ICON_SIZE_BUTTON );

		/* CET Regular (winter) Time */
		if( time_data.status & DCF77_Z2_CET )
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(cet_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
		else
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(cet_image), "media-playback-stop", GTK_ICON_SIZE_BUTTON );
	  } /* if( time_data.status & Z1_Z2_ERROR ) */

	  /* Transmitter operational status */
	  if( time_data.status & DCF77_TX_ABNL )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(txstat_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(txstat_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* Summer Time Imminent Announcement Flag */
	  if( time_data.status & DCF77_A1_CEST )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(sumtim_image), "dialog-warning", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(sumtim_image), "media-playback-stop", GTK_ICON_SIZE_BUTTON );

	  /* Leap Second Imminent Announcement Flag */
	  if( time_data.status & DCF77_A2_LEAP_SEC )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(leapsec_image), "dialog-warning", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(leapsec_image), "media-playback-stop", GTK_ICON_SIZE_BUTTON );
	} /* if( (time_data.station == DCF77) || ... */

	/* These are provided by RBU66 */
	if( time_data.station == RBU66 )
	{
	  /* Year parity error */
	  if( time_data.status & YEAR_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(year_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(year_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* Month parity error */
	  if( time_data.status & MONTH_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(month_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(month_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* Day of Month parity error */
	  if( time_data.status & MDAY_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(mday_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(mday_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* DeltaUT parity error */
	  if( time_data.status & RBU66_DUT_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(dut_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(dut_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* Tr. Julian Day parity error */
	  if( time_data.status & RBU66_TJD_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(tjd_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(tjd_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* Display RBU66 DUT1, TJD and DeltaUT data */
	  snprintf( time_date, sizeof(time_date), "DUT1: %+05.2f", time_data.DUT1 );
	  gtk_label_set_text( GTK_LABEL(rbu_dut1_label), time_date );
	  snprintf( time_date, sizeof(time_date), "T. Julian Day: %04d", time_data.tjd );
	  gtk_label_set_text( GTK_LABEL(tjd_label), time_date );
	  snprintf( time_date, sizeof(time_date), "DUT: %+02d", time_data.DUT );
	  gtk_label_set_text( GTK_LABEL(dut_label), time_date );

	} /* if( time_data.station == RBU66 ) */

	/* These are provided by MSF60 */
	if( time_data.station == MSF60 )
	{
	  /* Year parity error */
	  if( time_data.status & YEAR_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(year_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(year_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* Month & Day of Month parity error */
	  if( time_data.status & MDAY_ERROR )
		gtk_image_set_from_icon_name(
			GTK_IMAGE(mday_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  else
		gtk_image_set_from_icon_name(
			GTK_IMAGE(mday_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

	  /* If S.T. Warn and Effective are same, we have an error condition */
	  if( time_data.status & MSF60_SMTM_ERROR )
	  {
		gtk_image_set_from_icon_name(
			GTK_IMAGE(stwarn_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
		gtk_image_set_from_icon_name(
			GTK_IMAGE(steffect_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
	  }
	  else
	  {
		/* Summer Time Imminent warning */
		if( time_data.status & MSF60_SMTM_WARN )
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(stwarn_image), "dialog-warning", GTK_ICON_SIZE_BUTTON );
		else
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(stwarn_image), "media-playback-stop", GTK_ICON_SIZE_BUTTON );

		/* Summer Time Effective flag */
		if( time_data.status & MSF60_SMTM_EFFC )
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(steffect_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
		else
		  gtk_image_set_from_icon_name(
			  GTK_IMAGE(steffect_image), "media-playback-stop", GTK_ICON_SIZE_BUTTON );
	  } /* if( time_data.status & MSF_SMTM_ERR ) */

	  /* Display MSF60 DUT1 */
	  snprintf( time_date, sizeof(time_date), "DUT1: %+05.2f", time_data.DUT1 );
	  gtk_label_set_text( GTK_LABEL(msf_dut1_label), time_date );

	} /* if( time_data.station == MSF60 ) */

	/* Format and display date */
	strftime( time_date, sizeof(time_date),
		"<span size=\"large\" weight=\"bold\" font=\"mono\">%A %B %d %Y</span>",
		&(time_data.tim) );
	gtk_label_set_markup( GTK_LABEL(date_label), time_date );

  } /* if( tim.tm_sec == 0 ) */

} /* Sdrx_Display_Time() */

/*----------------------------------------------------------------------*/

/* Close_Time_Display()
 *
 * Frees time display widgets and resets time display
 */
  void
Close_Time_Display( void )
{
  ClearFlag( TIME_RECV_START );
  Free_Time_Buffers();
  if( isFlagClear(PLAYBACK_START) && isFlagClear(MONITORS_START) )
	Sdrx_Close_Device();
  init = TRUE;

} /* Close_Time_Display() */

/*----------------------------------------------------------------------*/

/* Sdrx_Display_Mode()
 *
 * Shows the demodulator mode on the mode menu button
 */
  void
Sdrx_Display_Mode( int mode )
{
  gchar *modes[MODULATION_MODES_NUM] = { MODULATION_MODES };
  GtkButton *menu_button = GTK_BUTTON(
	  lookup_widget(sdrx_main_window, "menu_button") );
  gtk_button_set_label( menu_button, modes[mode] );

} /* Sdrx_Display_Mode() */

/*----------------------------------------------------------------------*/

