/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "demodulate.h"
#include "shared.h"

/*----------------------------------------------------------------------*/

/* Calculate_Frequency_Offset()
 *
 * Calculates the frequency offset between
 * signal carrier and SDR center frequency
 */
  static inline double
Calculate_Frequency_Offset( double i, double q )
{
  double freq_offset, phi1, dphi;
  static double phi0 = 0.0;


  if( isFlagSet(ENABLE_FREQ_CORR) )
  {
	/* Phase angle of current I/Q sample */
	phi1 = atan2( q, i );

	/* Phase difference from previous to current I/Q sample */
	dphi = phi1 - phi0;

	/* Save current phase angle as the next previous */
	phi0 = phi1;

	/* Correct for discontinuity at +-180 deg phase angle */
	if( dphi > M_PI_2 ) dphi -= M_2PI;
	else if( dphi < -M_PI_2 ) dphi += M_2PI;

	/* Measured signal frequency */
	freq_offset = (double)SND_DSP_RATE * dphi / M_2PI;

  } /* if( isFlagSet(ENABLE_FREQ_CORR) ) */
  else freq_offset = 0.0;

  return( freq_offset );
} /* Calculate_Frequency_Offset() */

/*----------------------------------------------------------------------*/

/* Correct_Frequency_Offset()
 *
 * Corrects offset between Signal frequency and
 * Center frequency of tuner, e.e. AFC of sorts
 */
  static void
Correct_Frequency_Offset( double freq_offset, int len, int limit )
{
  /* Current and Previous Tuner center frequency */
  static gboolean clear = FALSE;
  static int count = 0;
  static double offset = 0.0;
  gchar text[8];

  /* Frequency deviation is averaged over 1 sec */
  if( isFlagSet(ENABLE_FREQ_CORR) )
  {
	/* Summate freq offset and count
	 * total length (number of samples) */
	offset += freq_offset;
	count  += len;

	/* Correct center frequency every 1 sec */
	if( count >= limit )
	{
	  /* Average frequency offset */
	  offset /= (double)count;

	  /* Limit max frequency offset correction */
	  if( offset > MAX_FREQUENCY_OFFSET )
		offset = MAX_FREQUENCY_OFFSET;
	  if( offset < -MAX_FREQUENCY_OFFSET )
		offset = -MAX_FREQUENCY_OFFSET;

	  /* Display and Round frequency offset to 0.9
	   * of its value for asymptotic correction */
	  snprintf( text, sizeof(text), "%d", (int)offset );
	  gtk_label_set_text( GTK_LABEL(freq_offset_label), text );
	  offset = round( offset * 0.9 );

	  /* Correct for frequency offset if not 0 */
	  if( offset )
	  {
		int center_freq = (int)rc_data.sdrx_center_freq;
		center_freq += (uint32_t)offset;
		if( center_freq < 0 ) center_freq = 0;
		rc_data.sdrx_center_freq = (uint32_t)center_freq;
		Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );
		Update_Spin_Dial( rc_data.sdrx_center_freq );
	  }

	  clear = TRUE;
	  count = 0;
	}
  } /* if(  isFlagSet(ENABLE_FREQ_CORR) ) */
  else if( clear )
  {
	gtk_label_set_text( GTK_LABEL(freq_offset_label), " - - - " );
	clear = FALSE;
  }

} /* Correct_Frequency_Offset() */

/*----------------------------------------------------------------------*/

/* Demodulated_Signal_Control()
 *
 * Controls the level of the demodulated signal
 * (in Audio AGC mode) and/or saves it in the
 * sound playback buffer
 */
  static inline void
Demodulated_Signal_Control( double demod_signal, int snd_buf_idx )
{
  double signal_ratio;


  /* Scale demodulated signal according to
   * modulation mode and SDR dongle type */
  demod_signal *= rc_data.demodulator_scale;

 /* Apply audio derived AGC if enabled */
  if( isFlagSet(TUNER_GAIN_ADAGC) )
  {
	/* Ratio of demodulated signal level to reference
	 * level. This is about 2/3 of max level the sound
	 * system can handle (+/- 32384 for 16-bit audio) */
	signal_ratio = fabs( demod_signal ) / ADAGC_REF_LEVEL;

	/* This is the AGC "attack" function */
	if( signal_ratio > rc_data.adagc_scale )
	  rc_data.adagc_scale = signal_ratio;
	else /* This the AGC "decay" function */
	  rc_data.adagc_scale *= rc_data.adagc_decay;

  } /* if( isFlagSet(TUNER_GAIN_ADAGC) ) */

  /* Scale demodulated signal as needed */
  demod_signal /= rc_data.adagc_scale;

  /* Fill sound buffer frames */
  playback_buf[demod_buf_num][snd_buf_idx]   = (short)demod_signal;
  playback_buf[demod_buf_num][snd_buf_idx+1] = (short)demod_signal;

} /* Demodulated_Signal_Control() */

/*----------------------------------------------------------------------*/

/* Playback_Control()
 *
 * Controls the rotation of demodulator
 * and sound write playback buffers
 */
  static void
Playback_Control( void )
{
  if( isFlagSet(PLAYBACK_SUSPEND) )
	return;

  /* Keep some distance between demod and sound buffer indices */
  demod_buf_num++;
  int diff = demod_buf_num - sound_buf_num;
  if( (diff >= -rc_data.playback_buffs / 4) &&
	  (diff <=  rc_data.playback_buffs / 4) )
	demod_buf_num = sound_buf_num + rc_data.playback_buffs / 2;

  /* Keep demod buffer index in range */
  if( demod_buf_num >= rc_data.playback_buffs )
	demod_buf_num -= rc_data.playback_buffs;

} /* Playback_Control() */

/*----------------------------------------------------------------------*/

/* I/Q Samples buffers for the LP Filters */
static double *demod_id = NULL, *demod_qd = NULL;

/* Alloc_Demod_Buffers()
 *
 * Allocates memory to demodulator
 * buffers used for local filtering
 */
  static void
Alloc_Demod_Buffers( void )
{
  if( (demod_id == NULL) || (demod_qd == NULL) )
  {
	/* Allocate I/Q double buffers */
	size_t req = (size_t)PLAYBACK_BUF_LEN * sizeof(double);
	mem_alloc( (void **)&demod_id, req );
	mem_alloc( (void **)&demod_qd, req );
  }
}

/*----------------------------------------------------------------------*/

/* Demodulate_FM()
 *
 * Demodulates FM Signals
 */
  gboolean
Demodulate_FM( int playback_buf_len )
{
  static int
	decimate_cnt = 0, /* Count of ADC samples to go into one sound sample */
	snd_buf_idx	 = 0, /* Index to the samples buffer for the sound card */
	sdrx_buf_idx = 0, /* Index to Sdrx's ADC samples buffer */
	bandwidth	 = 0; /* Saves current bandwidth value */

  /* Measured frequency deviation of sampled IF signal */
  static double freq_dev;

  /* Last measured phase angle of sampled IF signal */
  static double phi_0 = 0.0;

  double
	phi_1,	/* Current Phase angle of sampled IF signal */
	dphi,	/* Change in Phase angle of sampled IF signal */
	ifreq;	/* Instantaneous frequency of IF signal */

  /* Used in squelch code */
  static double dev_diff = 0.0, last_dev = 0.0;

  /* Demodulated Signal (Audio) */
  double demod_signal = 0.0;

  /* Demodulator filter data structs for samples buffers */
  static filter_data_t demod_filter_data_i, demod_filter_data_q;


  /* Prepare LP filters on B/W change */
  if( (bandwidth  != rc_data.demod_bandwidth) ||
	  isFlagSet(SDRX_NEW_BUFFER) )
  {
	double cutoff = 1.0;

	/* LP Filter cutoff must be specified taking into
	 * account the transition band as a fraction of Fc */
	cutoff  = (double)rc_data.demod_bandwidth / 2.0;
	cutoff /= (double)rc_data.sdrx_sample_rate;

	/* Initialize Demodulator I filter */
	demod_filter_data_i.cutoff   = cutoff;
	demod_filter_data_i.ripple   = FM_FILTER_RIPPLE;
	demod_filter_data_i.npoles   = FM_FILTER_POLES;
	demod_filter_data_i.type     = FILTER_LOWPASS;
	demod_filter_data_i.ring_idx = 0;
	demod_filter_data_i.samples_buf = sdrx_buf_i;
	demod_filter_data_i.samples_buf_len = rc_data.sdrx_buffer_len;
	Init_Chebyshev_Filter( &demod_filter_data_i );

	/* Initialize Demodulator Q filter */
	demod_filter_data_q.cutoff   = cutoff;
	demod_filter_data_q.ripple   = FM_FILTER_RIPPLE;
	demod_filter_data_q.npoles   = FM_FILTER_POLES;
	demod_filter_data_q.type     = FILTER_LOWPASS;
	demod_filter_data_q.ring_idx = 0;
	demod_filter_data_q.samples_buf = sdrx_buf_q;
	demod_filter_data_q.samples_buf_len = rc_data.sdrx_buffer_len;
	Init_Chebyshev_Filter( &demod_filter_data_q );

	bandwidth  = rc_data.demod_bandwidth;
	ClearFlag( SDRX_NEW_BUFFER );

  } /* if( bandwidth != rc_data.demod_bandwidth ) || */

  /* Demodulate filtered I/Q buffers */
  DSP_Filter( &demod_filter_data_i );
  DSP_Filter( &demod_filter_data_q );

  /* Fill buffer from Sdrx device. buf_len is in snd frames */
  playback_buf_len *= SND_NUM_CHANNELS;
  while( TRUE )
  {
	while( snd_buf_idx < playback_buf_len )
	{
	  /* Use rc_data.oversampling samples to produce
	   * an average of the frequency deviation */
	  freq_dev = 0.0;
	  while( decimate_cnt < rc_data.oversampling )
	  {
		/* Wait for sdrx buffer to refill when used up */
		if( sdrx_buf_idx >= rc_data.sdrx_buffer_len )
		{
		  sdrx_buf_idx = 0;
		  return( TRUE );
		}

		/* Phase angle of current I/Q sample */
		phi_1 = atan2( sdrx_buf_q[sdrx_buf_idx], sdrx_buf_i[sdrx_buf_idx] );
		sdrx_buf_idx++;

		/* Phase difference from previous to current I/Q sample */
		dphi  = phi_1 - phi_0;

		/* Save current phase angle as the next previous */
		phi_0 = phi_1;

		/* Correct for discontinuity at +-180 deg phase angle */
		if( dphi > M_PI_2 ) dphi -= M_2PI;
		else if( dphi < -M_PI_2 ) dphi += M_2PI;

		/* Measured frequency deviation */
		ifreq = (double)rc_data.sdrx_sample_rate * dphi / M_2PI;
		freq_dev += ifreq;

		/* Will correct frequency offset only if enabled */
		Correct_Frequency_Offset( ifreq, 1, rc_data.sdrx_sample_rate );

		decimate_cnt++;
	  } /* while( decimate_cnt < rc_data.oversampling ) */
	  decimate_cnt = 0;

	  /* Scale frequency deviation */
	  freq_dev /= (double)rc_data.oversampling;

	  /* Do Noise Squelch calculations */
	  if( rc_data.squelch_value < SQUELCH_THRESHOLD * 100 )
	  {
		double dev = freq_dev;
		dev_diff *= SQUELCH_WINDOW_MULT;
		dev_diff += fabs( dev - last_dev );
		dev_diff /= SQUELCH_WINDOW_LEN;
		last_dev = dev;
	  }
	  else dev_diff = 0.0;

	  /* Suppress output if squelch threshold passed */
	  if( dev_diff > rc_data.squelch_value )
		demod_signal = 0.0;
	  else
		demod_signal = freq_dev;

	  /* Process and save demodulated signal in playback buffer */
	  Demodulated_Signal_Control( demod_signal, snd_buf_idx );

	  snd_buf_idx += SND_NUM_CHANNELS;
	} /* while( snd_buf_idx < playback_buf_len ) */
	snd_buf_idx = 0;

	/* Advance demodulator and
	 * sound ring buffers index */
	Playback_Control();

  } /* while( TRUE ) */

  return( TRUE );
} /* Demodulate_FM() */

/*----------------------------------------------------------------------*/

/* Demodulate_AM()
 *
 * Demodulates AM Signals
 */
  gboolean
Demodulate_AM( int playback_buf_len )
{
  static double
	am_ave = 0.0; /* Average value of AM amplitude */

  double
	sig_ampl, /* AM Radio Signal Amplitude */
	ifreq;	  /* Instantaneous frequency of IF signal */

  static int
	decimate_cnt = 0, /* Count of ADC samples to go into one sound sample */
	snd_buf_idx	 = 0, /* Index to the samples buffer for the sound card */
	sdrx_buf_idx = 0, /* Index to Sdrx's ADC samples buffer */
	iqd_buf_idx	 = 0, /* Index to i and q buffers */
	bandwidth	 = 0; /* Saves current bandwidth value */

  /* Demodulator filter data structs for samples buffers */
  static filter_data_t demod_filter_data_i, demod_filter_data_q;

  /* Demodulated Signal (Audio) */
  double demod_signal;


  /* Allocate local filter buffers */
  Alloc_Demod_Buffers();

  /* Prepare LP filters on B/W change */
  if( bandwidth != rc_data.demod_bandwidth )
  {
	double cutoff = 1.0;

	/* LP Filter cutoff must be specified taking into
	 * account the transition band as a fraction of Fc */
	cutoff  = (double)rc_data.demod_bandwidth;
	cutoff /= (double)SND_DSP_RATE * 2.0;

	/* Initialize Demodulator I filter */
	demod_filter_data_i.cutoff   = cutoff;
	demod_filter_data_i.ripple   = AM_FILTER_RIPPLE;
	demod_filter_data_i.npoles   = AM_FILTER_POLES;
	demod_filter_data_i.type     = FILTER_LOWPASS;
	demod_filter_data_i.ring_idx = 0;
	demod_filter_data_i.samples_buf = demod_id;
	demod_filter_data_i.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &demod_filter_data_i );

	/* Initialize Demodulator Q filter */
	demod_filter_data_q.cutoff   = cutoff;
	demod_filter_data_q.ripple   = AM_FILTER_RIPPLE;
	demod_filter_data_q.npoles   = AM_FILTER_POLES;
	demod_filter_data_q.type     = FILTER_LOWPASS;
	demod_filter_data_q.ring_idx = 0;
	demod_filter_data_q.samples_buf = demod_qd;
	demod_filter_data_q.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &demod_filter_data_q );

	bandwidth = rc_data.demod_bandwidth;
  } /* if( bandwidth != rc_data.demod_bandwidth ) */

  /* Fill buffer from Sdrx device. buf_len is in snd frames */
  playback_buf_len *= SND_NUM_CHANNELS;
  while( TRUE )
  {
	while( snd_buf_idx < playback_buf_len )
	{
	  /* Use rc_data.oversampling samples to
	   * produce one carrier amplitude sample */
	  demod_id[iqd_buf_idx] = 0.0;
	  demod_qd[iqd_buf_idx] = 0.0;
	  while( decimate_cnt < rc_data.oversampling )
	  {
		/* Wait for sdrx buffer to refill */
		if( sdrx_buf_idx >= rc_data.sdrx_buffer_len )
		{
		  sdrx_buf_idx = 0;
		  return( TRUE );
		}

		/* Sum I/Q values as floats */
		demod_id[iqd_buf_idx] += sdrx_buf_i[sdrx_buf_idx];
		demod_qd[iqd_buf_idx] += sdrx_buf_q[sdrx_buf_idx];
		sdrx_buf_idx++;

		decimate_cnt++;
	  } /* while( decimate_cnt < rc_data.oversampling ) */
	  decimate_cnt = 0;

	  /* Measure and compensate frequency offset */
	  ifreq = Calculate_Frequency_Offset(
		  demod_id[iqd_buf_idx], demod_qd[iqd_buf_idx] );

	  /* Will correct frequency offset only if enabled */
	  Correct_Frequency_Offset( ifreq, 1, SND_DSP_RATE );

	  iqd_buf_idx++;
	  snd_buf_idx += SND_NUM_CHANNELS;
	} /* While( snd_buf_idx < playback_buf_len ) */
	iqd_buf_idx = 0;
	snd_buf_idx = 0;

	/* Demodulate filtered I/Q buffers */
	DSP_Filter( &demod_filter_data_i );
	DSP_Filter( &demod_filter_data_q );
	for( int idx = 0; idx < playback_buf_len; idx += SND_NUM_CHANNELS )
	{
	  /* Amplitude of AM radio signal */
	  sig_ampl = hypot( demod_id[iqd_buf_idx], demod_qd[iqd_buf_idx] );

	  /* Scale the measured amplitude */
	  sig_ampl /= (double)rc_data.oversampling;

	  /* Remove average (DC) value */
	  am_ave  = am_ave * AM_AVE_WINDOW_MUL + sig_ampl;
	  am_ave /= AM_AVE_WINDOW_LEN;

	  /* Fill sound buffer frames. Amplitude with average
	   * subtracted must be mult. by 2 to scale properly */
	  demod_signal = ( sig_ampl - am_ave );

	  /* Process and save demodulated signal in playback buffer */
	  Demodulated_Signal_Control( demod_signal, idx );

	  iqd_buf_idx++;
	} /* for( idx = 0; idx < playback_buf_len; idx += SND_NUM_CHN ) */
	iqd_buf_idx = 0; snd_buf_idx = 0;

	/* Advance demodulator ring buffers index */
	Playback_Control();

  } /* while( TRUE ) */

  return( TRUE );
} /* Demodulate_AM() */

/*----------------------------------------------------------------------*/

/* Demodulate_SSB()
 *
 * Demodulates SSB Signals
 */
  gboolean
Demodulate_SSB( int playback_buf_len )
{
  static int
	decimate_cnt = 0, /* Count of ADC samples to go into one sound sample */
	snd_buf_idx	 = 0, /* Index to the samples buffer for the sound card */
	sdrx_buf_idx = 0, /* Index to Sdrx's ADC samples buffer */
	iqd_buf_idx	 = 0, /* Index to i and q buffers */
	bandwidth	 = 0; /* Saves current bandwidth value */

  /* sinf/cosf tables, their length and index */
  static double *sinf = NULL, *cosf = NULL;
  static int trig_len, itr = 0, ifl = 0;
  int idx;

  double
	cutoff = 0.0,		/* Cutoff frequency of low pass filters */
	dphi   = 0.0,		/* Change in Phase angle of sampled IF signal */
	base_band = 0.0,	/* SSB Radio Signal's base band */
	ifreq,				/* Instantaneous frequency of IF signal */
	freq_offset;		/* Frequency offset between carrier and center freq */

  /* I/Q Samples buffers for the LP Filters */
  static double *afc_id = NULL, *afc_qd = NULL;

  /* Demodulator filter data structs for samples buffers */
  static filter_data_t demod_filter_data_i, demod_filter_data_q;

  /* Demodulator filter data structs for above samples buffers */
  static filter_data_t afc_filter_data_i, afc_filter_data_q;
  static size_t mem_req = 0;


  /* Allocate local filter buffers if needed */
  Alloc_Demod_Buffers();

  /* Allocate AFC I/Q double buffers */
  if( (afc_id == NULL) || (afc_qd == NULL) )
  {
	mem_req = (size_t)playback_buf_len * sizeof(double);
	mem_alloc( (void **)&afc_id, mem_req );
	mem_alloc( (void **)&afc_qd, mem_req );

	/* LP Filter cutoff must be specified taking into
	 * account the transition band as a fraction of Fc */
	cutoff  = AFC_BANDWIDTH;
	cutoff /= (double)SND_DSP_RATE * 2.0;

	/* Initialize Demodulator I filter */
	afc_filter_data_i.cutoff   = cutoff;
	afc_filter_data_i.ripple   = AFC_FILTER_RIPPLE;
	afc_filter_data_i.npoles   = AFC_FILTER_POLES;
	afc_filter_data_i.type     = FILTER_LOWPASS;
	afc_filter_data_i.ring_idx = 0;
	afc_filter_data_i.samples_buf = afc_id;
	afc_filter_data_i.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &afc_filter_data_i );

	/* Initialize Demodulator Q filter */
	afc_filter_data_q.cutoff   = cutoff;
	afc_filter_data_q.ripple   = AFC_FILTER_RIPPLE;
	afc_filter_data_q.npoles   = AFC_FILTER_POLES;
	afc_filter_data_q.type     = FILTER_LOWPASS;
	afc_filter_data_q.ring_idx = 0;
	afc_filter_data_q.samples_buf = afc_qd;
	afc_filter_data_q.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &afc_filter_data_q );
  } /* if( (afc_id == NULL) || (afc_qd == NULL) ) */

  /* Initialize on first call or change of params */
  static int mode = -1, weav = -1;
  if( ((mode != rc_data.modulation_mode)   ||
	   (weav != rc_data.weaver_frequency)) &&
	   (rc_data.weaver_frequency != 0.0) )
  {
	double phi = 0.0;
	int idx;

	/* Length of sinf/cosf trig tables */
	trig_len = SND_DSP_RATE / rc_data.weaver_frequency;

	/* Allocate trigonometric tables */
	size_t req = (size_t)trig_len * sizeof(double);
	mem_realloc( (void **)&sinf, req );
	mem_realloc( (void **)&cosf, req );

	/* Calculate trigonometric tables. "Negative" Weaver
	 * frequency is used for lower sideband demodulation */
	if( (rc_data.modulation_mode == USBW) ||
		(rc_data.modulation_mode == USBM) ||
		(rc_data.modulation_mode == USBN) ||
		(rc_data.modulation_mode == AMSU) )
	  dphi = M_2PI / (double)trig_len;
	else
	  if( (rc_data.modulation_mode == LSBW) ||
		  (rc_data.modulation_mode == LSBM) ||
		  (rc_data.modulation_mode == LSBN) ||
		  (rc_data.modulation_mode == AMSL) )
		dphi = -M_2PI / (double)trig_len;

	for( idx = 0; idx < trig_len; idx++ )
	{
	  sinf[idx] = sin( phi );
	  cosf[idx] = cos( phi );
	  phi += dphi;
	}

	mode = rc_data.modulation_mode;
	weav = rc_data.weaver_frequency;
  } /* if( mode != rc_data.modulation_mode ) */

  /* Prepare LP filters on B/W change */
  if( bandwidth != rc_data.demod_bandwidth )
  {
	/* LP Filter cutoff must be specified taking into
	 * account the transition band as a fraction of Fc */
	cutoff  = (double)rc_data.demod_bandwidth;
	cutoff /= (double)SND_DSP_RATE * 2.0;

	/* Initialize Demodulator I filter */
	demod_filter_data_i.cutoff   = cutoff;
	demod_filter_data_i.ripple   = SSB_FILTER_RIPPLE;
	demod_filter_data_i.npoles   = SSB_FILTER_POLES;
	demod_filter_data_i.type     = FILTER_LOWPASS;
	demod_filter_data_i.ring_idx = 0;
	demod_filter_data_i.samples_buf = demod_id;
	demod_filter_data_i.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &demod_filter_data_i );

	/* Initialize Demodulator Q filter */
	demod_filter_data_q.cutoff   = cutoff;
	demod_filter_data_q.ripple   = SSB_FILTER_RIPPLE;
	demod_filter_data_q.npoles   = SSB_FILTER_POLES;
	demod_filter_data_q.type     = FILTER_LOWPASS;
	demod_filter_data_q.ring_idx = 0;
	demod_filter_data_q.samples_buf = demod_qd;
	demod_filter_data_q.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &demod_filter_data_q );

	bandwidth = rc_data.demod_bandwidth;
  } /* if( bandwidth != rc_data.demod_bandwidth ) */

  /* Fill buffer from Sdrx device
   * playback_buf_len is in frames */
  playback_buf_len *= SND_NUM_CHANNELS;
  while( TRUE )
  {
	while( snd_buf_idx < playback_buf_len )
	{
	  /* Use rc_data.oversampling samples to
	   * produce one carrier amplitude sample */
	  demod_id[iqd_buf_idx] = 0.0;
	  demod_qd[iqd_buf_idx] = 0.0;
	  while( decimate_cnt < rc_data.oversampling )
	  {
		/* Wait for sdrx buffer to refill */
		if( sdrx_buf_idx >= rc_data.sdrx_buffer_len )
		{
		  sdrx_buf_idx = 0;
		  return( TRUE );
		}

		/* Sum I/Q values as floats */
		demod_id[iqd_buf_idx] += sdrx_buf_i[sdrx_buf_idx];
		demod_qd[iqd_buf_idx] += sdrx_buf_q[sdrx_buf_idx];
		sdrx_buf_idx++;

		decimate_cnt++;
	  } /* while( decimate_cnt < rc_data.oversampling ) */
	  decimate_cnt = 0;

	  iqd_buf_idx++;
	  snd_buf_idx += SND_NUM_CHANNELS;
	} /* while( snd_buf_idx < buf_len ) */
	iqd_buf_idx = 0;
	snd_buf_idx = 0;

	/* Measure and compensate frequency offset */
	if( isFlagSet(ENABLE_FREQ_CORR) )
	{
	  /* Translate frequency of I/Q sample stream by Weaver frequency.
	   * This is to shift the carrier frequency of AM signals to 0 Hz */
	  for( int idx = 0; idx < playback_buf_len; idx += SND_NUM_CHANNELS )
	  {
		afc_id[iqd_buf_idx] =
		  demod_id[iqd_buf_idx] * cosf[ifl] -
		  demod_qd[iqd_buf_idx] * sinf[ifl];
		afc_qd[iqd_buf_idx] =
		  demod_qd[iqd_buf_idx] * cosf[ifl] +
		  demod_id[iqd_buf_idx] * sinf[ifl];

		iqd_buf_idx++; ifl++;
		if( ifl >= trig_len ) ifl = 0;
	  } /* for( int idx = 0; idx < buf_len; idx += SND_NUM_CHN ) */
	  iqd_buf_idx = 0;

	  /* Low-pass filter AFC samples buffers */
	  DSP_Filter( &afc_filter_data_i );
	  DSP_Filter( &afc_filter_data_q );

	  /* Find frequency offset */
	  freq_offset = 0;
	  for( idx = 0; idx < playback_buf_len; idx += SND_NUM_CHANNELS )
	  {
		/* Measured signal frequency */
		ifreq = Calculate_Frequency_Offset(
			afc_id[iqd_buf_idx], afc_qd[iqd_buf_idx] );
		freq_offset += ifreq;
		iqd_buf_idx++;
	  }
	  iqd_buf_idx = 0;

	} /* if( isFlagSet(ENABLE_FREQ_CORR) ) */
	else freq_offset = 0.0;

	/* Will correct frequency offset only if enabled */
	Correct_Frequency_Offset(
		freq_offset, playback_buf_len / 2, SND_DSP_RATE );

	/* Demodulate filtered I/Q buffers */
	DSP_Filter( &demod_filter_data_i );
	DSP_Filter( &demod_filter_data_q );
	for( idx = 0; idx < playback_buf_len; idx += SND_NUM_CHANNELS )
	{
	  /* Apply Weaver SSB demodulator method to get base band */
	  base_band =
		demod_id[iqd_buf_idx] * sinf[itr] +
		demod_qd[iqd_buf_idx] * cosf[itr];
	  itr++;
	  if( itr >= trig_len ) itr = 0;

	  /* Scale the base band value */
	  base_band /= (double)rc_data.oversampling;

	  /* Process and save demodulated signal in playback buffer */
	  Demodulated_Signal_Control( base_band, idx );

	  iqd_buf_idx++;
	} /* for( idx = 0; idx < playback_buf_len; idx += SND_NUM_CHN ) */
	iqd_buf_idx = 0;
	snd_buf_idx = 0;

	/* Advance demodulator and
	 * sound ring buffers index */
	Playback_Control();

  } /* while( TRUE ) */

  return( TRUE );
} /* Demodulate_SSB() */

/*----------------------------------------------------------------------*/

/* Demodulate_CW()
 *
 * Demodulates CW Signals
 */
  gboolean
Demodulate_CW( int playback_buf_len )
{
  static int
	decimate_cnt = 0, /* Count of ADC samples to go into one sound sample */
	snd_buf_idx	 = 0, /* Index to the samples buffer for the sound card */
	sdrx_buf_idx = 0, /* Index to Sdrx's ADC samples buffer */
	iqd_buf_idx	 = 0, /* Index to i and q buffers */
	bandwidth	 = 0; /* Saves current bandwidth value */

  /* Demodulator filter data structs for samples buffers */
  static filter_data_t demod_filter_data_i, demod_filter_data_q;

  /* sinf/cosf tables, their length and index */
  static double *sinf = NULL, *cosf = NULL;
  static int trig_len, itr = 0;

  double
	base_band = 0.0,  /* CW Radio Signal's base band */
	dphi = 0.0,		  /* Change in Phase angle of sampled IF signal */
	ifreq;			  /* Instantaneous frequency of IF signal */


  /* Allocate local filter buffers */
  Alloc_Demod_Buffers();

  /* Initialize on first call or change of mode */
  static int mode = -1, weav = -1;
  if( ((mode != rc_data.modulation_mode) ||
	  (weav != rc_data.weaver_frequency)) &&
	  (rc_data.weaver_frequency != 0.0) )
  {
	double phi = 0.0;
	int idx;

	/* Length of sinf/cosf tables */
	trig_len = SND_DSP_RATE / rc_data.weaver_frequency;

	/* Allocate trigonometric tables */
	size_t req = (size_t)trig_len * sizeof(double);
	mem_realloc( (void **)&sinf, req );
	mem_realloc( (void **)&cosf, req );

	/* Calculate trigonometric tables. "Negative" Weaver
	 * frequency is used for lower sideband demodulation */
	if( (rc_data.modulation_mode == CWUW) ||
		(rc_data.modulation_mode == CWUM) ||
		(rc_data.modulation_mode == CWUN) )
	  dphi = M_2PI / (double)trig_len;
	else
	  if( (rc_data.modulation_mode == CWLW) ||
		  (rc_data.modulation_mode == CWLM) ||
		  (rc_data.modulation_mode == CWLN) )
		dphi = -M_2PI / (double)trig_len;
	for( idx = 0; idx < trig_len; idx++ )
	{
	  sinf[idx] = sin( phi );
	  cosf[idx] = cos( phi );
	  phi += dphi;
	}

	mode = rc_data.modulation_mode;
	weav = rc_data.weaver_frequency;
  } /* if( mode != rc_data.modulation_mode ) */

  /* Prepare LP filters on B/W change */
  if( bandwidth != rc_data.demod_bandwidth )
  {
	double cutoff = 1.0;

	/* LP Filter cutoff must be specified taking into
	 * account the transition band as a fraction of Fc */
	cutoff  = (double)rc_data.demod_bandwidth;
	cutoff /= (double)SND_DSP_RATE * 2.0;

	/* Initialize Demodulator I filter */
	demod_filter_data_i.cutoff = cutoff;
	demod_filter_data_i.ripple = CW_FILTER_RIPPLE;
	demod_filter_data_i.npoles = CW_FILTER_POLES;
	demod_filter_data_i.type   = FILTER_LOWPASS;
	demod_filter_data_i.ring_idx = 0;
	demod_filter_data_i.samples_buf = demod_id;
	demod_filter_data_i.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &demod_filter_data_i );

	/* Initialize Demodulator Q filter */
	demod_filter_data_q.cutoff = cutoff;
	demod_filter_data_q.ripple = CW_FILTER_RIPPLE;
	demod_filter_data_q.npoles = CW_FILTER_POLES;
	demod_filter_data_q.type   = FILTER_LOWPASS;
	demod_filter_data_q.ring_idx = 0;
	demod_filter_data_q.samples_buf = demod_qd;
	demod_filter_data_q.samples_buf_len = playback_buf_len;
	Init_Chebyshev_Filter( &demod_filter_data_q );

	bandwidth = rc_data.demod_bandwidth;
  } /* if( bandwidth != rc_data.demod_bandwidth ) */

  /* Fill buffer from Sdrx device. buf_len is in frames */
  playback_buf_len *= SND_NUM_CHANNELS;
  while( TRUE )
  {
	while( snd_buf_idx < playback_buf_len )
	{
	  /* Use rc_data.oversampling samples to
	   * produce one carrier amplitude sample */
	  demod_id[iqd_buf_idx] = 0.0;
	  demod_qd[iqd_buf_idx] = 0.0;
	  while( decimate_cnt < rc_data.oversampling )
	  {
		/* Wait for sdrx buffer to refill */
		if( sdrx_buf_idx >= rc_data.sdrx_buffer_len )
		{
		  sdrx_buf_idx = 0;
		  return( TRUE );
		}

		/* I/Q values as floats */
		demod_id[iqd_buf_idx] += sdrx_buf_i[sdrx_buf_idx];
		demod_qd[iqd_buf_idx] += sdrx_buf_q[sdrx_buf_idx];
		sdrx_buf_idx++;

		decimate_cnt++;
	  } /* while( decimate_cnt < rc_data.oversampling ) */
	  decimate_cnt = 0;

	  iqd_buf_idx++;
	  snd_buf_idx += SND_NUM_CHANNELS;
	} /* while( snd_buf_idx < buf_len ) */
	iqd_buf_idx = 0;
	snd_buf_idx = 0;

	/* Demodulate filtered I/Q buffers */
	DSP_Filter( &demod_filter_data_i );
	DSP_Filter( &demod_filter_data_q );
	for( int idx = 0; idx < playback_buf_len; idx += SND_NUM_CHANNELS )
	{
	  /* Apply Weaver Demodulation method for CW detection */
	  base_band =
		demod_id[iqd_buf_idx] * sinf[itr] +
		demod_qd[iqd_buf_idx] * cosf[itr];
	  itr++;
	  if( itr >= trig_len ) itr = 0;

	  /* Scale the base-band value */
	  base_band /= (double)rc_data.oversampling;

	  /* Process and save demodulated signal in playback buffer */
	  Demodulated_Signal_Control( base_band, idx );

	  /* Measure and compensate frequency offset */
	  ifreq = Calculate_Frequency_Offset(
		  demod_id[iqd_buf_idx], demod_qd[iqd_buf_idx] );

	  /* Will correct frequency offset only if enabled */
	  Correct_Frequency_Offset( ifreq, 1, SND_DSP_RATE );

	  iqd_buf_idx++;
	} /* for( idx = 0; idx < playback_buf_len; idx += SND_NUM_CHN ) */
	iqd_buf_idx = 0; snd_buf_idx = 0;

	/* Advance demodulator and
	 * sound ring buffers index */
	Playback_Control();

  } /* while( TRUE ) */

  return( TRUE );
}  /* Demodulate_CW() */

/*----------------------------------------------------------------------*/

