/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_H
#define COMMON_H	1

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <rtl-sdr.h>
#include <pthread.h>
#include <semaphore.h>
#include <alsa/asoundlib.h>
#include <mirsdrapi-rsp.h>
#include <perseus-sdr.h>
#include "interface.h"
#include "support.h"

/* Runtime config data */
typedef struct rc_data
{
  int
	dongle_type,      /* Type of SDR dongle or device */
	sdrx_buffer_len,  /* Sdrx samples buffer length */
	modulation_mode,  /* RF Carrier Modulation Mode */
	weaver_frequency, /* Phasing frequency used for SSB/CW demod (Weaver method) */
	weaver_offset,    /* Center frequency offset due to phasing frequency above  */
	oversampling,     /* Oversampling (Sdrx Sample-Rate/Sound Sample-Rate) */
	demod_bandwidth,  /* Bandwidth of filter used in demodulators in Hz */
	playback_buffs;   /* Number of playback ring buffers */

  uint32_t
	sdrx_sample_rate, /* Sampling Rate of SDR device */
	sdrx_center_freq; /* Center Frequency for SDR Tuner */

  /* librtlsdr Device index */
  uint32_t rtlsdr_device_index;

  /* Correction factor in ppm for RTLSDR synthesizer */
  int rtlsdr_freq_correction;

  /* Correction factor in ppm for SDRPlay synthesizer */
  double rsp1_freq_correction;

  /* Perseus Device index */
  int perseus_device_index;

  /* Correction factor in ppm for PERSEUS synthesizer */
  double perseus_freq_correction;

  /* Perseus attenuator value */
  double perseus_attenuator;

  /* S-Meter and Tuner Gain to display */
  double S_meter, smeter_offset, tuner_gain;

  double
	squelch_value,		/* Squelch threshold value */
	adagc_decay,		/* Audio Derived AGC decay factor */
	adagc_scale,		/* Scaling of demod signal before playback */
	fft_scale,			/* Scale factor to keep FFT values reasonable */
	fft_bandwidth,		/* FFT/Spectrum Roofing Filter Bandwidth in Hz */
	demodulator_scale;	/* Scale factor to keep demod o/p in range */

  /* GTK+ Monitor widgets data */
  gint
	spectrum_rowstride,
	spectrum_n_channels,
	signal_rowstride,
	signal_n_channels,
	monitor_width,
	spectrum_height,
	signal_height;

  char
	sound_card[32], /* Sound card name */
	pcm_device[32]; /* PCM device name */

} rc_data_t;

/* SDR dongle/device types */
enum
{
  DONGLE_TYPE_RTLSDR = 0,
  DONGLE_TYPE_SDRPLAY,
  DONGLE_TYPE_PERSEUS
};

/* Filter data struct */
typedef struct
{
  double
	cutoff,	/* Cutoff frequency as fraction of sample rate */
	ripple;	/* Passband ripple as a percentage */

  int
	npoles,	/* Number of poles, _must_ be even */
	type;	/* Filter type, as below this struct */

  /* a and b Coefficients of the filter */
  double *a, *b;

  /* Saved input and output values */
  double *x, *y;

  /* Ring buffer index for above */
  int ring_idx;

  /* Input samples buffer and its length */
  double *samples_buf;
  int samples_buf_len;

} filter_data_t;

/* Filter type for above struct */
enum
{
  FILTER_LOWPASS = 0,
  FILTER_HIGHPASS,
  FILTER_BANDPASS
};

/* Data used for decoding and display of Date/Time */
typedef struct
{
  /* Time station and decoded data bits */
  char station, data_bit_1, data_bit_2;

  /* Current second of time, truncated
   * Julian day DeltaUT and return status */
  int second, tjd, DUT, status;

  /* Current DUT1 value */
  double DUT1;

  /* Current decoded time */
  struct tm tim;

} time_data_t;

/* Time station index */
enum
{
  NONE = 0,
  MSF60,
  RBU66,
  DCF77,
  ALL162
};

/* Flow control flags */
#define SDRX_INITIALIZED	0x000001 /* sdrx was initialized OK */
#define RTLSDR_OPEN			0x000002 /* RTLSDR device initialized OK */
#define ENABLE_FREQ_CORR	0x000004 /* Enable Doppler shift correction */
#define TUNER_GAIN_MANUAL	0x000008 /* Enable manual Tuner Gain Control */
#define TUNER_GAIN_AUTO		0x000010 /* Enable manual Tuner Gain Control */
#define TUNER_GAIN_ADAGC	0x000020 /* Enable Audio Derived Gain Control */
#define MONITORS_START		0x000040 /* Start Spectrum and Spectrum displays */
#define PLAYBACK_START		0x000080 /* Start Sound Playback */
#define TIME_RECV_START		0x000100 /* Start Time Signal Receive/Decode */
#define PLAYBACK_STOP		0x000200 /* Stop Sound Playback */
#define RCCONFIG_SETUP		0x000400 /* Sdrx Configuration read and set-up */
#define OPER_PENDING		0x000800 /* Sdrx Configuration read and set-up */
#define FFT_INITIALIZED		0x001000 /* FFT functions initialized OK */
#define SAVE_STATIONS		0x002000 /* Save Stations File flag */
#define PLAYBACK_SUSPEND	0x004000 /* Suspend playback while changes are made */
#define PRESELECT_ENABLE	0x008000 /* Enable RX Front End Preselector */
#define MONITORS_INIT		0x010000 /* Initialize Spectrum Monitors */
#define PERSEUS_ATTEN_AUTO	0x020000 /* Perseus attenuator control auto */
#define SDRX_NEW_BUFFER		0x040000 /* Sdrx samples buffer changed */
#define PLOT_SCALES			0x080000 /* Plot scale and reticle in Spectrum display */

/* Return values */
#define ERROR		1
#define SUCCESS		0

/* Commands for dongle control */
#define ENABLE		1
#define DISABLE		0

/* Hide error dialog OK button */
#define HIDE_OK		TRUE
#define SHOW_OK		FALSE

/* Size of char arrays (strings) for error messages etc */
#define MESG_SIZE	128

/* Stations treeview up-down selection */
#define SELECT_ROW_UP		1
#define SELECT_ROW_DOWN		2

/* Factors used in calculating ADAGC decay constant */
#define ADAGC_DIVISOR		1000000.0

/* Should have been in math.h */
#ifndef M_2PI
  #define M_2PI		6.28318530717958647692
#endif

/*----------------------------------------------------------------------*/

/* Function prototypes produced by cproto */
/* callback_func.c */
gboolean Spectrum_Button_Toggled(gpointer togglebutton);
gboolean Playback_Button_Toggled(gpointer togglebutton);
gboolean Time_Togglebutton_Toggled(gpointer togglebutton);
void Time_Combobox_Changed(GtkComboBox *combobox);
void Spectrum_Drawingarea_Size_Alloc(char type, GtkWidget *widget, GdkRectangle *allocation);
void Signal_Drawingarea_Config_Event(GtkWidget *widget, GdkEventConfigure *event);
void Signal_Drawingarea_Expose_Event(void);
void FFT_Bandwidth_Changed(void *widget);
void Weaver_Combobox_Changed(GtkComboBox *combobox);
void Demodulator_Bandwidth_Combobox_Changed(GtkComboBox *combobox);
void Rate_Combobox_Changed(GtkComboBox *combobox);
void Mode_Combobox_Changed(int mode);
gboolean Frequency_Scroll_Event(GtkWidget *widget, GdkEvent *event);
gboolean Frequency_Button_Press_Event(GtkWidget *widget, GdkEventButton *event);
void Sdrx_Stop(void);
void Sdrx_Set_Center_Frequency(uint32_t center_freq);
void Sdrx_Change_Buffer(GtkComboBox *combobox);
void Sdrx_Set_Tuner_Gain(double gain);
void Sdrx_Set_Tuner_Gain_Mode(int mode);
void Sdrx_Close_Device(void);
/* callbacks.c */
void on_main_window_destroy(GtkObject *object, gpointer user_data);
gboolean on_main_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_freq_scroll_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_freq_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_center_freq_entry_activate(GtkEntry *entry, gpointer user_data);
gboolean on_center_freq_entry_scroll_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_afc_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_menu_button_clicked(GtkButton *button, gpointer user_data);
void on_weaver_combobox_changed(GtkComboBox *combobox, gpointer user_data);
void on_demod_bw_combobox_changed(GtkComboBox *combobox, gpointer user_data);
void on_smp_rate_combobox_changed(GtkComboBox *combobox, gpointer user_data);
void on_sdrx_buf_combobox_changed(GtkComboBox *combobox, gpointer user_data);
void on_fft_bw_combobox_changed(GtkComboBox *combobox, gpointer user_data);
void on_gain_hscale_value_changed(GtkRange *range, gpointer user_data);
void on_man_agc_radiobtn_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_auto_agc_radiobtn_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_audio_agc_radiobtn_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_agc_hscale_value_changed(GtkRange *range, gpointer user_data);
void on_squelch_hscale_value_changed(GtkRange *range, gpointer user_data);
void on_preamp_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_preselect_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_att_10db_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_att_20db_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_att_auto_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_time_button_clicked(GtkButton *button, gpointer user_data);
void on_stations_button_clicked(GtkButton *button, gpointer user_data);
void on_spectrum_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_playback_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
gboolean on_error_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_error_dialog_destroy(GtkObject *object, gpointer user_data);
void on_error_ok_button_clicked(GtkButton *button, gpointer user_data);
void on_error_quit_button_clicked(GtkButton *button, gpointer user_data);
void on_quit_cancel_button_clicked(GtkButton *button, gpointer user_data);
void on_quit_button_clicked(GtkButton *button, gpointer user_data);
void on_stations_window_destroy(GtkObject *object, gpointer user_data);
void on_stations_treeview_cursor_changed(GtkTreeView *treeview, gpointer user_data);
gboolean on_stations_treeview_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_down_button_clicked(GtkButton *button, gpointer user_data);
void on_up_button_clicked(GtkButton *button, gpointer user_data);
void on_delete_button_clicked(GtkButton *button, gpointer user_data);
void on_new_button_clicked(GtkButton *button, gpointer user_data);
void on_save_button_clicked(GtkButton *button, gpointer user_data);
void on_time_window_destroy(GtkObject *object, gpointer user_data);
void on_time_station_combobox_changed(GtkComboBox *combobox, gpointer user_data);
void on_time_receive_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_time_quit_button_clicked(GtkButton *button, gpointer user_data);
gboolean on_spectrum_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_spectrum_window_destroy(GtkObject *object, gpointer user_data);
gboolean on_signal_drawingarea_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
gboolean on_signal_drawingarea_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_signal_drawingarea_size_allocate(GtkWidget *widget, GdkRectangle *allocation, gpointer user_data);
gboolean on_signal_drawingarea_configure_event(GtkWidget *widget, GdkEventConfigure *event, gpointer user_data);
gboolean on_spectrum_drawingarea_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
gboolean on_spectrum_drawingarea_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_spectrum_drawingarea_size_allocate(GtkWidget *widget, GdkRectangle *allocation, gpointer user_data);
void on_usb_wide_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_usb_medium_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_usb_narrow_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_lsb_wide_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_lsb_medium_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_lsb_narrow_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cwu_wide_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cwu_medium_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cwu_narrow_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cwl_wide_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cwl_medium_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cwl_narrow_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_am_wide_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_am_narrow_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_amsu_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_amsl_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_fm_wide_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_fm_medium_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_fm_narrow1_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_fm_narrow2_activate(GtkMenuItem *menuitem, gpointer user_data);
/* demodulate.c */
gboolean Demodulate_FM(int playback_buf_len);
gboolean Demodulate_AM(int playback_buf_len);
gboolean Demodulate_SSB(int playback_buf_len);
gboolean Demodulate_CW(int playback_buf_len);
/* display.c */
void Error_Dialog(gchar *mesg, gboolean hide_ok);
void Display_Center_Frequency(uint32_t freq);
void Display_Frequency_Spin_Dial(GtkLabel *label, int data);
void Update_Spin_Dial(uint32_t freq);
void Combobox_Append_Text(void);
void Sdrx_Display_S_meter(double smeter);
void Sdrx_Display_Time(time_data_t time_data);
void Close_Time_Display(void);
void Sdrx_Display_Mode(int mode);
/* filters.c */
void Init_Chebyshev_Filter(filter_data_t *filter_data);
void DSP_Filter(filter_data_t *filter_data);
void Init_Roofing_Filter(void);
/* interface.c */
GtkWidget *create_main_window(void);
GtkWidget *create_error_dialog(void);
GtkWidget *create_quit_dialog(void);
GtkWidget *create_stations_window(void);
GtkWidget *create_time_window(void);
GtkWidget *create_spectrum_window(void);
GtkWidget *create_modes_menu(void);
/* main.c */
int main(int argc, char *argv[]);
/* monitors.c */
gboolean FFT_Initialize(int fft_length);
void FFT(int n, double *x, double *y);
void FFT_SDR(int fft_length);
void Display_Spectrum(void);
//void Plot_Horizontal_Scale(cairo_t *cr, int x, int y, int width, int *nval);
void Display_Signal(void);
void Tune_to_Monitor(double x);
void Sdrx_Frequency_Status(gboolean err);
/* perseus.c */
void Perseus_Set_Center_Frequency(uint32_t center_freq);
void Perseus_Set_Tuner_Gain_Mode(int mode);
gboolean Perseus_Set_Tuner_Gain(double gain);
gboolean Perseus_Set_Sample_Rate(void);
gboolean Perseus_Change_Buffer(GtkComboBox *combobox);
gboolean Perseus_Settings(uint8_t setting);
void Perseus_Close_Device(void);
gboolean Perseus_Initialize(void);
int Perseus_Sample_Rate_Idx(int rate);
int Perseus_Sample_Rate(int index);
void Perseus_ADAGC_Smeter(void);
/* rtlsdr.c */
void RtlSdr_Set_Center_Frequency(uint32_t center_freq);
void RtlSdr_Set_Tuner_Gain_Mode(int mode);
gboolean RtlSdr_Set_Tuner_Gain(double gain);
void RtlSdr_Get_Tuner_Gain(void);
gboolean RtlSdr_Set_Sample_Rate(void);
gboolean RtlSdr_Change_Buffer(GtkComboBox *combobox);
gboolean RtlSdr_Initialize(void);
void RtlSdr_Close_Device(void);
/* sdrplay.c */
void SDRPlay_Set_Center_Frequency(uint32_t center_freq);
void SDRPlay_Set_Tuner_Gain_Mode(int mode);
gboolean SDRPlay_Set_Tuner_Gain(double gain);
gboolean SDRPlay_Set_Sample_Rate(void);
gboolean SDRPlay_Change_Buffer(GtkComboBox *combobox);
void SDRPlay_Close_Device(void);
gboolean SDRPlay_Initialize(void);
/* shared.c */
/* sound.c */
gboolean Open_Playback(void);
gboolean DSP_Write(short *buffer, int buf_len);
void Close_Playback(void);
/* stations.c */
void List_Stations(void);
gboolean Save_Stations_File(void);
void Select_Treeview_Row(int direction);
void Stations_Cursor_Changed(GtkTreeView *treeview);
void Treeview_Button_Press(void);
void New_Station_Row(void);
void Delete_Station_Row(void);
/* support.c */
GtkWidget *lookup_widget(GtkWidget *widget, const gchar *widget_name);
void add_pixmap_directory(const gchar *directory);
GtkWidget *create_pixmap(GtkWidget *widget, const gchar *filename);
GdkPixbuf *create_pixbuf(const gchar *filename);
void glade_set_atk_action_description(AtkAction *action, const gchar *action_name, const gchar *description);
/* time_signal.c */
void Free_Time_Buffers(void);
void Receive_DCF77(void);
void Receive_ALL162(void);
void Receive_RBU66(void);
void Receive_MSF60(void);
/* utils.c */
int Load_Line(char *buff, FILE *pfile, char *mesg);
gboolean Load_Config(gpointer data);
void mem_alloc(void **ptr, size_t req);
void mem_realloc(void **ptr, size_t req);
void free_ptr(void **ptr);
int isFlagSet(int flag);
int isFlagClear(int flag);
void SetFlag(int flag);
void ClearFlag(int flag);
void ToggleFlag(int flag);
void Sdrx_New_Buffer(GtkComboBox *combobox);
void Set_FFT_Bandwidth_Combobox(uint32_t sample_rate);
gboolean Open_File(FILE **fp, char *fname, const char *mode);

/*----------------------------------------------------------------------*/

#endif /* End of common.h */

