/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "shared.h"

/*----------------------------------------------------------------------*/

/* Sound_Write_Async()
 *
 * Thread to asynchronously write sound buffer for replay
 */
  static void *
Sound_Write_Async( void *arg )
{
  while( isFlagSet(PLAYBACK_START) )
  {
	/* Write playback buffer to sound device */
	if( !DSP_Write(playback_buf[sound_buf_num], PLAYBACK_BUF_LEN) )
	  break;

	/* Advance sound playback buffer
	 * index and keep in limits */
	sound_buf_num++;
	if( sound_buf_num >= rc_data.playback_buffs )
	  sound_buf_num = 0;

  } /* while( isFlagSet(PLAYBACK_START) ) */

  ClearFlag( PLAYBACK_STOP );
  Close_Playback();
  pthread_exit( NULL );

  return( NULL );
} /* Sound_Write_Async() */

/*----------------------------------------------------------------------*/

/* Sdrx_Start()
 *
 * Controls the flow of sdrx
 */
  static gboolean
Sdrx_Start( gpointer data )
{
  /* Exit thread if Start toggled off */
  if( (isFlagClear(MONITORS_START) &&
	   isFlagClear(PLAYBACK_START) &&
	   isFlagClear(TIME_RECV_START)) ||
	   isFlagClear(SDRX_INITIALIZED) )
	return( FALSE );

  /* Wait on DSP data to be ready for processing */
  sem_wait( &pback_semaphore );

  /* Low pass filter I and Q data */
  DSP_Filter( &fft_filter_data_i );
  DSP_Filter( &fft_filter_data_q );

  /* Display FFT spectrum and signal */
  if( isFlagSet(MONITORS_START) )
  {
	/* Decimate input samples to match FFT bandwidth to user selected
	 * value. Default FFT bandwidth is equal to the ADC sample rate */
	int fft_decim = (int)
	  ( (double)rc_data.sdrx_sample_rate / rc_data.fft_bandwidth );

	/* Save samples for carrier fft */
	if( rc_data.sdrx_buffer_len >= FFT_INPUT_LENGTH * fft_decim )
	{
	  int buf_idx = 0;
	  for( int fft_idx = 0; fft_idx < FFT_INPUT_LENGTH; fft_idx++ )
	  {
		fft_in_i[fft_idx] = fft_in_q[fft_idx] = 0.0;

		for( int cnt = 0; cnt < fft_decim; cnt++ )
		{
		  fft_in_i[fft_idx] += sdrx_buf_i[buf_idx];
		  fft_in_q[fft_idx] += sdrx_buf_q[buf_idx];
		  buf_idx++;
		}
		fft_in_i[fft_idx] /= (double)fft_decim;
		fft_in_q[fft_idx] /= (double)fft_decim;

	  } /* for( int fft_idx = 0; fft_idx < FFT_INPUT_LENGTH; fft_idx++ ) */
	} /* if( isFlagSet(MONITORS_START) ) */

	/* Display FFT spectrum and signal */
	Display_Spectrum();
	Display_Signal();

  } /* if( isFlagSet(MONITORS_START) ) */

  /* Decode time and date from Time Stations */
  if( isFlagSet(TIME_RECV_START) )
  {
	if( Receive_Time_Station != NULL )
	  Receive_Time_Station();
  }

  /* Stream data from Device and demodulate */
  if( isFlagSet(PLAYBACK_START) )
  {
	/* Call appropriate demodulator function */
	if( !Demodulator(PLAYBACK_BUF_LEN) )
	  return( FALSE );

	/* Calculate relative signal strength for ADAGC */
	if( isFlagSet(TUNER_GAIN_ADAGC) )
	{
	  if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS )
		Perseus_ADAGC_Smeter();
	} /* if( isFlagSet(TUNER_GAIN_ADAGC) ) */

	/* Display tuner gain and signal level bar */
	Sdrx_Display_S_meter( rc_data.S_meter );

  } /* if( isFlagSet(PLAYBACK_START) ) */

  return( TRUE );
} /* Sdrx_Start() */

/*----------------------------------------------------------------------*/

/* Sdrx_Initialize()
 *
 * Initializes the API and the Device Hardware
 */
  static gboolean
Sdrx_Initialize( void )
{
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  if( !RtlSdr_Initialize() ) return( FALSE );
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  if( !SDRPlay_Initialize() ) return( FALSE );
	  break;

	case DONGLE_TYPE_PERSEUS:
	  if( !Perseus_Initialize() ) return( FALSE );
	  break;
  }

  return( TRUE );
} /* Sdrx_Initialize() */

/*----------------------------------------------------------------------*/

/* Sdrx_Set_Sample_Rate()
 *
 * Sets the Device ADC sample rate
 */
  static gboolean
Sdrx_Set_Sample_Rate( void )
{
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  if( !RtlSdr_Set_Sample_Rate() ) return( FALSE );
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  if( !SDRPlay_Set_Sample_Rate() ) return( FALSE );
	  break;

	case DONGLE_TYPE_PERSEUS:
	  if( !Perseus_Set_Sample_Rate() ) return( FALSE );
	  break;
  }

  return( TRUE );
} /* Sdrx_Set_Sample_Rate() */

/*----------------------------------------------------------------------*/

/* Spectrum_Button_Toggled()
 *
 * Handles the on_fft_togglebutton_toggled CB
 */
  gboolean
Spectrum_Button_Toggled( gpointer togglebutton )
{
  if( gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(togglebutton)) )
  {
	/* Create Spectrum Window */
	if( spectrum_window == NULL )
	{
	  spectrum_window = create_spectrum_window();
	  gtk_widget_show( spectrum_window );

	  /* Get spectrum and signal widgets */
	  signal_display   = lookup_widget( spectrum_window, "signal_drawingarea" );
	  spectrum_display = lookup_widget( spectrum_window, "spectrum_drawingarea" );
	}

	/* Initialize SDR device and sdrx */
	if( !Sdrx_Initialize() ) return( FALSE );
	SetFlag( MONITORS_START );

	/* Signal start of sdrx and displays */
	if( isFlagClear(PLAYBACK_START) &&
		isFlagClear(TIME_RECV_START) )
	  g_idle_add( Sdrx_Start, NULL );
  }
  else
  {
	/* Stop sdrx and cleanup */
	ClearFlag( MONITORS_START );
	ClearFlag( FFT_INITIALIZED );
	if( isFlagClear(PLAYBACK_START) &&
		isFlagClear(TIME_RECV_START) )
	  Sdrx_Close_Device();
  }

  return( FALSE );
} /* Spectrum_Button_Toggled() */

/*----------------------------------------------------------------------*/

/* Playback_Button_Toggled()
 *
 * Handles on_playback_togglebutton_toggled() C/B
 */
  gboolean
Playback_Button_Toggled( gpointer togglebutton )
{
  pthread_t pthread_id;

  if( gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(togglebutton)) )
  {
	/* Initialize SDR device */
	if( !Sdrx_Initialize() ) return( FALSE );

	/* Create a thread for async read from SDR device */
	if( Open_Playback() )
	{
	  /* Create a thread to play back demodulation buffer */
	  int ret = pthread_create( &pthread_id, NULL, Sound_Write_Async, NULL );
	  if( ret != SUCCESS )
	  {
		Error_Dialog(
			_("Failed to create Sound Async Write thread"), HIDE_OK );
		return( FALSE );
	  }

	  if( isFlagClear(MONITORS_START) &&
		  isFlagClear(TIME_RECV_START) )
		g_idle_add( Sdrx_Start, NULL );

	} /* if( Open_Playback() ) */
	else return( FALSE );

  } /* if( gtk_toggle_button_get_active() */
  else
  {
	ClearFlag( PLAYBACK_START );
	if( isFlagClear(MONITORS_START) &&
		isFlagClear(TIME_RECV_START) )
	  Sdrx_Close_Device();
  }

  return( FALSE );
} /* Playback_Button_Toggled() */

/*----------------------------------------------------------------------*/

/* Time_Togglebutton_Toggled()
 *
 * Handles the on_time_togglebutton_toggled() CB
 */
  gboolean
Time_Togglebutton_Toggled( gpointer togglebutton )
{
  if( gtk_toggle_button_get_active(
		GTK_TOGGLE_BUTTON(togglebutton)) )
  {
	/* Initialize SDR device and Sdrx */
	if( !Sdrx_Initialize() ) return( FALSE );
	SetFlag( TIME_RECV_START );

	/* Signal start of Sdrx and displays */
	if( isFlagClear(PLAYBACK_START) &&
		isFlagClear(MONITORS_START) )
	  g_idle_add( Sdrx_Start, NULL );
  }
  else
  {
	/* Stop Sdrx and cleanup */
	ClearFlag( TIME_RECV_START );
	Free_Time_Buffers();
	if( isFlagClear(PLAYBACK_START) &&
		isFlagClear(MONITORS_START) )
	  Sdrx_Close_Device();
  }

  return( FALSE );
} /* Time_Togglebutton_Toggled() */

/*----------------------------------------------------------------------*/

/* Time_Combobox_Changed()
 *
 * Handles the on_time_station_combobox_changed() CB
 */
  void
Time_Combobox_Changed( GtkComboBox *combobox )
{
  int modulation_mode, center_freq, station_idx;


  /* Free buffers and get the selected station */
  Free_Time_Buffers();
  station_idx = (int)gtk_combo_box_get_active( combobox );
  switch( station_idx )
  {
	case MSF60:
	  Receive_Time_Station = Receive_MSF60;
	  center_freq = MSF60_FREQUENCY;
	  modulation_mode  = CWUN;
	  gtk_widget_hide( lookup_widget(time_window, "dcf77_frame") );
	  gtk_widget_hide( lookup_widget(time_window, "rbu66_frame") );
	  gtk_widget_show( lookup_widget(time_window, "msf60_frame") );
	  gtk_window_set_title( GTK_WINDOW(time_window), "MSF 60kHz" );
	  break;

	case RBU66:
	  Receive_Time_Station = Receive_RBU66;
	  center_freq = RBU66_FREQUENCY;
	  modulation_mode  = CWUW;
	  gtk_widget_hide( lookup_widget(time_window, "dcf77_frame") );
	  gtk_widget_hide( lookup_widget(time_window, "msf60_frame") );
	  gtk_widget_show( lookup_widget(time_window, "rbu66_frame") );
	  gtk_window_set_title( GTK_WINDOW(time_window), "RBU 66.667kHz" );
	  break;

	case DCF77:
	  Receive_Time_Station = Receive_DCF77;
	  center_freq = DCF77_FREQUENCY;
	  modulation_mode  = CWUN;
	  gtk_widget_hide( lookup_widget(time_window, "rbu66_frame") );
	  gtk_widget_hide( lookup_widget(time_window, "msf60_frame") );
	  gtk_widget_show( lookup_widget(time_window, "dcf77_frame") );
	  gtk_window_set_title( GTK_WINDOW(time_window), "DCF77 77.5kHz" );
	  break;

	case ALL162:
	  Receive_Time_Station = Receive_ALL162;
	  center_freq = ALL162_FREQUENCY;
	  modulation_mode  = CWUN;
	  gtk_widget_hide( lookup_widget(time_window, "rbu66_frame") );
	  gtk_widget_hide( lookup_widget(time_window, "msf60_frame") );
	  gtk_widget_show( lookup_widget(time_window, "dcf77_frame") );
	  gtk_window_set_title( GTK_WINDOW(time_window), "Allouis 162kHz" );
	  break;

	default:
	  Receive_Time_Station = NULL;
	  return;
  }
  gtk_window_resize( GTK_WINDOW(time_window), 10, 10 );

  /* Set up center frequency on change */
  if( rc_data.sdrx_center_freq != center_freq )
  {
	rc_data.sdrx_center_freq = center_freq;
	Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );
  }

  /* Set up new mode on change */
  if( rc_data.modulation_mode != modulation_mode )
	Mode_Combobox_Changed( modulation_mode );

} /* Time_Combobox_Changed() */

/*----------------------------------------------------------------------*/

/* Spectrum_Drawingarea_Size_Alloc()
 *
 * Handles the on_spectrum_drawingarea_size_allocate CB
 */
  void
Spectrum_Drawingarea_Size_Alloc(
	char type,
	GtkWidget *widget,
	GdkRectangle *allocation )
{
  ClearFlag( MONITORS_START );
  ClearFlag( FFT_INITIALIZED );

  /* Initialize for spectrum allocation */
  if( type == SPECTRUM_DRAWINGAREA )
  {
	/* Destroy existing pixbuff */
	if( spectrum_pixbuf != NULL )
	{
	  g_object_unref( G_OBJECT(spectrum_pixbuf) );
	  spectrum_pixbuf = NULL;
	}

	/* Create spectrum pixbuf */
	spectrum_pixbuf = gdk_pixbuf_new(
		GDK_COLORSPACE_RGB, FALSE, 8,
		allocation->width, allocation->height );
	if( spectrum_pixbuf == NULL )
	{
	  spectrum_pixels         = NULL;
	  rc_data.monitor_width   = 0;
	  rc_data.spectrum_height = 0;
	  rc_data.spectrum_rowstride  = 0;
	  rc_data.spectrum_n_channels = 0;

	  Error_Dialog( _("Failed to create pixbuf for Spectrum Display"), HIDE_OK );
	  return;
	}

	/* Get spectrum pixbuf details and clear it */
	spectrum_pixels         = gdk_pixbuf_get_pixels( spectrum_pixbuf );
	rc_data.monitor_width   = gdk_pixbuf_get_width ( spectrum_pixbuf );
	rc_data.spectrum_height = gdk_pixbuf_get_height( spectrum_pixbuf );
	rc_data.spectrum_rowstride  = gdk_pixbuf_get_rowstride( spectrum_pixbuf );
	rc_data.spectrum_n_channels = gdk_pixbuf_get_n_channels( spectrum_pixbuf );
	gdk_pixbuf_fill( spectrum_pixbuf, 0 );

  } /* if( type == SPECTRUM_DRAWINGAREA ) */

  /* Initialize the FFT function after allocations complete */
  if( rc_data.monitor_width && rc_data.signal_height )
  {
	if( FFT_Initialize(FFT_INPUT_LENGTH) )
	  SetFlag( MONITORS_INIT );
  }

} /* Spectrum_Drawingarea_Size_Alloc() */

/*----------------------------------------------------------------------*/

/* Signal_Drawingarea_Config_Event()
 *
 * Creates a pixmap for the signal display
 */
  void
Signal_Drawingarea_Config_Event(
	GtkWidget *widget,
	GdkEventConfigure *event )
{
  /* Initialize for signal allocation */
  /* Create or resize signal pixmap */
  if( signal_pixmap != NULL )
  {
	g_object_unref( signal_pixmap );
	signal_pixmap = NULL;
  }

  signal_pixmap = gdk_pixmap_new(
	  widget->window, event->width, event->height, -1 );
  rc_data.signal_height = event->height;

} /* Signal_Drawingarea_Config_Event() */

/*----------------------------------------------------------------------*/

/* Signal_Drawingarea_Expose_Event()
 *
 * Handles the expose event for the signal drawingarea
 */
  void
Signal_Drawingarea_Expose_Event( void )
{
  /* Draw signal background */
  cairo_t *cr = gdk_cairo_create( signal_pixmap );
  cairo_set_source_rgb( cr, SIGNAL_BACKGND );
  cairo_rectangle( cr, 0.0, 0.0,
	  (double)rc_data.monitor_width,
	  (double)rc_data.signal_height );
  cairo_fill( cr );
  cairo_destroy( cr );

  SetFlag( PLOT_SCALES );

} /* Signal_Drawingarea_Expose_Event() */

/*----------------------------------------------------------------------*/

/* Clear_Playback_Buf()
 *
 * Clears the playback buffers on parameter changes
 */
  static void
Clear_Playback_Buf( void )
{
  int idx;
  size_t siz = 2 * PLAYBACK_BUF_LEN * sizeof( short );

  if( isFlagSet(SDRX_INITIALIZED) &&
	  isFlagSet(PLAYBACK_START) )
	for( idx = 0; idx < rc_data.playback_buffs; idx++ )
	  memset( (void *)playback_buf[idx], 0, siz );

} /* Clear_Playback_Buf() */

/*----------------------------------------------------------------------*/

/* FFT_Bandwidth_Changed
 *
 * Handles on_fft_bw_combobox_changed() CB
 */
  void
FFT_Bandwidth_Changed( void *widget )
{
  double bwd; /* Bandwidth */
  gchar *txt;

  if( isFlagSet(OPER_PENDING) )
	return;

  /* Get bandwidth in kHz and convert to Hz */
  txt = gtk_combo_box_get_active_text((GtkComboBox *)widget);
  if( txt != NULL )
  {
	bwd = atof( txt );
	bwd *= 1000.0;
	if( bwd <= 0.0 ) bwd = 1000.0;
	rc_data.fft_bandwidth = bwd;

	/* Init LPF filter if sample rate is specified */
	if( rc_data.sdrx_sample_rate ) Init_Roofing_Filter();

	SetFlag( PLOT_SCALES );
	g_free( txt );
  }

} /* FFT_Bandwidth_Changed() */

/*----------------------------------------------------------------------*/

/* Weaver_Combobox_Changed()
 *
 * Handles the weaver_combobox_changed callback
 */
  void
Weaver_Combobox_Changed( GtkComboBox *combobox )
{
  /* Available Weaver method phasing frequencies */
  char *weaver_frequency[] = { WEAVER_FREQUENCIES };

  /* Get selected Weaver frequency */
  int idx     = (int)gtk_combo_box_get_active( combobox );
  double freq = atof( weaver_frequency[idx] );
  size_t len  = strlen( weaver_frequency[idx] );

  /* If it is in kB, multiply by 1000 */
  if( weaver_frequency[idx][len-1] == 'k' ) freq *= 1000.0;
  rc_data.weaver_frequency = (int)freq;

  /* Set Weaver offset frequency */
  switch( rc_data.modulation_mode )
  {
	case USBW: case USBM: case USBN: case AMSU:
	  rc_data.weaver_offset = rc_data.weaver_frequency;
	  break;

  	case LSBW: case LSBM: case LSBN: case AMSL:
	  rc_data.weaver_offset = -rc_data.weaver_frequency;
	  break;

	default: rc_data.weaver_offset = 0;
  }

  /* Set center freq to account for SSB freq offset if relevant */
  Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );

} /* Weaver_Combobox_Changed() */

/*----------------------------------------------------------------------*/

/* Demodulator_Bandwidth_Combobox_Changed()
 *
 * Handles the demod_bw_combobox_changed callback
 */
  void
Demodulator_Bandwidth_Combobox_Changed( GtkComboBox *combobox )
{
  /* Available demodulator bandwidths */
  char *demod_bw[] = { DEMOD_BAND_WIDTHS };

  /* Get selected demodulator bandwidth */
  int idx     = (int)gtk_combo_box_get_active( combobox );
  double bndw = atof( demod_bw[idx] );
  size_t len  = strlen( demod_bw[idx] );

  /* If it is in kB, multiply by 1000 */
  if( demod_bw[idx][len-1] == 'k' ) bndw *= 1000.0;
  rc_data.demod_bandwidth = (int)bndw;

  /* Make the FFT bandwidth at least equal to demod bandwidth */
  idx = 0;
  gtk_combo_box_set_active( GTK_COMBO_BOX(fft_bw_combobox), idx );
  while( bndw > rc_data.fft_bandwidth )
  {
	static double bw = 0.0;

	/* Gone past last combobox entry but still bndw > rc_data.fft_bandwidth */
	gtk_combo_box_set_active( GTK_COMBO_BOX(fft_bw_combobox), ++idx );
	if( bw == rc_data.fft_bandwidth )
	{
	  /* Set the last (largest) FFT bandwidth available */
	  gtk_combo_box_set_active( GTK_COMBO_BOX(fft_bw_combobox), --idx );
	  break;
	}
	bw = rc_data.fft_bandwidth;
  } /* while( bndw > rc_data.fft_bandwidth ) */

} /* Demodulator_Bandwidth_Combobox_Changed() */

/*----------------------------------------------------------------------*/

/* Rate_Combobox_Changed()
 *
 * Handles the on_rate_combobox_changed() CB
 */
  void
Rate_Combobox_Changed( GtkComboBox *combobox )
{
  /* Clear play back buffer to avoid stutter etc */
  Clear_Playback_Buf();

  /* Get sample rate from combobox */
  int idx = gtk_combo_box_get_active( combobox );
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_SDRPLAY:
	  {
		/* Available sample rates for RSP1 */
		char *sdrplay_rates[] = { SDRPLAY_SAMPLE_RATES };
		rc_data.sdrx_sample_rate = (uint32_t)atoi( sdrplay_rates[idx] );
	  }
	  break;

	case DONGLE_TYPE_RTLSDR:
	  {
		/* Available sample rates for RTLSDR */
		char *rtlsdr_rates[]     = { RTLSDR_SAMPLE_RATES };
		rc_data.sdrx_sample_rate = (uint32_t)atoi( rtlsdr_rates[idx] );
	  }
	  break;

	case DONGLE_TYPE_PERSEUS:
	  rc_data.sdrx_sample_rate = (uint32_t)Perseus_Sample_Rate( idx );
	  break;
  } /* switch( rc_data.dongle_type ) */

  /* Set up main filter data and init filter */
  rc_data.sdrx_sample_rate *= (uint32_t)1000;

  /* Setup FFT Bandwidth combo box */
  Set_FFT_Bandwidth_Combobox( rc_data.sdrx_sample_rate );
  if( rc_data.sdrx_sample_rate ) Init_Roofing_Filter();

  /* Set the Sdrx ADC sample rate */
  Sdrx_Set_Sample_Rate();

} /* Rate_Combobox_Changed() */

/*----------------------------------------------------------------------*/

/* Mode_Combobox_Changed()
 *
 * Handles the on_mode_combobox_changed() CB
 */
  void
Mode_Combobox_Changed( int mode )
{
  int
	srate_idx   = 0,  /* Sample rate combobox index */
	dem_bw_idx  = 0,  /* Demodulator bandwidth combobox index */
	weaver_idx  = 0,  /* Weaver frequency combobox index */
	adc_buf_idx = 0;  /* ADC buffer size combobox index */

  /* Remembers previous state of an index */
  static int xidx = -1;

  /* If Weaver offset of center freq is needed */
  gboolean offset = FALSE;


  /* Clear play back buffer to avoid stutter etc */
  Clear_Playback_Buf();

  /* Get selected mode */
  rc_data.modulation_mode = mode;
  Sdrx_Display_Mode( mode );

  /* Set the relevant parameters for each demodulator mode */
  switch( rc_data.modulation_mode )
  {
	case WBFM:
	  dem_bw_idx = DEMOD_BW_200k;
	  weaver_idx = WEAVER_NONE;
	  offset = FALSE;
	  break;

  	case MBFM:
	  dem_bw_idx = DEMOD_BW_50k;
	  weaver_idx = WEAVER_NONE;
	  offset = FALSE;
	  break;

	case NBFM1:
	  dem_bw_idx = DEMOD_BW_25k;
	  weaver_idx = WEAVER_NONE;
	  offset = FALSE;
	  break;

	case NBFM2:
	  dem_bw_idx = DEMOD_BW_12k;
	  weaver_idx = WEAVER_NONE;
	  offset = FALSE;
	  break;

	case AMW:
	  dem_bw_idx = DEMOD_BW_8k;
	  weaver_idx = WEAVER_NONE;
	  offset = FALSE;
	  break;

	case AMN:
	  dem_bw_idx = DEMOD_BW_4500;
	  weaver_idx = WEAVER_NONE;
	  offset = FALSE;
	  break;

	case USBW: case USBM: case USBN:
	case LSBW: case LSBM: case LSBN:
	case CWUW: case CWLW: case CWUM:
	case CWLM: case CWUN: case CWLN:
	case AMSU: case AMSL:
	  offset = TRUE;
	  break;

  } /* switch( rc_data.modulation_mode ) */

  /* Set the relevant parameters for each demodulator mode */
  switch( rc_data.modulation_mode )
  {
	case WBFM: case MBFM:
	  adc_buf_idx = BUF_128;
	  switch( rc_data.dongle_type )
	  {
		case DONGLE_TYPE_SDRPLAY:
		  srate_idx = RSP_RATE_768;
		  break;

		case DONGLE_TYPE_RTLSDR:
		  srate_idx = RTL_RATE_1008;
		  break;

		case DONGLE_TYPE_PERSEUS:
		  srate_idx = Perseus_Sample_Rate_Idx( PERSEUS_RATE_192 );
		  break;
	  }
	  break;

	case NBFM1: case NBFM2:	case AMW:
	case AMN:   case AMSU:  case AMSL:
	case USBW:  case USBM:	case USBN:
	case LSBW:	case LSBM:  case LSBN:
	case CWUW:  case CWLW:	case CWUM:
	case CWLM:	case CWUN:  case CWLN:
	  adc_buf_idx = BUF_64;
	  switch( rc_data.dongle_type )
	  {
		case DONGLE_TYPE_SDRPLAY:
		  srate_idx = RSP_RATE_384;
		  break;

		case DONGLE_TYPE_RTLSDR:
		  srate_idx = RTL_RATE_288;
		  break;

		case DONGLE_TYPE_PERSEUS:
		  srate_idx = Perseus_Sample_Rate_Idx( PERSEUS_RATE_192 );
		  break;
	  }
	  break;

  } /* switch( rc_data.modulation_mode ) */

  /* Set the relevant parameters for each demodulator mode */
  switch( rc_data.modulation_mode )
  {
	case WBFM:
	  rc_data.demodulator_scale = WBFM_DEMOD_SCALE;
	  Demodulator = Demodulate_FM;
	  break;

	case MBFM:
	  rc_data.demodulator_scale = MBFM_DEMOD_SCALE;
	  Demodulator = Demodulate_FM;
	  break;

	case NBFM1:
	  rc_data.demodulator_scale = NBFM1_DEMOD_SCALE;
	  Demodulator = Demodulate_FM;
	  break;

	case NBFM2:
	  rc_data.demodulator_scale = NBFM2_DEMOD_SCALE;
	  Demodulator = Demodulate_FM;
	  break;

	case AMW: case AMN:
	  rc_data.demodulator_scale = AM_DEMOD_SCALE;
	  if( rc_data.dongle_type == DONGLE_TYPE_SDRPLAY )
		rc_data.demodulator_scale /= SDRPLAY_DEMOD_SCALE;
	  else if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS )
		rc_data.demodulator_scale /= PERSEUS_DEMOD_SCALE;
	  Demodulator = Demodulate_AM;
	  break;

	case USBW: case USBM: case USBN:
	case LSBW: case LSBM: case LSBN:
	case AMSU: case AMSL:
	  rc_data.demodulator_scale = SSB_DEMOD_SCALE;
	  if( rc_data.dongle_type == DONGLE_TYPE_SDRPLAY )
		rc_data.demodulator_scale /= SDRPLAY_DEMOD_SCALE;
	  else if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS )
		rc_data.demodulator_scale /= PERSEUS_DEMOD_SCALE;
	  Demodulator = Demodulate_SSB;
	  break;

	case CWUW: case CWLW: case CWUM:
	case CWLM: case CWUN: case CWLN:
	  rc_data.demodulator_scale = CW_DEMOD_SCALE;
	  if( rc_data.dongle_type == DONGLE_TYPE_SDRPLAY )
		rc_data.demodulator_scale /= SDRPLAY_DEMOD_SCALE;
	  else if( rc_data.dongle_type == DONGLE_TYPE_PERSEUS )
		rc_data.demodulator_scale /= PERSEUS_DEMOD_SCALE;
	  Demodulator = Demodulate_CW;
	  break;

  } /* switch( rc_data.modulation_mode ) */

  /* Set the relevant parameters for each demodulator mode */
  switch( rc_data.modulation_mode )
  {
	case USBW: case LSBW:
	case AMSU: case AMSL:
	  weaver_idx = WEAVER_2400;
	  dem_bw_idx = DEMOD_BW_4500;
	  break;

	case USBM: case LSBM:
	  weaver_idx = WEAVER_1600;
	  dem_bw_idx = DEMOD_BW_2800;
	  break;

	case USBN: case LSBN:
	  weaver_idx = WEAVER_1200;
	  dem_bw_idx = DEMOD_BW_2200;
	  break;

	case CWUW: case CWLW:
	  weaver_idx = WEAVER_600;
	  dem_bw_idx = DEMOD_BW_650;
	  break;

	case CWUM: case CWLM:
	  weaver_idx = WEAVER_600;
	  dem_bw_idx = DEMOD_BW_400;
	  break;

	case CWUN: case CWLN:
	  weaver_idx = WEAVER_600;
	  dem_bw_idx = DEMOD_BW_200;
	  break;

  } /* switch( rc_data.modulation_mode ) */

  if( isFlagClear(RCCONFIG_SETUP) )
	return;

  /* Set the parameters to the relevant combo boxes */
  gtk_combo_box_set_active(
	  GTK_COMBO_BOX(sdrx_buf_combobox), adc_buf_idx );
  gtk_combo_box_set_active(
	  GTK_COMBO_BOX(smp_rate_combobox), srate_idx );
  gtk_combo_box_set_active(
	  GTK_COMBO_BOX(demod_bw_combobox), dem_bw_idx );
  gtk_combo_box_set_active(
	  GTK_COMBO_BOX(weaver_combobox), weaver_idx );

  /* Calculate weaver offset if weaver_idx
   * doesn't change, as going from USBW to LSBW */
  if( (xidx == weaver_idx) && offset )
  {
	char   *weaver_frequency[] = { WEAVER_FREQUENCIES };
	double freq = atof( weaver_frequency[weaver_idx] );
	size_t len  = strlen( weaver_frequency[weaver_idx] );

	/* If it is in kB, multiply by 1000 */
	if( weaver_frequency[weaver_idx][len-1] == 'k' )
	  freq *= 1000.0;

	/* Set Weaver offset frequency */
	switch( rc_data.modulation_mode )
	{
	  case USBW: case USBM: case USBN: case AMSU:
		rc_data.weaver_offset = (int)freq;
		break;

	  case LSBW: case LSBM: case LSBN: case AMSL:
		rc_data.weaver_offset = -(int)freq;
		break;
	}

	/* Set center freq to account for SSB freq offset if relevant */
	Sdrx_Set_Center_Frequency( rc_data.sdrx_center_freq );
  }
  xidx = weaver_idx;

  if( !offset ) rc_data.weaver_offset = 0;

  if( (rc_data.modulation_mode == AMSU) || (rc_data.modulation_mode == AMSL) )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(afc_checkbutton), TRUE );
  else
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(afc_checkbutton), FALSE );

} /* Mode_Combobox_Changed() */

/*----------------------------------------------------------------------*/

/* Frequency_Scroll_Event()
 *
 * Handles scroll events on the Frequency "spin dial"
 */
  gboolean
Frequency_Scroll_Event( GtkWidget *widget, GdkEvent *event )
{
  const gchar *txt;
  int data;
  GtkLabel *label;


  /* Get the label from the event box widget */
  if( GTK_IS_BIN(widget) )
	label = GTK_LABEL( gtk_bin_get_child(GTK_BIN(widget)) );
  else return( FALSE );

  /* Get text from label and convert to number */
  txt  = gtk_label_get_text( label );
  data = atoi( txt );

  /* Change value of label data according to scroll direction */
  if( event->scroll.direction == GDK_SCROLL_UP )
	data++;
  else if( event->scroll.direction == GDK_SCROLL_DOWN )
	data--;

  Display_Frequency_Spin_Dial( label, data );

  return( TRUE );
} /* Frequency_Scroll_Event() */

/*----------------------------------------------------------------------*/

/* Frequency_Button_Press_Event()
 *
 * Handles button press events on the Frequency "spin dial"
 */
  gboolean
Frequency_Button_Press_Event( GtkWidget *widget, GdkEventButton *event )
{
  const gchar *txt;
  int data;
  GtkLabel *label;

  /* Get the label from the event box widget */
  if( GTK_IS_BIN(widget) )
	label = GTK_LABEL( gtk_bin_get_child(GTK_BIN(widget)) );
  else return( FALSE );

  /* Get text from label and convert to number */
  txt  = gtk_label_get_text( label );
  data = atoi( txt );

  /* Change value of label data
   * according to button pressed */
  if( event->button == 1 )
	data++;
  else if( event->button == 2 )
	data = 0;
  else if( event->button == 3 )
	data--;

  Display_Frequency_Spin_Dial( label, data );

  return( TRUE );
} /* Frequency_Button_Press_Event() */

/*----------------------------------------------------------------------*/

/* Sdrx_Stop()
 *
 * Runs down and stops Sdrx
 */
  void
Sdrx_Stop( void )
{
  ClearFlag( MONITORS_START );
  ClearFlag( PLAYBACK_START );
  ClearFlag( TIME_RECV_START );
  Sdrx_Close_Device();
  ClearFlag( SDRX_INITIALIZED );
  ClearFlag( FFT_INITIALIZED );

} /* Sdrx_Stop() */

/*----------------------------------------------------------------------*/

/* Sdrx_Set_Center_Frequency()
 *
 * Sets the Center Frequency of the Device Tuner
 */
  void
Sdrx_Set_Center_Frequency( uint32_t center_freq )
{
  /* Offset center freq for SSB operation to take
   * account of Weaver fold frequency offset */
  center_freq += rc_data.weaver_offset;

  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  RtlSdr_Set_Center_Frequency( center_freq );
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  SDRPlay_Set_Center_Frequency( center_freq );
	  break;

	case DONGLE_TYPE_PERSEUS:
	  Perseus_Set_Center_Frequency( center_freq );
	  break;
  }

  /* Display center frequency */
  Display_Center_Frequency( rc_data.sdrx_center_freq );
  SetFlag( PLOT_SCALES );

  return;
} /* Sdrx_Set_Center_Frequency() */

/*----------------------------------------------------------------------*/

/* Sdrx_Change_Buffer()
 *
 * Changes the sdrx buffer length and resets sdrx
 */
  void
Sdrx_Change_Buffer( GtkComboBox *combobox )
{
  Clear_Playback_Buf();

  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  RtlSdr_Change_Buffer( combobox );
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  SDRPlay_Change_Buffer( combobox );
	  break;

	case DONGLE_TYPE_PERSEUS:
	  Perseus_Change_Buffer( combobox );
	  break;
  }

  /* Setup FFT Bandwidth combo box */
  Set_FFT_Bandwidth_Combobox( rc_data.sdrx_sample_rate );
  if( rc_data.sdrx_sample_rate ) Init_Roofing_Filter();

} /* Sdrx_Change_Buffer() */

/*----------------------------------------------------------------------*/

/* Sdrx_Set_Tuner_Gain()
 *
 * Sets the Device Tuner's gain
 */
  void
Sdrx_Set_Tuner_Gain( double gain )
{
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  RtlSdr_Set_Tuner_Gain( gain );
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  SDRPlay_Set_Tuner_Gain( gain );
	  break;

	case DONGLE_TYPE_PERSEUS:
	  Perseus_Set_Tuner_Gain( gain );
	  break;
  }

} /* Sdrx_Set_Tuner_Gain() */

/*----------------------------------------------------------------------*/

/* Sdrx_Set_Tuner_Gain_Mode()
 *
 * Sets the Device Tuner's gain control mode
 */
  void
Sdrx_Set_Tuner_Gain_Mode( int mode )
{
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  RtlSdr_Set_Tuner_Gain_Mode( mode );
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  SDRPlay_Set_Tuner_Gain_Mode( mode );
	  break;

	case DONGLE_TYPE_PERSEUS:
	  Perseus_Set_Tuner_Gain_Mode( mode );
	  break;
  }

} /* Sdrx_Set_Tuner_Gain_Mode() */

/*----------------------------------------------------------------------*/

/* Sdrx_Close_Device()
 *
 * Closes and uninitializes the API and the Device
 */
  void
Sdrx_Close_Device( void )
{
  switch( rc_data.dongle_type )
  {
	case DONGLE_TYPE_RTLSDR:
	  RtlSdr_Close_Device();
	  break;

	case DONGLE_TYPE_SDRPLAY:
	  SDRPlay_Close_Device();
	  break;

	case DONGLE_TYPE_PERSEUS:
	  Perseus_Close_Device();
	  break;
  }

} /* Sdrx_Close_Device() */

/*----------------------------------------------------------------------*/

