/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "solar_lunar.h"
#include "shared.h"

/*------------------------------------------------------------------*/

/*  Calculate_Solar_Position()
 *
 *  Calculates solar position vector according to the
 *  algorithm in "Astronomical Algorithms" by Jean Meeus
 */

  void
Calculate_Solar_Position( double time, kinetics_t *solar_kinetics )
{
  double T, M, L, e, C, O, Lsa, nu, R, eps;

  T = ( time - 2451545.0 ) / 36525.0;
  M = 357.52911 + ( 35999.05029 - 0.0001537 * T ) * T;
  M = Radians( M );

  L = 280.46646 + ( 36000.76983 + 0.0003032 * T ) * T;
  L = Radians( L );

  e = 0.016708634 - ( 0.000042037 + 0.0000001267 * T ) * T;

  C = ( 1.914602 - (0.004817 + 0.000014 * T) * T) * sin(M) +
	(0.019993 - 0.000101 * T) * sin(2.0 * M) +
	0.000289 * sin(3.0 * M);
  C = Radians( C );

  O = 125.04 - 1934.136 * T;
  O = Radians( O );

  Lsa = L + C - Radians( 0.00569 ) - Radians( 0.00478 * sin(O) );

  nu = M + C;

  R = 1.000001018 * ( 1.0 - Sqr(e) ) / ( 1.0 + e * cos(nu) );

  eps =
	23.452294 -
	( 0.0130125 +
	  ( 0.00000164 - 0.000000503 * T ) * T ) * T +
	0.00256 * cos(O);
  eps = Radians( eps );

  R = AU * R;

  solar_kinetics->pos.x = R * cos(Lsa);
  solar_kinetics->pos.y = R * sin(Lsa) * cos(eps);
  solar_kinetics->pos.z = R * sin(Lsa) * sin(eps);
  solar_kinetics->pos.w = R;

  /* Solar velocity vector not used at the moment */
  solar_kinetics->vel.x = 0.0;
  solar_kinetics->vel.y = 0.0;
  solar_kinetics->vel.z = 0.0;
  solar_kinetics->vel.w = 0.0;

} /* End of Calculate_Solar_Position() */

/*------------------------------------------------------------------*/

/*  Satellite_Eclipsed()
 *
 *  Calculates stellite's eclipse status and depth
 */

  int
Satellite_Eclipsed(
	kinetics_t *sat_kinetics,
	kinetics_t *sol_kinetics,
	double *depth)
{
  double sd_sun, sd_earth, delta;
  vector_t Rho, earth;

  /* Determine partial eclipse */
  sd_earth = ArcSin(rc_data.radiusearthkm/sat_kinetics->pos.w);
  Vec_Sub( &sol_kinetics->pos, &sat_kinetics->pos, &Rho);
  sd_sun = ArcSin(sr/Rho.w);
  Scalar_Multiply( -1, &sat_kinetics->pos, &earth);
  delta = Angle( &sol_kinetics->pos, &earth);
  *depth = sd_earth - sd_sun - delta;

  if( sd_earth < sd_sun )
	return( 0 );
  else if( *depth >= 0 )
	return( 1 );
  else
	return( 0 );

} /* End of Function Sat _Eclipsed() */

/*------------------------------------------------------------------*/

/*  Lunar_Position()
 *
 *  Calculates azimuth and elevation of moon from observer.
 *  Adapted from code in the 'predict' tracker but is not in use.

  static void
Lunar_Position(
	double daynum,
	geodetic_t *obs_geodetic,
	vector_t *lunar_set)
{
  double
	t1, t2, t3, ew, en, e, n, si, co, ra, gh,
	lh, gm, de, se, gs, dc, el, az, mm, yr, ro;

  gm=24.0*(daynum-floor(daynum));
  ew=FMod2p(1.134193+daynum*0.229971506);
  mm=FMod2p(1.319238+daynum*0.228027135);
  t1=FMod2p(6.217512+daynum*0.01720196977);
  t2=2.0*FMod2p(2.550677+daynum*0.212768711);
  t3=FMod2p(4.7652214+daynum*0.230895723);
  ew=ew+0.01148*sin(t2)+0.10976*sin(mm);
  ew=ew-0.022235*sin(mm-t2)-0.003246*sin(t1);
  ew=ew+0.003735*sin(2.0*mm)-0.0019897*sin(2.0*t3);
  ew=ew-0.0010297*sin(2.0*mm-t2)-0.0009948*sin(mm+t1-t2);
  en=t3+0.0115070*sin(t2)+0.10873924*sin(mm);
  en=en-0.0222006*sin(mm-t2);
  en=0.0897797*sin(en)-0.002548*sin(t3-t2);

  si=zcosis*sin(en)+zsinis*cos(en)*sin(ew);
  co=sqrt(1.0-(si*si));
  dc=atan2(si,co);
  si=sin(ew)*zcosis-tan(en)*zsinis;
  co=cos(ew);
  ra=atan2(si,co);

  if (ra<0.0)
	ra+=twopi;

  t1=(double)(long)(daynum-39410.0);
  t2=floor((t1+32044.75)/36524.25);
  t2+=t1-t2/4.0+1486.0;
  yr=2084.0+floor((t2-122.1)/365.25);

  t1=yr-1.0;
  de=floor(365.25*(t1-1980.0))-floor(t1/100.0)+floor(t1/400.0)+381.0;
  t1=(de+29218.5)/36525.0;
  t1=6.6460656+t1*(2400.051262+t1*2.581e-5);
  se=t1-24.0*(yr-1900.0);
  t1=(se+0.0657098*(daynum-de)+gm*1.002738)/24.0;
  gs=24.0*(t1-floor(t1));

  t1=gs/24.0-ra/twopi;
  gh=twopi*(t1-floor(t1));

  n=obs_geodetic->lat; North latitude of tracking station
  e=obs_geodetic->lon ; East longitude of tracking station

  lh=gh+e;

  si=cos(lh)*cos(dc)*cos(n)+sin(dc)*sin(n);
  co=sqrt(1.0-(si*si));
  el=atan2(si,co);

  si=-sin(lh)*cos(dc)*cos(n);
  co=sin(dc)-sin(n)*sin(el);
  az=atan2(si,co);

  if (az<0.0)
	az+=twopi;

  ro=0.996986/(1.0+0.0549*cos(mm+0.10976*sin(mm)));
  el=el-0.0166*cos(el)/ro;

  lunar_set->x=az;
  lunar_set->y=el;

} End of Lunar_Position() */

/*------------------------------------------------------------------*/

/*  Calculate_Moon_Position()
 *
 *  Calculates azimuth, elevation and range to moon from observer.
 *  Based on an algorithm by Jean Meeus in Astronomical Algorithms.
 */

  void
Calculate_Moon_Position(
	double julian_utc,
	observer_status_t *obs_status,
	kinetics_t *obs_kinetics )
{
  /* Lunar perturbation constants for longitude */
  static const lunar_perturbation_t ptb_long[60] =
  {
	{0,	  0,	1,	 0,	6288774.0},
	{2,	  0,   -1,	 0,	1274027.0},
	{2,	  0,	0,	 0,	 658314.0},
	{0,	  0,	2,	 0,	 213618.0},
	{0,	  1,	0,	 0,	-185116.0},
	{0,	  0,	0,	 2,	-114332.0},
	{2,	  0,   -2,	 0,	  58793.0},
	{2,  -1,   -1,	 0,	  57066.0},
	{2,	  0,	1,	 0,	  53322.0},
	{2,  -1,	0,	 0,	  45758.0},
	{0,	  1,   -1,	 0,	 -40923.0},
	{1,	  0,	0,	 0,	 -34720.0},
	{0,	  1,	1,	 0,	 -30383.0},
	{2,	  0,	0,	-2,	  15327.0},
	{0,	  0,	1,	 2,	 -12528.0},
	{0,	  0,	1, 	-2,	  10980.0},
	{4,	  0,   -1,	 0,	  10675.0},
	{0,	  0,	3,	 0,	  10034.0},
	{4,	  0,   -2,	 0,	   8548.0},
	{2,	  1,   -1,	 0,	  -7888.0},
	{2,	  1,	0,	 0,	  -6766.0},
	{1,	  0,   -1,	 0,	  -5163.0},
	{1,	  1,	0,	 0,	   4987.0},
	{2,  -1,	1,	 0,	   4036.0},
	{2,	  0,	2,	 0,	   3994.0},
	{4,	  0,	0,	 0,	   3861.0},
	{2,	  0,   -3,	 0,	   3665.0},
	{0,	  1,   -2,	 0,	  -2689.0},
	{2,	  0,   -1,	 2,	  -2602.0},
	{2,  -1,   -2,	 0,	   2390.0},
	{1,	  0,	1,	 0,	  -2348.0},
	{2,  -2,	0,	 0,	   2236.0},
	{0,	  1,	2,	 0,	  -2120.0},
	{0,	  2,	0,	 0,	  -2069.0},
	{2,  -2,   -1,	 0,    2048.0},
	{2,	  0,	1, 	-2,	  -1773.0},
	{2,	  0,	0,	 2,	  -1595.0},
	{4,  -1,   -1,	 0,	   1215.0},
	{0,	  0,	2,	 2,	  -1110.0},
	{3,	  0,   -1,	 0,	   -892.0},
	{2,	  1,	1,	 0,	   -810.0},
	{4,  -1,   -2,	 0,     759.0},
	{0,	  2,   -1,	 0,    -713.0},
	{2,	  2,   -1,	 0,	   -700.0},
	{2,	  1,   -2,	 0,	    691.0},
	{2,  -1,	0, 	-2,	    596.0},
	{4,	  0,	1,	 0,		549.0},
	{0,	  0,	4,	 0,	    537.0},
	{4,  -1,	0,	 0,	    520.0},
	{1,	  0,   -2,	 0,	   -487.0},
	{2,	  1,	0, 	-2,	   -399.0},
	{0,	  0,	2, 	-2,	   -381.0},
	{1,	  1,	1,	 0,	    351.0},
	{3,	  0,   -2,	 0,	   -340.0},
	{4,	  0,   -3,	 0,	    330.0},
	{2,  -1,	2,	 0,	    327.0},
	{0,	  2,	1,	 0,	   -323.0},
	{1,	  1,   -1,	 0,	    299.0},
	{2,	  0,	3,	 0,	    294.0},
	{2,	  0,   -1, 	-2,	      0.0}
  };

  /* Lunar perturbation constants for distance */
  static const lunar_perturbation_t ptb_dist[60] =
  {
	{0,	  0,	1,	 0,-20905355.0},
	{2,	  0,   -1,	 0, -3699111.0},
	{2,	  0,	0,	 0,	-2955968.0},
	{0,	  0,	2,	 0,	 -569925.0},
	{0,	  1,	0,	 0,	   48888.0},
	{0,	  0,	0,	 2,	   -3149.0},
	{2,	  0,   -2,	 0,	  246158.0},
	{2,  -1,   -1,	 0,	 -152138.0},
	{2,	  0,	1,	 0,	 -170733.0},
	{2,  -1,	0,	 0,	 -204586.0},
	{0,	  1,   -1,	 0,	 -129620.0},
	{1,	  0,	0,	 0,	  108743.0},
	{0,	  1,	1,	 0,	  104755.0},
	{2,	  0,	0, 	-2,	   10321.0},
	{0,	  0,	1,	 2,	       0.0},
	{0,	  0,	1,	-2,	   79661.0},
	{4,	  0,   -1,	 0,	  -34782.0},
	{0,	  0,	3,	 0,	  -23210.0},
	{4,	  0,   -2,	 0,	  -21636.0},
	{2,	  1,   -1,	 0,	   24208.0},
	{2,	  1,	0,	 0,	   30824.0},
	{1,	  0,   -1,	 0,	   -8379.0},
	{1,	  1,	0,	 0,	  -16675.0},
	{2,  -1,	1,	 0,	  -12831.0},
	{2,	  0,	2,	 0,	  -10445.0},
	{4,	  0,	0,	 0,	  -11650.0},
	{2,	  0,   -3,	 0,	   14403.0},
	{0,	  1,   -2,	 0,	   -7003.0},
	{2,	  0,   -1,	 2,	       0.0},
	{2,  -1,   -2,	 0,	   10056.0},
	{1,	  0,	1,	 0,	    6322.0},
	{2,  -2,	0,	 0,	   -9884.0},
	{0,	  1,	2,	 0,	    5751.0},
	{0,	  2,	0,	 0,	       0.0},
	{2,  -2,   -1,	 0,	   -4950.0},
	{2,	  0,	1, 	-2,	    4130.0},
	{2,	  0,	0,	 2,	       0.0},
	{4,  -1,   -1,	 0,	   -3958.0},
	{0,	  0,	2,	 2,	       0.0},
	{3,	  0,   -1,	 0,	    3258.0},
	{2,	  1,	1,	 0,	    2616.0},
	{4,  -1,   -2,	 0,	   -1897.0},
	{0,	  2,   -1,	 0,	   -2117.0},
	{2,	  2,   -1,	 0,	    2354.0},
	{2,	  1,   -2,	 0,	       0.0},
	{2,  -1,	0, 	-2,	       0.0},
	{4,	  0,	1,	 0,	   -1423.0},
	{0,	  0,	4,	 0,	   -1117.0},
	{4,  -1,	0,	 0,	   -1571.0},
	{1,	  0,   -2,	 0,	   -1739.0},
	{2,	  1,	0, 	-2,	       0.0},
	{0,	  0,	2, 	-2,	   -4421.0},
	{1,	  1,	1,	 0,	       0.0},
	{3,	  0,   -2,	 0,	       0.0},
	{4,	  0,   -3,	 0,	       0.0},
	{2,  -1,	2,	 0,	       0.0},
	{0,	  2,	1,	 0,	    1165.0},
	{1,	  1,   -1,	 0,	       0.0},
	{2,	  0,	3,	 0,	       0.0},
	{2,	  0,   -1, 	-2,	    8752.0}
  };

  /* Lunar perturbation constants for latitude */
  static const lunar_perturbation_t ptb_latd[60] =
  {
	{0,   0,	0,	 1,	5128122.0},
	{0,	  0,	1,	 1,	 280602.0},
	{0,	  0,	1, 	-1,	 277693.0},
	{2,	  0,	0, 	-1,	 173237.0},
	{2,	  0,   -1,	 1,	  55413.0},
	{2,	  0,   -1, 	-1,	  46271.0},
	{2,   0,	0,	 1,	  32573.0},
	{0,   0,	2,	 1,	  17198.0},
	{2,	  0,	1, 	-1,	   9266.0},
	{0,	  0,	2, 	-1,	   8822.0},
	{2,  -1,	0, 	-1,	   8216.0},
	{2,   0,   -2, 	-1,	   4324.0},
	{2,   0,	1, 	 1,	   4200.0},
	{2,	  1,	0, 	-1,	  -3359.0},
	{2,  -1,   -1,	 1,	   2463.0},
	{2,  -1,	0,	 1,	   2211.0},
	{2,  -1,   -1, 	-1,	   2065.0},
	{0,	  1,   -1, 	-1,	  -1870.0},
	{4,	  0,   -1, 	-1,	   1828.0},
	{0,	  1,	0,	 1,	  -1794.0},
	{0,	  0,	0,	 3,	  -1749.0},
	{0,	  1,   -1,	 1,	  -1565.0},
	{1,	  0,	0,	 1,	  -1491.0},
	{0,	  1,	1,	 1,	  -1475.0},
	{0,	  1,	1, 	-1,	  -1410.0},
	{0,	  1,	0, 	-1,	  -1344.0},
	{1,	  0,	0, 	-1,	  -1335.0},
	{0,	  0,	3, 	 1,	   1107.0},
	{4,	  0,	0, 	-1,	   1021.0},
	{4,	  0,   -1,   1,	    833.0},
	{0,	  0,	1, 	-3,	    777.0},
	{4,	  0,   -2,	 1,	    671.0},
	{2,	  0,	0, 	-3,	    607.0},
	{2,	  0,	2, 	-1,	    596.0},
	{2,  -1,	1, 	-1,	    491.0},
	{2,   0,   -2,   1,	   -451.0},
	{0,	  0,	3, 	-1,	    439.0},
	{2,	  0,	2,	 1,	    422.0},
	{2,	  0,   -3, 	-1,	    421.0},
	{2,	  1,   -1,	 1,	   -366.0},
	{2,	  1,	0,	 1,	   -351.0},
	{4,	  0,	0,	 1,	    331.0},
	{2,  -1,	1,	 1,	    315.0},
	{2,  -2,	0, 	-1,	    302.0},
	{0,	  0,	1,   3,	   -283.0},
	{2,	  1,	1, 	-1,	   -229.0},
	{1,	  1,	0, 	-1,	    223.0},
	{1,	  1,	0,	 1,	    223.0},
	{0,	  1,   -2, 	-1,	   -220.0},
	{2,	  1,   -1, 	-1,	   -220.0},
	{1,	  0,	1,	 1,	   -185.0},
	{2,  -1,   -2, 	-1,	    181.0},
	{0,	  1,	2,	 1,	   -177.0},
	{4,	  0,   -2, 	-1,	    176.0},
	{4,  -1,   -1, 	-1,	    166.0},
	{1,	  0,	1, 	-1,	   -164.0},
	{4,	  0,	1, 	-1,	    132.0},
	{1,	  0,   -1, 	-1,	   -119.0},
	{4,  -1,	0, 	-1,	    115.0},
	{2,  -2,	0,	 1,	    107.0}
  };

  /* All the angles (arguments) converted to radians */
  static const double
	Lm_c0 = 218.3164477			* de2ra,
		  Lm_c1 = 481267.88123421		* de2ra,
		  Lm_c2 = -0.0015786			* de2ra,
		  Lm_c3 = 1.0 / 538841.0 		* de2ra,
		  Lm_c4 = 1.0 / 65194000.0	* de2ra,
		  Dm_c0 = 297.8501921			* de2ra,
		  Dm_c1 = 445267.1114034		* de2ra,
		  Dm_c2 = -0.0018819			* de2ra,
		  Dm_c3 = 1.0 / 545868.0		* de2ra,
		  Dm_c4 = 1.0 / 113065000.0	* de2ra,
		  Ms_c0 = 357.5291092			* de2ra,
		  Ms_c1 = 35999.0502909		* de2ra,
		  Ms_c2 = -0.0001536			* de2ra,
		  Ms_c3 = 1.0 / 24490000.0	* de2ra,
		  Mm_c0 = 134.9633964			* de2ra,
		  Mm_c1 = 477198.8675055		* de2ra,
		  Mm_c2 = 0.0087414			* de2ra,
		  Mm_c3 = 1.0 / 69699.0		* de2ra,
		  Mm_c4 = 1.0 / 14712000.0	* de2ra,
		  Fm_c0 = 93.2720950			* de2ra,
		  Fm_c1 = 483202.0175233		* de2ra,
		  Fm_c2 = -0.0036539			* de2ra,
		  Fm_c3 = 1.0 / 3526000.0		* de2ra,
		  Fm_c4 = 1.0 / 863310000.0	* de2ra,
		  A1_c0 = 119.75				* de2ra,
		  A1_c1 = 131.849				* de2ra,
		  A2_c0 = 53.09				* de2ra,
		  A2_c1 = 479264.290			* de2ra,
		  A3_c0 = 313.45				* de2ra,
		  A3_c1 = 481266.484			* de2ra,
		  eps_c0 = ( 23.0 + ( 26.0 + 21.448 / 60.0 ) / 60.0 ) * de2ra,
		  eps_c1 = -46.8150 / 3600.0 * de2ra,
		  eps_c2 = -0.00059 / 3600.0 * de2ra,
		  eps_c3 = 0.001813 / 3600.0 * de2ra;

  /* Terms used in position algorithm */
  double
	Lm, Dm, Ms, Mm, Fm, A1, A2, A3, Ee, Ee2,
	sum_long, ptlon, sum_dist, ptdis, sum_latd, ptlat,
	lambda, sinl, cosl, beta, sinb, cosb, delta;

  /* Terms used to find azimuth and elevation of moon */
  double
	eps, sinep, cosep, ra, dec, theta;

  /* Time in Julian centuries */
  double t;

  int idx, aMs;

  vector_t lunar_pos;

  /* Time in Julian centuries (?!) */
  t = ( julian_utc - 2451545.0 ) / 36525.0;

  /* Moon's mean longitude (in rads) */
  Lm = Lm_c0 + ( Lm_c1 + ( Lm_c2 + ( Lm_c3 + Lm_c4 * t ) * t ) * t ) * t;

  /* Moon's mean elongation (in rads) */
  Dm = Dm_c0 + ( Dm_c1 + ( Dm_c2 + ( Dm_c3 + Dm_c4 * t ) * t ) * t ) * t;

  /* Sun's mean anomaly (in rads) */
  Ms = Ms_c0 + ( Ms_c1 + ( Ms_c2 + Ms_c3 * t ) * t ) * t;

  /* Moon's mean anomaly (in rads) */
  Mm = Mm_c0 + ( Mm_c1 + ( Mm_c2 + ( Mm_c3 + Mm_c4 * t ) * t ) * t ) * t;

  /* Moon's argument of latitude */
  Fm = Fm_c0 + ( Fm_c1 + ( Fm_c2 + ( Fm_c3 + Fm_c4 * t ) * t ) * t ) * t;

  /* The three additional arguments (in rads) */
  A1 = A1_c0 + A1_c1 * t;
  A2 = A2_c0 + A2_c1 * t;
  A3 = A3_c0 + A3_c1 * t;

  /* Eccentricity of earth's orbit and its square */
  Ee  = 1.0 - ( 0.002516 + 0.0000074 * t ) * t;
  Ee2 = Ee * Ee;

  /* Sum of periodic terms for longitude, distance, latitude */
  sum_long = 0.0; sum_dist = 0.0; sum_latd = 0.0;
  for( idx = 0; idx < PERTB_NTERMS; idx++ )
  {
	/* Include the effect of changing eccentricity */
	aMs = abs( ptb_long[idx].Ms );
	ptlon = ptb_long[idx].a;
	ptdis = ptb_dist[idx].a;
	if( aMs == 1 )
	{
	  ptlon *= Ee;
	  ptdis *= Ee;
	}
	else if( aMs == 2 )
	{
	  ptlon *= Ee2;
	  ptdis *= Ee2;
	}

	aMs = abs( ptb_latd[idx].Ms );
	ptlat = ptb_latd[idx].a;
	if( aMs == 1 )
	  ptlat *= Ee;
	else if( aMs == 2 )
	  ptlat *= Ee2;

	/* Summate the perturbation terms */
	sum_long += ptlon * sin(
		(double)ptb_long[idx].Dm * Dm +
		(double)ptb_long[idx].Ms * Ms +
		(double)ptb_long[idx].Mm * Mm +
		(double)ptb_long[idx].Fm * Fm );

	sum_dist += ptdis * cos(
		(double)ptb_dist[idx].Dm * Dm +
		(double)ptb_dist[idx].Ms * Ms +
		(double)ptb_dist[idx].Mm * Mm +
		(double)ptb_dist[idx].Fm * Fm );

	sum_latd += ptlat * sin(
		(double)ptb_latd[idx].Dm * Dm +
		(double)ptb_latd[idx].Ms * Ms +
		(double)ptb_latd[idx].Mm * Mm +
		(double)ptb_latd[idx].Fm * Fm );

  } /* for( idx = 0; idx < PERTB_NTERMS; idx++) */

  /* Additive terms to include effects by
   * Venus, Jupiter and flattening of Earth */
  sum_long +=
	3958.0 * sin(A1) +
	1962.0 * sin(Lm - Fm) +
	318.0 * sin(A2);

  sum_latd +=
	-2235.0 * sin(Lm) +
	382.0 * sin(A3) +
	175.0 * sin(A1 - Fm) +
	175.0 * sin(A1 + Fm) +
	127.0 * sin(Lm - Mm) -
	115.0 * sin(Lm + Mm);

  /* Moon's position */
  lambda = Radians( sum_long / 1.0E6 ) + Lm;
  beta   = Radians( sum_latd / 1.0E6 );
  delta = 385000.56 + sum_dist / 1.0E3;

  /* Obliquity of the ecliptic */
  eps = eps_c0 + ( eps_c1 + ( eps_c2 + eps_c3 * t ) * t ) * t;

  /* Right ascension, declination, distance */
  cosl  = cos(lambda); sinl  = sin(lambda);
  cosep = cos(eps);    sinep = sin(eps);
  cosb  = cos(beta);   sinb  = sin(beta);
  ra    = atan2( sinl * cosep - tan(beta) * sinep, cosl );
  dec   = asin( sinb * cosep + cosb * sinep * sinl );

  /* Topocentric co-ordinates of Moon */
  RaDec_to_Topocentric( ra, dec, &lunar_pos );
  Scale_Vector( delta, &lunar_pos );

  /* Azimuth and elevation of Moon */
  theta = Theta( julian_utc, Degrees(obs_status->obs_geodetic.lon) );
  RaDec_to_AzEl(
	  ra, dec, theta,
	  &obs_status->obs_geodetic,
	  &obs_status->lunar_set );

  obs_status->lunar_set.z = sqrt(
	  Sqr( lunar_pos.x - obs_kinetics->pos.x ) +
	  Sqr( lunar_pos.y - obs_kinetics->pos.y ) +
	  Sqr( lunar_pos.z - obs_kinetics->pos.z ) );

  obs_status->lunar_set.x = Degrees( obs_status->lunar_set.x );
  obs_status->lunar_set.y = Degrees( obs_status->lunar_set.y );

} /* End of Calculate_Moon_Position() */

/*------------------------------------------------------------------------*/

