/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "rotor.h"
#include "shared.h"

/* Serial port File descriptor */
static int serial_fd = 0;

/* Original serial port options */
static struct termios serial_old_options;

/* Runtime config data */
extern rc_data_t rc_data;

/*------------------------------------------------------------------------*/

/*  Open_Rotor_Serial()
 *
 *  Opens Rotor's Serial Port device, returns the file
 *  descriptor on success or -1 on error
 */

  gboolean
Open_Rotor_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */
  int error = 0;

  char mesg[MESG_SIZE];
  size_t s = sizeof(mesg);

  /* Open Serial device, return on error */
  serial_fd = open( rc_data.rotor_serial, O_RDWR | O_NOCTTY );
  if( serial_fd < 0 )
  {
	serial_fd = 0;
	snprintf( mesg, s,
		_("%s: Unable to open device.\n"\
		  "Please quit and correct"),
		rc_data.rotor_serial );
	Error_Dialog( mesg, NONFATAL );
	return( FALSE );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  error = fcntl( serial_fd, F_SETLK, &lockinfo );
  if( error == -1 )
  {
	if( fcntl(serial_fd, F_GETLK, &lockinfo) != -1 )
	{
	  snprintf( mesg, s,
		  _("%s: Device already\n"\
			"locked by pid %d\n%s"),
		  rc_data.rotor_serial,
		  lockinfo.l_pid,
		  _("Please quit and correct") );
	  Error_Dialog( mesg, FATAL );
	}
	close( serial_fd );
	serial_fd = 0;
	return( FALSE );
  }

  /* Save the current serial port options */
  error = 0;
  error |= tcgetattr( serial_fd, &serial_old_options );

  /* Read the current serial port options */
  error |= tcgetattr( serial_fd, &new_options );

  /* Set the i/o baud rates to 9600 */
  error |= cfsetispeed( &new_options, B9600 );
  error |= cfsetospeed( &new_options, B9600 );

  /* Set options for 'raw' I/O mode */
  new_options.c_iflag &= (unsigned int)
	( ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON) );
  new_options.c_oflag &= (unsigned int)( ~OPOST );
  new_options.c_lflag &= (unsigned int)
	( ~( ECHO | ECHONL | ICANON | ISIG | IEXTEN) );
  new_options.c_cflag &= (unsigned int) ( ~(CSIZE | PARENB | CSTOPB) );
  new_options.c_cflag |= (unsigned int)( (CS8 | CLOCAL | CREAD) );

  /* Setup read() timeout to .5 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 5;

  /* Setup the new options for the port */
  error |= tcsetattr( serial_fd, TCSAFLUSH, &new_options );

  /* Disable Rotor control */
  if( error )
  {
	tcsetattr( serial_fd, TCSANOW, &serial_old_options );
	close( serial_fd );
	serial_fd = 0;
	return( FALSE );
  }

  return( TRUE );
} /* End of Open_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Rotor_Serial()
 *
 *  Reading Data from the Rotor's Serial Port
 */

  static char
Read_Rotor_Serial(void)
{
  char ch;

  if( read( serial_fd, &ch, 1 ) < 1 ) /* Error condition */
  {
	SetFlag( ROTOR_SERIAL_FAIL );
	return( -1 );
  }
  else
  {
	ClearFlag( ROTOR_SERIAL_FAIL );
	return( ch );
  }

} /* End of Read_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Close_Rotor_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Rotor_Serial( void )
{
  if( serial_fd > 0 )
  {
	tcsetattr( serial_fd, TCSANOW, &serial_old_options );
	close( serial_fd );
	serial_fd = 0;
  }

} /* End of Close_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Rotor_Serial()
 *
 *  Writing Data to the Rotor's Serial Port
 */

  static int
Write_Rotor_Serial( char *ch )
{
  if( write( serial_fd, ch, 1 ) < 1 ) /* Error condition */
  {
	SetFlag( ROTOR_SERIAL_FAIL );
	return( -1 );
  }
  else
  {
	ClearFlag( ROTOR_SERIAL_FAIL );
	return( 0 );
  }

} /* End of Write_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Rotor_Send_String()
 *
 *  Function to send a command or data string to
 *  the Yaesu GS-232 Az-El antenna rotor Controller
 */

  void
Rotor_Send_String( char *strn )
{
  int idx, len;  /* Index for loops etc */
  char ret = CR; /* CRURN keycode */

  len = (int)strlen( strn );
  for( idx = 0; idx < len; idx++ )
	Write_Rotor_Serial( &strn[ idx ] );

  /* Send a CRURN after string */
  Write_Rotor_Serial( &ret );

  /* Wait for CR response from controller */
  usleep(10000);

} /* End of Send_String */

/*------------------------------------------------------------------------*/

/*  Rotor_Read_String()
 *
 *  Function to read a data string from the
 *  Yaesu GS-232 Az-El antenna rotor Controller
 */

  void
Rotor_Read_String( char *strn, int len )
{
  char chr;
  int idx = 0;

  /* Read characters until CR is received. Ignore LF */
  while( ((chr = Read_Rotor_Serial()) != -1) && (chr != CR) && (idx < len) )
	if( chr != LF )
	  strn[ idx++ ] = chr;

  strn[ idx ]= '\0';

} /* End of Read_String */

/*------------------------------------------------------------------------*/

/*  Rotor_Send_Direction()
 *
 *  Send azimuth and elevation request to GS232 controller
 */

  void
Rotor_Send_Direction( int azim, int elev )
{
  /* String to enter Azim/Elev data  */
  /* for sending to GS232 controller */
  char strn[9];

  /* Correct azimuth value with the offset constant,   */
  /* which is due to some difference between the G5500 */
  /* Az/El rotor position transducer output volts and  */
  /* the volts expected by the GS232 firmware. (It now */
  /* seems that the GS232 treats 0 volts as Az 180 deg)*/
  azim += rc_data.azim_offset;

  /* Enter position data in 'strn' in the */
  /* 'W' command's format for sending out */
  strn[0] = 'W';
  snprintf( &strn[1], 4, "%03d", azim );
  strn[4] = ' ';
  snprintf( &strn[5], 4, "%03d", elev );
  Rotor_Send_String( strn );

} /* End of Rotor_Send_Direction() */

/*------------------------------------------------------------------------*/

/*  Rotor_Read_Direction()
 *
 *  Read azimuth and elevation data from GS232 controller
 */

  void
Rotor_Read_Direction( int *azim, int *elev )
{
  /* String to enter Azim/Elev data */
  /* read from GS232 controller     */
  char strn[11];

  /* Temporary buffer for cloning Az/El from strn */
  char buff[6];
  size_t s = sizeof(buff);

  /* Request and read rotor position string */
  Rotor_Send_String( "C2" );
  Rotor_Read_String( strn, 11 );

  /* Return parking values for Az/El if serial read fails */
  if( isFlagSet(ROTOR_SERIAL_FAIL) )
  {
	*azim = rc_data.azim_parking;
	*elev = rc_data.elev_parking;
  }
  else
  {
	/* Clone Azimuth value */
	Strlcpy( buff, strn, s );
	*azim = (int)( Modulus(atoi(buff) + rc_data.azim_offset, 360) );

	/* Clone Elevation value */
	Strlcpy( buff, &strn[5], s );
	*elev = atoi( buff );

  }  /* End of if( isFlagSet(ROTOR_SERIAL_FAIL) ) */

} /* End of Rotor_Read_Direction() */

/*------------------------------------------------------------------------*/

/*  Enable_Tracking()
 *
 *  Enables tracking of the selected satellite
 */

  void
Enable_Tracking( void )
{
  /* Open rotor's serial port if closed */
  if( isFlagClear(ROTORS_ENABLED) )
	if( Open_Rotor_Serial() )
	{
	  SetFlag(ROTORS_ENABLED | TRACKING_ENABLED);
	  ClearFlag( ROTORS_RUNNING );
	  Rotor_Send_String( "S" ); /* All stop */
	}

} /* Enable_Tracking() */

/*------------------------------------------------------------------*/

/*  Disable_Tracking()
 *
 *  Disables tracking of the selected satellite
 */

  void
Disable_Tracking( void )
{
  Close_Rotor_Serial();

  ClearFlag(
	  ROTORS_ENABLED   |
	  TRACKING_ENABLED |
	  REVERSE_TRACK    |
	  DIRECTION_ERROR );

} /* Disable_Tracking() */

/*------------------------------------------------------------------*/

