/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "observer.h"
#include "shared.h"

/*------------------------------------------------------------------*/

/*  Calculate_User_PosVel()
 *
 *  Function Calculate_User_PosVel passes the user's geodetic position
 *  and the time of interest and returns the ECI position and velocity
 *  of the observer. The velocity calculation assumes the geodetic
 *  position is stationary relative to the earth's surface.
 *  Reference:  The 1992 Astronomical Almanac, page K11.
 */

  void
Calculate_User_PosVel(
	double time,
	observer_status_t *obs_status,
	kinetics_t *obs_kinetics )
{
  double c, sq, achcp;

  obs_status->obs_geodetic.theta =
	FMod2p( ThetaG_JD(time) + obs_status->obs_geodetic.lon );
  c = 1.0 / sqrt(1.0 + f * (f - 2.0) * Sqr(sin(obs_status->obs_geodetic.lat)));
  sq = Sqr(1.0 - f) * c;
  achcp = (rc_data.radiusearthkm * c + obs_status->obs_geodetic.hgt) *
	cos(obs_status->obs_geodetic.lat);
  obs_kinetics->pos.x = achcp * cos(obs_status->obs_geodetic.theta);
  obs_kinetics->pos.y = achcp * sin(obs_status->obs_geodetic.theta);
  obs_kinetics->pos.z = (rc_data.radiusearthkm * sq + obs_status->obs_geodetic.hgt) *
	sin(obs_status->obs_geodetic.lat);
  obs_kinetics->vel.x = -mfactor * obs_kinetics->pos.y;
  obs_kinetics->vel.y =  mfactor * obs_kinetics->pos.x;
  obs_kinetics->vel.z =  0.0;

  Magnitude( &obs_kinetics->pos );
  Magnitude( &obs_kinetics->vel );

}  /* End of Calculate_User_PosVel() */

/*------------------------------------------------------------------*/

/*  Calculate_LatLonAlt()
 *
 *  Function Calculate_LatLonAlt will calculate the geodetic
 *  position of an object given its ECI position pos and time.
 *  It is intended to be used to determine the ground track of
 *  a satellite.  The calculations  assume the earth to be an
 *  oblate spheroid as defined in WGS '72.
 *  Reference:  The 1992 Astronomical Almanac, page K12.
 */

  static void
Calculate_LatLonAlt(
	double time,
	vector_t *pos,
	geodetic_t *geodetic )
{
  double r, e2, phi, c;

  geodetic->theta = AcTan(pos->y, pos->x);
  geodetic->lon = FMod2p(geodetic->theta - ThetaG_JD(time));
  r = sqrt(Sqr(pos->x) + Sqr(pos->y));
  e2 = f*(2.0 - f);
  geodetic->lat = AcTan(pos->z, r);

  do
  {
	phi = geodetic->lat;
	c = 1.0/sqrt(1.0 - e2*Sqr(sin(phi)));
	geodetic->lat = AcTan(pos->z + rc_data.radiusearthkm*c*e2*sin(phi),r);
  }
  while(fabs(geodetic->lat - phi) >= 1E-10);

  geodetic->hgt = r/cos(geodetic->lat) - rc_data.radiusearthkm*c;

  if( geodetic->lat > pio2 )
	geodetic->lat -= twopi;

} /* End of Calculate_LatLonAlt() */

/*------------------------------------------------------------------*/

/*  Calculate_Observer()
 *
 *  The procedures Calculate_Observer and Calculate_RADec calculate
 *  the *topocentric* coordinates of the object with ECI position,
 *  {pos}, and velocity, {vel}, from location {geodetic} at {time}.
 *  The {obs_set} returned for Calculate_Observer consists of azimuth,
 *  elevation, range, and range rate (in that order) with units of
 *  radians, radians, kilometers, and kilometers/second, respectively.
 *  The WGS '72 geoid is used.
 *
 *  The {obs_set} for Calculate_RADec consists of right ascension and
 *  declination (in that order) in radians.  Again, calculations are
 *  based on *topocentric* position using the WGS '72 geoid.
 */

  void
Calculate_Observer(
	kinetics_t *sat_kinetics,
	observer_status_t *obs_status,
	kinetics_t *obs_kinetics,
	vector_t *obs_set )
{
  double
	sin_lat,   cos_lat,
	sin_theta, cos_theta,
	el, azim,
	top_s, top_e, top_z;

  vector_t range, rgvel;


  Vec_Sub( &sat_kinetics->pos, &obs_kinetics->pos, &range );
  Vec_Sub( &sat_kinetics->vel, &obs_kinetics->vel, &rgvel );

  Magnitude( &range );

  sin_lat   = sin(obs_status->obs_geodetic.lat);
  cos_lat   = cos(obs_status->obs_geodetic.lat);
  sin_theta = sin(obs_status->obs_geodetic.theta);
  cos_theta = cos(obs_status->obs_geodetic.theta);

  top_s = sin_lat*cos_theta*range.x + sin_lat*sin_theta*range.y - cos_lat*range.z;
  top_e = -sin_theta*range.x + cos_theta*range.y;
  top_z = cos_lat*cos_theta*range.x + cos_lat*sin_theta*range.y + sin_lat*range.z;

  azim = atan(-top_e/top_s); /*Azimuth*/
  if( top_s > 0.0 ) azim = azim + pi;
  if(  azim < 0.0 ) azim = azim + twopi;

  el = ArcSin(top_z/range.w);

  obs_set->x = azim;      /* Azimuth (radians)  */
  obs_set->y = el;        /* Elevation (radians)*/
  obs_set->z = range.w;   /* Range (kilometers) */

  /* Range Rate (kilometers/second) */
  obs_set->w = Dot(&range, &rgvel)/range.w;

} /* End of Calculate_Observer() */

/*------------------------------------------------------------------*/

/*  Calculate_RADec()
 *
 *  See comments at start of Calculate_Obs()
 *
 *  Reference:  Methods of Orbit Determination by
 *  Pedro Ramon Escobal, pp. 401-402

  static void
Calculate_RADec(
	double time,
	kinetics_t *sat_kinetics,
	observer_status_t *obs_status,
	kinetics_t *obs_kinetics,
	vector_t *obs_set )
{
  Calculate_Observer(sat_kinetics, obs_status, obs_kinetics, obs_set);

  if( obs_set->y > 0. )
  {
	double
	  phi, theta, sin_theta, cos_theta, sin_phi, cos_phi,
	  az, el, Lxh, Lyh, Lzh, Sx, Ex, Zx, Sy, Ey, Zy, Sz, Ez, Zz,
	  Lx, Ly, Lz, cos_delta, sin_alpha, cos_alpha;

	az = obs_set->x;
	el = obs_set->y;
	phi   = obs_status->obs_geodetic.lat;
	theta = FMod2p(ThetaG_JD(time) + obs_status->obs_geodetic.lon);
	sin_theta = sin(theta);
	cos_theta = cos(theta);
	sin_phi = sin(phi);
	cos_phi = cos(phi);
	Lxh = -cos(az)*cos(el);
	Lyh =  sin(az)*cos(el);
	Lzh =  sin(el);
	Sx = sin_phi*cos_theta;
	Ex = -sin_theta;
	Zx = cos_theta*cos_phi;
	Sy = sin_phi*sin_theta;
	Ey = cos_theta;
	Zy = sin_theta*cos_phi;
	Sz = -cos_phi;
	Ez = 0;
	Zz = sin_phi;
	Lx = Sx*Lxh + Ex*Lyh + Zx*Lzh;
	Ly = Sy*Lxh + Ey*Lyh + Zy*Lzh;
	Lz = Sz*Lxh + Ez*Lyh + Zz*Lzh;
	obs_set->y = ArcSin(Lz);  Declination (radians)
	cos_delta = sqrt(1 - Sqr(Lz));
	sin_alpha = Ly/cos_delta;
	cos_alpha = Lx/cos_delta;
	obs_set->x = AcTan(sin_alpha,cos_alpha); Right Ascension (radians)
	obs_set->x = FMod2p(obs_set->x);

  }

}  End of Calculate_RADec(obs_set->y>0.) */

/*------------------------------------------------------------------*/

/*  RaDec_to_AzEl()
 *
 *  Trasformazione dal sistema equatoriale celeste RA, Dec a quello
 *  altazimuthale locale AZ, EL; angoli in radianti, theta tempo
 *  siderale locale.
 */

  void
RaDec_to_AzEl(
	double ra,
	double dec,
	double theta,
	geodetic_t *obs_geodetic,
	vector_t *lunar_set )
{
  double h;

  h = theta - ra;  /* Angolo orario */

  lunar_set->x = atan2(
	  sin(h), cos(h) * sin(obs_geodetic->lat) -
	  tan(dec) * cos(obs_geodetic->lat) ) + pi;

  lunar_set->y = asin(
	  sin(obs_geodetic->lat) * sin(dec) +
	  cos(obs_geodetic->lat) * cos(dec) * cos(h) );

} /* End of RaDec_to_AzEl() */

/*------------------------------------------------------------------------*/

/*  RaDec_to_Topocentric()
 *
 *  Trasformazione dal sistema equatoriale celeste RA, Dec a quello
 *  topocentrico IJK; angoli in radianti, la distanza dal centro del
 *  sistema topocentrico e` posta unitaria.
 */

  void
RaDec_to_Topocentric( double ra, double dec, vector_t *topocentric )
{
  topocentric->x = cos(ra);
  topocentric->y = sin(ra);
  topocentric->z = sin(dec);

} /* End of RaDec_to_Topocentric() */

/*------------------------------------------------------------------------*/

/*  Find_Horizon_Crossing()
 *
 *  Finds the horizon crossing time of a satellite
 *  as well as a possible crossing of the North line
 */

  static double
Find_Horizon_Crossing(
	double julian_utc,
	element_set_t *sat_elem_set,
	observer_status_t *obs_status,
	satellite_status_t *sat_status )
{
  double
	prev_elev, /* Satellite elevation memory */
	prev_azim, /* Satellite azimuth memory   */
	horizon,   /* Horizon threshold */
	angl_step, /* Azim/Elev angular step in horizon search */
	prev_step, /* Time-step memory */
	time_step; /* Time step of AOS/LOS search */

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity */
	sat_kinetics,   /* Satellite position/velocity */
	solar_kinetics; /*     Solar position/velocity */

  /* Satellite geodetic position */
  geodetic_t sat_geodetic;

  /* Azim/Elev/Range/Range rate of satellite */
  vector_t obs_set;

  /* Convert horizon to rad */
  horizon   = Radians( rc_data.horizon );
  angl_step = Radians( rc_data.angle_step );

  /* Convert time step to days */
  time_step = angl_step / sat_elem_set->no / xmnpda;

  /* Claculate orbital parameters to initialize search */
  Calculate_Orbit(julian_utc, sat_elem_set, &sat_geodetic, &sat_kinetics,
	  sat_status, obs_status, &obs_kinetics, &obs_set);

  /* Loop until horizon is crossed (new elev has different sign) */
  do
  {
	prev_azim = obs_set.x;
	prev_elev = obs_set.y;

	/* Increase search time-step at high elevations */
	julian_utc += time_step * ( 1. + 2. * fabs(obs_set.y) );

	Calculate_Orbit(julian_utc, sat_elem_set, &sat_geodetic, &sat_kinetics,
		sat_status, obs_status, &obs_kinetics, &obs_set);

	/* Check for visible pass and North crossing */
	/* while satellite is above the horizon      */
	if( (obs_set.y > horizon) && (prev_elev > horizon) )
	{
	  /* Calculate topocentric position of sun */
	  Calculate_Solar_Position( julian_utc, &solar_kinetics );

	  /* Calculate solar position and satellite eclipse depth, */
	  /* set or clear the satellite eclipsed flag accordingly  */
	  Calculate_Observer( &solar_kinetics, obs_status,
		  &obs_kinetics, &obs_status->solar_set );

	  /* Calculate satellite's illumination */
	  if( Satellite_Eclipsed(&sat_kinetics,
			&solar_kinetics,
			&sat_status->edepth) )
		Set_SatFlag( sat_status, SAT_ECLIPSED );
	  else
		Clear_SatFlag( sat_status, SAT_ECLIPSED );

	  /* If sun is 12 deg below horizon, set satellite visible flag */
	  if( (Degrees(obs_status->solar_set.y) < -12.0) &&
		  isSatFlagClear(sat_status, SAT_ECLIPSED) )
		Set_SatFlag( sat_status, PASS_VISIBLE );

	  /* If azimuth changes abruptly by more than 180 deg  */
	  /* then satellite crosses observer's North direction */
	  if( (fabs(prev_azim - obs_set.x) >= pi) )
		Set_SatFlag( sat_status, NORTH_CROSSING );

	} /* End of if( obs_set.y > 0. ) */

  }
  /* Loop until horizon is crossed (new elev has different sign) */
  while( (prev_elev * obs_set.y) > 0. );

  /* Update time-step to last value used */
  time_step *= 1. + 2. * fabs(prev_elev);

  /* Step back closer to horizon */
  time_step *= -fabs( obs_set.y / (obs_set.y - prev_elev) );
  julian_utc += time_step;

  /* Store current values */
  prev_elev = obs_set.y;
  prev_step = time_step;

  /* Initialize satellite status for horizon crossing iteration */
  Calculate_Orbit(julian_utc, sat_elem_set, &sat_geodetic, &sat_kinetics,
	  sat_status, obs_status, &obs_kinetics, &obs_set);

  /* Refine horizon crossing time by looping */
  /* while |elevation| >= horizon threshold  */
  while( fabs(obs_set.y) >= horizon )
  {
	/* Recalculate time step size as elev/elev rate */
	time_step *= fabs( obs_set.y / (obs_set.y - prev_elev) );

	/* Do not allow time-step to increase. This prevents   */
	/* instability and crashes with high elliptical orbits */
	if( fabs(time_step) > fabs(prev_step) )
	  time_step = prev_step;

	/* When horizon is crossed, reverse time-step */
	if( (prev_elev * obs_set.y) < 0. )
	  time_step = -time_step;

	julian_utc += time_step;

	/* Store current values */
	prev_elev = obs_set.y;
	prev_step = time_step;

	/* Calculate new satellite status */
	Calculate_Orbit(julian_utc, sat_elem_set, &sat_geodetic, &sat_kinetics,
		sat_status, obs_status, &obs_kinetics, &obs_set);
  }

  return( julian_utc );

} /* End of Find_Horizon_Crossing() */

/*------------------------------------------------------------------*/

/*  Find_Next_AOS_LOS()
 *
 *  Finds the next AOS/LOS of a satellite, after julian_utc
 */

  void
Find_Next_AOS_LOS(
	double julian_utc,
	element_set_t *sat_elem_set,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  double
	time_step,  /* Time step in min during AOS/LOS search */
	horizon;    /* Horizon threshold ( about 0.03 deg )   */

  /* Azim/Elev/Range/Range rate of satellite */
  vector_t obs_set;

  kinetics_t
	obs_kinetics, /*  Observer position/velocity */
	sat_kinetics; /* Satellite position/velocity */

  /* Satellite geodetic position */
  geodetic_t sat_geodetic;

  /* Convert HORIZON to rad, TIME_STEP to days */
  horizon = Radians( rc_data.horizon );
  time_step = rc_data.time_step/xmnpda;

  /* Calculate satellite status */
  Calculate_Orbit( julian_utc, sat_elem_set,
	  &sat_geodetic, &sat_kinetics,
	  sat_status, obs_status,
	  &obs_kinetics, &obs_set );

  /* If AOS/LOS is possible, find next AOS/LOS */
  if( isSatFlagClear(sat_status, NO_AOS_LOS) )
  {
	/* Move away from horizon */
	while( fabs(obs_set.y) <= horizon )
	{
	  julian_utc += time_step;
	  Calculate_Orbit( julian_utc, sat_elem_set,
		  &sat_geodetic, &sat_kinetics,
		  sat_status, obs_status,
		  &obs_kinetics, &obs_set );
	}

	/* If elev < 0 then search for next AOS, otherwise for LOS */
	if( obs_set.y < -horizon )
	{
	  /* Time of a horizon crossing event */
	  double event_time;

	  /* Find Next AOS time */
	  event_time = Find_Horizon_Crossing( julian_utc, sat_elem_set,
		  obs_status, sat_status );
	  sat_status->aos = event_time;

	  /* Move away from horizon */
	  do
	  {
		event_time += time_step;
		Calculate_Orbit( event_time, sat_elem_set,
			&sat_geodetic, &sat_kinetics,
			sat_status, obs_status,
			&obs_kinetics, &obs_set);
	  }
	  while( fabs(obs_set.y) <= horizon );

	  /* Find next LOS time */
	  Clear_SatFlag( sat_status, NORTH_CROSSING );
	  Clear_SatFlag( sat_status, PASS_VISIBLE );

	  event_time = Find_Horizon_Crossing( event_time, sat_elem_set,
		  obs_status, sat_status );
	  sat_status->los = event_time;
	}
	else if( obs_set.y > horizon )
	{
	  /* Time of a horizon crossing event */
	  double event_time;

	  /* Find Next LOS time */
	  Clear_SatFlag( sat_status, NORTH_CROSSING );
	  Clear_SatFlag( sat_status, PASS_VISIBLE );

	  event_time = Find_Horizon_Crossing( julian_utc, sat_elem_set,
		  obs_status, sat_status );
	  sat_status->los = event_time;

	  /* Move away from horizon */
	  do
	  {
		event_time += time_step;
		Calculate_Orbit( event_time, sat_elem_set,
			&sat_geodetic, &sat_kinetics,
			sat_status, obs_status,
			&obs_kinetics, &obs_set );
	  }
	  while( fabs(obs_set.y) <= horizon );

	  /* Find Next AOS time */
	  event_time = Find_Horizon_Crossing( event_time, sat_elem_set,
		  obs_status, sat_status );
	  sat_status->aos = event_time;
	}

  } /* End of if( isSatFlagClear(&sat_status, NO_AOS_LOS) ) */

} /* Find_Next_AOS_LOS() */

/*------------------------------------------------------------------*/

/*  Calculate_Orbit()
 *
 *  Calculates all data relevant to the position and velocity
 *  of a satellite and its position relative to the observer
 */

  void
Calculate_Orbit(
	double julian_utc,
	element_set_t *sat_elem_set,
	geodetic_t *sat_geodetic,
	kinetics_t *sat_kinetics,
	satellite_status_t *sat_status,
	observer_status_t *obs_status,
	kinetics_t *obs_kinetics,
	vector_t *obs_set )
{
  double
	age,    /* Days since epoch    */
	tsince; /* Minutes since epoch */

  /* Calculate observer topocentric position and velocity */
  Calculate_User_PosVel( julian_utc, obs_status, obs_kinetics );

  /* Calculate time since epoch in minutes */
  age = julian_utc - sat_elem_set->jdsatepoch;
  tsince = age * xmnpda;

  /* Call str6 sgp4() routine */
  sgp4( sat_elem_set, tsince, sat_kinetics );

  /* Scale position and velocity vectors to km and km/sec */
  Convert_Sat_State( sat_kinetics );

  /* Calculate scalar magnitude of velocity */
  Magnitude( &sat_kinetics->vel );

  /** All angles in rads. Distance in km. Velocity in km/s **/
  /* Calculate satellite Azi, Ele, Range and Range-rate */
  Calculate_Observer(sat_kinetics, obs_status, obs_kinetics, obs_set);

  /* Calculate satellite Lat North, Lon East and Alt. */
  Calculate_LatLonAlt( julian_utc, &sat_kinetics->pos, sat_geodetic );

  /* Calculate satellite phase in units/256 */
  sat_status->phs256 = sat_kinetics->phase * 256.0 / twopi;

} /* End of Calculate_Orbit() */

/*------------------------------------------------------------------*/

