/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "mode.h"
#include "shared.h"

/*------------------------------------------------------------------*/

/*  Calculate_Satellite_Status()
 *
 *  Does all the calculations needed to
 *  fill the satellite status structure
 */

  static void
Calculate_Satellite_Status(
	double julian_utc,
	element_set_t *sat_elem_set,
	kinetics_t *solar_kinetics,
	kinetics_t *obs_kinetics,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  double
	horizon, /* Horizon threshold */
	age;     /* Elapsed time after sat epoch */

  /* Satellite's predicted geodetic position */
  geodetic_t sat_geodetic;

  /* Satellite Azim, Elev, Range, Range rate */
  vector_t obs_set;

  /* Satellite position/velocity vectors */
  kinetics_t sat_kinetics;

  horizon = Radians(rc_data.horizon);

  /* Calculate all satellite orbital data */
  Calculate_Orbit( julian_utc, sat_elem_set,
	  &sat_geodetic, &sat_kinetics,
	  sat_status, obs_status,
	  obs_kinetics, &obs_set );

  /* Calculate solar position and satellite eclipse depth, */
  /* set or clear the satellite eclipsed flag accordingly  */
  Calculate_Observer( solar_kinetics, obs_status,
	  obs_kinetics, &obs_status->solar_set );

  /* Calculate satellite's illumination */
  if( Satellite_Eclipsed(&sat_kinetics, solar_kinetics, &sat_status->edepth) )
	Set_SatFlag( sat_status, SAT_ECLIPSED );
  else
	Clear_SatFlag( sat_status, SAT_ECLIPSED );

  /* If sun is 12 deg below horizon, set satellite visible flag */
  if( (Degrees(obs_status->solar_set.y) < -12.0) &&
	  (obs_set.y > -horizon) 			 &&
	  isSatFlagClear(sat_status, SAT_ECLIPSED) )
	Set_SatFlag( sat_status, SAT_VISIBLE );
  else
	Clear_SatFlag( sat_status, SAT_VISIBLE );

  /* Find next AOS/LOS in real time after horizon crossing */
  if( sat_status->elev * obs_set.y <= 0. )
	Find_Next_AOS_LOS(julian_utc, sat_elem_set, obs_status, sat_status);

  /* Calculate and fill in satellite status data */
  sat_status->azim    = Degrees(obs_set.x);
  sat_status->elev    = Degrees(obs_set.y);
  sat_status->range   = obs_set.z;
  sat_status->ploss   = 32.4 + 20.0 * log10(obs_set.z);
  sat_status->rrate   = obs_set.w;
  sat_status->doppler = (-sat_status->rrate * 1000.0) / Cl;
  sat_status->ssplat  = Degrees(sat_geodetic.lat);
  sat_status->ssplon  = Degrees(sat_geodetic.lon);
  sat_status->alt     = sat_geodetic.hgt;
  sat_status->ftprnt  = Footprint( &sat_kinetics.pos );
  sat_status->vel     = sat_kinetics.vel.w;
  age				  = julian_utc - sat_elem_set->jdsatepoch;
  sat_status->revnum  =
	floor( (sat_elem_set->no * xmnpda / twopi + age * sat_elem_set->bstar) *
		age + sat_elem_set->mo / twopi ) + sat_elem_set->revnum;

} /* End of  Calculate_Satellite_Status() */

/*------------------------------------------------------------------*/

/*  Single_Satellite_Mode()
 *
 *  Does all the calculations needed to fill the
 *  satellite and observer status structures and
 *  to display single observer/satellite situation
 */

  static void
Single_Satellite_Mode(
	element_set_t *sat_elem_set,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  /* First function call flag */
  static int first_call = TRUE;

  /* Julian UTC */
  double julian_utc;

  kinetics_t
	obs_kinetics,   /* Observer position/velocity vectors */
	solar_kinetics; /* Solar position/velocity vectors */

  /* Initialize */
  if( first_call )
  {
	first_call = FALSE;
	mode_index = 0;
  } /* if( first_call ) */

  /* Change satellite or location */
  switch( change_sat_loc )
  {
	case FORWD_SATELLITE : /* Request next satellite in TLE file */
	  Find_New_Satellite( FORWD_SATELLITE, sat_elem_set, sat_status, obs_status );

	  /* Re-initialize transceiver */
	  if( isFlagSet(CAT_SETUP) )
	  {
		mode_index = 0;
		Read_Transponder_Data( sat_status->name );
		Initialize_Tcvr( DEFAULT_MODE );
	  }
	  break;

	case BACK_SATELLITE : /* Request previous satellite in TLE file */
	  Find_New_Satellite( BACK_SATELLITE, sat_elem_set, sat_status, obs_status );

	  /* Re-initialize transceiver */
	  if( isFlagSet(CAT_SETUP) )
	  {
		mode_index = 0;
		Read_Transponder_Data( sat_status->name );
		Initialize_Tcvr( DEFAULT_MODE );
	  }
	  break;

	case FIRST_SATELLITE : /* Request first satellite in TLE file */
	  Find_New_Satellite( FIRST_SATELLITE, sat_elem_set, sat_status, obs_status );

	  /* Re-initialize transceiver */
	  if( isFlagSet(CAT_SETUP) )
	  {
		mode_index = 0;
		Read_Transponder_Data( sat_status->name );
		Initialize_Tcvr( DEFAULT_MODE );
	  }
	  break;

	case LAST_SATELLITE : /* Request last satellite in TLE file */
	  Find_New_Satellite( LAST_SATELLITE, sat_elem_set, sat_status, obs_status );

	  /* Re-initialize transceiver */
	  if( isFlagSet(CAT_SETUP) )
	  {
		mode_index = 0;
		Read_Transponder_Data( sat_status->name );
		Initialize_Tcvr( DEFAULT_MODE );
	  }
	  break;

	case BACK_OBSERVER : /* Load previous observer data from xsatcom.obs */
	  if( --obs_idx < 0 )
		obs_idx = num_obs - 1;
	  *obs_status = observer_data[obs_idx];

	  /* Pre-process satellite/observer data */
	  New_Status( FALSE, sat_elem_set, sat_status, obs_status );
	  break;

	case FORWD_OBSERVER : /* Load next observer data set from xsatcom.obs */
	  if( ++obs_idx == num_obs )
		obs_idx = 0;
	  *obs_status = observer_data[obs_idx];

	  /* Pre-process satellite/observer data */
	  New_Status( FALSE, sat_elem_set, sat_status, obs_status );
	  break;

	case FIRST_OBSERVER : /* Load first observer data from xsatcom.obs */
	  obs_idx = 0;
	  *obs_status = observer_data[obs_idx];

	  /* Pre-process satellite/observer data */
	  New_Status( FALSE, sat_elem_set, sat_status, obs_status );
	  break;

	case LAST_OBSERVER : /* Load last observer data from xsatcom.obs */
	  obs_idx = num_obs - 1;
	  *obs_status = observer_data[obs_idx];

	  /* Pre-process satellite/observer data */
	  New_Status( FALSE, sat_elem_set, sat_status, obs_status );

  } /* switch( change_sat_loc ) */

  /* Get UTC calendar and convert to Julian */
  Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
  julian_utc = Julian_Date( &obs_status->utc );

  /* Calculate topocentric position of sun */
  Calculate_Solar_Position( julian_utc, &solar_kinetics );

  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
  Calculate_Satellite_Status(
	  julian_utc, sat_elem_set,
	  &solar_kinetics, &obs_kinetics,
	  obs_status, sat_status );

  /* Calculate lunar azimuth and elevation */
  Calculate_Moon_Position( julian_utc, obs_status, &obs_kinetics );

  /* Display satellite status */
  Show_Satellite_Status( sat_status );

  /* Display observer status */
  Show_Observer_Status( satellite_track_builder, SATELLITE_TRACK, obs_status );

  /* Display satellite status with xplanet */
  if( isFlagSet(XPLANET_PLOT) || isFlagSet(MULTISAT_PLOT) )
	Display_Satellite( sat_status );

  /* Do rotor control if enabled */
  if( isFlagSet(ROTORS_ENABLED) )
  {
	/* Tracking resolution in deg */
	int resol = 1;

	if( isFlagSet(TRACK_POLAR) ) /* Track polar star */
	{
	  azimuth = 0.0;
	  elevation = Degrees(obs_status->obs_geodetic.lat);
	}
	else if( isFlagSet(TRACK_SUN) ) /* Track sun */
	{
	  azimuth = Degrees(obs_status->solar_set.x);
	  elevation = Degrees(obs_status->solar_set.y);
	}
	else if( isFlagSet(TRACK_MOON) ) /* Track moon */
	{
	  azimuth = obs_status->lunar_set.x;
	  elevation = obs_status->lunar_set.y;
	}
	else if( isFlagSet(PARK_ROTORS) ) /* Park rotors */
	{
	  azimuth = rc_data.azim_parking;
	  elevation = rc_data.elev_parking;
	}
	else if( isFlagSet(TRACKING_ENABLED) && /* Track a satellite */
		isFlagClear(TRACK_MANUAL) )
	{
	  azimuth = sat_status->azim;
	  elevation = sat_status->elev;
	  resol    = sat_status->resol;

	  if( isSatFlagSet( sat_status, NORTH_CROSSING) )
		SetFlag( REVERSE_TRACK );
	  else
		ClearFlag( REVERSE_TRACK );
	}

	/* Control and display rotor position and status */
	Show_Rotor_Control( azimuth, elevation, resol );

  } /* End of if( isFlagSet(ROTORS_ENABLED) ) */

  /* Activate transceiver CAT control if enabled */
  if( isFlagSet(CAT_SETUP) )
  {
	/* Control FT-847 parameters */
	CAT_Control( sat_status );

	/* Refresh CAT control window */
	Show_CAT_Control( &transceiver_status );
  }

} /* End of Single_Satellite_Mode() */

/*------------------------------------------------------------------*/

/*  Multi_Satellite_Mode()
 *
 *  Does all the calculations needed to fill the
 *  multi-satellite and observer status structures and
 *  to display single-observer/multi-satellite situation
 */

  static void
Multi_Satellite_Mode(
	element_set_t *sat_elem_set,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  /* Multi-sat window */
  static GtkTreeView      *multisat_treeview;
  static GtkTreeSelection *select;

  static GtkBuilder *builder = NULL;

  /* First function call flag */
  static int first_call = TRUE;

  /* Julian UTC */
  double julian_utc;

  int
	vis_sats, /* Number of visible satellites */
	icp,      /* Index for comparison loops   */
	idx;      /* General index for loops etc  */

  /* Satellite name selected */
  static char sat_name[NAME_SIZE];

  kinetics_t
	obs_kinetics,   /* Observer position/velocity vectors */
	solar_kinetics; /*    Solar position/velocity vectors */

  /* Multisatellite tracking status */
  static satellite_status_t *multisat_status = NULL;

  /* Change operating mode */
  if( isModeSet(END_MULTI_SAT) )
  {
	Clear_Mode( MULTI_SAT | END_MULTI_SAT );
	first_call = TRUE;
	sat_status->elev = 0.; /* Enable AOS/LOS recalculation */
	free_ptr( (void **)&multisat_status );
	g_object_unref( builder );
	return;
  }

  /* Initialize */
  if( first_call )
  {
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	/* Allocate memory for multisat_status */
	mem_alloc( (void **)&multisat_status,
		sizeof(satellite_status_t) * (size_t)num_sets,
		_("for multisat_status in Multi_Satellite_Mode()") );
	if( multisat_status == NULL )
	{
	  Set_Mode( END_MULTI_SAT );
	  return;
	}

	/* Create multi-sat window */
	multi_satellite = create_multi_sat( &builder );
	gtk_widget_show( multi_satellite );
	ClearFlag( FREEZE_MULTISAT );

	/* Create tree view and selection object */
	multisat_treeview = GTK_TREE_VIEW( Builder_Get_Object(
		  builder, "multisat_treeview") );
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (multisat_treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect( G_OBJECT(select), "changed",
		G_CALLBACK(tree_selection_changed_cb), sat_name );

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_("Satellite Name"),
		renderer, "text", NAME_COLUMN, NULL);

	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_toggle_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes ("AOS", renderer,
		"active", ACCESS_COLUMN, NULL);
	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_(" Azimuth "), renderer,
		"text", AZIM_COLUMN, NULL);

	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_("Elevation"), renderer,
		"text", ELEV_COLUMN, NULL);

	/* Add the column to the view. */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_("Range"), renderer,
		"text", RANGE_COLUMN, NULL);
	/* Add the column to the view. */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_("Phase"), renderer,
		"text", PHASE_COLUMN, NULL);
	/* Add the column to the view. */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell render */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_("Next AOS"), renderer,
		"text", AOS_COLUMN, NULL);
	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Create a cell render */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes (_("Next LOS"), renderer,
		"text", LOS_COLUMN, NULL);

	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multisat_treeview), column);

	/* Pre-process satellite/observer data */
	for( idx = 0; idx < num_sets; idx++ )
	  New_Status( TRUE, &element_sets[idx],
		  &multisat_status[idx], obs_status );

	Strlcpy( sat_name, sat_elem_set->name, sizeof(sat_name) );
	first_call = FALSE;

  } /* if( first_call ) */

  /* On satellite or observer change */
  if( change_sat_loc )
  {
	for( idx = 0; idx < num_sets; idx++ )
	  New_Status( TRUE, &element_sets[idx],
		  &multisat_status[idx], obs_status );

	Strlcpy( sat_name, sat_elem_set->name, sizeof(sat_name) );
  }
  /* Load new satellite data if selected in tree view */
  else if( strncmp(sat_name,
		sat_elem_set->name,
		strlen(sat_elem_set->name)) != 0 )
  {
	/* Load new satellite data */
	if( strlen(sat_name) == 0 ) return;
	Load_Tle_Set( sat_name, sat_elem_set );

	/* Pre-process satellite/observer data */
	New_Status( TRUE, sat_elem_set, sat_status, obs_status );

	/* Get UTC calendar and convert to Julian */
	Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
	julian_utc = Julian_Date( &obs_status->utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position( julian_utc, &solar_kinetics );

	/* Calculate single satellite status */
	Calculate_Satellite_Status(
		julian_utc, sat_elem_set,
		&solar_kinetics, &obs_kinetics,
		obs_status, sat_status );

	/* Re-initialize transceiver */
	if( isFlagSet(CAT_SETUP) )
	{
	  Read_Transponder_Data( sat_status->name );
	  mode_index = 0;
	  Initialize_Tcvr( DEFAULT_MODE );
	}

	change_sat_loc = TREE_SELECTION;
	gtk_widget_destroy( multi_satellite );
	Set_Mode( END_MULTI_SAT );

	return;
  } /* if( strcmp(sat_name, sat_elem_set->name) != 0 ) */

  /* Get UTC calendar and convert to Julian */
  Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
  julian_utc = Julian_Date( &obs_status->utc );

  /* Calculate topocentric position of sun */
  Calculate_Solar_Position(julian_utc, &solar_kinetics);

  /* Display observer status */
  Show_Observer_Status( builder, MULTI_SATELLITE, obs_status );
  Show_Satellite_Data(  builder, MULTI_SATELLITE, sat_status );

  /* Calculate multisatellite status */
  if( isFlagSet(FREEZE_MULTISAT) )
	return;
  vis_sats = 0;
  for( idx = 0; idx < num_sets; idx++ )
  {
	Calculate_Satellite_Status( julian_utc, &element_sets[idx],
		&solar_kinetics, &obs_kinetics,
		obs_status, &multisat_status[idx] );

	/* Count visible (elev>0) satellites */
	if( multisat_status[idx].elev > 0. )
	  vis_sats++;

	/* Clear satellite rank register */
	Clear_SatFlag( &multisat_status[idx], SAT_RANK_FLAGS );

  } /* End of for( idx = 0; idx <  num_sets; idx++ ) */

  /* Rank satellites: Top rank, visible satellites with furthest LOS */
  /* time. Lowest rank, invisible satellites with furthest AOS time. */
  /* Rank is entered in satellite status flags lowest byte.          */
  for( idx = 0; idx < num_sets; idx++ )
  {
	/* If sat below horizon, init. rank to number of visible sats */
	if( multisat_status[idx].elev <= 0. )
	  multisat_status[idx].flags += vis_sats;

	for( icp = 0; icp < num_sets; icp++ )
	{
	  /* If both sats visible, rank according to   */
	  /* > LOS and according to accessibility flag */
	  if( (multisat_status[icp].elev > 0.) &&
		  (multisat_status[idx].elev > 0.) )
	  {
		if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS)) &&
			(isSatFlagSet(&multisat_status[icp],NO_AOS_LOS)) &&
			(idx > icp) )
		  multisat_status[idx].flags++;

		if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS)) &&
			(isSatFlagClear(&multisat_status[icp],NO_AOS_LOS)) )
		  multisat_status[idx].flags++;

		if( (isSatFlagClear(&multisat_status[idx],NO_AOS_LOS)) &&
			(isSatFlagClear(&multisat_status[icp],NO_AOS_LOS)) )
		{
		  if(multisat_status[idx].los < multisat_status[icp].los)
			multisat_status[idx].flags++;
		  else if(
			  (multisat_status[idx].los == multisat_status[icp].los) &&
			  (idx > icp) )
			multisat_status[idx].flags++;
		}

	  } /* End of if( (multisat_status[icp].elev > 0.) && ... */


	  /* If both sats invisible, rank according to */
	  /* < AOS and according to accessibility flag */
	  if( (multisat_status[icp].elev <= 0.) &&
		  (multisat_status[idx].elev <= 0.) )
	  {
		if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS)) &&
			(isSatFlagSet(&multisat_status[icp],NO_AOS_LOS)) &&
			(idx > icp) )
		  multisat_status[idx].flags++;

		if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS)) &&
			(isSatFlagClear(&multisat_status[icp],NO_AOS_LOS)) )
		  multisat_status[idx].flags++;

		if( (isSatFlagClear(&multisat_status[idx],NO_AOS_LOS)) &&
			(isSatFlagClear(&multisat_status[icp],NO_AOS_LOS)) )
		{
		  if(multisat_status[idx].aos > multisat_status[icp].aos)
			multisat_status[idx].flags++;
		  else if(
			  (multisat_status[idx].aos == multisat_status[icp].aos) &&
			  (idx > icp) )
			multisat_status[idx].flags++;
		}

	  } /* End of if( (multisat_status[icp].elev <= 0.) && ... */

	} /* End of for( icp = 0; icp < num_sets; icp++ ) */

  } /* End of for( idx = 0; idx < num_sets; idx++ ) */

  /* Calculate single satellite status */
  Calculate_Satellite_Status( julian_utc, sat_elem_set,
	  &solar_kinetics, &obs_kinetics,
	  obs_status, sat_status );

  /* Display multi-satellite status */
  Show_Multisatellite_Status( multisat_treeview, multisat_status );

} /* End of Multi_Satellite_Mode() */

/*------------------------------------------------------------------*/

/*  Multi_Observer_Mode()
 *
 *  Single-satellite/Multi-observer tracking mode
 */

  static void
Multi_Observer_Mode(
	element_set_t *sat_elem_set,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  /* Multi-location window */
  static GtkTreeView      *multi_loc_treeview;
  static GtkTreeSelection *select;
  static GtkTreeStore     *store;
  GtkTreeIter iter1, iter2;
  static GtkTreePath *path = NULL;

  static GtkBuilder *builder = NULL;

  /* First function call flag */
  static int
	old_num_obs, /* To detect addition of a location */
	first_call = TRUE; /* First function call flag   */

  /* Index for loops etc */
  int idx;

  /* Used to detect location change */
  static char loc_name[NAME_SIZE];

  /* Julian UTC */
  double julian_utc;

  /* Multi-observer satellite status array */
  static satellite_status_t *mxsat_status = NULL;

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity vectors */
	solar_kinetics; /*     Solar position/velocity vectors */


  /* Change operating mode */
  if( isModeSet(END_MULTI_LOC) )
  {
	Clear_Mode( MULTI_LOC | END_MULTI_LOC );
	first_call = TRUE;
	g_object_unref( builder );
	return;
  }

  /* Initialize */
  if( first_call )
  {
	GtkTreeViewColumn *column = NULL;
	GtkCellRenderer *renderer;

	/* Create multi-loc window */
	multi_location = create_multi_loc( &builder );

	/* Create tree view and selection object */
	multi_loc_treeview =
	  GTK_TREE_VIEW( Builder_Get_Object(builder, "multiloc_treeview") );
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (multi_loc_treeview));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect( G_OBJECT(select), "changed",
		G_CALLBACK(tree_selection_changed_cb), loc_name );

	/*** Add the columns to the view ***/
	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes(
		"", renderer, "text", LABEL1_COLUMN, NULL);

	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multi_loc_treeview), column);

	/* Create a cell renderer */
	renderer = gtk_cell_renderer_text_new ();
	g_object_set (G_OBJECT (renderer),
		"foreground", "darkgreen", NULL);

	/* Create a column */
	column = gtk_tree_view_column_new_with_attributes(
		"", renderer, "text", DATA1_COLUMN, NULL);

	/* Add the column to the view */
	gtk_tree_view_append_column (GTK_TREE_VIEW (multi_loc_treeview), column);

	for( idx = 1; idx < NTREE_COLUMNS/2; idx++ )
	{
	  /* Create a cell renderer */
	  renderer = gtk_cell_renderer_text_new ();

	  /* Create a column */
	  column = gtk_tree_view_column_new_with_attributes (
		  "", renderer,
		  "text", 2*idx, NULL);

	  /* Add the column to the view */
	  gtk_tree_view_append_column (GTK_TREE_VIEW (multi_loc_treeview), column);

	  /* Create a cell renderer */
	  renderer = gtk_cell_renderer_text_new ();
	  g_object_set (G_OBJECT (renderer),
		  "foreground", "darkgreen", NULL);

	  /* Create a column */
	  column = gtk_tree_view_column_new_with_attributes ("", renderer,
		  "text", 2*idx+1, NULL);

	  /* Add the column to the view */
	  gtk_tree_view_append_column (GTK_TREE_VIEW (multi_loc_treeview), column);

	} /* for( idx = 1; idx < NTREE_COLUMNS/2; idx++ ) */

	/* Creat tree store */
	store = gtk_tree_store_new(
		NTREE_COLUMNS,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING );

	/* Initialize tree store */
	for( idx = 0; idx < num_obs; idx++ )
	{
	  gtk_tree_store_append (store, &iter1, NULL);
	  gtk_tree_store_set( store, &iter1,
		  LABEL1_COLUMN, observer_data[idx].loc_name,
		  -1 );

	  gtk_tree_store_append (store, &iter2, &iter1);
	  gtk_tree_store_set( store, &iter2,
		  LABEL1_COLUMN, _("Locator:"),
		  LABEL2_COLUMN, _("Longitude:"),
		  LABEL3_COLUMN, _("Latitude:"),
		  LABEL4_COLUMN, _("Height:"),
		  -1 );

	  gtk_tree_store_append (store, &iter2, &iter1);
	  gtk_tree_store_set( store, &iter2,
		  LABEL1_COLUMN, _("Date UTC:"),
		  LABEL2_COLUMN, _("Time UTC:"),
		  LABEL3_COLUMN, _("Sun Pos:"),
		  LABEL4_COLUMN, _("Moon Pos:"),
		  -1 );

	  gtk_tree_store_append (store, &iter2, &iter1);
	  gtk_tree_store_set( store, &iter2,
		  LABEL1_COLUMN, _("Satellite:"),
		  LABEL2_COLUMN, _("Azimuth:"),
		  LABEL3_COLUMN, _("Elevation:"),
		  LABEL4_COLUMN, _("Range:"),
		  -1 );

	  gtk_tree_store_append (store, &iter2, &iter1);
	  gtk_tree_store_set( store, &iter2,
		  LABEL1_COLUMN, _("Next AOS:"),
		  LABEL2_COLUMN, _("Next LOS:"),
		  LABEL3_COLUMN, _("Ovlap Fm:"),
		  LABEL4_COLUMN, _("Ovlap To:"),
		  -1 );

	} /* for( idx = 1; idx < num_obs; idx++ ) */

	gtk_tree_view_set_model( multi_loc_treeview, GTK_TREE_MODEL(store) );
	gtk_widget_show(multi_location);

	/* Process observers */
	satellite_status_t *ptr = (satellite_status_t *)realloc(
		mxsat_status, (size_t)num_obs * sizeof(satellite_status_t) );
	if( ptr == NULL )
	{
	  free( mxsat_status );
	  Error_Dialog( Abort_On_Error(-8), FATAL );
	  return;
	}
	mxsat_status = ptr;

	for( idx = 0; idx < num_obs; idx++ )
	  New_Status( TRUE, sat_elem_set,
		  &mxsat_status[idx],
		  &observer_data[idx] );

	Strlcpy( loc_name, obs_status->loc_name, sizeof(loc_name) );
	old_num_obs = num_obs;
	first_call = FALSE;

  } /* if( first_call ) */

  /* On addition of an observer */
  if( old_num_obs != num_obs )
  {
	old_num_obs = num_obs;

	/* Reallocate memory to buffer */
	satellite_status_t *ptr = (satellite_status_t *)realloc(
		mxsat_status, (size_t)num_obs * sizeof(satellite_status_t) );
	if( ptr == NULL )
	{
	  free( mxsat_status );
	  Error_Dialog( Abort_On_Error(-8), FATAL );
	  return;
	}
	mxsat_status = ptr;

	/* Append new location to store */
	gtk_tree_store_append (store, &iter1, NULL);
	gtk_tree_store_set( store, &iter1,
		LABEL1_COLUMN, observer_data[num_obs-1].loc_name,
		-1 );

	gtk_tree_store_append (store, &iter2, &iter1);
	gtk_tree_store_set( store, &iter2,
		LABEL1_COLUMN, _("Locator:"),
		LABEL2_COLUMN, _("Longitude:"),
		LABEL3_COLUMN, _("Latitude:"),
		LABEL4_COLUMN, _("Height:"),
		-1 );

	gtk_tree_store_append (store, &iter2, &iter1);
	gtk_tree_store_set( store, &iter2,
		LABEL1_COLUMN, _("Date UTC:"),
		LABEL2_COLUMN, _("Time UTC:"),
		LABEL3_COLUMN, _("Sun Pos:"),
		LABEL4_COLUMN, _("Moon Pos:"),
		-1 );

	gtk_tree_store_append (store, &iter2, &iter1);
	gtk_tree_store_set( store, &iter2,
		LABEL1_COLUMN, _("Satellite:"),
		LABEL2_COLUMN, _("Azimuth:"),
		LABEL3_COLUMN, _("Elevation:"),
		LABEL4_COLUMN, _("Range:"),
		-1 );

	gtk_tree_store_append (store, &iter2, &iter1);
	gtk_tree_store_set( store, &iter2,
		LABEL1_COLUMN, _("Next AOS:"),
		LABEL2_COLUMN, _("Next LOS:"),
		LABEL3_COLUMN, _("Ovlap Fm:"),
		LABEL4_COLUMN, _("Ovlap To:"),
		-1 );

	/* Process new observer */
	New_Status( TRUE, sat_elem_set,
		&mxsat_status[num_obs-1],
		&observer_data[num_obs-1] );

  } /* if( old_num_obs != num_obs ) */

  /* On satellite change */
  if( change_sat_loc )
  {
	/* Re-process observers */
	for( idx = 0; idx < num_obs; idx++ )
	  New_Status( TRUE, sat_elem_set, &mxsat_status[idx], &observer_data[idx] );

	Strlcpy( loc_name, obs_status->loc_name, sizeof(loc_name) );
  }
  /* On location change */
  else if( strncmp(obs_status->loc_name, loc_name, strlen(loc_name)) != 0 )
  {
	/* Load observer selected in tree view */
	Load_Observer_Set( loc_name, obs_status );

	/* Re-process observers */
	for( idx = 1; idx < num_obs; idx++ )
	  New_Status(
		  TRUE, sat_elem_set,
		  &mxsat_status[idx],
		  &observer_data[idx] );

	/* Re-process observer/satellite */
	New_Status( FALSE, sat_elem_set, sat_status, obs_status );

	change_sat_loc = TREE_SELECTION;
  }

  /* Set default satellite data */
  mxsat_status[0] = *sat_status;

  /* Calculate utc */
  Calendar_Time_Now( &observer_data[0].utc, &observer_data[0].localtm );
  julian_utc = Julian_Date( &observer_data[0].utc );

  /* Calculate topocentric position of sun */
  Calculate_Solar_Position( julian_utc, &solar_kinetics );

  path = gtk_tree_path_new_first();
  for( idx = 0; idx < num_obs; idx++ )
  {
	if( gtk_tree_view_row_expanded(multi_loc_treeview, path) )
	{
	  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	  Calculate_Satellite_Status(
		  julian_utc, sat_elem_set,
		  &solar_kinetics,
		  &obs_kinetics,
		  &observer_data[idx],
		  &mxsat_status[idx] );

	  /* Calculate lunar azimuth and elevation */
	  Calculate_Moon_Position(julian_utc, &observer_data[idx], &obs_kinetics);
	}

	gtk_tree_path_next( path );

  } /* for( idx = 1; idx < num_obs; idx++ ) */

  /* Display current observer and satellite status */
  Show_Multiobserver_Status( multi_loc_treeview, store, mxsat_status );
  Show_Observer_Status( builder, MULTI_LOCATION, obs_status );
  Show_Satellite_Data(  builder, MULTI_LOCATION, sat_status );

  gtk_tree_path_free(path);

} /* End of Multi_Observer_Mode() */

/*------------------------------------------------------------------*/

/*  Central_Mode()
 *
 *  Displays the main detailed menu startup screen of xsatcom.
 *  Runs as a 1-second timeout callback.
 */

  gboolean
Central_Mode( gpointer data )
{
  /* First function call flag */
  static gboolean block = FALSE, first_call = TRUE;


  /* Abort if it is re-entry due to timeout */
  if( block ) return( TRUE );
  block = TRUE;

  /*** Initialize for first call ***/
  if( first_call )
  {
	first_call = FALSE;

	/* Load 'Home' oserver form database */
	Load_Observer_Set( "Home", &observer_status);

	/* Input the default TLE set from TLE file  */
	Load_Tle_Set( "", &satellite_element_set );

	/* Pre-process satellite/observer data */
	New_Status(
		TRUE,
		&satellite_element_set,
		&satellite_status,
		&observer_status );

  } /* if( first_call ) */

  /* Reload data if new TLE set downloaded */
  if( change_sat_loc == DNLD_TLE_SETS )
  {
	char name[NAME_SIZE];

	/* Reload new TLE set */
	Strlcpy( name, satellite_element_set.name, sizeof(name) );
	Load_Tle_Set( name, &satellite_element_set );

	/* Pre-process satellite/observer data */
	New_Status(
		TRUE,
		&satellite_element_set,
		&satellite_status,
		&observer_status );
  }

  /*** Activate selected modes ***/

  /* Single satellite/observer mode */
  if( isModeSet(SINGLE_SAT) )
	Single_Satellite_Mode(
		&satellite_element_set,
		&observer_status,
		&satellite_status );

  /* Multi-satellite/single-observer mode */
  if( isModeSet(MULTI_SAT) )
	Multi_Satellite_Mode(
		&satellite_element_set,
		&observer_status,
		&satellite_status );

  /* Multi-observer/single-satellite mode */
  if( isModeSet(MULTI_LOC) )
	Multi_Observer_Mode(
		&satellite_element_set,
		&observer_status,
		&satellite_status );

  /* Pass prediction mode */
  if( isModeSet(PASS_PRED) && (pass_drawingarea != NULL) )
	gtk_widget_queue_draw( pass_drawingarea );

  /* Illumination prediction mode */
  if( isModeSet(ILLUM_PRED) && (illum_drawingarea != NULL) )
	gtk_widget_queue_draw( illum_drawingarea );

  block = FALSE;
  if( isModeClear(ILLUM_PRED) &&
	  isModeClear(PASS_PRED) )
	change_sat_loc = CLEAR_CHANGE;

  return( TRUE );

} /* End of Central_Mode() */

/*------------------------------------------------------------------*/

/*  Pass_Predict_Mode()
 *
 *  Predictions of satellite passes
 */

  void
Pass_Predict_Mode( cairo_t *cr )
{
  /* Number of predictions */
  static int num_pred;

  /* Pass predict window */
  static GtkBuilder *builder = NULL;

  int
	idx, /* Index for loops etc */
	num; /* Number of entries to pass_data */

  static double
	julian_utc, /* Julian UTC in calculations */
	time_now,   /* Real time julian UTC       */
	los_utc,    /* LOS time for start of prediction */
	aos_utc,    /* AOS time for start of prediction */
	from_utc,   /* Start time of pass predictions   */
	to_utc,     /* Ending time of pass predictions  */
	step_utc,   /* Time step for pass predictions   */
	old_utc,    /* UTC used in previous prediction  */
	old_los;    /* Stored LOS time, needed for flow control */

  /* Observer and solar position/velocity */
  kinetics_t
	solar_kinetics,
	obs_kinetics;

  /* Temporary observer and satellite status used in predictions */
  static satellite_status_t temp_sat_status;
  static observer_status_t  temp_obs_status;
  static satellite_status_t old_sat_status;

  /* Pass prediction data */
  static pass_data_t *pass_data = NULL;


  /* Signal to initialize */
  if( cr == NULL )
  {
	/* Pango layout for plot data */
	PangoLayout *layout;

	/* Create pass predict window */
	if( builder != NULL )
	{
	  g_object_unref( builder );
	  builder = NULL;
	}
	pass_predict = create_pass_pred( &builder );
	gtk_widget_show( pass_predict );
	pass_drawingarea = Builder_Get_Object( builder, "pass_drawingarea" );

	/* Calculate number of prediction points */
	layout = gtk_widget_create_pango_layout( pass_drawingarea, "X0" );
	pango_layout_get_pixel_size( layout, &num, &idx );
	GtkAllocation alloc;
	gtk_widget_get_allocation( pass_drawingarea, &alloc );
	num_pred = alloc.height / idx;
	g_object_unref( layout );

	/* Allocate memory for pass pred data */
	if( pass_data == NULL )
	  mem_alloc( (void **)&pass_data,
		  sizeof(pass_data_t) * (size_t)num_pred,
		  _("for pass_data in Pass_Predict_Mode()") );
	if( pass_data == NULL ) return;

	/* Signal pass predict mode ready */
	Set_Mode( PASS_PRED );

	/* Initialize for pass predictions */
	temp_sat_status = satellite_status;
	temp_obs_status = observer_status;
	ClearFlag( VISIBLE_PREDICT | USERTIME_PREDICT );

	return;
  } /* if( cr == NULL ) */

  /* On satellite or observer change */
  if( change_sat_loc )
  {
	ClearFlag( USERTIME_PREDICT );
	temp_sat_status = satellite_status;
	temp_obs_status = observer_status;
	change_sat_loc = CLEAR_CHANGE;
  }

  /* Move pass prediction forward */
  if( isModeSet(PASS_PRED_FWD) )
  {
	Clear_Mode( PASS_PRED_FWD );
	if( isFlagClear(USERTIME_PREDICT) )
	  SetFlag( USERTIME_PREDICT );

	/* Move ahead of pass prediction end time by ~5 min */
	from_utc = to_utc + 0.035;
  }

  /* Reset pass predictions to real time */
  if( isModeSet(PASS_PRED_RESET) )
  {
	Clear_Mode( PASS_PRED_RESET );
	ClearFlag( USERTIME_PREDICT );
  }

  /* Do accessible pass predictions */
  if( isModeSet(PASS_PRED_ACCESS) )
  {
	Clear_Mode( PASS_PRED_ACCESS );
	ClearFlag( VISIBLE_PREDICT | USERTIME_PREDICT );
	old_los = 0.;
  }

  /* Calculate time now as julian UTC */
  Calendar_Time_Now( &observer_status.utc, &observer_status.localtm );
  time_now = Julian_Date( &observer_status.utc );

  /* Calculate topocentric position of sun */
  Calculate_Solar_Position( time_now, &solar_kinetics );

  /* Display observer status */
  Show_Observer_Status( builder, PASS_PREDICT, &observer_status );
  Show_Satellite_Data(  builder, PASS_PREDICT, &satellite_status );

  /* If satellite is accessible and we have */
  /* AOS/LOS data then do pass predictions  */
  if( isSatFlagClear(&satellite_status, NO_AOS_LOS) )
  {
	/* Number of entries in prediction chart */
	num = num_pred-1;

	/* Initialize for real-time pass predictions */
	if( isFlagClear(USERTIME_PREDICT) )
	{
	  temp_sat_status = satellite_status;
	  temp_obs_status = observer_status;

	  /* If aos > los, satellite is above horizon.  */
	  /* Predictions are done starting in real time */
	  if( temp_sat_status.aos > temp_sat_status.los )
		aos_utc = time_now;
	  else
		aos_utc = temp_sat_status.aos;

	  los_utc = temp_sat_status.los;

	} /* End of if( isFlagClear(USERTIME_PREDICT) ) */
	else if( from_utc != old_utc )
	{
	  /* Enable AOS/LOS calculation */
	  temp_sat_status.elev = 0.;

	  /* Calculate topocentric position of sun */
	  Calculate_Solar_Position( from_utc, &solar_kinetics );

	  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	  Calculate_Satellite_Status(
		  from_utc,
		  &satellite_element_set,
		  &solar_kinetics,
		  &obs_kinetics,
		  &temp_obs_status,
		  &temp_sat_status );

	  /* If aos > los, satellite is above horizon.  */
	  /* Predictions are done starting in user time */
	  if( temp_sat_status.aos > temp_sat_status.los )
		aos_utc = from_utc;
	  else
		aos_utc = temp_sat_status.aos;

	  los_utc = temp_sat_status.los;

	} /* End of if( from_utc != old_utc ) */

	/* Look for (visually) visible passes.  */
	/* Loop till a visible pass is detected */
	if( isFlagSet(VISIBLE_PREDICT) &&
		isSatFlagClear(&temp_sat_status, PASS_VISIBLE) )
	{
	  if( old_los != temp_sat_status.los )
	  {
		old_los = temp_sat_status.los;
		while( isSatFlagClear(&temp_sat_status, PASS_VISIBLE) )
		{
		  /* Move 5 min ahead of current LOS time */
		  julian_utc = temp_sat_status.los + 0.0035;

		  /* Enable AOS/LOS calculation */
		  temp_sat_status.elev = 0.;

		  /* Calculate topocentric position of sun */
		  Calculate_Solar_Position( julian_utc, &solar_kinetics );

		  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
		  Calculate_Satellite_Status(
			  julian_utc,
			  &satellite_element_set,
			  &solar_kinetics,
			  &obs_kinetics,
			  &temp_obs_status,
			  &temp_sat_status );

		} /* End of while( isSatFlagClear(&temp_sat_status, ... ) */

		/* If aos > los, satellite is above horizon.  */
		/* Predictions are done starting in real time */
		if( temp_sat_status.aos > temp_sat_status.los )
		  aos_utc = time_now;
		else
		  aos_utc = temp_sat_status.aos;

		los_utc = temp_sat_status.los;

		/* Calculate time-step for pass plot */
		from_utc = aos_utc;
		to_utc   = los_utc;
		step_utc = (los_utc - aos_utc)/(double)(num-1);

	  } /* if( old_los != temp_sat_status.los ) */

	} /* End of isFlagSet(VISIBLE_PREDICT) ) */
	else
	{
	  /* Calculate time-step for pass plot */
	  from_utc = aos_utc;
	  to_utc   = los_utc;
	  step_utc = (los_utc - aos_utc)/(double)(num-1);
	}

	/* Refresh predictions if from_utc changes */
	if( from_utc != old_utc )
	{
	  /* Save temporary sat_status */
	  old_sat_status = temp_sat_status;

	  /* Setup start time and save original value */
	  julian_utc = old_utc = from_utc;

	  /* Suppress AOS/LOS computations while plotting pass */
	  Set_SatFlag( &temp_sat_status, NO_AOS_LOS );

	  /* Calculate pass prediction data */
	  for( idx = 0; idx < num; idx++ )
	  {
		/* Calculate topocentric position of sun */
		Calculate_Solar_Position( julian_utc, &solar_kinetics );

		/* Calculate sat az/el/range/lat/lon/alt/visibility etc */
		Calculate_Satellite_Status(
			julian_utc,
			&satellite_element_set,
			&solar_kinetics,
			&obs_kinetics,
			&temp_obs_status,
			&temp_sat_status );

		/* Enter pass prediction data */
		pass_data[idx].flags  = temp_sat_status.flags;
		pass_data[idx].elev   = temp_sat_status.elev;
		pass_data[idx].azim   = temp_sat_status.azim;
		pass_data[idx].range  = temp_sat_status.range;
		pass_data[idx].lon    = temp_sat_status.ssplon;
		pass_data[idx].lat    = temp_sat_status.ssplat;
		pass_data[idx].phs256 = temp_sat_status.phs256;
		pass_data[idx].edepth = temp_sat_status.edepth;
		pass_data[idx].julian_utc = julian_utc;

		julian_utc += step_utc;

	  } /* End of for( idx = 0; idx < num; idx++ ) */

	  /* Restore AOS/LOS search computations */
	  Clear_SatFlag( &temp_sat_status, NO_AOS_LOS );

	  /* Restore temporary sat_status */
	  temp_sat_status = old_sat_status;

	} /* End of if( from_utc != old_utc ) */

  } /* End of if(isSatFlagClear(sat_status, NO_AOS_LOS)) */
  else num = 0; /* No entries in prediction chart */

  Show_Pass_Predict( cr, builder, &satellite_status, pass_data, num );

} /* End of Pass_Predict_Mode() */

/*------------------------------------------------------------------*/

/*  Illumination_Predict_Mode()
 *
 *  Predicitons of satellite illumination
 */

static int illum[30];    /* Illumunation prediction values */
static double jdate[30]; /* Dates of illumination predictions */

  void
Illumination_Predict_Mode( cairo_t *cr )
{
  /* Pass predict window */
  static GtkBuilder *builder = NULL;

  /* Real time julian UTC */
  double time_now;

  /* Time to start predictions from */
  static double from_utc;

  /* Solar, observer and satellite position/velocity */
  kinetics_t solar_kinetics, sat_kinetics;


  /* Signal to initialize */
  if( cr == NULL )
  {
	/* Create illumination predict window and plots objects */
	if( builder != NULL )
	{
	  g_object_unref( builder );
	  builder = NULL;
	}
	illum_predict = create_illum_pred( &builder );
	gtk_widget_show( illum_predict );
	illum_drawingarea = Builder_Get_Object(	builder, "illum_drawingarea" );

	from_utc = Julian_Date( &observer_status.utc );
	Clear_Mode( ILLUM_PRED_DONE );
	Set_Mode( ILLUM_PRED );

	return;
  } /* if(  cr == NULL ) */

  Calendar_Time_Now( &observer_status.utc, &observer_status.localtm );
  time_now = Julian_Date( &observer_status.utc );

  /* Forward a month */
  if( isModeSet(ILLUM_PRED_FWD) )
  {
	from_utc += 30.0; /* + 30 days */
	Clear_Mode( ILLUM_PRED_DONE | ILLUM_PRED_FWD );
  }

  /* Back a month */
  if( isModeSet(ILLUM_PRED_BACK) )
  {
	from_utc -= 30.0; /* - 30 days */
	Clear_Mode( ILLUM_PRED_DONE | ILLUM_PRED_BACK );
  }

  /* Reset to real time */
  if( isModeSet(ILLUM_PRED_REDO) )
  {
	from_utc = time_now;
	Clear_Mode( ILLUM_PRED_DONE | ILLUM_PRED_REDO );
  }

  /* On satellite or observer change */
  if( change_sat_loc )
  {
	Clear_Mode( ILLUM_PRED_DONE );
	change_sat_loc = CLEAR_CHANGE;
  }

  /* Display observer status */
  Show_Observer_Status( builder, ILLUM_PREDICT, &observer_status );
  Show_Satellite_Data(  builder, ILLUM_PREDICT, &satellite_status );

  /* Do illumination predictions for a month */
  if( isModeClear(ILLUM_PRED_DONE) )
  {
	int
	  idx,	/* Index for loops etc */
	  ilm;	/* Index used in illumination predictions */

	double
	  julian_utc, /* Julian time UTC */
	  age,        /* Days since epoch    */
	  tsince;     /* Minutes since epoch */

	julian_utc = from_utc;
	for( idx = 0; (idx < 30); idx++ )
	{
	  illum[idx] = 0;
	  jdate[idx] = julian_utc;

	  for( ilm = 0; ilm < 1440; ilm++ )
	  {
		/* Calculate topocentric position of sun */
		Calculate_Solar_Position( julian_utc, &solar_kinetics );

		/* Calculate time since epoch in minutes */
		age = julian_utc - satellite_element_set.jdsatepoch;
		tsince = age * xmnpda;

		/* Call NORAD routines according to deep-space flag */
		sgp4( &satellite_element_set, tsince, &sat_kinetics );

		/* Scale position and velocity vectors to km and km/sec */
		Convert_Sat_State( &sat_kinetics );

		/* Calculate scalar magnitude of velocity */
		Magnitude( &sat_kinetics.vel );

		/* Calculate satellite's illumination */
		if( !Satellite_Eclipsed(
			  &sat_kinetics, &solar_kinetics,
			  &satellite_status.edepth) )
		  illum[idx]++;

		/* Step time by 1 minute of day */
		julian_utc += 6.944444E-4;

	  }/* End of for( ilm = 0; ilm < 1440; ilm++ ) */

	} /* End of for( idx = 0; idx < 30; idx++ ) */

	Set_Mode( ILLUM_PRED_DONE );
  } /* if( isModeClear(ILLUM_PRED_DONE) ) */

} /* End of Illumination_Predict_Mode() */

/*------------------------------------------------------------------*/

/* Draw_Illum_Predict()
 *
 * Draws the illumination predictions
 */
  void
Draw_Illum_Predict( cairo_t *cr )
{
  Illumination_Predict_Mode( cr );
  Show_Illumination_Predict( cr, jdate, illum );
}

/*------------------------------------------------------------------*/

/*  Manual_Position_Rotors()
 *
 *  Move az/el rotors to position usually related to a
 *  manual operation like up/down left/right or stop
 */

  void
Manual_Position_Rotors( int azim, int elev )
{
  /* Open rotor's serial port if closed */
  if( isFlagClear(ROTORS_ENABLED) )
	if( Open_Rotor_Serial() )
	{
	  SetFlag( ROTORS_ENABLED );
	  ClearFlag(
		  ROTORS_RUNNING   |
		  TRACKING_ENABLED |
		  REVERSE_TRACK );

	  Rotor_Send_String( "S" ); /* All stop */
	  Show_Rotor_Control( azim, elev, rc_data.high_resol );
	}

} /* End of Manual_Position_Rotors() */

/*------------------------------------------------------------------*/

/*  Auto_Track()
 *
 *  Enables or disables satellite tracking automatically

  static void
Auto_Track( double julian_utc, satellite_status_t *sat_status )
{
  Enable tracking about 5 min before AOS
  if( julian_utc >= (sat_status->aos - 0.0035) &&
	  isFlagClear(TRACKING_ENABLED) )
	Enable_Tracking();

  Disable tracking about 5 min after LOS
  if( julian_utc >= (sat_status->los + 0.0035) &&
	  isFlagSet(TRACKING_ENABLED) )
	Disable_Tracking();

} End of Auto_Track() */

/*------------------------------------------------------------------*/

/* Functions for testing and setting/clearing Mode flags
 *
 *  See xsatcom.h for definition of Mode control flags
 */

/* An int variable holding the single-bit flags */
static int Flags = SINGLE_SAT;

  int
isModeSet(int flag)
{
  return( (Flags & flag) == flag );
}

  int
isModeClear(int flag)
{
  return( (~Flags & flag) == flag );
}

  void
Set_Mode(int flag)
{
  Flags |= flag;
}

  void
Clear_Mode(int flag)
{
  Flags &= ~flag;
}

  void
Toggle_Mode(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

